// Copyright 2011-2020 Wason Technology, LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//    http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#include "HardwareTransport_libusb_private.h"
#include "HardwareTransport_private.h"
#include <boost/foreach.hpp>
#include <boost/range/algorithm.hpp>
#include <boost/range/adaptors.hpp>
#include <boost/locale.hpp>
#include <boost/algorithm/string.hpp>
#include <dlfcn.h>

// cSpell: ignore SYSFS, Wunused
#define SYSFS_MOUNT_PATH "/sys"
#define SYSFS_DEVICE_PATH SYSFS_MOUNT_PATH "/bus/usb/devices"
#define USB_MAX_DEPTH 7

namespace RobotRaconteur
{
namespace detail
{

// LibUsb_Functions
LibUsb_Functions::LibUsb_Functions()
{
    lib_handle = NULL;
    LIBUSB_FUNCTIONS_INIT(LIBUSB_FUNCTIONS_PTR_VOID);
}

bool LibUsb_Functions::LoadFunctions()
{
    if (lib_handle)
        throw InvalidOperationException("libusb functions already loaded");
    lib_handle = dlopen("libusb-1.0.so", RTLD_LAZY);
    if (!lib_handle)
        return false;

    LIBUSB_FUNCTIONS_INIT(LIBUSB_FUNCTIONS_PTR_INIT);

    return true;
}

LibUsb_Functions::~LibUsb_Functions()
{
    if (lib_handle)
    {
        dlclose(lib_handle);
        lib_handle = NULL;
    }
}
// End LibUsb_Functions

static void libusb_status_to_ec(libusb_transfer_status err, boost::system::error_code& ec)
{
    switch (err)
    {
    case LIBUSB_TRANSFER_COMPLETED:
        ec = boost::system::error_code();
        break;
    case LIBUSB_TRANSFER_TIMED_OUT:
        ec = boost::asio::error::make_error_code(boost::asio::error::timed_out);
        break;
    case LIBUSB_TRANSFER_CANCELLED:
        ec = boost::asio::error::make_error_code(boost::asio::error::operation_aborted);
        break;
    case LIBUSB_TRANSFER_NO_DEVICE:
        ec = boost::asio::error::make_error_code(boost::asio::error::host_not_found);
        break;
    default:
        ec = boost::asio::error::make_error_code(boost::asio::error::broken_pipe);
        break;
    }
}

#ifdef __GNUG__
#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wunused-function"
#endif
static void libusb_error_to_ec(int err, boost::system::error_code& ec)
{
    if (err >= 0)
    {
        ec = boost::system::error_code();
        return;
    }

    switch (err)
    {
    case LIBUSB_ERROR_TIMEOUT:
        ec = boost::asio::error::make_error_code(boost::asio::error::timed_out);
        break;
    case LIBUSB_ERROR_NO_DEVICE:
    case LIBUSB_ERROR_NOT_FOUND:
        ec = boost::asio::error::make_error_code(boost::asio::error::host_not_found);
        break;
    case LIBUSB_ERROR_BUSY:
        ec = boost::asio::error::make_error_code(boost::asio::error::connection_refused);
        break;
    default:
        ec = boost::asio::error::make_error_code(boost::asio::error::broken_pipe);
        break;
    }
}
#ifdef __GNUG__
#pragma GCC diagnostic pop
#endif

// LibUsb_Transfer

LibUsb_Transfer::LibUsb_Transfer(const RR_SHARED_PTR<LibUsb_Functions>& f,
                                 const RR_SHARED_PTR<libusb_device_handle>& device_handle,
                                 const RR_SHARED_PTR<LibUsbDeviceManager>& device_manager)
    : ref_count(0)
{
    this->f = f;
    this->device_handle = device_handle;
    this->transfer = NULL;
    this->device_manager = device_manager;

    thread_pool = device_manager->GetNode()->GetThreadPool();
}

LibUsb_Transfer::~LibUsb_Transfer()
{
    if (transfer != NULL)
    {
        f->libusb_free_transfer(transfer);
        transfer = NULL;
    }
}

void intrusive_ptr_release(LibUsb_Transfer* p)
{
    if (!p)
        return;
    p->ref_count--;
    if (p->ref_count == 0)
    {
        delete p; // NOLINT(cppcoreguidelines-owning-memory)
    }
}
void intrusive_ptr_add_ref(LibUsb_Transfer* p)
{
    if (!p)
        return;
    p->ref_count++;
}

LibUsb_Transfer_control::LibUsb_Transfer_control(const RR_SHARED_PTR<LibUsb_Functions>& f,
                                                 const RR_SHARED_PTR<libusb_device_handle>& device_handle,
                                                 const RR_SHARED_PTR<LibUsbDeviceManager>& device_manager)
    : LibUsb_Transfer(f, device_handle, device_manager)
{}

void LibUsb_Transfer_control::FillTransfer(uint8_t bmRequestType, uint8_t bRequest, uint16_t wValue, uint16_t wIndex,
                                           boost::asio::mutable_buffer& buf,
                                           boost::function<void(const boost::system::error_code&, size_t)> handler)
{
    size_t s = 8 + boost::asio::buffer_size(buf);
    transfer = f->libusb_alloc_transfer(0);
    if (!transfer)
        throw SystemResourceException("Memory error");
    this->temp_buf = boost::shared_array<uint8_t>(new uint8_t[s]);
    this->data_buf = buf;
    boost::asio::buffer_copy(boost::asio::buffer(this->temp_buf.get(), s) + 8, buf);
    libusb_fill_control_setup(this->temp_buf.get(), bmRequestType, bRequest, wValue, wIndex,
                              boost::asio::buffer_size(buf));

    libusb_fill_control_transfer(transfer, device_handle.get(), temp_buf.get(), &LibUsbDeviceManager::transfer_complete,
                                 this, 5000);

    this->handler = RR_MOVE(handler);
    ref_count++;
}

void LibUsb_Transfer_control::CompleteTransfer()
{
    RR_SHARED_PTR<ThreadPool> p = thread_pool.lock();
    if (!p)
        return;

    if (transfer->status != LIBUSB_TRANSFER_COMPLETED)
    {
        boost::system::error_code ec1;

        libusb_status_to_ec(transfer->status, ec1);

        p->Post(boost::bind(handler, ec1, 0));

        return;
    }

    size_t actual_length = transfer->actual_length;
    uint8_t* temp_buf1 = libusb_control_transfer_get_data(transfer);
    boost::asio::buffer_copy(data_buf, boost::asio::buffer(temp_buf1, actual_length));

    boost::system::error_code ec2;

    p->Post(boost::bind(handler, ec2, actual_length));
}

LibUsb_Transfer_control::~LibUsb_Transfer_control() {}

void LibUsb_Transfer_bulk::FillTransfer(uint8_t ep, boost::asio::mutable_buffer& buf,
                                        boost::function<void(const boost::system::error_code&, size_t)> handler)
{
    transfer = f->libusb_alloc_transfer(0);
    if (!transfer)
        throw SystemResourceException("Memory error");
    this->data_buf = buf;

    libusb_fill_bulk_transfer(transfer, device_handle.get(), ep, RR_BOOST_ASIO_BUFFER_CAST(uint8_t*, buf),
                              boost::numeric_cast<int>(boost::asio::buffer_size(buf)),
                              &LibUsbDeviceManager::transfer_complete, this, 0);

    this->handler = RR_MOVE(handler);
    ref_count++;
}

void LibUsb_Transfer_bulk::CompleteTransfer()
{
    RR_SHARED_PTR<ThreadPool> p = thread_pool.lock();
    if (!p)
        return;

    if (transfer->status != LIBUSB_TRANSFER_COMPLETED)
    {
        boost::system::error_code ec1;

        libusb_status_to_ec(transfer->status, ec1);

        p->Post(boost::bind(handler, ec1, 0));

        return;
    }

    size_t actual_length = transfer->actual_length;

    boost::system::error_code ec2;

    handler(ec2, actual_length);
}

LibUsb_Transfer_bulk::LibUsb_Transfer_bulk(const RR_SHARED_PTR<LibUsb_Functions>& f,
                                           const RR_SHARED_PTR<libusb_device_handle>& device_handle,
                                           const RR_SHARED_PTR<LibUsbDeviceManager>& device_manager)
    : LibUsb_Transfer(f, device_handle, device_manager)
{}

LibUsb_Transfer_bulk::~LibUsb_Transfer_bulk() {}

// End LibUsb_Transfer

// LibUsbDeviceManager

LibUsbDeviceManager::LibUsbDeviceManager(const RR_SHARED_PTR<HardwareTransport>& parent) : UsbDeviceManager(parent)
{
    running = false;
    hotplug_cb_handle = 0;
}

LibUsbDeviceManager::~LibUsbDeviceManager() {}

static void LibUsb_Functions_libusb_exit(const RR_SHARED_PTR<LibUsb_Functions>& f, libusb_context* c)
{
    f->libusb_exit(c);
}

static void LibUsb_Functions_libusb_unref(const RR_SHARED_PTR<LibUsb_Functions>& f, libusb_device* dev)
{
    f->libusb_unref_device(dev);
}

static UsbDeviceStatus LibUsbDevice_open_device(const RR_SHARED_PTR<LibUsbDeviceManager>& m,
                                                const RR_SHARED_PTR<LibUsb_Functions>& f,
                                                const RR_SHARED_PTR<libusb_device>& device,
                                                RR_SHARED_PTR<void>& device_handle)
{
    libusb_device_handle* dev_h = NULL;
    int res = f->libusb_open(device.get(), &dev_h);
    if (res != 0 || !dev_h)
    {
        if (res == LIBUSB_ERROR_BUSY)
        {
            return Busy;
        }
        if (res == LIBUSB_ERROR_ACCESS)
        {
            return Unauthorized;
        }
        return Error;
    }

    RR_WEAK_PTR<LibUsbDeviceManager> m1 = m;

    RR_SHARED_PTR<libusb_device_handle> dev_h1(
        dev_h, boost::bind(&LibUsbDeviceManager::LibUsbCloseDevice, m1, f, RR_BOOST_PLACEHOLDERS(_1)));

    device_handle = dev_h1;
    return Open;
}

bool LibUsbDeviceManager::InitUpdateDevices()
{
    if (!f)
    {
        RR_SHARED_PTR<LibUsb_Functions> f1 = RR_MAKE_SHARED<LibUsb_Functions>();
        if (!f1->LoadFunctions())
        {
            return false;
        }

        f = f1;
    }

    if (!context)
    {
        libusb_context* c = NULL;
        if (f->libusb_init(&c) != 0)
        {
            return false;
        }

        context.reset(c, boost::bind(&LibUsb_Functions_libusb_exit, f, RR_BOOST_PLACEHOLDERS(_1)));

        f->libusb_hotplug_register_callback(
            context.get(),
            static_cast<libusb_hotplug_event>(LIBUSB_HOTPLUG_EVENT_DEVICE_ARRIVED | LIBUSB_HOTPLUG_EVENT_DEVICE_LEFT),
            static_cast<libusb_hotplug_flag>(0), LIBUSB_HOTPLUG_MATCH_ANY, LIBUSB_HOTPLUG_MATCH_ANY,
            LIBUSB_HOTPLUG_MATCH_ANY, &LibUsbDeviceManager::OnUsbHotplugEvent, this, &hotplug_cb_handle);

        running = true;
        usb_thread = boost::thread(boost::bind(&LibUsbDeviceManager::UsbThread,
                                               RR_STATIC_POINTER_CAST<LibUsbDeviceManager>(shared_from_this())));
    }

    return true;
}

std::list<UsbDeviceManager_detected_device> LibUsbDeviceManager::GetDetectedDevicesPaths()
{
    std::list<UsbDeviceManager_detected_device> devices;

    libusb_device** list1 = NULL; // cppcheck-suppress nullPointerRedundantCheck
    ssize_t device_count = f->libusb_get_device_list(context.get(), &list1);
    if (list1 == NULL || device_count <= 0)
    {
        f->libusb_free_device_list(list1, 1);
        return devices;
    }

    for (ssize_t i = 0; i < device_count; i++)
    {
        int bus_num = f->libusb_get_bus_number(list1[i]);
        int port_num = f->libusb_get_device_address(list1[i]);
        if (bus_num <= 0 || port_num <= 0)
            continue;

        std::wstring ws_path = boost::lexical_cast<std::wstring>(bus_num) + std::wstring(L",") +
                               boost::lexical_cast<std::wstring>(port_num);

        libusb_device_descriptor device_descriptor = {};
        if (f->libusb_get_device_descriptor(list1[i], &device_descriptor) != 0)
        {
            continue;
        }

        libusb_config_descriptor* config_desc = NULL;
        if (f->libusb_get_active_config_descriptor(list1[i], &config_desc) != 0)
        {
            continue;
        }

        bool found_sysfs_bos_desc = false;
        std::vector<uint8_t> sysfs_bos_desc;
        std::vector<std::vector<uint8_t> > platform_bos_desc;
#ifdef ROBOTRACONTEUR_LINUX
        {
            std::vector<uint8_t> usb_port_numbers;
            usb_port_numbers.resize(16);
            uint8_t bus_num = f->libusb_get_bus_number(list1[i]);
            int port_count = f->libusb_get_port_numbers(list1[i], &usb_port_numbers[0], 16);
            if (port_count > 0)
            {
                usb_port_numbers.resize(port_count);
                std::vector<std::string> usb_port_numbers_str;
                for (size_t j = 0; j < usb_port_numbers.size(); j++)
                {
                    usb_port_numbers_str.push_back(boost::lexical_cast<std::string>((int)usb_port_numbers[j]));
                }
                std::string sysfs_dev_path = SYSFS_DEVICE_PATH;
                sysfs_dev_path +=
                    "/" + boost::lexical_cast<std::string>((int)bus_num) + "-" + boost::join(usb_port_numbers_str, ".");

                std::string bos_path = sysfs_dev_path + "/bos_descriptors";

                int f_bos = open(bos_path.c_str(), O_RDONLY);
                if (f_bos >= 0)
                {
                    sysfs_bos_desc.resize(UINT16_MAX);
                    int bos_len = (int)read(f_bos, &sysfs_bos_desc[0], UINT16_MAX);
                    if (bos_len > 0)
                    {
                        sysfs_bos_desc.resize(bos_len);
                        found_sysfs_bos_desc = true;
                    }
                    close(f_bos);

                    if (found_sysfs_bos_desc)
                    {
                        size_t p = 0;

                        while (p < sysfs_bos_desc.size() - 3)
                        {
                            struct libusb_bos_dev_capability_descriptor* cap =
                                reinterpret_cast<libusb_bos_dev_capability_descriptor*>(&sysfs_bos_desc[p]);
                            size_t len = cap->bLength;
                            if (len == 0)
                            {
                                break;
                            }
                            if (p + len > sysfs_bos_desc.size())
                            {
                                break;
                            }

                            if (cap->bDescriptorType == LIBUSB_DT_DEVICE_CAPABILITY && cap->bDevCapabilityType == 0x05)
                            {
                                if (cap->bLength < 20)
                                {
                                    break;
                                }
                                std::vector<uint8_t> desc(&sysfs_bos_desc[p + 4], &sysfs_bos_desc[p + len]);
                                platform_bos_desc.push_back(desc);
                            }
                            p += len;
                        }
                    }
                }
            }
        }
#endif

        if (!found_sysfs_bos_desc)
        {
            libusb_device_handle* device_handle = NULL;
            if (f->libusb_open(list1[i], &device_handle) != 0)
            {
                f->libusb_free_config_descriptor(config_desc);
                continue;
            }

            libusb_bos_descriptor* bos_desc = NULL;
            if (f->libusb_get_bos_descriptor(device_handle, &bos_desc) != 0)
            {
                f->libusb_free_config_descriptor(config_desc);
                f->libusb_free_bos_descriptor(bos_desc);
                f->libusb_close(device_handle);
                continue;
            }
            for (uint8_t j = 0; j < bos_desc->bNumDeviceCaps; j++)
            {
                libusb_bos_dev_capability_descriptor* cap = bos_desc->dev_capability[j];
                if (cap->bDescriptorType == LIBUSB_DT_DEVICE_CAPABILITY && cap->bDevCapabilityType == 0x05)
                {
                    if (cap->bLength < 8)
                    {
                        continue;
                    }
                    std::vector<uint8_t> desc(cap->dev_capability_data + 1,
                                              cap->dev_capability_data + cap->bLength - 3);
                    platform_bos_desc.push_back(RR_MOVE(desc));
                }
            }

            f->libusb_free_bos_descriptor(bos_desc);
            f->libusb_close(device_handle);
        }

        uint8_t interface_ = 0;
        uint8_t rr_desc_vendor_code = 0;
        bool rr_found = false;

        for (size_t j = 0; j < platform_bos_desc.size(); j++)
        {
            std::vector<uint8_t>& desc = platform_bos_desc[j];
            if (desc.size() != 20)
            {
                continue;
            }
            if (memcmp(&desc[0], RR_USB_BOS_PLATFORM_ROBOTRACONTEUR_UUID, 16) != 0)
            {
                continue;
            }

            // Version check 0x0100
            if (desc[17] != 0x01 || desc[16] != 0x00)
            {
                continue;
            }
            interface_ = desc[18];
            rr_desc_vendor_code = desc[19];
            rr_found = true;
        }

        if (!rr_found)
        {
            RR_SHARED_PTR<HardwareTransport> t = GetParent();
            for (uint8_t j = 0; j < config_desc->bNumInterfaces; j++)
            {
                if (t->IsValidUsbDevice(device_descriptor.idVendor, device_descriptor.idProduct, j))
                {
                    rr_found = true;
                    interface_ = j;
                    break;
                }
            }
        }

        if (rr_found)
        {

            if (config_desc->interface[interface_].num_altsetting < 1)
            {
                continue;
            }

            const libusb_interface_descriptor& inter = config_desc->interface[interface_].altsetting[0];

            if (device_descriptor.bDeviceClass != 0xFF && device_descriptor.bDeviceClass != 0x00)
            {
                rr_found = false;
            }

            if (inter.bInterfaceClass != 0xFF)
            {
                rr_found = false;
            }
        }

        f->libusb_free_config_descriptor(config_desc);

        if (!rr_found)
            continue;

        RR_SHARED_PTR<libusb_device> dev_sp(list1[i],
                                            boost::bind(&LibUsb_Functions_libusb_unref, f, RR_BOOST_PLACEHOLDERS(_1)));
        f->libusb_ref_device(dev_sp.get());
        UsbDeviceManager_detected_device d;
        d.handle = dev_sp;
        d.interface_ = interface_;
        d.path = RR_MOVE(ws_path);
        d.rr_desc_vendor_code = rr_desc_vendor_code;
        devices.push_back(RR_MOVE(d));
    }

    f->libusb_free_device_list(list1, 1);

    return devices;
}

RR_SHARED_PTR<UsbDevice> LibUsbDeviceManager::CreateDevice(const UsbDeviceManager_detected_device& device)
{
    return RR_MAKE_SHARED<LibUsbDevice>(RR_STATIC_POINTER_CAST<LibUsbDeviceManager>(shared_from_this()), f, device);
}

void LibUsbDeviceManager::UsbThread()
{
    RR_SHARED_PTR<libusb_context> context = this->context;

    if (!context)
        return;

    while (true)
    {
        {
            boost::mutex::scoped_lock lock(manager_transfer_lock);
            if (!running)
            {
                return;
            }
        }

        int rv = f->libusb_handle_events(context.get());
        if (rv < 0)
        {
            if (rv != LIBUSB_ERROR_TIMEOUT)
            {
                {
                    boost::mutex::scoped_lock lock(this_lock);

                    BOOST_FOREACH (RR_SHARED_PTR<UsbDevice>& d, devices | boost::adaptors::map_values)
                    {
                        d->Close();
                    }

                    devices.clear();

                    context.reset();
                    hotplug_cb_handle = 0;
                    try
                    {
                        std::string message =
                            "Internal usb event loop failure: " + boost::lexical_cast<std::string>(rv);
                        ConnectionException exp(message);
                        RobotRaconteurNode::TryHandleException(node, &exp);
                    }
                    catch (std::exception&)
                    {}
                }
                return;
            }
        }
    }
}

void LibUsb_Functions_libusb_close(const RR_SHARED_PTR<LibUsb_Functions>& f, libusb_device_handle* h)
{
    f->libusb_close(h);
}

void LibUsbDeviceManager::LibUsbCloseDevice(RR_WEAK_PTR<LibUsbDeviceManager> d,
                                            const RR_SHARED_PTR<LibUsb_Functions>& f, libusb_device_handle* h)
{
    RR_SHARED_PTR<LibUsbDeviceManager> d1 = d.lock();
    if (!d1)
    {
        // We lost the USB device manager...
        f->libusb_close(h);
        return;
    }

    {
        boost::mutex::scoped_lock lock(d1->manager_transfer_lock);
        d1->closing_device_handles.erase(h);
    }

    if (boost::this_thread::get_id() == d1->usb_thread.get_id())
    {
        // Send the close request to the thread pool to avoid deadlock
        RobotRaconteurNode::TryPostToThreadPool(d1->node, boost::bind(&LibUsb_Functions_libusb_close, f, h), true);
    }
    else
    {
        f->libusb_close(h);
    }
}

int LibUsbDeviceManager::OnUsbHotplugEvent(libusb_context* ctx, libusb_device* device, libusb_hotplug_event event,
                                           void* user_data)
{
    RR_UNUSED(ctx);
    RR_UNUSED(device);
    RR_UNUSED(event);
    RR_UNUSED(user_data);
    return 0;
}

void LibUsbDeviceManager::submit_transfer(boost::intrusive_ptr<LibUsb_Transfer>& transfer)
{
    {
        boost::mutex::scoped_lock lock(manager_transfer_lock);

        if (!running)
        {
            transfer->transfer->actual_length = 0;
            transfer->transfer->status = LIBUSB_TRANSFER_ERROR;
            RobotRaconteurNode::TryPostToThreadPool(node, boost::bind(&LibUsb_Transfer::CompleteTransfer, transfer),
                                                    true);
            return;
        }

        if (!closing_device_handles.empty())
        {
            if (closing_device_handles.find(transfer->device_handle.get()) != closing_device_handles.end())
            {
                transfer->transfer->actual_length = 0;
                transfer->transfer->status = LIBUSB_TRANSFER_ERROR;
                RobotRaconteurNode::TryPostToThreadPool(node, boost::bind(&LibUsb_Transfer::CompleteTransfer, transfer),
                                                        true);
                return;
            }
        }

        manager_transfer_list.push_back(*transfer);
    }
    int rv = f->libusb_submit_transfer(transfer->transfer);
    if (rv != 0)
    {
        transfer->transfer->actual_length = 0;
        transfer->transfer->status = LIBUSB_TRANSFER_ERROR;
        RobotRaconteurNode::TryPostToThreadPool(node, boost::bind(&LibUsb_Transfer::CompleteTransfer, transfer), true);
    }
}

void LibUsbDeviceManager::transfer_complete(libusb_transfer* transfer)
{
    boost::intrusive_ptr<LibUsb_Transfer> t1(static_cast<LibUsb_Transfer*>(transfer->user_data), false);

    {
        RR_SHARED_PTR<LibUsbDeviceManager> m = t1->device_manager.lock();
        if (m)
        {
            boost::mutex::scoped_lock lock(m->manager_transfer_lock);
            m->manager_transfer_list.erase(m->manager_transfer_list.iterator_to(*t1));

            if (!m->closing_device_handles.empty())
            {
                std::map<libusb_device_handle*, boost::function<void()> >::iterator e =
                    m->closing_device_handles.find(t1->device_handle.get());
                if (e != m->closing_device_handles.end())
                {
                    int count = 0;
                    for (LibUsb_Transfer::manager_transfer_list_t::iterator e2 = m->manager_transfer_list.begin();
                         e2 != m->manager_transfer_list.end(); e2++)
                    {
                        if (e2->device_handle == t1->device_handle)
                        {
                            count++;
                        }
                    }
                    if (count == 0)
                    {
                        boost::function<void()> draw_down_handler = e->second;
                        e->second.clear();
                        if (draw_down_handler)
                        {
                            try
                            {
                                RobotRaconteurNode::TryPostToThreadPool(m->GetNode(), draw_down_handler, true);
                            }
                            catch (std::exception&)
                            {}
                        }
                    }
                }
            }
        }
    }

    t1->CompleteTransfer();
}

void LibUsbDeviceManager::DrawDownRequests(const RR_SHARED_PTR<libusb_device_handle>& h,
                                           boost::function<void()> handler)
{
    // TODO: draw down requests
    // GetNode()->GetThreadPool()->Post(handler);

    boost::mutex::scoped_lock lock(manager_transfer_lock);

    if (closing_device_handles.find(h.get()) != closing_device_handles.end())
    {
        RobotRaconteurNode::TryPostToThreadPool(node, handler, true);
        return;
    }

    int count = 0;
    for (LibUsb_Transfer::manager_transfer_list_t::iterator e = manager_transfer_list.begin();
         e != manager_transfer_list.end(); e++)
    {
        if (e->device_handle == h)
        {
            f->libusb_cancel_transfer(e->transfer);
            count++;
        }
    }

    if (count == 0)
    {
        // This will most likely never happen...
        RobotRaconteurNode::TryPostToThreadPool(node, handler, true);
    }
    else
    {
        closing_device_handles.insert(std::make_pair(h.get(), handler));
    }
}

void LibUsbDeviceManager::Shutdown()
{
    bool r = false;
    {
        boost::mutex::scoped_lock lock(manager_transfer_lock);
        r = running;
        running = false;
    }
    UsbDeviceManager::Shutdown();

    if (r)
    {
        if (f)
        {
            f->libusb_hotplug_deregister_callback(context.get(), hotplug_cb_handle);
        }
        usb_thread.join();
        context.reset();
    }
}

// End LibUsbDeviceManager

// LibUsbDevice_Initialize

LibUsbDevice_Initialize::LibUsbDevice_Initialize(const RR_SHARED_PTR<UsbDevice>& parent,
                                                 const RR_SHARED_PTR<LibUsb_Functions>& f,
                                                 const UsbDeviceManager_detected_device& detected_device)
    : UsbDevice_Initialize(parent, detected_device)
{
    this->f = f;
    this->m = RR_STATIC_POINTER_CAST<LibUsbDeviceManager>(parent->GetParent());
}

void LibUsbDevice_Initialize::AsyncControlTransfer(
    uint8_t bmRequestType, uint8_t bRequest, uint16_t wValue, uint16_t wIndex, boost::asio::mutable_buffer& buf,
    boost::function<void(const boost::system::error_code&, size_t)> handler, const RR_SHARED_PTR<void>& dev_h)
{
    boost::mutex::scoped_lock lock(this_lock);
    AsyncControlTransferNoLock(bmRequestType, bRequest, wValue, wIndex, buf, handler, dev_h);
}

// Call with lock
void LibUsbDevice_Initialize::AsyncControlTransferNoLock(
    uint8_t bmRequestType, uint8_t bRequest, uint16_t wValue, uint16_t wIndex, boost::asio::mutable_buffer& buf,
    boost::function<void(const boost::system::error_code&, size_t)> handler, const RR_SHARED_PTR<void>& dev_h)
{
    RR_SHARED_PTR<LibUsbDeviceManager> m1 = m.lock();
    if (!m1)
    {
        throw InvalidOperationException("Device manager lost");
    }
    RR_SHARED_PTR<libusb_device_handle> h = RR_STATIC_POINTER_CAST<libusb_device_handle>(dev_h);
    boost::intrusive_ptr<LibUsb_Transfer> t(new LibUsb_Transfer_control(f, h, m1));
    boost::static_pointer_cast<LibUsb_Transfer_control>(t)->FillTransfer(bmRequestType, bRequest, wValue, wIndex, buf,
                                                                         handler);

    m1->submit_transfer(t);
}

// Call with lock
UsbDeviceStatus LibUsbDevice_Initialize::OpenDevice(RR_SHARED_PTR<void>& dev_h)
{
    RR_SHARED_PTR<LibUsbDeviceManager> m1 = m.lock();
    if (!m1)
        return Error;
    return LibUsbDevice_open_device(m1, f, RR_STATIC_POINTER_CAST<libusb_device>(detected_device.handle), dev_h);
}

// Call with lock
UsbDeviceStatus LibUsbDevice_Initialize::ReadPipeSettings(const RR_SHARED_PTR<void>& dev_h,
                                                          RR_SHARED_PTR<UsbDevice_Settings>& settings)
{

    RR_SHARED_PTR<libusb_device_handle> h = RR_STATIC_POINTER_CAST<libusb_device_handle>(dev_h);

    libusb_config_descriptor* config_desc = NULL;
    int desired_config = 0;
    if (f->libusb_get_config_descriptor(f->libusb_get_device(h.get()), 0, &config_desc) != 0)
    {
        return Error;
    }

    // NOLINTNEXTLINE(clang-analyzer-deadcode.DeadStores)
    desired_config = config_desc->bConfigurationValue;

    const libusb_interface_descriptor* interface_desc = NULL;
    for (uint8_t i = 0; i < config_desc->bNumInterfaces; i++)
    {
        if (config_desc->interface[i].num_altsetting < 1)
        {
            continue;
        }
        if (config_desc->interface[i].altsetting[0].bInterfaceNumber != settings->interface_number)
        {
            continue;
        }
        interface_desc = &config_desc->interface[i].altsetting[0];
    }

    if (!interface_desc)
    {
        f->libusb_free_config_descriptor(config_desc);
        return Error;
    }

    bool in_found = false;
    bool out_found = false;

    for (uint8_t i = 0; i < interface_desc->bNumEndpoints; i++)
    {
        const libusb_endpoint_descriptor* ep = &interface_desc->endpoint[i];
        if (USB_ENDPOINT_TYPE_BULK(ep->bmAttributes) && USB_ENDPOINT_DIRECTION_IN(ep->bEndpointAddress) && !in_found)
        {
            in_found = true;
            settings->in_pipe_id = ep->bEndpointAddress;
            settings->in_pipe_maxpacket = ep->wMaxPacketSize;
            settings->in_pipe_buffer_size = settings->in_pipe_maxpacket;
        }

        if (USB_ENDPOINT_TYPE_BULK(ep->bmAttributes) && USB_ENDPOINT_DIRECTION_OUT(ep->bEndpointAddress) && !out_found)
        {
            out_found = true;
            settings->out_pipe_id = ep->bEndpointAddress;
            settings->out_pipe_maxpacket = ep->wMaxPacketSize;
            settings->out_pipe_buffer_size = settings->out_pipe_maxpacket;
        }
    }

    settings->interface_alt_setting = interface_desc->bAlternateSetting;
    settings->device_desired_config = desired_config;

    f->libusb_free_config_descriptor(config_desc);

    if (!in_found && !out_found)
    {
        return Error;
    }

    return Open;
}

UsbDeviceStatus LibUsbDevice_Initialize::ReadInterfaceSettings(const RR_SHARED_PTR<void>& dev_h,
                                                               RR_SHARED_PTR<UsbDevice_Settings>& settings)
{
    RR_UNUSED(dev_h);
    RR_UNUSED(settings);

    // This function is not used in libusb
    return Open;
}

// End LibUsbDevice_Initialize

// LibUsbDevice_Claim
LibUsbDevice_Claim::LibUsbDevice_Claim(const RR_SHARED_PTR<UsbDevice>& parent, const RR_SHARED_PTR<LibUsb_Functions>& f,
                                       const UsbDeviceManager_detected_device& detected_device)
    : UsbDevice_Claim(parent, detected_device)
{
    this->f = f;
    m = RR_STATIC_POINTER_CAST<LibUsbDeviceManager>(parent->GetParent());
}
LibUsbDevice_Claim::~LibUsbDevice_Claim() {}

void LibUsbDevice_Claim::AsyncControlTransfer(uint8_t bmRequestType, uint8_t bRequest, uint16_t wValue, uint16_t wIndex,
                                              boost::asio::mutable_buffer& buf,
                                              boost::function<void(const boost::system::error_code&, size_t)> handler,
                                              const RR_SHARED_PTR<void>& dev_h)
{
    boost::mutex::scoped_lock lock(this_lock);
    AsyncControlTransferNoLock(bmRequestType, bRequest, wValue, wIndex, buf, handler, dev_h);
}

void LibUsbDevice_Claim::AsyncControlTransferNoLock(
    uint8_t bmRequestType, uint8_t bRequest, uint16_t wValue, uint16_t wIndex, boost::asio::mutable_buffer& buf,
    boost::function<void(const boost::system::error_code&, size_t)> handler, const RR_SHARED_PTR<void>& dev_h)
{
    RR_SHARED_PTR<LibUsbDeviceManager> m1 = m.lock();
    if (!m1)
    {
        throw InvalidOperationException("Device manager lost");
    }
    RR_SHARED_PTR<void> dev_h2 = dev_h;
    if (!dev_h2)
    {
        dev_h2 = device_handle;
    }

    if (!device_handle)
    {
        RobotRaconteurNode::TryPostToThreadPool(node, boost::bind(handler, boost::asio::error::broken_pipe, 0), true);
        return;
    }

    RR_SHARED_PTR<libusb_device_handle> h = RR_STATIC_POINTER_CAST<libusb_device_handle>(dev_h2);
    boost::intrusive_ptr<LibUsb_Transfer> t(new LibUsb_Transfer_control(f, h, m1));
    boost::static_pointer_cast<LibUsb_Transfer_control>(t)->FillTransfer(bmRequestType, bRequest, wValue, wIndex, buf,
                                                                         handler);

    m1->submit_transfer(t);
}

void LibUsbDevice_Claim::AsyncReadPipe(uint8_t ep, boost::asio::mutable_buffer& buf,
                                       boost::function<void(const boost::system::error_code&, size_t)> handler)
{
    boost::mutex::scoped_lock lock(this_lock);
    AsyncPipeOp(ep, buf, handler);
}

void LibUsbDevice_Claim::AsyncReadPipeNoLock(uint8_t ep, boost::asio::mutable_buffer& buf,
                                             boost::function<void(const boost::system::error_code&, size_t)> handler)
{
    AsyncPipeOp(ep, buf, handler);
}

void LibUsbDevice_Claim::AsyncWritePipe(uint8_t ep, boost::asio::mutable_buffer& buf,
                                        boost::function<void(const boost::system::error_code&, size_t)> handler)
{
    boost::mutex::scoped_lock lock(this_lock);
    AsyncPipeOp(ep, buf, handler);
}

void LibUsbDevice_Claim::AsyncWritePipeNoLock(uint8_t ep, boost::asio::mutable_buffer& buf,
                                              boost::function<void(const boost::system::error_code&, size_t)> handler)
{
    AsyncPipeOp(ep, buf, handler);
}

void LibUsbDevice_Claim::AsyncPipeOp(uint8_t ep, boost::asio::mutable_buffer& buf,
                                     boost::function<void(const boost::system::error_code&, size_t)> handler)
{
    RR_SHARED_PTR<LibUsbDeviceManager> m1 = m.lock();
    if (!m1)
    {
        throw InvalidOperationException("Device manager lost");
    }

    if (!device_handle)
    {
        RobotRaconteurNode::TryPostToThreadPool(node, boost::bind(handler, boost::asio::error::broken_pipe, 0), true);
        return;
    }

    RR_SHARED_PTR<libusb_device_handle> h = RR_STATIC_POINTER_CAST<libusb_device_handle>(device_handle);
    boost::intrusive_ptr<LibUsb_Transfer> t(new LibUsb_Transfer_bulk(f, h, m1));
    boost::static_pointer_cast<LibUsb_Transfer_bulk>(t)->FillTransfer(ep, buf, handler);

    m1->submit_transfer(t);
}

// Call with lock
UsbDeviceStatus LibUsbDevice_Claim::ClaimDevice(RR_SHARED_PTR<void>& dev_h)
{
    RR_SHARED_PTR<LibUsbDeviceManager> m1 = m.lock();
    if (!m1)
        return Error;
    UsbDeviceStatus res =
        LibUsbDevice_open_device(m1, f, RR_STATIC_POINTER_CAST<libusb_device>(detected_device.handle), dev_h);
    if (res != Open)
    {
        return res;
    }

    RR_SHARED_PTR<libusb_device_handle> dev_h1 = RR_STATIC_POINTER_CAST<libusb_device_handle>(dev_h);

    int current_config = 0;
    if (f->libusb_get_configuration(dev_h1.get(), &current_config) != 0)
    {
        return Error;
    }

    if (current_config != settings->device_desired_config)
    {
        f->libusb_set_configuration(dev_h1.get(), 0);
    }

    int res1 = f->libusb_claim_interface(dev_h1.get(), settings->interface_number);
    if (res1 != 0)
    {
        if (res1 == LIBUSB_ERROR_BUSY)
        {
            return Busy;
        }

        if (res1 == LIBUSB_ERROR_ACCESS)
        {
            return Unauthorized;
        }

        return Error;
    }

    // f->libusb_clear_halt(dev_h1.get(), settings->in_pipe_id);
    // f->libusb_clear_halt(dev_h1.get(), settings->out_pipe_id);

    device_handle = dev_h1;
    return Open;
}

// Call with lock
void LibUsbDevice_Claim::ReleaseClaim()
{
    if (!device_handle)
        return;

    f->libusb_release_interface(device_handle.get(), settings->interface_number);

    device_handle.reset();
}

void LibUsbDevice_Claim::DrawDownRequests(boost::function<void()> handler)
{
    if (!device_handle)
        return;

    // TODO: Draw down active requests

    // f->libusb_(device_handle.get(), settings->in_pipe_id);
    // f->WinUsb_AbortPipe(device_handle.get(), settings->out_pipe_id);

    RR_SHARED_PTR<LibUsbDeviceManager> m;
    try
    {
        m = RR_STATIC_POINTER_CAST<LibUsbDeviceManager>(GetParent()->GetParent());
    }
    catch (std::exception&)
    {}

    if (m)
    {
        m->DrawDownRequests(device_handle, handler);
    }
    else
    {
        RobotRaconteurNode::TryPostToThreadPool(node, handler, true);
    }
}

void LibUsbDevice_Claim::ClearHalt(uint8_t ep)
{
    boost::mutex::scoped_lock lock(this_lock);
    if (!device_handle)
        return;
    f->libusb_clear_halt(device_handle.get(), ep);
}

// End LibUsbDevice_Claim

// LibUsbDevice

LibUsbDevice::LibUsbDevice(const RR_SHARED_PTR<LibUsbDeviceManager>& parent, const RR_SHARED_PTR<LibUsb_Functions>& f,
                           const UsbDeviceManager_detected_device& device)
    : UsbDevice(parent, device)
{

    this->f = f;
}

LibUsbDevice::~LibUsbDevice() {}

RR_SHARED_PTR<UsbDevice_Initialize> LibUsbDevice::CreateInitialize()
{
    return RR_MAKE_SHARED<LibUsbDevice_Initialize>(shared_from_this(), f, detected_device);
}
RR_SHARED_PTR<UsbDevice_Claim> LibUsbDevice::CreateClaim()
{
    return RR_MAKE_SHARED<LibUsbDevice_Claim>(shared_from_this(), f, detected_device);
}

// End LibUsbDevice
} // namespace detail
} // namespace RobotRaconteur
