#    Polite.pm: safe read/write to pipes in a cooperating
#    environment where all readers/writers politely lock the files
#    before reading/writing to them
#    
#    Copyright (C) 1999  Stefano Ghirlanda, stefano@zool.su.se
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

package LyX::Polite;
use strict;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

require Exporter;

@ISA=('Exporter');
@EXPORT=qw(pipe_read pipe_write wait_lock);
$VERSION = '0.01';

use Fcntl ':flock';
use POSIX;
use Carp;
use strict;

# reading from a pipe, wants the full filename in input
sub pipe_read {
    croak "pipe_read: more than one argument" if scalar @_ > 1;
    my $pipe = shift;
    croak "file $pipe does not exist" unless -e $pipe;
    croak "file $pipe is not a named pipe" unless -p $pipe;
    my $line = "";
    my $piece;
    while ($line !~ /\n$/) {
	sysopen PIPE, $pipe, O_RDONLY  or croak "can't open $pipe$!";
	flock PIPE, LOCK_EX or croak "can't lock $pipe: $!";
	my $read = sysread PIPE, $piece, 1024 or croak "can't read from $pipe";
	close PIPE or croak "can't close $pipe: $!";
	redo unless (defined $read && $read>0);
	$line .= $piece;
    }
    return $line;
}


# writing to a pipe, wants full filename and scalar to be written 
sub pipe_write {
    croak "pipe_write: more than two arguments" if scalar @_ > 2;
    my ($pipe, $line) = @_;
    croak "file $pipe does not exist" unless -e $pipe;
    croak "file $pipe is not a named pipe" unless -p $pipe;
    sysopen PIPE, $pipe, O_WRONLY or croak "can't open $pipe: $!";
    flock PIPE, LOCK_EX or croak "can't lock $pipe: $!";
    my $old = select PIPE;
    $| = 1;
    select $old;
    my $headache =  ($line !~ /\n$/);
    $line .= "\n" if $headache; # :-)
    syswrite PIPE, $line, length $line or croak "can't write to $pipe: $!";
    close PIPE or croak "can't close $pipe: $!";
}


# waits until someone else locks a file, or for a maximum of
# seconds passed as second argument (default 1 second)
sub wait_lock {
    croak "wait_lock: more than two arguments" if scalar @_ >2;
    my $pipe = shift;
    croak "file $pipe does not exist" unless -e $pipe;
    croak "file $pipe is not a named pipe" unless -p $pipe;
    my $timeout = shift || 1;
    eval {
	# SIGALRM is used to time out, see perlipc manpage
	local $SIG{ALRM} = 'IGNORE';
	alarm $timeout;
	while (1) {
	    sysopen PIPE, $pipe, O_WRONLY or croak "can't open $pipe: $!";
	    my $lock = flock PIPE, LOCK_EX|LOCK_NB; #try to lock the file
	    if ($lock) { #we got the lock: no one had it
		close PIPE or croak "can't close $pipe: $!";
	    } else { #someome else has locked the file
		last;
	    }
	}
	alarm 0;
    };

    # inform caller of what has happened: undef if we timed out,
    # 1 if someome got the lock
    return if $@ =~ /alarm clock restart/;
    return 1;
}

1;
__END__

=head1 NAME

LyX::Polite - three functions to 'safely' write to pipes

=head1 SYNOPSIS

C<my $line = pipe_read($pipename)>

C<pipe_write($pipename, $string)>

C<wait_lock($pipename, $timeout)>

=head1 DESCRIPTION

I<NOTE:> If you wnat to write a LyX client in Perl, there is no need to use LyX::Polite, use LyX::Client instead! See L<LyX::Client>.

LyX::Polite provides three functions to write to and read from named pipes (used as a mechanism for interprocess communication) when there might be conflict between different programs trying to access the pipes.

The mechanism is that all functions honor file locking. Both read and write operations block until an exclusive lock on the pipe is obtained. Note that this is not useful if there are processes not honoring file locking trying to use the pipe.

All read/write operation are made I<linewise>, and '\n' is assumed as line-termination character (because it like like this in the LyX messaging protocol).

=head1 USAGE

=head2 C<my $line = pipe_read($pipename)>

Reads a line (i.e.: until a newline) from the pipe whose filename is $pipename.

=head2 C<pipe_write($pipename, $string)>

Writes $string to the pipe whose name is $pipename. A newline is automatically added to $string, if not present.

=head2 C<wait_lock($pipename, $timeout)>

Waits until another process gets an exclusive lock on $pipename, but only for $timeout seconds at most. If $timeout is not specified, it defaults to 1 second.

The need of this function arises in a fairly LyX-specific context. See the LyX online documentation, chapter 4 of the Customization document, if you want to understand what this is about.

=head1 MISSING FEATURES

It is not possible to change what a "line is", i.e. the functions want a newline to terminate lines.

=head1 BUGS

The code has been tested only on Linux, but LyX and Perl run on other platforms as well. Contact the S<AUTHOR> if you find portability problems and/or want to help in solving them.

You might want to look at the LyX::Polite code if you want to port the functions to another programming language or address portability issues. Both are very welcome.

When some processes do not honor file locking, things are likely to go wrong.

=head1 SEE ALSO

See L<lyx> and its online documentation, particularly chapter 4 of the Customization document.

See L<LyX::Client> if you want to write a LyX client in Perl.

See sysread(), syswrite(), sysopen(), flock() in L<perfunc> if you want to understand the code and/or port it.

=head1 AUTHOR

Please send to Stefano Ghirlanda, stefano@zool.su.se, any bug reports
and suggestions.

=head1 COPYRIGHT

Copyright 1999 Stefano Ghirlanda.

Redistributable under the terms of the General Public License, version 2.0
or later, at your choice.


