// SPDX-License-Identifier: GPL-2.0
/*
 * Ingenic JZ4755 SoC CGU driver
 * Heavily based on JZ4725b CGU driver
 *
 * Copyright (C) 2022 Siarhei Volkau
 * Author: Siarhei Volkau <lis8215@gmail.com>
 */

#include <linux/clk-provider.h>
#include <linux/delay.h>
#include <linux/of.h>

#include <dt-bindings/clock/ingenic,jz4755-cgu.h>

#include "cgu.h"
#include "pm.h"

/* CGU register offsets */
#define CGU_REG_CPCCR		0x00
#define CGU_REG_CPPCR		0x10
#define CGU_REG_CLKGR		0x20
#define CGU_REG_OPCR		0x24
#define CGU_REG_I2SCDR		0x60
#define CGU_REG_LPCDR		0x64
#define CGU_REG_MSCCDR		0x68
#define CGU_REG_SSICDR		0x74
#define CGU_REG_CIMCDR		0x7C

static struct ingenic_cgu *cgu;

static const s8 pll_od_encoding[4] = {
	0x0, 0x1, -1, 0x3,
};

static const u8 jz4755_cgu_cpccr_div_table[] = {
	1, 2, 3, 4, 6, 8,
};

static const u8 jz4755_cgu_pll_half_div_table[] = {
	2, 1,
};

static const struct ingenic_cgu_clk_info jz4755_cgu_clocks[] = {

	/* External clocks */

	[JZ4755_CLK_EXT] = { "ext", CGU_CLK_EXT },
	[JZ4755_CLK_OSC32K] = { "osc32k", CGU_CLK_EXT },

	[JZ4755_CLK_PLL] = {
		"pll", CGU_CLK_PLL,
		.parents = { JZ4755_CLK_EXT, },
		.pll = {
			.reg = CGU_REG_CPPCR,
			.rate_multiplier = 1,
			.m_shift = 23,
			.m_bits = 9,
			.m_offset = 2,
			.n_shift = 18,
			.n_bits = 5,
			.n_offset = 2,
			.od_shift = 16,
			.od_bits = 2,
			.od_max = 4,
			.od_encoding = pll_od_encoding,
			.stable_bit = 10,
			.bypass_reg = CGU_REG_CPPCR,
			.bypass_bit = 9,
			.enable_bit = 8,
		},
	},

	/* Muxes & dividers */

	[JZ4755_CLK_PLL_HALF] = {
		"pll half", CGU_CLK_DIV,
		.parents = { JZ4755_CLK_PLL, },
		.div = {
			CGU_REG_CPCCR, 21, 1, 1, -1, -1, -1, 0,
			jz4755_cgu_pll_half_div_table,
		},
	},

	[JZ4755_CLK_EXT_HALF] = {
		"ext half", CGU_CLK_DIV,
		.parents = { JZ4755_CLK_EXT, },
		.div = {
			CGU_REG_CPCCR, 30, 1, 1, -1, -1, -1, 0,
			NULL,
		},
	},

	[JZ4755_CLK_CCLK] = {
		"cclk", CGU_CLK_DIV,
		.parents = { JZ4755_CLK_PLL, },
		.div = {
			CGU_REG_CPCCR, 0, 1, 4, 22, -1, -1, 0,
			jz4755_cgu_cpccr_div_table,
		},
	},

	[JZ4755_CLK_H0CLK] = {
		"hclk", CGU_CLK_DIV,
		.parents = { JZ4755_CLK_PLL, },
		.div = {
			CGU_REG_CPCCR, 4, 1, 4, 22, -1, -1, 0,
			jz4755_cgu_cpccr_div_table,
		},
	},

	[JZ4755_CLK_PCLK] = {
		"pclk", CGU_CLK_DIV,
		.parents = { JZ4755_CLK_PLL, },
		.div = {
			CGU_REG_CPCCR, 8, 1, 4, 22, -1, -1, 0,
			jz4755_cgu_cpccr_div_table,
		},
	},

	[JZ4755_CLK_MCLK] = {
		"mclk", CGU_CLK_DIV,
		.parents = { JZ4755_CLK_PLL, },
		.div = {
			CGU_REG_CPCCR, 12, 1, 4, 22, -1, -1, 0,
			jz4755_cgu_cpccr_div_table,
		},
	},

	[JZ4755_CLK_H1CLK] = {
		"h1clk", CGU_CLK_DIV,
		.parents = { JZ4755_CLK_PLL, },
		.div = {
			CGU_REG_CPCCR, 16, 1, 4, 22, -1, -1, 0,
			jz4755_cgu_cpccr_div_table,
		},
	},

	[JZ4755_CLK_UDC] = {
		"udc", CGU_CLK_MUX | CGU_CLK_DIV | CGU_CLK_GATE,
		.parents = { JZ4755_CLK_EXT_HALF, JZ4755_CLK_PLL_HALF, },
		.mux = { CGU_REG_CPCCR, 29, 1 },
		.div = { CGU_REG_CPCCR, 23, 1, 6, -1, -1, -1 },
		.gate = { CGU_REG_CLKGR, 10 },
	},

	[JZ4755_CLK_LCD] = {
		"lcd", CGU_CLK_DIV | CGU_CLK_GATE,
		.parents = { JZ4755_CLK_PLL_HALF, },
		.div = { CGU_REG_LPCDR, 0, 1, 11, -1, -1, -1 },
		.gate = { CGU_REG_CLKGR, 9 },
	},

	[JZ4755_CLK_MMC] = {
		"mmc", CGU_CLK_DIV,
		.parents = { JZ4755_CLK_PLL_HALF, },
		.div = { CGU_REG_MSCCDR, 0, 1, 5, -1, -1, -1 },
	},

	[JZ4755_CLK_I2S] = {
		"i2s", CGU_CLK_MUX | CGU_CLK_DIV,
		.parents = { JZ4755_CLK_EXT_HALF, JZ4755_CLK_PLL_HALF, },
		.mux = { CGU_REG_CPCCR, 31, 1 },
		.div = { CGU_REG_I2SCDR, 0, 1, 9, -1, -1, -1 },
	},

	[JZ4755_CLK_SPI] = {
		"spi", CGU_CLK_DIV | CGU_CLK_GATE,
		.parents = { JZ4755_CLK_PLL_HALF, },
		.div = { CGU_REG_SSICDR, 0, 1, 4, -1, -1, -1 },
		.gate = { CGU_REG_CLKGR, 4 },
	},

	[JZ4755_CLK_TVE] = {
		"tve", CGU_CLK_MUX | CGU_CLK_GATE,
		.parents = { JZ4755_CLK_LCD, JZ4755_CLK_EXT, },
		.mux = { CGU_REG_LPCDR, 31, 1 },
		.gate = { CGU_REG_CLKGR, 18 },
	},

	[JZ4755_CLK_RTC] = {
		"rtc", CGU_CLK_MUX | CGU_CLK_GATE,
		.parents = { JZ4755_CLK_EXT512, JZ4755_CLK_OSC32K, },
		.mux = { CGU_REG_OPCR, 2, 1},
		.gate = { CGU_REG_CLKGR, 2 },
	},

	[JZ4755_CLK_CIM] = {
		"cim", CGU_CLK_DIV | CGU_CLK_GATE,
		.parents = { JZ4755_CLK_PLL_HALF, },
		.div = { CGU_REG_CIMCDR, 0, 1, 8, -1, -1, -1 },
		.gate = { CGU_REG_CLKGR, 8 },
	},

	/* Gate-only clocks */

	[JZ4755_CLK_UART0] = {
		"uart0", CGU_CLK_GATE,
		.parents = { JZ4755_CLK_EXT_HALF, },
		.gate = { CGU_REG_CLKGR, 0 },
	},

	[JZ4755_CLK_UART1] = {
		"uart1", CGU_CLK_GATE,
		.parents = { JZ4755_CLK_EXT_HALF, },
		.gate = { CGU_REG_CLKGR, 14 },
	},

	[JZ4755_CLK_UART2] = {
		"uart2", CGU_CLK_GATE,
		.parents = { JZ4755_CLK_EXT_HALF, },
		.gate = { CGU_REG_CLKGR, 15 },
	},

	[JZ4755_CLK_ADC] = {
		"adc", CGU_CLK_GATE,
		.parents = { JZ4755_CLK_EXT_HALF, },
		.gate = { CGU_REG_CLKGR, 7 },
	},

	[JZ4755_CLK_AIC] = {
		"aic", CGU_CLK_GATE,
		.parents = { JZ4755_CLK_EXT_HALF, },
		.gate = { CGU_REG_CLKGR, 5 },
	},

	[JZ4755_CLK_I2C] = {
		"i2c", CGU_CLK_GATE,
		.parents = { JZ4755_CLK_EXT_HALF, },
		.gate = { CGU_REG_CLKGR, 3 },
	},

	[JZ4755_CLK_BCH] = {
		"bch", CGU_CLK_GATE,
		.parents = { JZ4755_CLK_H1CLK, },
		.gate = { CGU_REG_CLKGR, 11 },
	},

	[JZ4755_CLK_TCU] = {
		"tcu", CGU_CLK_GATE,
		.parents = { JZ4755_CLK_EXT, },
		.gate = { CGU_REG_CLKGR, 1 },
	},

	[JZ4755_CLK_DMA] = {
		"dma", CGU_CLK_GATE,
		.parents = { JZ4755_CLK_PCLK, },
		.gate = { CGU_REG_CLKGR, 12 },
	},

	[JZ4755_CLK_MMC0] = {
		"mmc0", CGU_CLK_GATE,
		.parents = { JZ4755_CLK_MMC, },
		.gate = { CGU_REG_CLKGR, 6 },
	},

	[JZ4755_CLK_MMC1] = {
		"mmc1", CGU_CLK_GATE,
		.parents = { JZ4755_CLK_MMC, },
		.gate = { CGU_REG_CLKGR, 16 },
	},

	[JZ4755_CLK_AUX_CPU] = {
		"aux_cpu", CGU_CLK_GATE,
		.parents = { JZ4755_CLK_H1CLK, },
		.gate = { CGU_REG_CLKGR, 24 },
	},

	[JZ4755_CLK_AHB1] = {
		"ahb1", CGU_CLK_GATE,
		.parents = { JZ4755_CLK_H1CLK, },
		.gate = { CGU_REG_CLKGR, 23 },
	},

	[JZ4755_CLK_IDCT] = {
		"idct", CGU_CLK_GATE,
		.parents = { JZ4755_CLK_H1CLK, },
		.gate = { CGU_REG_CLKGR, 22 },
	},

	[JZ4755_CLK_DB] = {
		"db", CGU_CLK_GATE,
		.parents = { JZ4755_CLK_H1CLK, },
		.gate = { CGU_REG_CLKGR, 21 },
	},

	[JZ4755_CLK_ME] = {
		"me", CGU_CLK_GATE,
		.parents = { JZ4755_CLK_H1CLK, },
		.gate = { CGU_REG_CLKGR, 20 },
	},

	[JZ4755_CLK_MC] = {
		"mc", CGU_CLK_GATE,
		.parents = { JZ4755_CLK_H1CLK, },
		.gate = { CGU_REG_CLKGR, 19 },
	},

	[JZ4755_CLK_TSSI] = {
		"tssi", CGU_CLK_GATE,
		.parents = { JZ4755_CLK_EXT_HALF/* not sure */, },
		.gate = { CGU_REG_CLKGR, 17 },
	},

	[JZ4755_CLK_IPU] = {
		"ipu", CGU_CLK_GATE,
		.parents = { JZ4755_CLK_PLL_HALF/* not sure */, },
		.gate = { CGU_REG_CLKGR, 13 },
	},

	[JZ4755_CLK_EXT512] = {
		"ext/512", CGU_CLK_FIXDIV,
		.parents = { JZ4755_CLK_EXT, },

		.fixdiv = { 512 },
	},

	[JZ4755_CLK_UDC_PHY] = {
		"udc_phy", CGU_CLK_GATE,
		.parents = { JZ4755_CLK_EXT_HALF, },
		.gate = { CGU_REG_OPCR, 6, true },
	},
};

static void __init jz4755_cgu_init(struct device_node *np)
{
	int retval;

	cgu = ingenic_cgu_new(jz4755_cgu_clocks,
			      ARRAY_SIZE(jz4755_cgu_clocks), np);
	if (!cgu) {
		pr_err("%s: failed to initialise CGU\n", __func__);
		return;
	}

	retval = ingenic_cgu_register_clocks(cgu);
	if (retval)
		pr_err("%s: failed to register CGU Clocks\n", __func__);

	ingenic_cgu_register_syscore_ops(cgu);
}
/*
 * CGU has some children devices, this is useful for probing children devices
 * in the case where the device node is compatible with "simple-mfd".
 */
CLK_OF_DECLARE_DRIVER(jz4755_cgu, "ingenic,jz4755-cgu", jz4755_cgu_init);
