/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2025 Univ. Grenoble Alpes, CNRS, Grenoble INP - UGA, TIMC, 38000 Grenoble, France
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

#ifndef __CPP_HOTPLUG_ACTION_EXTENSION__
#define __CPP_HOTPLUG_ACTION_EXTENSION__

#include "HotPlugActionExtension.h"

#include <QFileSystemWatcher>
#include <QMutex>

namespace camitk {

/**
 * @brief An ActionExtension that can be created on the fly from a camitk extension file
 *
 */
class CAMITK_API CppHotPlugActionExtension : public HotPlugActionExtension {
    Q_OBJECT

public:
    /// constructor
    CppHotPlugActionExtension(const QString& camitkFilePath, bool forceRebuild = false);

    /// destructor
    ~CppHotPlugActionExtension() override;

    /// get the programming language of this extension
    virtual QString getProgrammingLanguage() const override {
        return "C++";
    }

    /// instantiate all the C++ actions (might generate some warnings).
    /// @note this method must set the successfullyLoaded to false and return false when some action could not
    /// be instantiated
    /// @param progressMinimum is the current initial progress value
    /// @param progressMaximum is the maximum progress bar value to use when all actions are initialized
    virtual bool initActions(int progressMinimum = 0, int progressMaximum = 100) override;

    /// add this file to the file system watcher
    /// In fact only one library is watched (the last to register)
    /// As all the library should be updated at the same time during a rebuild
    /// only one file is to be watched: the extension only needs to be
    /// informed once (this will avoid race condition in the
    /// signal sent by fileSystemWatcher)
    void watchActionLibrary(QString libraryPath, HotPlugAction* action);

private slots:
    /// when the file watcher detect something changed
    void extensionRebuilt(const QString& path, int timerCallCount = 0);

    /// ensure all files in watchedUserActionLibraries are watched by fileSystemWatcher
    void updateWatchedFiles();

private:
    /// rebuild the extension using CMake to configure+build
    virtual void rebuild();

    /// block the extensionRebuilt() body instructions so that only one rebuild is done at the same time
    QMutex reloadMutex;

    /// @brief watch for the library changes and reload when needed
    /// new HotPlugAction register themselves by calling watchActionLibrary()
    QFileSystemWatcher fileSystemWatcher;

    /// key = shared library file path to watch corresponding to the HotPlugAction
    QMap<QString, HotPlugAction*> watchedUserActionLibraries;
};

} // namespace camitk

#endif // __CPP_HOTPLUG_ACTION_EXTENSION__
