/*
   Copyright The containerd Authors.

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
*/

package tasks

import (
	"errors"
	"fmt"
	"os"
	"text/tabwriter"

	"github.com/containerd/containerd/v2/cmd/ctr/commands"
	"github.com/containerd/typeurl/v2"
	"github.com/urfave/cli/v2"
)

var psCommand = &cli.Command{
	Name:      "ps",
	Usage:     "List processes for container",
	ArgsUsage: "CONTAINER",
	Action: func(cliContext *cli.Context) error {
		id := cliContext.Args().First()
		if id == "" {
			return errors.New("container id must be provided")
		}
		client, ctx, cancel, err := commands.NewClient(cliContext)
		if err != nil {
			return err
		}
		defer cancel()
		container, err := client.LoadContainer(ctx, id)
		if err != nil {
			return err
		}
		task, err := container.Task(ctx, nil)
		if err != nil {
			return err
		}
		processes, err := task.Pids(ctx)
		if err != nil {
			return err
		}
		w := tabwriter.NewWriter(os.Stdout, 1, 8, 4, ' ', 0)
		fmt.Fprintln(w, "PID\tINFO")
		for _, ps := range processes {
			var info interface{} = "-"
			if ps.Info != nil {
				info, err = typeurl.UnmarshalAny(ps.Info)
				if err != nil {
					return err
				}
			}
			if _, err := fmt.Fprintf(w, "%d\t%+v\n", ps.Pid, info); err != nil {
				return err
			}
		}
		return w.Flush()
	},
}
