/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * filename: xtheme.c                                                      *
 *                                                                         *
 * UTIL C-source: Medical Image Conversion Utility                         *
 *                                                                         *
 * purpose      : CSS theme related routines                               *
 *                                                                         *
 * project      : (X)MedCon by Erik Nolf                                   *
 *                                                                         *
 * Functions    : XMdcConfigureCssTheme()     - Configure CSS theming      *
 *                XMdcRemoveCssTheme()        - Remove CSS file in profile *
 *                XMdcParseCssTheme()         - Parse  CSS theme from ...  *
 *                XMDcWriteCssTheme()         - Write  CSS file in profile *
 *                                                                         *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/*
 */

/*
   Copyright (C) 1997-2026 by Erik Nolf

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 2, or (at your option) any later
   version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
   Public License for more details.

   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation, Inc.,
   59 Place - Suite 330, Boston, MA 02111-1307, USA.  */

/****************************************************************************
                              H E A D E R S
****************************************************************************/

#include "m-depend.h"

#include "xmedcon.h"

/****************************************************************************
                              D E F I N E S
****************************************************************************/

static GtkCssProvider *provider=NULL;

/****************************************************************************
                            F U N C T I O N S
****************************************************************************/

void XMdcConfigureCssTheme(void)
{
  GdkDisplay *display;
  GdkScreen *screen;

  /* create provider */
  provider = gtk_css_provider_new();
  display  = gdk_display_get_default();
  screen   = gdk_display_get_default_screen(display);

  gtk_style_context_add_provider_for_screen(
      screen,
      GTK_STYLE_PROVIDER(provider),
      GTK_STYLE_PROVIDER_PRIORITY_APPLICATION
  );

  XMdcRemoveCssTheme(XMEDCONUSR);
  
  XMdcParseCssTheme(XMEDCONUSR);

  XMdcWriteCssTheme(XMEDCONUSR);

  g_object_unref(provider);
}

void XMdcRemoveCssTheme(char *h)
{

  /*remove theme when requested */
  if (XMDC_THEME_ACTION & MDC_THEME_REMOVE) {

    if (h != NULL) {
      sprintf(xmdcstr,"%s%c.xmedcon.css",h,MDC_PATH_DELIM_CHR);
      remove(xmdcstr);

      MdcDebugPrint("CSS removed from %s",xmdcstr);
    }
  }
}

void XMdcParseCssTheme(char *h)
{
  GFile *css_fp = NULL;
  GError *error = NULL;
  const char *themestr;

  /* PASS #1: selected theme from command line option */
  if (XMDC_THEME_ACTION & MDC_THEME_LOAD) {

    switch (XMDC_THEME_CHOICE) {

      case MDC_THEME_SYSTEM:      themestr = mdc_theme_system;
                                  break;
      case MDC_THEME_HUNDLIVDUSK: themestr = mdc_theme_hundlivdusk;
                                  break;
      case MDC_THEME_SPEEDOMETER: themestr = mdc_theme_speedometer;
                                  break;
      case MDC_THEME_BLUEDARK:    themestr = mdc_theme_bluedark;
                                  break;
      case MDC_THEME_BLUELIGHT:   themestr = mdc_theme_bluelight;
                                  break;
      case MDC_THEME_TURBOVISION: themestr = mdc_theme_turbovision;
                                  break;
      case MDC_THEME_HUNDLIVDAWN: themestr = mdc_theme_hundlivdawn;
                                  break;
      default: themestr = NULL;
    }

    if (themestr != NULL) {

      MdcDebugPrint("CSS load from inline theme #%d",XMDC_THEME_CHOICE);

      gtk_css_provider_load_from_data(provider, themestr, strlen(themestr), &error);

      return;
    }

  }

  /* PASS #2: check for file in user home or profile */
  if (h != NULL) {

    sprintf(xmdcstr,"%s%c.xmedcon.css",h,MDC_PATH_DELIM_CHR);

    if (MdcFileExists(xmdcstr)) {

      MdcDebugPrint("CSS file found for user = %s",xmdcstr);

      css_fp = g_file_new_for_path(xmdcstr);
      gtk_css_provider_load_from_file(provider, css_fp, &error);

      return;
    }
  }

  /* PASS #3: try XMEDCONCSS environment variable */
  if (XMEDCONCSS != NULL) {

    if (MdcFileExists(XMEDCONCSS)) {
      sprintf(xmdcstr,"%s",XMEDCONCSS);

      MdcDebugPrint("CSS file found in XMEDCONCSS = %s",xmdcstr);

      css_fp = g_file_new_for_path(xmdcstr);
      gtk_css_provider_load_from_file(provider, css_fp, &error);

      return;
    }
  }

  /* PASS #4: try hardcoded path */
  if (MdcFileExists(XMDCCSS)) {

    MdcDebugPrint("CSS file found in XMDCCSS = %s",XMDCCSS);

    css_fp = g_file_new_for_path(XMDCCSS);
    gtk_css_provider_load_from_file(provider, css_fp, &error);

    return;
  }

  /* PASS #5: now we go for current favourite */
  MdcDebugPrint("CSS load from inline favourite");
  themestr = mdc_theme_hundlivdusk;
  gtk_css_provider_load_from_data(provider, themestr, strlen(themestr), &error);

}

void XMdcWriteCssTheme(char *h)
{
  FILE *fp;
  char *css_str;

  /* write theme to profile when requested */
  if (XMDC_THEME_ACTION & MDC_THEME_WRITE) {

    if (h != NULL) {

       sprintf(xmdcstr,"%s%c.xmedcon.css",h,MDC_PATH_DELIM_CHR);

       MdcDebugPrint("CSS write to file %s",xmdcstr);

       fp = fopen(xmdcstr,"w");
       if (fp != NULL) {
         css_str = gtk_css_provider_to_string(provider);
         if (css_str != NULL) {
           fprintf(fp,"%s",css_str);
           free(css_str);
         }
         fclose(fp);
       }
    }
  }
}

