/* asn.c
 *
 * Copyright (C) 2006-2025 wolfSSL Inc.
 *
 * This file is part of wolfSSL.
 *
 * wolfSSL is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * wolfSSL is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1335, USA
 */

/*
 * DESCRIPTION
 * This library provides the interface to Abstract Syntax Notation One (ASN.1)
 * objects.
 * ASN.1 is a standard interface description language for defining data
 * structures that can be serialized and deserialized in a cross-platform way.
 *
 * Encoding of ASN.1 is either using Basic Encoding Rules (BER) or
 * Distinguished Encoding Rules (DER). DER has only one possible encoding for a
 * ASN.1 description and the data.
 * Encode using DER and decode BER or DER.
 *
 * Provides routines to convert BER into DER. Replaces indefinite length
 * encoded items with explicit lengths.
 */

#include <wolfssl/wolfcrypt/libwolfssl_sources.h>

/*
ASN Options:
 * NO_ASN_TIME_CHECK: Disables ASN time checks (avoiding the ASN_BEFORE_DATE_E
 * and ASN_AFTER_DATE_E errors). Safer ways to avoid date errors would be to
 * set the WOLFSSL_LOAD_FLAG_DATE_ERR_OKAY flag when calling the _ex versions of
 * cert loading functions or to define the WOLFSSL_NO_OCSP_DATE_CHECK macro to
 * skip OCSP date errors. Defining NO_ASN_TIME_CHECK will skip ALL date checks
 * and could pose a security risk.
 * NO_ASN_TIME: Disables time parts of the ASN code for systems without an RTC
    or wishing to save space.
 * IGNORE_NAME_CONSTRAINTS: Skip ASN name checks.
 * ASN_DUMP_OID: Allows dump of OID information for debugging.
 * RSA_DECODE_EXTRA: Decodes extra information in RSA public key.
 * WOLFSSL_CERT_GEN: Cert generation. Saves extra certificate info in GetName.
 * WOLFSSL_NO_ASN_STRICT: Disable strict RFC compliance checks to
    restore 3.13.0 behavior.
 * WOLFSSL_ASN_ALLOW_0_SERIAL: Even if WOLFSSL_NO_ASN_STRICT is not defined,
    allow a length=1, but zero value serial number.
 * WOLFSSL_NO_OCSP_OPTIONAL_CERTS: Skip optional OCSP certs (responder issuer
    must still be trusted)
 * WOLFSSL_NO_TRUSTED_CERTS_VERIFY: Workaround for situation where entire cert
    chain is not loaded. This only matches on subject and public key and
    does not perform a PKI validation, so it is not a secure solution.
    Only enabled for OCSP.
 * WOLFSSL_NO_OCSP_ISSUER_CHECK: Can be defined for backwards compatibility to
    disable checking of https://www.rfc-editor.org/rfc/rfc6960#section-4.2.2.2.
 * WOLFSSL_SMALL_CERT_VERIFY: Verify the certificate signature without using
    DecodedCert. Doubles up on some code but allows smaller dynamic memory
    usage.
 * WOLFSSL_NO_OCSP_DATE_CHECK: Disable date checks for OCSP responses. This
    may be required when the system's real-time clock is not very accurate.
    It is recommended to enforce the nonce check instead if possible.
 * WOLFSSL_NO_CRL_DATE_CHECK: Disable date checks for CRL's.
 * WOLFSSL_NO_CRL_NEXT_DATE: Do not fail if CRL next date is missing
 * WOLFSSL_FORCE_OCSP_NONCE_CHECK: Require nonces to be available in OCSP
    responses. The nonces are optional and may not be supported by all
    responders. If it can be ensured that the used responder sends nonces this
    option may improve security.
 * WOLFSSL_ASN_TEMPLATE: Encoding and decoding using a template.
 * WOLFSSL_DEBUG_ASN_TEMPLATE: Enables debugging output when using ASN.1
    templates.
 * WOLFSSL_ASN_TEMPLATE_TYPE_CHECK: Use ASN functions to better test compiler
    type issues for testing
 * CRLDP_VALIDATE_DATA: For ASN template only, validates the reason data
 * WOLFSSL_AKID_NAME: Enable support for full AuthorityKeyIdentifier extension.
    Only supports copying full AKID from an existing certificate.
 * WOLFSSL_CUSTOM_OID: Enable custom OID support for subject and request
    extensions
 * WOLFSSL_HAVE_ISSUER_NAMES: Store pointers to issuer name components and their
    lengths and encodings.
 * WOLFSSL_SUBJ_DIR_ATTR: Enable support for SubjectDirectoryAttributes
    extension.
 * WOLFSSL_SUBJ_INFO_ACC: Enable support for SubjectInfoAccess extension.
 * WOLFSSL_FPKI: Enable support for FPKI (Federal PKI) extensions.
 * WOLFSSL_CERT_NAME_ALL: Adds more certificate name capability at the
    cost of taking up more memory. Adds initials, givenname, dnQualifer for
    example.
 * WC_ASN_HASH_SHA256: Force use of SHA2-256 for the internal hash ID calcs.
 * WOLFSSL_ALLOW_ENCODING_CA_FALSE: Allow encoding BasicConstraints CA:FALSE
 *  which is discouraged by X.690 specification - default values shall not
 *  be encoded.
 * NO_TIME_SIGNEDNESS_CHECK: Disabled the time_t signedness check.
 * WOLFSSL_ECC_SIGALG_PARAMS_NULL_ALLOWED: Allows the ECDSA/EdDSA signature
 *  algorithms in certificates to have NULL parameter instead of empty.
 *  DO NOT enable this unless required for interoperability.
 * WOLFSSL_ASN_EXTRA: Make more ASN.1 APIs available regardless of internal
 *  usage.
*/

#ifndef NO_RSA
    #include <wolfssl/wolfcrypt/rsa.h>
    #if defined(WOLFSSL_XILINX_CRYPT) || defined(WOLFSSL_CRYPTOCELL)
        extern int wc_InitRsaHw(RsaKey* key);
    #endif
#endif

#ifndef NO_ASN

#include <wolfssl/wolfcrypt/asn.h>
#include <wolfssl/wolfcrypt/coding.h>
#include <wolfssl/wolfcrypt/md2.h>
#include <wolfssl/wolfcrypt/hmac.h>
#include <wolfssl/wolfcrypt/pwdbased.h>
#include <wolfssl/wolfcrypt/des3.h>
#include <wolfssl/wolfcrypt/aes.h>
#include <wolfssl/wolfcrypt/rc2.h>
#include <wolfssl/wolfcrypt/wc_encrypt.h>

#include <wolfssl/wolfcrypt/random.h>
#include <wolfssl/wolfcrypt/hash.h>
#ifdef NO_INLINE
    #include <wolfssl/wolfcrypt/misc.h>
#else
    #define WOLFSSL_MISC_INCLUDED
    #include <wolfcrypt/src/misc.c>
#endif

#ifndef NO_RC4
    #include <wolfssl/wolfcrypt/arc4.h>
#endif

#if defined(WOLFSSL_SHA512) || defined(WOLFSSL_SHA384)
    #include <wolfssl/wolfcrypt/sha512.h>
#endif

#ifndef NO_SHA256
    #include <wolfssl/wolfcrypt/sha256.h>
#endif

#ifdef HAVE_ECC
    #include <wolfssl/wolfcrypt/ecc.h>
#endif

#ifdef WOLFSSL_SM2
    #include <wolfssl/wolfcrypt/sm2.h>
#endif

#ifdef HAVE_ED25519
    #include <wolfssl/wolfcrypt/ed25519.h>
#endif
#ifdef HAVE_CURVE25519
    #include <wolfssl/wolfcrypt/curve25519.h>
#endif

#ifdef HAVE_ED448
    #include <wolfssl/wolfcrypt/ed448.h>
#endif
#ifdef HAVE_CURVE448
    #include <wolfssl/wolfcrypt/curve448.h>
#endif

#if defined(HAVE_FALCON)
    #include <wolfssl/wolfcrypt/falcon.h>
#endif
#if defined(HAVE_DILITHIUM)
    #include <wolfssl/wolfcrypt/dilithium.h>
#endif
#if defined(HAVE_SPHINCS)
    #include <wolfssl/wolfcrypt/sphincs.h>
#endif

#ifdef WOLFSSL_QNX_CAAM
    #include <wolfssl/wolfcrypt/port/caam/wolfcaam.h>
#endif

#if defined(WOLFSSL_RENESAS_FSPSM_TLS) || defined(WOLFSSL_RENESAS_TSIP_TLS)
    #include <wolfssl/wolfcrypt/port/Renesas/renesas_cmn.h>
#endif

#ifndef NO_DSA
    #include <wolfssl/wolfcrypt/dsa.h>
#else
    typedef void* DsaKey;
#endif

#ifdef WOLF_CRYPTO_CB
    #include <wolfssl/wolfcrypt/cryptocb.h>
#endif

#ifndef WOLFCRYPT_ONLY
    #include <wolfssl/internal.h>
#endif

#if defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)
    #include <wolfssl/openssl/objects.h>
#endif

#if (defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)) && \
        !defined(WOLFCRYPT_ONLY)
    #define WOLFSSL_X509_NAME_AVAILABLE
#endif

#ifdef _MSC_VER
    /* 4996 warning to use MS extensions e.g., strcpy_s instead of XSTRNCPY */
    #pragma warning(disable: 4996)
#endif

#define ERROR_OUT(err, eLabel) { ret = (err); goto eLabel; }

#if !defined(NO_SKID) && (!defined(HAVE_FIPS) || !defined(HAVE_FIPS_VERSION))
    #if !defined(HAVE_SELFTEST) || (defined(HAVE_SELFTEST) && \
                                   (!defined(HAVE_SELFTEST_VERSION) || \
                                    HAVE_SELFTEST_VERSION < 2))
    #ifndef WOLFSSL_AES_KEY_SIZE_ENUM
    #define WOLFSSL_AES_KEY_SIZE_ENUM
    enum Asn_Misc {
        AES_IV_SIZE         = 16,
        AES_128_KEY_SIZE    = 16,
        AES_192_KEY_SIZE    = 24,
        AES_256_KEY_SIZE    = 32
    };
    #endif
    #endif /* HAVE_SELFTEST */
#endif

#if defined(WOLFSSL_ASN_PRINT) || defined(WOLFSSL_DEBUG_ASN_TEMPLATE)

/* String representations of tags. */
static const char* tagString[4][32] = {
    /* Universal */
    {
        "EOC",
        "BOOLEAN",
        "INTEGER",
        "BIT STRING",
        "OCTET STRING",
        "NULL",
        "OBJECT ID",
        "ObjectDescriptor",
        "INSTANCE OF",
        "REAL",
        "ENUMERATED",
        "EMBEDDED PDV",
        "UT8String",
        "RELATIVE-OID",
        "(0x0e) 14",
        "(0x0f) 15",
        "SEQUENCE",
        "SET",
        "NumericString",
        "PrintableString",
        "T61String",
        "VideotexString",
        "IA5String",
        "UTCTime",
        "GeneralizedTime",
        "GraphicString",
        "ISO646String",
        "GeneralString",
        "UniversalString",
        "CHARACTER STRING",
        "BMPString",
        "(0x1f) 31",
    },
    /* Application */
    {
         "[A 0]",  "[A 1]",  "[A 2]",  "[A 3]",
         "[A 4]",  "[A 5]",  "[A 6]",  "[A 7]",
         "[A 8]",  "[A 9]", "[A 10]", "[A 11]",
        "[A 12]", "[A 13]", "[A 14]", "[A 15]",
        "[A 16]", "[A 17]", "[A 18]", "[A 19]",
        "[A 20]", "[A 21]", "[A 22]", "[A 23]",
        "[A 24]", "[A 25]", "[A 26]", "[A 27]",
        "[A 28]", "[A 20]", "[A 30]", "[A 31]"
    },
    /* Context-Specific */
    {
         "[0]",  "[1]",  "[2]",  "[3]",  "[4]",  "[5]",  "[6]",  "[7]",
         "[8]",  "[9]", "[10]", "[11]", "[12]", "[13]", "[14]", "[15]",
        "[16]", "[17]", "[18]", "[19]", "[20]", "[21]", "[22]", "[23]",
        "[24]", "[25]", "[26]", "[27]", "[28]", "[20]", "[30]", "[31]"
    },
    /* Private */
    {
         "[P 0]",  "[P 1]",  "[P 2]",  "[P 3]",
         "[P 4]",  "[P 5]",  "[P 6]",  "[P 7]",
         "[P 8]",  "[P 9]", "[P 10]", "[P 11]",
        "[P 12]", "[P 13]", "[P 14]", "[P 15]",
        "[P 16]", "[P 17]", "[P 18]", "[P 19]",
        "[P 20]", "[P 21]", "[P 22]", "[P 23]",
        "[P 24]", "[P 25]", "[P 26]", "[P 27]",
        "[P 28]", "[P 20]", "[P 30]", "[P 31]"
    }
};

/* Converts a tag byte to string.
 *
 * @param [in] tag  BER tag value to interpret.
 * @return  String corresponding to tag.
 */
static const char* TagString(byte tag)
{
    return tagString[tag >> 6][tag & ASN_TYPE_MASK];
}

#endif


/* Calculates the minimum number of bytes required to encode the value.
 *
 * @param [in] value  Value to be encoded.
 * @return  Number of bytes to encode value.
 */
static word32 BytePrecision(word32 value)
{
    word32 i;
    for (i = (word32)sizeof(value); i; --i)
        if (value >> ((i - 1) * WOLFSSL_BIT_SIZE))
            break;

    return i;
}

/* DER encodes the length value in output buffer.
 *
 *    0 ->  2^7-1: <len byte>.
 *  2^7 ->       : <0x80 + #bytes> <len big-endian bytes>
 *
 * @param [in]      length  Value to encode.
 * @param [in, out] output  Buffer to encode into.
 * @return  Number of bytes used in encoding.
 */
WOLFSSL_LOCAL word32 SetASNLength(word32 length, byte* output)
{
    word32 i = 0;

    if (length < ASN_LONG_LENGTH)
        output[i++] = (byte)length;
    else {
        word32 j;

        output[i++] = (byte)(BytePrecision(length) | ASN_LONG_LENGTH);

        for (j = BytePrecision(length); j; --j) {
            output[i] = (byte)(length >> ((j - 1) * WOLFSSL_BIT_SIZE));
            i++;
        }
    }

    return i;
}

#ifdef WC_ASN_RUNTIME_DATE_CHECK_CONTROL
static int AsnSkipDateCheck = 0;
int wc_AsnSetSkipDateCheck(int skip_p)
{
    AsnSkipDateCheck = (skip_p != 0);
    return 0;
}
int wc_AsnGetSkipDateCheck(void)
{
    return AsnSkipDateCheck;
}
#else
#define AsnSkipDateCheck 0
int wc_AsnSetSkipDateCheck(int skip_p)
{
    (void)skip_p;
    return NOT_COMPILED_IN;
}
int wc_AsnGetSkipDateCheck(void)
{
    return 0;
}
#endif

#ifdef WOLFSSL_ASN_TEMPLATE
/* Calculate the size of a DER encoded length value.
 *
 *    0 ->  2^7-1: <length byte>.
 *  2^7 ->       : <0x80 + #bytes> <big-endian length bytes>
 *
 * @param [in] length  Value to encode.
 * @return  Number of bytes required to encode.
 */
static word32 SizeASNLength(word32 length)
{
    return 1 + ((length >= ASN_LONG_LENGTH) ? BytePrecision(length) : 0);
}

/* Calculate the size of a DER encoded header.
 *
 * Header = Tag | Encoded length
 *
 * @param [in] length  Length value to encode.
 * @return  Number of bytes required to encode a DER header.
 */
#define SizeASNHeader(length) \
    (1 + SizeASNLength(length))
#endif

#ifdef WOLFSSL_ASN_TEMPLATE
#ifdef WOLFSSL_SMALL_STACK
    /* Declare the variable that is the dynamic data for decoding BER data.
     *
     * @param [in] name  Variable name to declare.
     * @param [in] cnt   Number of elements required.
     */
    #define DECL_ASNGETDATA(name, cnt)                                         \
        ASNGetData* name = NULL

    /* Allocates the dynamic BER decoding data.
     *
     * @param [in]      name  Variable name to declare.
     * @param [in]      cnt   Number of elements required.
     * @param [in, out] err   Error variable.
     * @param [in]      heap  Dynamic memory allocation hint.
     */
    #define ALLOC_ASNGETDATA(name, cnt, err, heap)                             \
    do {                                                                       \
        if ((err) == 0) {                                                      \
            (name) = (ASNGetData*)XMALLOC(sizeof(ASNGetData) * (cnt), (heap),  \
                                        DYNAMIC_TYPE_TMP_BUFFER);              \
            if ((name) == NULL) {                                              \
                (err) = MEMORY_E;                                              \
            }                                                                  \
        }                                                                      \
    }                                                                          \
    while (0)

    /* Allocates the dynamic BER decoding data and clears the memory.
     *
     * @param [in]      name  Variable name to declare.
     * @param [in]      cnt   Number of elements required.
     * @param [in, out] err   Error variable.
     * @param [in]      heap  Dynamic memory allocation hint.
     */
    #define CALLOC_ASNGETDATA(name, cnt, err, heap)                            \
    do {                                                                       \
        ALLOC_ASNGETDATA(name, cnt, err, heap);                                \
        if ((err) == 0) {                                                      \
            XMEMSET((name), 0, sizeof(ASNGetData) * (cnt));                    \
        }                                                                      \
    }                                                                          \
    while (0)

    /* Disposes of the dynamic BER decoding data.
     *
     * @param [in]      name  Variable name to declare.
     * @param [in]      heap  Dynamic memory allocation hint.
     */
    #define FREE_ASNGETDATA(name, heap)                                        \
    do {                                                                       \
        if ((name) != NULL) {                                                  \
            XFREE((name), (heap), DYNAMIC_TYPE_TMP_BUFFER);                    \
        }                                                                      \
    }                                                                          \
    while (0)

    /* Declare the variable that is the dynamic data for encoding DER data.
     *
     * @param [in] name  Variable name to declare.
     * @param [in] cnt   Number of elements required.
     */
    #define DECL_ASNSETDATA(name, cnt)                                         \
        ASNSetData* name = NULL

    /* Allocates the dynamic DER encoding data.
     *
     * @param [in]      name  Variable name to declare.
     * @param [in]      cnt   Number of elements required.
     * @param [in, out] err   Error variable.
     * @param [in]      heap  Dynamic memory allocation hint.
     */
    #define ALLOC_ASNSETDATA(name, cnt, err, heap)                             \
    do {                                                                       \
        if ((err) == 0) {                                                      \
            (name) = (ASNSetData*)XMALLOC(sizeof(ASNGetData) * (cnt), (heap),  \
                                    DYNAMIC_TYPE_TMP_BUFFER);                  \
            if ((name) == NULL) {                                              \
                (err) = MEMORY_E;                                              \
            }                                                                  \
        }                                                                      \
    }                                                                          \
    while (0)

    /* Allocates the dynamic DER encoding data and clears the memory.
     *
     * @param [in]      name  Variable name to declare.
     * @param [in]      cnt   Number of elements required.
     * @param [in, out] err   Error variable.
     * @param [in]      heap  Dynamic memory allocation hint.
     */
    #define CALLOC_ASNSETDATA(name, cnt, err, heap)                            \
    do {                                                                       \
        ALLOC_ASNSETDATA(name, cnt, err, heap);                                \
        if ((err) == 0) {                                                      \
            XMEMSET(name, 0, sizeof(ASNSetData) * (cnt));                      \
        }                                                                      \
    }                                                                          \
    while (0)

    /* Disposes of the dynamic DER encoding data.
     *
     * @param [in]      name  Variable name to declare.
     * @param [in]      heap  Dynamic memory allocation hint.
     */
    #define FREE_ASNSETDATA(name, heap)                                        \
    do {                                                                       \
        if ((name) != NULL) {                                                  \
            XFREE(name, heap, DYNAMIC_TYPE_TMP_BUFFER);                        \
        }                                                                      \
    }                                                                          \
    while (0)
#else
    /* Declare the variable that is the dynamic data for decoding BER data.
     *
     * @param [in] name  Variable name to declare.
     * @param [in] cnt   Number of elements required.
     */
    #define DECL_ASNGETDATA(name, cnt)                  \
        ASNGetData name[cnt]

    /* No implementation as declaration is static.
     *
     * @param [in]      name  Variable name to declare.
     * @param [in]      cnt   Number of elements required.
     * @param [in, out] err   Error variable.
     * @param [in]      heap  Dynamic memory allocation hint.
     */
    #define ALLOC_ASNGETDATA(name, cnt, err, heap) WC_DO_NOTHING

    /* Clears the memory of the dynamic BER encoding data.
     *
     * @param [in]      name  Variable name to declare.
     * @param [in]      cnt   Number of elements required.
     * @param [in, out] err   Error variable.
     * @param [in]      heap  Dynamic memory allocation hint.
     */
    #define CALLOC_ASNGETDATA(name, cnt, err, heap)     \
        XMEMSET(name, 0, sizeof(name))

    /* No implementation as declaration is static.
     *
     * @param [in]      name  Variable name to declare.
     * @param [in]      heap  Dynamic memory allocation hint.
     */
    #define FREE_ASNGETDATA(name, heap) WC_DO_NOTHING

    /* Declare the variable that is the dynamic data for encoding DER data.
     *
     * @param [in] name  Variable name to declare.
     * @param [in] cnt   Number of elements required.
     */
    #define DECL_ASNSETDATA(name, cnt)                  \
        ASNSetData name[cnt]

    /* No implementation as declaration is static.
     *
     * @param [in]      name  Variable name to declare.
     * @param [in]      cnt   Number of elements required.
     * @param [in, out] err   Error variable.
     * @param [in]      heap  Dynamic memory allocation hint.
     */
    #define ALLOC_ASNSETDATA(name, cnt, err, heap) WC_DO_NOTHING

    /* Clears the memory of the dynamic BER encoding data.
     *
     * @param [in]      name  Variable name to declare.
     * @param [in]      cnt   Number of elements required.
     * @param [in, out] err   Error variable.
     * @param [in]      heap  Dynamic memory allocation hint.
     */
    #define CALLOC_ASNSETDATA(name, cnt, err, heap)     \
        XMEMSET(name, 0, sizeof(name))

    /* No implementation as declaration is static.
     *
     * @param [in]      name  Variable name to declare.
     * @param [in]      heap  Dynamic memory allocation hint.
     */
    #define FREE_ASNSETDATA(name, heap) WC_DO_NOTHING
#endif


#ifdef DEBUG_WOLFSSL
    /* Enable this when debugging the parsing or creation of ASN.1 data. */
    #if 0
        #define WOLFSSL_DEBUG_ASN_TEMPLATE
    #endif
#endif

#ifdef WOLFSSL_DEBUG_ASN_TEMPLATE

#include <stdarg.h>

/* Log a message that has the printf format string.
 *
 * @param [in] <va_args>  printf style arguments.
 */
#define WOLFSSL_MSG_VSNPRINTF(...)                    \
    do {                                              \
      char line[81];                                  \
      snprintf(line, sizeof(line) - 1, __VA_ARGS__);  \
      line[sizeof(line) - 1] = '\0';                  \
      WOLFSSL_MSG(line);                              \
    }                                                 \
    while (0)
#endif

/* Returns whether ASN.1 item is an integer and the Most-Significant Bit is set.
 *
 * @param [in] asn     ASN.1 items to encode.
 * @param [in] data_a  Data to place in each item. Lengths set were not known.
 * @param [in] i       Index of item to check.
 * @return  1 when ASN.1 item is an integer and MSB is 1.
 * @return  0 otherwise.
 */
#define ASNIntMSBSet(asn, data_a, i)                  \
    (((asn)[i].tag == ASN_INTEGER) &&                 \
      ((data_a)[i].data.buffer.data != NULL &&        \
      ((data_a)[i].data.buffer.data[0] & 0x80) == 0x80))


/* Calculate the size of a DER encoded number.
 *
 * @param [in] n     Number to be encoded.
 * @param [in] bits  Maximum number of bits to encode.
 * @param [in] tag   BER tag e.g. INTEGER, BIT_STRING, etc.
 * @return  Number of bytes to the ASN.1 item.
 */
static word32 SizeASN_Num(word32 n, int bits, byte tag)
{
    int    j;
    word32 len;

    len = 1 + 1 + (word32)bits / 8;
    /* Discover actual size by checking for high zeros. */
    for (j = bits - 8; j > 0; j -= 8) {
        if (n >> j)
            break;
        len--;
    }
    if (tag == ASN_BIT_STRING)
        len++;
    else if ((tag == ASN_INTEGER) && (((n >> j) & 0x80) == 0x80))
        len++;

    return len;
}

/* Calculate the size of the data in the constructed item based on the
 * length of the ASN.1 items below.
 *
 * @param [in]      asn    ASN.1 items to encode.
 * @param [in, out] data   Data to place in each item. Lengths set were not
 *                         known.
 * @param [in]      idx    Index of item working on.
 */
static void SizeASN_CalcDataLength(const ASNItem* asn, ASNSetData *data,
                                   int idx, int maxIdx)
{
    int j;

    data[idx].data.buffer.length = 0;
    /* Sum the item length of all items underneath. */
    for (j = idx + 1; j < maxIdx; j++) {
        /* Stop looking if the next ASN.1 is same level or higher. */
        if (asn[j].depth <= asn[idx].depth)
            break;
        /* Only add in length if it is one level below. */
        if (asn[j].depth - 1 == asn[idx].depth) {
            data[idx].data.buffer.length += data[j].length;
            /* The length of a header only item doesn't include the data unless
             * a replacement buffer is supplied.
             */
            if (asn[j].headerOnly && data[j].data.buffer.data == NULL &&
                    data[j].dataType != ASN_DATA_TYPE_REPLACE_BUFFER) {
                data[idx].data.buffer.length += data[j].data.buffer.length;
            }
        }
    }
}

/* Calculate the size of the DER encoding.
 *
 * Call SetASN_Items() to write encoding to a buffer.
 *
 * @param [in]      asn    ASN.1 items to encode.
 * @param [in, out] data   Data to place in each item. Lengths set where not
 *                         known.
 * @param [in]      count  Count of items to encode.
 * @param [out]     encSz  Length of the DER encoding.
 * @return  0 on success.
 * @return  BAD_STATE_E when the data type is not supported.
 */
int SizeASN_Items(const ASNItem* asn, ASNSetData *data, int count, int* encSz)
{
    int    i;
    word32 sz = 0;
    word32 len;
    word32 dataLen;
    int    length;

#ifdef WOLFSSL_DEBUG_ASN_TEMPLATE
    WOLFSSL_ENTER("SizeASN_Items");
#endif

    if (asn == NULL || data == NULL || count <= 0 || encSz == NULL) {
        WOLFSSL_MSG("bad arguments in SizeASN_Items");
        return BAD_FUNC_ARG;
    }

    for (i = count - 1; i >= 0; i--) {
        /* Skip this ASN.1 item when encoding. */
        if (data[i].noOut) {
            /* Set the offset to the current size - used in writing DER. */
            data[i].offset = sz;
            continue;
        }

        len = 0;
        switch (data[i].dataType) {
            /* Calculate the size of the number of different sizes. */
            case ASN_DATA_TYPE_WORD8:
                len = SizeASN_Num(data[i].data.u8, 8, asn[i].tag);
                break;
            case ASN_DATA_TYPE_WORD16:
                len = SizeASN_Num(data[i].data.u16, 16, asn[i].tag);
                break;
        #ifdef WOLFSSL_ASN_TEMPLATE_NEED_SET_INT32
            /* Not used yet! */
            case ASN_DATA_TYPE_WORD32:
                len = SizeASN_Num(data[i].data.u32, 32, asn[i].tag);
                break;
        #endif

            case ASN_DATA_TYPE_MP:
                /* Calculate the size of the MP integer data. */
                length = mp_unsigned_bin_size(data[i].data.mp);
                length += mp_leading_bit(data[i].data.mp) ? 1 : 0;
                len = (word32)SizeASNHeader((word32)length) + (word32)length;
                break;

            case ASN_DATA_TYPE_REPLACE_BUFFER:
                /* Buffer is put in directly - use the length. */
                len = data[i].data.buffer.length;
                break;

            case ASN_DATA_TYPE_NONE:
                /* Calculate the size based on the data to be included.
                 * Mostly used for constructed items.
                 */
                if (asn[i].headerOnly) {
                    if (data[i].data.buffer.data != NULL) {
                        /* Force all child nodes to be ignored. Buffer
                         * overwrites children. */
                        {
                            int ii;
                            for (ii = i + 1; ii < count; ii++) {
                                if (asn[ii].depth <= asn[i].depth)
                                    break;
                                sz -= data[ii].length;
                                data[ii].noOut = 1;
                            }
                        }
                    }
                    else {
                        /* Calculate data length from items below if no buffer
                         * supplied. */
                        SizeASN_CalcDataLength(asn, data, i, count);
                    }
                }
                if (asn[i].tag == ASN_BOOLEAN) {
                    dataLen = 1;
                }
                else {
                    dataLen = data[i].data.buffer.length;
                }
                /* BIT_STRING and INTEGER have one byte prepended. */
                if ((asn[i].tag == ASN_BIT_STRING) ||
                                                   ASNIntMSBSet(asn, data, i)) {
                    dataLen++;
                    /* ASN.1 items are below and cannot include extra byte. */
                    if (asn[i].headerOnly) {
                        len++;
                    }
                }
                /* Add in the size of tag and length. */
                len += SizeASNHeader(dataLen);
                /* Include data in length if not header only or if
                 * buffer supplied. */
                if (!asn[i].headerOnly || data[i].data.buffer.data != NULL) {
                    len += dataLen;
                }
                break;

        #ifdef DEBUG_WOLFSSL
            default:
            #ifdef WOLFSSL_DEBUG_ASN_TEMPLATE
                WOLFSSL_MSG_VSNPRINTF("%2d: %d", i, data[i].dataType);
                WOLFSSL_MSG("Bad data type");
            #endif
                return BAD_STATE_E;
        #endif
        }

        /* Set the total length of the item. */
        data[i].length = len;
        /* Add length to total size. */
        sz += len;
        /* Set the offset to the current size - used in writing DER. */
        data[i].offset = sz;

    #ifdef WOLFSSL_DEBUG_ASN_TEMPLATE
        WOLFSSL_MSG_VSNPRINTF("%2d: %4d %4d %c %*s %-16s", i,
                data[i].offset, data[i].length, asn[i].constructed ? '+' : ' ',
                asn[i].depth, "", TagString(asn[i].tag));
    #endif
    }

    *encSz = (int)sz;
    return 0;
}

/* Create the DER encoding of a number.
 *
 * Assumes that the out buffer is large enough for encoding.
 *
 * @param [in] n     Number to be encoded.
 * @param [in] bits  Maximum number of bits to encode.
 * @param [in] tag   DER tag e.g. INTEGER, BIT_STRING, etc.
 */
static void SetASN_Num(word32 n, int bits, byte* out, byte tag)
{
    int    j;
    word32 idx;
    byte   len;

    /* Encoding: Tag (1 byte) | Length (1 byte) | Data (number) */

    /* Data will start at index 2 unless BIT_STRING or INTEGER */
    idx = 2;

    /* Set the length of the number based on maximum bit length. */
    len = (byte)(bits / 8);
    /* Discover actual size by checking for leading zero bytes. */
    for (j = bits - 8; j > 0; j -= 8) {
        if ((n >> j) != 0) {
            break;
        }
        len--;
    }
    /* Keep j, index of first non-zero byte, for writing out. */

    /* A BIT_STRING has the number of unused bits in last byte prepended to
     * data.
     */
    if (tag == ASN_BIT_STRING) {
        byte unusedBits = 0;
        byte lastByte = (byte)(n >> j);

        /* Quick check last bit. */
        if ((lastByte & 0x01) == 0x00) {
            unusedBits++;
            /* Check each bit for first least significant bit set. */
            while (((lastByte >> unusedBits) & 0x01) == 0x00)
                unusedBits++;
        }
        /* Add unused bits byte. */
        len++;
        out[idx++] = unusedBits;
    }

    /* An INTEGER has a prepended byte if MSB of number is 1 - makes encoded
     * value positive. */
    if ((tag == ASN_INTEGER) && (((n >> j) & 0x80) == 0x80)) {
        len++;
        out[idx++] = 0;
    }

    /* Go back and put in length. */
    out[1] = len;
    /* Place in the required bytes of the number. */
    for (; j >= 0; j -= 8)
        out[idx++] = (byte)(n >> j);
}

/* Creates the DER encoding of the ASN.1 items.
 *
 * Assumes the output buffer is large enough to hold encoding.
 * Must call SizeASN_Items() to determine size of encoding and offsets.
 *
 * @param [in]      asn     ASN.1 items to encode.
 * @param [in]      data    Data to place in each item.
 * @param [in]      count   Count of items to encode.
 * @param [in, out] output  Buffer to write encoding into.
 * @return  Size of the DER encoding in bytes.
 */
int SetASN_Items(const ASNItem* asn, ASNSetData *data, int count, byte* output)
{
    int    i;
    int    length;
    int    err;
    word32 sz;
    word32 idx;
    byte*  out;

#ifdef WOLFSSL_DEBUG_ASN_TEMPLATE
    WOLFSSL_ENTER("SetASN_Items");
#endif

    /* Offset of first item is the total length.
     * SizeASN_Items() calculated this. */
    sz = data[0].offset;

    /* Write out each item. */
    for (i = 0; i < count; i++) {
        /* Skip items not writing out. */
        if (data[i].noOut)
            continue;

        /* Start position to write item based on reverse offsets. */
        out = output + sz - data[i].offset;
        /* Index from start of item out. */
        idx = 0;

        if (data[i].dataType != ASN_DATA_TYPE_REPLACE_BUFFER) {
            /* Put in the tag - not dumping in DER from buffer. */
            out[idx++] = asn[i].tag |
                         (asn[i].constructed ? ASN_CONSTRUCTED : 0);
        }

    #ifdef WOLFSSL_DEBUG_ASN_TEMPLATE
        WOLFSSL_MSG_VSNPRINTF("%2d: %4d %4d %c %*s %-16s", i,
                sz - data[i].offset,
                data[i].length, asn[i].constructed ? '+' : ' ', asn[i].depth,
                "", TagString(asn[i].tag));
    #endif

        switch (data[i].dataType) {
            /* Write out the length and data of a number. */
            case ASN_DATA_TYPE_WORD8:
                SetASN_Num(data[i].data.u8, 8, out, asn[i].tag);
                break;
            case ASN_DATA_TYPE_WORD16:
                SetASN_Num(data[i].data.u16, 16, out, asn[i].tag);
                break;
        #ifdef WOLFSSL_ASN_TEMPLATE_NEED_SET_INT32
            /* Not used yet! */
            case ASN_DATA_TYPE_WORD32:
                SetASN_Num(data[i].data.u32, 32, out, asn[i].tag);
                break;
        #endif

            /* Write out the length and data of a multi-precision number. */
            case ASN_DATA_TYPE_MP:
                /* Get length in bytes. */
                length = mp_unsigned_bin_size(data[i].data.mp);
                /* Add one for leading zero to make encoding a positive num. */
                length += mp_leading_bit(data[i].data.mp) ? 1 : 0;
                /* Write out length. */
                idx += SetASNLength((word32)length, out + idx);
                /* Write out leading zero to make positive. */
                if (mp_leading_bit(data[i].data.mp)) {
                    out[idx++] = 0;
                }
                /* Encode number in big-endian byte array. */
                err = mp_to_unsigned_bin(data[i].data.mp, out + idx);
                if (err != MP_OKAY) {
                    WOLFSSL_MSG("SetASN_Items: Failed to write mp_int");
                    return MP_TO_E;
                }
                break;

            case ASN_DATA_TYPE_REPLACE_BUFFER:
                if (data[i].data.buffer.data == NULL) {
                    /* Return pointer for caller to use. */
                    data[i].data.buffer.data = out + idx;
                }
                else {
                    /* Dump in the DER encoded data. */
                    XMEMCPY(out + idx, data[i].data.buffer.data,
                            data[i].data.buffer.length);
                }
                break;

            case ASN_DATA_TYPE_NONE:
                if (asn[i].tag == ASN_BOOLEAN) {
                    /* Always one byte of data. */
                    out[idx++] = 1;
                    /* TRUE = 0xff, FALSE = 0x00 */
                    out[idx] = data[i].data.u8 ? 0xffU : 0x00U;
                }
                else if (asn[i].tag == ASN_TAG_NULL) {
                    /* NULL tag is always a zero length item. */
                    out[idx] = 0;
                }
                else {
                    word32 dataLen = data[i].data.buffer.length;
                    /* Add one to data length for BIT_STRING unused bits and
                     * INTEGER leading zero to make positive.
                     */
                    if ((asn[i].tag == ASN_BIT_STRING) ||
                                                   ASNIntMSBSet(asn, data, i)) {
                        dataLen++;
                    }
                    /* Write out length. */
                    idx += SetASNLength(dataLen, out + idx);
                    if ((asn[i].tag == ASN_BIT_STRING) ||
                                                   ASNIntMSBSet(asn, data, i)) {
                       /* Write out leading byte. BIT_STRING has no unused bits
                        * - use number data types if needed. */
                        out[idx++] = 0x00;
                    }
                    /* Record pointer for caller if data not supplied. */
                    if (data[i].data.buffer.data == NULL) {
                        data[i].data.buffer.data = out + idx;
                    }
                    /* Copy supplied data if not putting out header only or
                     * if buffer supplied. */
                    else if (!asn[i].headerOnly ||
                            data[i].data.buffer.data != NULL) {
                        /* Allow data to come from output buffer. */
                        XMEMMOVE(out + idx, data[i].data.buffer.data,
                                 data[i].data.buffer.length);
                    }
                }
                break;

        #ifdef DEBUG_WOLFSSL
            default:
            #ifdef WOLFSSL_DEBUG_ASN_TEMPLATE
                WOLFSSL_MSG_VSNPRINTF("Bad data type: %d", data[i].dataType);
            #endif
                return BAD_STATE_E;
        #endif
        }
    }

    return (int)sz;
}


static int GetOID(const byte* input, word32* inOutIdx, word32* oid,
                  word32 oidType, int length);

/* Maximum supported depth in ASN.1 description. */
#define GET_ASN_MAX_DEPTH          7
/* Maximum number of checked numbered choices. Only one of the items with the
 * number is allowed.
 */
#define GET_ASN_MAX_CHOICES        2

/* Use existing function to decode BER length encoding. */
#define GetASN_Length GetLength_ex

/* Check an INTEGER's first byte - must be a positive number.
 *
 * @param [in] input    BER encoded data.
 * @param [in] idx      Index of BIT_STRING data.
 * @param [in] length   Length of input data.
 * @param [in] positive Indicates number must be positive.
 * @return  0 on success.
 * @return  ASN_PARSE_E when 0 is not required but seen.
 * @return  ASN_EXPECT_0_E when 0 is required and not seen.
 */
static int GetASN_Integer(const byte* input, word32 idx, int length,
                          int positive)
{
#if (!defined(HAVE_SELFTEST) && !defined(HAVE_FIPS)) || \
    (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION > 2))
    /* Check contents consist of one or more octets. */
    if (length == 0) {
        WOLFSSL_MSG("Zero length INTEGER not allowed");
        return ASN_PARSE_E;
    }
#endif
    if (input[idx] == 0) {
        /* Check leading zero byte required. */
        if ((length > 1) && ((input[idx + 1] & 0x80) == 0)) {
            WOLFSSL_MSG("Zero not required on INTEGER");
        #ifndef WOLFSSL_ASN_INT_LEAD_0_ANY
            return ASN_PARSE_E;
        #endif
        }
    }
    /* check for invalid padding on negative integer.
     * c.f. X.690 (ISO/IEC 8825-2:2003 (E)) 10.4.6; RFC 5280 4.1
     */
    else if ((length > 1) && (input[idx] == 0xff) &&
             ((input[idx + 1] & 0x80) != 0)) {
        WOLFSSL_MSG("Bad INTEGER encoding of negative");
    #ifndef WOLFSSL_ASN_INT_LEAD_0_ANY
        return ASN_EXPECT_0_E;
    #endif /* WOLFSSL_ASN_INT_LEAD_0_ANY */
    }
    /* Check whether a leading zero byte was required. */
    else if (positive && (input[idx] & 0x80)) {
        WOLFSSL_MSG("INTEGER is negative");
    #ifndef WOLFSSL_ASN_INT_LEAD_0_ANY
        return ASN_EXPECT_0_E;
    #endif /* WOLFSSL_ASN_INT_LEAD_0_ANY */
    }

    return 0;
}

/* Check a BIT_STRING's first byte - unused bits.
 *
 * @param [in] input   BER encoded data.
 * @param [in] idx     Index of BIT_STRING data.
 * @param [in] length  Length of input data.
 * @return  0 on success.
 * @return  ASN_PARSE_E when unused bits is invalid.
 */
int GetASN_BitString(const byte* input, word32 idx, int length)
{
#if (!defined(HAVE_SELFTEST) && !defined(HAVE_FIPS)) || \
    (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION > 2))
    /* Check contents consist of one or more octets. */
    if (length == 0) {
    #ifdef WOLFSSL_DEBUG_ASN_TEMPLATE
        WOLFSSL_MSG("Zero length BIT STRING not allowed");
    #endif
        return ASN_PARSE_E;
    }
#endif
    /* Ensure unused bits value is valid range. */
    if (input[idx] > 7) {
    #ifdef WOLFSSL_DEBUG_ASN_TEMPLATE
        WOLFSSL_MSG_VSNPRINTF("BIT STRING unused bits too big: %d > 7",
                input[idx]);
    #endif
        return ASN_PARSE_E;
    }
    /* Ensure unused bits are zero. */
    if ((byte)(input[idx + (word32)length - 1] << (8 - input[idx])) != 0) {
    #ifdef WOLFSSL_DEBUG_ASN_TEMPLATE
        WOLFSSL_MSG_VSNPRINTF("BIT STRING unused bits used: %d %02x",
                input[idx], input[idx + length - 1]);
    #endif
        return ASN_PARSE_E;
    }

    return 0;
}

#ifndef WOLFSSL_NO_ASN_STRICT
/* Check a UTF8STRING's data is valid.
 *
 * @param [in] input   BER encoded data.
 * @param [in] idx     Index of UTF8STRING data.
 * @param [in] length  Length of input data.
 * @return  0 on success.
 * @return  ASN_PARSE_E when data is invalid.
 */
static int GetASN_UTF8String(const byte* input, word32 idx, int length)
{
    int ret = 0;
    word32 i = 0;

    while ((ret == 0) && ((int)i < length)) {
        int cnt;

        /* Check code points and get count of following bytes. */
        if ((input[idx + i] & 0x80) == 0x00) {
            cnt = 0;
        }
        else if ((input[idx + i] & 0xe0) == 0xc0) {
            cnt = 1;
        }
        else if ((input[idx + i] & 0xf0) == 0xe0) {
            cnt = 2;
        }
        else if ((input[idx + i] & 0xf8) == 0xf0) {
            cnt = 3;
        }
        else {
            WOLFSSL_MSG("Invalid character in UTF8STRING\n");
            ret = ASN_PARSE_E;
            break;
        }

        /* Have checked first byte. */
        i++;
        /* Check each following byte. */
        for (; cnt > 0; cnt--) {
            /* Check we have enough data. */
            if ((int)i == length) {
                WOLFSSL_MSG("Missing character in UTF8STRING\n");
                ret = ASN_PARSE_E;
                break;
            }
            /* Check following byte has top bit set. */
            if ((input[idx + i] & 0x80) != 0x80) {
                WOLFSSL_MSG("Invalid character in UTF8STRING\n");
                ret = ASN_PARSE_E;
                break;
            }
            i++;
        }
    }

    return ret;
}
#endif

/* Check an OBJECT IDENTIFIER's data is valid.
 *
 * X.690 8.19
 *
 * @param [in] input   BER encoded data.
 * @param [in] idx     Index of OBJECT IDENTIFIER data.
 * @param [in] length  Length of input data.
 * @return  0 on success.
 * @return  ASN_PARSE_E when data is invalid.
 */
static int GetASN_ObjectId(const byte* input, word32 idx, int length)
{
    int ret = 0;

    /* OID data must be at least 3 bytes. */
    if (length < 3) {
    #ifdef WOLFSSL_DEBUG_ASN_TEMPLATE
        WOLFSSL_MSG_VSNPRINTF("OID length must be 3 or more: %d", length);
    #else
        WOLFSSL_MSG("OID length less than 3");
    #endif
        ret = ASN_PARSE_E;
    }
    /* Last octet of a sub-identifier has bit 8 clear. Last octet must be last
     * of a subidentifier. Ensure last octet hasn't got top bit set.
     */
    else if ((input[(int)idx + length - 1] & 0x80) == 0x80) {
        WOLFSSL_MSG("OID last octet has top bit set");
        ret = ASN_PARSE_E;
    }

    return ret;
}

/* Get the ASN.1 items from the BER encoding.
 *
 * @param [in] asn         ASN.1 item expected.
 * @param [in] data        Data array to place found item into.
 * @param [in] input       BER encoded data.
 * @param [in] idx         Starting index of item data.
 * @param [in] len         Length of input buffer upto end of this item's data.
 * @param [in] zeroPadded  INTEGER was zero padded to make positive.
 * @return  0 on success.
 * @return  ASN_PARSE_E when BER encoded data is invalid.
 * @return  ASN_EXPECT_0_E when NULL tagged item has a non-zero length.
 * @return  MP_INIT_E when the unable to initialize an mp_int.
 * @return  ASN_GETINT_E when the unable to convert data to an mp_int.
 * @return  BAD_STATE_E when the data type is not supported.
 * @return  ASN_UNKNOWN_OID_E when the OID cannot be verified.
 */
static int GetASN_StoreData(const ASNItem* asn, ASNGetData* data,
                            const byte* input, word32 idx, int len,
                            int zeroPadded)
{
    int i;
    int err;

    /* Parse data based on data type to extract. */
    switch (data->dataType) {
        /* Parse a data into a number of specified bits. */
        case ASN_DATA_TYPE_WORD8:
            /* Check data is small enough to fit. */
            if (len != 1) {
            #ifdef WOLFSSL_DEBUG_ASN_TEMPLATE
                WOLFSSL_MSG_VSNPRINTF("Expecting one byte: %d", len);
            #endif
                return ASN_PARSE_E;
            }
            if ((asn->tag != ASN_BOOLEAN) && (!zeroPadded) &&
                (input[idx] >= 0x80U)) {
            #ifdef WOLFSSL_DEBUG_ASN_TEMPLATE
                WOLFSSL_MSG_VSNPRINTF("Unexpected negative INTEGER value");
            #endif
                return ASN_EXPECT_0_E;
            }
            /* Fill number with all of data. */
            *data->data.u8 = input[idx];
            break;
        case ASN_DATA_TYPE_WORD16:
            /* Check data is small enough to fit. */
            if (len == 0 || len > 2) {
            #ifdef WOLFSSL_DEBUG_ASN_TEMPLATE
                WOLFSSL_MSG_VSNPRINTF("Expecting 1 or 2 bytes: %d", len);
            #endif
                return ASN_PARSE_E;
            }
            if (!zeroPadded && (input[idx] >= 0x80U)) {
            #ifdef WOLFSSL_DEBUG_ASN_TEMPLATE
                WOLFSSL_MSG_VSNPRINTF("Unexpected negative INTEGER value");
            #endif
                return ASN_EXPECT_0_E;
            }
            /* Fill number with all of data. */
            *data->data.u16 = 0;
            for (i = 0; i < len; i++) {
                *data->data.u16 = (word16)(*data->data.u16 << 8U);
                *data->data.u16 = (word16)(*data->data.u16 | input[idx + (word32)i]);
            }
            break;
        case ASN_DATA_TYPE_WORD32:
            /* Check data is small enough to fit. */
            if (len == 0 || len > 4) {
            #ifdef WOLFSSL_DEBUG_ASN_TEMPLATE
                WOLFSSL_MSG_VSNPRINTF("Expecting 1 to 4 bytes: %d", len);
            #endif
                return ASN_PARSE_E;
            }
            if (!zeroPadded && (input[idx] >= 0x80U)) {
            #ifdef WOLFSSL_DEBUG_ASN_TEMPLATE
                WOLFSSL_MSG_VSNPRINTF("Unexpected negative INTEGER value");
            #endif
                return ASN_EXPECT_0_E;
            }
            /* Fill number with all of data. */
            *data->data.u32 = 0;
            for (i = 0; i < len; i++) {
                *data->data.u32 <<= 8;
                *data->data.u32 |= input[idx + (word32)i] ;
            }
            break;

        case ASN_DATA_TYPE_BUFFER:
            /* Check buffer is big enough to hold data. */
            if (len > (int)*data->data.buffer.length) {
            #ifdef WOLFSSL_DEBUG_ASN_TEMPLATE
                WOLFSSL_MSG_VSNPRINTF("Buffer too small for data: %d %d", len,
                        *data->data.buffer.length);
            #endif
                return BUFFER_E;
            }
            /* Copy in data and record actual length seen. */
            XMEMCPY(data->data.buffer.data, input + idx, (size_t)len);
            *data->data.buffer.length = (word32)len;
            break;

        case ASN_DATA_TYPE_EXP_BUFFER:
            /* Check data is same size expected. */
            if (len != (int)data->data.ref.length) {
            #ifdef WOLFSSL_DEBUG_ASN_TEMPLATE
                WOLFSSL_MSG_VSNPRINTF("Data not expected length: %d %d", len,
                        data->data.ref.length);
            #endif
                return ASN_PARSE_E;
            }
            /* Check data is same as expected. */
            if (XMEMCMP(data->data.ref.data, input + idx, (size_t)len) != 0) {
            #ifdef WOLFSSL_DEBUG_ASN_TEMPLATE
                WOLFSSL_MSG("Data not as expected");
            #endif
                return ASN_PARSE_E;
            }
            break;

        case ASN_DATA_TYPE_MP:
        case ASN_DATA_TYPE_MP_POS_NEG:
            /* Initialize mp_int and read in big-endian byte array. */
            if (mp_init(data->data.mp) != MP_OKAY) {
            #ifdef WOLFSSL_DEBUG_ASN_TEMPLATE
                WOLFSSL_MSG_VSNPRINTF("Failed to init mp: %p", data->data.mp);
            #endif
                return MP_INIT_E;
            }
            FALL_THROUGH;
        case ASN_DATA_TYPE_MP_INITED:
            err = mp_read_unsigned_bin(data->data.mp, (byte*)input + idx,
                                       (word32)len);
            if (err != 0) {
            #ifdef WOLFSSL_DEBUG_ASN_TEMPLATE
                WOLFSSL_MSG_VSNPRINTF("Failed to read mp: %d", err);
            #endif
                mp_clear(data->data.mp);
                return ASN_GETINT_E;
            }
        #ifdef HAVE_WOLF_BIGINT
            err = wc_bigint_from_unsigned_bin(&data->data.mp->raw, input + idx,
                    len);
            if (err != 0) {
            #ifdef WOLFSSL_DEBUG_ASN_TEMPLATE
                WOLFSSL_MSG_VSNPRINTF("Failed to create bigint: %d", err);
            #endif
                mp_clear(data->data.mp);
                return ASN_GETINT_E;
            }
        #endif /* HAVE_WOLF_BIGINT */

        #ifdef WOLFSSL_SP_INT_NEGATIVE
            /* Don't always read as positive. */
            if ((data->dataType == ASN_DATA_TYPE_MP_POS_NEG) && (!zeroPadded) &&
                (input[idx] & 0x80)) {
                #ifdef MP_NEG
                    data->data.mp->sign = MP_NEG;
                #else
                    #ifdef OPENSSL_EXTRA
                        /* public API wolfSSL_ASN1_INTEGER_get() depends
                         * indirectly on negative bignum handling here.
                         */
                        #error OPENSSL_EXTRA requires negative bignum support.
                    #endif
                    #ifdef WOLFSSL_DEBUG_ASN_TEMPLATE
                    WOLFSSL_MSG_VSNPRINTF("ASN negative integer without bignum support.");
                    #endif
                    mp_clear(data->data.mp);
                    return ASN_GETINT_E;
                #endif
            }
        #else
            (void)zeroPadded;
        #endif
            break;

        case ASN_DATA_TYPE_CHOICE:
            /* Check if tag matched any of the choices specified. */
            for (i = 0; data->data.choice[i] != 0; i++)
                if (data->data.choice[i] == data->tag)
                    break;
            if (data->data.choice[i] == 0) {
            #ifdef WOLFSSL_DEBUG_ASN_TEMPLATE
                WOLFSSL_MSG("Tag didn't match a choice");
            #endif
                return ASN_PARSE_E;
            }

            /* Store data pointer and length for caller. */
            data->data.ref.data = input + idx;
            data->data.ref.length = (word32)len;
            break;

        case ASN_DATA_TYPE_NONE:
            /* Default behaviour based on tag. */
            if (asn->tag == ASN_BOOLEAN) {
                /* BOOLEAN has only one byte of data in BER. */
                if (len != 1) {
                #ifdef WOLFSSL_DEBUG_ASN_TEMPLATE
                    WOLFSSL_MSG_VSNPRINTF("BOOLEAN length too long: %d", len);
                #endif
                    return ASN_PARSE_E;
                }
                if (data->data.u8 == NULL)
                    return BAD_STATE_E;
                /* Store C boolean value. */
                *data->data.u8 = (input[idx] != 0);
                break;
            }
            if (asn->tag == ASN_TAG_NULL) {
                /* NULL has no data in BER. */
                if (len != 0) {
                #ifdef WOLFSSL_DEBUG_ASN_TEMPLATE
                    WOLFSSL_MSG_VSNPRINTF("NULL length too long: %d", len);
                #endif
                    return ASN_EXPECT_0_E;
                }
                data->data.ref.data = input + idx;
                break;
            }
            if (asn->tag == ASN_OBJECT_ID) {
                word32 oidIdx = 0;
                /* Store OID data pointer and length */
                data->data.oid.data = input + idx;
                data->data.oid.length = (word32)len;
                /* Get the OID sum. */
                err = GetOID(input + idx, &oidIdx, &data->data.oid.sum,
                        data->data.oid.type, len);
                if (err < 0) {
                #ifdef WOLFSSL_DEBUG_ASN_TEMPLATE
                    WOLFSSL_MSG_VSNPRINTF("OID check failed: %d", err);
                #endif
                    return err;
                }
                break;
            }

            /* Otherwise store data pointer and length. */
            data->data.ref.data = input + idx;
            data->data.ref.length = (word32)len;
            break;

    #ifdef DEBUG_WOLFSSL
        default:
            /* Bad ASN data type. */
        #ifdef WOLFSSL_DEBUG_ASN_TEMPLATE
            WOLFSSL_MSG_VSNPRINTF("Bad data type: %d", data->dataType);
        #endif
            return BAD_STATE_E;
    #endif
    }

    return 0;
}

/* Get the ASN.1 items from the BER encoding.
 *
 * @param [in]      asn       ASN.1 items expected.
 * @param [in]      data      Data array to place found items into.
 * @param [in]      count     Count of items to parse.
 * @param [in]      complete  Whether the whole buffer is to be used up.
 * @param [in]      input     BER encoded data.
 * @param [in, out] inOutIdx  On in, starting index of data.
 *                            On out, end of parsed data.
 * @param [in]      length    Length of input buffer.
 * @return  0 on success.
 * @return  ASN_PARSE_E when BER encoded data does not match ASN.1 items or
 *          is invalid.
 * @return  BUFFER_E when data in buffer is too small.
 * @return  ASN_OBJECT_ID_E when the expected OBJECT_ID tag is not found.
 * @return  ASN_BITSTR_E when the expected BIT_STRING tag is not found.
 * @return  ASN_EXPECT_0_E when the INTEGER has the MSB set or NULL has a
 *          non-zero length.
 * @return  MP_INIT_E when the unable to initialize an mp_int.
 * @return  ASN_GETINT_E when the unable to convert data to an mp_int.
 * @return  BAD_STATE_E when the data type is not supported.
 * @return  ASN_UNKNOWN_OID_E when the OID cannot be verified.
 */
int GetASN_Items(const ASNItem* asn, ASNGetData *data, int count, int complete,
                 const byte* input, word32* inOutIdx, word32 length)
{
    int    i;
    int    j;
    int    err;
    int    len;
    /* Current index into buffer. */
    word32 idx = *inOutIdx;
    /* Declare the end index array. */
    word32 endIdx[GET_ASN_MAX_DEPTH];
    /* Set choices to -1 to indicate they haven't been seen or found. */
    signed char   choiceMet[GET_ASN_MAX_CHOICES] = { -1, -1 };
    /* Not matching a choice right now. */
    int    choice = 0;
    /* Current depth of ASN.1 item. */
    int    depth;
    /* Minimum depth value seen. */
    int    minDepth;
    /* Integer had a zero prepended. */
    int    zeroPadded;
    word32 tmpW32Val;
    signed char tmpScharVal;

#ifdef WOLFSSL_DEBUG_ASN_TEMPLATE
    WOLFSSL_ENTER("GetASN_Items");
#endif

    /* Set the end index at each depth to be the length. */
    for (i=0; i<GET_ASN_MAX_DEPTH; i++) {
        endIdx[i] = length;
    }

    /* Start depth at first items depth. */
    minDepth = depth = asn[0].depth;
    /* Check every ASN.1 item. */
    for (i = 0; i < count; i++) {
        /* Store offset of ASN.1 item. */
        data[i].offset = idx;
        /* Length of data in ASN.1 item starts empty. */
        data[i].length = 0;
        /* Get current item depth. */
        depth = asn[i].depth;
    #ifdef WOLFSSL_DEBUG_ASN_TEMPLATE
        if (depth > GET_ASN_MAX_DEPTH) {
            WOLFSSL_MSG("Depth in template too large");
            return ASN_PARSE_E;
        }
    #endif
        /* Keep track of minimum depth. */
        if (depth < minDepth) {
            minDepth = depth;
        }

        /* Reset choice if different from previous. */
        if (choice > 0 && asn[i].optional != choice) {
            choice = 0;
        }
        /* Check if first of numbered choice. */
        if (choice == 0 && asn[i].optional > 1) {
            choice = asn[i].optional;
            tmpScharVal = choiceMet[choice - 2];
            XFENCE(); /* Prevent memory access */
            if (tmpScharVal == -1) {
                /* Choice seen but not found a match yet. */
                choiceMet[choice - 2] = 0;
            }
        }

        /* Check for end of data or not a choice and tag not matching. */
        tmpW32Val = endIdx[depth];
        XFENCE(); /* Prevent memory access */
        if (idx == tmpW32Val || (data[i].dataType != ASN_DATA_TYPE_CHOICE &&
                              (input[idx] & ~ASN_CONSTRUCTED) != asn[i].tag)) {
            if (asn[i].optional) {
                /* Skip over ASN.1 items underneath this optional item. */
                for (j = i + 1; j < count; j++) {
                    if (asn[i].depth >= asn[j].depth)
                        break;
                    data[j].offset = idx;
                    data[j].length = 0;
                }
                i = j - 1;
                continue;
            }

            /* Check for end of data. */
            if (idx == length) {
        #ifdef WOLFSSL_DEBUG_ASN_TEMPLATE
                WOLFSSL_MSG_VSNPRINTF(
                    "%2d: %4d %4d %c %*s %-16s%*s  (index past end)",
                    i, data[i].offset, data[i].length,
                    asn[i].constructed ? '+' : ' ', asn[i].depth, "",
                    TagString(asn[i].tag), 6 - asn[i].depth, "");
                WOLFSSL_MSG_VSNPRINTF("Index past end of data: %d %d", idx,
                        length);
        #endif
                return BUFFER_E;
            }
        #ifdef WOLFSSL_DEBUG_ASN_TEMPLATE
            /* Show expected versus found. */
            WOLFSSL_MSG_VSNPRINTF(
                "%2d: %4d %4d %c %*s %-16s%*s  Tag=0x%02x (%s)",
                i, data[i].offset, data[i].length,
                asn[i].constructed ? '+' : ' ', asn[i].depth, "",
                TagString(asn[i].tag), 6 - asn[i].depth, "",
                input[idx], TagString(input[idx]));
        #endif
            /* Check for end of data at this depth. */
            if (idx == endIdx[depth]) {
            #ifdef WOLFSSL_DEBUG_ASN_TEMPLATE
                WOLFSSL_MSG_VSNPRINTF("Index past outer item: %d %d", idx,
                        endIdx[depth]);
            #endif
                return ASN_PARSE_E;
            }

            /* Expecting an OBJECT_ID */
            if (asn[i].tag == ASN_OBJECT_ID) {
            #ifdef WOLFSSL_DEBUG_ASN_TEMPLATE
                WOLFSSL_MSG("Expecting OBJECT ID");
            #endif
                return ASN_OBJECT_ID_E;
            }
            /* Expecting a BIT_STRING */
            if (asn[i].tag == ASN_BIT_STRING) {
            #ifdef WOLFSSL_DEBUG_ASN_TEMPLATE
                WOLFSSL_MSG("Expecting BIT STRING");
            #endif
                return ASN_BITSTR_E;
            }
            /* Not the expected tag. */
        #ifdef WOLFSSL_DEBUG_ASN_TEMPLATE
            WOLFSSL_MSG("Bad tag");
        #endif
            return ASN_PARSE_E;
        }

        /* Store found tag in data. */
        data[i].tag = input[idx];
        XFENCE(); /* Prevent memory access */
        if (data[i].dataType != ASN_DATA_TYPE_CHOICE) {
            int constructed = (input[idx] & ASN_CONSTRUCTED) == ASN_CONSTRUCTED;
            /* Check constructed match expected for non-choice ASN.1 item. */
            if (asn[i].constructed != constructed) {
            #ifdef WOLFSSL_DEBUG_ASN_TEMPLATE
                WOLFSSL_MSG_VSNPRINTF(
                        "%2d: %4d %4d %c %*s %-16s%*s  Tag=0x%02x (%s)",
                        i, data[i].offset, data[i].length,
                        asn[i].constructed ? '+' : ' ', asn[i].depth, "",
                        TagString(asn[i].tag), 6 - asn[i].depth, "",
                        input[idx], TagString(input[idx]));
                if (!constructed) {
                    WOLFSSL_MSG("Not constructed");
                }
                else {
                    WOLFSSL_MSG("Not expected to be constructed");
                }
            #endif
                return ASN_PARSE_E;
            }
        }
        /* Move index to start of length. */
        idx++;
        /* Get the encoded length. */
        if (GetASN_Length(input, &idx, &len, endIdx[depth], 1) < 0) {
        #ifdef WOLFSSL_DEBUG_ASN_TEMPLATE
            WOLFSSL_MSG_VSNPRINTF("%2d: idx=%d len=%d end=%d", i, idx, len,
                    endIdx[depth]);
        #endif
            return ASN_PARSE_E;
        }
        /* Store length of data. */
        data[i].length = (word32)len;
        /* Note the max length of items under this one. */
        endIdx[depth + 1] = idx + (word32)len;
        if (choice > 1) {
            /* Note we found a number choice. */
            choiceMet[choice - 2] = 1;
        }

    #ifdef WOLFSSL_DEBUG_ASN_TEMPLATE
        WOLFSSL_MSG_VSNPRINTF("%2d: %4d %4d %c %*s %-16s", i,
                data[i].offset, data[i].length, asn[i].constructed ? '+' : ' ',
                asn[i].depth, "", TagString(data[i].tag));
    #endif

        /* Assume no zero padding on INTEGER. */
        zeroPadded = 0;
        /* Check data types that prepended a byte. */
        if (asn[i].tag == ASN_INTEGER) {
            /* Check validity of first byte. */
            err = GetASN_Integer(input, idx, len,
                    data[i].dataType == ASN_DATA_TYPE_MP ||
                    data[i].dataType == ASN_DATA_TYPE_MP_INITED);
            if (err != 0)
                return err;
            if (len > 1 && input[idx] == 0) {
                zeroPadded = 1;
                /* Move over prepended byte. */
                idx++;
                len--;
            }
        }
        else if (asn[i].tag == ASN_BIT_STRING) {
            /* Check prepended byte is correct. */
            err = GetASN_BitString(input, idx, len);
            if (err != 0)
                return err;
            /* Move over prepended byte. */
            idx++;
            len--;
        }
    #ifndef WOLFSSL_NO_ASN_STRICT
        else if ((asn[i].tag == ASN_UTF8STRING) ||
                 (data[i].tag == ASN_UTF8STRING)) {
            /* Check validity of data. */
            err = GetASN_UTF8String(input, idx, len);
            if (err != 0)
                return err;
        }
    #endif
        else if (asn[i].tag == ASN_OBJECT_ID) {
            /* Check validity of data. */
            err = GetASN_ObjectId(input, idx, len);
            if (err != 0)
                return err;
        }

        /* Don't parse data if only header required. */
        if (asn[i].headerOnly) {
            /* Store reference to data and length. */
            data[i].data.ref.data = input + idx;
            data[i].data.ref.length = (word32)len;
            continue;
        }

        /* Store the data at idx in the ASN data item. */
        err = GetASN_StoreData(&asn[i], &data[i], input, idx, len, zeroPadded);
        if (err != 0) {
            return err;
        }

        /* Move index to next item. */
        idx += (word32)len;

        /* When matched numbered choice ... */
        if (asn[i].optional > 1) {
            /* Skip over other ASN.1 items of the same number. */
            for (j = i + 1; j < count; j++) {
                if (asn[j].depth <= asn[i].depth &&
                                           asn[j].optional != asn[i].optional) {
                   break;
                }
            }
            i = j - 1;
        }
    }

    if (complete) {
        /* When expecting ASN.1 items to completely use data, check we did. */
        for (j = depth; j > minDepth; j--) {
            if (idx < endIdx[j]) {
            #ifdef WOLFSSL_DEBUG_ASN_TEMPLATE
                WOLFSSL_MSG_VSNPRINTF(
                    "More data in constructed item at depth: %d", j - 1);
            #endif
                return ASN_PARSE_E;
            }
        }
    }

    /* Check all choices where met - found an item for them. */
    for (j = 0; j < GET_ASN_MAX_CHOICES; j++) {
        if (choiceMet[j] == 0) {
        #ifdef WOLFSSL_DEBUG_ASN_TEMPLATE
            WOLFSSL_MSG_VSNPRINTF("No choice seen: %d", j + 2);
        #endif
            return ASN_PARSE_E;
        }
    }

    /* Return index after ASN.1 data has been parsed. */
    *inOutIdx = idx;

    return 0;
}

#ifdef WOLFSSL_DEBUG_ASN_TEMPLATE
/* Calculate the size of the DER encoding.
 *
 * Call SetASN_Items() to write encoding to a buffer.
 *
 * @param [in]      asn    ASN.1 items to encode.
 * @param [in, out] data   Data to place in each item. Lengths set were not
 *                         known.
 * @param [in]      count  Count of items to encode.
 * @param [out]     len    Length of the DER encoding.
 * @return  Size of the DER encoding in bytes.
 */
static int SizeASN_ItemsDebug(const char* name, const ASNItem* asn,
    ASNSetData *data, int count, int* encSz)
{
    WOLFSSL_MSG_VSNPRINTF("TEMPLATE: %s", name);
    return SizeASN_Items(asn, data, count, encSz);
}

/* Creates the DER encoding of the ASN.1 items.
 *
 * Assumes the output buffer is large enough to hold encoding.
 * Must call SizeASN_Items() to determine size of encoding and offsets.
 *
 * Displays the template name first.
 *
 * @param [in]      name    Name of ASN.1 template.
 * @param [in]      asn     ASN.1 items to encode.
 * @param [in]      data    Data to place in each item.
 * @param [in]      count   Count of items to encode.
 * @param [in, out] output  Buffer to write encoding into.
 * @return  Size of the DER encoding in bytes.
 */
static int SetASN_ItemsDebug(const char* name, const ASNItem* asn,
    ASNSetData *data, int count, byte* output)
{
    WOLFSSL_MSG_VSNPRINTF("TEMPLATE: %s", name);
    return SetASN_Items(asn, data, count, output);
}

/* Get the ASN.1 items from the BER encoding.
 *
 * Displays the template name first.
 *
 * @param [in]      name      Name of ASN.1 template.
 * @param [in]      asn       ASN.1 items expected.
 * @param [in]      data      Data array to place found items into.
 * @param [in]      count     Count of items to parse.
 * @param [in]      complete  Whether the whole buffer is to be used up.
 * @param [in]      input     BER encoded data.
 * @param [in, out] inOutIdx  On in, starting index of data.
 *                            On out, end of parsed data.
 * @param [in]      maxIdx    Maximum index of input data.
 * @return  0 on success.
 * @return  ASN_PARSE_E when BER encoded data does not match ASN.1 items or
 * is invalid.
 * @return  BUFFER_E when data in buffer is too small.
 * @return  ASN_OBJECT_ID_E when the expected OBJECT_ID tag is not found.
 * @return  ASN_BITSTR_E when the expected BIT_STRING tag is not found.
 * @return  ASN_EXPECT_0_E when the INTEGER has the MSB set or NULL has a
 *          non-zero length.
 * @return  MP_INIT_E when the unable to initialize an mp_int.
 * @return  ASN_GETINT_E when the unable to convert data to an mp_int.
 * @return  BAD_STATE_E when the data type is not supported.
 */
static int GetASN_ItemsDebug(const char* name, const ASNItem* asn,
    ASNGetData *data, int count, int complete, const byte* input,
    word32* inOutIdx, word32 maxIdx)
{
    WOLFSSL_MSG_VSNPRINTF("TEMPLATE: %s", name);
    return GetASN_Items(asn, data, count, complete, input, inOutIdx, maxIdx);
}

/* Calculate the size of the DER encoding.
 *
 * Call SetASN_Items() to write encoding to a buffer.
 *
 * @param [in]      asn    ASN.1 items to encode.
 * @param [in, out] data   Data to place in each item. Lengths set were not
 *                         known.
 * @param [in]      count  Count of items to encode.
 * @param [out]     len    Length of the DER encoding.
 * @return  Size of the DER encoding in bytes.
 */
#define SizeASN_Items(asn, data, count, encSz)  \
    SizeASN_ItemsDebug(#asn, asn, data, count, encSz)

/* Creates the DER encoding of the ASN.1 items.
 *
 * Assumes the output buffer is large enough to hold encoding.
 * Must call SizeASN_Items() to determine size of encoding and offsets.
 *
 * Displays the template name first.
 *
 * @param [in]      name    Name of ASN.1 template.
 * @param [in]      asn     ASN.1 items to encode.
 * @param [in]      data    Data to place in each item.
 * @param [in]      count   Count of items to encode.
 * @param [in, out] output  Buffer to write encoding into.
 * @return  Size of the DER encoding in bytes.
 */
#define SetASN_Items(asn, data, count, output)  \
    SetASN_ItemsDebug(#asn, asn, data, count, output)

/* Get the ASN.1 items from the BER encoding.
 *
 * Displays the template name first.
 *
 * @param [in]      name      Name of ASN.1 template.
 * @param [in]      asn       ASN.1 items expected.
 * @param [in]      data      Data array to place found items into.
 * @param [in]      count     Count of items to parse.
 * @param [in]      complete  Whether the whole buffer is to be used up.
 * @param [in]      input     BER encoded data.
 * @param [in, out] inOutIdx  On in, starting index of data.
 *                            On out, end of parsed data.
 * @param [in]      maxIdx    Maximum index of input data.
 * @return  0 on success.
 * @return  ASN_PARSE_E when BER encoded data does not match ASN.1 items or
 * is invalid.
 * @return  BUFFER_E when data in buffer is too small.
 * @return  ASN_OBJECT_ID_E when the expected OBJECT_ID tag is not found.
 * @return  ASN_BITSTR_E when the expected BIT_STRING tag is not found.
 * @return  ASN_EXPECT_0_E when the INTEGER has the MSB set or NULL has a
 *          non-zero length.
 * @return  MP_INIT_E when the unable to initialize an mp_int.
 * @return  ASN_GETINT_E when the unable to convert data to an mp_int.
 * @return  BAD_STATE_E when the data type is not supported.
 */
#define GetASN_Items(asn, data, count, complete, input, inOutIdx, maxIdx)  \
    GetASN_ItemsDebug(#asn, asn, data, count, complete, input, inOutIdx, maxIdx)
#endif /* WOLFSSL_DEBUG_ASN_TEMPLATE */

/* Decode a BER encoded constructed sequence.
 *
 * @param [in]       input     Buffer of BER encoded data.
 * @param [in, out]  inOutIdx  On in, index to start decoding from.
 *                             On out, index of next encoded byte.
 * @param [out]      len       Length of data under SEQUENCE.
 * @param [in]       maxIdx    Maximum index of data. Index of byte after SEQ.
 * @param [in]       complete  All data used with SEQUENCE and data under.
 * @return  0 on success.
 * @return  BUFFER_E when not enough data to complete decode.
 * @return  ASN_PARSE when decoding failed.
 */
static int GetASN_Sequence(const byte* input, word32* inOutIdx, int* len,
                           word32 maxIdx, int complete)
{
    int ret = 0;
    word32 idx = *inOutIdx;

    /* Check buffer big enough for tag. */
    if (idx + 1 > maxIdx) {
        ret = BUFFER_E;
    }
    /* Check it is a constructed SEQUENCE. */
    if ((ret == 0) && (input[idx++] != (ASN_SEQUENCE | ASN_CONSTRUCTED))) {
        ret = ASN_PARSE_E;
    }
    /* Get the length. */
    if ((ret == 0) && (GetASN_Length(input, &idx, len, maxIdx, 1) < 0)) {
        ret = ASN_PARSE_E;
    }
    /* Check all data used if complete set. */
    if ((ret == 0) && complete && (idx + (word32)*len != maxIdx)) {
        ret = ASN_PARSE_E;
    }
    if (ret == 0) {
        /* Return index of next byte of encoded data. */
        *inOutIdx = idx;
    }

    return ret;
}


#ifdef WOLFSSL_ASN_TEMPLATE_TYPE_CHECK
/* Setup ASN data item to get an 8-bit number.
 *
 * @param [in] dataASN  Dynamic ASN data item.
 * @param [in] num      Pointer to an 8-bit variable.
 */
void GetASN_Int8Bit(ASNGetData *dataASN, byte* num)
{
    dataASN->dataType = ASN_DATA_TYPE_WORD8;
    dataASN->data.u8  = num;
}

/* Setup ASN data item to get a 16-bit number.
 *
 * @param [in] dataASN  Dynamic ASN data item.
 * @param [in] num      Pointer to a 16-bit variable.
 */
void GetASN_Int16Bit(ASNGetData *dataASN, word16* num)
{
    dataASN->dataType = ASN_DATA_TYPE_WORD16;
    dataASN->data.u16 = num;
}

/* Setup ASN data item to get a 32-bit number.
 *
 * @param [in] dataASN  Dynamic ASN data item.
 * @param [in] num      Pointer to a 32-bit variable.
 */
void GetASN_Int32Bit(ASNGetData *dataASN, word32* num)
{
    dataASN->dataType = ASN_DATA_TYPE_WORD32;
    dataASN->data.u32 = num;
}

/* Setup ASN data item to get data into a buffer of a specific length.
 *
 * @param [in] dataASN  Dynamic ASN data item.
 * @param [in] data     Buffer to hold data.
 * @param [in] length   Length of buffer in bytes.
 */
void GetASN_Buffer(ASNGetData *dataASN, byte* data, word32* length)
{
    dataASN->dataType           = ASN_DATA_TYPE_BUFFER;
    dataASN->data.buffer.data   = data;
    dataASN->data.buffer.length = length;
}

/* Setup ASN data item to check parsed data against expected buffer.
 *
 * @param [in] dataASN  Dynamic ASN data item.
 * @param [in] data     Buffer containing expected data.
 * @param [in] length   Length of buffer in bytes.
 */
void GetASN_ExpBuffer(ASNGetData *dataASN, const byte* data, word32 length)
{
    dataASN->dataType        = ASN_DATA_TYPE_EXP_BUFFER;
    dataASN->data.ref.data   = data;
    dataASN->data.ref.length = length;
}

/* Setup ASN data item to get a number into an mp_int.
 *
 * @param [in] dataASN  Dynamic ASN data item.
 * @param [in] num      Multi-precision number object.
 */
void GetASN_MP(ASNGetData *dataASN, mp_int* num)
{
    dataASN->dataType = ASN_DATA_TYPE_MP;
    dataASN->data.mp  = num;
}

/* Setup ASN data item to get a number into an mp_int that is initialized.
 *
 * @param [in] dataASN  Dynamic ASN data item.
 * @param [in] num      Multi-precision number object.
 */
void GetASN_MP_Inited(ASNGetData *dataASN, mp_int* num)
{
    dataASN->dataType = ASN_DATA_TYPE_MP_INITED;
    dataASN->data.mp  = num;
}

/* Setup ASN data item to get a positive or negative number into an mp_int.
 *
 * @param [in] dataASN  Dynamic ASN data item.
 * @param [in] num      Multi-precision number object.
 */
void GetASN_MP_PosNeg(ASNGetData *dataASN, mp_int* num)
{
    dataASN->dataType = ASN_DATA_TYPE_MP_POS_NEG;
    dataASN->data.mp  = num;
}

/* Setup ASN data item to be a choice of tags.
 *
 * @param [in] dataASN  Dynamic ASN data item.
 * @param [in] options  0 terminated list of tags that are valid.
 */
void GetASN_Choice(ASNGetData *dataASN, const byte* options)
{
    dataASN->dataType    = ASN_DATA_TYPE_CHOICE;
    dataASN->data.choice = options;
}

/* Setup ASN data item to get a boolean value.
 *
 * @param [in] dataASN  Dynamic ASN data item.
 * @param [in] num      Pointer to an 8-bit variable.
 */
void GetASN_Boolean(ASNGetData *dataASN, byte* num)
{
    dataASN->dataType    = ASN_DATA_TYPE_NONE;
    dataASN->data.choice = num;
}

/* Setup ASN data item to be a an OID of a specific type.
 *
 * @param [in] dataASN  Dynamic ASN data item.
 * @param [in] oidType  Type of OID to expect.
 */
void GetASN_OID(ASNGetData *dataASN, int oidType)
{
    dataASN->data.oid.type = oidType;
}

/* Get the data and length from an ASN data item.
 *
 * @param [in]  dataASN  Dynamic ASN data item.
 * @param [out] data     Pointer to data of item.
 * @param [out] length   Length of buffer in bytes.
 */
void GetASN_GetConstRef(ASNGetData * dataASN, const byte** data, word32* length)
{
    *data   = dataASN->data.ref.data;
    *length = dataASN->data.ref.length;
}

/* Get the data and length from an ASN data item.
 *
 * @param [in]  dataASN  Dynamic ASN data item.
 * @param [out] data     Pointer to data of item.
 * @param [out] length   Length of buffer in bytes.
 */
void GetASN_GetRef(ASNGetData * dataASN, byte** data, word32* length)
{
    *data   = (byte*)dataASN->data.ref.data;
    *length =        dataASN->data.ref.length;
}

/* Get the data and length from an ASN data item that is an OID.
 *
 * @param [in]  dataASN  Dynamic ASN data item.
 * @param [out] data     Pointer to .
 * @param [out] length   Length of buffer in bytes.
 */
void GetASN_OIDData(ASNGetData * dataASN, byte** data, word32* length)
{
    *data   = (byte*)dataASN->data.oid.data;
    *length =        dataASN->data.oid.length;
}

/* Setup an ASN data item to set a boolean.
 *
 * @param [in] dataASN  Dynamic ASN data item.
 * @param [in] val      Boolean value.
 */
void SetASN_Boolean(ASNSetData *dataASN, byte val)
{
    dataASN->dataType = ASN_DATA_TYPE_NONE;
    dataASN->data.u8  = val;
}

/* Setup an ASN data item to set an 8-bit number.
 *
 * @param [in] dataASN  Dynamic ASN data item.
 * @param [in] num      8-bit number to set.
 */
void SetASN_Int8Bit(ASNSetData *dataASN, byte num)
{
    dataASN->dataType = ASN_DATA_TYPE_WORD8;
    dataASN->data.u8  = num;
}

/* Setup an ASN data item to set a 16-bit number.
 *
 * @param [in] dataASN  Dynamic ASN data item.
 * @param [in] num      16-bit number to set.
 */
void SetASN_Int16Bit(ASNSetData *dataASN, word16 num)
{
    dataASN->dataType = ASN_DATA_TYPE_WORD16;
    dataASN->data.u16 = num;
}

/* Setup an ASN data item to set the data in a buffer.
 *
 * @param [in] dataASN  Dynamic ASN data item.
 * @param [in] data     Buffer containing data to set.
 * @param [in] length   Length of data in buffer in bytes.
 */
void SetASN_Buffer(ASNSetData *dataASN, const byte* data, word32 length)
{
    dataASN->data.buffer.data   = data;
    dataASN->data.buffer.length = length;
}

/* Setup an ASN data item to set the DER encode data in a buffer.
 *
 * @param [in] dataASN  Dynamic ASN data item.
 * @param [in] data     Buffer containing BER encoded data to set.
 * @param [in] length   Length of data in buffer in bytes.
 */
void SetASN_ReplaceBuffer(ASNSetData *dataASN, const byte* data, word32 length)
{
    dataASN->dataType           = ASN_DATA_TYPE_REPLACE_BUFFER;
    dataASN->data.buffer.data   = data;
    dataASN->data.buffer.length = length;
}

/* Setup an ASN data item to set an multi-precision number.
 *
 * @param [in] dataASN  Dynamic ASN data item.
 * @param [in] num      Multi-precision number.
 */
void SetASN_MP(ASNSetData *dataASN, mp_int* num)
{
    dataASN->dataType = ASN_DATA_TYPE_MP;
    dataASN->data.mp  = num;
}

/* Setup an ASN data item to set an OID based on id and type.
 *
 * oid and oidType pair are unique.
 *
 * @param [in] dataASN  Dynamic ASN data item.
 * @param [in] oid      OID identifier.
 * @param [in] oidType  Type of OID.
 */
void SetASN_OID(ASNSetData *dataASN, int oid, int oidType)
{
    dataASN->data.buffer.data = OidFromId(oid, oidType,
                                                  &dataASN->data.buffer.length);
}
#endif /* WOLFSSL_ASN_TEMPLATE_TYPE_CHECK */

#ifdef CRLDP_VALIDATE_DATA
/* Get the data of the BIT_STRING as a 16-bit number.
 *
 * @param [in]  dataASN  Dynamic ASN data item.
 * @param [out] val      ASN.1 item's data as a 16-bit number.
 * @return  0 on success.
 * @return  ASN_PARSE_E when BITSTRING value is more than 2 bytes.
 * @return  ASN_PARSE_E when unused bits of BITSTRING is invalid.
 */
static int GetASN_BitString_Int16Bit(ASNGetData* dataASN, word16* val)
{
    int ret;
    int i;
    const byte* input = dataASN->data.ref.data;
    int length = dataASN->data.ref.length;

    /* Validate the BIT_STRING data. */
    ret = GetASN_BitString(input, 0, length);
    if (ret == 0) {
        /* Skip unused bits byte. */
        input++;
        length--;

        /* Check the data is usable. */
        if (length == 0 || length > 2) {
#ifdef WOLFSSL_DEBUG_ASN_TEMPLATE
            WOLFSSL_MSG_VSNPRINTF("Expecting 1 or 2 bytes: %d", length);
#endif
            ret = ASN_PARSE_E;
        }
    }
    if (ret == 0) {
        /* Fill 16-bit var with all the data. */
        *val = 0;
        for (i = 0; i < length; i++) {
            *val <<= 8;
            *val |= input[i];
        }
    }
    return ret;
}
#endif /* CRLDP_VALIDATE_DATA */

#endif /* WOLFSSL_ASN_TEMPLATE */


/* Decode the BER/DER length field.
 *
 * @param [in]      input     BER encoded data.
 * @param [in, out] inOutIdx  On in, starting index of length.
 *                            On out, end of parsed length.
 * @param [out]     len       Length value decoded.
 * @param [in]      maxIdx    Maximum index of input data.
 * @return  Length on success.
 * @return  ASN_PARSE_E if the encoding is invalid.
 * @return  BUFFER_E when not enough data to complete decode.
 */
int GetLength(const byte* input, word32* inOutIdx, int* len, word32 maxIdx)
{
    return GetLength_ex(input, inOutIdx, len, maxIdx, 1);
}


/* Decode the BER/DER length field and check the length is valid on request.
 *
 * BER/DER has Type-Length-Value triplets.
 * When requested will check that the Length decoded, indicating the number
 * of bytes in the Value, is available in the buffer after the Length bytes.
 *
 * Only supporting a length upto INT_MAX.
 *
 * @param [in]      input     BER encoded data.
 * @param [in, out] inOutIdx  On in, starting index of length.
 *                            On out, end of parsed length.
 * @param [out]     len       Length value decoded.
 * @param [in]      maxIdx    Maximum index of input data.
 * @param [in]      check     Whether to check the buffer has at least the
 *                            decoded length of bytes remaining.
 * @return  Length on success.
 * @return  ASN_PARSE_E if the encoding is invalid.
 * @return  BUFFER_E when not enough data to complete decode.
 */
int GetLength_ex(const byte* input, word32* inOutIdx, int* len, word32 maxIdx,
                 int check)
{
    int     length = 0;
    word32  idx = (word32)*inOutIdx;
    byte    b;

    /* Ensure zero return length on error. */
    *len = 0;

    /* Check there is at least one byte available containing length information.
     */
    if ((idx + 1) > maxIdx) {
        WOLFSSL_MSG("GetLength - bad index on input");
        return BUFFER_E;
    }

    /* Get the first length byte. */
    b = input[idx++];
    /* Check if the first byte indicates the count of bytes. */
    if (b >= ASN_LONG_LENGTH) {
        /* Bottom 7 bits are the number of bytes to calculate length with.
         * Note: 0 indicates indefinite length encoding *not* 0 bytes of length.
         */
        int bytes = (int)(b & 0x7F);
        int minLen;

        /* Calculate minimum length to be encoded with bytes. */
        if (b == ASN_INDEF_LENGTH) {
            /* Indefinite length encoding - no length bytes. */
            minLen = 0;
        }
        else if (bytes == 1) {
            minLen = 0x80;
        }
        /* Only support up to the number of bytes that fit into return var. */
        else if (bytes > (int)sizeof(length)) {
            WOLFSSL_MSG("GetLength - overlong data length spec");
            return ASN_PARSE_E;
        }
        else {
            minLen = 1 << ((bytes - 1) * 8);
        }

        /* Check the number of bytes required are available. */
        if ((idx + (word32)bytes) > maxIdx) {
            WOLFSSL_MSG("GetLength - bad long length");
            return BUFFER_E;
        }

        /* Big-endian encoding of number. */
        while (bytes--) {
            b = input[idx++];
            length = (length << 8) | b;
        }
        /* Negative value indicates we overflowed the signed int. */
        if (length < 0) {
            return ASN_PARSE_E;
        }
        /* Don't allow lengths that are longer than strictly required. */
        if (length < minLen) {
            return ASN_PARSE_E;
        }
    }
    else {
        /* Length in first byte. */
        length = b;
    }

    /* When requested, check the buffer has at least length bytes left. */
    if (check && ((idx + (word32)length) > maxIdx)) {
        WOLFSSL_MSG("GetLength - value exceeds buffer length");
        return BUFFER_E;
    }

    /* Return index after length encoding. */
    *inOutIdx = idx;
    /* Return length if valid. */
    if (length > 0) {
        *len = length;
    }

    /* Return length calculated or error code. */
    return length;
}


/* Gets the tag of next BER/DER encoded item.
 *
 * Checks there is enough data in the buffer for the tag byte.
 *
 * @param [in]      input     BER encoded data.
 * @param [in, out] inOutIdx  On in, starting index of tag.
 *                            On out, end of parsed tag.
 * @param [out]     tag       Tag value found.
 * @param [in]      maxIdx    Maximum index of input data.
 *
 * return  0 on success
 * return  BAD_FUNC_ARG when tag, inOutIdx or input is NULL.
 * return  BUFFER_E when not enough space in buffer for tag.
 */
int GetASNTag(const byte* input, word32* inOutIdx, byte* tag, word32 maxIdx)
{
    int ret = 0;
    word32 idx = 0;

    /* Check validity of parameters. */
    if ((tag == NULL) || (inOutIdx == NULL) || (input == NULL)) {
        ret = BAD_FUNC_ARG;
    }
    if (ret == 0) {
        /* Get index and ensure space for tag. */
        idx = *inOutIdx;
        if (idx + ASN_TAG_SZ > maxIdx) {
            WOLFSSL_MSG("Buffer too small for ASN tag");
            ret = BUFFER_E;
        }
    }
    if (ret == 0) {
        /* Return the tag and the index after tag. */
        *tag = input[idx];
        *inOutIdx = idx + ASN_TAG_SZ;
    }
    /* Return error code. */
    return ret;
}


/* Decode the DER/BER header (Type-Length) and check the length when requested.
 *
 * BER/DER has Type-Length-Value triplets.
 * Check that the tag/type is the required value.
 * When requested will check that the Length decoded, indicating the number
 * of bytes in the Value, is available in the buffer after the Length bytes.
 *
 * Only supporting a length upto INT_MAX.
 *
 * @param [in]      input     Buffer holding DER/BER encoded data.
 * @param [in]      tag       ASN.1 tag value expected in header.
 * @param [in, out] inOutIdx  On in, starting index of header.
 *                            On out, end of parsed header.
 * @param [out]     len       Number of bytes in the ASN.1 data.
 * @param [in]      maxIdx    Length of data in buffer.
 * @param [in]      check     Whether to check the buffer has at least the
 *                            decoded length of bytes remaining.
 * @return  Number of bytes in the ASN.1 data on success.
 * @return  BUFFER_E when there is not enough data to parse.
 * @return  ASN_PARSE_E when the expected tag is not found or length is invalid.
 */
static int GetASNHeader_ex(const byte* input, byte tag, word32* inOutIdx,
                           int* len, word32 maxIdx, int check)
{
    int    ret = 0;
    word32 idx = *inOutIdx;
    byte   tagFound;
    int    length = 0;

    /* Get tag/type. */
    if (GetASNTag(input, &idx, &tagFound, maxIdx) != 0) {
        ret = ASN_PARSE_E;
    }
    /* Ensure tag is the expected value. */
    if ((ret == 0) && (tagFound != tag)) {
        ret = ASN_PARSE_E;
    }
    /* Get the encoded length. */
    if ((ret == 0) && (GetLength_ex(input, &idx, &length, maxIdx, check) < 0)) {
        ret = ASN_PARSE_E;
    }
    if (ret == 0 && tag == ASN_OBJECT_ID) {
        if (length < 3) {
            /* OID data must be at least 3 bytes. */
            WOLFSSL_MSG("OID length less than 3");
            ret = ASN_PARSE_E;
        }
        else if ((input[(int)idx + length - 1] & 0x80) == 0x80) {
            /* Last octet of a sub-identifier has bit 8 clear. Last octet must
             * be last of a subidentifier. Ensure last octet hasn't got top bit
             * set. */
            WOLFSSL_MSG("OID last octet has top bit set");
            ret = ASN_PARSE_E;
        }
    }
    if (ret == 0) {
        /* Return the length of data and index after header. */
        *len      = length;
        *inOutIdx = idx;
        ret = length;
    }
    /* Return number of data bytes or error code. */
    return ret;
}


/* Decode the DER/BER header (Type-Length) and check the length.
 *
 * BER/DER has Type-Length-Value triplets.
 * Check that the tag/type is the required value.
 * Checks that the Length decoded, indicating the number of bytes in the Value,
 * is available in the buffer after the Length bytes.
 *
 * @param [in]      input     Buffer holding DER/BER encoded data.
 * @param [in]      tag       ASN.1 tag value expected in header.
 * @param [in, out] inOutIdx  On in, starting index of header.
 *                            On out, end of parsed header.
 * @param [out]     len       Number of bytes in the ASN.1 data.
 * @param [in]      maxIdx    Length of data in buffer.
 * @return  Number of bytes in the ASN.1 data on success.
 * @return  BUFFER_E when there is not enough data to parse.
 * @return  ASN_PARSE_E when the expected tag is not found or length is invalid.
 */
int GetASNHeader(const byte* input, byte tag, word32* inOutIdx, int* len,
                        word32 maxIdx)
{
    return GetASNHeader_ex(input, tag, inOutIdx, len, maxIdx, 1);
}

#ifndef WOLFSSL_ASN_TEMPLATE
static int GetHeader(const byte* input, byte* tag, word32* inOutIdx, int* len,
                     word32 maxIdx, int check)
{
    word32 idx = *inOutIdx;
    int    length;

    if ((idx + 1) > maxIdx)
        return BUFFER_E;

    *tag = input[idx++];

    if (GetLength_ex(input, &idx, &length, maxIdx, check) < 0)
        return ASN_PARSE_E;

    *len      = length;
    *inOutIdx = idx;
    return length;
}
#endif

/* Decode the header of a BER/DER encoded SEQUENCE.
 *
 * @param [in]      input     Buffer holding DER/BER encoded data.
 * @param [in, out] inOutIdx  On in, starting index of header.
 *                            On out, end of parsed header.
 * @param [out]     len       Number of bytes in the ASN.1 data.
 * @param [in]      maxIdx    Length of data in buffer.
 * @return  Number of bytes in the ASN.1 data on success.
 * @return  BUFFER_E when there is not enough data to parse.
 * @return  ASN_PARSE_E when the tag is not a SEQUENCE or length is invalid.
 */
int GetSequence(const byte* input, word32* inOutIdx, int* len,
                           word32 maxIdx)
{
    return GetASNHeader(input, ASN_SEQUENCE | ASN_CONSTRUCTED, inOutIdx, len,
                        maxIdx);
}

/* Decode the header of a BER/DER encoded SEQUENCE.
 *
 * @param [in]      input     Buffer holding DER/BER encoded data.
 * @param [in, out] inOutIdx  On in, starting index of header.
 *                            On out, end of parsed header.
 * @param [out]     len       Number of bytes in the ASN.1 data.
 * @param [in]      maxIdx    Length of data in buffer.
 * @param [in]      check     Whether to check the buffer has at least the
 *                            decoded length of bytes remaining.
 * @return  Number of bytes in the ASN.1 data on success.
 * @return  BUFFER_E when there is not enough data to parse.
 * @return  ASN_PARSE_E when the tag is not a SEQUENCE or length is invalid.
 */
int GetSequence_ex(const byte* input, word32* inOutIdx, int* len,
                           word32 maxIdx, int check)
{
    return GetASNHeader_ex(input, ASN_SEQUENCE | ASN_CONSTRUCTED, inOutIdx, len,
                        maxIdx, check);
}

/**
 * Index a SEQUENCE OF object to get to a specific element.
 *
 * @param[in] seqOf Buffer holding DER/BER SEQUENCE OF object.
 * @param[in] seqOfSz Size of the seqOf SEQUENCE OF object.
 * @param[in] seqIndex Index of the SEQUENCE OF element being requested.
 * @param[out] out Buffer in which to store pointer to the <seqIndex>th element
 * of the SEQUENCE OF object.
 * @param[out] outSz Buffer in which to store the length of the <seqIndex>th
 * element of the SEQUENCE OF object.
 *
 * @return 0 on success.
 * @return BUFFER_E when there is not enough data to parse.
 * @return BAD_INDEX_E when the given seqIndex is out of range.
 * @return ASN_PARSE_E when the seqOf is not in the expected format.
 */
int wc_IndexSequenceOf(const byte * seqOf, word32 seqOfSz, size_t seqIndex,
        const byte ** out, word32 * outSz)
{
    int length;
    word32 seqOfIdx = 0U;
    byte tagFound;
    size_t i;
    word32 elementIdx = 0U;
    int ret = 0;

    /* Validate the SEQUENCE OF header. */
    if (GetSequence(seqOf, &seqOfIdx, &length, seqOfSz) < 0) {
        ret = ASN_PARSE_E;
    }
    else {
        seqOfSz = seqOfIdx + (word32)length;

        for (i = 0U; i <= seqIndex; i++) {
            if (seqOfIdx >= seqOfSz) {
                ret = BAD_INDEX_E;
                break;
            }

            elementIdx = seqOfIdx;

            /* Validate the element tag. */
            if (GetASNTag(seqOf, &seqOfIdx, &tagFound, seqOfSz) != 0) {
                ret = ASN_PARSE_E;
                break;
            }

            /* Validate and get the element's encoded length. */
            if (GetLength(seqOf, &seqOfIdx, &length, seqOfSz) < 0) {
                ret = ASN_PARSE_E;
                break;
            }

            seqOfIdx += (word32)length;
        }
    }

    /* If the tag and length checks above passed then we've found the requested
     * element and validated it fits within seqOfSz. */
    if (ret == 0) {
        *out = &seqOf[elementIdx];
        *outSz = (seqOfIdx - elementIdx);
    }

    return ret;
}

/* Decode the header of a BER/DER encoded SET.
 *
 * @param [in]      input     Buffer holding DER/BER encoded data.
 * @param [in, out] inOutIdx  On in, starting index of header.
 *                            On out, end of parsed header.
 * @param [out]     len       Number of bytes in the ASN.1 data.
 * @param [in]      maxIdx    Length of data in buffer.
 * @return  Number of bytes in the ASN.1 data on success.
 * @return  BUFFER_E when there is not enough data to parse.
 * @return  ASN_PARSE_E when the tag is not a SET or length is invalid.
 */
int GetSet(const byte* input, word32* inOutIdx, int* len,
                        word32 maxIdx)
{
    return GetASNHeader(input, ASN_SET | ASN_CONSTRUCTED, inOutIdx, len,
                        maxIdx);
}

/* Decode the header of a BER/DER encoded SET.
 *
 * @param [in]      input     Buffer holding DER/BER encoded data.
 * @param [in, out] inOutIdx  On in, starting index of header.
 *                            On out, end of parsed header.
 * @param [out]     len       Number of bytes in the ASN.1 data.
 * @param [in]      maxIdx    Length of data in buffer.
 * @param [in]      check     Whether to check the buffer has at least the
 *                            decoded length of bytes remaining.
 * @return  Number of bytes in the ASN.1 data on success.
 * @return  BUFFER_E when there is not enough data to parse.
 * @return  ASN_PARSE_E when the tag is not a SET or length is invalid.
 */
int GetSet_ex(const byte* input, word32* inOutIdx, int* len,
                        word32 maxIdx, int check)
{
    return GetASNHeader_ex(input, ASN_SET | ASN_CONSTRUCTED, inOutIdx, len,
                        maxIdx, check);
}

#if !defined(WOLFSSL_ASN_TEMPLATE) || defined(HAVE_OCSP)
/* Decode the BER/DER encoded NULL.
 *
 * No data in a NULL ASN.1 item.
 * Ensure that the all fields are as expected and move index past the element.
 *
 * @param [in]      input     Buffer holding DER/BER encoded data.
 * @param [in, out] inOutIdx  On in, starting index of NULL item.
 *                            On out, end of parsed NULL item.
 * @param [in]      maxIdx    Length of data in buffer.
 * @return  0 on success.
 * @return  BUFFER_E when there is not enough data to parse.
 * @return  ASN_TAG_NULL_E when the NULL tag is not found.
 * @return  ASN_EXPECT_0_E when the length is not zero.
 */
static int GetASNNull(const byte* input, word32* inOutIdx, word32 maxIdx)
{
    int ret = 0;
    word32 idx = *inOutIdx;

    /* Check buffer has enough data for a NULL item. */
    if ((idx + 2) > maxIdx) {
        ret = BUFFER_E;
    }
    /* Check the tag is NULL. */
    if ((ret == 0) && (input[idx++] != ASN_TAG_NULL)) {
        ret = ASN_TAG_NULL_E;
    }
    /* Check the length is zero. */
    if ((ret == 0) && (input[idx++] != 0)) {
        ret = ASN_EXPECT_0_E;
    }
    if (ret == 0) {
        /* Return the index after NULL tag. */
        *inOutIdx = idx;
    }
    /* Return error code. */
    return ret;
}
#endif

#ifndef WOLFSSL_ASN_TEMPLATE
/* Set the DER/BER encoding of the ASN.1 NULL element.
 *
 * output  Buffer to write into.
 * returns the number of bytes added to the buffer.
 */
static int SetASNNull(byte* output)
{
    output[0] = ASN_TAG_NULL;
    output[1] = 0;

    return 2;
}
#endif

#ifndef NO_CERTS
#ifndef WOLFSSL_ASN_TEMPLATE
/* Get the DER/BER encoding of an ASN.1 BOOLEAN.
 *
 * input     Buffer holding DER/BER encoded data.
 * inOutIdx  Current index into buffer to parse.
 * maxIdx    Length of data in buffer.
 * returns BUFFER_E when there is not enough data to parse.
 *         ASN_PARSE_E when the BOOLEAN tag is not found or length is not 1.
 *         Otherwise, 0 to indicate the value was false and 1 to indicate true.
 */
static int GetBoolean(const byte* input, word32* inOutIdx, word32 maxIdx)
{
    word32 idx = *inOutIdx;
    byte   b;

    if ((idx + 3) > maxIdx)
        return BUFFER_E;

    b = input[idx++];
    if (b != ASN_BOOLEAN)
        return ASN_PARSE_E;

    if (input[idx++] != 1)
        return ASN_PARSE_E;

    b = input[idx++] != 0;

    *inOutIdx = idx;
    return b;
}
#endif
#endif /* !NO_CERTS*/


/* Decode the header of a BER/DER encoded OCTET STRING.
 *
 * @param [in]      input     Buffer holding DER/BER encoded data.
 * @param [in, out] inOutIdx  On in, starting index of header.
 *                            On out, end of parsed header.
 * @param [out]     len       Number of bytes in the ASN.1 data.
 * @param [in]      maxIdx    Length of data in buffer.
 * @return  Number of bytes in the ASN.1 data on success.
 * @return  BUFFER_E when there is not enough data to parse.
 * @return  ASN_PARSE_E when the tag is not a OCTET STRING or length is invalid.
 */
int GetOctetString(const byte* input, word32* inOutIdx, int* len, word32 maxIdx)
{
    return GetASNHeader(input, ASN_OCTET_STRING, inOutIdx, len, maxIdx);
}

/* Get the DER/BER encoding of an ASN.1 INTEGER header.
 *
 * Removes the leading zero byte when found.
 *
 * input     Buffer holding DER/BER encoded data.
 * inOutIdx  Current index into buffer to parse.
 * len       The number of bytes in the ASN.1 data (excluding any leading zero).
 * maxIdx    Length of data in buffer.
 * returns BUFFER_E when there is not enough data to parse.
 *         ASN_PARSE_E when the INTEGER tag is not found, length is invalid,
 *         or invalid use of or missing leading zero.
 *         Otherwise, 0 to indicate success.
 */
int GetASNInt(const byte* input, word32* inOutIdx, int* len,
                     word32 maxIdx)
{
    int    ret;

    ret = GetASNHeader(input, ASN_INTEGER, inOutIdx, len, maxIdx);
    if (ret < 0)
        return ret;

    if (*len > 0) {
#ifndef WOLFSSL_ASN_INT_LEAD_0_ANY
        /* check for invalid padding on negative integer.
         * c.f. X.690 (ISO/IEC 8825-2:2003 (E)) 10.4.6; RFC 5280 4.1
         */
        if (*len > 1) {
            if ((input[*inOutIdx] == 0xff) && (input[*inOutIdx + 1] & 0x80)) {
                WOLFSSL_MSG("Bad INTEGER encoding of negative");
                return ASN_EXPECT_0_E;
            }
        }
#endif

        /* remove leading zero, unless there is only one 0x00 byte */
        if ((input[*inOutIdx] == 0x00) && (*len > 1)) {
            (*inOutIdx)++;
            (*len)--;

#ifndef WOLFSSL_ASN_INT_LEAD_0_ANY
            if (*len > 0 && (input[*inOutIdx] & 0x80) == 0) {
                WOLFSSL_MSG("INTEGER is negative");
                return ASN_EXPECT_0_E;
            }
#endif
        }
    }

    return 0;
}

#ifndef WOLFSSL_ASN_TEMPLATE
#if !defined(NO_CERTS) && defined(WOLFSSL_CUSTOM_CURVES)
/* Get the DER/BER encoding of an ASN.1 INTEGER that has a value of no more than
 * 7 bits.
 *
 * input     Buffer holding DER/BER encoded data.
 * inOutIdx  Current index into buffer to parse.
 * maxIdx    Length of data in buffer.
 * returns BUFFER_E when there is not enough data to parse.
 *         ASN_PARSE_E when the INTEGER tag is not found or length is invalid.
 *         Otherwise, the 7-bit value.
 */
static int GetInteger7Bit(const byte* input, word32* inOutIdx, word32 maxIdx)
{
    word32 idx = *inOutIdx;
    byte   b;

    if ((idx + 3) > maxIdx)
        return BUFFER_E;

    if (GetASNTag(input, &idx, &b, maxIdx) != 0)
        return ASN_PARSE_E;
    if (b != ASN_INTEGER)
        return ASN_PARSE_E;
    if (input[idx++] != 1)
        return ASN_PARSE_E;
    b = input[idx++];

    *inOutIdx = idx;
    return b;
}
#endif /* !NO_CERTS */

#if ((defined(WC_RSA_PSS) && !defined(NO_RSA)) || !defined(NO_CERTS))
/* Get the DER/BER encoding of an ASN.1 INTEGER that has a value of no more than
 * 16 bits.
 *
 * input     Buffer holding DER/BER encoded data.
 * inOutIdx  Current index into buffer to parse.
 * maxIdx    Length of data in buffer.
 * returns BUFFER_E when there is not enough data to parse.
 *         ASN_PARSE_E when the INTEGER tag is not found or length is invalid.
 *         Otherwise, the 16-bit value.
 */
static int GetInteger16Bit(const byte* input, word32* inOutIdx, word32 maxIdx)
{
    word32 idx = *inOutIdx;
    byte tag;
    word16 n;

    if ((idx + 2) > maxIdx)
        return BUFFER_E;

    if (GetASNTag(input, &idx, &tag, maxIdx) != 0)
        return ASN_PARSE_E;
    if (tag != ASN_INTEGER)
        return ASN_PARSE_E;
    if (input[idx] == 1) {
        idx++;
        if ((idx + 1) > maxIdx) {
            return ASN_PARSE_E;
        }
        n = input[idx++];
    }
    else if (input[idx] == 2) {
        idx++;
        if ((idx + 2) > maxIdx) {
            return ASN_PARSE_E;
        }
        n = input[idx++];
        n = (n << 8) | input[idx++];
    }
    else
        return ASN_PARSE_E;

    *inOutIdx = idx;
    return n;
}
#endif /* WC_RSA_PSS && !NO_RSA */
#endif /* !WOLFSSL_ASN_TEMPLATE */

#if !defined(NO_DSA) && !defined(NO_SHA)
static const char sigSha1wDsaName[] = "SHAwDSA";
static const char sigSha256wDsaName[] = "SHA256wDSA";
#endif /* NO_DSA */
#ifndef NO_RSA
#ifdef WOLFSSL_MD2
    static const char  sigMd2wRsaName[] = "md2WithRSAEncryption";
#endif
#ifndef NO_MD5
    static const char  sigMd5wRsaName[] = "md5WithRSAEncryption";
#endif
#ifndef NO_SHA
    static const char  sigSha1wRsaName[] = "sha1WithRSAEncryption";
#endif
#ifdef WOLFSSL_SHA224
    static const char sigSha224wRsaName[] = "sha224WithRSAEncryption";
#endif
#ifndef NO_SHA256
    static const char sigSha256wRsaName[] = "sha256WithRSAEncryption";
#endif
#ifdef WOLFSSL_SHA384
    static const char sigSha384wRsaName[] = "sha384WithRSAEncryption";
#endif
#ifdef WOLFSSL_SHA512
    static const char sigSha512wRsaName[] = "sha512WithRSAEncryption";
#endif
#ifdef WOLFSSL_SHA3
#ifndef WOLFSSL_NOSHA3_224
    static const char sigSha3_224wRsaName[] = "sha3_224WithRSAEncryption";
#endif
#ifndef WOLFSSL_NOSHA3_256
    static const char sigSha3_256wRsaName[] = "sha3_256WithRSAEncryption";
#endif
#ifndef WOLFSSL_NOSHA3_384
    static const char sigSha3_384wRsaName[] = "sha3_384WithRSAEncryption";
#endif
#ifndef WOLFSSL_NOSHA3_512
    static const char sigSha3_512wRsaName[] = "sha3_512WithRSAEncryption";
#endif
#endif
#ifdef WC_RSA_PSS
    static const char sigRsaSsaPssName[] = "rsassaPss";
#endif
#endif /* NO_RSA */
#ifdef HAVE_ECC
#ifndef NO_SHA
    static const char sigSha1wEcdsaName[] = "SHAwECDSA";
#endif
#ifdef WOLFSSL_SHA224
    static const char sigSha224wEcdsaName[] = "SHA224wECDSA";
#endif
#ifndef NO_SHA256
    static const char sigSha256wEcdsaName[] = "SHA256wECDSA";
#endif
#ifdef WOLFSSL_SHA384
    static const char sigSha384wEcdsaName[] = "SHA384wECDSA";
#endif
#ifdef WOLFSSL_SHA512
    static const char sigSha512wEcdsaName[] = "SHA512wECDSA";
#endif
#ifdef WOLFSSL_SHA3
#ifndef WOLFSSL_NOSHA3_224
    static const char sigSha3_224wEcdsaName[] = "SHA3_224wECDSA";
#endif
#ifndef WOLFSSL_NOSHA3_256
    static const char sigSha3_256wEcdsaName[] = "SHA3_256wECDSA";
#endif
#ifndef WOLFSSL_NOSHA3_384
    static const char sigSha3_384wEcdsaName[] = "SHA3_384wECDSA";
#endif
#ifndef WOLFSSL_NOSHA3_512
    static const char sigSha3_512wEcdsaName[] = "SHA3_512wECDSA";
#endif
#endif
#endif /* HAVE_ECC */
static const char sigUnknownName[] = "Unknown";


/* Get the human readable string for a signature type
 *
 * oid  Oid value for signature
 */
const char* GetSigName(int oid) {
    switch (oid) {
    #if !defined(NO_DSA) && !defined(NO_SHA)
        case CTC_SHAwDSA:
            return sigSha1wDsaName;
        case CTC_SHA256wDSA:
            return sigSha256wDsaName;
    #endif /* NO_DSA && NO_SHA */
    #ifndef NO_RSA
        #ifdef WOLFSSL_MD2
        case CTC_MD2wRSA:
            return sigMd2wRsaName;
        #endif
        #ifndef NO_MD5
        case CTC_MD5wRSA:
            return sigMd5wRsaName;
        #endif
        #ifndef NO_SHA
        case CTC_SHAwRSA:
            return sigSha1wRsaName;
        #endif
        #ifdef WOLFSSL_SHA224
        case CTC_SHA224wRSA:
            return sigSha224wRsaName;
        #endif
        #ifndef NO_SHA256
        case CTC_SHA256wRSA:
            return sigSha256wRsaName;
        #endif
        #ifdef WOLFSSL_SHA384
        case CTC_SHA384wRSA:
            return sigSha384wRsaName;
        #endif
        #ifdef WOLFSSL_SHA512
        case CTC_SHA512wRSA:
            return sigSha512wRsaName;
        #endif
        #ifdef WOLFSSL_SHA3
        #ifndef WOLFSSL_NOSHA3_224
        case CTC_SHA3_224wRSA:
            return sigSha3_224wRsaName;
        #endif
        #ifndef WOLFSSL_NOSHA3_256
        case CTC_SHA3_256wRSA:
            return sigSha3_256wRsaName;
        #endif
        #ifndef WOLFSSL_NOSHA3_384
        case CTC_SHA3_384wRSA:
            return sigSha3_384wRsaName;
        #endif
        #ifndef WOLFSSL_NOSHA3_512
        case CTC_SHA3_512wRSA:
            return sigSha3_512wRsaName;
        #endif
        #endif
        #ifdef WC_RSA_PSS
        case CTC_RSASSAPSS:
            return sigRsaSsaPssName;
        #endif
    #endif /* NO_RSA */
    #ifdef HAVE_ECC
        #ifndef NO_SHA
        case CTC_SHAwECDSA:
            return sigSha1wEcdsaName;
        #endif
        #ifdef WOLFSSL_SHA224
        case CTC_SHA224wECDSA:
            return sigSha224wEcdsaName;
        #endif
        #ifndef NO_SHA256
        case CTC_SHA256wECDSA:
            return sigSha256wEcdsaName;
        #endif
        #ifdef WOLFSSL_SHA384
        case CTC_SHA384wECDSA:
            return sigSha384wEcdsaName;
        #endif
        #ifdef WOLFSSL_SHA512
        case CTC_SHA512wECDSA:
            return sigSha512wEcdsaName;
        #endif
        #ifdef WOLFSSL_SHA3
        #ifndef WOLFSSL_NOSHA3_224
        case CTC_SHA3_224wECDSA:
            return sigSha3_224wEcdsaName;
        #endif
        #ifndef WOLFSSL_NOSHA3_256
        case CTC_SHA3_256wECDSA:
            return sigSha3_256wEcdsaName;
        #endif
        #ifndef WOLFSSL_NOSHA3_384
        case CTC_SHA3_384wECDSA:
            return sigSha3_384wEcdsaName;
        #endif
        #ifndef WOLFSSL_NOSHA3_512
        case CTC_SHA3_512wECDSA:
            return sigSha3_512wEcdsaName;
        #endif
        #endif
    #endif /* HAVE_ECC */
        default:
            return sigUnknownName;
    }
}


#if !defined(WOLFSSL_ASN_TEMPLATE) || defined(HAVE_PKCS7) || \
    defined(OPENSSL_EXTRA)
#if !defined(NO_DSA) || defined(HAVE_ECC) || !defined(NO_CERTS) || \
   (!defined(NO_RSA) && \
        (defined(WOLFSSL_CERT_GEN) || \
        ((defined(WOLFSSL_KEY_GEN) || defined(OPENSSL_EXTRA)))))
/* Set the DER/BER encoding of the ASN.1 INTEGER header.
 *
 * When output is NULL, calculate the header length only.
 *
 * @param [in]  len        Length of INTEGER data in bytes.
 * @param [in]  firstByte  First byte of data, most significant byte of integer,
 *                         to encode.
 * @param [out] output     Buffer to write into.
 * @return  Number of bytes added to the buffer.
 */
int SetASNInt(int len, byte firstByte, byte* output)
{
    int idx = 0;

    if (output) {
        /* Write out tag. */
        output[idx] = ASN_INTEGER;
    }
    /* Step over tag. */
    idx += ASN_TAG_SZ;
    /* Check if first byte has top bit set in which case a 0 is needed to
     * maintain positive value. */
    if (firstByte & 0x80) {
        /* Add pre-prepended byte to length of data in INTEGER. */
        len++;
    }
    /* Encode length - passing NULL for output will not encode. */
    idx += (int)SetLength((word32)len, output ? output + idx : NULL);
    /* Put out prepended 0 as well. */
    if (firstByte & 0x80) {
        if (output) {
            /* Write out 0 byte. */
            output[idx] = 0x00;
        }
        /* Update index. */
        idx++;
    }

    /* Return index after header. */
    return idx;
}
#endif
#endif

#ifndef WOLFSSL_ASN_TEMPLATE
#if !defined(NO_DSA) || defined(HAVE_ECC) || (defined(WOLFSSL_CERT_GEN) && \
    !defined(NO_RSA)) || ((defined(WOLFSSL_KEY_GEN) || \
    (!defined(NO_DH) && defined(WOLFSSL_DH_EXTRA)) || \
    defined(OPENSSL_EXTRA)) && !defined(NO_RSA))
/* Set the DER/BER encoding of the ASN.1 INTEGER element with an mp_int.
 * The number is assumed to be positive.
 *
 * n       Multi-precision integer to encode.
 * maxSz   Maximum size of the encoded integer.
 *         A negative value indicates no check of length requested.
 * output  Buffer to write into.
 * returns BUFFER_E when the data is too long for the buffer.
 *         MP_TO_E when encoding the integer fails.
 *         Otherwise, the number of bytes added to the buffer.
 */
static int SetASNIntMP(mp_int* n, int maxSz, byte* output)
{
    int idx = 0;
    int leadingBit;
    int length;

    leadingBit = mp_leading_bit(n);
    length = mp_unsigned_bin_size(n);
    if (maxSz >= 0 && (1 + length + (leadingBit ? 1 : 0)) > maxSz)
        return BUFFER_E;
    idx = SetASNInt(length, (byte)(leadingBit ? 0x80U : 0x00U), output);
    if (maxSz >= 0 && (idx + length) > maxSz)
        return BUFFER_E;

    if (output) {
        int err = mp_to_unsigned_bin(n, output + idx);
        if (err != MP_OKAY)
            return MP_TO_E;
    }
    idx += length;

    return idx;
}
#endif
#endif /* !WOLFSSL_ASN_TEMPLATE */

#ifdef WOLFSSL_ASN_TEMPLATE
/* ASN.1 template for an INTEGER. */
static const ASNItem intASN[] = {
/* INT */ { 0, ASN_INTEGER, 0, 0, 0 }
};
enum {
    INTASN_IDX_INT = 0
};

/* Number of items in ASN.1 template for an INTEGER. */
#define intASN_Length (sizeof(intASN) / sizeof(ASNItem))
#endif /* WOLFSSL_ASN_TEMPLATE */

/* Windows header clash for WinCE using GetVersion */
/* Decode Version - one byte INTEGER.
 *
 * @param [in]      input     Buffer of BER data.
 * @param [in, out] inOutIdx  On in, start of encoded Version.
 *                            On out, start of next encode ASN.1 item.
 * @param [out]     version   Number encoded in INTEGER.
 * @param [in]      maxIdx    Maximum index of data in buffer.
 * @return  0 on success.
 * @return  ASN_PARSE_E when encoding is invalid.
 * @return  BUFFER_E when data in buffer is too small.
 * @return  ASN_EXPECT_0_E when the most significant bit is set.
 */
int GetMyVersion(const byte* input, word32* inOutIdx,
                               int* version, word32 maxIdx)
{
#ifndef WOLFSSL_ASN_TEMPLATE
    word32 idx = *inOutIdx;
    byte   tag;

    if ((idx + MIN_VERSION_SZ) > maxIdx)
        return ASN_PARSE_E;

    if (GetASNTag(input, &idx, &tag, maxIdx) != 0)
        return ASN_PARSE_E;

    if (tag != ASN_INTEGER)
        return ASN_PARSE_E;

    if (input[idx++] != 0x01)
        return ASN_VERSION_E;

    *version  = input[idx++];
    *inOutIdx = idx;

    return *version;
#else
    ASNGetData dataASN[intASN_Length];
    int ret;
    byte num = 0;

    /* Clear dynamic data and set the version number variable. */
    XMEMSET(dataASN, 0, sizeof(dataASN));
    GetASN_Int8Bit(&dataASN[INTASN_IDX_INT], &num);
    /* Decode the version (INTEGER). */
    ret = GetASN_Items(intASN, dataASN, intASN_Length, 0, input, inOutIdx,
                       maxIdx);
    if (ret == 0) {
        /* Return version through variable and return value. */
        *version = num;
        ret = num;
    }
    return ret;
#endif /* WOLFSSL_ASN_TEMPLATE */
}


#if !defined(NO_PWDBASED) || defined(WOLFSSL_ASN_EXTRA)
/* Decode small positive integer, 32 bits or less.
 *
 * @param [in]      input     Buffer of BER data.
 * @param [in, out] inOutIdx  On in, start of encoded INTEGER.
 *                            On out, start of next encode ASN.1 item.
 * @param [out]     number    Number encoded in INTEGER.
 * @param [in]      maxIdx    Maximum index of data in buffer.
 * @return  0 on success.
 * @return  ASN_PARSE_E when encoding is invalid.
 * @return  BUFFER_E when data in buffer is too small.
 * @return  ASN_EXPECT_0_E when the most significant bit is set.
 */
int GetShortInt(const byte* input, word32* inOutIdx, int* number, word32 maxIdx)
{
#ifndef WOLFSSL_ASN_TEMPLATE
    word32 idx = *inOutIdx;
    word32 len;
    byte   tag;

    *number = 0;

    /* check for type and length bytes */
    if ((idx + 2) > maxIdx)
        return BUFFER_E;

    if (GetASNTag(input, &idx, &tag, maxIdx) != 0)
        return ASN_PARSE_E;

    if (tag != ASN_INTEGER)
        return ASN_PARSE_E;

    len = input[idx++];
    if (len > 4)
        return ASN_PARSE_E;

    if (len + idx > maxIdx)
        return ASN_PARSE_E;

    if (input[idx] >= 0x80U) {
        /* This function only expects positive INTEGER values. */
        return ASN_EXPECT_0_E;
    }

    while (len--) {
        *number  = *number << 8 | input[idx++];
    }

    *inOutIdx = idx;

    return *number;
#else
    ASNGetData dataASN[intASN_Length];
    int ret;
    word32 num = 0;

    /* Clear dynamic data and set the 32-bit number variable. */
    XMEMSET(dataASN, 0, sizeof(dataASN));
    GetASN_Int32Bit(&dataASN[INTASN_IDX_INT], &num);
    /* Decode the short int (INTEGER). */
    ret = GetASN_Items(intASN, dataASN, intASN_Length, 0, input, inOutIdx,
                       maxIdx);
    if (ret == 0) {
        /* Return number through variable and return value. */
        *number = (int)num;
        ret = (int)num;
    }
    return ret;
#endif
}
#endif /* !NO_PWDBASED || WOLFSSL_ASN_EXTRA */


#ifndef NO_PWDBASED
#if !defined(WOLFSSL_ASN_TEMPLATE) || defined(HAVE_PKCS8) || \
     defined(HAVE_PKCS12)
/* Set small integer, 32 bits or less. DER encoding with no leading 0s
 * returns total amount written including ASN tag and length byte on success */
int SetShortInt(byte* output, word32* inOutIdx, word32 number, word32 maxIdx)
{
    word32 idx = *inOutIdx;
    word32 len;
    int    i;
    word32 extraByte = 0;

    if (number == 0)
        len = 1;
    else
        len = BytePrecision(number);

    /* clarify the len range to prepare for the next right bit shifting */
    if (len < 1 || len > sizeof(number)) {
        return ASN_PARSE_E;
    }
    if (number >> (WOLFSSL_BIT_SIZE * len - 1)) {
        /* Need one byte of zero value not to be negative number */
        extraByte = 1;
    }

    /* check for room for type and length bytes. */
    if ((idx + 2 + extraByte + len) > maxIdx)
        return BUFFER_E;

    /* check that MAX_SHORT_SZ allows this size of ShortInt. */
    if (2 + extraByte + len > MAX_SHORT_SZ)
        return ASN_PARSE_E;

    output[idx++] = ASN_INTEGER;
    output[idx++] = (byte)(len + extraByte);
    if (extraByte) {
        output[idx++] = 0x00;
    }

    for (i = (int)len - 1; i >= 0; --i)
        output[idx++] = (byte)(number >> (i * WOLFSSL_BIT_SIZE));

    len = idx - *inOutIdx;
    *inOutIdx = idx;

    return (int)len;
}
#endif /* !WOLFSSL_ASN_TEMPLATE || HAVE_PKCS8 || HAVE_PKCS12 */
#endif /* !NO_PWDBASED */

#if !defined(WOLFSSL_ASN_TEMPLATE) && !defined(NO_CERTS)
/* May not have one, not an error */
static int GetExplicitVersion(const byte* input, word32* inOutIdx, int* version,
                              word32 maxIdx)
{
    word32 idx = *inOutIdx;
    byte tag;

    WOLFSSL_ENTER("GetExplicitVersion");

    if (GetASNTag(input, &idx, &tag, maxIdx) != 0)
        return ASN_PARSE_E;

    if (tag == (ASN_CONTEXT_SPECIFIC | ASN_CONSTRUCTED)) {
        int ret;

        *inOutIdx = ++idx;  /* skip header */
        ret = GetMyVersion(input, inOutIdx, version, maxIdx);
        if (ret >= 0) {
            /* check if version is expected value rfc 5280 4.1 {0, 1, 2} */
            if (*version > MAX_X509_VERSION || *version < MIN_X509_VERSION) {
                WOLFSSL_MSG("Unexpected certificate version");
                WOLFSSL_ERROR_VERBOSE(ASN_VERSION_E);
                ret = ASN_VERSION_E;
            }
        }
        return ret;
    }

    /* go back as is */
    *version = 0;

    return 0;
}
#endif

/* Decode small integer, 32 bits or less.
 *
 * mp_int is initialized.
 *
 * @param [out]     mpi       mp_int to hold number.
 * @param [in]      input     Buffer of BER data.
 * @param [in, out] inOutIdx  On in, start of encoded INTEGER.
 *                            On out, start of next encode ASN.1 item.
 * @param [in]      maxIdx    Maximum index of data in buffer.
 * @return  0 on success.
 * @return  ASN_PARSE_E when encoding is invalid.
 * @return  BUFFER_E when data in buffer is too small.
 * @return  ASN_EXPECT_0_E when the most significant bit is set.
 * @return  MP_INIT_E when the unable to initialize an mp_int.
 * @return  ASN_GETINT_E when the unable to convert data to an mp_int.
 */
int GetInt(mp_int* mpi, const byte* input, word32* inOutIdx, word32 maxIdx)
{
#ifndef WOLFSSL_ASN_TEMPLATE
    word32 idx = *inOutIdx;
    int    ret;
    int    length;

    ret = GetASNInt(input, &idx, &length, maxIdx);
    if (ret != 0)
        return ret;

    if (mp_init(mpi) != MP_OKAY)
        return MP_INIT_E;

    if (mp_read_unsigned_bin(mpi, input + idx, (word32)length) != 0) {
        mp_clear(mpi);
        return ASN_GETINT_E;
    }

#ifdef HAVE_WOLF_BIGINT
    if (wc_bigint_from_unsigned_bin(&mpi->raw, input + idx, length) != 0) {
        mp_clear(mpi);
        return ASN_GETINT_E;
    }
#endif /* HAVE_WOLF_BIGINT */

    *inOutIdx = idx + (word32)length;

    return 0;
#else
    ASNGetData dataASN[intASN_Length];

    /* Clear dynamic data and set the mp_int to fill with value. */
    XMEMSET(dataASN, 0, sizeof(dataASN));
    GetASN_MP_PosNeg(&dataASN[INTASN_IDX_INT], mpi);
    /* Decode the big number (INTEGER). */
    return GetASN_Items(intASN, dataASN, intASN_Length, 0, input, inOutIdx,
                        maxIdx);
#endif
}

#if (defined(HAVE_ECC) || !defined(NO_DSA)) && !defined(WOLFSSL_ASN_TEMPLATE)
static int GetIntPositive(mp_int* mpi, const byte* input, word32* inOutIdx,
    word32 maxIdx, int initNum)
{
    word32 idx = *inOutIdx;
    int    ret;
    int    length;

    ret = GetASNInt(input, &idx, &length, maxIdx);
    if (ret != 0)
        return ret;

    /* should not be hit but adding in an additional sanity check */
    if (idx + length > maxIdx) {
        return MP_INIT_E;
    }

    if ((input[idx] & 0x80) == 0x80) {
        if (idx < 1) {
            /* needs at least one byte for length value */
            return MP_INIT_E;
        }

        if (input[idx - 1] != 0x00) {
            return MP_INIT_E;
        }
    }

    if (initNum) {
        if (mp_init(mpi) != MP_OKAY)
            return MP_INIT_E;
    }

    if (mp_read_unsigned_bin(mpi, input + idx, (word32)length) != 0) {
        mp_clear(mpi);
        return ASN_GETINT_E;
    }

#ifdef HAVE_WOLF_BIGINT
    if (wc_bigint_from_unsigned_bin(&mpi->raw, input + idx, length) != 0) {
        mp_clear(mpi);
        return ASN_GETINT_E;
    }
#endif /* HAVE_WOLF_BIGINT */

    *inOutIdx = idx + (word32)length;

    return 0;
}
#endif /* (ECC || !NO_DSA) && !WOLFSSL_ASN_TEMPLATE */

#ifndef WOLFSSL_ASN_TEMPLATE
#if !defined(NO_RSA) || !defined(NO_DSA)
static int SkipInt(const byte* input, word32* inOutIdx, word32 maxIdx)
{
    word32 idx = *inOutIdx;
    int    ret;
    int    length;

    ret = GetASNInt(input, &idx, &length, maxIdx);
    if (ret != 0)
        return ret;

    *inOutIdx = idx + (word32)length;

    return 0;
}
#endif
#endif /* !WOLFSSL_ASN_TEMPLATE */

#ifdef WOLFSSL_ASN_TEMPLATE
/* ASN.1 template for a BIT_STRING. */
static const ASNItem bitStringASN[] = {
/* BIT_STR */ { 0, ASN_BIT_STRING, 0, 1, 0 }
};
enum {
    BITSTRINGASN_IDX_BIT_STR = 0
};

/* Number of items in ASN.1 template for a BIT_STRING. */
#define bitStringASN_Length (sizeof(bitStringASN) / sizeof(ASNItem))
#endif

/* Decode and check the BIT_STRING is valid. Return length and unused bits.
 *
 * @param [in]      input       Buffer holding BER encoding.
 * @param [in, out] inOutIdx    On in, start of BIT_STRING.
 *                              On out, start of ASN.1 item after BIT_STRING.
 * @param [out]     len         Length of BIT_STRING data.
 * @param [in]      maxIdx      Maximum index of data in buffer.
 * @param [in]      zeroBits    Indicates whether zero unused bits is expected.
 * @param [in]      unusedBits  Number of unused bits in last byte.
 * @return  0 on success.
 * @return  ASN_PARSE_E when encoding is invalid.
 * @return  ASN_BITSTR_E when the expected BIT_STRING tag is not found.
 * @return  BUFFER_E when data in buffer is too small.
 * @return  ASN_EXPECT_0_E when unused bits is not zero when expected.
 */
int CheckBitString(const byte* input, word32* inOutIdx, int* len,
                          word32 maxIdx, int zeroBits, byte* unusedBits)
{
#ifndef WOLFSSL_ASN_TEMPLATE
    word32 idx = *inOutIdx;
    int    length;
    byte   b;

    if (GetASNTag(input, &idx, &b, maxIdx) != 0) {
        return ASN_BITSTR_E;
    }

    if (b != ASN_BIT_STRING) {
        return ASN_BITSTR_E;
    }

    if (GetLength(input, &idx, &length, maxIdx) < 0)
        return ASN_PARSE_E;

    /* extra sanity check that length is greater than 0 */
    if (length <= 0) {
        WOLFSSL_MSG("Error length was 0 in CheckBitString");
        return BUFFER_E;
    }

    if (idx + 1 > maxIdx) {
        WOLFSSL_MSG("Attempted buffer read larger than input buffer");
        return BUFFER_E;
    }

    b = input[idx];
    if (zeroBits && (b != 0x00))
        return ASN_EXPECT_0_E;
    if (b >= 0x08)
        return ASN_PARSE_E;
    if (b != 0) {
        if ((byte)(input[idx + (word32)length - 1] << (8 - b)) != 0)
            return ASN_PARSE_E;
    }
    idx++;
    length--; /* length has been checked for greater than 0 */

    *inOutIdx = idx;
    if (len != NULL)
        *len = length;
    if (unusedBits != NULL)
        *unusedBits = b;

    return 0;
#else
    ASNGetData dataASN[bitStringASN_Length];
    int ret;
    int bits = 0;

    /* Parse BIT_STRING and check validity of unused bits. */
    XMEMSET(dataASN, 0, sizeof(dataASN));
    /* Decode BIT_STRING. */
    ret = GetASN_Items(bitStringASN, dataASN, bitStringASN_Length, 0, input,
            inOutIdx, maxIdx);
    if (ret == 0) {
        /* Get unused bits from dynamic ASN.1 data. */
        bits = GetASNItem_UnusedBits(dataASN[BITSTRINGASN_IDX_BIT_STR]);
        /* Check unused bits is 0 when expected. */
        if (zeroBits && (bits != 0)) {
            ret = ASN_EXPECT_0_E;
        }
    }
    if (ret == 0) {
        /* Return length of data and unused bits if required. */
        if (len != NULL) {
            *len = (int)dataASN[BITSTRINGASN_IDX_BIT_STR].data.ref.length;
        }
        if (unusedBits != NULL) {
            *unusedBits = (byte)bits;
        }
    }

    return ret;
#endif
}

/* RSA (with CertGen or KeyGen) OR ECC OR ED25519 OR ED448 (with CertGen or
 * KeyGen) */
#if (!defined(NO_RSA) && \
     (defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_KEY_GEN) || \
      defined(OPENSSL_EXTRA))) || \
    (defined(HAVE_ECC) && defined(HAVE_ECC_KEY_EXPORT)) || \
    ((defined(HAVE_ED25519) || defined(HAVE_ED448)) && \
     (defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_KEY_GEN) || \
      defined(OPENSSL_EXTRA))) || \
    (defined(WC_ENABLE_ASYM_KEY_EXPORT) && !defined(NO_CERTS)) || \
    (!defined(NO_DSA) && !defined(HAVE_SELFTEST) && defined(WOLFSSL_KEY_GEN)) || \
    (!defined(NO_DH) && defined(WOLFSSL_DH_EXTRA))

/* Set the DER/BER encoding of the ASN.1 BIT STRING header.
 *
 * When output is NULL, calculate the header length only.
 *
 * @param [in]  len         Length of BIT STRING data.
 *                          That is, the number of least significant zero bits
 *                          before a one.
 *                          The last byte is the most-significant non-zero byte
 *                          of a number.
 * @param [out] output      Buffer to write into.
 * @return  Number of bytes added to the buffer.
 */
word32 SetBitString(word32 len, byte unusedBits, byte* output)
{
    word32 idx = 0;

    if (output) {
        /* Write out tag. */
        output[idx] = ASN_BIT_STRING;
    }
    /* Step over tag. */
    idx += ASN_TAG_SZ;

    /* Encode length - passing NULL for output will not encode.
     * Add one to length for unused bits. */
    idx += SetLength(len + 1, output ? output + idx : NULL);
    if (output) {
        /* Write out unused bits. */
        output[idx] = unusedBits;
    }
    /* Skip over unused bits. */
    idx++;

    /* Return index after header. */
    return idx;
}
#endif /* !NO_RSA || HAVE_ECC || HAVE_ED25519 || HAVE_ED448 */

#ifdef ASN_BER_TO_DER

#ifndef BER_OCTET_LENGTH
    #define BER_OCTET_LENGTH 4096
#endif

/* sets the terminating 0x00 0x00 at the end of an indefinite length
 * returns the number of bytes written */
word32 SetIndefEnd(byte* output)
{
    byte terminate[ASN_INDEF_END_SZ] = { 0x00, 0x00 };

    if (output != NULL) {
        XMEMCPY(output, terminate, ASN_INDEF_END_SZ);
    }

    return (word32)ASN_INDEF_END_SZ;
}


/* Breaks an octet string up into chunks for use with streaming
 * returns 0 on success and updates idx */
int StreamOctetString(const byte* inBuf, word32 inBufSz, byte* out,
                      word32* outSz, word32* idx)
{
    word32 i  = 0;
    word32 outIdx = *idx;
    byte* tmp = out;

    if (tmp) tmp += outIdx;

    while (i < inBufSz) {
        word32 ret, sz;

        sz = BER_OCTET_LENGTH;

        if ((sz + i) > inBufSz) {
            sz = inBufSz - i;
        }

        ret = SetOctetString(sz, tmp);
        if (ret > 0) {
            outIdx += ret;
        }

        if (tmp) {
            if ((word32)ret + sz + i + outIdx > *outSz) {
                return BUFFER_E;
            }
            XMEMCPY(tmp + ret, inBuf + i, sz);
            tmp += sz + ret;
        }
        outIdx += sz;
        i      += sz;
    }

    if (tmp) {
        *idx = outIdx;
        return 0;
    }
    else {
        *outSz = outIdx;
        return WC_NO_ERR_TRACE(LENGTH_ONLY_E);
    }
}


/* Convert BER to DER */

/* Pull information from the ASN.1 BER encoded item header */
static int GetBerHeader(const byte* data, word32* idx, word32 maxIdx,
                        byte* pTag, word32* pLen, int* indef)
{
    int len = 0;
    byte tag;
    word32 i = *idx;

    *indef = 0;

    /* Check there is enough data for a minimal header */
    if (i + 2 > maxIdx) {
        return ASN_PARSE_E;
    }

    /* Retrieve tag */
    tag = data[i++];

    /* Indefinite length handled specially */
    if (data[i] == ASN_INDEF_LENGTH) {
        /* Check valid tag for indefinite */
        if (((tag & 0xc0) == 0) && ((tag & ASN_CONSTRUCTED) == 0x00)) {
            return ASN_PARSE_E;
        }
        i++;
        *indef = 1;
    }
    else if (GetLength(data, &i, &len, maxIdx) < 0) {
        return ASN_PARSE_E;
    }

    /* Return tag, length and index after BER item header */
    *pTag = tag;
    *pLen = (word32)len;
    *idx = i;
    return 0;
}

#ifndef INDEF_ITEMS_MAX
#define INDEF_ITEMS_MAX       20
#endif

/* Indef length item data */
typedef struct Indef {
    word32 start;
    int depth;
    int headerLen;
    word32 len;
} Indef;

/* Indef length items */
typedef struct IndefItems
{
    Indef len[INDEF_ITEMS_MAX];
    int cnt;
    int idx;
    int depth;
} IndefItems;


/* Get header length of current item */
static int IndefItems_HeaderLen(IndefItems* items)
{
    return items->len[items->idx].headerLen;
}

/* Get data length of current item */
static word32 IndefItems_Len(IndefItems* items)
{
    return items->len[items->idx].len;
}

/* Add a indefinite length item */
static int IndefItems_AddItem(IndefItems* items, word32 start)
{
    int ret = 0;
    int i;

    if (items->cnt == INDEF_ITEMS_MAX) {
        ret = MEMORY_E;
    }
    else {
        i = items->cnt++;
        items->len[i].start = start;
        items->len[i].depth = items->depth++;
        items->len[i].headerLen = 1;
        items->len[i].len = 0;
        items->idx = i;
    }

    return ret;
}

/* Increase data length of current item */
static void IndefItems_AddData(IndefItems* items, word32 length)
{
    items->len[items->idx].len += length;
}

/* Update header length of current item to reflect data length */
static void IndefItems_UpdateHeaderLen(IndefItems* items)
{
    items->len[items->idx].headerLen +=
                               (int)SetLength(items->len[items->idx].len, NULL);
}

/* Go to indefinite parent of current item */
static void IndefItems_Up(IndefItems* items)
{
    int i;
    int depth = items->len[items->idx].depth - 1;

    for (i = items->cnt - 1; i >= 0; i--) {
        if (items->len[i].depth == depth) {
            break;
        }
    }
    items->idx = i;
    items->depth = depth + 1;
}

/* Calculate final length by adding length of indefinite child items */
static void IndefItems_CalcLength(IndefItems* items)
{
    int i;
    int idx = items->idx;

    for (i = idx + 1; i < items->cnt; i++) {
        if (items->len[i].depth == items->depth) {
            items->len[idx].len += (word32)items->len[i].headerLen;
            items->len[idx].len += items->len[i].len;
        }
    }
    items->len[idx].headerLen += (int)SetLength(items->len[idx].len, NULL);
}

/* Add more data to indefinite length item */
static void IndefItems_MoreData(IndefItems* items, word32 length)
{
    if (items->cnt > 0 && items->idx >= 0) {
        items->len[items->idx].len += length;
    }
}

/* Convert a BER encoding with indefinite length items to DER.
 *
 * ber    BER encoded data.
 * berSz  Length of BER encoded data.
 * der    Buffer to hold DER encoded version of data.
 *        NULL indicates only the length is required.
 * derSz  The size of the buffer to hold the DER encoded data.
 *        Will be set if der is NULL, otherwise the value is checked as der is
 *        filled.
 * returns ASN_PARSE_E if the BER data is invalid and BAD_FUNC_ARG if ber or
 * derSz are NULL.
 */
int wc_BerToDer(const byte* ber, word32 berSz, byte* der, word32* derSz)
{
    int ret = 0;
    word32 i, j;
    WC_DECLARE_VAR(indefItems, IndefItems, 1, 0);
    byte tag, basic;
    word32 length;
    int indef;

    if (ber == NULL || derSz == NULL)
        return BAD_FUNC_ARG;

    WC_ALLOC_VAR_EX(indefItems, IndefItems, 1, NULL, DYNAMIC_TYPE_TMP_BUFFER,
    {
        ret=MEMORY_E;
        goto end;
    });

    XMEMSET(indefItems, 0, sizeof(*indefItems));

    /* Calculate indefinite item lengths */
    for (i = 0; i < berSz; ) {
        word32 start = i;

        /* Get next BER item */
        ret = GetBerHeader(ber, &i, berSz, &tag, &length, &indef);
        if (ret != 0) {
            goto end;
        }

        if (indef) {
            /* Indefinite item - add to list */
            ret = IndefItems_AddItem(indefItems, i);
            if (ret != 0) {
                goto end;
            }

            if ((tag & 0xC0) == 0 &&
                tag != (ASN_SEQUENCE | ASN_CONSTRUCTED) &&
                tag != (ASN_SET      | ASN_CONSTRUCTED)) {
                /* Constructed basic type - get repeating tag */
                basic = (byte)(tag & (~ASN_CONSTRUCTED));

                /* Add up lengths of each item below */
                for (; i < berSz; ) {
                    /* Get next BER_item */
                    ret = GetBerHeader(ber, &i, berSz, &tag, &length, &indef);
                    if (ret != 0) {
                        goto end;
                    }

                    /* End of content closes item */
                    if (tag == ASN_EOC) {
                        /* Must be zero length */
                        if (length != 0) {
                            ret = ASN_PARSE_E;
                            goto end;
                        }
                        break;
                    }

                    /* Must not be indefinite and tag must match parent */
                    if (indef || tag != basic) {
                        ret = ASN_PARSE_E;
                        goto end;
                    }

                    /* Add to length */
                    IndefItems_AddData(indefItems, length);
                    /* Skip data */
                    i += length;
                }

                /* Ensure we got an EOC and not end of data */
                if (tag != ASN_EOC) {
                    ret = ASN_PARSE_E;
                    goto end;
                }

                /* Set the header length to include the length field */
                IndefItems_UpdateHeaderLen(indefItems);
                /* Go to indefinite parent item */
                IndefItems_Up(indefItems);
            }
        }
        else if (tag == ASN_EOC) {
            /* End-of-content must be 0 length */
            if (length != 0) {
                ret = ASN_PARSE_E;
                goto end;
            }
            /* Check there is an item to close - missing EOC */
            if (indefItems->depth == 0) {
                ret = ASN_PARSE_E;
                goto end;
            }

            /* Finish calculation of data length for indefinite item */
            IndefItems_CalcLength(indefItems);
            /* Go to indefinite parent item */
            IndefItems_Up(indefItems);
        }
        else {
            /* Known length item to add in - make sure enough data for it */
            if (i + length > berSz) {
                ret = ASN_PARSE_E;
                goto end;
            }

            /* Include all data - can't have indefinite inside definite */
            i += length;
            /* Add entire item to current indefinite item */
            IndefItems_MoreData(indefItems, i - start);
        }
    }
    /* Check we had a EOC for each indefinite item */
    if (indefItems->depth != 0) {
        ret = ASN_PARSE_E;
        goto end;
    }

    /* Write out DER */

    j = 0;
    /* Reset index */
    indefItems->idx = 0;
    for (i = 0; i < berSz; ) {
        word32 start = i;

        /* Get item - checked above */
        (void)GetBerHeader(ber, &i, berSz, &tag, &length, &indef);
        if (indef) {
            if (der != NULL) {
                /* Check enough space for header */
                if (j + (word32)IndefItems_HeaderLen(indefItems) > *derSz) {
                    ret = BUFFER_E;
                    goto end;
                }

                if ((tag & 0xC0) == 0 &&
                    tag != (ASN_SEQUENCE | ASN_CONSTRUCTED) &&
                    tag != (ASN_SET      | ASN_CONSTRUCTED)) {
                    /* Remove constructed tag for basic types */
                    tag &= (byte)~ASN_CONSTRUCTED;
                }
                /* Add tag and length */
                der[j] = tag;
                (void)SetLength(IndefItems_Len(indefItems), der + j + 1);
            }
            /* Add header length of indefinite item */
            j += (word32)IndefItems_HeaderLen(indefItems);

            if ((tag & 0xC0) == 0 &&
                tag != (ASN_SEQUENCE | ASN_CONSTRUCTED) &&
                tag != (ASN_SET      | ASN_CONSTRUCTED)) {
                /* For basic type - get each child item and add data */
                for (; i < berSz; ) {
                    (void)GetBerHeader(ber, &i, berSz, &tag, &length, &indef);
                    if (tag == ASN_EOC) {
                        break;
                    }
                    if (der != NULL) {
                        if (j + length > *derSz) {
                            ret = BUFFER_E;
                            goto end;
                        }
                        XMEMCPY(der + j, ber + i, length);
                    }
                    j += length;
                    i += length;
                }
            }

            /* Move to next indef item in list */
            indefItems->idx++;
        }
        else if (tag == ASN_EOC) {
            /* End-Of-Content is not written out in DER */
        }
        else {
            /* Write out definite length item as is. */
            i += length;
            if (der != NULL) {
                /* Ensure space for item */
                if (j + i - start > *derSz) {
                    ret = BUFFER_E;
                    goto end;
                }
                /* Copy item as is */
                XMEMCPY(der + j, ber + start, i - start);
            }
            j += i - start;
        }
    }

    /* Return the length of the DER encoded ASN.1 */
    *derSz = j;
    if (der == NULL) {
        ret = WC_NO_ERR_TRACE(LENGTH_ONLY_E);
    }
end:
    WC_FREE_VAR_EX(indefItems, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    return ret;
}
#endif

#ifndef WOLFSSL_ASN_TEMPLATE
#if defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_CERT_GEN)
/* Set the DER/BER encoding of the ASN.1 BIT_STRING with a 16-bit value.
 *
 * val         16-bit value to encode.
 * output      Buffer to write into.
 * returns the number of bytes added to the buffer.
 */
static word32 SetBitString16Bit(word16 val, byte* output)
{
    word32 idx;
    int    len;
    byte   lastByte;
    byte   unusedBits = 0;

    if ((val >> 8) != 0) {
        len = 2;
        lastByte = (byte)(val >> 8);
    }
    else {
        len = 1;
        lastByte = (byte)val;
    }

    while (((lastByte >> unusedBits) & 0x01) == 0x00)
        unusedBits++;

    idx = SetBitString((word32)len, unusedBits, output);
    output[idx++] = (byte)val;
    if (len > 1)
        output[idx++] = (byte)(val >> 8);

    return idx;
}
#endif /* WOLFSSL_CERT_EXT || WOLFSSL_CERT_GEN */
#endif /* !WOLFSSL_ASN_TEMPLATE */

/* hashType */
#ifdef WOLFSSL_MD2
    static const byte hashMd2hOid[] = {42, 134, 72, 134, 247, 13, 2, 2};
#endif
#ifndef NO_MD5
    static const byte hashMd5hOid[] = {42, 134, 72, 134, 247, 13, 2, 5};
#endif
#ifndef NO_SHA
    static const byte hashSha1hOid[] = {43, 14, 3, 2, 26};
#endif
#ifdef WOLFSSL_SHA224
    static const byte hashSha224hOid[] = {96, 134, 72, 1, 101, 3, 4, 2, 4};
#endif
#ifndef NO_SHA256
    static const byte hashSha256hOid[] = {96, 134, 72, 1, 101, 3, 4, 2, 1};
#endif
#ifdef WOLFSSL_SHA384
    static const byte hashSha384hOid[] = {96, 134, 72, 1, 101, 3, 4, 2, 2};
#endif
#ifdef WOLFSSL_SHA512
    static const byte hashSha512hOid[] = {96, 134, 72, 1, 101, 3, 4, 2, 3};
    #ifndef WOLFSSL_NOSHA512_224
    static const byte hashSha512_224hOid[] = {96, 134, 72, 1, 101, 3, 4, 2, 5};
    #endif
    #ifndef WOLFSSL_NOSHA512_256
    static const byte hashSha512_256hOid[] = {96, 134, 72, 1, 101, 3, 4, 2, 6};
    #endif
#endif
#ifdef WOLFSSL_SHA3
#ifndef WOLFSSL_NOSHA3_224
    static const byte hashSha3_224hOid[] = {96, 134, 72, 1, 101, 3, 4, 2, 7};
#endif /* WOLFSSL_NOSHA3_224 */
#ifndef WOLFSSL_NOSHA3_256
    static const byte hashSha3_256hOid[] = {96, 134, 72, 1, 101, 3, 4, 2, 8};
#endif /* WOLFSSL_NOSHA3_256 */
#ifndef WOLFSSL_NOSHA3_384
    static const byte hashSha3_384hOid[] = {96, 134, 72, 1, 101, 3, 4, 2, 9};
#endif /* WOLFSSL_NOSHA3_384 */
#ifndef WOLFSSL_NOSHA3_512
    static const byte hashSha3_512hOid[] = {96, 134, 72, 1, 101, 3, 4, 2, 10};
#endif /* WOLFSSL_NOSHA3_512 */
#endif /* WOLFSSL_SHA3 */

/* hmacType */
#ifndef NO_HMAC
    #ifdef WOLFSSL_SHA224
    static const byte hmacSha224Oid[] = {42, 134, 72, 134, 247, 13, 2, 8};
    #endif
    #ifndef NO_SHA256
    static const byte hmacSha256Oid[] = {42, 134, 72, 134, 247, 13, 2, 9};
    #endif
    #ifdef WOLFSSL_SHA384
    static const byte hmacSha384Oid[] = {42, 134, 72, 134, 247, 13, 2, 10};
    #endif
    #ifdef WOLFSSL_SHA512
    static const byte hmacSha512Oid[] = {42, 134, 72, 134, 247, 13, 2, 11};
    #endif
#endif

/* sigType */
#if !defined(NO_DSA) && !defined(NO_SHA)
    static const byte sigSha1wDsaOid[] = {42, 134, 72, 206, 56, 4, 3};
    static const byte sigSha256wDsaOid[] = {96, 134, 72, 1, 101, 3, 4, 3, 2};
#endif /* NO_DSA */
#ifndef NO_RSA
    #ifdef WOLFSSL_MD2
    static const byte sigMd2wRsaOid[] = {42, 134, 72, 134, 247, 13, 1, 1, 2};
    #endif
    #ifndef NO_MD5
    static const byte sigMd5wRsaOid[] = {42, 134, 72, 134, 247, 13, 1, 1, 4};
    #endif
    #ifndef NO_SHA
    static const byte sigSha1wRsaOid[] = {42, 134, 72, 134, 247, 13, 1, 1, 5};
    #endif
    #ifdef WOLFSSL_SHA224
    static const byte sigSha224wRsaOid[] = {42, 134, 72, 134, 247, 13, 1, 1,14};
    #endif
    #ifndef NO_SHA256
    static const byte sigSha256wRsaOid[] = {42, 134, 72, 134, 247, 13, 1, 1,11};
    #endif
    #ifdef WOLFSSL_SHA384
    static const byte sigSha384wRsaOid[] = {42, 134, 72, 134, 247, 13, 1, 1,12};
    #endif
    #ifdef WOLFSSL_SHA512
    static const byte sigSha512wRsaOid[] = {42, 134, 72, 134, 247, 13, 1, 1,13};
    #endif
    #ifdef WOLFSSL_SHA3
    #ifndef WOLFSSL_NOSHA3_224
    static const byte sigSha3_224wRsaOid[] = {96, 134, 72, 1, 101, 3, 4, 3, 13};
    #endif
    #ifndef WOLFSSL_NOSHA3_256
    static const byte sigSha3_256wRsaOid[] = {96, 134, 72, 1, 101, 3, 4, 3, 14};
    #endif
    #ifndef WOLFSSL_NOSHA3_384
    static const byte sigSha3_384wRsaOid[] = {96, 134, 72, 1, 101, 3, 4, 3, 15};
    #endif
    #ifndef WOLFSSL_NOSHA3_512
    static const byte sigSha3_512wRsaOid[] = {96, 134, 72, 1, 101, 3, 4, 3, 16};
    #endif
    #endif
    #ifdef WC_RSA_PSS
    static const byte sigRsaSsaPssOid[] = {42, 134, 72, 134, 247, 13, 1, 1, 10};
    #endif
#endif /* NO_RSA */
#ifdef HAVE_ECC
    #ifndef NO_SHA
    static const byte sigSha1wEcdsaOid[] = {42, 134, 72, 206, 61, 4, 1};
    #endif
    #ifdef WOLFSSL_SHA224
    static const byte sigSha224wEcdsaOid[] = {42, 134, 72, 206, 61, 4, 3, 1};
    #endif
    #ifndef NO_SHA256
    static const byte sigSha256wEcdsaOid[] = {42, 134, 72, 206, 61, 4, 3, 2};
    #endif
    #ifdef WOLFSSL_SHA384
    static const byte sigSha384wEcdsaOid[] = {42, 134, 72, 206, 61, 4, 3, 3};
    #endif
    #ifdef WOLFSSL_SHA512
    static const byte sigSha512wEcdsaOid[] = {42, 134, 72, 206, 61, 4, 3, 4};
    #endif
    #ifdef WOLFSSL_SHA3
    #ifndef WOLFSSL_NOSHA3_224
    static const byte sigSha3_224wEcdsaOid[] = {96, 134, 72, 1, 101, 3, 4, 3, 9};
    #endif
    #ifndef WOLFSSL_NOSHA3_256
    static const byte sigSha3_256wEcdsaOid[] = {96, 134, 72, 1, 101, 3, 4, 3, 10};
    #endif
    #ifndef WOLFSSL_NOSHA3_384
    static const byte sigSha3_384wEcdsaOid[] = {96, 134, 72, 1, 101, 3, 4, 3, 11};
    #endif
    #ifndef WOLFSSL_NOSHA3_512
    static const byte sigSha3_512wEcdsaOid[] = {96, 134, 72, 1, 101, 3, 4, 3, 12};
    #endif
    #endif
    #if defined(WOLFSSL_SM2) && defined(WOLFSSL_SM3)
    /* 0x2A, 0x81, 0x1C, 0xCF, 0x55, 0x01, 0x83, 0x75 */
    static const byte sigSm3wSm2Oid[] = {42, 129, 28, 207, 85, 1, 131, 117};
    #endif
#endif /* HAVE_ECC */
#ifdef HAVE_ED25519
    static const byte sigEd25519Oid[] = {43, 101, 112};
#endif /* HAVE_ED25519 */
#ifdef HAVE_ED448
    static const byte sigEd448Oid[] = {43, 101, 113};
#endif /* HAVE_ED448 */
#ifdef HAVE_FALCON
    /* Falcon Level 1: 1 3 9999 3 6 */
    static const byte sigFalcon_Level1Oid[] = {43, 206, 15, 3, 6};

    /* Falcon Level 5: 1 3 9999 3 9 */
    static const byte sigFalcon_Level5Oid[] = {43, 206, 15, 3, 9};
#endif /* HAVE_FACON */
#ifdef HAVE_DILITHIUM
#ifdef WOLFSSL_DILITHIUM_FIPS204_DRAFT
    /* Dilithium Level 2: 1.3.6.1.4.1.2.267.12.4.4 */
    static const byte sigDilithium_Level2Oid[] =
        {43, 6, 1, 4, 1, 2, 130, 11, 12, 4, 4};

    /* Dilithium Level 3: 1.3.6.1.4.1.2.267.12.6.5 */
    static const byte sigDilithium_Level3Oid[] =
        {43, 6, 1, 4, 1, 2, 130, 11, 12, 6, 5};

    /* Dilithium Level 5: 1.3.6.1.4.1.2.267.12.8.7 */
    static const byte sigDilithium_Level5Oid[] =
        {43, 6, 1, 4, 1, 2, 130, 11, 12, 8, 7};
#endif /* WOLFSSL_DILITHIUM_FIPS204_DRAFT */

    /* ML-DSA Level 2: 2.16.840.1.101.3.4.3.17 */
    static const byte sigMlDsa_Level2Oid[] =
        {96, 134, 72, 1, 101, 3, 4, 3, 17};

    /* ML-DSA Level 3: 2.16.840.1.101.3.4.3.18 */
    static const byte sigMlDsa_Level3Oid[] =
        {96, 134, 72, 1, 101, 3, 4, 3, 18};

    /* ML-DSA Level 5: 2.16.840.1.101.3.4.3.19 */
    static const byte sigMlDsa_Level5Oid[] =
        {96, 134, 72, 1, 101, 3, 4, 3, 19};
#endif /* HAVE_DILITHIUM */
#ifdef HAVE_SPHINCS
    /* Sphincs Fast Level 1: 1 3 9999 6 7 4 */
    static const byte sigSphincsFast_Level1Oid[] =
        {43, 206, 15, 6, 7, 4};

    /* Sphincs Fast Level 3: 1 3 9999 6 8 3 */
    static const byte sigSphincsFast_Level3Oid[] =
        {43, 206, 15, 6, 8, 3};

    /* Sphincs Fast Level 5: 1 3 9999 6 9 3 */
    static const byte sigSphincsFast_Level5Oid[] =
        {43, 206, 15, 6, 9, 3};

    /* Sphincs Small Level 1: 1 3 9999 6 7 10 */
    static const byte sigSphincsSmall_Level1Oid[] =
        {43, 206, 15, 6, 7, 10};

    /* Sphincs Small Level 3: 1 3 9999 6 8 7 */
    static const byte sigSphincsSmall_Level3Oid[] =
        {43, 206, 15, 6, 8, 7};

    /* Sphincs Small Level 5: 1 3 9999 6 9 7 */
    static const byte sigSphincsSmall_Level5Oid[] =
        {43, 206, 15, 6, 9, 7};
#endif /* HAVE_SPHINCS */

/* keyType */
#ifndef NO_DSA
    static const byte keyDsaOid[] = {42, 134, 72, 206, 56, 4, 1};
#endif /* NO_DSA */
#ifndef NO_RSA
    static const byte keyRsaOid[] = {42, 134, 72, 134, 247, 13, 1, 1, 1};
#ifdef WC_RSA_PSS
    static const byte keyRsaPssOid[] = {42, 134, 72, 134, 247, 13, 1, 1, 10};
#endif
#endif /* NO_RSA */
#ifdef HAVE_ECC
    static const byte keyEcdsaOid[] = {42, 134, 72, 206, 61, 2, 1};
#endif /* HAVE_ECC */
#ifdef HAVE_ED25519
    static const byte keyEd25519Oid[] = {43, 101, 112};
#endif /* HAVE_ED25519 */
#ifdef HAVE_CURVE25519
    static const byte keyCurve25519Oid[] = {43, 101, 110};
#endif
#ifdef HAVE_ED448
    static const byte keyEd448Oid[] = {43, 101, 113};
#endif /* HAVE_ED448 */
#ifdef HAVE_CURVE448
    static const byte keyCurve448Oid[] = {43, 101, 111};
#endif /* HAVE_CURVE448 */
#ifndef NO_DH
    static const byte keyDhOid[] = {42, 134, 72, 134, 247, 13, 1, 3, 1};
#endif /* !NO_DH */
#ifdef HAVE_FALCON
    /* Falcon Level 1: 1 3 9999 3 6 */
    static const byte keyFalcon_Level1Oid[] = {43, 206, 15, 3, 6};

    /* Falcon Level 5: 1 3 9999 3 9 */
    static const byte keyFalcon_Level5Oid[] = {43, 206, 15, 3, 9};
#endif /* HAVE_FALCON */
#ifdef HAVE_DILITHIUM
#ifdef WOLFSSL_DILITHIUM_FIPS204_DRAFT
    /* Dilithium Level 2: 1.3.6.1.4.1.2.267.12.4.4 */
    static const byte keyDilithium_Level2Oid[] =
        {43, 6, 1, 4, 1, 2, 130, 11, 12, 4, 4};

    /* Dilithium Level 3: 1.3.6.1.4.1.2.267.12.6.5 */
    static const byte keyDilithium_Level3Oid[] =
        {43, 6, 1, 4, 1, 2, 130, 11, 12, 6, 5};

    /* Dilithium Level 5: 1.3.6.1.4.1.2.267.12.8.7 */
    static const byte keyDilithium_Level5Oid[] =
        {43, 6, 1, 4, 1, 2, 130, 11, 12, 8, 7};
#endif

    /* ML-DSA Level 2: 2.16.840.1.101.3.4.3.17 */
    static const byte keyMlDsa_Level2Oid[] =
        {96, 134, 72, 1, 101, 3, 4, 3, 17};

    /* ML-DSA Level 3: 2.16.840.1.101.3.4.3.18 */
    static const byte keyMlDsa_Level3Oid[] =
        {96, 134, 72, 1, 101, 3, 4, 3, 18};

    /* ML-DSA Level 5: 2.16.840.1.101.3.4.3.19 */
    static const byte keyMlDsa_Level5Oid[] =
        {96, 134, 72, 1, 101, 3, 4, 3, 19};
#endif /* HAVE_DILITHIUM */
#ifdef HAVE_SPHINCS
    /* Sphincs Fast Level 1: 1 3 9999 6 7 4 */
    static const byte keySphincsFast_Level1Oid[] =
        {43, 206, 15, 6, 7, 4};

    /* Sphincs Fast Level 3: 1 3 9999 6 8 3 */
    static const byte keySphincsFast_Level3Oid[] =
        {43, 206, 15, 6, 8, 3};

    /* Sphincs Fast Level 5: 1 3 9999 6 9 3 */
    static const byte keySphincsFast_Level5Oid[] =
        {43, 206, 15, 6, 9, 3};

    /* Sphincs Small Level 1: 1 3 9999 6 7 10 */
    static const byte keySphincsSmall_Level1Oid[] =
        {43, 206, 15, 6, 7, 10};

    /* Sphincs Small Level 3: 1 3 9999 6 8 7 */
    static const byte keySphincsSmall_Level3Oid[] =
        {43, 206, 15, 6, 8, 7};

    /* Sphincs Small Level 5: 1 3 9999 6 9 7 */
    static const byte keySphincsSmall_Level5Oid[] =
        {43, 206, 15, 6, 9, 7};
#endif /* HAVE_SPHINCS */

/* curveType */
#ifdef HAVE_ECC
    /* See "ecc_sets" table in ecc.c */
#endif /* HAVE_ECC */

#ifdef HAVE_AES_CBC
/* blkType */
    #ifdef WOLFSSL_AES_128
    static const byte blkAes128CbcOid[] = {96, 134, 72, 1, 101, 3, 4, 1, 2};
    #endif
    #ifdef WOLFSSL_AES_192
    static const byte blkAes192CbcOid[] = {96, 134, 72, 1, 101, 3, 4, 1, 22};
    #endif
    #ifdef WOLFSSL_AES_256
    static const byte blkAes256CbcOid[] = {96, 134, 72, 1, 101, 3, 4, 1, 42};
    #endif
#endif /* HAVE_AES_CBC */
#ifdef HAVE_AESGCM
    #ifdef WOLFSSL_AES_128
    static const byte blkAes128GcmOid[] = {96, 134, 72, 1, 101, 3, 4, 1, 6};
    #endif
    #ifdef WOLFSSL_AES_192
    static const byte blkAes192GcmOid[] = {96, 134, 72, 1, 101, 3, 4, 1, 26};
    #endif
    #ifdef WOLFSSL_AES_256
    static const byte blkAes256GcmOid[] = {96, 134, 72, 1, 101, 3, 4, 1, 46};
    #endif
#endif /* HAVE_AESGCM */
#ifdef HAVE_AESCCM
    #ifdef WOLFSSL_AES_128
    static const byte blkAes128CcmOid[] = {96, 134, 72, 1, 101, 3, 4, 1, 7};
    #endif
    #ifdef WOLFSSL_AES_192
    static const byte blkAes192CcmOid[] = {96, 134, 72, 1, 101, 3, 4, 1, 27};
    #endif
    #ifdef WOLFSSL_AES_256
    static const byte blkAes256CcmOid[] = {96, 134, 72, 1, 101, 3, 4, 1, 47};
    #endif
#endif /* HAVE_AESCCM */

#ifndef NO_DES3
    static const byte blkDesCbcOid[]  = {43, 14, 3, 2, 7};
    static const byte blkDes3CbcOid[] = {42, 134, 72, 134, 247, 13, 3, 7};
#endif

/* keyWrapType */
#ifdef WOLFSSL_AES_128
    static const byte wrapAes128Oid[] = {96, 134, 72, 1, 101, 3, 4, 1, 5};
#endif
#ifdef WOLFSSL_AES_192
    static const byte wrapAes192Oid[] = {96, 134, 72, 1, 101, 3, 4, 1, 25};
#endif
#ifdef WOLFSSL_AES_256
    static const byte wrapAes256Oid[] = {96, 134, 72, 1, 101, 3, 4, 1, 45};
#endif
#ifdef HAVE_PKCS7
/* From RFC 3211 */
static const byte wrapPwriKekOid[] = {42, 134, 72, 134, 247, 13, 1, 9, 16, 3,9};
#endif

/* cmsKeyAgreeType */
#ifndef NO_SHA
    static const byte dhSinglePass_stdDH_sha1kdf_Oid[]   =
                                          {43, 129, 5, 16, 134, 72, 63, 0, 2};
#endif
#ifdef WOLFSSL_SHA224
    static const byte dhSinglePass_stdDH_sha224kdf_Oid[] = {43, 129, 4, 1, 11, 0};
#endif
#ifndef NO_SHA256
    static const byte dhSinglePass_stdDH_sha256kdf_Oid[] = {43, 129, 4, 1, 11, 1};
#endif
#ifdef WOLFSSL_SHA384
    static const byte dhSinglePass_stdDH_sha384kdf_Oid[] = {43, 129, 4, 1, 11, 2};
#endif
#ifdef WOLFSSL_SHA512
    static const byte dhSinglePass_stdDH_sha512kdf_Oid[] = {43, 129, 4, 1, 11, 3};
#endif

/* ocspType */
#ifdef HAVE_OCSP
    static const byte ocspBasicOid[]    = {43, 6, 1, 5, 5, 7, 48, 1, 1};
    static const byte ocspNonceOid[]    = {43, 6, 1, 5, 5, 7, 48, 1, 2};
    static const byte ocspNoCheckOid[]  = {43, 6, 1, 5, 5, 7, 48, 1, 5};
#endif /* HAVE_OCSP */

/* certExtType */
static const byte extBasicCaOid[] = {85, 29, 19};
static const byte extAltNamesOid[] = {85, 29, 17};
static const byte extCrlDistOid[] = {85, 29, 31};
static const byte extAuthInfoOid[] = {43, 6, 1, 5, 5, 7, 1, 1};
static const byte extAuthKeyOid[] = {85, 29, 35};
static const byte extSubjKeyOid[] = {85, 29, 14};
static const byte extCertPolicyOid[] = {85, 29, 32};
static const byte extKeyUsageOid[] = {85, 29, 15};
static const byte extInhibitAnyOid[] = {85, 29, 54};
static const byte extExtKeyUsageOid[] = {85, 29, 37};
#ifndef IGNORE_NAME_CONSTRAINTS
    static const byte extNameConsOid[] = {85, 29, 30};
#endif
#ifdef HAVE_CRL
static const byte extCrlNumberOid[] = {85, 29, 20};
#endif
#ifdef WOLFSSL_SUBJ_DIR_ATTR
    static const byte extSubjDirAttrOid[] = {85, 29, 9};
#endif
#ifdef WOLFSSL_SUBJ_INFO_ACC
    static const byte extSubjInfoAccessOid[] = {43, 6, 1, 5, 5, 7, 1, 11};
#endif

/* certAuthInfoType */
static const byte extAuthInfoOcspOid[] = {43, 6, 1, 5, 5, 7, 48, 1};
static const byte extAuthInfoCaIssuerOid[] = {43, 6, 1, 5, 5, 7, 48, 2};
#ifdef WOLFSSL_SUBJ_INFO_ACC
    static const byte extAuthInfoCaRespOid[] = {43, 6, 1, 5, 5, 7, 48, 5};
#endif /* WOLFSSL_SUBJ_INFO_ACC */

/* certPolicyType */
static const byte extCertPolicyAnyOid[] = {85, 29, 32, 0};
static const byte extCertPolicyIsrgDomainValid[] =
    {43, 6, 1, 4, 1, 130, 223, 19, 1, 1, 1};
#ifdef WOLFSSL_FPKI
#define CERT_POLICY_TYPE_OID_BASE(num) {96, 134, 72, 1, 101, 3, 2, 1, 3, num}
    static const byte extCertPolicyFpkiHighAssuranceOid[] =
            CERT_POLICY_TYPE_OID_BASE(4);
    static const byte extCertPolicyFpkiCommonHardwareOid[] =
            CERT_POLICY_TYPE_OID_BASE(7);
    static const byte extCertPolicyFpkiMediumHardwareOid[] =
            CERT_POLICY_TYPE_OID_BASE(12);
    static const byte extCertPolicyFpkiCommonAuthOid[] =
            CERT_POLICY_TYPE_OID_BASE(13);
    static const byte extCertPolicyFpkiCommonHighOid[] =
            CERT_POLICY_TYPE_OID_BASE(16);
    static const byte extCertPolicyFpkiCommonDevicesHardwareOid[] =
            CERT_POLICY_TYPE_OID_BASE(36);
    static const byte extCertPolicyFpkiCommonPivContentSigningOid[] =
            CERT_POLICY_TYPE_OID_BASE(39);
    static const byte extCertPolicyFpkiPivAuthOid[] =
            CERT_POLICY_TYPE_OID_BASE(40);
    static const byte extCertPolicyFpkiPivAuthHwOid[] =
            CERT_POLICY_TYPE_OID_BASE(41);
    static const byte extCertPolicyFpkiPiviAuthOid[] =
            CERT_POLICY_TYPE_OID_BASE(45);

    /* Federal PKI Test OIDs - 2.16.840.1.101.3.2.1.48.x */
    #define TEST_CERT_POLICY_TYPE_OID_BASE(num) {96, 134, 72, 1, 101, 3, 2, 1, 48, num}
    static const byte extCertPolicyFpkiAuthTestOid[] =
        TEST_CERT_POLICY_TYPE_OID_BASE(11);
    static const byte extCertPolicyFpkiCardauthTestOid[] =
        TEST_CERT_POLICY_TYPE_OID_BASE(13);
    static const byte extCertPolicyFpkiPivContentTestOid[] =
        TEST_CERT_POLICY_TYPE_OID_BASE(86);
    static const byte extCertPolicyFpkiAuthDerivedTestOid[] =
        TEST_CERT_POLICY_TYPE_OID_BASE(109);
    static const byte extCertPolicyFpkiAuthDerivedHwTestOid[] =
        TEST_CERT_POLICY_TYPE_OID_BASE(110);

    /* DoD PKI OIDs - 2.16.840.1.101.2.1.11.X */
    #define DOD_POLICY_TYPE_OID_BASE(num) {96, 134, 72, 1, 101, 2, 1, 11, num}
    static const byte extCertPolicyDodMediumOid[] =
            DOD_POLICY_TYPE_OID_BASE(5);
    static const byte extCertPolicyDodMediumHardwareOid[] =
            DOD_POLICY_TYPE_OID_BASE(9);
    static const byte extCertPolicyDodPivAuthOid[] =
            DOD_POLICY_TYPE_OID_BASE(10);
    static const byte extCertPolicyDodMediumNpeOid[] =
            DOD_POLICY_TYPE_OID_BASE(17);
    static const byte extCertPolicyDodMedium2048Oid[] =
            DOD_POLICY_TYPE_OID_BASE(18);
    static const byte extCertPolicyDodMediumHardware2048Oid[] =
            DOD_POLICY_TYPE_OID_BASE(19);
    static const byte extCertPolicyDodPivAuth2048Oid[] =
            DOD_POLICY_TYPE_OID_BASE(20);
    static const byte extCertPolicyDodPeerInteropOid[] =
            DOD_POLICY_TYPE_OID_BASE(31);
    static const byte extCertPolicyDodMediumNpe112Oid[] =
            DOD_POLICY_TYPE_OID_BASE(36);
    static const byte extCertPolicyDodMediumNpe128Oid[] =
            DOD_POLICY_TYPE_OID_BASE(37);
    static const byte extCertPolicyDodMediumNpe192Oid[] =
            DOD_POLICY_TYPE_OID_BASE(38);
    static const byte extCertPolicyDodMedium112Oid[] =
            DOD_POLICY_TYPE_OID_BASE(39);
    static const byte extCertPolicyDodMedium128Oid[] =
            DOD_POLICY_TYPE_OID_BASE(40);
    static const byte extCertPolicyDodMedium192Oid[] =
            DOD_POLICY_TYPE_OID_BASE(41);
    static const byte extCertPolicyDodMediumHardware112Oid[] =
            DOD_POLICY_TYPE_OID_BASE(42);
    static const byte extCertPolicyDodMediumHardware128Oid[] =
            DOD_POLICY_TYPE_OID_BASE(43);
    static const byte extCertPolicyDodMediumHardware192Oid[] =
            DOD_POLICY_TYPE_OID_BASE(44);
    static const byte extCertPolicyDodAdminOid[] =
            DOD_POLICY_TYPE_OID_BASE(59);
    static const byte extCertPolicyDodInternalNpe112Oid[] =
            DOD_POLICY_TYPE_OID_BASE(60);
    static const byte extCertPolicyDodInternalNpe128Oid[] =
            DOD_POLICY_TYPE_OID_BASE(61);
    static const byte extCertPolicyDodInternalNpe192Oid[] =
            DOD_POLICY_TYPE_OID_BASE(62);

    /* ECA PKI OIDs - 2.16.840.1.101.3.2.1.12.X */
    #define ECA_POLICY_TYPE_OID_BASE(num) {96, 134, 72, 1, 101, 3, 2, 1, 12, num}
    static const byte extCertPolicyEcaMediumOid[] =
            ECA_POLICY_TYPE_OID_BASE(1);
    static const byte extCertPolicyEcaMediumHardwareOid[] =
            ECA_POLICY_TYPE_OID_BASE(2);
    static const byte extCertPolicyEcaMediumTokenOid[] =
            ECA_POLICY_TYPE_OID_BASE(3);
    static const byte extCertPolicyEcaMediumSha256Oid[] =
            ECA_POLICY_TYPE_OID_BASE(4);
    static const byte extCertPolicyEcaMediumTokenSha256Oid[] =
            ECA_POLICY_TYPE_OID_BASE(5);
    static const byte extCertPolicyEcaMediumHardwarePiviOid[] =
            ECA_POLICY_TYPE_OID_BASE(6);
    static const byte extCertPolicyEcaContentSigningPiviOid[] =
            ECA_POLICY_TYPE_OID_BASE(8);
    static const byte extCertPolicyEcaMediumDeviceSha256Oid[] =
            ECA_POLICY_TYPE_OID_BASE(9);
    static const byte extCertPolicyEcaMediumHardwareSha256Oid[] =
            ECA_POLICY_TYPE_OID_BASE(10);

    /* Department of State PKI OIDs - 2.16.840.1.101.3.2.1.6.X */
    #define STATE_POLICY_TYPE_OID_BASE(num) {96, 134, 72, 1, 101, 3, 2, 1, 6, num}
    static const byte extCertPolicyStateBasicOid[] =
            STATE_POLICY_TYPE_OID_BASE(1);
    static const byte extCertPolicyStateLowOid[] =
            STATE_POLICY_TYPE_OID_BASE(2);
    static const byte extCertPolicyStateModerateOid[] =
            STATE_POLICY_TYPE_OID_BASE(3);
    static const byte extCertPolicyStateHighOid[] =
            STATE_POLICY_TYPE_OID_BASE(4);
    static const byte extCertPolicyStateMedHwOid[] =
            STATE_POLICY_TYPE_OID_BASE(12);
    static const byte extCertPolicyStateMediumDeviceHardwareOid[] =
            STATE_POLICY_TYPE_OID_BASE(38);

    /* U.S. Treasury SSP PKI OIDs - 2.16.840.1.101.3.2.1.5.X */
    #define TREASURY_POLICY_TYPE_OID_BASE(num) {96, 134, 72, 1, 101, 3, 2, 1, 5, num}
    static const byte extCertPolicyTreasuryMediumHardwareOid[] =
            TREASURY_POLICY_TYPE_OID_BASE(4);
    static const byte extCertPolicyTreasuryHighOid[] =
            TREASURY_POLICY_TYPE_OID_BASE(5);
    static const byte extCertPolicyTreasuryPiviHardwareOid[] =
            TREASURY_POLICY_TYPE_OID_BASE(10);
    static const byte extCertPolicyTreasuryPiviContentSigningOid[] =
            TREASURY_POLICY_TYPE_OID_BASE(12);

    /* Boeing PKI OIDs - 1.3.6.1.4.1.73.15.3.1.X */
    #define BOEING_POLICY_TYPE_OID_BASE(num) {43, 6, 1, 4, 1, 73, 15, 3, 1, num}
    static const byte extCertPolicyBoeingMediumHardwareSha256Oid[] =
            BOEING_POLICY_TYPE_OID_BASE(12);
    static const byte extCertPolicyBoeingMediumHardwareContentSigningSha256Oid[] =
            BOEING_POLICY_TYPE_OID_BASE(17);

    /* Carillon Federal Services OIDs - 1.3.6.1.4.1.45606.3.1.X */
    #define CARILLON_POLICY_TYPE_OID_BASE(num) {43, 6, 1, 4, 1, 130, 228, 38, 3, 1, num}
    static const byte extCertPolicyCarillonMediumhw256Oid[] =
            CARILLON_POLICY_TYPE_OID_BASE(12);
    static const byte extCertPolicyCarillonAivhwOid[] =
            CARILLON_POLICY_TYPE_OID_BASE(20);
    static const byte extCertPolicyCarillonAivcontentOid[] =
            CARILLON_POLICY_TYPE_OID_BASE(22);

    /* Carillon Information Security OIDs - 1.3.6.1.4.1.25054.3.1.X */
    #define CIS_POLICY_TYPE_OID_BASE(num) {43, 6, 1, 4, 1, 129, 195, 94, 3, 1, num}
    static const byte extCertPolicyCisMediumhw256Oid[] =
            CIS_POLICY_TYPE_OID_BASE(12);
    static const byte extCertPolicyCisMeddevhw256Oid[] =
            CIS_POLICY_TYPE_OID_BASE(14);
    static const byte extCertPolicyCisIcecapHwOid[] =
            CIS_POLICY_TYPE_OID_BASE(20);
    static const byte extCertPolicyCisIcecapContentOid[] =
            CIS_POLICY_TYPE_OID_BASE(22);

    /* CertiPath Bridge OIDs - 1.3.6.1.4.1.24019.1.1.1.X */
    #define CERTIPATH_POLICY_TYPE_OID_BASE(num) {43, 6, 1, 4, 1, 129, 187, 83, 1, 1, 1, num}
    static const byte extCertPolicyCertipathMediumhwOid[] =
            CERTIPATH_POLICY_TYPE_OID_BASE(2);
    static const byte extCertPolicyCertipathHighhwOid[] =
            CERTIPATH_POLICY_TYPE_OID_BASE(3);
    static const byte extCertPolicyCertipathIcecapHwOid[] =
            CERTIPATH_POLICY_TYPE_OID_BASE(7);
    static const byte extCertPolicyCertipathIcecapContentOid[] =
            CERTIPATH_POLICY_TYPE_OID_BASE(9);
    static const byte extCertPolicyCertipathVarMediumhwOid[] =
            CERTIPATH_POLICY_TYPE_OID_BASE(18);
    static const byte extCertPolicyCertipathVarHighhwOid[] =
            CERTIPATH_POLICY_TYPE_OID_BASE(19);

    /* TSCP Bridge OIDs - 1.3.6.1.4.1.38099.1.1.1.X */
    #define TSCP_POLICY_TYPE_OID_BASE(num) {43, 6, 1, 4, 1, 130, 169, 83, 1, 1, 1, num}
    static const byte extCertPolicyTscpMediumhwOid[] =
            TSCP_POLICY_TYPE_OID_BASE(2);
    static const byte extCertPolicyTscpPiviOid[] =
            TSCP_POLICY_TYPE_OID_BASE(5);
    static const byte extCertPolicyTscpPiviContentOid[] =
            TSCP_POLICY_TYPE_OID_BASE(7);

    /* DigiCert NFI PKI OIDs - 2.16.840.1.113733.1.7.23.3.1.X */
    #define DIGICERT_NFI_POLICY_TYPE_OID_BASE(num) {96, 134, 72, 1, 134, 248, 69, 1, 7, 23, 3, 1, num}
    static const byte extCertPolicyDigicertNfiMediumHardwareOid[] =
            DIGICERT_NFI_POLICY_TYPE_OID_BASE(7);
    static const byte extCertPolicyDigicertNfiAuthOid[] =
            DIGICERT_NFI_POLICY_TYPE_OID_BASE(13);
    static const byte extCertPolicyDigicertNfiPiviHardwareOid[] =
            DIGICERT_NFI_POLICY_TYPE_OID_BASE(18);
    static const byte extCertPolicyDigicertNfiPiviContentSigningOid[] =
            DIGICERT_NFI_POLICY_TYPE_OID_BASE(20);
    static const byte extCertPolicyDigicertNfiMediumDevicesHardwareOid[] =
            DIGICERT_NFI_POLICY_TYPE_OID_BASE(36);

    /* Entrust Managed Services NFI PKI OIDs - 2.16.840.1.114027.200.3.10.7.X */
    #define ENTRUST_NFI_POLICY_TYPE_OID_BASE(num) {96, 134, 72, 1, 134, 250, 107, 129, 72, 3, 10, 7, num}
    static const byte extCertPolicyEntrustNfiMediumHardwareOid[] =
            ENTRUST_NFI_POLICY_TYPE_OID_BASE(2);
    static const byte extCertPolicyEntrustNfiMediumAuthenticationOid[] =
            ENTRUST_NFI_POLICY_TYPE_OID_BASE(4);
    static const byte extCertPolicyEntrustNfiPiviHardwareOid[] =
            ENTRUST_NFI_POLICY_TYPE_OID_BASE(6);
    static const byte extCertPolicyEntrustNfiPiviContentSigningOid[] =
            ENTRUST_NFI_POLICY_TYPE_OID_BASE(9);
    static const byte extCertPolicyEntrustNfiMediumDevicesHwOid[] =
            ENTRUST_NFI_POLICY_TYPE_OID_BASE(16);

    /* Exostar LLC PKI OIDs - 1.3.6.1.4.1.13948.1.1.1.X */
    #define EXOSTAR_POLICY_TYPE_OID_BASE(num) {43, 6, 1, 4, 1, 236, 124, 1, 1, 1, num}
    static const byte extCertPolicyExostarMediumHardwareSha2Oid[] =
            EXOSTAR_POLICY_TYPE_OID_BASE(6);

    /* IdenTrust NFI OIDs - 2.16.840.1.113839.0.100.X.Y */
    #define IDENTRUST_POLICY_TYPE_OID_BASE(num1, num2) {96, 134, 72, 1, 134, 249, 47, 0, 100, num1, num2}
    static const byte extCertPolicyIdentrustMediumhwSignOid[] =
            IDENTRUST_POLICY_TYPE_OID_BASE(12, 1);
    static const byte extCertPolicyIdentrustMediumhwEncOid[] =
            IDENTRUST_POLICY_TYPE_OID_BASE(12, 2);
    static const byte extCertPolicyIdentrustPiviHwIdOid[] =
            IDENTRUST_POLICY_TYPE_OID_BASE(18, 0);
    static const byte extCertPolicyIdentrustPiviHwSignOid[] =
            IDENTRUST_POLICY_TYPE_OID_BASE(18, 1);
    static const byte extCertPolicyIdentrustPiviHwEncOid[] =
            IDENTRUST_POLICY_TYPE_OID_BASE(18, 2);
    static const byte extCertPolicyIdentrustPiviContentOid[] =
            IDENTRUST_POLICY_TYPE_OID_BASE(20, 1);

    /* Lockheed Martin PKI OIDs - 1.3.6.1.4.1.103.100.1.1.3.X */
    #define LOCKHEED_POLICY_TYPE_OID_BASE(num) {43, 6, 1, 4, 1, 103, 100, 1, 1, 3, num}
    static const byte extCertPolicyLockheedMediumAssuranceHardwareOid[] =
            LOCKHEED_POLICY_TYPE_OID_BASE(3);

    /* Northrop Grumman PKI OIDs - 1.3.6.1.4.1.16334.509.2.X */
    #define NORTHROP_POLICY_TYPE_OID_BASE(num) {43, 6, 1, 4, 1, 255, 78, 131, 125, 2, num}
    static const byte extCertPolicyNorthropMediumAssurance256HardwareTokenOid[] =
            NORTHROP_POLICY_TYPE_OID_BASE(8);
    static const byte extCertPolicyNorthropPiviAssurance256HardwareTokenOid[] =
            NORTHROP_POLICY_TYPE_OID_BASE(9);
    static const byte extCertPolicyNorthropPiviAssurance256ContentSigningOid[] =
            NORTHROP_POLICY_TYPE_OID_BASE(11);
    static const byte extCertPolicyNorthropMediumAssurance384HardwareTokenOid[] =
            NORTHROP_POLICY_TYPE_OID_BASE(14);

    /* Raytheon PKI OIDs - 1.3.6.1.4.1.1569.10.1.X and 1.3.6.1.4.1.26769.10.1.X */
    #define RAYTHEON_POLICY_TYPE_OID_BASE(num) {43, 6, 1, 4, 1, 140, 33, 10, 1, num}
    static const byte extCertPolicyRaytheonMediumHardwareOid[] =
            RAYTHEON_POLICY_TYPE_OID_BASE(12);
    static const byte extCertPolicyRaytheonMediumDeviceHardwareOid[] =
            RAYTHEON_POLICY_TYPE_OID_BASE(18);

    #define RAYTHEON_SHA2_POLICY_TYPE_OID_BASE(num) {43, 6, 1, 4, 1, 129, 209, 17, 10, 1, num}
    static const byte extCertPolicyRaytheonSha2MediumHardwareOid[] =
            RAYTHEON_SHA2_POLICY_TYPE_OID_BASE(12);
    static const byte extCertPolicyRaytheonSha2MediumDeviceHardwareOid[] =
            RAYTHEON_SHA2_POLICY_TYPE_OID_BASE(18);

    /* WidePoint NFI PKI OIDs - 1.3.6.1.4.1.3922.1.1.1.X */
    #define WIDEPOINT_NFI_POLICY_TYPE_OID_BASE(num) {43, 6, 1, 4, 1, 158, 82, 1, 1, 1, num}
    static const byte extCertPolicyWidepointNfiMediumHardwareOid[] =
            WIDEPOINT_NFI_POLICY_TYPE_OID_BASE(12);
    static const byte extCertPolicyWidepointNfiPiviHardwareOid[] =
            WIDEPOINT_NFI_POLICY_TYPE_OID_BASE(18);
    static const byte extCertPolicyWidepointNfiPiviContentSigningOid[] =
            WIDEPOINT_NFI_POLICY_TYPE_OID_BASE(20);
    static const byte extCertPolicyWidepointNfiMediumDevicesHardwareOid[] =
            WIDEPOINT_NFI_POLICY_TYPE_OID_BASE(38);

    /* Australian Defence Organisation PKI OIDs - 1.2.36.1.334.1.2.X.X */
    #define ADO_POLICY_TYPE_OID_BASE(type, num) {42, 36, 1, 130, 78, 1, 2, type, num}
    static const byte extCertPolicyAdoIndividualMediumAssuranceOid[] =
            ADO_POLICY_TYPE_OID_BASE(1, 2);
    static const byte extCertPolicyAdoIndividualHighAssuranceOid[] =
            ADO_POLICY_TYPE_OID_BASE(1, 3);
    static const byte extCertPolicyAdoResourceMediumAssuranceOid[] =
            ADO_POLICY_TYPE_OID_BASE(2, 2);

    /* Comodo Ltd PKI OID  1.3.6.1.4.1.6449.1.2.1.3.4 */
    #define COMODO_POLICY_TYPE_OID_BASE(num) {43, 6, 1, 4, 1, 178, 49, 1, 2, 1, 3, num}
    static const byte extCertPolicyComodoLtdOid[] =
            COMODO_POLICY_TYPE_OID_BASE(4);

    /* Netherlands Ministry of Defence PKI OIDs - 2.16.528.1.1003.1.2.5.X */
    #define NL_MOD_POLICY_TYPE_OID_BASE(num) {96, 132, 16, 1, 135, 107, 1, 2, 5, num}
    static const byte extCertPolicyNlModAuthenticityOid[] =
            NL_MOD_POLICY_TYPE_OID_BASE(1);
    static const byte extCertPolicyNlModIrrefutabilityOid[] =
            NL_MOD_POLICY_TYPE_OID_BASE(2);
    static const byte extCertPolicyNlModConfidentialityOid[] =
            NL_MOD_POLICY_TYPE_OID_BASE(3);
#endif /* WOLFSSL_FPKI */

/* certAltNameType */
static const byte extAltNamesHwNameOid[] = {43, 6, 1, 5, 5, 7, 8, 4};

/* certKeyUseType */
static const byte extExtKeyUsageAnyOid[] = {85, 29, 37, 0};
static const byte extExtKeyUsageServerAuthOid[]   = {43, 6, 1, 5, 5, 7, 3, 1};
static const byte extExtKeyUsageClientAuthOid[]   = {43, 6, 1, 5, 5, 7, 3, 2};
static const byte extExtKeyUsageCodeSigningOid[]  = {43, 6, 1, 5, 5, 7, 3, 3};
static const byte extExtKeyUsageEmailProtectOid[] = {43, 6, 1, 5, 5, 7, 3, 4};
static const byte extExtKeyUsageTimestampOid[]    = {43, 6, 1, 5, 5, 7, 3, 8};
static const byte extExtKeyUsageOcspSignOid[]     = {43, 6, 1, 5, 5, 7, 3, 9};
#ifdef WOLFSSL_WOLFSSH
#define EXT_KEY_USAGE_OID_BASE(num) {43, 6, 1, 5, 5, 7, 3, num}
    static const byte extExtKeyUsageSshClientAuthOid[] =
            EXT_KEY_USAGE_OID_BASE(21);
    static const byte extExtKeyUsageSshMSCLOid[] =
            {43, 6, 1, 4, 1, 130, 55, 20, 2, 2};
    static const byte extExtKeyUsageSshKpClientAuthOid[] =
            {43, 6, 1, 5, 2, 3, 4};
#endif /* WOLFSSL_WOLFSSH */

#ifdef WOLFSSL_SUBJ_DIR_ATTR
#define SUBJ_DIR_ATTR_TYPE_OID_BASE(num) {43, 6, 1, 5, 5, 7, 9, num}
    static const byte extSubjDirAttrDobOid[] = SUBJ_DIR_ATTR_TYPE_OID_BASE(1);
    static const byte extSubjDirAttrPobOid[] = SUBJ_DIR_ATTR_TYPE_OID_BASE(2);
    static const byte extSubjDirAttrGenderOid[] =
            SUBJ_DIR_ATTR_TYPE_OID_BASE(3);
    static const byte extSubjDirAttrCocOid[] = SUBJ_DIR_ATTR_TYPE_OID_BASE(4);
    static const byte extSubjDirAttrCorOid[] = SUBJ_DIR_ATTR_TYPE_OID_BASE(5);
#endif

#if defined(WOLFSSL_CERT_REQ) || defined(WOLFSSL_CERT_GEN) || \
    defined(WOLFSSL_ASN_TEMPLATE) || defined(OPENSSL_EXTRA) || \
    defined(OPENSSL_EXTRA_X509_SMALL)
/* csrAttrType */
#define CSR_ATTR_TYPE_OID_BASE(num) {42, 134, 72, 134, 247, 13, 1, 9, num}
#if !defined(WOLFSSL_CERT_REQ) || defined(WOLFSSL_CERT_GEN) || \
    defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL) || \
    defined(WOLFSSL_ASN_TEMPLATE)
static const byte attrEmailOid[] =             CSR_ATTR_TYPE_OID_BASE(1);
#endif
#ifdef WOLFSSL_CERT_REQ
static const byte attrUnstructuredNameOid[] =  CSR_ATTR_TYPE_OID_BASE(2);
static const byte attrPkcs9ContentTypeOid[] =  CSR_ATTR_TYPE_OID_BASE(3);
static const byte attrChallengePasswordOid[] = CSR_ATTR_TYPE_OID_BASE(7);
static const byte attrExtensionRequestOid[] =  CSR_ATTR_TYPE_OID_BASE(14);
static const byte attrSerialNumberOid[] = {85, 4, 5};
static const byte attrDnQualifier[] = {85, 4, 46};
static const byte attrInitals[] = {85, 4, 43};
static const byte attrSurname[] = {85, 4, 4};
static const byte attrGivenName[] = {85, 4, 42};
#endif
#endif

/* kdfType */
static const byte pbkdf2Oid[] = {42, 134, 72, 134, 247, 13, 1, 5, 12};

/* PKCS5 */
#if !defined(NO_DES3) && !defined(NO_MD5)
static const byte pbeMd5Des[] = {42, 134, 72, 134, 247, 13, 1, 5, 3};
#endif
#if !defined(NO_DES3) && !defined(NO_SHA)
static const byte pbeSha1Des[] = {42, 134, 72, 134, 247, 13, 1, 5, 10};
#endif
static const byte pbes2[] = {42, 134, 72, 134, 247, 13, 1, 5, 13};

/* PKCS12 */
#if !defined(NO_RC4) && !defined(NO_SHA)
static const byte pbeSha1RC4128[] = {42, 134, 72, 134, 247, 13, 1, 12, 1, 1};
#endif
#if !defined(NO_DES3) && !defined(NO_SHA)
static const byte pbeSha1Des3[] = {42, 134, 72, 134, 247, 13, 1, 12, 1, 3};
#endif
#if defined(WC_RC2) && !defined(NO_SHA)
static const byte pbe40Rc2Cbc[] = {42, 134, 72, 134, 247, 13, 1, 12, 1, 6};
#endif

#ifdef HAVE_LIBZ
/* zlib compression */
static const byte zlibCompress[] = {42, 134, 72, 134, 247, 13, 1, 9, 16, 3, 8};
#endif
#ifdef WOLFSSL_APACHE_HTTPD
/* tlsExtType */
static const byte tlsFeatureOid[] = {43, 6, 1, 5, 5, 7, 1, 24};
/* certNameType */
static const byte dnsSRVOid[] = {43, 6, 1, 5, 5, 7, 8, 7};
#endif

#if defined(WOLFSSL_CERT_REQ) || defined(WOLFSSL_CERT_GEN) || \
    defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL) || \
    defined(WOLFSSL_ASN_TEMPLATE)
/* Pilot attribute types (0.9.2342.19200300.100.1.*) */
#define PLT_ATTR_TYPE_OID_BASE(num) {9, 146, 38, 137, 147, 242, 44, 100, 1, num}
static const byte uidOid[] = PLT_ATTR_TYPE_OID_BASE(1); /* user id */
static const byte rfc822Mlbx[] = PLT_ATTR_TYPE_OID_BASE(3); /* RFC822 mailbox */
static const byte fvrtDrk[] = PLT_ATTR_TYPE_OID_BASE(5);/* favourite drink*/
#endif

#if defined(WOLFSSL_CERT_GEN) || \
    defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL) || \
    defined(WOLFSSL_ASN_TEMPLATE)
static const byte dcOid[] = {9, 146, 38, 137, 147, 242, 44, 100, 1, 25}; /* domain component */
#endif


/* Looks up the ID/type of an OID.
 *
 * When known returns the OID as a byte array and its length.
 * ID-type are unique.
 *
 * Use oidIgnoreType to autofail.
 *
 * Note that while this function currently handles a large
 * number of FPKI certificate policy OIDs, these OIDs are not
 * currently being handled in the code, they are just recognized
 * as valid OIDs.
 *
 * @param [in]  id     OID id.
 * @param [in]  type   Type of OID (enum Oid_Types).
 * @param [out] oidSz  Length of OID byte array returned.
 * @return  Array of bytes for the OID.
 * @return  NULL when ID/type not recognized.
 */
const byte* OidFromId(word32 id, word32 type, word32* oidSz)
{
    const byte* oid = NULL;

    *oidSz = 0;

    switch (type) {

        case oidHashType:
            switch (id) {
            #ifdef WOLFSSL_MD2
                case MD2h:
                    oid = hashMd2hOid;
                    *oidSz = sizeof(hashMd2hOid);
                    break;
            #endif
            #ifndef NO_MD5
                case MD5h:
                    oid = hashMd5hOid;
                    *oidSz = sizeof(hashMd5hOid);
                    break;
            #endif
            #ifndef NO_SHA
                case SHAh:
                    oid = hashSha1hOid;
                    *oidSz = sizeof(hashSha1hOid);
                    break;
            #endif
            #ifdef WOLFSSL_SHA224
                case SHA224h:
                    oid = hashSha224hOid;
                    *oidSz = sizeof(hashSha224hOid);
                    break;
            #endif
            #ifndef NO_SHA256
                case SHA256h:
                    oid = hashSha256hOid;
                    *oidSz = sizeof(hashSha256hOid);
                    break;
            #endif
            #ifdef WOLFSSL_SHA384
                case SHA384h:
                    oid = hashSha384hOid;
                    *oidSz = sizeof(hashSha384hOid);
                    break;
            #endif
            #ifdef WOLFSSL_SHA512
                #ifndef WOLFSSL_NOSHA512_224
                case SHA512_224h:
                    oid = hashSha512_224hOid;
                    *oidSz = sizeof(hashSha512_224hOid);
                    break;
                #endif
                #ifndef WOLFSSL_NOSHA512_256
                case SHA512_256h:
                    oid = hashSha512_256hOid;
                    *oidSz = sizeof(hashSha512_256hOid);
                    break;
                #endif
                case SHA512h:
                    oid = hashSha512hOid;
                    *oidSz = sizeof(hashSha512hOid);
                    break;
            #endif
            #ifdef WOLFSSL_SHA3
            #ifndef WOLFSSL_NOSHA3_224
                case SHA3_224h:
                    oid = hashSha3_224hOid;
                    *oidSz = sizeof(hashSha3_224hOid);
                    break;
            #endif /* WOLFSSL_NOSHA3_224 */
            #ifndef WOLFSSL_NOSHA3_256
                case SHA3_256h:
                    oid = hashSha3_256hOid;
                    *oidSz = sizeof(hashSha3_256hOid);
                    break;
            #endif /* WOLFSSL_NOSHA3_256 */
            #ifndef WOLFSSL_NOSHA3_384
                case SHA3_384h:
                    oid = hashSha3_384hOid;
                    *oidSz = sizeof(hashSha3_384hOid);
                    break;
            #endif /* WOLFSSL_NOSHA3_384 */
            #ifndef WOLFSSL_NOSHA3_512
                case SHA3_512h:
                    oid = hashSha3_512hOid;
                    *oidSz = sizeof(hashSha3_512hOid);
                    break;
            #endif /* WOLFSSL_NOSHA3_512 */
            #endif /* WOLFSSL_SHA3 */
                default:
                    break;
            }
            break;

        case oidSigType:
            switch (id) {
                #if !defined(NO_DSA) && !defined(NO_SHA)
                case CTC_SHAwDSA:
                    oid = sigSha1wDsaOid;
                    *oidSz = sizeof(sigSha1wDsaOid);
                    break;
                case CTC_SHA256wDSA:
                    oid = sigSha256wDsaOid;
                    *oidSz = sizeof(sigSha256wDsaOid);
                    break;
                #endif /* NO_DSA */
                #ifndef NO_RSA
                #ifdef WOLFSSL_MD2
                case CTC_MD2wRSA:
                    oid = sigMd2wRsaOid;
                    *oidSz = sizeof(sigMd2wRsaOid);
                    break;
                #endif
                #ifndef NO_MD5
                case CTC_MD5wRSA:
                    oid = sigMd5wRsaOid;
                    *oidSz = sizeof(sigMd5wRsaOid);
                    break;
                #endif
                #ifndef NO_SHA
                case CTC_SHAwRSA:
                    oid = sigSha1wRsaOid;
                    *oidSz = sizeof(sigSha1wRsaOid);
                    break;
                #endif
                #ifdef WOLFSSL_SHA224
                case CTC_SHA224wRSA:
                    oid = sigSha224wRsaOid;
                    *oidSz = sizeof(sigSha224wRsaOid);
                    break;
                #endif
                #ifndef NO_SHA256
                case CTC_SHA256wRSA:
                    oid = sigSha256wRsaOid;
                    *oidSz = sizeof(sigSha256wRsaOid);
                    break;
                #endif
                #ifdef WOLFSSL_SHA384
                case CTC_SHA384wRSA:
                    oid = sigSha384wRsaOid;
                    *oidSz = sizeof(sigSha384wRsaOid);
                    break;
                #endif
                #ifdef WOLFSSL_SHA512
                case CTC_SHA512wRSA:
                    oid = sigSha512wRsaOid;
                    *oidSz = sizeof(sigSha512wRsaOid);
                    break;
                #endif /* WOLFSSL_SHA512 */
                #ifdef WOLFSSL_SHA3
                #ifndef WOLFSSL_NOSHA3_224
                case CTC_SHA3_224wRSA:
                    oid = sigSha3_224wRsaOid;
                    *oidSz = sizeof(sigSha3_224wRsaOid);
                    break;
                #endif
                #ifndef WOLFSSL_NOSHA3_256
                case CTC_SHA3_256wRSA:
                    oid = sigSha3_256wRsaOid;
                    *oidSz = sizeof(sigSha3_256wRsaOid);
                    break;
                #endif
                #ifndef WOLFSSL_NOSHA3_384
                case CTC_SHA3_384wRSA:
                    oid = sigSha3_384wRsaOid;
                    *oidSz = sizeof(sigSha3_384wRsaOid);
                    break;
                #endif
                #ifndef WOLFSSL_NOSHA3_512
                case CTC_SHA3_512wRSA:
                    oid = sigSha3_512wRsaOid;
                    *oidSz = sizeof(sigSha3_512wRsaOid);
                    break;
                #endif
                #endif
                #ifdef WC_RSA_PSS
                case CTC_RSASSAPSS:
                    oid = sigRsaSsaPssOid;
                    *oidSz = sizeof(sigRsaSsaPssOid);
                    break;
                #endif
                #endif /* NO_RSA */
                #ifdef HAVE_ECC
                #ifndef NO_SHA
                case CTC_SHAwECDSA:
                    oid = sigSha1wEcdsaOid;
                    *oidSz = sizeof(sigSha1wEcdsaOid);
                    break;
                #endif
                #ifdef WOLFSSL_SHA224
                case CTC_SHA224wECDSA:
                    oid = sigSha224wEcdsaOid;
                    *oidSz = sizeof(sigSha224wEcdsaOid);
                    break;
                #endif
                #ifndef NO_SHA256
                case CTC_SHA256wECDSA:
                    oid = sigSha256wEcdsaOid;
                    *oidSz = sizeof(sigSha256wEcdsaOid);
                    break;
                #endif
                #ifdef WOLFSSL_SHA384
                case CTC_SHA384wECDSA:
                    oid = sigSha384wEcdsaOid;
                    *oidSz = sizeof(sigSha384wEcdsaOid);
                    break;
                #endif
                #ifdef WOLFSSL_SHA512
                case CTC_SHA512wECDSA:
                    oid = sigSha512wEcdsaOid;
                    *oidSz = sizeof(sigSha512wEcdsaOid);
                    break;
                #endif
                #ifdef WOLFSSL_SHA3
                #ifndef WOLFSSL_NOSHA3_224
                case CTC_SHA3_224wECDSA:
                    oid = sigSha3_224wEcdsaOid;
                    *oidSz = sizeof(sigSha3_224wEcdsaOid);
                    break;
                #endif
                #ifndef WOLFSSL_NOSHA3_256
                case CTC_SHA3_256wECDSA:
                    oid = sigSha3_256wEcdsaOid;
                    *oidSz = sizeof(sigSha3_256wEcdsaOid);
                    break;
                #endif
                #ifndef WOLFSSL_NOSHA3_384
                case CTC_SHA3_384wECDSA:
                    oid = sigSha3_384wEcdsaOid;
                    *oidSz = sizeof(sigSha3_384wEcdsaOid);
                    break;
                #endif
                #ifndef WOLFSSL_NOSHA3_512
                case CTC_SHA3_512wECDSA:
                    oid = sigSha3_512wEcdsaOid;
                    *oidSz = sizeof(sigSha3_512wEcdsaOid);
                    break;
                #endif
                #endif
                #if defined(WOLFSSL_SM2) && defined(WOLFSSL_SM3)
                case CTC_SM3wSM2:
                    oid = sigSm3wSm2Oid;
                    *oidSz = sizeof(sigSm3wSm2Oid);
                    break;
                #endif
                #endif /* HAVE_ECC */
                #ifdef HAVE_ED25519
                case CTC_ED25519:
                    oid = sigEd25519Oid;
                    *oidSz = sizeof(sigEd25519Oid);
                    break;
                #endif
                #ifdef HAVE_ED448
                case CTC_ED448:
                    oid = sigEd448Oid;
                    *oidSz = sizeof(sigEd448Oid);
                    break;
                #endif
                #ifdef HAVE_FALCON
                case CTC_FALCON_LEVEL1:
                    oid = sigFalcon_Level1Oid;
                    *oidSz = sizeof(sigFalcon_Level1Oid);
                    break;
                case CTC_FALCON_LEVEL5:
                    oid = sigFalcon_Level5Oid;
                    *oidSz = sizeof(sigFalcon_Level5Oid);
                    break;
                #endif /* HAVE_FALCON */
            #ifdef HAVE_DILITHIUM
                #ifdef WOLFSSL_DILITHIUM_FIPS204_DRAFT
                case CTC_DILITHIUM_LEVEL2:
                    oid = sigDilithium_Level2Oid;
                    *oidSz = sizeof(sigDilithium_Level2Oid);
                    break;
                case CTC_DILITHIUM_LEVEL3:
                    oid = sigDilithium_Level3Oid;
                    *oidSz = sizeof(sigDilithium_Level3Oid);
                    break;
                case CTC_DILITHIUM_LEVEL5:
                    oid = sigDilithium_Level5Oid;
                    *oidSz = sizeof(sigDilithium_Level5Oid);
                    break;
                #endif
                case CTC_ML_DSA_LEVEL2:
                    oid = sigMlDsa_Level2Oid;
                    *oidSz = sizeof(sigMlDsa_Level2Oid);
                    break;
                case CTC_ML_DSA_LEVEL3:
                    oid = sigMlDsa_Level3Oid;
                    *oidSz = sizeof(sigMlDsa_Level3Oid);
                    break;
                case CTC_ML_DSA_LEVEL5:
                    oid = sigMlDsa_Level5Oid;
                    *oidSz = sizeof(sigMlDsa_Level5Oid);
                    break;
            #endif /* HAVE_DILITHIUM */
                #ifdef HAVE_SPHINCS
                case CTC_SPHINCS_FAST_LEVEL1:
                    oid = sigSphincsFast_Level1Oid;
                    *oidSz = sizeof(sigSphincsFast_Level1Oid);
                    break;
                case CTC_SPHINCS_FAST_LEVEL3:
                    oid = sigSphincsFast_Level3Oid;
                    *oidSz = sizeof(sigSphincsFast_Level3Oid);
                    break;
                case CTC_SPHINCS_FAST_LEVEL5:
                    oid = sigSphincsFast_Level5Oid;
                    *oidSz = sizeof(sigSphincsFast_Level5Oid);
                    break;
                case CTC_SPHINCS_SMALL_LEVEL1:
                    oid = sigSphincsSmall_Level1Oid;
                    *oidSz = sizeof(sigSphincsSmall_Level1Oid);
                    break;
                case CTC_SPHINCS_SMALL_LEVEL3:
                    oid = sigSphincsSmall_Level3Oid;
                    *oidSz = sizeof(sigSphincsSmall_Level3Oid);
                    break;
                case CTC_SPHINCS_SMALL_LEVEL5:
                    oid = sigSphincsSmall_Level5Oid;
                    *oidSz = sizeof(sigSphincsSmall_Level5Oid);
                    break;
                #endif /* HAVE_SPHINCS */
                default:
                    break;
            }
            break;

        case oidKeyType:
            switch (id) {
                #ifndef NO_DSA
                case DSAk:
                    oid = keyDsaOid;
                    *oidSz = sizeof(keyDsaOid);
                    break;
                #endif /* NO_DSA */
            #ifndef NO_RSA
                case RSAk:
                    oid = keyRsaOid;
                    *oidSz = sizeof(keyRsaOid);
                    break;
                #ifdef WC_RSA_PSS
                case RSAPSSk:
                    oid = keyRsaPssOid;
                    *oidSz = sizeof(keyRsaPssOid);
                    break;
                #endif
            #endif /* NO_RSA */
                #ifdef HAVE_ECC
                case ECDSAk:
                    oid = keyEcdsaOid;
                    *oidSz = sizeof(keyEcdsaOid);
                    break;
                #endif /* HAVE_ECC */
                #ifdef HAVE_ED25519
                case ED25519k:
                    oid = keyEd25519Oid;
                    *oidSz = sizeof(keyEd25519Oid);
                    break;
                #endif /* HAVE_ED25519 */
                #ifdef HAVE_CURVE25519
                case X25519k:
                    oid = keyCurve25519Oid;
                    *oidSz = sizeof(keyCurve25519Oid);
                    break;
                #endif /* HAVE_CURVE25519 */
                #ifdef HAVE_ED448
                case ED448k:
                    oid = keyEd448Oid;
                    *oidSz = sizeof(keyEd448Oid);
                    break;
                #endif /* HAVE_ED448 */
                #ifdef HAVE_CURVE448
                case X448k:
                    oid = keyCurve448Oid;
                    *oidSz = sizeof(keyCurve448Oid);
                    break;
                #endif /* HAVE_CURVE448 */
                #ifndef NO_DH
                case DHk:
                    oid = keyDhOid;
                    *oidSz = sizeof(keyDhOid);
                    break;
                #endif /* !NO_DH */
                #ifdef HAVE_FALCON
                case FALCON_LEVEL1k:
                    oid = keyFalcon_Level1Oid;
                    *oidSz = sizeof(keyFalcon_Level1Oid);
                    break;
                case FALCON_LEVEL5k:
                    oid = keyFalcon_Level5Oid;
                    *oidSz = sizeof(keyFalcon_Level5Oid);
                    break;
                #endif /* HAVE_FALCON */
            #ifdef HAVE_DILITHIUM
                #ifdef WOLFSSL_DILITHIUM_FIPS204_DRAFT
                case DILITHIUM_LEVEL2k:
                    oid = keyDilithium_Level2Oid;
                    *oidSz = sizeof(keyDilithium_Level2Oid);
                    break;
                case DILITHIUM_LEVEL3k:
                    oid = keyDilithium_Level3Oid;
                    *oidSz = sizeof(keyDilithium_Level3Oid);
                    break;
                case DILITHIUM_LEVEL5k:
                    oid = keyDilithium_Level5Oid;
                    *oidSz = sizeof(keyDilithium_Level5Oid);
                    break;
                #endif
                case ML_DSA_LEVEL2k:
                    oid = keyMlDsa_Level2Oid;
                    *oidSz = sizeof(keyMlDsa_Level2Oid);
                    break;
                case ML_DSA_LEVEL3k:
                    oid = keyMlDsa_Level3Oid;
                    *oidSz = sizeof(keyMlDsa_Level3Oid);
                    break;
                case ML_DSA_LEVEL5k:
                    oid = keyMlDsa_Level5Oid;
                    *oidSz = sizeof(keyMlDsa_Level5Oid);
                    break;
            #endif /* HAVE_DILITHIUM */
                #ifdef HAVE_SPHINCS
                case SPHINCS_FAST_LEVEL1k:
                    oid = keySphincsFast_Level1Oid;
                    *oidSz = sizeof(keySphincsFast_Level1Oid);
                    break;
                case SPHINCS_FAST_LEVEL3k:
                    oid = keySphincsFast_Level3Oid;
                    *oidSz = sizeof(keySphincsFast_Level3Oid);
                    break;
                case SPHINCS_FAST_LEVEL5k:
                    oid = keySphincsFast_Level5Oid;
                    *oidSz = sizeof(keySphincsFast_Level5Oid);
                    break;
                case SPHINCS_SMALL_LEVEL1k:
                    oid = keySphincsSmall_Level1Oid;
                    *oidSz = sizeof(keySphincsSmall_Level1Oid);
                    break;
                case SPHINCS_SMALL_LEVEL3k:
                    oid = keySphincsSmall_Level3Oid;
                    *oidSz = sizeof(keySphincsSmall_Level3Oid);
                    break;
                case SPHINCS_SMALL_LEVEL5k:
                    oid = keySphincsSmall_Level5Oid;
                    *oidSz = sizeof(keySphincsSmall_Level5Oid);
                    break;
                #endif /* HAVE_SPHINCS */
                default:
                    break;
            }
            break;

        #ifdef HAVE_ECC
        case oidCurveType:
            if (wc_ecc_get_oid(id, &oid, oidSz) < 0) {
                WOLFSSL_MSG("ECC OID not found");
            }
            break;
        #endif /* HAVE_ECC */

        case oidBlkType:
            switch (id) {
    #ifdef HAVE_AES_CBC
        #ifdef WOLFSSL_AES_128
                case AES128CBCb:
                    oid = blkAes128CbcOid;
                    *oidSz = sizeof(blkAes128CbcOid);
                    break;
        #endif
        #ifdef WOLFSSL_AES_192
                case AES192CBCb:
                    oid = blkAes192CbcOid;
                    *oidSz = sizeof(blkAes192CbcOid);
                    break;
        #endif
        #ifdef WOLFSSL_AES_256
                case AES256CBCb:
                    oid = blkAes256CbcOid;
                    *oidSz = sizeof(blkAes256CbcOid);
                    break;
        #endif
    #endif /* HAVE_AES_CBC */
    #ifdef HAVE_AESGCM
        #ifdef WOLFSSL_AES_128
                case AES128GCMb:
                    oid = blkAes128GcmOid;
                    *oidSz = sizeof(blkAes128GcmOid);
                    break;
        #endif
        #ifdef WOLFSSL_AES_192
                case AES192GCMb:
                    oid = blkAes192GcmOid;
                    *oidSz = sizeof(blkAes192GcmOid);
                    break;
        #endif
        #ifdef WOLFSSL_AES_256
                case AES256GCMb:
                    oid = blkAes256GcmOid;
                    *oidSz = sizeof(blkAes256GcmOid);
                    break;
        #endif
    #endif /* HAVE_AESGCM */
    #ifdef HAVE_AESCCM
        #ifdef WOLFSSL_AES_128
                case AES128CCMb:
                    oid = blkAes128CcmOid;
                    *oidSz = sizeof(blkAes128CcmOid);
                    break;
        #endif
        #ifdef WOLFSSL_AES_192
                case AES192CCMb:
                    oid = blkAes192CcmOid;
                    *oidSz = sizeof(blkAes192CcmOid);
                    break;
        #endif
        #ifdef WOLFSSL_AES_256
                case AES256CCMb:
                    oid = blkAes256CcmOid;
                    *oidSz = sizeof(blkAes256CcmOid);
                    break;
        #endif
    #endif /* HAVE_AESCCM */
    #ifndef NO_DES3
                case DESb:
                    oid = blkDesCbcOid;
                    *oidSz = sizeof(blkDesCbcOid);
                    break;
                case DES3b:
                    oid = blkDes3CbcOid;
                    *oidSz = sizeof(blkDes3CbcOid);
                    break;
    #endif /* !NO_DES3 */
                default:
                    break;
            }
            break;

        #ifdef HAVE_OCSP
        case oidOcspType:
            switch (id) {
                case OCSP_BASIC_OID:
                    oid = ocspBasicOid;
                    *oidSz = sizeof(ocspBasicOid);
                    break;
                case OCSP_NONCE_OID:
                    oid = ocspNonceOid;
                    *oidSz = sizeof(ocspNonceOid);
                    break;
                default:
                    break;
            }
            break;
        #endif /* HAVE_OCSP */

        case oidCertExtType:
            switch (id) {
                case BASIC_CA_OID:
                    oid = extBasicCaOid;
                    *oidSz = sizeof(extBasicCaOid);
                    break;
                case ALT_NAMES_OID:
                    oid = extAltNamesOid;
                    *oidSz = sizeof(extAltNamesOid);
                    break;
                case CRL_DIST_OID:
                    oid = extCrlDistOid;
                    *oidSz = sizeof(extCrlDistOid);
                    break;
                case AUTH_INFO_OID:
                    oid = extAuthInfoOid;
                    *oidSz = sizeof(extAuthInfoOid);
                    break;
                case AUTH_KEY_OID:
                    oid = extAuthKeyOid;
                    *oidSz = sizeof(extAuthKeyOid);
                    break;
                case SUBJ_KEY_OID:
                    oid = extSubjKeyOid;
                    *oidSz = sizeof(extSubjKeyOid);
                    break;
                case CERT_POLICY_OID:
                    oid = extCertPolicyOid;
                    *oidSz = sizeof(extCertPolicyOid);
                    break;
                case KEY_USAGE_OID:
                    oid = extKeyUsageOid;
                    *oidSz = sizeof(extKeyUsageOid);
                    break;
                case INHIBIT_ANY_OID:
                    oid = extInhibitAnyOid;
                    *oidSz = sizeof(extInhibitAnyOid);
                    break;
                case EXT_KEY_USAGE_OID:
                    oid = extExtKeyUsageOid;
                    *oidSz = sizeof(extExtKeyUsageOid);
                    break;
            #ifndef IGNORE_NAME_CONSTRAINTS
                case NAME_CONS_OID:
                    oid = extNameConsOid;
                    *oidSz = sizeof(extNameConsOid);
                    break;
            #endif
            #ifdef HAVE_OCSP
                case OCSP_NOCHECK_OID:
                    oid = ocspNoCheckOid;
                    *oidSz = sizeof(ocspNoCheckOid);
                    break;
            #endif
            #ifdef WOLFSSL_SUBJ_DIR_ATTR
                case SUBJ_DIR_ATTR_OID:
                    oid = extSubjDirAttrOid;
                    *oidSz = sizeof(extSubjDirAttrOid);
                    break;
            #endif
            #ifdef WOLFSSL_SUBJ_INFO_ACC
                case SUBJ_INFO_ACC_OID:
                    oid = extSubjInfoAccessOid;
                    *oidSz = sizeof(extSubjInfoAccessOid);
                    break;
            #endif
                default:
                    break;
            }
            break;

        case oidCrlExtType:
            #ifdef HAVE_CRL
            switch (id) {
                case AUTH_KEY_OID:
                    oid = extAuthKeyOid;
                    *oidSz = sizeof(extAuthKeyOid);
                    break;
                case CRL_NUMBER_OID:
                    oid = extCrlNumberOid;
                    *oidSz = sizeof(extCrlNumberOid);
                    break;
                default:
                    break;
            }
            #endif
            break;

        case oidCertAuthInfoType:
            switch (id) {
                case AIA_OCSP_OID:
                    oid = extAuthInfoOcspOid;
                    *oidSz = sizeof(extAuthInfoOcspOid);
                    break;
                case AIA_CA_ISSUER_OID:
                    oid = extAuthInfoCaIssuerOid;
                    *oidSz = sizeof(extAuthInfoCaIssuerOid);
                    break;
                #ifdef WOLFSSL_SUBJ_INFO_ACC
                case AIA_CA_REPO_OID:
                    oid = extAuthInfoCaRespOid;
                    *oidSz = sizeof(extAuthInfoCaRespOid);
                    break;
                #endif /* WOLFSSL_SUBJ_INFO_ACC */
                default:
                    break;
            }
            break;

        case oidCertPolicyType:
            switch (id) {
                case CP_ANY_OID:
                    oid = extCertPolicyAnyOid;
                    *oidSz = sizeof(extCertPolicyAnyOid);
                    break;
                case CP_ISRG_DOMAIN_VALID:
                    oid = extCertPolicyIsrgDomainValid;
                    *oidSz = sizeof(extCertPolicyIsrgDomainValid);
                    break;
                #if defined(WOLFSSL_FPKI)
                case CP_FPKI_HIGH_ASSURANCE_OID:
                    oid = extCertPolicyFpkiHighAssuranceOid;
                    *oidSz = sizeof(extCertPolicyFpkiHighAssuranceOid);
                    break;
                case CP_FPKI_COMMON_HARDWARE_OID:
                    oid = extCertPolicyFpkiCommonHardwareOid;
                    *oidSz = sizeof(extCertPolicyFpkiCommonHardwareOid);
                    break;
                case CP_FPKI_MEDIUM_HARDWARE_OID:
                    oid = extCertPolicyFpkiMediumHardwareOid;
                    *oidSz = sizeof(extCertPolicyFpkiMediumHardwareOid);
                    break;
                case CP_FPKI_COMMON_HIGH_OID:
                    oid = extCertPolicyFpkiCommonHighOid;
                    *oidSz = sizeof(extCertPolicyFpkiCommonHighOid);
                    break;
                case CP_FPKI_COMMON_DEVICES_HARDWARE_OID:
                    oid = extCertPolicyFpkiCommonDevicesHardwareOid;
                    *oidSz = sizeof(extCertPolicyFpkiCommonDevicesHardwareOid);
                    break;
                case CP_FPKI_COMMON_PIV_CONTENT_SIGNING_OID:
                    oid = extCertPolicyFpkiCommonPivContentSigningOid;
                    *oidSz = sizeof(extCertPolicyFpkiCommonPivContentSigningOid);
                    break;
                case CP_FPKI_COMMON_AUTH_OID:
                    oid = extCertPolicyFpkiCommonAuthOid;
                    *oidSz = sizeof(extCertPolicyFpkiCommonAuthOid);
                    break;
                case CP_FPKI_PIV_AUTH_OID:
                    oid = extCertPolicyFpkiPivAuthOid;
                    *oidSz = sizeof(extCertPolicyFpkiPivAuthOid);
                    break;
                case CP_FPKI_PIV_AUTH_HW_OID: /* collision with AES256CBCb */
                    oid = extCertPolicyFpkiPivAuthHwOid;
                    *oidSz = sizeof(extCertPolicyFpkiPivAuthHwOid);
                    break;
                case CP_FPKI_PIVI_AUTH_OID:
                    oid = extCertPolicyFpkiPiviAuthOid;
                    *oidSz = sizeof(extCertPolicyFpkiPiviAuthOid);
                    break;
                case CP_FPKI_AUTH_TEST_OID:
                    oid = extCertPolicyFpkiAuthTestOid;
                    *oidSz = sizeof(extCertPolicyFpkiAuthTestOid);
                    break;
                case CP_FPKI_CARDAUTH_TEST_OID:
                    oid = extCertPolicyFpkiCardauthTestOid;
                    *oidSz = sizeof(extCertPolicyFpkiCardauthTestOid);
                    break;
                case CP_FPKI_PIV_CONTENT_TEST_OID:
                    oid = extCertPolicyFpkiPivContentTestOid;
                    *oidSz = sizeof(extCertPolicyFpkiPivContentTestOid);
                    break;
                case CP_FPKI_PIV_AUTH_DERIVED_TEST_OID:
                    oid = extCertPolicyFpkiAuthDerivedTestOid;
                    *oidSz = sizeof(extCertPolicyFpkiAuthDerivedTestOid);
                    break;
                case CP_FPKI_PIV_AUTH_DERIVED_HW_TEST_OID:
                    oid = extCertPolicyFpkiAuthDerivedHwTestOid;
                    *oidSz = sizeof(extCertPolicyFpkiAuthDerivedHwTestOid);
                    break;
                case CP_DOD_MEDIUM_OID:
                    oid = extCertPolicyDodMediumOid;
                    *oidSz = sizeof(extCertPolicyDodMediumOid);
                    break;
                case CP_DOD_MEDIUM_HARDWARE_OID:
                    oid = extCertPolicyDodMediumHardwareOid;
                    *oidSz = sizeof(extCertPolicyDodMediumHardwareOid);
                    break;
                case CP_DOD_PIV_AUTH_OID:
                    oid = extCertPolicyDodPivAuthOid;
                    *oidSz = sizeof(extCertPolicyDodPivAuthOid);
                    break;
                case CP_DOD_MEDIUM_NPE_OID:
                    oid = extCertPolicyDodMediumNpeOid;
                    *oidSz = sizeof(extCertPolicyDodMediumNpeOid);
                    break;
                case CP_DOD_MEDIUM_2048_OID:
                    oid = extCertPolicyDodMedium2048Oid;
                    *oidSz = sizeof(extCertPolicyDodMedium2048Oid);
                    break;
                case CP_DOD_MEDIUM_HARDWARE_2048_OID:
                    oid = extCertPolicyDodMediumHardware2048Oid;
                    *oidSz = sizeof(extCertPolicyDodMediumHardware2048Oid);
                    break;
                case CP_DOD_PIV_AUTH_2048_OID:
                    oid = extCertPolicyDodPivAuth2048Oid;
                    *oidSz = sizeof(extCertPolicyDodPivAuth2048Oid);
                    break;
                case CP_DOD_PEER_INTEROP_OID:
                    oid = extCertPolicyDodPeerInteropOid;
                    *oidSz = sizeof(extCertPolicyDodPeerInteropOid);
                    break;
                case CP_DOD_MEDIUM_NPE_112_OID:
                    oid = extCertPolicyDodMediumNpe112Oid;
                    *oidSz = sizeof(extCertPolicyDodMediumNpe112Oid);
                    break;
                case CP_DOD_MEDIUM_NPE_128_OID:
                    oid = extCertPolicyDodMediumNpe128Oid;
                    *oidSz = sizeof(extCertPolicyDodMediumNpe128Oid);
                    break;
                case CP_DOD_MEDIUM_NPE_192_OID:
                    oid = extCertPolicyDodMediumNpe192Oid;
                    *oidSz = sizeof(extCertPolicyDodMediumNpe192Oid);
                    break;
                case CP_DOD_MEDIUM_112_OID:
                    oid = extCertPolicyDodMedium112Oid;
                    *oidSz = sizeof(extCertPolicyDodMedium112Oid);
                    break;
                case CP_DOD_MEDIUM_128_OID:
                    oid = extCertPolicyDodMedium128Oid;
                    *oidSz = sizeof(extCertPolicyDodMedium128Oid);
                    break;
                case CP_DOD_MEDIUM_192_OID:
                    oid = extCertPolicyDodMedium192Oid;
                    *oidSz = sizeof(extCertPolicyDodMedium192Oid);
                    break;
                case CP_DOD_MEDIUM_HARDWARE_112_OID:
                    oid = extCertPolicyDodMediumHardware112Oid;
                    *oidSz = sizeof(extCertPolicyDodMediumHardware112Oid);
                    break;
                case CP_DOD_MEDIUM_HARDWARE_128_OID:
                    oid = extCertPolicyDodMediumHardware128Oid;
                    *oidSz = sizeof(extCertPolicyDodMediumHardware128Oid);
                    break;
                case CP_DOD_MEDIUM_HARDWARE_192_OID:
                    oid = extCertPolicyDodMediumHardware192Oid;
                    *oidSz = sizeof(extCertPolicyDodMediumHardware192Oid);
                    break;
                case CP_DOD_ADMIN_OID:
                    oid = extCertPolicyDodAdminOid;
                    *oidSz = sizeof(extCertPolicyDodAdminOid);
                    break;
                case CP_DOD_INTERNAL_NPE_112_OID:
                    oid = extCertPolicyDodInternalNpe112Oid;
                    *oidSz = sizeof(extCertPolicyDodInternalNpe112Oid);
                    break;
                case CP_DOD_INTERNAL_NPE_128_OID:
                    oid = extCertPolicyDodInternalNpe128Oid;
                    *oidSz = sizeof(extCertPolicyDodInternalNpe128Oid);
                    break;
                case CP_DOD_INTERNAL_NPE_192_OID:
                    oid = extCertPolicyDodInternalNpe192Oid;
                    *oidSz = sizeof(extCertPolicyDodInternalNpe192Oid);
                    break;
                case CP_ECA_MEDIUM_OID:
                    oid = extCertPolicyEcaMediumOid;
                    *oidSz = sizeof(extCertPolicyEcaMediumOid);
                    break;
                case CP_ECA_MEDIUM_HARDWARE_OID:
                    oid = extCertPolicyEcaMediumHardwareOid;
                    *oidSz = sizeof(extCertPolicyEcaMediumHardwareOid);
                    break;
                case CP_ECA_MEDIUM_TOKEN_OID:
                    oid = extCertPolicyEcaMediumTokenOid;
                    *oidSz = sizeof(extCertPolicyEcaMediumTokenOid);
                    break;
                case CP_ECA_MEDIUM_SHA256_OID:
                    oid = extCertPolicyEcaMediumSha256Oid;
                    *oidSz = sizeof(extCertPolicyEcaMediumSha256Oid);
                    break;
                case CP_ECA_MEDIUM_TOKEN_SHA256_OID:
                    oid = extCertPolicyEcaMediumTokenSha256Oid;
                    *oidSz = sizeof(extCertPolicyEcaMediumTokenSha256Oid);
                    break;
                case CP_ECA_MEDIUM_HARDWARE_PIVI_OID:
                    oid = extCertPolicyEcaMediumHardwarePiviOid;
                    *oidSz = sizeof(extCertPolicyEcaMediumHardwarePiviOid);
                    break;
                case CP_ECA_CONTENT_SIGNING_PIVI_OID:
                    oid = extCertPolicyEcaContentSigningPiviOid;
                    *oidSz = sizeof(extCertPolicyEcaContentSigningPiviOid);
                    break;
                case CP_ECA_MEDIUM_DEVICE_SHA256_OID:
                    oid = extCertPolicyEcaMediumDeviceSha256Oid;
                    *oidSz = sizeof(extCertPolicyEcaMediumDeviceSha256Oid);
                    break;
                case CP_ECA_MEDIUM_HARDWARE_SHA256_OID:
                    oid = extCertPolicyEcaMediumHardwareSha256Oid;
                    *oidSz = sizeof(extCertPolicyEcaMediumHardwareSha256Oid);
                    break;

                /* Department of State PKI OIDs */
                case CP_STATE_BASIC_OID:
                    oid = extCertPolicyStateBasicOid;
                    *oidSz = sizeof(extCertPolicyStateBasicOid);
                    break;
                case CP_STATE_LOW_OID:
                    oid = extCertPolicyStateLowOid;
                    *oidSz = sizeof(extCertPolicyStateLowOid);
                    break;
                case CP_STATE_MODERATE_OID:
                    oid = extCertPolicyStateModerateOid;
                    *oidSz = sizeof(extCertPolicyStateModerateOid);
                    break;
                case CP_STATE_HIGH_OID:
                    oid = extCertPolicyStateHighOid;
                    *oidSz = sizeof(extCertPolicyStateHighOid);
                    break;
                case CP_STATE_MEDHW_OID:
                    oid = extCertPolicyStateMedHwOid;
                    *oidSz = sizeof(extCertPolicyStateMedHwOid);
                    break;
                case CP_STATE_MEDDEVHW_OID:
                    oid = extCertPolicyStateMediumDeviceHardwareOid;
                    *oidSz = sizeof(extCertPolicyStateMediumDeviceHardwareOid);
                    break;

                /* U.S. Treasury SSP PKI OIDs */
                case CP_TREAS_MEDIUMHW_OID:
                    oid = extCertPolicyTreasuryMediumHardwareOid;
                    *oidSz = sizeof(extCertPolicyTreasuryMediumHardwareOid);
                    break;
                case CP_TREAS_HIGH_OID:
                    oid = extCertPolicyTreasuryHighOid;
                    *oidSz = sizeof(extCertPolicyTreasuryHighOid);
                    break;
                case CP_TREAS_PIVI_HW_OID:
                    oid = extCertPolicyTreasuryPiviHardwareOid;
                    *oidSz = sizeof(extCertPolicyTreasuryPiviHardwareOid);
                    break;
                case CP_TREAS_PIVI_CONTENT_OID:
                    oid = extCertPolicyTreasuryPiviContentSigningOid;
                    *oidSz = sizeof(extCertPolicyTreasuryPiviContentSigningOid);
                    break;

                /* Boeing PKI OIDs */
                case CP_BOEING_MEDIUMHW_SHA256_OID:
                    oid = extCertPolicyBoeingMediumHardwareSha256Oid;
                    *oidSz = sizeof(extCertPolicyBoeingMediumHardwareSha256Oid);
                    break;
                case CP_BOEING_MEDIUMHW_CONTENT_SHA256_OID:
                    oid = extCertPolicyBoeingMediumHardwareContentSigningSha256Oid;
                    *oidSz = sizeof(extCertPolicyBoeingMediumHardwareContentSigningSha256Oid);
                    break;

                /* DigiCert NFI PKI OIDs */
                case CP_DIGICERT_NFSSP_MEDIUMHW_OID:
                    oid = extCertPolicyDigicertNfiMediumHardwareOid;
                    *oidSz = sizeof(extCertPolicyDigicertNfiMediumHardwareOid);
                    break;
                case CP_DIGICERT_NFSSP_AUTH_OID:
                    oid = extCertPolicyDigicertNfiAuthOid;
                    *oidSz = sizeof(extCertPolicyDigicertNfiAuthOid);
                    break;
                case CP_DIGICERT_NFSSP_PIVI_HW_OID:
                    oid = extCertPolicyDigicertNfiPiviHardwareOid;
                    *oidSz = sizeof(extCertPolicyDigicertNfiPiviHardwareOid);
                    break;
                case CP_DIGICERT_NFSSP_PIVI_CONTENT_OID:
                    oid = extCertPolicyDigicertNfiPiviContentSigningOid;
                    *oidSz = sizeof(extCertPolicyDigicertNfiPiviContentSigningOid);
                    break;
                case CP_DIGICERT_NFSSP_MEDDEVHW_OID:
                    oid = extCertPolicyDigicertNfiMediumDevicesHardwareOid;
                    *oidSz = sizeof(extCertPolicyDigicertNfiMediumDevicesHardwareOid);
                    break;

                /* Entrust Managed Services NFI PKI OIDs */
                case CP_ENTRUST_NFSSP_MEDIUMHW_OID:
                    oid = extCertPolicyEntrustNfiMediumHardwareOid;
                    *oidSz = sizeof(extCertPolicyEntrustNfiMediumHardwareOid);
                    break;
                case CP_ENTRUST_NFSSP_MEDAUTH_OID:
                    oid = extCertPolicyEntrustNfiMediumAuthenticationOid;
                    *oidSz = sizeof(extCertPolicyEntrustNfiMediumAuthenticationOid);
                    break;
                case CP_ENTRUST_NFSSP_PIVI_HW_OID:
                    oid = extCertPolicyEntrustNfiPiviHardwareOid;
                    *oidSz = sizeof(extCertPolicyEntrustNfiPiviHardwareOid);
                    break;
                case CP_ENTRUST_NFSSP_PIVI_CONTENT_OID:
                    oid = extCertPolicyEntrustNfiPiviContentSigningOid;
                    *oidSz = sizeof(extCertPolicyEntrustNfiPiviContentSigningOid);
                    break;
                case CP_ENTRUST_NFSSP_MEDDEVHW_OID:
                    oid = extCertPolicyEntrustNfiMediumDevicesHwOid;
                    *oidSz = sizeof(extCertPolicyEntrustNfiMediumDevicesHwOid);
                    break;

                /* Exostar LLC PKI OIDs */
                case CP_EXOSTAR_MEDIUMHW_SHA2_OID:
                    oid = extCertPolicyExostarMediumHardwareSha2Oid;
                    *oidSz = sizeof(extCertPolicyExostarMediumHardwareSha2Oid);
                    break;

                /* Lockheed Martin PKI OIDs */
                case CP_LOCKHEED_MEDIUMHW_OID:
                    oid = extCertPolicyLockheedMediumAssuranceHardwareOid;
                    *oidSz = sizeof(extCertPolicyLockheedMediumAssuranceHardwareOid);
                    break;

                /* Northrop Grumman PKI OIDs */
                case CP_NORTHROP_MEDIUM_256_HW_OID:
                    oid = extCertPolicyNorthropMediumAssurance256HardwareTokenOid;
                    *oidSz = sizeof(extCertPolicyNorthropMediumAssurance256HardwareTokenOid);
                    break;
                case CP_NORTHROP_PIVI_256_HW_OID:
                    oid = extCertPolicyNorthropPiviAssurance256HardwareTokenOid;
                    *oidSz = sizeof(extCertPolicyNorthropPiviAssurance256HardwareTokenOid);
                    break;
                case CP_NORTHROP_PIVI_256_CONTENT_OID:
                    oid = extCertPolicyNorthropPiviAssurance256ContentSigningOid;
                    *oidSz = sizeof(extCertPolicyNorthropPiviAssurance256ContentSigningOid);
                    break;
                case CP_NORTHROP_MEDIUM_384_HW_OID:
                    oid = extCertPolicyNorthropMediumAssurance384HardwareTokenOid;
                    *oidSz = sizeof(extCertPolicyNorthropMediumAssurance384HardwareTokenOid);
                    break;

                /* Raytheon PKI OIDs */
                case CP_RAYTHEON_MEDIUMHW_OID:
                    oid = extCertPolicyRaytheonMediumHardwareOid;
                    *oidSz = sizeof(extCertPolicyRaytheonMediumHardwareOid);
                    break;
                case CP_RAYTHEON_MEDDEVHW_OID:
                    oid = extCertPolicyRaytheonMediumDeviceHardwareOid;
                    *oidSz = sizeof(extCertPolicyRaytheonMediumDeviceHardwareOid);
                    break;
                case CP_RAYTHEON_SHA2_MEDIUMHW_OID:
                    oid = extCertPolicyRaytheonSha2MediumHardwareOid;
                    *oidSz = sizeof(extCertPolicyRaytheonSha2MediumHardwareOid);
                    break;
                case CP_RAYTHEON_SHA2_MEDDEVHW_OID:
                    oid = extCertPolicyRaytheonSha2MediumDeviceHardwareOid;
                   *oidSz = sizeof(extCertPolicyRaytheonSha2MediumDeviceHardwareOid);
                    break;

                /* WidePoint NFI PKI OIDs */
                case CP_WIDEPOINT_MEDIUMHW_OID:
                    oid = extCertPolicyWidepointNfiMediumHardwareOid;
                    *oidSz = sizeof(extCertPolicyWidepointNfiMediumHardwareOid);
                    break;
                case CP_WIDEPOINT_PIVI_HW_OID:
                    oid = extCertPolicyWidepointNfiPiviHardwareOid;
                    *oidSz = sizeof(extCertPolicyWidepointNfiPiviHardwareOid);
                    break;
                case CP_WIDEPOINT_PIVI_CONTENT_OID:
                    oid = extCertPolicyWidepointNfiPiviContentSigningOid;
                    *oidSz = sizeof(extCertPolicyWidepointNfiPiviContentSigningOid);
                    break;
                case CP_WIDEPOINT_MEDDEVHW_OID:
                    oid = extCertPolicyWidepointNfiMediumDevicesHardwareOid;
                    *oidSz = sizeof(extCertPolicyWidepointNfiMediumDevicesHardwareOid);
                    break;

                /* Australian Defence Organisation PKI OIDs */
                case CP_ADO_MEDIUM_OID:
                    oid = extCertPolicyAdoIndividualMediumAssuranceOid;
                    *oidSz = sizeof(extCertPolicyAdoIndividualMediumAssuranceOid);
                    break;
                case CP_ADO_HIGH_OID:
                    oid = extCertPolicyAdoIndividualHighAssuranceOid;
                    *oidSz = sizeof(extCertPolicyAdoIndividualHighAssuranceOid);
                    break;
                case CP_ADO_RESOURCE_MEDIUM_OID:
                    oid = extCertPolicyAdoResourceMediumAssuranceOid;
                    *oidSz = sizeof(extCertPolicyAdoResourceMediumAssuranceOid);
                    break;

                /* Netherlands Ministry of Defence PKI OIDs */
                case CP_NL_MOD_AUTH_OID:
                    oid = extCertPolicyNlModAuthenticityOid;
                    *oidSz = sizeof(extCertPolicyNlModAuthenticityOid);
                    break;
                case CP_NL_MOD_IRREFUT_OID:
                    oid = extCertPolicyNlModIrrefutabilityOid;
                    *oidSz = sizeof(extCertPolicyNlModIrrefutabilityOid);
                    break;
                case CP_NL_MOD_CONFID_OID:
                    oid = extCertPolicyNlModConfidentialityOid;
                    *oidSz = sizeof(extCertPolicyNlModConfidentialityOid);
                    break;

                /* IdenTrust NFI OIDs */
                case CP_IDENTRUST_MEDIUMHW_SIGN_OID:
                    oid = extCertPolicyIdentrustMediumhwSignOid;
                    *oidSz = sizeof(extCertPolicyIdentrustMediumhwSignOid);
                    break;
                case CP_IDENTRUST_MEDIUMHW_ENC_OID:
                    oid = extCertPolicyIdentrustMediumhwEncOid;
                    *oidSz = sizeof(extCertPolicyIdentrustMediumhwEncOid);
                    break;
                case CP_IDENTRUST_PIVI_HW_ID_OID:
                    oid = extCertPolicyIdentrustPiviHwIdOid;
                    *oidSz = sizeof(extCertPolicyIdentrustPiviHwIdOid);
                    break;
                case CP_IDENTRUST_PIVI_HW_SIGN_OID:
                    oid = extCertPolicyIdentrustPiviHwSignOid;
                    *oidSz = sizeof(extCertPolicyIdentrustPiviHwSignOid);
                    break;
                case CP_IDENTRUST_PIVI_HW_ENC_OID:
                    oid = extCertPolicyIdentrustPiviHwEncOid;
                    *oidSz = sizeof(extCertPolicyIdentrustPiviHwEncOid);
                    break;
                case CP_IDENTRUST_PIVI_CONTENT_OID:
                    oid = extCertPolicyIdentrustPiviContentOid;
                    *oidSz = sizeof(extCertPolicyIdentrustPiviContentOid);
                    break;

                /* TSCP Bridge OIDs */
                case CP_TSCP_MEDIUMHW_OID:
                    oid = extCertPolicyTscpMediumhwOid;
                    *oidSz = sizeof(extCertPolicyTscpMediumhwOid);
                    break;
                case CP_TSCP_PIVI_OID:
                    oid = extCertPolicyTscpPiviOid;
                    *oidSz = sizeof(extCertPolicyTscpPiviOid);
                    break;
                case CP_TSCP_PIVI_CONTENT_OID:
                    oid = extCertPolicyTscpPiviContentOid;
                    *oidSz = sizeof(extCertPolicyTscpPiviContentOid);
                    break;

                /* Carillon Federal Services OIDs */
                case CP_CARILLON_MEDIUMHW_256_OID:
                    oid = extCertPolicyCarillonMediumhw256Oid;
                    *oidSz = sizeof(extCertPolicyCarillonMediumhw256Oid);
                    break;
                case CP_CARILLON_AIVHW_OID:
                    oid = extCertPolicyCarillonAivhwOid;
                    *oidSz = sizeof(extCertPolicyCarillonAivhwOid);
                    break;
                case CP_CARILLON_AIVCONTENT_OID:
                    oid = extCertPolicyCarillonAivcontentOid;
                    *oidSz = sizeof(extCertPolicyCarillonAivcontentOid);
                    break;

                /* Carillon Information Security OIDs */
                case CP_CIS_MEDIUMHW_256_OID:
                    oid = extCertPolicyCisMediumhw256Oid;
                    *oidSz = sizeof(extCertPolicyCisMediumhw256Oid);
                    break;
                case CP_CIS_MEDDEVHW_256_OID:
                    oid = extCertPolicyCisMeddevhw256Oid;
                    *oidSz = sizeof(extCertPolicyCisMeddevhw256Oid);
                    break;
                case CP_CIS_ICECAP_HW_OID:
                    oid = extCertPolicyCisIcecapHwOid;
                    *oidSz = sizeof(extCertPolicyCisIcecapHwOid);
                    break;
                case CP_CIS_ICECAP_CONTENT_OID:
                    oid = extCertPolicyCisIcecapContentOid;
                    *oidSz = sizeof(extCertPolicyCisIcecapContentOid);
                    break;

                /* CertiPath Bridge OIDs */
                case CP_CERTIPATH_MEDIUMHW_OID:
                    oid = extCertPolicyCertipathMediumhwOid;
                    *oidSz = sizeof(extCertPolicyCertipathMediumhwOid);
                    break;
                case CP_CERTIPATH_HIGHHW_OID:
                    oid = extCertPolicyCertipathHighhwOid;
                    *oidSz = sizeof(extCertPolicyCertipathHighhwOid);
                    break;
                case CP_CERTIPATH_ICECAP_HW_OID:
                    oid = extCertPolicyCertipathIcecapHwOid;
                    *oidSz = sizeof(extCertPolicyCertipathIcecapHwOid);
                    break;
                case CP_CERTIPATH_ICECAP_CONTENT_OID:
                    oid = extCertPolicyCertipathIcecapContentOid;
                    *oidSz = sizeof(extCertPolicyCertipathIcecapContentOid);
                    break;
                case CP_CERTIPATH_VAR_MEDIUMHW_OID:
                    oid = extCertPolicyCertipathVarMediumhwOid;
                    *oidSz = sizeof(extCertPolicyCertipathVarMediumhwOid);
                    break;
                case CP_CERTIPATH_VAR_HIGHHW_OID:
                    oid = extCertPolicyCertipathVarHighhwOid;
                    *oidSz = sizeof(extCertPolicyCertipathVarHighhwOid);
                    break;
                case CP_COMODO_OID:
                    oid = extCertPolicyComodoLtdOid;
                    *oidSz = sizeof(extCertPolicyComodoLtdOid);
                    break;
                /* FPKI OIDs */
                #endif /* WOLFSSL_FPKI */
                default:
                    break;
            }
            break;

        case oidCertAltNameType:
            switch (id) {
                case HW_NAME_OID:
                    oid = extAltNamesHwNameOid;
                    *oidSz = sizeof(extAltNamesHwNameOid);
                    break;
                default:
                    break;
            }
            break;

        case oidCertKeyUseType:
            switch (id) {
                case EKU_ANY_OID:
                    oid = extExtKeyUsageAnyOid;
                    *oidSz = sizeof(extExtKeyUsageAnyOid);
                    break;
                case EKU_SERVER_AUTH_OID:
                    oid = extExtKeyUsageServerAuthOid;
                    *oidSz = sizeof(extExtKeyUsageServerAuthOid);
                    break;
                case EKU_CLIENT_AUTH_OID:
                    oid = extExtKeyUsageClientAuthOid;
                    *oidSz = sizeof(extExtKeyUsageClientAuthOid);
                    break;
                case EKU_CODESIGNING_OID:
                    oid = extExtKeyUsageCodeSigningOid;
                    *oidSz = sizeof(extExtKeyUsageCodeSigningOid);
                    break;
                case EKU_EMAILPROTECT_OID:
                    oid = extExtKeyUsageEmailProtectOid;
                    *oidSz = sizeof(extExtKeyUsageEmailProtectOid);
                    break;
                case EKU_TIMESTAMP_OID:
                    oid = extExtKeyUsageTimestampOid;
                    *oidSz = sizeof(extExtKeyUsageTimestampOid);
                    break;
                case EKU_OCSP_SIGN_OID:
                    oid = extExtKeyUsageOcspSignOid;
                    *oidSz = sizeof(extExtKeyUsageOcspSignOid);
                    break;
                #ifdef WOLFSSL_WOLFSSH
                case EKU_SSH_CLIENT_AUTH_OID:
                    oid = extExtKeyUsageSshClientAuthOid;
                    *oidSz = sizeof(extExtKeyUsageSshClientAuthOid);
                    break;
                case EKU_SSH_MSCL_OID:
                    oid = extExtKeyUsageSshMSCLOid;
                    *oidSz = sizeof(extExtKeyUsageSshMSCLOid);
                    break;
                case EKU_SSH_KP_CLIENT_AUTH_OID:
                    oid = extExtKeyUsageSshKpClientAuthOid;
                    *oidSz = sizeof(extExtKeyUsageSshKpClientAuthOid);
                    break;
                #endif /* WOLFSSL_WOLFSSH */
                default:
                    break;
            }
            break;

        case oidKdfType:
            switch (id) {
                case PBKDF2_OID:
                    oid = pbkdf2Oid;
                    *oidSz = sizeof(pbkdf2Oid);
                    break;
                default:
                    break;
            }
            break;

        case oidPBEType:
            switch (id) {
        #if !defined(NO_SHA) && !defined(NO_RC4)
                case PBE_SHA1_RC4_128_SUM:
                case PBE_SHA1_RC4_128:
                    oid = pbeSha1RC4128;
                    *oidSz = sizeof(pbeSha1RC4128);
                    break;
        #endif
        #if !defined(NO_MD5) && !defined(NO_DES3)
                case PBE_MD5_DES_SUM:
                case PBE_MD5_DES:
                    oid = pbeMd5Des;
                    *oidSz = sizeof(pbeMd5Des);
                    break;

        #endif
        #if !defined(NO_SHA) && !defined(NO_DES3)
                case PBE_SHA1_DES_SUM:
                case PBE_SHA1_DES:
                    oid = pbeSha1Des;
                    *oidSz = sizeof(pbeSha1Des);
                    break;

        #endif
        #if !defined(NO_SHA) && !defined(NO_DES3)
                case PBE_SHA1_DES3_SUM:
                case PBE_SHA1_DES3:
                    oid = pbeSha1Des3;
                    *oidSz = sizeof(pbeSha1Des3);
                    break;
        #endif
        #if !defined(NO_SHA) && defined(WC_RC2)
                case PBE_SHA1_40RC2_CBC_SUM:
                case PBE_SHA1_40RC2_CBC:
                    oid = pbe40Rc2Cbc;
                    *oidSz = sizeof(pbe40Rc2Cbc);
                    break;
        #endif
                case PBES2_SUM:
                case PBES2:
                    oid = pbes2;
                    *oidSz = sizeof(pbes2);
                    break;
                default:
                    break;
            }
            break;

        case oidKeyWrapType:
            switch (id) {
            #ifdef WOLFSSL_AES_128
                case AES128_WRAP:
                    oid = wrapAes128Oid;
                    *oidSz = sizeof(wrapAes128Oid);
                    break;
            #endif
            #ifdef WOLFSSL_AES_192
                case AES192_WRAP:
                    oid = wrapAes192Oid;
                    *oidSz = sizeof(wrapAes192Oid);
                    break;
            #endif
            #ifdef WOLFSSL_AES_256
                case AES256_WRAP:
                    oid = wrapAes256Oid;
                    *oidSz = sizeof(wrapAes256Oid);
                    break;
            #endif
            #ifdef HAVE_PKCS7
                case PWRI_KEK_WRAP:
                    oid = wrapPwriKekOid;
                    *oidSz = sizeof(wrapPwriKekOid);
                    break;
            #endif
                default:
                    break;
            }
            break;

        case oidCmsKeyAgreeType:
            switch (id) {
            #ifndef NO_SHA
                case dhSinglePass_stdDH_sha1kdf_scheme:
                    oid = dhSinglePass_stdDH_sha1kdf_Oid;
                    *oidSz = sizeof(dhSinglePass_stdDH_sha1kdf_Oid);
                    break;
            #endif
            #ifdef WOLFSSL_SHA224
                case dhSinglePass_stdDH_sha224kdf_scheme:
                    oid = dhSinglePass_stdDH_sha224kdf_Oid;
                    *oidSz = sizeof(dhSinglePass_stdDH_sha224kdf_Oid);
                    break;
            #endif
            #ifndef NO_SHA256
                case dhSinglePass_stdDH_sha256kdf_scheme:
                    oid = dhSinglePass_stdDH_sha256kdf_Oid;
                    *oidSz = sizeof(dhSinglePass_stdDH_sha256kdf_Oid);
                    break;
            #endif
            #ifdef WOLFSSL_SHA384
                case dhSinglePass_stdDH_sha384kdf_scheme:
                    oid = dhSinglePass_stdDH_sha384kdf_Oid;
                    *oidSz = sizeof(dhSinglePass_stdDH_sha384kdf_Oid);
                    break;
            #endif
            #ifdef WOLFSSL_SHA512
                case dhSinglePass_stdDH_sha512kdf_scheme:
                    oid = dhSinglePass_stdDH_sha512kdf_Oid;
                    *oidSz = sizeof(dhSinglePass_stdDH_sha512kdf_Oid);
                    break;
            #endif
                default:
                    break;
            }
            break;

#ifndef NO_HMAC
        case oidHmacType:
            switch (id) {
        #ifdef WOLFSSL_SHA224
                case HMAC_SHA224_OID:
                    oid = hmacSha224Oid;
                    *oidSz = sizeof(hmacSha224Oid);
                    break;
        #endif
        #ifndef NO_SHA256
                case HMAC_SHA256_OID:
                    oid = hmacSha256Oid;
                    *oidSz = sizeof(hmacSha256Oid);
                    break;
        #endif
        #ifdef WOLFSSL_SHA384
                case HMAC_SHA384_OID:
                    oid = hmacSha384Oid;
                    *oidSz = sizeof(hmacSha384Oid);
                    break;
        #endif
        #ifdef WOLFSSL_SHA512
                case HMAC_SHA512_OID:
                    oid = hmacSha512Oid;
                    *oidSz = sizeof(hmacSha512Oid);
                    break;
        #endif
                default:
                    break;
            }
            break;
#endif /* !NO_HMAC */

#ifdef HAVE_LIBZ
        case oidCompressType:
            switch (id) {
                case ZLIBc:
                    oid = zlibCompress;
                    *oidSz = sizeof(zlibCompress);
                    break;
                default:
                    break;
            }
            break;
#endif /* HAVE_LIBZ */
#ifdef WOLFSSL_APACHE_HTTPD
        case oidCertNameType:
            switch (id) {
                 case WC_NID_id_on_dnsSRV:
                    oid = dnsSRVOid;
                    *oidSz = sizeof(dnsSRVOid);
                    break;
                default:
                    break;
            }
            break;
        case oidTlsExtType:
            switch (id) {
                case TLS_FEATURE_OID:
                    oid = tlsFeatureOid;
                    *oidSz = sizeof(tlsFeatureOid);
                    break;
                default:
                    break;
            }
            break;
#endif /* WOLFSSL_APACHE_HTTPD */
#ifdef WOLFSSL_CERT_REQ
        case oidCsrAttrType:
            switch (id) {
                case GIVEN_NAME_OID:
                    oid = attrGivenName;
                    *oidSz = sizeof(attrGivenName);
                    break;
                case SURNAME_OID:
                    oid = attrSurname;
                    *oidSz = sizeof(attrSurname);
                    break;
                case INITIALS_OID:
                    oid = attrInitals;
                    *oidSz = sizeof(attrInitals);
                    break;
                case DNQUALIFIER_OID:
                    oid = attrDnQualifier;
                    *oidSz = sizeof(attrDnQualifier);
                    break;
                case UNSTRUCTURED_NAME_OID:
                    oid = attrUnstructuredNameOid;
                    *oidSz = sizeof(attrUnstructuredNameOid);
                    break;
                case PKCS9_CONTENT_TYPE_OID:
                    oid = attrPkcs9ContentTypeOid;
                    *oidSz = sizeof(attrPkcs9ContentTypeOid);
                    break;
                case CHALLENGE_PASSWORD_OID:
                    oid = attrChallengePasswordOid;
                    *oidSz = sizeof(attrChallengePasswordOid);
                    break;
                case SERIAL_NUMBER_OID:
                    oid = attrSerialNumberOid;
                    *oidSz = sizeof(attrSerialNumberOid);
                    break;
                case USER_ID_OID:
                    oid = uidOid;
                    *oidSz = sizeof(uidOid);
                    break;
                case EXTENSION_REQUEST_OID:
                    oid = attrExtensionRequestOid;
                    *oidSz = sizeof(attrExtensionRequestOid);
                    break;
                default:
                    break;
            }
            break;
#endif
#ifdef WOLFSSL_SUBJ_DIR_ATTR
        case oidSubjDirAttrType:
            switch (id) {
                case SDA_DOB_OID:
                    oid = extSubjDirAttrDobOid;
                    *oidSz = sizeof(extSubjDirAttrDobOid);
                    break;
                case SDA_POB_OID:
                    oid = extSubjDirAttrPobOid;
                    *oidSz = sizeof(extSubjDirAttrPobOid);
                    break;
                case SDA_GENDER_OID:
                    oid = extSubjDirAttrGenderOid;
                    *oidSz = sizeof(extSubjDirAttrGenderOid);
                    break;
                case SDA_COC_OID:
                    oid = extSubjDirAttrCocOid;
                    *oidSz = sizeof(extSubjDirAttrCocOid);
                    break;
                case SDA_COR_OID:
                    oid = extSubjDirAttrCorOid;
                    *oidSz = sizeof(extSubjDirAttrCorOid);
                    break;
                default:
                    break;
            }
            break;
#endif /* WOLFSSL_SUBJ_DIR_ATTR */
        case oidIgnoreType:
        default:
            break;
    }

    return oid;
}

#ifdef HAVE_ECC

/* Check the OID id is for a known elliptic curve.
 *
 * @param [in]  oid  OID id.
 * @return  ECC set id on success.
 * @return  ECC_CURVE_OID_E when OID id is 0 or not supported.
 */
static int CheckCurve(word32 oid)
{
    int ret;
    word32 oidSz;

    /* Lookup OID id. */
    ret = wc_ecc_get_oid(oid, NULL, &oidSz);
    /* Check for error or zero length OID size (can't get OID for encoding). */
    if ((ret < 0) || (oidSz == 0)) {
        WOLFSSL_MSG("CheckCurve not found");
        WOLFSSL_ERROR_VERBOSE(ECC_CURVE_OID_E);
        ret = ECC_CURVE_OID_E;
    }

    /* Return ECC set id or error code. */
    return ret;
}

#endif

#ifdef HAVE_OID_ENCODING
/* Encode dotted form of OID into byte array version.
 *
 * @param [in]      in     Dotted form of OID.
 * @param [in]      inSz   Count of numbers in dotted form.
 * @param [in]      out    Buffer to hold OID.
 * @param [in, out] outSz  On in, size of buffer.
 *                         On out, number of bytes in buffer.
 * @return  0 on success
 * @return  BAD_FUNC_ARG when in or outSz is NULL.
 * @return  BUFFER_E when buffer too small.
 */
int wc_EncodeObjectId(const word16* in, word32 inSz, byte* out, word32* outSz)
{
    return EncodeObjectId(in, inSz, out, outSz);
}

int EncodeObjectId(const word16* in, word32 inSz, byte* out, word32* outSz)
{
    int i, x, len;
    word32 d, t;

    /* check args */
    if (in == NULL || outSz == NULL || inSz <= 0) {
        return BAD_FUNC_ARG;
    }

    /* compute length of encoded OID */
    d = ((word32)in[0] * 40) + in[1];
    len = 0;
    for (i = 1; i < (int)inSz; i++) {
        x = 0;
        t = d;
        while (t) {
            x++;
            t >>= 1;
        }
        len += (x / 7) + ((x % 7) ? 1 : 0) + (d == 0 ? 1 : 0);

        if (i < (int)inSz - 1) {
            d = in[i + 1];
        }
    }

    if (out) {
        /* verify length */
        if ((int)*outSz < len) {
            return BUFFER_E; /* buffer provided is not large enough */
        }

        /* calc first byte */
        d = ((word32)in[0] * 40) + in[1];

        /* encode bytes */
        x = 0;
        for (i = 1; i < (int)inSz; i++) {
            if (d) {
                int y = x, z;
                byte mask = 0;
                while (d) {
                    out[x++] = (byte)((d & 0x7F) | mask);
                    d     >>= 7;
                    mask  |= 0x80;  /* upper bit is set on all but the last byte */
                }
                /* now swap bytes y...x-1 */
                z = x - 1;
                while (y < z) {
                    mask = out[y];
                    out[y] = out[z];
                    out[z] = mask;
                    ++y;
                    --z;
                }
            }
            else {
              out[x++] = 0x00; /* zero value */
            }

            /* next word */
            if (i < (int)inSz - 1) {
                d = in[i + 1];
            }
        }
    }

    /* return length */
    *outSz = (word32)len;

    return 0;
}
#endif /* HAVE_OID_ENCODING */

#if defined(HAVE_OID_DECODING) || defined(WOLFSSL_ASN_PRINT)
/* Encode dotted form of OID into byte array version.
 *
 * @param [in]      in     Byte array containing OID.
 * @param [in]      inSz   Size of OID in bytes.
 * @param [in]      out    Array to hold dotted form of OID.
 * @param [in, out] outSz  On in, number of elements in array.
 *                         On out, count of numbers in dotted form.
 * @return  0 on success
 * @return  BAD_FUNC_ARG when in or outSz is NULL.
 * @return  BUFFER_E when dotted form buffer too small.
 */
int DecodeObjectId(const byte* in, word32 inSz, word16* out, word32* outSz)
{
    int x = 0, y = 0;
    word32 t = 0;

    /* check args */
    if (in == NULL || outSz == NULL) {
        return BAD_FUNC_ARG;
    }

    /* decode bytes */
    while (inSz--) {
        t = (t << 7) | (in[x] & 0x7F);
        if (!(in[x] & 0x80)) {
            if (y >= (int)*outSz) {
                return BUFFER_E;
            }
            if (y == 0) {
                out[0] = (word16)(t / 40);
                out[1] = (word16)(t % 40);
                y = 2;
            }
            else {
                out[y++] = (word16)t;
            }
            t = 0; /* reset tmp */
        }
        x++;
    }

    /* return length */
    *outSz = (word32)y;

    return 0;
}
#endif /* HAVE_OID_DECODING || WOLFSSL_ASN_PRINT */

/* Decode the header of a BER/DER encoded OBJECT ID.
 *
 * @param [in]      input     Buffer holding DER/BER encoded data.
 * @param [in, out] inOutIdx  On in, starting index of header.
 *                            On out, end of parsed header.
 * @param [out]     len       Number of bytes in the ASN.1 data.
 * @param [in]      maxIdx    Length of data in buffer.
 * @return  0 on success.
 * @return  BUFFER_E when there is not enough data to parse.
 * @return  ASN_PARSE_E when the tag is not a OBJECT ID or length is invalid.
 */
int GetASNObjectId(const byte* input, word32* inOutIdx, int* len, word32 maxIdx)
{
    int ret = GetASNHeader(input, ASN_OBJECT_ID, inOutIdx, len, maxIdx);
    if (ret > 0) {
        /* Only return 0 on success. */
        ret = 0;
    }
    return ret;
}

/* Set the DER/BER encoding of the ASN.1 OBJECT ID header.
 *
 * When output is NULL, calculate the header length only.
 *
 * @param [in]  len        Length of OBJECT ID data in bytes.
 * @param [out] output     Buffer to write into.
 * @return  Number of bytes added to the buffer.
 */
int SetObjectId(int len, byte* output)
{
    int idx = 0;

    if (output) {
        /* Write out tag. */
        output[idx] = ASN_OBJECT_ID;
    }
    /* Skip tag. */
    idx += ASN_TAG_SZ;
    /* Encode length - passing NULL for output will not encode. */
    idx += (int)SetLength((word32)len, output ? output + idx : NULL);

    /* Return index after header. */
    return idx;
}

#ifdef ASN_DUMP_OID
/* Dump the OID information.
 *
 * Decode the OID too if function available.
 *
 * @param [in] oidData  OID data from buffer.
 * @param [in] oidSz    Size of OID data in buffer.
 * @param [in] oid      OID id.
 * @param [in] oidType  Type of OID.
 * @return  0 on success.
 * @return  BUFFER_E when not enough bytes for proper decode.
 *          (HAVE_OID_DECODING)
 */
static int DumpOID(const byte* oidData, word32 oidSz, word32 oid,
                   word32 oidType)
{
    int    ret = 0;
    word32 i;

    /* support for dumping OID information */
    printf("OID (Type %d, Sz %d, Sum %d): ", oidType, oidSz, oid);
    /* Dump bytes in decimal. */
    for (i = 0; i < oidSz; i++) {
        printf("%d, ", oidData[i]);
    }
    printf("\n");
    /* Dump bytes in hexadecimal. */
    for (i = 0; i < oidSz; i++) {
        printf("%02x, ", oidData[i]);
    }
    printf("\n");

    #ifdef HAVE_OID_DECODING
    {
        word16 decOid[MAX_OID_SZ];
        word32 decOidSz = sizeof(decOid);
        /* Decode the OID into dotted form. */
        ret = DecodeObjectId(oidData, oidSz, decOid, &decOidSz);
        if (ret == 0) {
            printf("  Decoded (Sz %d): ", decOidSz);
            for (i=0; i<decOidSz; i++) {
                printf("%d.", decOid[i]);
            }
            printf("\n");
        }
        else {
            printf("DecodeObjectId failed: %d\n", ret);
        }
    }
    #endif /* HAVE_OID_DECODING */

    return ret;
}
#endif /* ASN_DUMP_OID */

#ifdef WOLFSSL_OLD_OID_SUM
#ifdef WOLFSSL_FPKI
/* Handles the large number of collisions from FPKI certificate policy
 * OID sums.  Returns a special value (100000 + actual sum) if a
 * collision is detected.
 * @param [in]      oid      Buffer holding OID.
 * @param [in]      oidSz    Length of OID data in buffer.
 * @param [in]      oidSum   The sum of the OID being passed in.
 */
static word32 fpkiCertPolOid(const byte* oid, word32 oidSz, word32 oidSum) {

    switch (oidSum) {
        case CP_ADO_MEDIUM_OID:
            if ((word32)sizeof(extCertPolicyComodoLtdOid) == (word32)oidSz &&
            XMEMCMP(oid, extCertPolicyComodoLtdOid,
            sizeof(extCertPolicyComodoLtdOid)) == 0)
                return CP_COMODO_OID;
            break;
        case CP_FPKI_HIGH_ASSURANCE_OID:
            if ((word32)sizeof(extCertPolicyStateBasicOid) == (word32)oidSz &&
            XMEMCMP(oid, extCertPolicyStateBasicOid,
            sizeof(extCertPolicyStateBasicOid)) == 0)
                return CP_STATE_BASIC_OID;
            break;
        case CP_FPKI_COMMON_DEVICES_HARDWARE_OID:
            if ((word32)sizeof(extCertPolicyDodPeerInteropOid) == (word32)oidSz &&
            XMEMCMP(oid, extCertPolicyDodPeerInteropOid,
            sizeof(extCertPolicyDodPeerInteropOid)) == 0)
                return CP_DOD_PEER_INTEROP_OID;
            break;
        case CP_FPKI_PIV_AUTH_HW_OID:
            if ((word32)sizeof(extCertPolicyDodMediumNpe112Oid) == (word32)oidSz &&
            XMEMCMP(oid, extCertPolicyDodMediumNpe112Oid,
            sizeof(extCertPolicyDodMediumNpe112Oid)) == 0)
                return CP_DOD_MEDIUM_NPE_112_OID;
            else if ((word32)sizeof(extCertPolicyStateMediumDeviceHardwareOid) == (word32)oidSz &&
            XMEMCMP(oid, extCertPolicyStateMediumDeviceHardwareOid,
            sizeof(extCertPolicyStateMediumDeviceHardwareOid)) == 0)
                return CP_STATE_MEDDEVHW_OID;
            break;
        case CP_FPKI_PIVI_AUTH_OID:
            if ((word32)sizeof(extCertPolicyDodMedium128Oid) == (word32)oidSz &&
            XMEMCMP(oid, extCertPolicyDodMedium128Oid,
            sizeof(extCertPolicyDodMedium128Oid)) == 0)
                return CP_DOD_MEDIUM_128_OID;
            break;
        case CP_FPKI_COMMON_PIVI_CONTENT_SIGNING_OID:
            if ((word32)sizeof(extCertPolicyDodMediumHardware112Oid) == (word32)oidSz &&
                XMEMCMP(oid, extCertPolicyDodMediumHardware112Oid,
                sizeof(extCertPolicyDodMediumHardware112Oid)) == 0)
                    return CP_DOD_MEDIUM_HARDWARE_112_OID;
            else if ((word32)sizeof(extCertPolicyCertipathHighhwOid) == (word32)oidSz &&
                XMEMCMP(oid, extCertPolicyCertipathHighhwOid,
                sizeof(extCertPolicyCertipathHighhwOid)) == 0)
                    return CP_CERTIPATH_HIGHHW_OID;
            break;
        case CP_DOD_MEDIUM_OID:
            if ((word32)sizeof(extCertPolicyEcaMediumOid) == (word32)oidSz &&
            XMEMCMP(oid, extCertPolicyEcaMediumOid,
            sizeof(extCertPolicyEcaMediumOid)) == 0)
                return CP_ECA_MEDIUM_OID;
            break;
        case CP_FPKI_COMMON_AUTH_OID:
            if ((word32)sizeof(extCertPolicyEcaMediumSha256Oid) == (word32)oidSz &&
            XMEMCMP(oid, extCertPolicyEcaMediumSha256Oid,
            sizeof(extCertPolicyEcaMediumSha256Oid)) == 0)
                return CP_ECA_MEDIUM_SHA256_OID;
            break;
        case CP_FPKI_MEDIUM_HARDWARE_OID:
            if ((word32)sizeof(extCertPolicyEcaMediumTokenOid) == (word32)oidSz &&
            XMEMCMP(oid, extCertPolicyEcaMediumTokenOid,
            sizeof(extCertPolicyEcaMediumTokenOid)) == 0)
                return CP_ECA_MEDIUM_TOKEN_OID;
            else if ((word32)sizeof(extCertPolicyTreasuryPiviHardwareOid) == (word32)oidSz &&
            XMEMCMP(oid, extCertPolicyTreasuryPiviHardwareOid,
            sizeof(extCertPolicyTreasuryPiviHardwareOid)) == 0)
                return CP_TREAS_PIVI_HW_OID;
            break;
        case CP_DOD_MEDIUM_HARDWARE_OID:
            if ((word32)sizeof(extCertPolicyEcaMediumTokenSha256Oid) == (word32)oidSz &&
            XMEMCMP(oid, extCertPolicyEcaMediumTokenSha256Oid,
            sizeof(extCertPolicyEcaMediumTokenSha256Oid)) == 0)
                return CP_ECA_MEDIUM_TOKEN_SHA256_OID;
            else if ((word32)sizeof(extCertPolicyTreasuryPiviContentSigningOid) == (word32)oidSz &&
            XMEMCMP(oid, extCertPolicyTreasuryPiviContentSigningOid,
            sizeof(extCertPolicyTreasuryPiviContentSigningOid)) == 0)
                return CP_TREAS_PIVI_CONTENT_OID;
            break;
        case CP_DOD_PIV_AUTH_OID:
            if ((word32)sizeof(extCertPolicyEcaMediumHardwarePiviOid) == (word32)oidSz &&
            XMEMCMP(oid, extCertPolicyEcaMediumHardwarePiviOid,
            sizeof(extCertPolicyEcaMediumHardwarePiviOid)) == 0)
                return CP_ECA_MEDIUM_HARDWARE_PIVI_OID;
            else if ((word32)sizeof(extCertPolicyStateMedHwOid) == (word32)oidSz &&
            XMEMCMP(oid, extCertPolicyStateMedHwOid,
            sizeof(extCertPolicyStateMedHwOid)) == 0)
                return CP_STATE_MEDHW_OID;
            break;
        case CP_FPKI_COMMON_HARDWARE_OID:
            if ((word32)sizeof(extCertPolicyStateHighOid) == (word32)oidSz &&
            XMEMCMP(oid, extCertPolicyStateHighOid,
            sizeof(extCertPolicyStateHighOid)) == 0)
                return CP_STATE_HIGH_OID;
            else if ((word32)sizeof(extCertPolicyTreasuryHighOid) == (word32)oidSz &&
            XMEMCMP(oid, extCertPolicyTreasuryHighOid,
            sizeof(extCertPolicyTreasuryHighOid)) == 0)
                return CP_TREAS_HIGH_OID;
            break;
        case CP_ECA_MEDIUM_HARDWARE_OID:
            if ((word32)sizeof(extCertPolicyExostarMediumHardwareSha2Oid) == (word32)oidSz &&
            XMEMCMP(oid, extCertPolicyExostarMediumHardwareSha2Oid,
            sizeof(extCertPolicyExostarMediumHardwareSha2Oid)) == 0)
                return CP_EXOSTAR_MEDIUMHW_SHA2_OID;
            break;
        case CP_ADO_HIGH_OID:
            if ((word32)sizeof(extCertPolicyAdoResourceMediumAssuranceOid) == (word32)oidSz &&
            XMEMCMP(oid, extCertPolicyAdoResourceMediumAssuranceOid,
            sizeof(extCertPolicyAdoResourceMediumAssuranceOid)) == 0)
                return CP_ADO_RESOURCE_MEDIUM_OID;
            break;
        case CP_DOD_ADMIN_OID:
            if ((word32)sizeof(extCertPolicyCarillonAivcontentOid) == (word32)oidSz &&
            XMEMCMP(oid, extCertPolicyCarillonAivcontentOid,
            sizeof(extCertPolicyCarillonAivcontentOid)) == 0)
                return CP_CARILLON_AIVCONTENT_OID;
            break;
        case CP_TREAS_MEDIUMHW_OID:
            if ((word32)sizeof(extCertPolicyStateModerateOid) == (word32)oidSz &&
            XMEMCMP(oid, extCertPolicyStateModerateOid,
            sizeof(extCertPolicyStateModerateOid)) == 0)
                return CP_STATE_MODERATE_OID;
            break;
        case CP_CIS_ICECAP_HW_OID:
            if ((word32)sizeof(extCertPolicyNlModIrrefutabilityOid) == (word32)oidSz &&
            XMEMCMP(oid, extCertPolicyNlModIrrefutabilityOid,
            sizeof(extCertPolicyNlModIrrefutabilityOid)) == 0)
                return CP_NL_MOD_IRREFUT_OID;
            break;
        case CP_DOD_MEDIUM_192_OID:
            if ((word32)sizeof(extCertPolicyCertipathMediumhwOid) == (word32)oidSz &&
            XMEMCMP(oid, extCertPolicyCertipathMediumhwOid,
            sizeof(extCertPolicyCertipathMediumhwOid)) == 0)
                return CP_CERTIPATH_MEDIUMHW_OID;
            break;
        case CP_CARILLON_AIVHW_OID:
            if ((word32)sizeof(extCertPolicyCertipathVarMediumhwOid) == (word32)oidSz &&
            XMEMCMP(oid, extCertPolicyCertipathVarMediumhwOid,
            sizeof(extCertPolicyCertipathVarMediumhwOid)) == 0)
                return CP_CERTIPATH_VAR_MEDIUMHW_OID;
            break;
        case CP_ISRG_DOMAIN_VALID:
            if ((word32)sizeof(extCertPolicyEcaContentSigningPiviOid) == (word32)oidSz &&
            XMEMCMP(oid, extCertPolicyEcaContentSigningPiviOid,
            sizeof(extCertPolicyEcaContentSigningPiviOid)) == 0)
                return CP_ECA_CONTENT_SIGNING_PIVI_OID;
            break;
        default:
            break;
    }

    return 0;
}
#endif
#endif /* WOLFSSL_OLD_OID_SUM */

word32 wc_oid_sum(const byte* input, int length)
{
    int i;
    word32 oid = 0;
#ifndef WOLFSSL_OLD_OID_SUM
    int shift = 0;
#endif

    /* Check for valid input. */
    if (input == NULL || length > MAX_OID_SZ) {
        WOLFSSL_MSG("wc_oid_sum: invalid args");
        return 0;
    }

    /* Sum it up for now. */
    for (i = 0; i < length; i++) {
    #ifdef WOLFSSL_OLD_OID_SUM
        oid += (word32)input[i];
    #else
        oid ^= ((word32)(~input[i])) << shift;
        shift = (shift + 8) & 0x1f;
    #endif
    }
#ifndef WOLFSSL_OLD_OID_SUM
    oid &= 0x7fffffff;
#endif

    return oid;
}

/* Get the OID data and verify it is of the type specified when compiled in.
 *
 * @param [in]      input     Buffer holding OID.
 * @param [in, out] inOutIdx  On in, starting index of OID.
 *                            On out, end of parsed OID.
 * @param [out]     oid       OID id.
 * @param [in]      oidType   Expected type of OID. Define NO_VERIFY_OID to
 *                            not compile in check.
 * @param [in]      length    Length of OID data in buffer.
 * @return  0 on success.
 * @return  ASN_UNKNOWN_OID_E when OID is not recognized.
 * @return  BUFFER_E when not enough bytes for proper decode. (ASN_DUMP_OID and
 *          HAVE_OID_DECODING)
 */
static int GetOID(const byte* input, word32* inOutIdx, word32* oid,
                  word32 oidType, int length)
{
    int    ret = 0;
    word32 idx = *inOutIdx;
#ifndef NO_VERIFY_OID
    word32 actualOidSz;
    const byte* actualOid;
    const byte* checkOid = NULL;
    word32 checkOidSz;
#endif /* NO_VERIFY_OID */
#ifdef WOLFSSL_OLD_OID_SUM
#if defined(HAVE_SPHINCS) || defined(WOLFSSL_FPKI)
    word32 found_collision = 0;
#endif
#endif
    (void)oidType;
    *oid = 0;

#if !defined(NO_VERIFY_OID) || defined(WOLFSSL_FPKI)
    /* Keep references to OID data and length for check. */
    actualOid = &input[idx];
    actualOidSz = (word32)length;
#endif /* NO_VERIFY_OID */

#ifdef WOLFSSL_OLD_OID_SUM
#if defined(HAVE_SPHINCS)
    /* Since we are summing it up, there could be collisions...and indeed there
     * are: SPHINCS_FAST_LEVEL1 and SPHINCS_FAST_LEVEL3.
     *
     * We will look for the special case of SPHINCS_FAST_LEVEL3 and set *oid to
     * 283 instead of 281; 282 is taken.
     *
     * These hacks will hopefully disappear when new standardized OIDs appear.
     */
    if (idx + (word32)sizeof(sigSphincsFast_Level3Oid) < (word32)length &&
            XMEMCMP(&input[idx], sigSphincsFast_Level3Oid,
               sizeof(sigSphincsFast_Level3Oid)) == 0) {
        found_collision = SPHINCS_FAST_LEVEL3k;
    }
#endif /* HAVE_SPHINCS */
#endif

    *oid = wc_oid_sum(actualOid, (int)actualOidSz);
    idx += actualOidSz;

#ifdef WOLFSSL_OLD_OID_SUM
#ifdef WOLFSSL_FPKI
    /* Due to the large number of OIDs for FPKI certificate policy, there
       are multiple collsisions.  Handle them in a dedicated function,
       if a collision is detected, the OID is adjusted. */
    if (oidType == oidCertPolicyType) {
        found_collision = fpkiCertPolOid(actualOid, actualOidSz, *oid);
    }
#endif

#if defined(HAVE_SPHINCS) || defined(WOLFSSL_FPKI)
    if (found_collision) {
        *oid = found_collision;
    }
#endif /* HAVE_SPHINCS */
#endif

    /* Return the index after the OID data. */
    *inOutIdx = idx;

#ifndef NO_VERIFY_OID
    /* 'Ignore' type means we don't care which OID it is. */
    if (oidType != oidIgnoreType) {
        /* Get the OID data for the id-type. */
        checkOid = OidFromId(*oid, oidType, &checkOidSz);

#ifdef WOLFSSL_OLD_OID_SUM
    #if defined(WOLFSSL_FPKI)
        /* Handle OID sum collision of
            AES256CBCb (454) 2.16.840.1.101.3.4.1.42
            CP_FPKI_PIV_AUTH_HW_OID (454) 2.16.840.1.101.3.2.1.3.41
        */
        #if defined(HAVE_AES_CBC) && defined(WOLFSSL_AES_256)
        if ((actualOidSz == (word32)sizeof(blkAes256CbcOid)) &&
                (XMEMCMP(actualOid, blkAes256CbcOid,
                 sizeof(blkAes256CbcOid)) == 0)) {

            checkOid   = blkAes256CbcOid;
            checkOidSz = sizeof(blkAes256CbcOid);
        }
        #endif /* HAVE_AES_CBC */
    #endif /* WOLFSSL_FPKI */
#endif

    #ifdef ASN_DUMP_OID
        /* Dump out the data for debug. */
        ret = DumpOID(actualOid, actualOidSz, *oid, oidType);
    #endif

        /* TODO: Want to fail when checkOid is NULL.
         * Can't as too many situations where unknown OID is to be
         * supported. Extra parameter for must not be NULL?
         */
        /* Check that the OID data matches what we found for the OID id. */
        if ((ret == 0) && (checkOid != NULL) && ((checkOidSz != actualOidSz) ||
                (XMEMCMP(actualOid, checkOid, checkOidSz) != 0))) {
            WOLFSSL_MSG("OID Check Failed");
            WOLFSSL_ERROR_VERBOSE(ASN_UNKNOWN_OID_E);
            ret = ASN_UNKNOWN_OID_E;
        }
    }
#endif /* NO_VERIFY_OID */

    return ret;
}

#ifdef WOLFSSL_ASN_TEMPLATE
/* ASN.1 template for an OBJECT_ID. */
static const ASNItem objectIdASN[] = {
/* OID */ { 0, ASN_OBJECT_ID, 0, 0, 0 }
};
enum {
    OBJECTIDASN_IDX_OID = 0
};

/* Number of items in ASN.1 template for an OBJECT_ID. */
#define objectIdASN_Length (sizeof(objectIdASN) / sizeof(ASNItem))
#endif

/* Get the OID id/sum from the BER encoded OBJECT_ID.
 *
 * @param [in]      input     Buffer holding BER encoded data.
 * @param [in, out] inOutIdx  On in, start of OBJECT_ID.
 *                            On out, start of ASN.1 item after OBJECT_ID.
 * @param [out]     oid       Id of OID in OBJECT_ID data.
 * @param [in]      oidType   Type of OID to expect.
 * @param [in]      maxIdx    Maximum index of data in buffer.
 * @return  0 on success.
 * @return  ASN_PARSE_E when encoding is invalid.
 * @return  ASN_UNKNOWN_OID_E when the OID cannot be verified.
 */
int GetObjectId(const byte* input, word32* inOutIdx, word32* oid,
                                  word32 oidType, word32 maxIdx)
{
#ifndef WOLFSSL_ASN_TEMPLATE
    int ret, length;

    WOLFSSL_ENTER("GetObjectId");

    ret = GetASNObjectId(input, inOutIdx, &length, maxIdx);
    if (ret != 0)
        return ret;

    return GetOID(input, inOutIdx, oid, oidType, length);
#else
    ASNGetData dataASN[objectIdASN_Length];
    int ret;

    WOLFSSL_ENTER("GetObjectId");

    /* Clear dynamic data and set OID type expected. */
    XMEMSET(dataASN, 0, sizeof(dataASN));
    GetASN_OID(&dataASN[OBJECTIDASN_IDX_OID], oidType);
    /* Decode OBJECT_ID. */
    ret = GetASN_Items(objectIdASN, dataASN, objectIdASN_Length, 0, input,
                       inOutIdx, maxIdx);
    if (ret == 0) {
        /* Return the id/sum. */
        *oid = dataASN[OBJECTIDASN_IDX_OID].data.oid.sum;
    }

    return ret;
#endif /* WOLFSSL_ASN_TEMPLATE */
}

#ifndef WOLFSSL_ASN_TEMPLATE
static int SkipObjectId(const byte* input, word32* inOutIdx, word32 maxIdx)
{
    word32 idx = *inOutIdx;
    int    length;
    int ret;

    ret = GetASNObjectId(input, &idx, &length, maxIdx);
    if (ret != 0)
        return ret;

    idx += (word32)length;
    *inOutIdx = idx;

    return 0;
}
#endif

#ifdef WOLFSSL_ASN_TEMPLATE
/* ASN.1 template for an algorithm identifier. */
static const ASNItem algoIdASN[] = {
/*  SEQ  */    { 0, ASN_SEQUENCE, 1, 1, 0 },
/*  OID  */        { 1, ASN_OBJECT_ID, 0, 0, 0 },
/*  NULL */        { 1, ASN_TAG_NULL, 0, 0, 1 },
};
enum {
    ALGOIDASN_IDX_SEQ = 0,
    ALGOIDASN_IDX_OID,
    ALGOIDASN_IDX_NULL
};

/* Number of items in ASN.1 template for an algorithm identifier. */
#define algoIdASN_Length (sizeof(algoIdASN) / sizeof(ASNItem))
#endif

static int GetAlgoIdImpl(const byte* input, word32* inOutIdx, word32* oid,
                     word32 oidType, word32 maxIdx, byte *absentParams)
{
#ifndef WOLFSSL_ASN_TEMPLATE
    int    length;
    word32 idx = *inOutIdx;
    int    ret;
    *oid = 0;

    WOLFSSL_ENTER("GetAlgoId");

    if (GetSequence(input, &idx, &length, maxIdx) < 0)
        return ASN_PARSE_E;

    if (GetObjectId(input, &idx, oid, oidType, maxIdx) < 0)
        return ASN_OBJECT_ID_E;

    /* could have NULL tag and 0 terminator, but may not */
    if (idx < maxIdx) {
        word32 localIdx = idx; /*use localIdx to not advance when checking tag*/
        byte   tag;

        if (GetASNTag(input, &localIdx, &tag, maxIdx) == 0) {
            if (tag == ASN_TAG_NULL) {
                ret = GetASNNull(input, &idx, maxIdx);
                if (ret != 0)
                    return ret;

                if (absentParams != NULL) {
                    *absentParams = FALSE;
                }
            }
        }
    }

    *inOutIdx = idx;

    return 0;
#else
    DECL_ASNGETDATA(dataASN, algoIdASN_Length);
    int ret = 0;

    WOLFSSL_ENTER("GetAlgoId");

    CALLOC_ASNGETDATA(dataASN, algoIdASN_Length, ret, NULL);
    if (ret == 0) {
        /* Set OID type expected. */
        GetASN_OID(&dataASN[ALGOIDASN_IDX_OID], oidType);
        /* Decode the algorithm identifier. */
        ret = GetASN_Items(algoIdASN, dataASN, algoIdASN_Length, 0, input,
            inOutIdx, maxIdx);
    }
    if (ret == 0) {
        /* Return the OID id/sum. */
        *oid = dataASN[ALGOIDASN_IDX_OID].data.oid.sum;

        if ((absentParams != NULL) &&
            (dataASN[ALGOIDASN_IDX_NULL].tag == ASN_TAG_NULL)) {
            *absentParams = FALSE;
        }
    }

    FREE_ASNGETDATA(dataASN, NULL);
    return ret;
#endif /* WOLFSSL_ASN_TEMPLATE */
}

/* Get the OID id/sum from the BER encoding of an algorithm identifier.
 *
 * NULL tag is skipped if present.
 *
 * @param [in]      input     Buffer holding BER encoded data.
 * @param [in, out] inOutIdx  On in, start of algorithm identifier.
 *                            On out, start of ASN.1 item after algorithm id.
 * @param [out]     oid       Id of OID in algorithm identifier data.
 * @param [in]      oidType   Type of OID to expect.
 * @param [in]      maxIdx    Maximum index of data in buffer.
 * @return  0 on success.
 * @return  ASN_PARSE_E when encoding is invalid.
 * @return  ASN_UNKNOWN_OID_E when the OID cannot be verified.
 */
int GetAlgoId(const byte* input, word32* inOutIdx, word32* oid,
                     word32 oidType, word32 maxIdx)
{
    return GetAlgoIdImpl(input, inOutIdx, oid, oidType, maxIdx, NULL);
}

int GetAlgoIdEx(const byte* input, word32* inOutIdx, word32* oid,
                     word32 oidType, word32 maxIdx, byte *absentParams)
{
    /* Assume absent until proven otherwise */
    if (absentParams != NULL) {
        *absentParams = TRUE;
    }

    return GetAlgoIdImpl(input, inOutIdx, oid, oidType, maxIdx, absentParams);
}

#ifndef NO_RSA

#ifdef WC_RSA_PSS
/* RFC 8017 - PKCS #1 has RSA PSS parameter ASN definition. */

/* Convert a hash OID to a hash type.
 *
 * @param  [in]   oid   Hash OID.
 * @param  [out]  type  Hash type.
 * @return  0 on success.
 * @return  ASN_PARSE_E when hash OID not supported for RSA PSS.
 */
static int RsaPssHashOidToType(word32 oid, enum wc_HashType* type)
{
    int ret = 0;

    switch (oid) {
    /* SHA-1 is missing as it is the default is not allowed to appear. */
#ifdef WOLFSSL_SHA224
    case SHA224h:
        *type = WC_HASH_TYPE_SHA224;
        break;
#endif
#ifndef NO_SHA256
    case SHA256h:
        *type = WC_HASH_TYPE_SHA256;
        break;
#endif
#ifdef WOLFSSL_SHA384
    case SHA384h:
        *type = WC_HASH_TYPE_SHA384;
        break;
#endif
#ifdef WOLFSSL_SHA512
    case SHA512h:
        *type = WC_HASH_TYPE_SHA512;
        break;
    /* TODO: SHA512_224h */
    /* TODO: SHA512_256h */
#endif
    default:
        ret = ASN_PARSE_E;
        break;
    }

    return ret;
}

/* Convert a hash OID to a MGF1 type.
 *
 * @param  [in]   oid   Hash OID.
 * @param  [out]  mgf   MGF type.
 * @return  0 on success.
 * @return  ASN_PARSE_E when hash OID not supported for RSA PSS.
 */
static int RsaPssHashOidToMgf1(word32 oid, int* mgf)
{
    int ret = 0;

    switch (oid) {
    /* SHA-1 is missing as it is the default is not allowed to appear. */
#ifdef WOLFSSL_SHA224
    case SHA224h:
        *mgf = WC_MGF1SHA224;
        break;
#endif
#ifndef NO_SHA256
    case SHA256h:
        *mgf = WC_MGF1SHA256;
        break;
#endif
#ifdef WOLFSSL_SHA384
    case SHA384h:
        *mgf = WC_MGF1SHA384;
        break;
#endif
#ifdef WOLFSSL_SHA512
    case SHA512h:
        *mgf = WC_MGF1SHA512;
        break;
    /* TODO: SHA512_224h */
    /* TODO: SHA512_256h */
#endif
    default:
        ret = ASN_PARSE_E;
        break;
    }

    return ret;
}

#if !defined(NO_CERTS) && !defined(NO_ASN_CRYPT)

/* Convert a hash OID to a fake signature OID.
 *
 * @param  [in]   oid     Hash OID.
 * @param  [out]  sigOid  Signature OID to pass wto HashForSignature().
 * @return  0 on success.
 * @return  ASN_PARSE_E when hash OID not supported for RSA PSS.
 */
static int RsaPssHashOidToSigOid(word32 oid, word32* sigOid)
{
    int ret = 0;

    switch (oid) {
#ifndef NO_SHA
    case WC_HASH_TYPE_SHA:
        *sigOid = CTC_SHAwRSA;
        break;
#endif
#ifdef WOLFSSL_SHA224
    case WC_HASH_TYPE_SHA224:
        *sigOid = CTC_SHA224wRSA;
        break;
#endif
#ifndef NO_SHA256
    case WC_HASH_TYPE_SHA256:
        *sigOid = CTC_SHA256wRSA;
        break;
#endif
#ifdef WOLFSSL_SHA384
    case WC_HASH_TYPE_SHA384:
        *sigOid = CTC_SHA384wRSA;
        break;
#endif
#ifdef WOLFSSL_SHA512
    case WC_HASH_TYPE_SHA512:
        *sigOid = CTC_SHA512wRSA;
        break;
#endif
    /* TODO: SHA512_224h */
    /* TODO: SHA512_256h */
    /* Not supported by HashForSignature() */
    default:
        ret = ASN_PARSE_E;
        break;
    }

    return ret;
}
#endif

#ifdef WOLFSSL_ASN_TEMPLATE
/* ASN tag for hashAlgorithm. */
#define ASN_TAG_RSA_PSS_HASH        (ASN_CONTEXT_SPECIFIC | 0)
/* ASN tag for maskGenAlgorithm. */
#define ASN_TAG_RSA_PSS_MGF         (ASN_CONTEXT_SPECIFIC | 1)
/* ASN tag for saltLength. */
#define ASN_TAG_RSA_PSS_SALTLEN     (ASN_CONTEXT_SPECIFIC | 2)
/* ASN tag for trailerField. */
#define ASN_TAG_RSA_PSS_TRAILER     (ASN_CONTEXT_SPECIFIC | 3)

/* ASN.1 template for RSA PSS parameters. */
static const ASNItem rsaPssParamsASN[] = {
/*  SEQ         */  { 0, ASN_SEQUENCE, 1, 1, 0 },
/*  HASH        */      { 1, ASN_TAG_RSA_PSS_HASH, 1, 1, 1 },
/*  HASHSEQ     */          { 2, ASN_SEQUENCE, 1, 1, 0 },
/*  HASHOID     */              { 3, ASN_OBJECT_ID, 0, 0, 0 },
/*  HASHNULL    */              { 3, ASN_TAG_NULL, 0, 0, 1 },
/*  MGF         */      { 1, ASN_TAG_RSA_PSS_MGF, 1, 1, 1 },
/*  MGFSEQ      */          { 2, ASN_SEQUENCE, 1, 1, 0 },
/*  MGFOID      */              { 3, ASN_OBJECT_ID, 0, 0, 0 },
/*  MGFPARAM    */              { 3, ASN_SEQUENCE, 1, 1, 0 },
/*  MGFHOID     */                  { 4, ASN_OBJECT_ID, 0, 0, 0 },
/*  MGFHNULL    */                  { 4, ASN_TAG_NULL, 0, 0, 1 },
/*  SALTLEN     */      { 1, ASN_TAG_RSA_PSS_SALTLEN, 1, 1, 1 },
/*  SALTLENINT  */          { 2, ASN_INTEGER, 0, 0, 0 },
/*  TRAILER     */      { 1, ASN_TAG_RSA_PSS_TRAILER, 1, 1, 1 },
/*  TRAILERINT  */          { 2, ASN_INTEGER, 0, 0, 0 },
};
enum {
    RSAPSSPARAMSASN_IDX_SEQ = 0,
    RSAPSSPARAMSASN_IDX_HASH,
    RSAPSSPARAMSASN_IDX_HASHSEQ,
    RSAPSSPARAMSASN_IDX_HASHOID,
    RSAPSSPARAMSASN_IDX_HASHNULL,
    RSAPSSPARAMSASN_IDX_MGF,
    RSAPSSPARAMSASN_IDX_MGFSEQ,
    RSAPSSPARAMSASN_IDX_MGFOID,
    RSAPSSPARAMSASN_IDX_MGFPARAM,
    RSAPSSPARAMSASN_IDX_MGFHOID,
    RSAPSSPARAMSASN_IDX_MGFHNULL,
    RSAPSSPARAMSASN_IDX_SALTLEN,
    RSAPSSPARAMSASN_IDX_SALTLENINT,
    RSAPSSPARAMSASN_IDX_TRAILER,
    RSAPSSPARAMSASN_IDX_TRAILERINT
};

/* Number of items in ASN.1 template for an algorithm identifier. */
#define rsaPssParamsASN_Length (sizeof(rsaPssParamsASN) / sizeof(ASNItem))
#else
/* ASN tag for hashAlgorithm. */
#define ASN_TAG_RSA_PSS_HASH        (ASN_CONTEXT_SPECIFIC | ASN_CONSTRUCTED | 0)
/* ASN tag for maskGenAlgorithm. */
#define ASN_TAG_RSA_PSS_MGF         (ASN_CONTEXT_SPECIFIC | ASN_CONSTRUCTED | 1)
/* ASN tag for saltLength. */
#define ASN_TAG_RSA_PSS_SALTLEN     (ASN_CONTEXT_SPECIFIC | ASN_CONSTRUCTED | 2)
/* ASN tag for trailerField. */
#define ASN_TAG_RSA_PSS_TRAILER     (ASN_CONTEXT_SPECIFIC | ASN_CONSTRUCTED | 3)
#endif

/* Decode the RSA PSS parameters.
 *
 * @param  [in]   params   Buffer holding BER encoded RSA PSS parameters.
 * @param  [in]   sz       Size of data in buffer in bytes.
 * @param  [out]  hash     Hash algorithm to use on message.
 * @param  [out]  mgf      MGF algorithm to use with PSS padding.
 * @param  [out]  saltLen  Length of salt in PSS padding.
 * @return  BAD_FUNC_ARG when the params is NULL.
 * @return  ASN_PARSE_E when the decoding fails.
 * @return  0 on success.
 */
static int DecodeRsaPssParams(const byte* params, word32 sz,
    enum wc_HashType* hash, int* mgf, int* saltLen)
{
#ifndef WOLFSSL_ASN_TEMPLATE
    int ret = 0;
    word32 idx = 0;
    int len = 0;
    word32 oid = 0;
    byte tag;
    int length;

    if (params == NULL) {
        ret = BAD_FUNC_ARG;
    }
    if ((ret == 0) && (GetSequence_ex(params, &idx, &len, sz, 1) < 0)) {
        ret = ASN_PARSE_E;
    }
    if (ret == 0) {
        if ((idx < sz) && (params[idx] == ASN_TAG_RSA_PSS_HASH)) {
            /* Hash algorithm to use on message. */
            if (GetHeader(params, &tag, &idx, &length, sz, 0) < 0) {
                ret = ASN_PARSE_E;
            }
            if (ret == 0) {
                if (GetAlgoId(params, &idx, &oid, oidHashType, sz) < 0) {
                    ret = ASN_PARSE_E;
                }
            }
            if (ret == 0) {
                ret = RsaPssHashOidToType(oid, hash);
            }
        }
        else {
            /* Default hash algorithm. */
            *hash = WC_HASH_TYPE_SHA;
        }
    }
    if (ret == 0) {
        if ((idx < sz) && (params[idx] == ASN_TAG_RSA_PSS_MGF)) {
            /* MGF and hash algorithm to use with padding. */
            if (GetHeader(params, &tag, &idx, &length, sz, 0) < 0) {
                ret = ASN_PARSE_E;
            }
            if (ret == 0) {
                if (GetAlgoId(params, &idx, &oid, oidIgnoreType, sz) < 0) {
                    ret = ASN_PARSE_E;
                }
            }
            if ((ret == 0) && (oid != MGF1_OID)) {
                ret = ASN_PARSE_E;
            }
            if (ret == 0) {
                ret = GetAlgoId(params, &idx, &oid, oidHashType, sz);
                if (ret == 0) {
                    ret = RsaPssHashOidToMgf1(oid, mgf);
                }
            }
        }
        else {
            /* Default MGF/Hash algorithm. */
            *mgf = WC_MGF1SHA1;
        }
    }
    if (ret == 0) {
        if ((idx < sz) && (params[idx] == ASN_TAG_RSA_PSS_SALTLEN)) {
            /* Salt length to use with padding. */
            if (GetHeader(params, &tag, &idx, &length, sz, 0) < 0) {
                ret = ASN_PARSE_E;
            }
            if (ret == 0) {
                ret = GetInteger16Bit(params, &idx, sz);
                if (ret >= 0) {
                    *saltLen = ret;
                    ret = 0;
                }
            }
        }
        else {
            /* Default salt length. */
            *saltLen = 20;
        }
    }
    if (ret == 0) {
        if ((idx < sz) && (params[idx] == ASN_TAG_RSA_PSS_TRAILER)) {
            /* Unused - trialerField. */
            if (GetHeader(params, &tag, &idx, &length, sz, 0) < 0) {
                ret = ASN_PARSE_E;
            }
            if (ret == 0) {
                ret = GetInteger16Bit(params, &idx, sz);
                if (ret > 0) {
                    ret = 0;
                }
            }
        }
    }
    if ((ret == 0) && (idx != sz)) {
        ret = ASN_PARSE_E;
    }

    return ret;
#else
    DECL_ASNGETDATA(dataASN, rsaPssParamsASN_Length);
    int ret = 0;
    word16 sLen = 20;

    if (params == NULL) {
        ret = BAD_FUNC_ARG;
    }

    CALLOC_ASNGETDATA(dataASN, rsaPssParamsASN_Length, ret, NULL);
    if (ret == 0) {
        word32 inOutIdx = 0;
        /* Default values. */
        *hash = WC_HASH_TYPE_SHA;
        *mgf = WC_MGF1SHA1;

        /* Set OID type expected. */
        GetASN_OID(&dataASN[RSAPSSPARAMSASN_IDX_HASHOID], oidHashType);
        GetASN_OID(&dataASN[RSAPSSPARAMSASN_IDX_MGFHOID], oidHashType);
        /* Place the salt length into 16-bit var sLen. */
        GetASN_Int16Bit(&dataASN[RSAPSSPARAMSASN_IDX_SALTLENINT], &sLen);
        /* Decode the algorithm identifier. */
        ret = GetASN_Items(rsaPssParamsASN, dataASN, rsaPssParamsASN_Length, 1,
            params, &inOutIdx, sz);
    }
    if ((ret == 0) && (dataASN[RSAPSSPARAMSASN_IDX_HASHOID].tag != 0)) {
        word32 oid = dataASN[RSAPSSPARAMSASN_IDX_HASHOID].data.oid.sum;
        ret = RsaPssHashOidToType(oid, hash);
    }
    if ((ret == 0) && (dataASN[RSAPSSPARAMSASN_IDX_MGFHOID].tag != 0)) {
        word32 oid = dataASN[RSAPSSPARAMSASN_IDX_MGFHOID].data.oid.sum;
        ret = RsaPssHashOidToMgf1(oid, mgf);
    }
    if (ret == 0) {
        *saltLen = sLen;
    }

    FREE_ASNGETDATA(dataASN, NULL);
    return ret;
#endif /* WOLFSSL_ASN_TEMPLATE */
}
#endif /* WC_RSA_PSS */

#if defined(WOLFSSL_ASN_TEMPLATE) || (!defined(NO_CERTS) && \
    (defined(WOLFSSL_KEY_GEN) || defined(OPENSSL_EXTRA) || \
     defined(WOLFSSL_KCAPI_RSA) || defined(WOLFSSL_SE050)))
/* Byte offset of numbers in RSA key. */
size_t rsaIntOffset[] = {
    WC_OFFSETOF(RsaKey, n),
    WC_OFFSETOF(RsaKey, e),
#ifndef WOLFSSL_RSA_PUBLIC_ONLY
    WC_OFFSETOF(RsaKey, d),
    WC_OFFSETOF(RsaKey, p),
    WC_OFFSETOF(RsaKey, q),
#if defined(WOLFSSL_KEY_GEN) || defined(OPENSSL_EXTRA) || !defined(RSA_LOW_MEM)
    WC_OFFSETOF(RsaKey, dP),
    WC_OFFSETOF(RsaKey, dQ),
    WC_OFFSETOF(RsaKey, u)
#endif
#endif
};

/* Get a number from the RSA key based on an index.
 *
 * Order: { n, e, d, p, q, dP, dQ, u }
 *
 * Caller must ensure index is not invalid!
 *
 * @param [in] key  RSA key object.
 * @param [in] idx  Index of number.
 * @return  A pointer to an mp_int when valid index.
 * @return  NULL when invalid index.
 */
static mp_int* GetRsaInt(RsaKey* key, int idx)
{
    /* Cast key to byte array to and use offset to get to mp_int field. */
    return (mp_int*)(((byte*)key) + rsaIntOffset[idx]);
}
#endif

#ifdef WOLFSSL_ASN_TEMPLATE
/* ASN.1 template for an RSA private key.
 * PKCS #1: RFC 8017, A.1.2 - RSAPrivateKey
 */
static const ASNItem rsaKeyASN[] = {
/*  SEQ */    { 0, ASN_SEQUENCE, 1, 1, 0 },
/*  VER */        { 1, ASN_INTEGER, 0, 0, 0 },
                /* Integers need to be in this specific order
                 * as asn code depends on this. */
/*  N   */        { 1, ASN_INTEGER, 0, 0, 0 },
/*  E   */        { 1, ASN_INTEGER, 0, 0, 0 },
#if !defined(WOLFSSL_RSA_PUBLIC_ONLY) || defined(WOLFSSL_KEY_GEN)
/*  D   */        { 1, ASN_INTEGER, 0, 0, 0 },
/*  P   */        { 1, ASN_INTEGER, 0, 0, 0 },
/*  Q   */        { 1, ASN_INTEGER, 0, 0, 0 },
/*  DP  */        { 1, ASN_INTEGER, 0, 0, 0 },
/*  DQ  */        { 1, ASN_INTEGER, 0, 0, 0 },
/*  U   */        { 1, ASN_INTEGER, 0, 0, 0 },
                /* otherPrimeInfos  OtherPrimeInfos OPTIONAL
                 * v2 - multiprime */
#endif
};
enum {
    RSAKEYASN_IDX_SEQ = 0,
    RSAKEYASN_IDX_VER,
    /* Integers need to be in this specific order
     * as asn code depends on this. */
    RSAKEYASN_IDX_N,
    RSAKEYASN_IDX_E,
#if !defined(WOLFSSL_RSA_PUBLIC_ONLY) || defined(WOLFSSL_KEY_GEN)
    RSAKEYASN_IDX_D,
    RSAKEYASN_IDX_P,
    RSAKEYASN_IDX_Q,
    RSAKEYASN_IDX_DP,
    RSAKEYASN_IDX_DQ,
    RSAKEYASN_IDX_U,
#endif
    WOLF_ENUM_DUMMY_LAST_ELEMENT(RSAKEYASN_IDX)
};

/* Number of items in ASN.1 template for an RSA private key. */
#define rsaKeyASN_Length (sizeof(rsaKeyASN) / sizeof(ASNItem))
#endif

/* Decode RSA private key.
 *
 * PKCS #1: RFC 8017, A.1.2 - RSAPrivateKey
 *
 * Compiling with WOLFSSL_RSA_PUBLIC_ONLY will result in only the public fields
 * being extracted.
 *
 * @param [in]      input     Buffer holding BER encoded data.
 * @param [in, out] inOutIdx  On in, start of RSA private key.
 *                            On out, start of ASN.1 item after RSA private key.
 * @param [in, out] key       RSA key object. May be NULL.
 * @param [out]     keySz     Size of key in bytes. May be NULL.
 * @param [in]      inSz      Number of bytes in buffer.
 * @return  0 on success.
 * @return  BAD_FUNC_ARG when input or inOutIdx is NULL.
 * @return  BAD_FUNC_ARG when key and keySz are NULL.
 * @return  ASN_PARSE_E when BER encoded data does not match ASN.1 items or
 *          is invalid.
 * @return  BUFFER_E when data in buffer is too small.
 * @return  ASN_EXPECT_0_E when the INTEGER has the MSB set or NULL has a
 *          non-zero length.
 * @return  MP_INIT_E when the unable to initialize an mp_int.
 * @return  ASN_GETINT_E when the unable to convert data to an mp_int.
 */
static int _RsaPrivateKeyDecode(const byte* input, word32* inOutIdx,
    RsaKey* key, int* keySz, word32 inSz)
{
#ifndef WOLFSSL_ASN_TEMPLATE
    int version, length;
    word32 algId = 0;

    if (inOutIdx == NULL || input == NULL || (key == NULL && keySz == NULL)) {
        return BAD_FUNC_ARG;
    }

    /* if has pkcs8 header skip it */
    if (ToTraditionalInline_ex(input, inOutIdx, inSz, &algId) < 0) {
        /* ignore error, did not have pkcs8 header */
    }

    if (GetSequence(input, inOutIdx, &length, inSz) < 0)
        return ASN_PARSE_E;

    if (GetMyVersion(input, inOutIdx, &version, inSz) < 0)
        return ASN_PARSE_E;

    if (key == NULL) {
        int i;

        /* Modulus */
        if (GetASNInt(input, inOutIdx, keySz, inSz) < 0) {
            return ASN_PARSE_E;
        }
        *inOutIdx += (word32)*keySz;
        for (i = 1; i < RSA_INTS; i++) {
            if (SkipInt(input, inOutIdx, inSz) < 0) {
                return ASN_RSA_KEY_E;
            }
        }
    }
    else {
        key->type = RSA_PRIVATE;

    #ifdef WOLFSSL_CHECK_MEM_ZERO
        mp_memzero_add("Decode RSA key d", &key->d);
        mp_memzero_add("Decode RSA key p", &key->p);
        mp_memzero_add("Decode RSA key q", &key->q);
    #if (defined(WOLFSSL_KEY_GEN) || defined(OPENSSL_EXTRA) || \
        !defined(RSA_LOW_MEM)) && !defined(WOLFSSL_RSA_PUBLIC_ONLY)
        mp_memzero_add("Decode RSA key dP", &key->dP);
        mp_memzero_add("Decode RSA key dQ", &key->dQ);
        mp_memzero_add("Decode RSA key u", &key->u);
    #endif
    #endif

        if (GetInt(&key->n,  input, inOutIdx, inSz) < 0 ||
            GetInt(&key->e,  input, inOutIdx, inSz) < 0 ||
    #ifndef WOLFSSL_RSA_PUBLIC_ONLY
            GetInt(&key->d,  input, inOutIdx, inSz) < 0 ||
            GetInt(&key->p,  input, inOutIdx, inSz) < 0 ||
            GetInt(&key->q,  input, inOutIdx, inSz) < 0
    #else
            SkipInt(input, inOutIdx, inSz) < 0 ||
            SkipInt(input, inOutIdx, inSz) < 0 ||
            SkipInt(input, inOutIdx, inSz) < 0
    #endif
           ) {
                return ASN_RSA_KEY_E;
           }
    #if (defined(WOLFSSL_KEY_GEN) || defined(OPENSSL_EXTRA) || !defined(RSA_LOW_MEM)) \
        && !defined(WOLFSSL_RSA_PUBLIC_ONLY)
        if (GetInt(&key->dP, input, inOutIdx, inSz) < 0 ||
            GetInt(&key->dQ, input, inOutIdx, inSz) < 0 ||
            GetInt(&key->u,  input, inOutIdx, inSz) < 0 )  return ASN_RSA_KEY_E;
    #else
        if (SkipInt(input, inOutIdx, inSz) < 0 ||
            SkipInt(input, inOutIdx, inSz) < 0 ||
            SkipInt(input, inOutIdx, inSz) < 0 )  return ASN_RSA_KEY_E;
    #endif

    #if defined(WOLFSSL_XILINX_CRYPT) || defined(WOLFSSL_CRYPTOCELL)
        if (wc_InitRsaHw(key) != 0) {
            return BAD_STATE_E;
        }
    #endif
    }

    return 0;
#else
    DECL_ASNGETDATA(dataASN, rsaKeyASN_Length);
    int        ret = 0;
    byte       version = (byte)-1;
#if defined(HAVE_PKCS8) || defined(HAVE_PKCS12)
    word32 algId = 0;
#endif
    void*      heap = NULL;

    /* Check validity of parameters. */
    if ((inOutIdx == NULL) || (input == NULL) || ((key == NULL) &&
            (keySz == NULL))) {
        ret = BAD_FUNC_ARG;
    }

    if ((ret == 0) && (key != NULL)) {
        heap = key->heap;
    }

#if defined(HAVE_PKCS8) || defined(HAVE_PKCS12)
    if (ret == 0) {
        /* if has pkcs8 header skip it */
        if (ToTraditionalInline_ex(input, inOutIdx, inSz, &algId) < 0) {
            /* ignore error, did not have pkcs8 header */
        }
    }
#endif

    (void)heap;
    CALLOC_ASNGETDATA(dataASN, rsaKeyASN_Length, ret, heap);

    if (ret == 0) {
        /* Register variable to hold version field. */
        GetASN_Int8Bit(&dataASN[RSAKEYASN_IDX_VER], &version);
        /* Setup data to store INTEGER data in mp_int's in RSA object. */
    #if defined(WOLFSSL_RSA_PUBLIC_ONLY)
        #define RSA_ASN_INTS        RSA_PUB_INTS
        /* Not extracting all data from BER encoding. */
        #define RSA_ASN_COMPLETE    0
    #else
        #define RSA_ASN_INTS        RSA_INTS
        /* Extracting all data from BER encoding. */
        #define RSA_ASN_COMPLETE    1
    #endif
        if (key != NULL) {
            int i;
            /* Extract all public fields. */
            for (i = 0; i < RSA_ASN_INTS; i++) {
                GetASN_MP(&dataASN[(byte)RSAKEYASN_IDX_N + i],
                    GetRsaInt(key, i));
            }
        }
        /* Parse BER encoding for RSA private key. */
        ret = GetASN_Items(rsaKeyASN, dataASN, rsaKeyASN_Length,
            RSA_ASN_COMPLETE, input, inOutIdx, inSz);
    }
    /* Check version: 0 - two prime, 1 - multi-prime
     * Multi-prime has optional sequence after coefficient for extra primes.
     * If extra primes, parsing will fail as not all the buffer was used.
     */
    if ((ret == 0) && (version > PKCS1v1)) {
        ret = ASN_PARSE_E;
    }
    if ((ret == 0) && (key != NULL)) {
    #if !defined(WOLFSSL_RSA_PUBLIC_ONLY)
        /* RSA key object has all private key values. */
        key->type = RSA_PRIVATE;
    #else
        /* RSA key object has all public key values. */
        key->type = RSA_PUBLIC;
    #endif

    #ifdef WOLFSSL_XILINX_CRYPT
        if (wc_InitRsaHw(key) != 0)
            ret = BAD_STATE_E;
    #endif
    }
    else if (ret == 0) {
        /* Not filling in key but do want key size. */
        *keySz = (int)dataASN[(byte)RSAKEYASN_IDX_N].length;
        /* Check whether first byte of data is 0x00 and drop it. */
        if (input[(int)dataASN[RSAKEYASN_IDX_E].offset - *keySz] == 0) {
            (*keySz)--;
        }
    }

    FREE_ASNGETDATA(dataASN, heap);
    return ret;
#endif /* WOLFSSL_ASN_TEMPLATE */
}

/* Decode RSA private key.
 *
 * PKCS #1: RFC 8017, A.1.2 - RSAPrivateKey
 *
 * Compiling with WOLFSSL_RSA_PUBLIC_ONLY will result in only the public fields
 * being extracted.
 *
 * @param [in]      input     Buffer holding BER encoded data.
 * @param [in, out] inOutIdx  On in, start of RSA private key.
 *                            On out, start of ASN.1 item after RSA private key.
 * @param [in, out] key       RSA key object.
 * @param [in]      inSz      Number of bytes in buffer.
 * @return  0 on success.
 * @return  BAD_FUNC_ARG when input, inOutIdx or key is NULL.
 * @return  ASN_PARSE_E when BER encoded data does not match ASN.1 items or
 *          is invalid.
 * @return  BUFFER_E when data in buffer is too small.
 * @return  ASN_EXPECT_0_E when the INTEGER has the MSB set or NULL has a
 *          non-zero length.
 * @return  MP_INIT_E when the unable to initialize an mp_int.
 * @return  ASN_GETINT_E when the unable to convert data to an mp_int.
 */
int wc_RsaPrivateKeyDecode(const byte* input, word32* inOutIdx, RsaKey* key,
    word32 inSz)
{
    if (key == NULL) {
        return BAD_FUNC_ARG;
    }
    return _RsaPrivateKeyDecode(input, inOutIdx, key, NULL, inSz);
}

/* Valdidate RSA private key ASN.1 encoding.
 *
 * PKCS #1: RFC 8017, A.1.2 - RSAPrivateKey
 *
 * Compiling with WOLFSSL_RSA_PUBLIC_ONLY will result in only the public fields
 * being extracted.
 *
 * @param [in]      input     Buffer holding BER encoded data.
 * @param [in, out] inOutIdx  On in, start of RSA private key.
 *                            On out, start of ASN.1 item after RSA private key.
 * @param [in]      inSz      Number of bytes in buffer.
 * @return  0 on success.
 * @return  BAD_FUNC_ARG when input, inOutIdx or keySz is NULL.
 * @return  ASN_PARSE_E when BER encoded data does not match ASN.1 items or
 *          is invalid.
 * @return  BUFFER_E when data in buffer is too small.
 * @return  ASN_EXPECT_0_E when the INTEGER has the MSB set or NULL has a
 *          non-zero length.
 * @return  MP_INIT_E when the unable to initialize an mp_int.
 * @return  ASN_GETINT_E when the unable to convert data to an mp_int.
 */
int wc_RsaPrivateKeyValidate(const byte* input, word32* inOutIdx, int* keySz,
     word32 inSz)
{
    return _RsaPrivateKeyDecode(input, inOutIdx, NULL, keySz, inSz);
}
#endif /* NO_RSA */

#ifdef WOLFSSL_ASN_TEMPLATE
/* ASN.1 template for a PKCS #8 key.
 * Ignoring optional attributes and public key.
 * PKCS #8: RFC 5958, 2 - PrivateKeyInfo
 */
static const ASNItem pkcs8KeyASN[] = {
/*  SEQ                 */    { 0, ASN_SEQUENCE, 1, 1, 0 },
/*  VER                 */        { 1, ASN_INTEGER, 0, 0, 0 },
/*  PKEY_ALGO_SEQ       */        { 1, ASN_SEQUENCE, 1, 1, 0 },
/*  PKEY_ALGO_OID_KEY   */            { 2, ASN_OBJECT_ID, 0, 0, 0 },
/*  PKEY_ALGO_OID_CURVE */            { 2, ASN_OBJECT_ID, 0, 0, 1 },
/*  PKEY_ALGO_NULL      */            { 2, ASN_TAG_NULL, 0, 0, 1 },
#ifdef WC_RSA_PSS
/*  PKEY_ALGO_PARAM_SEQ */            { 2, ASN_SEQUENCE, 1, 0, 1 },
#endif
/*  PKEY_DATA           */        { 1, ASN_OCTET_STRING, 0, 0, 0 },
/*  OPTIONAL Attributes IMPLICIT [0] */
                                  { 1, ASN_CONTEXT_SPECIFIC | 0, 1, 0, 1 },
/* [[2: publicKey        [1] PublicKey OPTIONAL ]] */
};
enum {
    PKCS8KEYASN_IDX_SEQ = 0,
    PKCS8KEYASN_IDX_VER,
    PKCS8KEYASN_IDX_PKEY_ALGO_SEQ,
    PKCS8KEYASN_IDX_PKEY_ALGO_OID_KEY,
    PKCS8KEYASN_IDX_PKEY_ALGO_OID_CURVE,
    PKCS8KEYASN_IDX_PKEY_ALGO_NULL,
#ifdef WC_RSA_PSS
    PKCS8KEYASN_IDX_PKEY_ALGO_PARAM_SEQ,
#endif
    PKCS8KEYASN_IDX_PKEY_DATA,
    PKCS8KEYASN_IDX_PKEY_ATTRIBUTES,
    WOLF_ENUM_DUMMY_LAST_ELEMENT(PKCS8KEYASN_IDX)
};

/* Number of items in ASN.1 template for a PKCS #8 key. */
#define pkcs8KeyASN_Length (sizeof(pkcs8KeyASN) / sizeof(ASNItem))
#endif

/* Remove PKCS #8 header around an RSA, ECDSA, Ed25519, or Ed448.
 *
 * @param [in]       input     Buffer holding BER data.
 * @param [in, out]  inOutIdx  On in, start of PKCS #8 encoding.
 *                             On out, start of encoded key.
 * @param [in]       sz        Size of data in buffer.
 * @param [out]      algId     Key's algorithm id from PKCS #8 header.
 * @param [out]      eccOid    ECC curve OID.
 * @return  Length of key data on success.
 * @return  BAD_FUNC_ARG when input or inOutIdx is NULL.
 * @return  ASN_PARSE_E when BER encoded data does not match ASN.1 items or
 *          is invalid.
 * @return  BUFFER_E when data in buffer is too small.
 * @return  ASN_OBJECT_ID_E when the expected OBJECT_ID tag is not found.
 * @return  ASN_EXPECT_0_E when the INTEGER has the MSB set or NULL has a
 *          non-zero length.
 */
int ToTraditionalInline_ex2(const byte* input, word32* inOutIdx, word32 sz,
                            word32* algId, word32* eccOid)
{
#ifndef WOLFSSL_ASN_TEMPLATE
    word32 idx;
    int    version, length;
    int    ret;
    byte   tag;

    if (input == NULL || inOutIdx == NULL)
        return BAD_FUNC_ARG;

    idx = *inOutIdx;

    if (GetSequence(input, &idx, &length, sz) < 0)
        return ASN_PARSE_E;

    if (GetMyVersion(input, &idx, &version, sz) < 0)
        return ASN_PARSE_E;

    if (GetAlgoId(input, &idx, algId, oidKeyType, sz) < 0)
        return ASN_PARSE_E;

    if (GetASNTag(input, &idx, &tag, sz) < 0)
        return ASN_PARSE_E;
    idx = idx - 1; /* reset idx after finding tag */

#if defined(WC_RSA_PSS) && !defined(NO_RSA)
    if (*algId == RSAPSSk && tag == (ASN_SEQUENCE | ASN_CONSTRUCTED)) {
        word32 seqIdx = idx;
        int seqLen;
        /* Not set when -1. */
        enum wc_HashType hash = WC_HASH_TYPE_NONE;
        int mgf = -1;
        int saltLen = 0;

        if (GetSequence(input, &idx, &seqLen, sz) < 0) {
            return ASN_PARSE_E;
        }
        /* Get the private key parameters. */
        ret = DecodeRsaPssParams(input + seqIdx,
            seqLen + idx - seqIdx, &hash, &mgf, &saltLen);
        if (ret != 0) {
            return ASN_PARSE_E;
        }
        /* TODO: store parameters so that usage can be checked. */
        idx += seqLen;
    }
#endif /* WC_RSA_PSS && !NO_RSA */

    if (tag == ASN_OBJECT_ID) {
        if ((*algId == ECDSAk) && (eccOid != NULL)) {
            if (GetObjectId(input, &idx, eccOid, oidCurveType, sz) < 0)
                return ASN_PARSE_E;
        }
        else {
            if (SkipObjectId(input, &idx, sz) < 0)
                return ASN_PARSE_E;
        }
    }

    ret = GetOctetString(input, &idx, &length, sz);
    if (ret < 0) {
        if (ret == WC_NO_ERR_TRACE(BUFFER_E))
            return ASN_PARSE_E;
        /* Some private keys don't expect an octet string */
        WOLFSSL_MSG("Couldn't find Octet string");
    }

    *inOutIdx = idx;

    return length;
#else
    DECL_ASNGETDATA(dataASN, pkcs8KeyASN_Length);
    int ret = 0;
    word32 oid = 9;
    byte version = 0;
    word32 idx;

    (void)eccOid;

    /* Check validity of parameters. */
    if (input == NULL || inOutIdx == NULL) {
        return BAD_FUNC_ARG;
    }

    idx = *inOutIdx;

    CALLOC_ASNGETDATA(dataASN, pkcs8KeyASN_Length, ret, NULL);

    if (ret == 0) {
        /* Get version, check key type and curve type. */
        GetASN_Int8Bit(&dataASN[PKCS8KEYASN_IDX_VER], &version);
        GetASN_OID(&dataASN[PKCS8KEYASN_IDX_PKEY_ALGO_OID_KEY], oidKeyType);
        GetASN_OID(&dataASN[PKCS8KEYASN_IDX_PKEY_ALGO_OID_CURVE], oidCurveType);
        /* Parse data. */
        ret = GetASN_Items(pkcs8KeyASN, dataASN, pkcs8KeyASN_Length, 1, input,
                           &idx, sz);
    }

    if (ret == 0) {
        /* Key type OID. */
        oid = dataASN[PKCS8KEYASN_IDX_PKEY_ALGO_OID_KEY].data.oid.sum;

        /* Version 1 includes an optional public key.
         * If public key is included then the parsing will fail as it did not
         * use all the data.
         */
        if (version > PKCS8v1) {
            ret = ASN_PARSE_E;
        }
    }
    if (ret == 0) {
        switch (oid) {
    #ifndef NO_RSA
            case RSAk:
                /* Must have NULL item but not OBJECT_ID item. */
                if ((dataASN[PKCS8KEYASN_IDX_PKEY_ALGO_NULL].tag == 0) ||
                    (dataASN[PKCS8KEYASN_IDX_PKEY_ALGO_OID_CURVE].tag != 0)) {
                    ret = ASN_PARSE_E;
                }
                break;
        #ifdef WC_RSA_PSS
            case RSAPSSk:
                /* Must not have NULL item. */
                if (dataASN[PKCS8KEYASN_IDX_PKEY_ALGO_NULL].tag != 0) {
                    ret = ASN_PARSE_E;
                }
                if (dataASN[PKCS8KEYASN_IDX_PKEY_ALGO_PARAM_SEQ].tag != 0) {
                    enum wc_HashType hash;
                    int mgf;
                    int saltLen;
                    const byte* params = GetASNItem_Addr(
                        dataASN[PKCS8KEYASN_IDX_PKEY_ALGO_PARAM_SEQ], input);
                    word32 paramsSz = GetASNItem_Length(
                        dataASN[PKCS8KEYASN_IDX_PKEY_ALGO_PARAM_SEQ], input);

                    /* Validate the private key parameters. */
                    ret = DecodeRsaPssParams(params, paramsSz, &hash, &mgf,
                        &saltLen);
                    if (ret != 0) {
                        return ASN_PARSE_E;
                    }
                    /* TODO: store parameters so that usage can be checked. */
                }
                break;
        #endif
    #endif
        #ifdef HAVE_ECC
            case ECDSAk:
                /* Must not have NULL item. */
                if (dataASN[PKCS8KEYASN_IDX_PKEY_ALGO_NULL].tag != 0) {
                    ret = ASN_PARSE_E;
                }
                if (eccOid != NULL) {
                    ASNGetData* oidCurve =
                        &dataASN[PKCS8KEYASN_IDX_PKEY_ALGO_OID_CURVE];
                    *eccOid = oidCurve->data.oid.sum;
                }
                break;
        #endif
        #ifdef HAVE_ED25519
            case ED25519k:
                /* Neither NULL item nor OBJECT_ID item allowed. */
                if ((dataASN[PKCS8KEYASN_IDX_PKEY_ALGO_NULL].tag != 0) ||
                    (dataASN[PKCS8KEYASN_IDX_PKEY_ALGO_OID_CURVE].tag != 0)) {
                    ret = ASN_PARSE_E;
                }
                break;
        #endif
        #ifdef HAVE_CURVE25519
            case X25519k:
                /* Neither NULL item nor OBJECT_ID item allowed. */
                if ((dataASN[PKCS8KEYASN_IDX_PKEY_ALGO_NULL].tag != 0) ||
                    (dataASN[PKCS8KEYASN_IDX_PKEY_ALGO_OID_CURVE].tag != 0)) {
                    ret = ASN_PARSE_E;
                }
                break;
        #endif
        #ifdef HAVE_ED448
            case ED448k:
                /* Neither NULL item nor OBJECT_ID item allowed. */
                if ((dataASN[PKCS8KEYASN_IDX_PKEY_ALGO_NULL].tag != 0) ||
                    (dataASN[PKCS8KEYASN_IDX_PKEY_ALGO_OID_CURVE].tag != 0)) {
                    ret = ASN_PARSE_E;
                }
                break;
        #endif
        #ifdef HAVE_CURVE448
            case X448k:
                /* Neither NULL item nor OBJECT_ID item allowed. */
                if ((dataASN[PKCS8KEYASN_IDX_PKEY_ALGO_NULL].tag != 0) ||
                    (dataASN[PKCS8KEYASN_IDX_PKEY_ALGO_OID_CURVE].tag != 0)) {
                    ret = ASN_PARSE_E;
                }
                break;
        #endif
        #ifndef NO_DH
            case DHk:
                /* Neither NULL item nor OBJECT_ID item allowed. */
                if ((dataASN[PKCS8KEYASN_IDX_PKEY_ALGO_NULL].tag != 0) ||
                    (dataASN[PKCS8KEYASN_IDX_PKEY_ALGO_OID_CURVE].tag != 0)) {
                    ret = ASN_PARSE_E;
                }
                break;
        #endif
            /* DSAk not supported. */
            /* Falcon, Dilithium and Sphincs not supported. */
            /* Ignore OID lookup failures. */
            default:
                break;
        }
    }
    if (ret == 0) {
        /* Return algorithm id of internal key. */
        *algId = oid;
        /* Return index to start of internal key. */
        *inOutIdx = GetASNItem_DataIdx(dataASN[PKCS8KEYASN_IDX_PKEY_DATA], input);
        /* Return value is length of internal key. */
        ret = (int)dataASN[PKCS8KEYASN_IDX_PKEY_DATA].data.ref.length;
    }

    FREE_ASNGETDATA(dataASN, NULL);
    return ret;
#endif
}

/* Remove PKCS #8 header around an RSA, ECDSA, Ed25519, or Ed448.
 *
 * @param [in]       input     Buffer holding BER data.
 * @param [in, out]  inOutIdx  On in, start of PKCS #8 encoding.
 *                             On out, start of encoded key.
 * @param [in]       sz        Size of data in buffer.
 * @param [out]      algId     Key's algorithm id from PKCS #8 header.
 * @return  Length of key data on success.
 * @return  BAD_FUNC_ARG when input or inOutIdx is NULL.
 * @return  ASN_PARSE_E when BER encoded data does not match ASN.1 items or
 *          is invalid.
 * @return  BUFFER_E when data in buffer is too small.
 * @return  ASN_OBJECT_ID_E when the expected OBJECT_ID tag is not found.
 * @return  ASN_EXPECT_0_E when the INTEGER has the MSB set or NULL has a
 *          non-zero length.
 */
int ToTraditionalInline_ex(const byte* input, word32* inOutIdx, word32 sz,
                           word32* algId)
{
    return ToTraditionalInline_ex2(input, inOutIdx, sz, algId, NULL);
}


/* TODO: test case  */
int ToTraditionalInline(const byte* input, word32* inOutIdx, word32 sz)
{
    word32 oid;

    return ToTraditionalInline_ex(input, inOutIdx, sz, &oid);
}

#if defined(HAVE_PKCS8) || defined(HAVE_PKCS12)

/* Remove PKCS8 header, move beginning of traditional to beginning of input */
int ToTraditional_ex(byte* input, word32 sz, word32* algId)
{
    word32 inOutIdx = 0;
    int    length;

    if (input == NULL)
        return BAD_FUNC_ARG;

    length = ToTraditionalInline_ex(input, &inOutIdx, sz, algId);
    if (length < 0)
        return length;

    if ((word32)length + inOutIdx > sz)
        return BUFFER_E;

    XMEMMOVE(input, input + inOutIdx, (size_t)length);

    return length;
}

int ToTraditional(byte* input, word32 sz)
{
    word32 oid;

    return ToTraditional_ex(input, sz, &oid);
}

#endif /* HAVE_PKCS8 || HAVE_PKCS12 */

#if defined(HAVE_PKCS8)

int wc_GetPkcs8TraditionalOffset(byte* input, word32* inOutIdx, word32 sz)
{
    int length;
    word32 algId;

    if (input == NULL || inOutIdx == NULL || (*inOutIdx > sz))
        return BAD_FUNC_ARG;

    length = ToTraditionalInline_ex(input, inOutIdx, sz, &algId);

    return length;
}

int wc_CreatePKCS8Key(byte* out, word32* outSz, byte* key, word32 keySz,
        int algoID, const byte* curveOID, word32 oidSz)
{
#ifndef WOLFSSL_ASN_TEMPLATE
    word32 keyIdx = 0;
    word32 tmpSz  = 0;
    word32 sz;
    word32 tmpAlgId = 0;

    /* If out is NULL then return the max size needed
     * + 2 for ASN_OBJECT_ID and ASN_OCTET_STRING tags */
    if (out == NULL && outSz != NULL) {
        *outSz = keySz + MAX_SEQ_SZ + MAX_VERSION_SZ + MAX_ALGO_SZ
                 + MAX_LENGTH_SZ + MAX_LENGTH_SZ + 2;

        if (curveOID != NULL)
            *outSz += oidSz + MAX_LENGTH_SZ + 1;

        WOLFSSL_MSG("Checking size of PKCS8");

        return WC_NO_ERR_TRACE(LENGTH_ONLY_E);
    }

    WOLFSSL_ENTER("wc_CreatePKCS8Key");

    if (key == NULL || out == NULL || outSz == NULL) {
        return BAD_FUNC_ARG;
    }

    /* check the buffer has enough room for largest possible size */
    if (curveOID != NULL) {
        if (*outSz < (keySz + MAX_SEQ_SZ + MAX_VERSION_SZ + MAX_ALGO_SZ
               + MAX_LENGTH_SZ + MAX_LENGTH_SZ + 3 + oidSz + MAX_LENGTH_SZ))
            return BUFFER_E;
    }
    else {
        oidSz = 0; /* with no curveOID oid size must be 0 */
        if (*outSz < (keySz + MAX_SEQ_SZ + MAX_VERSION_SZ + MAX_ALGO_SZ
                  + MAX_LENGTH_SZ + MAX_LENGTH_SZ + 2))
            return BUFFER_E;
    }

    /* sanity check: make sure the key doesn't already have a PKCS 8 header */
    if (ToTraditionalInline_ex(key, &keyIdx, keySz, &tmpAlgId) >= 0) {
        (void)tmpAlgId;
        return ASN_PARSE_E;
    }

    /* PrivateKeyInfo ::= SEQUENCE */
    keyIdx = MAX_SEQ_SZ; /* save room for sequence */

    /*  version Version
     *  no header information just INTEGER */
    sz = (word32)SetMyVersion(PKCS8v0, out + keyIdx, 0);
    tmpSz += sz; keyIdx += sz;
    /*  privateKeyAlgorithm PrivateKeyAlgorithmIdentifier */
    sz = 0; /* set sz to 0 and get privateKey oid buffer size needed */
    if (curveOID != NULL && oidSz > 0) {
        byte buf[MAX_LENGTH_SZ];
        sz = SetLength(oidSz, buf);
        sz += 1; /* plus one for ASN object id */
    }
    sz = (word32)SetAlgoID(algoID, out + keyIdx, oidKeyType, (int)(oidSz + sz));
    tmpSz += sz; keyIdx += sz;

    /*  privateKey          PrivateKey *
     * pkcs8 ecc uses slightly different format. Places curve oid in
     * buffer */
    if (curveOID != NULL && oidSz > 0) {
        sz = (word32)SetObjectId((int)oidSz, out + keyIdx);
        keyIdx += sz; tmpSz += sz;
        XMEMCPY(out + keyIdx, curveOID, oidSz);
        keyIdx += oidSz; tmpSz += oidSz;
    }

    sz = (word32)SetOctetString(keySz, out + keyIdx);
    keyIdx += sz; tmpSz += sz;
    XMEMCPY(out + keyIdx, key, keySz);
    tmpSz += keySz;

    /*  attributes          optional
     * No attributes currently added */

    /* rewind and add sequence */
    sz = SetSequence(tmpSz, out);
    XMEMMOVE(out + sz, out + MAX_SEQ_SZ, tmpSz);

    *outSz = tmpSz + sz;
    return (int)(tmpSz + sz);
#else
    /* pkcs8KeyASN_Length-1, the -1 is because we are not adding the optional
     * set of attributes */
    DECL_ASNSETDATA(dataASN, pkcs8KeyASN_Length-1);
    int sz = 0;
    int ret = 0;
    word32 keyIdx = 0;
    word32 tmpAlgId = 0;

    WOLFSSL_ENTER("wc_CreatePKCS8Key");

    /* Check validity of parameters. */
    if (out == NULL && outSz != NULL) {
    }
    else if (key == NULL || out == NULL || outSz == NULL) {
        ret = BAD_FUNC_ARG;
    }

#ifndef WOLFSSL_NO_ASN_STRICT
    /* Sanity check: make sure key doesn't have PKCS #8 header. */
    if (ToTraditionalInline_ex(key, &keyIdx, keySz, &tmpAlgId) >= 0) {
        (void)tmpAlgId;
        ret = ASN_PARSE_E;
    }
#else
    (void)keyIdx;
    (void)tmpAlgId;
#endif

    CALLOC_ASNSETDATA(dataASN, pkcs8KeyASN_Length-1, ret, NULL);

    if (ret == 0) {
        /* Only support default PKCS #8 format - v0. */
        SetASN_Int8Bit(&dataASN[PKCS8KEYASN_IDX_VER], PKCS8v0);
        /* Set key OID that corresponds to key data. */
        SetASN_OID(&dataASN[PKCS8KEYASN_IDX_PKEY_ALGO_OID_KEY], (word32)algoID,
            oidKeyType);
        if (curveOID != NULL && oidSz > 0) {
            /* ECC key and curveOID set to write. */
            SetASN_Buffer(&dataASN[PKCS8KEYASN_IDX_PKEY_ALGO_OID_CURVE],
                curveOID, oidSz);
        }
        else {
            /* EC curve OID to encode. */
            dataASN[PKCS8KEYASN_IDX_PKEY_ALGO_OID_CURVE].noOut = 1;
        }
        /* Only RSA keys have NULL tagged item after OID. */
        dataASN[PKCS8KEYASN_IDX_PKEY_ALGO_NULL].noOut = (algoID != RSAk);
    #ifdef WC_RSA_PSS
        dataASN[PKCS8KEYASN_IDX_PKEY_ALGO_PARAM_SEQ].noOut = 1;
    #endif
        /* Set key data to encode. */
        SetASN_Buffer(&dataASN[PKCS8KEYASN_IDX_PKEY_DATA], key, keySz);

        /* Get the size of the DER encoding. */
        ret = SizeASN_Items(pkcs8KeyASN, dataASN, pkcs8KeyASN_Length-1, &sz);
    }
    if ((ret == 0) || (ret == WC_NO_ERR_TRACE(LENGTH_ONLY_E))) {
        /* Always return the calculated size. */
        *outSz = (word32)sz;
    }
    /* Check for buffer to encoded into. */
    if ((ret == 0) && (out == NULL)) {
        WOLFSSL_MSG("Checking size of PKCS8");
        ret = WC_NO_ERR_TRACE(LENGTH_ONLY_E);
    }
    if (ret == 0) {
        /*  Encode PKCS #8 key into buffer. */
        SetASN_Items(pkcs8KeyASN, dataASN, pkcs8KeyASN_Length-1, out);
        ret = sz;
    }

    FREE_ASNSETDATA(dataASN, NULL);
    return ret;
#endif /* WOLFSSL_ASN_TEMPLATE */
}

#endif /* HAVE_PKCS8 */

#if defined(HAVE_PKCS12) || !defined(NO_CHECK_PRIVATE_KEY)
/* check that the private key is a pair for the public key
 * return 1 (true) on match
 * return 0 or negative value on failure/error
 *
 * privKey   : buffer holding DER format private key
 * privKeySz : size of private key buffer
 * pubKey    : buffer holding DER format public key
 * pubKeySz  : size of public key buffer
 * ks        : type of key
 * heap      : heap hint to use */
int wc_CheckPrivateKey(const byte* privKey, word32 privKeySz,
                       const byte* pubKey, word32 pubKeySz, enum Key_Sum ks,
                       void* heap)
{
    int ret;
    (void)privKeySz;
    (void)pubKeySz;
    (void)ks;

    if (privKey == NULL || pubKey == NULL) {
        return BAD_FUNC_ARG;
    }

    #if !defined(NO_RSA) && !defined(NO_ASN_CRYPT)
    /* test if RSA key */
    if (ks == RSAk
    #ifdef WC_RSA_PSS
        || ks == RSAPSSk
    #endif
        ) {
    #ifdef WOLFSSL_SMALL_STACK
        RsaKey* a;
        RsaKey* b = NULL;
    #else
        RsaKey a[1], b[1];
    #endif
        word32 keyIdx = 0;

    #ifdef WOLFSSL_SMALL_STACK
        a = (RsaKey*)XMALLOC(sizeof(RsaKey), NULL, DYNAMIC_TYPE_RSA);
        if (a == NULL)
            return MEMORY_E;
        b = (RsaKey*)XMALLOC(sizeof(RsaKey), NULL, DYNAMIC_TYPE_RSA);
        if (b == NULL) {
            XFREE(a, NULL, DYNAMIC_TYPE_RSA);
            return MEMORY_E;
        }
    #endif

        if ((ret = wc_InitRsaKey(a, heap)) < 0) {
            WC_FREE_VAR_EX(b, NULL, DYNAMIC_TYPE_RSA);
            WC_FREE_VAR_EX(a, NULL, DYNAMIC_TYPE_RSA);
            return ret;
        }
        if ((ret = wc_InitRsaKey(b, heap)) < 0) {
            wc_FreeRsaKey(a);
            WC_FREE_VAR_EX(b, NULL, DYNAMIC_TYPE_RSA);
            WC_FREE_VAR_EX(a, NULL, DYNAMIC_TYPE_RSA);
            return ret;
        }
        if ((ret = wc_RsaPrivateKeyDecode(privKey, &keyIdx, a, privKeySz)) == 0) {
            WOLFSSL_MSG("Checking RSA key pair");
            keyIdx = 0; /* reset to 0 for parsing public key */

            if ((ret = wc_RsaPublicKeyDecode(pubKey, &keyIdx, b,
                    pubKeySz)) == 0) {
                /* both keys extracted successfully now check n and e
                 * values are the same. This is dereferencing RsaKey */
                if (mp_cmp(&(a->n), &(b->n)) != MP_EQ ||
                    mp_cmp(&(a->e), &(b->e)) != MP_EQ) {
                    ret = MP_CMP_E;
                    WOLFSSL_ERROR_VERBOSE(ret);
                }
                else
                    ret = 1;
            }
            else {
                WOLFSSL_ERROR_VERBOSE(ret);
            }
        }
        wc_FreeRsaKey(b);
        wc_FreeRsaKey(a);
        WC_FREE_VAR_EX(b, NULL, DYNAMIC_TYPE_RSA);
        WC_FREE_VAR_EX(a, NULL, DYNAMIC_TYPE_RSA);
    }
    else
    #endif /* !NO_RSA && !NO_ASN_CRYPT */

    #if defined(HAVE_ECC) && defined(HAVE_ECC_KEY_EXPORT) && !defined(NO_ASN_CRYPT)
    if (ks == ECDSAk) {
    #ifdef WOLFSSL_SMALL_STACK
        ecc_key* key_pair;
        byte*    privDer;
    #else
        ecc_key  key_pair[1];
        byte     privDer[MAX_ECC_BYTES];
    #endif
        word32   privSz = MAX_ECC_BYTES;
        word32   keyIdx = 0;

    #ifdef WOLFSSL_SMALL_STACK
        key_pair = (ecc_key*)XMALLOC(sizeof(ecc_key), NULL, DYNAMIC_TYPE_ECC);
        if (key_pair == NULL)
            return MEMORY_E;
        privDer = (byte*)XMALLOC(MAX_ECC_BYTES, NULL, DYNAMIC_TYPE_TMP_BUFFER);
        if (privDer == NULL) {
            XFREE(key_pair, NULL, DYNAMIC_TYPE_ECC);
            return MEMORY_E;
        }
    #endif

        if ((ret = wc_ecc_init_ex(key_pair, heap, INVALID_DEVID)) < 0) {
            WC_FREE_VAR_EX(privDer, NULL, DYNAMIC_TYPE_TMP_BUFFER);
            WC_FREE_VAR_EX(key_pair, NULL, DYNAMIC_TYPE_ECC);
            return ret;
        }

        if ((ret = wc_EccPrivateKeyDecode(privKey, &keyIdx, key_pair,
                privKeySz)) == 0) {
            WOLFSSL_MSG("Checking ECC key pair");

            if ((ret = wc_ecc_export_private_only(key_pair, privDer, &privSz))
                                                                         == 0) {
            #ifdef WOLFSSL_CHECK_MEM_ZERO
                wc_MemZero_Add("wc_CheckPrivateKey privDer", privDer, privSz);
            #endif
                wc_ecc_free(key_pair);
                ret = wc_ecc_init_ex(key_pair, heap, INVALID_DEVID);
                if (ret == 0) {
                    ret = wc_ecc_import_private_key(privDer,
                                            privSz, pubKey,
                                            pubKeySz, key_pair);
                }

                /* public and private extracted successfully now check if is
                 * a pair and also do sanity checks on key. wc_ecc_check_key
                 * checks that private * base generator equals pubkey */
                if (ret == 0) {
                    if ((ret = wc_ecc_check_key(key_pair)) == 0) {
                        ret = 1;
                    }
                    else {
                        WOLFSSL_ERROR_VERBOSE(ret);
                    }
                }
                ForceZero(privDer, privSz);
            }
        }
        else {
            WOLFSSL_ERROR_VERBOSE(ret);
        }
        wc_ecc_free(key_pair);
    #ifdef WOLFSSL_SMALL_STACK
        XFREE(privDer, NULL, DYNAMIC_TYPE_TMP_BUFFER);
        XFREE(key_pair, NULL, DYNAMIC_TYPE_ECC);
    #elif defined(WOLFSSL_CHECK_MEM_ZERO)
        wc_MemZero_Check(privDer, MAX_ECC_BYTES);
    #endif
    }
    else
    #endif /* HAVE_ECC && HAVE_ECC_KEY_EXPORT && !NO_ASN_CRYPT */

    #if defined(HAVE_ED25519) && defined(HAVE_ED25519_KEY_IMPORT) && !defined(NO_ASN_CRYPT)
    if (ks == ED25519k) {
        WC_DECLARE_VAR(key_pair, ed25519_key, 1, 0);
        word32       keyIdx = 0;

        WC_ALLOC_VAR_EX(key_pair, ed25519_key, 1, NULL, DYNAMIC_TYPE_ED25519,
            return MEMORY_E);

        if ((ret = wc_ed25519_init_ex(key_pair, heap, INVALID_DEVID)) < 0) {
            WC_FREE_VAR_EX(key_pair, NULL, DYNAMIC_TYPE_ED25519);
            return ret;
        }
        if ((ret = wc_Ed25519PrivateKeyDecode(privKey, &keyIdx, key_pair,
                privKeySz)) == 0) {
            WOLFSSL_MSG("Checking ED25519 key pair");
            keyIdx = 0;
            if ((ret = wc_ed25519_import_public(pubKey, pubKeySz,
                    key_pair)) == 0) {
                /* public and private extracted successfully no check if is
                 * a pair and also do sanity checks on key. wc_ecc_check_key
                 * checks that private * base generator equals pubkey */
                if ((ret = wc_ed25519_check_key(key_pair)) == 0) {
                    ret = 1;
                }
                else {
                    WOLFSSL_ERROR_VERBOSE(ret);
                }
            }
        }
        else {
            WOLFSSL_ERROR_VERBOSE(ret);
        }
        wc_ed25519_free(key_pair);
        WC_FREE_VAR_EX(key_pair, NULL, DYNAMIC_TYPE_ED25519);
    }
    else
    #endif /* HAVE_ED25519 && HAVE_ED25519_KEY_IMPORT && !NO_ASN_CRYPT */

    #if defined(HAVE_ED448) && defined(HAVE_ED448_KEY_IMPORT) && !defined(NO_ASN_CRYPT)
    if (ks == ED448k) {
        WC_DECLARE_VAR(key_pair, ed448_key, 1, 0);
        word32     keyIdx = 0;

        WC_ALLOC_VAR_EX(key_pair, ed448_key, 1, NULL, DYNAMIC_TYPE_ED448,
            return MEMORY_E);

        if ((ret = wc_ed448_init_ex(key_pair, heap, INVALID_DEVID)) < 0) {
            WC_FREE_VAR_EX(key_pair, NULL, DYNAMIC_TYPE_ED448);
            return ret;
        }
        if ((ret = wc_Ed448PrivateKeyDecode(privKey, &keyIdx, key_pair,
                privKeySz)) == 0) {
            WOLFSSL_MSG("Checking ED448 key pair");
            keyIdx = 0;
            if ((ret = wc_ed448_import_public(pubKey, pubKeySz,
                    key_pair)) == 0) {
                /* public and private extracted successfully no check if is
                 * a pair and also do sanity checks on key. wc_ecc_check_key
                 * checks that private * base generator equals pubkey */
                if ((ret = wc_ed448_check_key(key_pair)) == 0) {
                    ret = 1;
                }
                else {
                    WOLFSSL_ERROR_VERBOSE(ret);
                }
            }
        }
        else {
            WOLFSSL_ERROR_VERBOSE(ret);
        }
        wc_ed448_free(key_pair);
        WC_FREE_VAR_EX(key_pair, NULL, DYNAMIC_TYPE_ED448);
    }
    else
    #endif /* HAVE_ED448 && HAVE_ED448_KEY_IMPORT && !NO_ASN_CRYPT */
    #if defined(HAVE_FALCON)
    if ((ks == FALCON_LEVEL1k) || (ks == FALCON_LEVEL5k)) {
        WC_DECLARE_VAR(key_pair, falcon_key, 1, 0);
        word32     keyIdx = 0;

        WC_ALLOC_VAR_EX(key_pair, falcon_key, 1, NULL, DYNAMIC_TYPE_FALCON,
            return MEMORY_E);
        ret = wc_falcon_init(key_pair);
        if (ret  < 0) {
            WC_FREE_VAR_EX(key_pair, NULL, DYNAMIC_TYPE_FALCON);
            return ret;
        }

        if (ks == FALCON_LEVEL1k) {
            ret = wc_falcon_set_level(key_pair, 1);
        }
        else if (ks == FALCON_LEVEL5k) {
            ret = wc_falcon_set_level(key_pair, 5);
        }

        if (ret  < 0) {
            WC_FREE_VAR_EX(key_pair, NULL, DYNAMIC_TYPE_FALCON);
            return ret;
        }
        if ((ret = wc_Falcon_PrivateKeyDecode(privKey, &keyIdx, key_pair,
                                             privKeySz)) == 0) {
            WOLFSSL_MSG("Checking Falcon key pair");
            keyIdx = 0;
            if ((ret = wc_falcon_import_public(pubKey, pubKeySz,
                                               key_pair)) == 0) {
                /* Public and private extracted successfully. Sanity check. */
                if ((ret = wc_falcon_check_key(key_pair)) == 0) {
                    ret = 1;
                }
                else {
                    WOLFSSL_ERROR_VERBOSE(ret);
                }
            }
        }
        else {
            WOLFSSL_ERROR_VERBOSE(ret);
        }
        wc_falcon_free(key_pair);
        WC_FREE_VAR_EX(key_pair, NULL, DYNAMIC_TYPE_FALCON);
    }
    else
    #endif /* HAVE_FALCON */
#if defined(HAVE_DILITHIUM) && !defined(WOLFSSL_DILITHIUM_NO_SIGN) && \
    !defined(WOLFSSL_DILITHIUM_NO_VERIFY) && !defined(WOLFSSL_DILITHIUM_NO_ASN1)
    if ((ks == ML_DSA_LEVEL2k) ||
        (ks == ML_DSA_LEVEL3k) ||
        (ks == ML_DSA_LEVEL5k)
    #ifdef WOLFSSL_DILITHIUM_FIPS204_DRAFT
     || (ks == DILITHIUM_LEVEL2k)
     || (ks == DILITHIUM_LEVEL3k)
     || (ks == DILITHIUM_LEVEL5k)
    #endif
        ) {
        WC_DECLARE_VAR(key_pair, dilithium_key, 1, 0);
        word32     keyIdx = 0;

        WC_ALLOC_VAR_EX(key_pair, dilithium_key, 1, NULL,
            DYNAMIC_TYPE_DILITHIUM, return MEMORY_E);
        ret = wc_dilithium_init(key_pair);
        if (ret  < 0) {
            WC_FREE_VAR_EX(key_pair, NULL, DYNAMIC_TYPE_DILITHIUM);
            return ret;
        }


        if (ks == ML_DSA_LEVEL2k) {
            ret = wc_dilithium_set_level(key_pair, WC_ML_DSA_44);
        }
        else if (ks == ML_DSA_LEVEL3k) {
            ret = wc_dilithium_set_level(key_pair, WC_ML_DSA_65);
        }
        else if (ks == ML_DSA_LEVEL5k) {
            ret = wc_dilithium_set_level(key_pair, WC_ML_DSA_87);
        }
    #ifdef WOLFSSL_DILITHIUM_FIPS204_DRAFT
        else if (ks == DILITHIUM_LEVEL2k) {
            ret = wc_dilithium_set_level(key_pair, WC_ML_DSA_44_DRAFT);
        }
        else if (ks == DILITHIUM_LEVEL3k) {
            ret = wc_dilithium_set_level(key_pair, WC_ML_DSA_65_DRAFT);
        }
        else if (ks == DILITHIUM_LEVEL5k) {
            ret = wc_dilithium_set_level(key_pair, WC_ML_DSA_87_DRAFT);
        }
    #endif

        if (ret  < 0) {
            WC_FREE_VAR_EX(key_pair, NULL, DYNAMIC_TYPE_DILITHIUM);
            return ret;
        }
        if ((ret = wc_Dilithium_PrivateKeyDecode(privKey, &keyIdx, key_pair,
                                             privKeySz)) == 0) {
            WOLFSSL_MSG("Checking Dilithium key pair");
            keyIdx = 0;
            if ((ret = wc_dilithium_import_public(pubKey, pubKeySz,
                                               key_pair)) == 0) {
                /* Public and private extracted successfully. Sanity check. */
                if ((ret = wc_dilithium_check_key(key_pair)) == 0)
                    ret = 1;
            }
        }
        wc_dilithium_free(key_pair);
        WC_FREE_VAR_EX(key_pair, NULL, DYNAMIC_TYPE_DILITHIUM);
    }
    else
#endif /* HAVE_DILITHIUM && !WOLFSSL_DILITHIUM_VERIFY_ONLY */
    #if defined(HAVE_SPHINCS)
    if ((ks == SPHINCS_FAST_LEVEL1k) ||
        (ks == SPHINCS_FAST_LEVEL3k) ||
        (ks == SPHINCS_FAST_LEVEL5k) ||
        (ks == SPHINCS_SMALL_LEVEL1k) ||
        (ks == SPHINCS_SMALL_LEVEL3k) ||
        (ks == SPHINCS_SMALL_LEVEL5k)) {
        WC_DECLARE_VAR(key_pair, sphincs_key, 1, 0);
        word32     keyIdx = 0;

        WC_ALLOC_VAR_EX(key_pair, sphincs_key, 1, NULL, DYNAMIC_TYPE_SPHINCS,
            return MEMORY_E);
        ret = wc_sphincs_init(key_pair);
        if (ret  < 0) {
            WC_FREE_VAR_EX(key_pair, NULL, DYNAMIC_TYPE_SPHINCS);
            return ret;
        }

        if (ks == SPHINCS_FAST_LEVEL1k) {
            ret = wc_sphincs_set_level_and_optim(key_pair, 1, FAST_VARIANT);
        }
        else if (ks == SPHINCS_FAST_LEVEL3k) {
            ret = wc_sphincs_set_level_and_optim(key_pair, 3, FAST_VARIANT);
        }
        else if (ks == SPHINCS_FAST_LEVEL5k) {
            ret = wc_sphincs_set_level_and_optim(key_pair, 5, FAST_VARIANT);
        }
        else if (ks == SPHINCS_SMALL_LEVEL1k) {
            ret = wc_sphincs_set_level_and_optim(key_pair, 1, SMALL_VARIANT);
        }
        else if (ks == SPHINCS_SMALL_LEVEL3k) {
            ret = wc_sphincs_set_level_and_optim(key_pair, 3, SMALL_VARIANT);
        }
        else if (ks == SPHINCS_SMALL_LEVEL5k) {
            ret = wc_sphincs_set_level_and_optim(key_pair, 5, SMALL_VARIANT);
        }

        if (ret  < 0) {
            WC_FREE_VAR_EX(key_pair, NULL, DYNAMIC_TYPE_SPHINCS);
            return ret;
        }
        if ((ret = wc_Sphincs_PrivateKeyDecode(privKey, &keyIdx, key_pair,
                                             privKeySz)) == 0) {
            WOLFSSL_MSG("Checking Sphincs key pair");
            keyIdx = 0;
            if ((ret = wc_sphincs_import_public(pubKey, pubKeySz,
                                               key_pair)) == 0) {
                /* Public and private extracted successfully. Sanity check. */
                if ((ret = wc_sphincs_check_key(key_pair)) == 0)
                    ret = 1;
            }
        }
        wc_sphincs_free(key_pair);
        WC_FREE_VAR_EX(key_pair, NULL, DYNAMIC_TYPE_SPHINCS);
    }
    else
    #endif /* HAVE_SPHINCS */
    {
        ret = 0;
    }
    (void)ks;
    (void)heap;

    return ret;
}

/* check that the private key is a pair for the public key in certificate
 * return 1 (true) on match
 * return 0 or negative value on failure/error
 *
 * key      : buffer holding DER format key
 * keySz    : size of key buffer
 * der      : a initialized and parsed DecodedCert holding a certificate
 * checkAlt : indicate if we check primary or alternative key
 */
int wc_CheckPrivateKeyCert(const byte* key, word32 keySz, DecodedCert* der,
                           int checkAlt, void* heap)
{
    int ret = 0;

    if (key == NULL || der == NULL) {
        return BAD_FUNC_ARG;
    }

#ifdef WOLFSSL_DUAL_ALG_CERTS
    if (checkAlt && der->sapkiDer != NULL) {
        /* We have to decode the public key first */
        /* Default to max pub key size. */
        word32 pubKeyLen = MAX_PUBLIC_KEY_SZ;
        byte* decodedPubKey = (byte*)XMALLOC(pubKeyLen, heap,
                                             DYNAMIC_TYPE_PUBLIC_KEY);
        if (decodedPubKey == NULL) {
            ret = MEMORY_E;
        }
        if (ret == 0) {
            if (der->sapkiOID == RSAk || der->sapkiOID == ECDSAk) {
                /* Simply copy the data */
                XMEMCPY(decodedPubKey, der->sapkiDer, der->sapkiLen);
                pubKeyLen = der->sapkiLen;
            }
            else {
            #if defined(WC_ENABLE_ASYM_KEY_IMPORT)
                word32 idx = 0;
                ret = DecodeAsymKeyPublic(der->sapkiDer, &idx, der->sapkiLen,
                                          decodedPubKey, &pubKeyLen,
                                          der->sapkiOID);
            #else
                ret = NOT_COMPILED_IN;
            #endif /* WC_ENABLE_ASYM_KEY_IMPORT */
            }
        }
        if (ret == 0) {
            ret = wc_CheckPrivateKey(key, keySz, decodedPubKey, pubKeyLen,
                                     (enum Key_Sum) der->sapkiOID, heap);
        }
        XFREE(decodedPubKey, heap, DYNAMIC_TYPE_PUBLIC_KEY);
    }
    else
#endif
    {
        ret = wc_CheckPrivateKey(key, keySz, der->publicKey,
                der->pubKeySize, (enum Key_Sum) der->keyOID, heap);
    }

    (void)checkAlt;

    return ret;
}

#endif /* HAVE_PKCS12 || !NO_CHECK_PRIVATE_KEY */

#ifndef NO_PWDBASED

#if defined(HAVE_PKCS8) || defined(HAVE_PKCS12)
/* Check the PBE algorithm is supported and return wolfSSL id, version and block
 * size of encryption algorithm.
 *
 * When PBES2, version is PKCS5v2, CheckAlgoV2() must be called to get id and
 * blockSz based on encryption algorithm.
 *
 * @param [in]  first    First byte of OID to use in check.
 * @param [in]  second   Second byte of OID to use in check.
 * @param [out] id       wolfSSL id for PBE algorithm.
 * @param [out] version  Version of PBE OID:
 *                       PKCS12v1 (PBE), PKCS5 (PBES1), PKCS5v2 (PBES2).
 * @param [out] blockSz  Block size of encryption algorithm.
 * @return  0 on success.
 * @return  ALGO_ID_E when OID not supported.
 * @return  ASN_INPUT_E when first byte is invalid.
 */
static int CheckAlgo(int first, int second, int* id, int* version, int* blockSz)
{
    int ret = 0;

    (void)id;
    (void)blockSz;

    *version = -1;

    /* pkcs-12 1 = pkcs-12PbeIds */
    if (first == 1) {
        /* PKCS #12: Appendix C */
        switch (second) {
#if !defined(NO_SHA)
    #ifndef NO_RC4
        case PBE_SHA1_RC4_128:
            *id = PBE_SHA1_RC4_128;
            *version = PKCS12v1;
            if (blockSz != NULL) {
                *blockSz = 1;
            }
            break;
    #endif
    #ifndef NO_DES3
        case PBE_SHA1_DES3:
            *id = PBE_SHA1_DES3;
            *version = PKCS12v1;
            if (blockSz != NULL) {
                *blockSz = DES_BLOCK_SIZE;
            }
            break;
    #endif
    #ifdef WC_RC2
        case PBE_SHA1_40RC2_CBC:
            *id = PBE_SHA1_40RC2_CBC;
            *version = PKCS12v1;
            if (blockSz != NULL) {
                *blockSz = RC2_BLOCK_SIZE;
            }
            break;
    #endif
#endif /* !NO_SHA */
        default:
            ret = ALGO_ID_E;
            break;
        }
    }
    else if (first != PKCS5) {
        /* Bad OID. */
        ret = ASN_INPUT_E;
    }
    /* PKCS #5 PBES2: Appendix A.4
     * pkcs-5 13 = id-PBES2 */
    else if (second == PBES2) {
        *version = PKCS5v2;
        /* Id and block size come from CheckAlgoV2() */
    }
    else  {
        /* PKCS #5 PBES1: Appendix A.3 */
        /* see RFC 2898 for ids */
        switch (second) {
    #ifndef NO_DES3
        #ifndef NO_MD5
        case PBES1_MD5_DES:
            *id = PBE_MD5_DES;
            *version = PKCS5;
            if (blockSz != NULL) {
                *blockSz = DES_BLOCK_SIZE;
            }
            break;
        #endif
        #ifndef NO_SHA
        case PBES1_SHA1_DES:
            *id = PBE_SHA1_DES;
            *version = PKCS5;
            if (blockSz != NULL) {
                *blockSz = DES_BLOCK_SIZE;
            }
            break;
        #endif
    #endif /* !NO_DES3 */
        default:
            ret = ALGO_ID_E;
            break;
        }
    }

    /* Return error code. */
    return ret;
}

#endif /* HAVE_PKCS8 || HAVE_PKCS12 */

#ifdef HAVE_PKCS8

/* Check the encryption algorithm with PBES2 is supported and return block size
 * and wolfSSL id for the PBE.
 *
 * @param [in]  oid      Encryption algorithm OID id.
 * @param [out] id       wolfSSL id for PBE algorithm.
 * @param [out] version  Version of PBE OID:
 *                       PKCS12v1 (PBE), PKCS5 (PBES1), PKCS5v2 (PBES2).
 * @return  0 on success.
 * @return  ALGO_ID_E when encryption algorithm is not supported with PBES2.
 */
static int CheckAlgoV2(int oid, int* id, int* blockSz)
{
    int ret = 0;

    (void)id;
    (void)blockSz;

    switch (oid) {
#if !defined(NO_DES3) && !defined(NO_SHA)
    case DESb:
        *id = PBE_SHA1_DES;
        if (blockSz != NULL) {
            *blockSz = DES_BLOCK_SIZE;
        }
        break;
    case DES3b:
        *id = PBE_SHA1_DES3;
        if (blockSz != NULL) {
            *blockSz = DES_BLOCK_SIZE;
        }
        break;
#endif
#ifdef WOLFSSL_AES_256
    case AES256CBCb:
        *id = PBE_AES256_CBC;
        if (blockSz != NULL) {
            *blockSz = WC_AES_BLOCK_SIZE;
        }
        break;
#endif
#ifdef WOLFSSL_AES_128
    case AES128CBCb:
        *id = PBE_AES128_CBC;
        if (blockSz != NULL) {
            *blockSz = WC_AES_BLOCK_SIZE;
        }
        break;
#endif
    default:
        WOLFSSL_MSG("No PKCS v2 algo found");
        ret = ALGO_ID_E;
        break;
    }

    /* Return error code. */
    return ret;
}

#endif /* HAVE_PKCS8 */

#if defined(HAVE_PKCS8) || defined(HAVE_PKCS12)

int wc_GetKeyOID(byte* key, word32 keySz, const byte** curveOID, word32* oidSz,
        int* algoID, void* heap)
{
    word32 tmpIdx = 0;

    if (key == NULL || algoID == NULL)
        return BAD_FUNC_ARG;

    *algoID = 0;

    #if !defined(NO_RSA) && !defined(NO_ASN_CRYPT)
    {
        RsaKey *rsa = (RsaKey *)XMALLOC(sizeof *rsa, heap, DYNAMIC_TYPE_TMP_BUFFER);
        if (rsa == NULL)
            return MEMORY_E;

        wc_InitRsaKey(rsa, heap);
        if (wc_RsaPrivateKeyDecode(key, &tmpIdx, rsa, keySz) == 0) {
            *algoID = RSAk;
        }
        else {
            WOLFSSL_MSG("Not RSA DER key");
        }
        wc_FreeRsaKey(rsa);
        XFREE(rsa, heap, DYNAMIC_TYPE_TMP_BUFFER);
    }
    #endif /* !NO_RSA && !NO_ASN_CRYPT */
    #if defined(HAVE_ECC) && !defined(NO_ASN_CRYPT)
    if (*algoID == 0) {
        ecc_key *ecc = (ecc_key *)XMALLOC(sizeof *ecc, heap, DYNAMIC_TYPE_TMP_BUFFER);
        if (ecc == NULL)
            return MEMORY_E;

        tmpIdx = 0;
        wc_ecc_init_ex(ecc, heap, INVALID_DEVID);
        if (wc_EccPrivateKeyDecode(key, &tmpIdx, ecc, keySz) == 0) {
            *algoID = ECDSAk;

            /* now find oid */
            if (wc_ecc_get_oid(ecc->dp->oidSum, curveOID, oidSz) < 0) {
                WOLFSSL_MSG("Error getting ECC curve OID");
                wc_ecc_free(ecc);
                XFREE(ecc, heap, DYNAMIC_TYPE_TMP_BUFFER);
                return BAD_FUNC_ARG;
            }
        }
        else {
            WOLFSSL_MSG("Not ECC DER key either");
        }
        wc_ecc_free(ecc);
        XFREE(ecc, heap, DYNAMIC_TYPE_TMP_BUFFER);
    }
#endif /* HAVE_ECC && !NO_ASN_CRYPT */
#if defined(HAVE_ED25519) && defined(HAVE_ED25519_KEY_IMPORT) && !defined(NO_ASN_CRYPT)
    if (*algoID == 0) {
        ed25519_key *ed25519 = (ed25519_key *)XMALLOC(sizeof *ed25519, heap,
            DYNAMIC_TYPE_TMP_BUFFER);
        if (ed25519 == NULL)
            return MEMORY_E;

        tmpIdx = 0;
        if (wc_ed25519_init_ex(ed25519, heap, INVALID_DEVID) == 0) {
            if (wc_Ed25519PrivateKeyDecode(key, &tmpIdx, ed25519, keySz) == 0) {
                *algoID = ED25519k;
            }
            else {
                WOLFSSL_MSG("Not ED25519 DER key");
            }
            wc_ed25519_free(ed25519);
        }
        else {
            WOLFSSL_MSG("GetKeyOID wc_ed25519_init failed");
        }
        XFREE(ed25519, heap, DYNAMIC_TYPE_TMP_BUFFER);
    }
#endif /* HAVE_ED25519 && HAVE_ED25519_KEY_IMPORT && !NO_ASN_CRYPT */
#if defined(HAVE_ED448) && defined(HAVE_ED448_KEY_IMPORT) && !defined(NO_ASN_CRYPT)
    if (*algoID == 0) {
        ed448_key *ed448 = (ed448_key *)XMALLOC(sizeof *ed448, heap,
            DYNAMIC_TYPE_TMP_BUFFER);
        if (ed448 == NULL)
            return MEMORY_E;

        tmpIdx = 0;
        if (wc_ed448_init(ed448) == 0) {
            if (wc_Ed448PrivateKeyDecode(key, &tmpIdx, ed448, keySz) == 0) {
                *algoID = ED448k;
            }
            else {
                WOLFSSL_MSG("Not ED448 DER key");
            }
            wc_ed448_free(ed448);
        }
        else {
            WOLFSSL_MSG("GetKeyOID wc_ed448_init failed");
        }
        XFREE(ed448, heap, DYNAMIC_TYPE_TMP_BUFFER);
    }
#endif /* HAVE_ED448 && HAVE_ED448_KEY_IMPORT && !NO_ASN_CRYPT */
#if defined(HAVE_FALCON)
    if (*algoID == 0) {
        falcon_key *falcon = (falcon_key *)XMALLOC(sizeof(*falcon), heap,
            DYNAMIC_TYPE_TMP_BUFFER);
        if (falcon == NULL)
            return MEMORY_E;

        if (wc_falcon_init(falcon) != 0) {
            tmpIdx = 0;
            if (wc_falcon_set_level(falcon, 1) == 0) {
                if (wc_Falcon_PrivateKeyDecode(key, &tmpIdx, falcon, keySz)
                    == 0) {
                    *algoID = FALCON_LEVEL1k;
                }
                else {
                    WOLFSSL_MSG("Not Falcon Level 1 DER key");
                }
            }
            else if (wc_falcon_set_level(falcon, 5) == 0) {
                if (wc_Falcon_PrivateKeyDecode(key, &tmpIdx, falcon, keySz)
                    == 0) {
                    *algoID = FALCON_LEVEL5k;
                }
                else {
                    WOLFSSL_MSG("Not Falcon Level 5 DER key");
                }
            }
            else {
                WOLFSSL_MSG("GetKeyOID falcon initialization failed");
            }
            wc_falcon_free(falcon);
        }
        XFREE(falcon, heap, DYNAMIC_TYPE_TMP_BUFFER);
    }
#endif /* HAVE_FALCON */
#if defined(HAVE_DILITHIUM) && !defined(WOLFSSL_DILITHIUM_NO_SIGN) && \
    !defined(WOLFSSL_DILITHIUM_NO_VERIFY) && !defined(WOLFSSL_DILITHIUM_NO_ASN1)
    if (*algoID == 0) {
        dilithium_key *dilithium = (dilithium_key *)XMALLOC(sizeof(*dilithium),
             heap, DYNAMIC_TYPE_TMP_BUFFER);
        if (dilithium == NULL)
            return MEMORY_E;

        /* wc_dilithium_init() returns 0 on success and a non-zero value on
         * failure. */
        if (wc_dilithium_init(dilithium) == 0) {
            if ((*algoID == 0) &&
                (wc_dilithium_set_level(dilithium, WC_ML_DSA_44) == 0)) {
                tmpIdx = 0;
                if (wc_Dilithium_PrivateKeyDecode(key, &tmpIdx, dilithium,
                        keySz) == 0) {
                    *algoID = ML_DSA_LEVEL2k;
                }
                else {
                    WOLFSSL_MSG("Not Dilithium Level 2 DER key");
                }
            }
            if ((*algoID == 0) &&
                (wc_dilithium_set_level(dilithium, WC_ML_DSA_65) == 0)) {
                tmpIdx = 0;
                if (wc_Dilithium_PrivateKeyDecode(key, &tmpIdx, dilithium,
                        keySz) == 0) {
                    *algoID = ML_DSA_LEVEL3k;
                }
                else {
                    WOLFSSL_MSG("Not Dilithium Level 3 DER key");
                }
            }
            if ((*algoID == 0) &&
                (wc_dilithium_set_level(dilithium, WC_ML_DSA_87) == 0)) {
                tmpIdx = 0;
                if (wc_Dilithium_PrivateKeyDecode(key, &tmpIdx, dilithium,
                        keySz) == 0) {
                    *algoID = ML_DSA_LEVEL5k;
                }
                else {
                    WOLFSSL_MSG("Not Dilithium Level 5 DER key");
                }
            }
            else {
                WOLFSSL_MSG("GetKeyOID dilithium initialization failed");
            }
            wc_dilithium_free(dilithium);
        }
        XFREE(dilithium, heap, DYNAMIC_TYPE_TMP_BUFFER);
    }
#endif /* HAVE_DILITHIUM && !WOLFSSL_DILITHIUM_VERIFY_ONLY */
#if defined(HAVE_SPHINCS)
    if (*algoID == 0) {
        sphincs_key *sphincs = (sphincs_key *)XMALLOC(sizeof(*sphincs),
             heap, DYNAMIC_TYPE_TMP_BUFFER);
        if (sphincs == NULL)
            return MEMORY_E;

        if (wc_sphincs_init(sphincs) != 0) {
            tmpIdx = 0;
            if (wc_sphincs_set_level_and_optim(sphincs, 1, FAST_VARIANT)
                == 0) {
                if (wc_Sphincs_PrivateKeyDecode(key, &tmpIdx, sphincs,
                    keySz) == 0) {
                    *algoID = SPHINCS_FAST_LEVEL1k;
                }
                else {
                    WOLFSSL_MSG("Not Sphincs-fast Level 1 DER key");
                }
            }
            else if (wc_sphincs_set_level_and_optim(sphincs, 3, FAST_VARIANT)
                == 0) {
                if (wc_Sphincs_PrivateKeyDecode(key, &tmpIdx, sphincs,
                    keySz) == 0) {
                    *algoID = SPHINCS_FAST_LEVEL3k;
                }
                else {
                    WOLFSSL_MSG("Not Sphincs-fast Level 3 DER key");
                }
            }
            else if (wc_sphincs_set_level_and_optim(sphincs, 5, FAST_VARIANT)
                == 0) {
                if (wc_Sphincs_PrivateKeyDecode(key, &tmpIdx, sphincs,
                    keySz) == 0) {
                    *algoID = SPHINCS_FAST_LEVEL5k;
                }
                else {
                    WOLFSSL_MSG("Not Sphincs-fast Level 5 DER key");
                }
            }
            else if (wc_sphincs_set_level_and_optim(sphincs, 1, SMALL_VARIANT)
                == 0) {
                if (wc_Sphincs_PrivateKeyDecode(key, &tmpIdx, sphincs,
                    keySz) == 0) {
                    *algoID = SPHINCS_SMALL_LEVEL1k;
                }
                else {
                    WOLFSSL_MSG("Not Sphincs-small Level 1 DER key");
                }
            }
            else if (wc_sphincs_set_level_and_optim(sphincs, 3, SMALL_VARIANT)
                == 0) {
                if (wc_Sphincs_PrivateKeyDecode(key, &tmpIdx, sphincs,
                    keySz) == 0) {
                    *algoID = SPHINCS_SMALL_LEVEL3k;
                }
                else {
                    WOLFSSL_MSG("Not Sphincs-small Level 3 DER key");
                }
            }
            else if (wc_sphincs_set_level_and_optim(sphincs, 5, SMALL_VARIANT)
                == 0) {
                if (wc_Sphincs_PrivateKeyDecode(key, &tmpIdx, sphincs,
                    keySz) == 0) {
                    *algoID = SPHINCS_SMALL_LEVEL5k;
                }
                else {
                    WOLFSSL_MSG("Not Sphincs-small Level 5 DER key");
                }
            }
            else {
                WOLFSSL_MSG("GetKeyOID sphincs initialization failed");
            }
            wc_sphincs_free(sphincs);
        }
        XFREE(sphincs, heap, DYNAMIC_TYPE_TMP_BUFFER);
    }
#endif /* HAVE_SPHINCS */

    /* if flag is not set then this is not a key that we understand. */
    if (*algoID == 0) {
        WOLFSSL_MSG("Bad key DER or compile options");
        return BAD_FUNC_ARG;
    }

    (void)tmpIdx;
    (void)curveOID;
    (void)oidSz;
    (void)keySz;
    (void)heap;

    return 1;
}

#endif /* HAVE_PKCS8 || HAVE_PKCS12 */

#ifdef WOLFSSL_ASN_TEMPLATE
#if defined(HAVE_PKCS8) || defined(HAVE_PKCS12)
/* ASN.1 template for PBES2 parameters.
 * PKCS #5: RFC 8018, A.4 - PBES2-params without outer SEQUENCE
 *                    A.2 - PBKDF2-params
 *                    B.2 - Encryption schemes
 *                    C   - AlgorithmIdentifier
 */
static const ASNItem pbes2ParamsASN[] = {
/* KDF_SEQ                */ { 0, ASN_SEQUENCE, 1, 1, 0 },
               /* PBKDF2 */
/* KDF_OID                */     { 1, ASN_OBJECT_ID, 0, 0, 0 },
/* PBKDF2_PARAMS_SEQ      */     { 1, ASN_SEQUENCE, 1, 1, 0 },
                   /* Salt */
/* PBKDF2_PARAMS_SALT     */         { 2, ASN_OCTET_STRING, 0, 0, 0 },
                   /* Iteration count */
/* PBKDF2_PARAMS_ITER     */         { 2, ASN_INTEGER, 0, 0, 0 },
                   /* Key length */
/* PBKDF2_PARAMS_KEYLEN   */         { 2, ASN_INTEGER, 0, 0, 1 },
                   /* PRF - default is HMAC-SHA1 */
/* PBKDF2_PARAMS_PRF      */         { 2, ASN_SEQUENCE, 1, 1, 1 },
/* PBKDF2_PARAMS_PRF_OID  */             { 3, ASN_OBJECT_ID, 0, 0, 0 },
/* PBKDF2_PARAMS_PRF_NULL */             { 3, ASN_TAG_NULL, 0, 0, 1 },
/* ENCS_SEQ               */ { 0, ASN_SEQUENCE, 1, 1, 0 },
                   /* Encryption algorithm */
/* ENCS_OID               */   { 1, ASN_OBJECT_ID, 0, 0, 0 },
                   /* IV for CBC */
/* ENCS_PARAMS            */   { 1, ASN_OCTET_STRING, 0, 0, 0 },
};
enum {
    PBES2PARAMSASN_IDX_KDF_SEQ = 0,
    PBES2PARAMSASN_IDX_KDF_OID,
    PBES2PARAMSASN_IDX_PBKDF2_PARAMS_SEQ,
    PBES2PARAMSASN_IDX_PBKDF2_PARAMS_SALT,
    PBES2PARAMSASN_IDX_PBKDF2_PARAMS_ITER,
    PBES2PARAMSASN_IDX_PBKDF2_PARAMS_KEYLEN,
    PBES2PARAMSASN_IDX_PBKDF2_PARAMS_PRF,
    PBES2PARAMSASN_IDX_PBKDF2_PARAMS_PRF_OID,
    PBES2PARAMSASN_IDX_PBKDF2_PARAMS_PRF_NULL,
    PBES2PARAMSASN_IDX_ENCS_SEQ,
    PBES2PARAMSASN_IDX_ENCS_OID,
    PBES2PARAMSASN_IDX_ENCS_PARAMS
};

/* Number of items in ASN.1 template for PBES2 parameters. */
#define pbes2ParamsASN_Length (sizeof(pbes2ParamsASN) / sizeof(ASNItem))

/* ASN.1 template for PBES1 parameters.
 * PKCS #5: RFC 8018, A.3. - PBEParameter without outer SEQUENCE
 */
static const ASNItem pbes1ParamsASN[] = {
            /* Salt */
/* SALT */    { 0, ASN_OCTET_STRING, 0, 0, 0 },
            /* Iteration count */
/* ITER */    { 0, ASN_INTEGER, 0, 0, 0 },
};
enum {
    PBES1PARAMSASN_IDX_SALT = 0,
    PBES1PARAMSASN_IDX_ITER
};

/* Number of items in ASN.1 template for PBES1 parameters. */
#define pbes1ParamsASN_Length (sizeof(pbes1ParamsASN) / sizeof(ASNItem))
#endif /* HAVE_PKCS8 || HAVE_PKCS12 */
#endif /* WOLFSSL_ASN_TEMPLATE */

#ifdef HAVE_PKCS8

/*
 * Equivalent to calling TraditionalEnc with the same parameters but with
 * encAlgId set to 0. This function must be kept alive because it's sometimes
 * part of the API (WOLFSSL_ASN_API).
 */
int UnTraditionalEnc(byte* key, word32 keySz, byte* out, word32* outSz,
        const char* password, int passwordSz, int vPKCS, int vAlgo,
        byte* salt, word32 saltSz, int itt, WC_RNG* rng, void* heap)
{
    return TraditionalEnc(key, keySz, out, outSz, password, passwordSz,
                vPKCS, vAlgo, 0, salt, saltSz, itt, rng, heap);
}

static int GetAlgoV2(int encAlgId, const byte** oid, int *len, int* id,
                     int *blkSz)
{
    int ret = 0;

    switch (encAlgId) {
#if !defined(NO_DES3) && !defined(NO_SHA)
    case DESb:
        *len = sizeof(blkDesCbcOid);
        *oid = blkDesCbcOid;
        *id = PBE_SHA1_DES;
        *blkSz = 8;
        break;
    case DES3b:
        *len = sizeof(blkDes3CbcOid);
        *oid = blkDes3CbcOid;
        *id = PBE_SHA1_DES3;
        *blkSz = 8;
        break;
#endif
#if defined(WOLFSSL_AES_128) && defined(HAVE_AES_CBC)
    case AES128CBCb:
        *len = sizeof(blkAes128CbcOid);
        *oid = blkAes128CbcOid;
        *id = PBE_AES128_CBC;
        *blkSz = 16;
        break;
#endif
#if defined(WOLFSSL_AES_256) && defined(HAVE_AES_CBC)
    case AES256CBCb:
        *len = sizeof(blkAes256CbcOid);
        *oid = blkAes256CbcOid;
        *id = PBE_AES256_CBC;
        *blkSz = 16;
        break;
#endif
    default:
        (void)len;
        (void)oid;
        (void)id;
        (void)blkSz;
        ret = ALGO_ID_E;
    }

    return ret;
}

int wc_EncryptPKCS8Key_ex(byte* key, word32 keySz, byte* out, word32* outSz,
        const char* password, int passwordSz, int vPKCS, int pbeOid,
        int encAlgId, byte* salt, word32 saltSz, int itt, int hmacOid,
        WC_RNG* rng, void* heap)
{
    WC_DECLARE_VAR(saltTmp, byte, MAX_SALT_SIZE, 0);
    int genSalt = 0;
    int ret = 0;
    int version = 0;
    int pbeId = 0;
    int blockSz = 0;
    const byte* encOid = NULL;
    int encOidSz = 0;
    word32 padSz = 0;
    word32 innerLen = 0;
    const byte* pbeOidBuf = NULL;
    word32 pbeOidBufSz = 0;
    word32 pbeLen = 0;
    word32 kdfLen = 0;
    word32 encLen = 0;
    byte cbcIv[MAX_IV_SIZE];
    word32 idx = 0;
    word32 encIdx = 0;
    const byte* hmacOidBuf = NULL;
    word32 hmacOidBufSz = 0;
    byte tmpShort[MAX_SHORT_SZ];
    word32 tmpIdx = 0;

    (void)heap;

    WOLFSSL_ENTER("wc_EncryptPKCS8Key_ex");

    if (key == NULL || outSz == NULL || password == NULL) {
        ret = BAD_FUNC_ARG;
    }

    if (ret == 0) {
        ret = CheckAlgo(vPKCS, pbeOid, &pbeId, &version, &blockSz);
    }
    if (ret == 0 && (salt == NULL || saltSz == 0)) {
        genSalt = 1;
        saltSz = 8;
    }
    if (ret == 0 && version == PKCS5v2) {
        ret = GetAlgoV2(encAlgId, &encOid, &encOidSz, &pbeId, &blockSz);
    }
    if (ret == 0) {
        padSz = (word32)((blockSz - ((int)keySz & (blockSz - 1))) &
            (blockSz - 1));
        ret = SetShortInt(tmpShort, &tmpIdx, (word32)itt, MAX_SHORT_SZ);
        if (ret > 0) {
            /* inner = OCT salt INT itt */
            innerLen = 2 + saltSz + (word32)ret;
            ret = 0;
        }
    }
    if (ret == 0) {
        if (version != PKCS5v2) {
            pbeOidBuf = OidFromId((word32)pbeId, oidPBEType, &pbeOidBufSz);
            /* pbe = OBJ pbse1 SEQ [ inner ] */
            pbeLen = 2 + pbeOidBufSz + 2 + innerLen;
        }
        else {
            if (hmacOid > 0) {
                hmacOidBuf = OidFromId((word32)hmacOid, oidHmacType,
                                &hmacOidBufSz);
                innerLen += 2 + 2 + hmacOidBufSz;
            }
            pbeOidBuf = pbes2;
            pbeOidBufSz = sizeof(pbes2);
            /* kdf = OBJ pbkdf2 [ SEQ innerLen ] */
            kdfLen = 2U + (word32)sizeof(pbkdf2Oid) + 2U + innerLen;
            /* enc = OBJ enc_alg OCT iv */
            encLen = 2U + (word32)encOidSz + 2U + (word32)blockSz;
            /* pbe = OBJ pbse2 SEQ [ SEQ [ kdf ] SEQ [ enc ] ] */
            pbeLen = 2U + (word32)sizeof(pbes2) + 2U + 2U + kdfLen + 2U +
                encLen;

            ret = wc_RNG_GenerateBlock(rng, cbcIv, (word32)blockSz);
        }
    }
    if (ret == 0) {
        /* outerLen = length of PBE encoding + octet string data */
        /* Plus 2 for tag and length for pbe */
        word32 outerLen = 2 + pbeLen;
        /* Octet string tag, length */
        outerLen += 1 + SetLength(keySz + padSz, NULL);
        /* Octet string bytes */
        outerLen += keySz + padSz;
        if (out == NULL) {
            /* Sequence tag, length */
            *outSz = 1 + SetLength(outerLen, NULL) + outerLen;
            return WC_NO_ERR_TRACE(LENGTH_ONLY_E);
        }
        SetOctetString(keySz + padSz, out);

        idx += SetSequence(outerLen, out + idx);

        encIdx = idx + outerLen - keySz - padSz;
        /* Put Encrypted content in place. */
        XMEMCPY(out + encIdx, key, keySz);
        if (padSz > 0) {
            XMEMSET(out + encIdx + keySz, (int)padSz, padSz);
            keySz += padSz;
        }

        if (genSalt == 1) {
            WC_ALLOC_VAR_EX(saltTmp, byte, saltSz, heap,
                DYNAMIC_TYPE_TMP_BUFFER, ret=MEMORY_E);
            if (WC_VAR_OK(saltTmp))
            {
                salt = saltTmp;
                if ((ret = wc_RNG_GenerateBlock(rng, saltTmp, saltSz)) != 0) {
                    WOLFSSL_MSG("Error generating random salt");
                }
            }
        }
    }
    if (ret == 0) {
        ret = wc_CryptKey(password, passwordSz, salt, (int)saltSz, itt, pbeId,
                  out + encIdx, (int)keySz, version, cbcIv, 1, hmacOid);
    }
    if (ret == 0) {
        if (version != PKCS5v2) {
            /* PBE algorithm */
            idx += SetSequence(pbeLen, out + idx);
            idx += (word32)SetObjectId((int)pbeOidBufSz, out + idx);
            XMEMCPY(out + idx, pbeOidBuf, pbeOidBufSz);
            idx += pbeOidBufSz;
        }
        else {
            /* PBES2 algorithm identifier */
            idx += SetSequence(pbeLen, out + idx);
            idx += (word32)SetObjectId((int)pbeOidBufSz, out + idx);
            XMEMCPY(out + idx, pbeOidBuf, pbeOidBufSz);
            idx += pbeOidBufSz;
            /* PBES2 Parameters: SEQ [ kdf ] SEQ [ enc ] */
            idx += SetSequence(2 + kdfLen + 2 + encLen, out + idx);
            /* KDF Algorithm Identifier */
            idx += SetSequence(kdfLen, out + idx);
            idx += (word32)SetObjectId((int)sizeof(pbkdf2Oid), out + idx);
            XMEMCPY(out + idx, pbkdf2Oid, sizeof(pbkdf2Oid));
            idx += (word32)sizeof(pbkdf2Oid);
        }
        idx += SetSequence(innerLen, out + idx);
        idx += SetOctetString(saltSz, out + idx);
        XMEMCPY(out + idx, salt, saltSz); idx += saltSz;
        ret = SetShortInt(out, &idx, (word32)itt, *outSz);
        if (ret > 0)
            ret = 0;
        if (version == PKCS5v2) {
            if (hmacOid > 0) {
                idx += SetSequence(2+hmacOidBufSz, out + idx);
                idx += (word32)SetObjectId((int)hmacOidBufSz, out + idx);
                XMEMCPY(out + idx, hmacOidBuf, hmacOidBufSz);
                idx += (word32)hmacOidBufSz;
            }
        }
    }
    if (ret == 0) {
        if (version == PKCS5v2) {
            /* Encryption Algorithm Identifier */
            idx += SetSequence(encLen, out + idx);
            idx += (word32)SetObjectId(encOidSz, out + idx);
            XMEMCPY(out + idx, encOid, (size_t)encOidSz);
            idx += (word32)encOidSz;
            /* Encryption Algorithm Parameter: CBC IV */
            idx += SetOctetString((word32)blockSz, out + idx);
            XMEMCPY(out + idx, cbcIv, (size_t)blockSz);
            idx += (word32)blockSz;
        }
        idx += SetOctetString(keySz, out + idx);
        /* Default PRF - no need to write out OID */
        idx += keySz;

        ret = (int)idx;
    }

    WC_FREE_VAR_EX(saltTmp, heap, DYNAMIC_TYPE_TMP_BUFFER);

    WOLFSSL_LEAVE("wc_EncryptPKCS8Key_ex", ret);

    return ret;
}

int wc_EncryptPKCS8Key(byte* key, word32 keySz, byte* out, word32* outSz,
        const char* password, int passwordSz, int vPKCS, int pbeOid,
        int encAlgId, byte* salt, word32 saltSz, int itt, WC_RNG* rng,
        void* heap)
{
    return wc_EncryptPKCS8Key_ex(key, keySz, out, outSz, password, passwordSz,
                vPKCS, pbeOid, encAlgId, salt, saltSz, itt, 0, rng, heap);
}

int wc_DecryptPKCS8Key(byte* input, word32 sz, const char* password,
        int passwordSz)
{
    int ret;
    int length;
    word32 inOutIdx = 0;

    if (input == NULL || password == NULL) {
        return BAD_FUNC_ARG;
    }

    if (GetSequence(input, &inOutIdx, &length, sz) < 0) {
        ret = ASN_PARSE_E;
    }
    else {
        ret = DecryptContent(input + inOutIdx, sz - inOutIdx, password,
                passwordSz);
        if (ret > 0) {
            XMEMMOVE(input, input + inOutIdx, (size_t)ret);
        }
    }

    if (ret > 0) {
        /* DecryptContent will decrypt the data, but it will leave any padding
         * bytes intact. This code calculates the length without the padding
         * and we return that to the user. */
        inOutIdx = 0;
        if (GetSequence(input, &inOutIdx, &length, (word32)ret) < 0) {
            ret = ASN_PARSE_E;
        }
        else {
            ret = (int)inOutIdx + length;
        }
    }

    return ret;
}

/* Takes an unencrypted, traditional DER-encoded key and converts it to a PKCS#8
 * encrypted key. If out is not NULL, it will hold the encrypted key. If it's
 * NULL, LENGTH_ONLY_E will be returned and outSz will have the required out
 * buffer size. */
int TraditionalEnc_ex(byte* key, word32 keySz, byte* out, word32* outSz,
        const char* password, int passwordSz, int vPKCS, int vAlgo,
        int encAlgId, byte* salt, word32 saltSz, int itt, int hmacOid,
        WC_RNG* rng, void* heap)
{
    int ret = 0;
    byte *pkcs8Key = NULL;
    word32 pkcs8KeySz = 0;
    int algId = 0;
    const byte* curveOid = NULL;
    word32 curveOidSz = 0;

    if (ret == 0) {
        /* check key type and get OID if ECC */
        ret = wc_GetKeyOID(key, keySz, &curveOid, &curveOidSz, &algId, heap);
        if (ret == 1)
            ret = 0;
    }
    if (ret == 0) {
        ret = wc_CreatePKCS8Key(NULL, &pkcs8KeySz, key, keySz, algId, curveOid,
                                                                    curveOidSz);
        if (ret == WC_NO_ERR_TRACE(LENGTH_ONLY_E))
            ret = 0;
    }
    if (ret == 0) {
        pkcs8Key = (byte*)XMALLOC(pkcs8KeySz, heap, DYNAMIC_TYPE_TMP_BUFFER);
        if (pkcs8Key == NULL)
            ret = MEMORY_E;
    }
    if (ret == 0) {
        ret = wc_CreatePKCS8Key(pkcs8Key, &pkcs8KeySz, key, keySz, algId,
            curveOid, curveOidSz);
        if (ret >= 0) {
            pkcs8KeySz = (word32)ret;
            ret = 0;
        }
    }
#ifdef WOLFSSL_CHECK_MEM_ZERO
    if (ret == 0) {
        wc_MemZero_Add("TraditionalEnc pkcs8Key", pkcs8Key, pkcs8KeySz);
    }
#endif
    if (ret == 0) {
        ret = wc_EncryptPKCS8Key_ex(pkcs8Key, pkcs8KeySz, out, outSz, password,
            passwordSz, vPKCS, vAlgo, encAlgId, salt, saltSz, itt, hmacOid, rng,
            heap);
    }

    if (pkcs8Key != NULL) {
        ForceZero(pkcs8Key, pkcs8KeySz);
        XFREE(pkcs8Key, heap, DYNAMIC_TYPE_TMP_BUFFER);
    }

    (void)rng;

    return ret;
}

/* Takes an unencrypted, traditional DER-encoded key and converts it to a PKCS#8
 * encrypted key. If out is not NULL, it will hold the encrypted key. If it's
 * NULL, LENGTH_ONLY_E will be returned and outSz will have the required out
 * buffer size. */
int TraditionalEnc(byte* key, word32 keySz, byte* out, word32* outSz,
        const char* password, int passwordSz, int vPKCS, int vAlgo,
        int encAlgId, byte* salt, word32 saltSz, int itt, WC_RNG* rng,
        void* heap)
{
    return TraditionalEnc_ex(key, keySz, out, outSz, password, passwordSz,
                vPKCS, vAlgo, encAlgId, salt, saltSz, itt, 0, rng, heap);

}

/* Same as TraditionalEnc, but in the public API. */
int wc_CreateEncryptedPKCS8Key(byte* key, word32 keySz, byte* out,
        word32* outSz, const char* password, int passwordSz, int vPKCS,
        int pbeOid, int encAlgId, byte* salt, word32 saltSz, int itt,
        WC_RNG* rng, void* heap)
{
    return TraditionalEnc(key, keySz, out, outSz, password, passwordSz, vPKCS,
        pbeOid, encAlgId, salt, saltSz, itt, rng, heap);
}


#ifdef WOLFSSL_ASN_TEMPLATE
/* ASN.1 template for PKCS #8/#7 encrypted key for decrypting
 * PKCS #8: RFC 5958, 3 - EncryptedPrivateKeyInfo without outer SEQUENCE
 * PKCS #7: RFC 2315, 10.1 - EncryptedContentInfo without outer SEQUENCE
 */
static const ASNItem pkcs8DecASN[] = {
/* ENCALGO_SEQ    */ { 1, ASN_SEQUENCE, 1, 1, 0 },
/* ENCALGO_OID    */     { 2, ASN_OBJECT_ID, 0, 0, 0 },
/* ENCALGO_PARAMS */     { 2, ASN_SEQUENCE, 1, 0, 0 },
            /* PKCS #7 */
/* ENCCONTENT     */ { 1, ASN_CONTEXT_SPECIFIC | ASN_ENC_CONTENT,
                                       0, 0, 2 },
            /* PKCS #8 */
/* ENCDATA        */ { 1, ASN_OCTET_STRING, 0, 0, 2 },
};
enum {
    PKCS8DECASN_IDX_ENCALGO_SEQ = 0,
    PKCS8DECASN_IDX_ENCALGO_OID,
    PKCS8DECASN_IDX_ENCALGO_PARAMS,
    PKCS8DECASN_IDX_ENCCONTENT,
    PKCS8DECASN_IDX_ENCDATA
};

/* Number of items in ASN.1 template for PKCS #8/#7 encrypted key. */
#define pkcs8DecASN_Length (sizeof(pkcs8DecASN) / sizeof(ASNItem))
#endif

/* Decrypt data using PBE algorithm.
 *
 * PKCS #8: RFC 5958, 3 - EncryptedPrivateKeyInfo without outer SEQUENCE
 * PKCS #7: RFC 2315, 10.1 - EncryptedContentInfo without outer SEQUENCE
 *
 * Note: input buffer is overwritten with decrypted data!
 *
 * Salt is in KDF parameters and IV is PBE parameters when needed.
 *
 * @param [in] input       Data to decrypt and unwrap.
 * @param [in] sz          Size of encrypted data.
 * @param [in] password    Password to derive encryption key with.
 * @param [in] passwordSz  Size of password in bytes.
 * @return  Length of decrypted data on success.
 * @return  MEMORY_E when dynamic memory allocation fails.
 * @return  ASN_PARSE_E when BER encoded data does not match ASN.1 items or
 *          is invalid.
 * @return  BUFFER_E when data in buffer is too small.
 * @return  ASN_OBJECT_ID_E when the expected OBJECT_ID tag is not found.
 * @return  Other when decryption fails.
 */
int DecryptContent(byte* input, word32 sz, const char* password, int passwordSz)
{
#ifndef WOLFSSL_ASN_TEMPLATE
    word32 inOutIdx = 0, seqEnd, oid, shaOid = 0, seqPkcs5End = sz;
    int    ret = 0, first, second, length = 0, version, saltSz, id = 0;
    int    iterations = 0, keySz = 0;
#ifdef WOLFSSL_SMALL_STACK
    byte*  salt = NULL;
    byte*  cbcIv = NULL;
#else
    byte   salt[MAX_SALT_SIZE];
    byte   cbcIv[MAX_IV_SIZE];
#endif
    byte   tag;

    if (passwordSz < 0) {
        WOLFSSL_MSG("Bad password size");
        return BAD_FUNC_ARG;
    }

    if (GetAlgoId(input, &inOutIdx, &oid, oidIgnoreType, sz) < 0) {
        ERROR_OUT(ASN_PARSE_E, exit_dc);
    }

    first  = input[inOutIdx - 2];   /* PKCS version always 2nd to last byte */
    second = input[inOutIdx - 1];   /* version.algo, algo id last byte */

    if (CheckAlgo(first, second, &id, &version, NULL) < 0) {
        ERROR_OUT(ASN_INPUT_E, exit_dc); /* Algo ID error */
    }

    if (version == PKCS5v2) {
        if (GetSequence(input, &inOutIdx, &length, sz) < 0) {
            ERROR_OUT(ASN_PARSE_E, exit_dc);
        }
        seqPkcs5End = inOutIdx + length;

        if (GetAlgoId(input, &inOutIdx, &oid, oidKdfType, sz) < 0) {
            ERROR_OUT(ASN_PARSE_E, exit_dc);
        }

        if (oid != PBKDF2_OID) {
            ERROR_OUT(ASN_PARSE_E, exit_dc);
        }
    }

    if (GetSequence(input, &inOutIdx, &length, sz) <= 0) {
        ERROR_OUT(ASN_PARSE_E, exit_dc);
    }
    /* Find the end of this SEQUENCE so we can check for the OPTIONAL and
     * DEFAULT items. */
    seqEnd = inOutIdx + (word32)length;

    ret = GetOctetString(input, &inOutIdx, &saltSz, seqEnd);
    if (ret < 0)
        goto exit_dc;

    if (saltSz > MAX_SALT_SIZE) {
        ERROR_OUT(ASN_PARSE_E, exit_dc);
    }

    WC_ALLOC_VAR_EX(salt, byte, MAX_SALT_SIZE, NULL, DYNAMIC_TYPE_TMP_BUFFER,
        ERROR_OUT(MEMORY_E,exit_dc));

    XMEMCPY(salt, &input[inOutIdx], (size_t)saltSz);
    inOutIdx += (word32)saltSz;

    if (GetShortInt(input, &inOutIdx, &iterations, seqEnd) < 0) {
        ERROR_OUT(ASN_PARSE_E, exit_dc);
    }

    /* OPTIONAL key length */
    if (seqEnd > inOutIdx) {
        word32 localIdx = inOutIdx;

        if (GetASNTag(input, &localIdx, &tag, seqEnd) < 0) {
            ERROR_OUT(ASN_PARSE_E, exit_dc);
        }

        if (tag == ASN_INTEGER &&
                GetShortInt(input, &inOutIdx, &keySz, seqEnd) < 0) {
            ERROR_OUT(ASN_PARSE_E, exit_dc);
        }
    }

    /* DEFAULT HMAC is SHA-1 */
    if (seqEnd > inOutIdx) {
        if (GetAlgoId(input, &inOutIdx, &oid, oidHmacType, seqEnd) < 0) {
            ERROR_OUT(ASN_PARSE_E, exit_dc);
        }

        shaOid = oid;
    }

    WC_ALLOC_VAR_EX(cbcIv, byte, MAX_IV_SIZE, NULL, DYNAMIC_TYPE_TMP_BUFFER,
        ERROR_OUT(MEMORY_E,exit_dc));

    if (version == PKCS5v2) {
        /* get encryption algo */
        if (GetAlgoId(input, &inOutIdx, &oid, oidBlkType, seqPkcs5End) < 0) {
            ERROR_OUT(ASN_PARSE_E, exit_dc);
        }

        if (CheckAlgoV2((int)oid, &id, NULL) < 0) {
            ERROR_OUT(ASN_PARSE_E, exit_dc); /* PKCS v2 algo id error */
        }

        if (shaOid == 0)
            shaOid = oid;

        ret = GetOctetString(input, &inOutIdx, &length, seqPkcs5End);
        if (ret < 0)
            goto exit_dc;

        if (length > MAX_IV_SIZE) {
            ERROR_OUT(ASN_PARSE_E, exit_dc);
        }

        XMEMCPY(cbcIv, &input[inOutIdx], (size_t)length);
        inOutIdx += (word32)length;
    }

    if (GetASNTag(input, &inOutIdx, &tag, sz) < 0) {
        ERROR_OUT(ASN_PARSE_E, exit_dc);
    }

    if (tag != (ASN_CONTEXT_SPECIFIC | 0) && tag != ASN_OCTET_STRING) {
        ERROR_OUT(ASN_PARSE_E, exit_dc);
    }

    if (GetLength(input, &inOutIdx, &length, sz) < 0) {
        ERROR_OUT(ASN_PARSE_E, exit_dc);
    }

    ret = wc_CryptKey(password, passwordSz, salt, saltSz, iterations, id,
                   input + inOutIdx, length, version, cbcIv, 0, (int)shaOid);

exit_dc:
    WC_FREE_VAR_EX(salt, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    WC_FREE_VAR_EX(cbcIv, NULL, DYNAMIC_TYPE_TMP_BUFFER);

    if (ret == 0) {
        XMEMMOVE(input, input + inOutIdx, (size_t)length);
        ret = length;
    }

    return ret;
#else
    /* pbes2ParamsASN longer than pkcs8DecASN_Length/pbes1ParamsASN_Length. */
    DECL_ASNGETDATA(dataASN, pbes2ParamsASN_Length);
    int    ret = 0;
    int    id = 0;
    int    version = 0;
    word32 idx = 0;
    word32 pIdx = 0;
    word32 iterations = 0;
    word32 keySz = 0;
    word32 saltSz = 0;
    word32 shaOid = 0;
    byte*  salt = NULL;
    byte*  key = NULL;
    byte   cbcIv[MAX_IV_SIZE];
    byte*  params = NULL;

    WOLFSSL_ENTER("DecryptContent");

    CALLOC_ASNGETDATA(dataASN, pbes2ParamsASN_Length, ret, NULL);

    if (ret == 0) {
        /* Check OID is a PBE Type */
        GetASN_OID(&dataASN[PKCS8DECASN_IDX_ENCALGO_OID], oidPBEType);
        ret = GetASN_Items(pkcs8DecASN, dataASN, pkcs8DecASN_Length, 0, input,
                           &idx, sz);
    }
    if (ret == 0) {
        /* Check the PBE algorithm and get the version and id. */
        idx = dataASN[PKCS8DECASN_IDX_ENCALGO_OID].data.oid.length;
        /* Second last byte: 1 (PKCS #12 PBE Id) or 5 (PKCS #5)
         * Last byte: Alg or PBES2 */
        ret = CheckAlgo(
            dataASN[PKCS8DECASN_IDX_ENCALGO_OID].data.oid.data[idx - 2],
            dataASN[PKCS8DECASN_IDX_ENCALGO_OID].data.oid.data[idx - 1], &id,
            &version, NULL);
    }
    if (ret == 0) {
        /* Get the parameters data. */
        GetASN_GetRef(&dataASN[PKCS8DECASN_IDX_ENCALGO_PARAMS], &params, &sz);
        /* Having a numbered choice means none or both will have errored out. */
        if (dataASN[PKCS8DECASN_IDX_ENCCONTENT].tag != 0)
            GetASN_GetRef(&dataASN[PKCS8DECASN_IDX_ENCCONTENT], &key, &keySz);
        else if (dataASN[PKCS8DECASN_IDX_ENCDATA].tag != 0)
            GetASN_GetRef(&dataASN[PKCS8DECASN_IDX_ENCDATA], &key, &keySz);
        else
            ret = ASN_RSA_KEY_E;
    }

    if (ret == 0) {
        if (version != PKCS5v2) {
            /* Initialize for PBES1 parameters and put iterations in var. */
            XMEMSET(dataASN, 0, sizeof(*dataASN) * pbes1ParamsASN_Length);
            GetASN_Int32Bit(&dataASN[PBES1PARAMSASN_IDX_ITER], &iterations);
            /* Parse the PBES1 parameters. */
            ret = GetASN_Items(pbes1ParamsASN, dataASN, pbes1ParamsASN_Length,
                               0, params, &pIdx, sz);
            if (ret == 0) {
                /* Get the salt data. */
                GetASN_GetRef(&dataASN[PBES1PARAMSASN_IDX_SALT], &salt,
                    &saltSz);
            }
        }
        else {
            word32 ivSz = MAX_IV_SIZE;

            /* Initialize for PBES2 parameters. Put iterations in var; match
             * KDF, HMAC and cipher, and copy CBC into buffer. */
            XMEMSET(dataASN, 0, sizeof(*dataASN) * pbes2ParamsASN_Length);
            GetASN_ExpBuffer(&dataASN[PBES2PARAMSASN_IDX_KDF_OID], pbkdf2Oid,
                sizeof(pbkdf2Oid));
            GetASN_Int32Bit(&dataASN[PBES2PARAMSASN_IDX_PBKDF2_PARAMS_ITER],
                &iterations);
            GetASN_OID(&dataASN[PBES2PARAMSASN_IDX_PBKDF2_PARAMS_PRF_OID],
                oidHmacType);
            GetASN_OID(&dataASN[PBES2PARAMSASN_IDX_ENCS_OID], oidBlkType);
            GetASN_Buffer(&dataASN[PBES2PARAMSASN_IDX_ENCS_PARAMS], cbcIv,
                &ivSz);
            /* Parse the PBES2 parameters  */
            ret = GetASN_Items(pbes2ParamsASN, dataASN, pbes2ParamsASN_Length,
                               0, params, &pIdx, sz);
            if (ret == 0) {
                /* Get the salt data. */
                GetASN_GetRef(&dataASN[PBES2PARAMSASN_IDX_PBKDF2_PARAMS_SALT],
                    &salt, &saltSz);
                /* Get the digest and encryption algorithm id. */
                shaOid = dataASN[PBES2PARAMSASN_IDX_PBKDF2_PARAMS_PRF_OID].data.oid.sum; /* Default HMAC-SHA1 */
                id     = (int)dataASN[PBES2PARAMSASN_IDX_ENCS_OID].data.oid.sum;
                /* Convert encryption algorithm to a PBE algorithm if needed. */
                CheckAlgoV2(id, &id, NULL);
            }
        }
    }

    if (ret == 0) {
        /* Decrypt the key. */
        ret = wc_CryptKey(
            password, passwordSz, salt, (int)saltSz, (int)iterations, id, key,
            (int)keySz, version, cbcIv, 0, (int)shaOid);
    }
    if (ret == 0) {
        /* Copy the decrypted key into the input (inline). */
        XMEMMOVE(input, key, keySz);
        ret = (int)keySz;
    }

    FREE_ASNGETDATA(dataASN, NULL);
    return ret;
#endif
}

/* Decrypt data using PBE algorithm and get key from PKCS#8 wrapping.
 *
 * PKCS #8: RFC 5958, 3 - EncryptedPrivateKeyInfo
 * PKCS #7: RFC 2315, 10.1 - EncryptedContentInfo
 *
 * Note: input buffer is overwritten with decrypted key!
 *
 * Salt is in KDF parameters and IV is PBE parameters when needed.
 *
 * @param [in]  input       Data to decrypt and unwrap.
 * @param [in]  sz          Size of encrypted data.
 * @param [in]  password    Password to derive encryption key with.
 * @param [in]  passwordSz  Size of password in bytes.
 * @param [out] algId       Key algorithm from PKCS#8 wrapper.
 * @return  Length of decrypted data on success.
 * @return  MEMORY_E when dynamic memory allocation fails.
 * @return  ASN_PARSE_E when BER encoded data does not match ASN.1 items or
 *          is invalid.
 * @return  BUFFER_E when data in buffer is too small.
 * @return  ASN_OBJECT_ID_E when the expected OBJECT_ID tag is not found.
 * @return  Other when decryption fails.
 */
int ToTraditionalEnc(byte* input, word32 sz, const char* password,
                     int passwordSz, word32* algId)
{
    int ret;

    ret = wc_DecryptPKCS8Key(input, sz, password, passwordSz);
    if (ret > 0) {
        ret = ToTraditional_ex(input, (word32)ret, algId);
    }

    return ret;
}

#endif /* HAVE_PKCS8 */

#ifdef HAVE_PKCS12

#ifdef WOLFSSL_ASN_TEMPLATE
/* ASN.1 template for PKCS #8 encrypted key with PBES2 parameters.
 * PKCS #8: RFC 5958, 3 - EncryptedPrivateKeyInfo
 * PKCS #5: RFC 8018, A.4 - PBES2
 */
static const ASNItem p8EncPbes2ASN[] = {
/* SEQ */           { 0, ASN_SEQUENCE, 1, 1, 0 },
/* ALGO_SEQ */          { 1, ASN_SEQUENCE, 1, 1, 0 },
                    /* PBE algorithm */
/* ALGO_OID */              { 2, ASN_OBJECT_ID, 0, 0, 0 },
/* ALGO_PARAMS_SEQ */       { 2, ASN_SEQUENCE, 1, 1, 0 },
/* ALGO_PARAMS_KDF_SEQ */       { 3, ASN_SEQUENCE, 1, 1, 0 },
               /* PBKDF2 */
/* ALGO_PARAMS_KDF_OID */           { 4, ASN_OBJECT_ID, 0, 0, 0 },
/* ALGO_PARAMS_PBKDF2_SEQ */        { 4, ASN_SEQUENCE, 1, 1, 0 },
                   /* Salt */
/* ALGO_PARAMS_PBKDF2_SALT */           { 5, ASN_OCTET_STRING, 0, 0, 0 },
                   /* Iteration count */
/* ALGO_PARAMS_PBKDF2_ITER */           { 5, ASN_INTEGER, 0, 0, 0 },
                   /* Key length */
/* ALGO_PARAMS_PBKDF2_KEYLEN */         { 5, ASN_INTEGER, 0, 0, 1 },
                   /* PRF - default is HMAC-SHA1 */
/* ALGO_PARAMS_PBKDF2_PRF */            { 5, ASN_SEQUENCE, 1, 1, 1 },
/* ALGO_PARAMS_PBKDF2_PRF_OID */            { 6, ASN_OBJECT_ID, 0, 0, 0 },
/* ALGO_PARAMS_PBKDF2_PRF_NULL */           { 6, ASN_TAG_NULL, 0, 0, 1 },
/* ALGO_ENCS_SEQ */             { 3, ASN_SEQUENCE, 1, 1, 0 },
                   /* Encryption algorithm */
/* ALGO_ENCS_OID */                 { 4, ASN_OBJECT_ID, 0, 0, 0 },
                   /* IV for CBC */
/* ALGO_ENCS_PARAMS */              { 4, ASN_OCTET_STRING, 0, 0, 0 },
/* ENCDATA */           { 1, (ASN_CONTEXT_SPECIFIC | 0), 0, 0, 0 },
};
enum {
    P8ENCPBES2ASN_IDX_SEQ = 0,
    P8ENCPBES2ASN_IDX_ALGO_SEQ,
    P8ENCPBES2ASN_IDX_ALGO_OID,
    P8ENCPBES2ASN_IDX_ALGO_PARAMS_SEQ,
    P8ENCPBES2ASN_IDX_ALGO_PARAMS_KDF_SEQ,
    P8ENCPBES2ASN_IDX_ALGO_PARAMS_KDF_OID,
    P8ENCPBES2ASN_IDX_ALGO_PARAMS_PBKDF2_SEQ,
    P8ENCPBES2ASN_IDX_ALGO_PARAMS_PBKDF2_SALT,
    P8ENCPBES2ASN_IDX_ALGO_PARAMS_PBKDF2_ITER,
    P8ENCPBES2ASN_IDX_ALGO_PARAMS_PBKDF2_KEYLEN,
    P8ENCPBES2ASN_IDX_ALGO_PARAMS_PBKDF2_PRF,
    P8ENCPBES2ASN_IDX_ALGO_PARAMS_PBKDF2_PRF_OID,
    P8ENCPBES2ASN_IDX_ALGO_PARAMS_PBKDF2_PRF_NULL,
    P8ENCPBES2ASN_IDX_ALGO_ENCS_SEQ,
    P8ENCPBES2ASN_IDX_ALGO_ENCS_OID,
    P8ENCPBES2ASN_IDX_ALGO_ENCS_PARAMS,
    P8ENCPBES2ASN_IDX_ENCDATA
};

#define p8EncPbes2ASN_Length (sizeof(p8EncPbes2ASN) / sizeof(ASNItem))
#endif /* WOLFSSL_ASN_TEMPLATE */

static int EncryptContentPBES2(byte* input, word32 inputSz, byte* out,
        word32* outSz, const char* password, int passwordSz, int encAlgId,
        byte* salt, word32 saltSz, int itt, int hmacOid, WC_RNG* rng,
        void* heap)
{
    int ret = 0;
#ifndef WOLFSSL_ASN_TEMPLATE
    (void)input;
    (void)inputSz;
    (void)out;
    (void)outSz;
    (void)password;
    (void)passwordSz;
    (void)encAlgId;
    (void)salt;
    (void)saltSz;
    (void)itt;
    (void)hmacOid;
    (void)rng;
    (void)heap;
    ret = ASN_VERSION_E;
#else /* WOLFSSL_ASN_TEMPLATE */
    /* PBES2 is only supported when enabling the ASN template */

    DECL_ASNSETDATA(dataASN, p8EncPbes2ASN_Length);
    const byte* blkOidBuf = NULL;
    int blkOidSz = 0;
    int pbesId = -1;
    int blockSz = 0;
    int asnSz = 0;
    word32 pkcs8Sz = 0;
    byte* cbcIv = NULL;
    byte* saltEnc = NULL;
    int genSalt = (salt == NULL || saltSz == 0);

    WOLFSSL_ENTER("EncryptContentPBES2");

    /* Must have a output size to return or check. */
    if (outSz == NULL) {
        ret = BAD_FUNC_ARG;
    }
    if ((ret == 0) && genSalt) {
        salt = NULL;
        saltSz = PKCS5V2_SALT_SZ;
        /* Salt generated into encoding below. */
    }
    /* Check salt size is valid. */
    if ((ret == 0) && (saltSz > MAX_SALT_SIZE)) {
        ret = ASN_PARSE_E;
    }
    if ((ret == 0) && GetAlgoV2(encAlgId, &blkOidBuf, &blkOidSz, &pbesId,
        &blockSz) < 0) {
        ret = ASN_INPUT_E;
    }
    CALLOC_ASNSETDATA(dataASN, p8EncPbes2ASN_Length, ret, heap);

    if (ret == 0) {
        /* Setup data to go into encoding including PBE algorithm, salt,
         * iteration count, and padded key length. */
        SetASN_OID(&dataASN[P8ENCPBES2ASN_IDX_ALGO_OID], (word32)PBES2,
                   oidPBEType);
        SetASN_Buffer(&dataASN[P8ENCPBES2ASN_IDX_ALGO_PARAMS_KDF_OID],
            pbkdf2Oid, sizeof(pbkdf2Oid));
        SetASN_Buffer(&dataASN[P8ENCPBES2ASN_IDX_ALGO_PARAMS_PBKDF2_SALT], NULL,
            saltSz);
        SetASN_Int16Bit(&dataASN[P8ENCPBES2ASN_IDX_ALGO_PARAMS_PBKDF2_ITER],
            (word16)itt);
        dataASN[P8ENCPBES2ASN_IDX_ALGO_PARAMS_PBKDF2_KEYLEN].noOut = 1;
        if (hmacOid > 0) {
            const byte* hmacOidBuf = NULL;
            word32 hmacOidSz = 0;
            hmacOidBuf = OidFromId((word32)hmacOid, oidHmacType, &hmacOidSz);
            if (hmacOidBuf == NULL) {
                ret = ASN_PARSE_E;
            }
            if (ret == 0) {
                SetASN_Buffer(
                    &dataASN[P8ENCPBES2ASN_IDX_ALGO_PARAMS_PBKDF2_PRF_OID],
                    hmacOidBuf, hmacOidSz);
            }
        }
        else {
            /* SHA1 will be used as default without PRF parameters */
            dataASN[P8ENCPBES2ASN_IDX_ALGO_PARAMS_PBKDF2_PRF].noOut = 1;
            dataASN[P8ENCPBES2ASN_IDX_ALGO_PARAMS_PBKDF2_PRF_OID].noOut = 1;
            dataASN[P8ENCPBES2ASN_IDX_ALGO_PARAMS_PBKDF2_PRF_NULL].noOut = 1;
        }
        SetASN_Buffer(&dataASN[P8ENCPBES2ASN_IDX_ALGO_ENCS_OID], blkOidBuf,
            blkOidSz);
        SetASN_Buffer(&dataASN[P8ENCPBES2ASN_IDX_ALGO_ENCS_PARAMS], NULL,
            blockSz);
        pkcs8Sz = wc_PkcsPad(NULL, inputSz, (word32)blockSz);
        SetASN_Buffer(&dataASN[P8ENCPBES2ASN_IDX_ENCDATA], NULL, pkcs8Sz);

        /* Calculate size of encoding. */
        ret = SizeASN_Items(p8EncPbes2ASN + P8ENCPBES2ASN_IDX_ALGO_SEQ,
                dataASN + P8ENCPBES2ASN_IDX_ALGO_SEQ,
                (int)(p8EncPbes2ASN_Length - P8ENCPBES2ASN_IDX_ALGO_SEQ),
                &asnSz);
    }
    /* Return size when no output buffer. */
    if ((ret == 0) && (out == NULL)) {
        *outSz = (word32)asnSz;
        ret = WC_NO_ERR_TRACE(LENGTH_ONLY_E);
    }
    /* Check output buffer is big enough for encoded data. */
    if ((ret == 0) && (asnSz > (int)*outSz)) {
        ret = BAD_FUNC_ARG;
    }
    if (ret == 0) {
        /* Encode PKCS#8 key. */
        SetASN_Items(p8EncPbes2ASN + P8ENCPBES2ASN_IDX_ALGO_SEQ,
                 dataASN + P8ENCPBES2ASN_IDX_ALGO_SEQ,
                 (int)(p8EncPbes2ASN_Length - P8ENCPBES2ASN_IDX_ALGO_SEQ),
                 out);

        saltEnc = (byte*)
            dataASN[P8ENCPBES2ASN_IDX_ALGO_PARAMS_PBKDF2_SALT].data.buffer.data;
        if (genSalt) {
            /* Generate salt into encoding. */
            ret = wc_RNG_GenerateBlock(rng, saltEnc, saltSz);
        }
        else {
            XMEMCPY(saltEnc, salt, saltSz);
        }
    }
    if (ret == 0) {
        cbcIv = (byte*)
            dataASN[P8ENCPBES2ASN_IDX_ALGO_ENCS_PARAMS].data.buffer.data;
        ret = wc_RNG_GenerateBlock(rng, cbcIv, (word32)blockSz);
    }
    if (ret == 0) {
        /* Store PKCS#8 key in output buffer. */
        byte* pkcs8 = (byte*)
            dataASN[P8ENCPBES2ASN_IDX_ENCDATA].data.buffer.data;
        XMEMCPY(pkcs8, input, inputSz);
        (void)wc_PkcsPad(pkcs8, inputSz, (word32)blockSz);

        /* Encrypt PKCS#8 key inline. */
        ret = wc_CryptKey(password, passwordSz, saltEnc, (int)saltSz, itt,
            pbesId, pkcs8, (int)pkcs8Sz, PKCS5v2, cbcIv, 1, hmacOid);
    }
    if (ret == 0) {
        /* Returning size on success. */
        ret = asnSz;
    }

    FREE_ASNSETDATA(dataASN, heap);
    (void)heap;
#endif /* WOLFSSL_ASN_TEMPLATE */
    return ret;
}

#ifdef WOLFSSL_ASN_TEMPLATE
/* ASN.1 template for PKCS #8 encrypted key with PBES1 parameters.
 * PKCS #8: RFC 5958, 3 - EncryptedPrivateKeyInfo
 * PKCS #5: RFC 8018, A.3 - PBEParameter
 */
static const ASNItem p8EncPbes1ASN[] = {
/* SEQ                   */ { 0, ASN_SEQUENCE, 1, 1, 0 },
/* ENCALGO_SEQ           */     { 1, ASN_SEQUENCE, 1, 1, 0 },
                    /* PBE algorithm */
/* ENCALGO_OID           */         { 2, ASN_OBJECT_ID, 0, 0, 0 },
/* ENCALGO_PBEPARAM_SEQ  */         { 2, ASN_SEQUENCE, 1, 1, 0 },
                        /* Salt */
/* ENCALGO_PBEPARAM_SALT */             { 3, ASN_OCTET_STRING, 0, 0, 0 },
                        /* Iteration Count */
/* ENCALGO_PBEPARAM_ITER */             { 3, ASN_INTEGER, 0, 0, 0 },
/* ENCDATA               */     { 1, (ASN_CONTEXT_SPECIFIC | 0), 0, 0, 0 },
};
enum {
    P8ENCPBES1ASN_IDX_SEQ = 0,
    P8ENCPBES1ASN_IDX_ENCALGO_SEQ,
    P8ENCPBES1ASN_IDX_ENCALGO_OID,
    P8ENCPBES1ASN_IDX_ENCALGO_PBEPARAM_SEQ,
    P8ENCPBES1ASN_IDX_ENCALGO_PBEPARAM_SALT,
    P8ENCPBES1ASN_IDX_ENCALGO_PBEPARAM_ITER,
    P8ENCPBES1ASN_IDX_ENCDATA
};

#define p8EncPbes1ASN_Length (sizeof(p8EncPbes1ASN) / sizeof(ASNItem))
#endif /* WOLFSSL_ASN_TEMPLATE */

/* Wrap a private key in PKCS#8 and encrypt.
 *
 * Used for PKCS#12 and PKCS#7.
 * vPKCS is the version of PKCS to use.
 * vAlgo is the algorithm version to use.
 *
 * When salt is NULL, a random number is generated.
 *
 * data returned is :
 * [ seq - obj [ seq -salt,itt]] , construct with encrypted data
 *
 * @param [in]  input       Data to encrypt.
 * @param [in]  inputSz     Length of data in bytes.
 * @param [out] out         Buffer to write wrapped encrypted data into.
 * @param [out] outSz       Length of encrypted data in bytes.
 * @param [in]  password    Password used to create encryption key.
 * @param [in]  passwordSz  Length of password in bytes.
 * @param [in]  vPKCS       First byte used to determine PBE algorithm.
 * @param [in]  vAlgo       Second byte used to determine PBE algorithm.
 * @param [in]  encAlgId    Encryption Algorithm for PBES2.
 * @param [in]  salt        Salt to use with KDF.
 * @param [in]  saltSz      Length of salt in bytes.
 * @param [in]  itt         Number of iterations to use in KDF.
 * @param [in]  hmacOid     HMAC Algorithm for PBES2.
 * @param [in]  rng         Random number generator to use to generate salt.
 * @param [in]  heap        Dynamic memory allocator hint.
 * @return  The size of encrypted data on success
 * @return  LENGTH_ONLY_E when out is NULL and able to encode.
 * @return  ASN_PARSE_E when the salt size is too large.
 * @return  ASN_VERSION_E when attempting to use a PBES2 algorithm (use
 *          TraditionalEnc).
 * @return  MEMORY_E when dynamic memory allocation fails.
 * @return  Other when encryption or random number generation fails.
 */
int EncryptContent(byte* input, word32 inputSz, byte* out, word32* outSz,
        const char* password, int passwordSz, int vPKCS, int vAlgo,
        int encAlgId, byte* salt, word32 saltSz, int itt, int hmacOid,
        WC_RNG* rng, void* heap)
{
#ifndef WOLFSSL_ASN_TEMPLATE
    word32 sz;
    word32 inOutIdx = 0;
    word32 tmpIdx   = 0;
    word32 totalSz  = 0;
    word32 seqSz;
    word32 innerSz;
    int    ret;
    int    version, id = PBE_NONE, blockSz = 0;
#ifdef WOLFSSL_SMALL_STACK
    byte*  saltTmp = NULL;
    byte*  cbcIv   = NULL;
#else
    byte   saltTmp[MAX_SALT_SIZE];
    byte   cbcIv[MAX_IV_SIZE];
#endif
    byte   seq[MAX_SEQ_SZ];
    byte   shr[MAX_SHORT_SZ];
    word32 maxShr = MAX_SHORT_SZ;
    word32 algoSz;
    const  byte* algoName;

    (void)encAlgId;
    (void)hmacOid;
    (void)heap;

    (void)EncryptContentPBES2;

    WOLFSSL_ENTER("EncryptContent");

    if (CheckAlgo(vPKCS, vAlgo, &id, &version, &blockSz) < 0)
        return ASN_INPUT_E;  /* Algo ID error */

    if (version == PKCS5v2) {
        WOLFSSL_MSG("PKCS#5 version 2 not supported yet");
        return BAD_FUNC_ARG;
    }

    if (saltSz > MAX_SALT_SIZE)
        return ASN_PARSE_E;

    if (outSz == NULL) {
        return BAD_FUNC_ARG;
    }

    /* calculate size */
    /* size of constructed string at end */
    sz = wc_PkcsPad(NULL, inputSz, (word32)blockSz);
    totalSz  = ASN_TAG_SZ;
    totalSz += SetLength(sz, seq);
    totalSz += sz;

    /* size of sequence holding object id and sub sequence of salt and itt */
    algoName = OidFromId((word32)id, oidPBEType, &algoSz);
    if (algoName == NULL) {
        WOLFSSL_MSG("Unknown Algorithm");
        return 0;
    }
    innerSz = (word32)SetObjectId((int)algoSz, seq);
    innerSz += algoSz;

    /* get subsequence of salt and itt */
    if (salt == NULL || saltSz == 0) {
        sz = 8;
    }
    else {
        sz = saltSz;
    }
    seqSz  = SetOctetString(sz, seq);
    seqSz += sz;

    tmpIdx = 0;
    ret = SetShortInt(shr, &tmpIdx, (word32)itt, maxShr);
    if (ret >= 0) {
        seqSz += (word32)ret;
    }
    else {
        return ret;
    }
    innerSz += seqSz + SetSequence(seqSz, seq);
    totalSz += innerSz + SetSequence(innerSz, seq);

    if (out == NULL) {
        *outSz = totalSz;
        return WC_NO_ERR_TRACE(LENGTH_ONLY_E);
    }

    inOutIdx = 0;
    if (totalSz > *outSz)
        return BUFFER_E;

    inOutIdx += SetSequence(innerSz, out + inOutIdx);
    inOutIdx += (word32)SetObjectId((int)algoSz, out + inOutIdx);
    XMEMCPY(out + inOutIdx, algoName, algoSz);
    inOutIdx += algoSz;
    inOutIdx += SetSequence(seqSz, out + inOutIdx);

    /* create random salt if one not provided */
    if (salt == NULL || saltSz == 0) {
        saltSz = 8;
        WC_ALLOC_VAR_EX(saltTmp, byte, saltSz, heap, DYNAMIC_TYPE_TMP_BUFFER,
            return MEMORY_E);
        salt = saltTmp;

        if ((ret = wc_RNG_GenerateBlock(rng, saltTmp, saltSz)) != 0) {
            WOLFSSL_MSG("Error generating random salt");
            WC_FREE_VAR_EX(saltTmp, heap, DYNAMIC_TYPE_TMP_BUFFER);
            return ret;
        }
    }
    inOutIdx += SetOctetString(saltSz, out + inOutIdx);
    if (saltSz + inOutIdx > *outSz) {
        WC_FREE_VAR_EX(saltTmp, heap, DYNAMIC_TYPE_TMP_BUFFER);
        return BUFFER_E;
    }
    XMEMCPY(out + inOutIdx, salt, saltSz);
    inOutIdx += saltSz;

    /* place iteration setting in buffer */
    ret = SetShortInt(out, &inOutIdx, (word32)itt, *outSz);
    if (ret < 0) {
        WC_FREE_VAR_EX(saltTmp, heap, DYNAMIC_TYPE_TMP_BUFFER);
        return ret;
    }

    if (inOutIdx + 1 > *outSz) {
        WC_FREE_VAR_EX(saltTmp, heap, DYNAMIC_TYPE_TMP_BUFFER);
        return BUFFER_E;
    }
    out[inOutIdx++] = ASN_CONTEXT_SPECIFIC | 0;

    /* get pad size and verify buffer room */
    sz = wc_PkcsPad(NULL, inputSz, (word32)blockSz);
    if (sz + inOutIdx > *outSz) {
        WC_FREE_VAR_EX(saltTmp, heap, DYNAMIC_TYPE_TMP_BUFFER);
        return BUFFER_E;
    }
    inOutIdx += SetLength(sz, out + inOutIdx);

    /* copy input to output buffer and pad end */
    XMEMCPY(out + inOutIdx, input, inputSz);
    sz = wc_PkcsPad(out + inOutIdx, inputSz, (word32)blockSz);
#ifdef WOLFSSL_SMALL_STACK
    cbcIv = (byte*)XMALLOC(MAX_IV_SIZE, heap, DYNAMIC_TYPE_TMP_BUFFER);
    if (cbcIv == NULL) {
        XFREE(saltTmp, heap, DYNAMIC_TYPE_TMP_BUFFER);
        return MEMORY_E;
    }
#endif

    /* encrypt */
    if ((ret = wc_CryptKey(password, passwordSz, salt, (int)saltSz, itt, id,
                   out + inOutIdx, (int)sz, version, cbcIv, 1, 0)) < 0) {

        WC_FREE_VAR_EX(cbcIv, heap, DYNAMIC_TYPE_TMP_BUFFER);
        WC_FREE_VAR_EX(saltTmp, heap, DYNAMIC_TYPE_TMP_BUFFER);
        return ret;  /* encrypt failure */
    }

    WC_FREE_VAR_EX(cbcIv, heap, DYNAMIC_TYPE_TMP_BUFFER);
    WC_FREE_VAR_EX(saltTmp, heap, DYNAMIC_TYPE_TMP_BUFFER);

    (void)rng;

    return (int)(inOutIdx + sz);
#else /* WOLFSSL_ASN_TEMPLATE */
    /* PBES2 is only supported when enabling the ASN template */

    DECL_ASNSETDATA(dataASN, p8EncPbes1ASN_Length);
    int ret = 0;
    int sz = 0;
    int version = 0;
    int id = -1;
    int blockSz = 0;
    word32 pkcs8Sz = 0;

    (void)heap;

    WOLFSSL_ENTER("EncryptContent");

    /* Must have a output size to return or check. */
    if (outSz == NULL) {
        ret = BAD_FUNC_ARG;
    }
    /* Check salt size is valid. */
    if ((ret == 0) && (saltSz > MAX_SALT_SIZE)) {
        ret = ASN_PARSE_E;
    }
    /* Get algorithm parameters for algorithm identifier. */
    if ((ret == 0) && CheckAlgo(vPKCS, vAlgo, &id, &version, &blockSz) < 0) {
        ret = ASN_INPUT_E;
    }
    /* Check PKCS #5 version - only PBSE1 parameters supported. */
    if ((ret == 0) && (version == PKCS5v2)) {
        return EncryptContentPBES2(input, inputSz, out, outSz, password,
            passwordSz, encAlgId, salt, saltSz, itt, hmacOid, rng, heap);
    }

    CALLOC_ASNSETDATA(dataASN, p8EncPbes1ASN_Length, ret, heap);

    if (ret == 0) {
        /* Setup data to go into encoding including PBE algorithm, salt,
         * iteration count, and padded key length. */
        SetASN_OID(&dataASN[P8ENCPBES1ASN_IDX_ENCALGO_OID], (word32)id,
                   oidPBEType);
        if (salt == NULL || saltSz == 0) {
            salt = NULL;
            saltSz = PKCS5_SALT_SZ;
            /* Salt generated into encoding below. */
        }
        SetASN_Buffer(&dataASN[P8ENCPBES1ASN_IDX_ENCALGO_PBEPARAM_SALT],
                salt, saltSz);
        SetASN_Int16Bit(&dataASN[P8ENCPBES1ASN_IDX_ENCALGO_PBEPARAM_ITER],
                        (word16)itt);
        pkcs8Sz = wc_PkcsPad(NULL, inputSz, (word32)blockSz);
        SetASN_Buffer(&dataASN[P8ENCPBES1ASN_IDX_ENCDATA], NULL, pkcs8Sz);

        /* Calculate size of encoding. */
        ret = SizeASN_Items(p8EncPbes1ASN + P8ENCPBES1ASN_IDX_ENCALGO_SEQ,
                dataASN + P8ENCPBES1ASN_IDX_ENCALGO_SEQ,
                (int)(p8EncPbes1ASN_Length - P8ENCPBES1ASN_IDX_ENCALGO_SEQ),
                &sz);
    }
    /* Return size when no output buffer. */
    if ((ret == 0) && (out == NULL)) {
        *outSz = (word32)sz;
        ret = WC_NO_ERR_TRACE(LENGTH_ONLY_E);
    }
    /* Check output buffer is big enough for encoded data. */
    if ((ret == 0) && (sz > (int)*outSz)) {
        ret = BAD_FUNC_ARG;
    }
    if (ret == 0) {
        /* Encode PKCS#8 key. */
        SetASN_Items(p8EncPbes1ASN + P8ENCPBES1ASN_IDX_ENCALGO_SEQ,
                 dataASN + P8ENCPBES1ASN_IDX_ENCALGO_SEQ,
                 (int)(p8EncPbes1ASN_Length - P8ENCPBES1ASN_IDX_ENCALGO_SEQ),
                 out);

        if (salt == NULL) {
            /* Generate salt into encoding. */
            salt = (byte*)dataASN[P8ENCPBES1ASN_IDX_ENCALGO_PBEPARAM_SALT].
                data.buffer.data;
            ret = wc_RNG_GenerateBlock(rng, salt, saltSz);
        }
    }
    if (ret == 0) {
        byte cbcIv[MAX_IV_SIZE];
        /* Store PKCS#8 key in output buffer. */
        byte* pkcs8 =
            (byte*)dataASN[P8ENCPBES1ASN_IDX_ENCDATA].data.buffer.data;
        XMEMCPY(pkcs8, input, inputSz);
        (void)wc_PkcsPad(pkcs8, inputSz, (word32)blockSz);

        /* Encrypt PKCS#8 key inline. */
        ret = wc_CryptKey(password, passwordSz, salt, (int)saltSz, itt, id,
                          pkcs8, (int)pkcs8Sz, version, cbcIv, 1, 0);
    }
    if (ret == 0) {
        /* Returning size on success. */
        ret = sz;
    }

    FREE_ASNSETDATA(dataASN, heap);
    return ret;
#endif /* WOLFSSL_ASN_TEMPLATE */
}


#endif /* HAVE_PKCS12 */
#endif /* NO_PWDBASED */

/* Block padding used for PKCS#5, PKCS#7, PKCS#8 and PKCS#12.
 *
 * The length of padding is the value of each padding byte.
 *
 * When buf is NULL, the padded size is returned.
 *
 * @param [in, out] buf      Buffer of data to be padded. May be NULL.
 * @param [in]      sz       Size of data in bytes.
 * @param [in]      blockSz  Size of block, in bytes, which buffer size must be
 *                           a multiple of. Assumed to be less than 256 and
 *                           a power of 2.
 * @return  Size of padded buffer in bytes.
 */
word32 wc_PkcsPad(byte* buf, word32 sz, word32 blockSz)
{
    /* Calculate number of padding bytes. */
    word32 padSz = blockSz - (sz & (blockSz - 1));

    /* Pad with padSz byte. */
    if (buf != NULL) {
        word32 i;
        for (i = 0; i < padSz; i++) {
            buf[sz+i] = (byte)(padSz & 0xFF);
        }
    }

    /* Return padded buffer size in bytes. */
    return sz + padSz;
}

#ifndef NO_RSA
#ifdef WOLFSSL_ASN_TEMPLATE
/* ASN.1 template for an RSA public key.
 * X.509: RFC 5280, 4.1 - SubjectPublicKeyInfo
 * PKCS #1: RFC 8017, A.1.1 - RSAPublicKey
 */
static const ASNItem rsaPublicKeyASN[] = {
/*  SEQ            */ { 0, ASN_SEQUENCE, 1, 1, 0 },
/*  ALGOID_SEQ     */     { 1, ASN_SEQUENCE, 1, 1, 0 },
/*  ALGOID_OID     */         { 2, ASN_OBJECT_ID, 0, 0, 0 },
/*  ALGOID_NULL    */         { 2, ASN_TAG_NULL, 0, 0, 1 },
#ifdef WC_RSA_PSS
/*  ALGOID_P_SEQ   */         { 2, ASN_SEQUENCE, 1, 0, 1 },
#endif
/*  PUBKEY         */     { 1, ASN_BIT_STRING, 0, 1, 0 },
                                                  /* RSAPublicKey */
/*  PUBKEY_RSA_SEQ */         { 2, ASN_SEQUENCE, 1, 1, 0 },
/*  PUBKEY_RSA_N   */             { 3, ASN_INTEGER, 0, 0, 0 },
/*  PUBKEY_RSA_E   */             { 3, ASN_INTEGER, 0, 0, 0 },
};
enum {
    RSAPUBLICKEYASN_IDX_SEQ = 0,
    RSAPUBLICKEYASN_IDX_ALGOID_SEQ,
    RSAPUBLICKEYASN_IDX_ALGOID_OID,
    RSAPUBLICKEYASN_IDX_ALGOID_NULL,
#ifdef WC_RSA_PSS
    RSAPUBLICKEYASN_IDX_ALGOID_P_SEQ,
#endif
    RSAPUBLICKEYASN_IDX_PUBKEY,
    RSAPUBLICKEYASN_IDX_PUBKEY_RSA_SEQ,
    RSAPUBLICKEYASN_IDX_PUBKEY_RSA_N,
    RSAPUBLICKEYASN_IDX_PUBKEY_RSA_E
};

/* Number of items in ASN.1 template for an RSA public key. */
#define rsaPublicKeyASN_Length (sizeof(rsaPublicKeyASN) / sizeof(ASNItem))
#endif

#if defined(WOLFSSL_RENESAS_TSIP_TLS) || defined(WOLFSSL_RENESAS_FSPSM_TLS)
/* This function is to retrieve key position information in a cert.*
 * The information will be used to call TSIP TLS-linked API for    *
 * certificate verification.                                       */
static int RsaPublicKeyDecodeRawIndex(const byte* input, word32* inOutIdx,
                                      word32 inSz, word32* key_n,
                                      word32* key_n_len, word32* key_e,
                                      word32* key_e_len)
{
#ifndef WOLFSSL_ASN_TEMPLATE
    int ret = 0;
    int length = 0;

#if defined(OPENSSL_EXTRA) || defined(RSA_DECODE_EXTRA)
    byte b;
#endif

    if (input == NULL || inOutIdx == NULL)
        return BAD_FUNC_ARG;

    if (GetSequence(input, inOutIdx, &length, inSz) < 0)
        return ASN_PARSE_E;

#if defined(OPENSSL_EXTRA) || defined(RSA_DECODE_EXTRA)
    if ((*inOutIdx + 1) > inSz)
        return BUFFER_E;

    b = input[*inOutIdx];
    if (b != ASN_INTEGER) {
        /* not from decoded cert, will have algo id, skip past */
        if (GetSequence(input, inOutIdx, &length, inSz) < 0)
            return ASN_PARSE_E;

        if (SkipObjectId(input, inOutIdx, inSz) < 0)
            return ASN_PARSE_E;

        /* Option NULL ASN.1 tag */
        if (*inOutIdx  >= inSz) {
            return BUFFER_E;
        }
        if (input[*inOutIdx] == ASN_TAG_NULL) {
            ret = GetASNNull(input, inOutIdx, inSz);
            if (ret != 0)
                return ret;
        }
        /* TODO: support RSA PSS */

        /* should have bit tag length and seq next */
        ret = CheckBitString(input, inOutIdx, NULL, inSz, 1, NULL);
        if (ret != 0)
            return ret;

        if (GetSequence(input, inOutIdx, &length, inSz) < 0)
            return ASN_PARSE_E;
    }
#endif /* OPENSSL_EXTRA */

    /* Get modulus */
    ret = GetASNInt(input, inOutIdx, &length, inSz);
    *key_n += *inOutIdx;
    if (ret < 0) {
        return ASN_RSA_KEY_E;
    }
    if (key_n_len)
        *key_n_len = length;
    *inOutIdx += length;

    /* Get exponent */
    ret = GetASNInt(input, inOutIdx, &length, inSz);
    *key_e += *inOutIdx;
    if (ret < 0) {
        return ASN_RSA_KEY_E;
    }
    if (key_e_len)
        *key_e_len = length;
    return ret;
#else
    int ret = 0;
    const byte*  n   = NULL;
    const byte*  e   = NULL; /* pointer to modulus/exponent */
    word32 rawIndex = 0;

    ret = wc_RsaPublicKeyDecode_ex(input, inOutIdx, (word32)inSz,
                                                &n, key_n_len, &e, key_e_len);
    if (ret == 0) {
        /* convert pointer to offset */
        if (key_n != NULL) {
            rawIndex = n - input;
            *key_n += rawIndex;
        }
        if (key_e != NULL) {
            rawIndex = e - input;
            *key_e += rawIndex;
        }
    }
    return ret;
#endif

}
#endif /* WOLFSSL_RENESAS_TSIP */
/* Decode RSA public key.
 *
 * X.509: RFC 5280, 4.1 - SubjectPublicKeyInfo
 * PKCS #1: RFC 8017, A.1.1 - RSAPublicKey
 *
 * @param [in]      input     Buffer holding BER encoded data.
 * @param [in, out] inOutIdx  On in, start of RSA public key.
 *                            On out, start of ASN.1 item after RSA public key.
 * @param [in]      inSz      Number of bytes in buffer.
 * @param [out]     n         Pointer to modulus in buffer.
 * @param [out]     nSz       Size of modulus in bytes.
 * @param [out]     e         Pointer to exponent in buffer.
 * @param [out]     eSz       Size of exponent in bytes.
 * @return  0 on success.
 * @return  ASN_PARSE_E when BER encoded data does not match ASN.1 items or
 *          is invalid.
 * @return  BUFFER_E when data in buffer is too small.
 * @return  ASN_OBJECT_ID_E when the expected OBJECT_ID tag is not found.
 * @return  ASN_EXPECT_0_E when the INTEGER has the MSB set or NULL has a
 *          non-zero length.
 * @return  ASN_BITSTR_E when the expected BIT_STRING tag is not found.
 * @return  ASN_UNKNOWN_OID_E when the OID cannot be verified.
 */
int wc_RsaPublicKeyDecode_ex(const byte* input, word32* inOutIdx, word32 inSz,
    const byte** n, word32* nSz, const byte** e, word32* eSz)
{
#ifndef WOLFSSL_ASN_TEMPLATE
    int ret = 0;
    int length = 0;
#if defined(OPENSSL_EXTRA) || defined(RSA_DECODE_EXTRA)
    word32 localIdx;
    byte   tag;
#endif

    if (input == NULL || inOutIdx == NULL)
        return BAD_FUNC_ARG;

    if (GetSequence(input, inOutIdx, &length, inSz) < 0)
        return ASN_PARSE_E;

#if defined(OPENSSL_EXTRA) || defined(RSA_DECODE_EXTRA)
    localIdx = *inOutIdx;
    if (GetASNTag(input, &localIdx, &tag, inSz) < 0)
        return BUFFER_E;

    if (tag != ASN_INTEGER) {
        /* not from decoded cert, will have algo id, skip past */
        if (GetSequence(input, inOutIdx, &length, inSz) < 0)
            return ASN_PARSE_E;

        if (SkipObjectId(input, inOutIdx, inSz) < 0)
            return ASN_PARSE_E;

        /* Option NULL ASN.1 tag */
        if (*inOutIdx  >= inSz) {
            return BUFFER_E;
        }

        localIdx = *inOutIdx;
        if (GetASNTag(input, &localIdx, &tag, inSz) < 0)
            return ASN_PARSE_E;

        if (tag == ASN_TAG_NULL) {
            ret = GetASNNull(input, inOutIdx, inSz);
            if (ret != 0)
                return ret;
        }
    #ifdef WC_RSA_PSS
        /* Skip RSA PSS parameters. */
        else if (tag == (ASN_SEQUENCE | ASN_CONSTRUCTED)) {
            if (GetSequence(input, inOutIdx, &length, inSz) < 0)
                return ASN_PARSE_E;
            *inOutIdx += length;
        }
    #endif

        /* should have bit tag length and seq next */
        ret = CheckBitString(input, inOutIdx, NULL, inSz, 1, NULL);
        if (ret != 0)
            return ret;

        if (GetSequence(input, inOutIdx, &length, inSz) < 0)
            return ASN_PARSE_E;
    }
#endif /* OPENSSL_EXTRA */

    /* Get modulus */
    ret = GetASNInt(input, inOutIdx, &length, inSz);
    if (ret < 0) {
        return ASN_RSA_KEY_E;
    }
    if (nSz)
        *nSz = (word32)length;
    if (n)
        *n = &input[*inOutIdx];
    *inOutIdx += (word32)length;

    /* Get exponent */
    ret = GetASNInt(input, inOutIdx, &length, inSz);
    if (ret < 0) {
        return ASN_RSA_KEY_E;
    }
    if (eSz)
        *eSz = (word32)length;
    if (e)
        *e = &input[*inOutIdx];
    *inOutIdx += (word32)length;

    return ret;
#else
    DECL_ASNGETDATA(dataASN, rsaPublicKeyASN_Length);
    int ret = 0;
#ifdef WC_RSA_PSS
    word32 oid = RSAk;
#endif

    /* Check validity of parameters. */
    if (input == NULL || inOutIdx == NULL) {
        ret = BAD_FUNC_ARG;
    }

    CALLOC_ASNGETDATA(dataASN, rsaPublicKeyASN_Length, ret, NULL);

    if (ret == 0) {
        /* Try decoding PKCS #1 public key by ignoring rest of ASN.1. */
        ret = GetASN_Items(&rsaPublicKeyASN[RSAPUBLICKEYASN_IDX_PUBKEY_RSA_SEQ],
           &dataASN[RSAPUBLICKEYASN_IDX_PUBKEY_RSA_SEQ],
           (int)(rsaPublicKeyASN_Length - RSAPUBLICKEYASN_IDX_PUBKEY_RSA_SEQ),
           0, input, inOutIdx, inSz);
        if (ret != 0) {
            /* Didn't work - try whole SubjectKeyInfo instead. */
        #ifdef WC_RSA_PSS
            /* Could be RSA or RSA PSS key. */
            GetASN_OID(&dataASN[RSAPUBLICKEYASN_IDX_ALGOID_OID], oidKeyType);
        #else
            /* Set the OID to expect. */
            GetASN_ExpBuffer(&dataASN[RSAPUBLICKEYASN_IDX_ALGOID_OID],
                    keyRsaOid, sizeof(keyRsaOid));
        #endif
            /* Decode SubjectKeyInfo. */
            ret = GetASN_Items(rsaPublicKeyASN, dataASN,
                               rsaPublicKeyASN_Length, 1, input, inOutIdx,
                               inSz);
        }
    }
#ifdef WC_RSA_PSS
    if ((ret == 0) && (dataASN[RSAPUBLICKEYASN_IDX_ALGOID_OID].tag != 0)) {
        /* Two possible OIDs supported - RSA and RSA PSS. */
        oid = dataASN[RSAPUBLICKEYASN_IDX_ALGOID_OID].data.oid.sum;
        if ((oid != RSAk) && (oid != RSAPSSk)) {
            ret = ASN_PARSE_E;
        }
    }
    if ((ret == 0) && (dataASN[RSAPUBLICKEYASN_IDX_ALGOID_P_SEQ].tag != 0)) {
        /* Can't have NULL and SEQ. */
        if (dataASN[RSAPUBLICKEYASN_IDX_ALGOID_NULL].tag != 0) {
            ret = ASN_PARSE_E;
        }
        /* SEQ present only with RSA PSS. */
        if ((ret == 0) && (oid != RSAPSSk)) {
            ret = ASN_PARSE_E;
        }
        if (ret == 0) {
            enum wc_HashType hash;
            int mgf;
            int saltLen;
            const byte* params = GetASNItem_Addr(
                dataASN[RSAPUBLICKEYASN_IDX_ALGOID_P_SEQ], input);
            word32 paramsSz = GetASNItem_Length(
                dataASN[RSAPUBLICKEYASN_IDX_ALGOID_P_SEQ], input);

            /* Validate the private key parameters. */
            ret = DecodeRsaPssParams(params, paramsSz, &hash, &mgf, &saltLen);
            /* TODO: store parameters so that usage can be checked. */
        }
    }
#endif
    if (ret == 0) {
        /* Return the buffers and lengths asked for. */
        if (n != NULL) {
            *n   = dataASN[RSAPUBLICKEYASN_IDX_PUBKEY_RSA_N].data.ref.data;
        }
        if (nSz != NULL) {
            *nSz = dataASN[RSAPUBLICKEYASN_IDX_PUBKEY_RSA_N].data.ref.length;
        }
        if (e != NULL) {
            *e   = dataASN[RSAPUBLICKEYASN_IDX_PUBKEY_RSA_E].data.ref.data;
        }
        if (eSz != NULL) {
            *eSz = dataASN[RSAPUBLICKEYASN_IDX_PUBKEY_RSA_E].data.ref.length;
        }
    }

    FREE_ASNGETDATA(dataASN, NULL);
    return ret;
#endif /* WOLFSSL_ASN_TEMPLATE */
}

/* Decode RSA public key.
 *
 * X.509: RFC 5280, 4.1 - SubjectPublicKeyInfo
 * PKCS #1: RFC 8017, A.1.1 - RSAPublicKey
 *
 * @param [in]      input     Buffer holding BER encoded data.
 * @param [in, out] inOutIdx  On in, start of RSA public key.
 *                            On out, start of ASN.1 item after RSA public key.
 * @param [in, out] key       RSA key object.
 * @param [in]      inSz      Number of bytes in buffer.
 * @return  0 on success.
 * @return  ASN_PARSE_E when BER encoded data does not match ASN.1 items or
 *          is invalid.
 * @return  BUFFER_E when data in buffer is too small.
 * @return  ASN_OBJECT_ID_E when the expected OBJECT_ID tag is not found.
 * @return  ASN_EXPECT_0_E when the INTEGER has the MSB set or NULL has a
 *          non-zero length.
 * @return  ASN_BITSTR_E when the expected BIT_STRING tag is not found.
 * @return  ASN_UNKNOWN_OID_E when the OID cannot be verified.
 */
int wc_RsaPublicKeyDecode(const byte* input, word32* inOutIdx, RsaKey* key,
                       word32 inSz)
{
    int ret;
    const byte *n = NULL, *e = NULL;
    word32 nSz = 0, eSz = 0;

    if (key == NULL)
        return BAD_FUNC_ARG;

    ret = wc_RsaPublicKeyDecode_ex(input, inOutIdx, inSz, &n, &nSz, &e, &eSz);
    if (ret == 0) {
        ret = wc_RsaPublicKeyDecodeRaw(n, nSz, e, eSz, key);
    }

    return ret;
}
#endif /* !NO_RSA */

#ifndef NO_DH
#if defined(WOLFSSL_DH_EXTRA)
/*
 * Decodes DH public key to fill specified DhKey.
 *
 * return 0 on success, negative on failure
 */
int wc_DhPublicKeyDecode(const byte* input, word32* inOutIdx,
                DhKey* key, word32 inSz)
{
    int ret = 0;
    int length;
    word32 oid = 0;

    if (input == NULL || inOutIdx == NULL || key == NULL || inSz == 0)
        return BAD_FUNC_ARG;

    if (GetSequence(input, inOutIdx, &length, inSz) < 0)
        return ASN_PARSE_E;

    if (GetSequence(input, inOutIdx, &length, inSz) < 0)
        return ASN_PARSE_E;

    ret = GetObjectId(input, inOutIdx, &oid, oidKeyType, inSz);
    if (oid != DHk || ret < 0)
        return ASN_DH_KEY_E;

    if (GetSequence(input, inOutIdx, &length, inSz) < 0)
        return ASN_PARSE_E;

    if (GetInt(&key->p, input, inOutIdx, inSz) < 0)
        return ASN_DH_KEY_E;

    if (GetInt(&key->g, input, inOutIdx, inSz) < 0) {
        mp_clear(&key->p);
        return ASN_DH_KEY_E;
    }
    ret = (CheckBitString(input, inOutIdx, &length, inSz, 0, NULL) == 0);
    if (ret > 0) {
        /* Found Bit String WOLFSSL_DH_EXTRA is required to access DhKey.pub */
        if (GetInt(&key->pub, input, inOutIdx, inSz) < 0) {
            mp_clear(&key->p);
            mp_clear(&key->g);
            return ASN_DH_KEY_E;
        }
    }
    else {
        mp_clear(&key->p);
        mp_clear(&key->g);
        return ASN_DH_KEY_E;
    }
    return 0;
}
#endif /* WOLFSSL_DH_EXTRA */

#ifdef WOLFSSL_ASN_TEMPLATE
/* ASN.1 template for DH key.
 * PKCS #3, 9 - DHParameter.
 * (Also in: RFC 2786, 3)
 */
static const ASNItem dhParamASN[] = {
/* SEQ     */    { 0, ASN_SEQUENCE, 1, 1, 0 },
                /* prime */
/* PRIME   */        { 1, ASN_INTEGER, 0, 0, 0 },
                /* base */
/* BASE    */        { 1, ASN_INTEGER, 0, 0, 0 },
                /* privateValueLength */
/* PRIVLEN */        { 1, ASN_INTEGER, 0, 0, 1 },
};
enum {
    DHPARAMASN_IDX_SEQ = 0,
    DHPARAMASN_IDX_PRIME,
    DHPARAMASN_IDX_BASE,
    DHPARAMASN_IDX_PRIVLEN
};

/* Number of items in ASN.1 template for DH key. */
#define dhParamASN_Length (sizeof(dhParamASN) / sizeof(ASNItem))

#ifdef WOLFSSL_DH_EXTRA
/* ASN.1 template for DH key wrapped in PKCS #8 or SubjectPublicKeyInfo.
 * PKCS #8: RFC 5208, 5 - PrivateKeyInfo
 * X.509: RFC 5280, 4.1 - SubjectPublicKeyInfo
 * RFC 3279, 2.3.3 - DH in SubjectPublicKeyInfo
 */
static const ASNItem dhKeyPkcs8ASN[] = {
/* SEQ                  */ { 0, ASN_SEQUENCE, 1, 1, 0 },
/* VER                  */     { 1, ASN_INTEGER, 0, 0, 1 },
/* PKEYALGO_SEQ         */     { 1, ASN_SEQUENCE, 1, 1, 0 },
/* PKEYALGO_OID         */         { 2, ASN_OBJECT_ID, 0, 0, 0 },
                                                     /* DHParameter */
/* PKEYALGO_PARAM_SEQ   */         { 2, ASN_SEQUENCE, 1, 1, 0 },
                                                         /* p */
/* PKEYALGO_PARAM_P     */             { 3, ASN_INTEGER, 0, 0, 0 },
                                                         /* g */
/* PKEYALGO_PARAM_G     */             { 3, ASN_INTEGER, 0, 0, 0 },
                                                         /* q - factor of p-1 */
/* PKEYALGO_PARAM_Q     */             { 3, ASN_INTEGER, 0, 0, 1 },
                                                         /* j - subgroup factor */
/* PKEYALGO_PARAM_J     */             { 3, ASN_INTEGER, 0, 0, 1 },
                                                         /* ValidationParms */
/* PKEYALGO_PARAM_VALID */             { 3, ASN_SEQUENCE, 0, 0, 1 },
                                                 /* PrivateKey - PKCS #8 */
/* PKEY_STR             */     { 1, ASN_OCTET_STRING, 0, 1, 2 },
/* PKEY_INT             */         { 2, ASN_INTEGER, 0, 0, 0 },
                                                 /* PublicKey - SubjectPublicKeyInfo. */
/* PUBKEY_STR           */     { 1, ASN_BIT_STRING, 0, 1, 2 },
/* PUBKEY_INT           */         { 2, ASN_INTEGER, 0, 0, 0 },
};
enum {
    DHKEYPKCS8ASN_IDX_SEQ = 0,
    DHKEYPKCS8ASN_IDX_VER,
    DHKEYPKCS8ASN_IDX_PKEYALGO_SEQ,
    DHKEYPKCS8ASN_IDX_PKEYALGO_OID,
    DHKEYPKCS8ASN_IDX_PKEYALGO_PARAM_SEQ,
    DHKEYPKCS8ASN_IDX_PKEYALGO_PARAM_P,
    DHKEYPKCS8ASN_IDX_PKEYALGO_PARAM_G,
    DHKEYPKCS8ASN_IDX_PKEYALGO_PARAM_Q,
    DHKEYPKCS8ASN_IDX_PKEYALGO_PARAM_J,
    DHKEYPKCS8ASN_IDX_PKEYALGO_PARAM_VALID,
    DHKEYPKCS8ASN_IDX_PKEY_STR,
    DHKEYPKCS8ASN_IDX_PKEY_INT,
    DHKEYPKCS8ASN_IDX_PUBKEY_STR,
    DHKEYPKCS8ASN_IDX_PUBKEY_INT
};

#define dhKeyPkcs8ASN_Length (sizeof(dhKeyPkcs8ASN) / sizeof(ASNItem))
#endif
#endif

/* Decodes either PKCS#3 DH parameters or PKCS#8 DH key file (WOLFSSL_DH_EXTRA).
 *
 * See also wc_DhParamsLoad(). Loads directly into buffers rather than key
 * object.
 *
 * @param [in]      input     BER/DER encoded data.
 * @param [in, out] inOutIdx  On in, start of DH key data.
 *                            On out, end of DH key data.
 * @param [in, out] key       DH key object.
 * @param [in]      inSz      Size of data in bytes.
 * @return  0 on success.
 * @return  BAD_FUNC_ARG when input, inOutIDx or key is NULL.
 * @return  MEMORY_E when dynamic memory allocation fails.
 * @return  ASN_PARSE_E when BER encoded data does not match ASN.1 items or
 *          is invalid.
 * @return  BUFFER_E when data in buffer is too small.
 * @return  ASN_OBJECT_ID_E when the expected OBJECT_ID tag is not found.
 * @return  ASN_BITSTR_E when the expected BIT_STRING tag is not found.
 * @return  ASN_EXPECT_0_E when the INTEGER has the MSB set or NULL has a
 *          non-zero length.
 * @return  MP_INIT_E when the unable to initialize an mp_int.
 * @return  ASN_GETINT_E when the unable to convert data to an mp_int.
 * @return  ASN_UNKNOWN_OID_E when the OID cannot be verified.
 */
int wc_DhKeyDecode(const byte* input, word32* inOutIdx, DhKey* key, word32 inSz)
{
#ifndef WOLFSSL_ASN_TEMPLATE
    int ret = 0;
    int length;
#ifdef WOLFSSL_DH_EXTRA
    #if !defined(HAVE_FIPS) || \
        (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION > 2))
    word32 oid = 0, temp = 0;
    #endif
#endif

    WOLFSSL_ENTER("wc_DhKeyDecode");

    if (inOutIdx == NULL)
        return BAD_FUNC_ARG;

    if (GetSequence(input, inOutIdx, &length, inSz) < 0)
        return ASN_PARSE_E;

#ifdef WOLFSSL_DH_EXTRA
    #if !defined(HAVE_FIPS) || \
        (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION > 2))
    temp = *inOutIdx;
    #endif
#endif
    /* Assume input started after 1.2.840.113549.1.3.1 dhKeyAgreement */
    if (GetInt(&key->p, input, inOutIdx, inSz) < 0) {
        ret = ASN_DH_KEY_E;
    }
    if (ret == 0 && GetInt(&key->g, input, inOutIdx, inSz) < 0) {
        mp_clear(&key->p);
        ret = ASN_DH_KEY_E;
    }

#ifdef WOLFSSL_DH_EXTRA
    #if !defined(HAVE_FIPS) || \
        (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION > 2))
    /* If ASN_DH_KEY_E: Check if input started at beginning of key */
    if (ret == WC_NO_ERR_TRACE(ASN_DH_KEY_E)) {
        *inOutIdx = temp;

        /* the version (0) - private only (for public skip) */
        if (GetASNInt(input, inOutIdx, &length, inSz) == 0) {
            *inOutIdx += (word32)length;
        }

        /* Size of dhKeyAgreement section */
        if (GetSequence(input, inOutIdx, &length, inSz) < 0)
            return ASN_PARSE_E;

        /* Check for dhKeyAgreement */
        ret = GetObjectId(input, inOutIdx, &oid, oidKeyType, inSz);
        if (oid != DHk || ret < 0)
            return ASN_DH_KEY_E;

        if (GetSequence(input, inOutIdx, &length, inSz) < 0)
            return ASN_PARSE_E;

        if (GetInt(&key->p, input, inOutIdx, inSz) < 0) {
            return ASN_DH_KEY_E;
        }
        if (ret == 0 && GetInt(&key->g, input, inOutIdx, inSz) < 0) {
            mp_clear(&key->p);
            return ASN_DH_KEY_E;
        }
    }

    temp = *inOutIdx;
    ret = (CheckBitString(input, inOutIdx, &length, inSz, 0, NULL) == 0);
    if (ret > 0) {
        /* Found Bit String */
        if (GetInt(&key->pub, input, inOutIdx, inSz) == 0) {
            WOLFSSL_MSG("Found Public Key");
            ret = 0;
        }
    } else {
        *inOutIdx = temp;
        ret = (GetOctetString(input, inOutIdx, &length, inSz) >= 0);
        if (ret > 0) {
            /* Found Octet String */
            if (GetInt(&key->priv, input, inOutIdx, inSz) == 0) {
                WOLFSSL_MSG("Found Private Key");

                /* Compute public */
                ret = mp_exptmod(&key->g, &key->priv, &key->p, &key->pub);
            }
        } else {
            /* Don't use length from failed CheckBitString/GetOctetString */
            *inOutIdx = temp;
            ret = 0;
        }
    }
    #endif /* !HAVE_FIPS || HAVE_FIPS_VERSION > 2 */
#endif /* WOLFSSL_DH_EXTRA */

    WOLFSSL_LEAVE("wc_DhKeyDecode", ret);

    return ret;
#else
#ifdef WOLFSSL_DH_EXTRA
    DECL_ASNGETDATA(dataASN, dhKeyPkcs8ASN_Length);
#else
    DECL_ASNGETDATA(dataASN, dhParamASN_Length);
#endif
    int ret = 0;

    /* Check input parameters are valid. */
    if ((input == NULL) || (inOutIdx == NULL) || (key == NULL)) {
        ret = BAD_FUNC_ARG;
    }

#ifdef WOLFSSL_DH_EXTRA
    ALLOC_ASNGETDATA(dataASN, dhKeyPkcs8ASN_Length, ret, key->heap);
#else
    ALLOC_ASNGETDATA(dataASN, dhParamASN_Length, ret, key->heap);
#endif

    if (ret == 0) {
        /* Initialize data and set mp_ints to hold p and g. */
        XMEMSET(dataASN, 0, sizeof(*dataASN) * dhParamASN_Length);
        GetASN_MP(&dataASN[DHPARAMASN_IDX_PRIME], &key->p);
        GetASN_MP(&dataASN[DHPARAMASN_IDX_BASE], &key->g);
        /* Try simple PKCS #3 template. */
        ret = GetASN_Items(dhParamASN, dataASN, dhParamASN_Length, 1, input,
                           inOutIdx, inSz);
#ifdef WOLFSSL_DH_EXTRA
        if (ret != 0) {
            mp_free(&key->p);
            mp_free(&key->g);

            /* Initialize data and set mp_ints to hold p, g, q, priv and pub. */
            XMEMSET(dataASN, 0, sizeof(*dataASN) * dhKeyPkcs8ASN_Length);
            GetASN_ExpBuffer(&dataASN[DHKEYPKCS8ASN_IDX_PKEYALGO_OID],
                    keyDhOid, sizeof(keyDhOid));
            GetASN_MP(&dataASN[DHKEYPKCS8ASN_IDX_PKEYALGO_PARAM_P], &key->p);
            GetASN_MP(&dataASN[DHKEYPKCS8ASN_IDX_PKEYALGO_PARAM_G], &key->g);
            GetASN_MP(&dataASN[DHKEYPKCS8ASN_IDX_PKEYALGO_PARAM_Q], &key->q);
            GetASN_MP(&dataASN[DHKEYPKCS8ASN_IDX_PKEY_INT], &key->priv);
            GetASN_MP(&dataASN[DHKEYPKCS8ASN_IDX_PUBKEY_INT], &key->pub);
            /* Try PKCS #8 wrapped template. */
            ret = GetASN_Items(dhKeyPkcs8ASN, dataASN, dhKeyPkcs8ASN_Length, 1,
                               input, inOutIdx, inSz);
            if (ret == 0) {
                /* VERSION only present in PKCS #8 private key structure */
                if ((dataASN[DHKEYPKCS8ASN_IDX_PKEY_INT].length != 0) &&
                        (dataASN[DHKEYPKCS8ASN_IDX_VER].length == 0)) {
                    ret = ASN_PARSE_E;
                }
                else if ((dataASN[DHKEYPKCS8ASN_IDX_PUBKEY_INT].length != 0) &&
                        (dataASN[DHKEYPKCS8ASN_IDX_VER].length != 0)) {
                    ret = ASN_PARSE_E;
                }
            }
            if ((ret == 0) && mp_iszero(&key->pub)) {
                ret = mp_exptmod(&key->g, &key->priv, &key->p, &key->pub);
            }
        }
#endif
    }

    FREE_ASNGETDATA(dataASN, key->heap);
    return ret;
#endif /* WOLFSSL_ASN_TEMPLATE */
}

#ifdef WOLFSSL_DH_EXTRA

/* Export DH Key (private or public) */
int wc_DhKeyToDer(DhKey* key, byte* output, word32* outSz, int exportPriv)
{
#ifndef WOLFSSL_ASN_TEMPLATE
    int ret, privSz = 0, pubSz = 0;
    word32 keySz, idx, len, total;

    if (key == NULL || outSz == NULL) {
        return BAD_FUNC_ARG;
    }

    /* determine size */
    if (exportPriv) {
        /* octet string: priv */
        privSz = SetASNIntMP(&key->priv, -1, NULL);
        if (privSz < 0)
            return privSz;
        idx = 1 + SetLength((word32)privSz, NULL) +
            (word32)privSz; /* +1 for ASN_OCTET_STRING */
    }
    else {
        /* bit string: public */
        pubSz = SetASNIntMP(&key->pub, -1, NULL);
        if (pubSz < 0)
            return pubSz;
        idx = SetBitString((word32)pubSz, 0, NULL) + (word32)pubSz;
    }
    keySz = idx;

    /* DH Parameters sequence with P and G */
    total = 0;
    ret = wc_DhParamsToDer(key, NULL, &total);
    if (ret != WC_NO_ERR_TRACE(LENGTH_ONLY_E))
        return ret;
    idx += total;

    /* object dhKeyAgreement 1.2.840.113549.1.3.1 */
    idx += (word32)SetObjectId(sizeof(keyDhOid), NULL);
    idx += (word32)sizeof(keyDhOid);
    len = idx - keySz;
    /* sequence - all but pub/priv */
    idx += SetSequence(len, NULL);
    if (exportPriv) {
        /* version: 0 (ASN_INTEGER, 0x01, 0x00) */
        idx += 3;
    }
    /* sequence */
    total = idx + SetSequence(idx, NULL);

    /* if no output, then just getting size */
    if (output == NULL) {
        *outSz = total;
        return WC_NO_ERR_TRACE(LENGTH_ONLY_E);
    }

    /* make sure output fits in buffer */
    if (total > *outSz) {
        return BUFFER_E;
    }
    total = idx;

    /* sequence */
    idx = SetSequence(total, output);
    if (exportPriv) {
        /* version: 0 */
        idx += (word32)SetMyVersion(0, output + idx, 0);
    }
    /* sequence - all but pub/priv */
    idx += SetSequence(len, output + idx);
    /* object dhKeyAgreement 1.2.840.113549.1.3.1 */
    idx += (word32)SetObjectId(sizeof(keyDhOid), output + idx);
    XMEMCPY(output + idx, keyDhOid, sizeof(keyDhOid));
    idx += sizeof(keyDhOid);

    /* DH Parameters sequence with P and G */
    total = *outSz - idx;
    ret = wc_DhParamsToDer(key, output + idx, &total);
    if (ret < 0)
        return ret;
    idx += total;

    /* octet string: priv */
    if (exportPriv) {
        idx += (word32)SetOctetString((word32)privSz, output + idx);
        idx += (word32)SetASNIntMP(&key->priv, -1, output + idx);
    }
    else {
        /* bit string: public */
        idx += (word32)SetBitString((word32)pubSz, 0, output + idx);
        idx += (word32)SetASNIntMP(&key->pub, -1, output + idx);
    }
    *outSz = idx;

    return (int)idx;
#else
    ASNSetData dataASN[dhKeyPkcs8ASN_Length];
    int ret = 0;
    int sz;

    WOLFSSL_ENTER("wc_DhKeyToDer");

    XMEMSET(dataASN, 0, sizeof(dataASN));
    SetASN_Int8Bit(&dataASN[DHKEYPKCS8ASN_IDX_VER], 0);
    SetASN_OID(&dataASN[DHKEYPKCS8ASN_IDX_PKEYALGO_OID], DHk, oidKeyType);
    /* Set mp_int containing p and g. */
    SetASN_MP(&dataASN[DHKEYPKCS8ASN_IDX_PKEYALGO_PARAM_P], &key->p);
    SetASN_MP(&dataASN[DHKEYPKCS8ASN_IDX_PKEYALGO_PARAM_G], &key->g);
    dataASN[DHKEYPKCS8ASN_IDX_PKEYALGO_PARAM_Q].noOut = 1;
    dataASN[DHKEYPKCS8ASN_IDX_PKEYALGO_PARAM_J].noOut = 1;
    dataASN[DHKEYPKCS8ASN_IDX_PKEYALGO_PARAM_VALID].noOut = 1;

    if (exportPriv) {
        SetASN_MP(&dataASN[DHKEYPKCS8ASN_IDX_PKEY_INT], &key->priv);
        dataASN[DHKEYPKCS8ASN_IDX_PUBKEY_STR].noOut = 1;
        dataASN[DHKEYPKCS8ASN_IDX_PUBKEY_INT].noOut = 1;
    }
    else {
        dataASN[DHKEYPKCS8ASN_IDX_VER].noOut = 1;
        dataASN[DHKEYPKCS8ASN_IDX_PKEY_STR].noOut = 1;
        dataASN[DHKEYPKCS8ASN_IDX_PKEY_INT].noOut = 1;
        SetASN_MP(&dataASN[DHKEYPKCS8ASN_IDX_PUBKEY_INT], &key->pub);
    }

    /* Calculate the size of the DH parameters. */
    ret = SizeASN_Items(dhKeyPkcs8ASN, dataASN, dhKeyPkcs8ASN_Length, &sz);
    if (output == NULL) {
        *outSz = (word32)sz;
        ret = WC_NO_ERR_TRACE(LENGTH_ONLY_E);
    }
    /* Check buffer is big enough for encoding. */
    if ((ret == 0) && ((int)*outSz < sz)) {
        ret = BUFFER_E;
    }
    if (ret == 0) {
        /* Encode the DH parameters into buffer. */
        SetASN_Items(dhKeyPkcs8ASN, dataASN, dhKeyPkcs8ASN_Length, output);
        /* Set the actual encoding size. */
        *outSz = (word32)sz;
        /* Return the actual encoding size. */
        ret = sz;
    }

    return ret;
#endif
}

int wc_DhPubKeyToDer(DhKey* key, byte* out, word32* outSz)
{
    return wc_DhKeyToDer(key, out, outSz, 0);
}
int wc_DhPrivKeyToDer(DhKey* key, byte* out, word32* outSz)
{
    return wc_DhKeyToDer(key, out, outSz, 1);
}


/* Convert DH key parameters to DER format, write to output (outSz)
 * If output is NULL then max expected size is set to outSz and LENGTH_ONLY_E is
 * returned.
 *
 * Note : static function due to redefinition complications with DhKey and FIPS
 * version 2 build.
 *
 * return bytes written on success */
int wc_DhParamsToDer(DhKey* key, byte* output, word32* outSz)
{
#ifndef WOLFSSL_ASN_TEMPLATE
    int ret;
    word32 idx, total;

    if (key == NULL || outSz == NULL) {
        return BAD_FUNC_ARG;
    }

    /* determine size */
    /* integer - g */
    ret = SetASNIntMP(&key->g, -1, NULL);
    if (ret < 0)
        return ret;
    idx = (word32)ret;
    /* integer - p */
    ret = SetASNIntMP(&key->p, -1, NULL);
    if (ret < 0)
        return ret;
    idx += (word32)ret;
    total = idx;
     /* sequence */
    idx += SetSequence(idx, NULL);

    if (output == NULL) {
        *outSz = idx;
        return WC_NO_ERR_TRACE(LENGTH_ONLY_E);
    }
    /* make sure output fits in buffer */
    if (idx > *outSz) {
        return BUFFER_E;
    }


    /* write DH parameters */
    /* sequence - for P and G only */
    idx = SetSequence(total, output);
    /* integer - p */
    ret = SetASNIntMP(&key->p, -1, output + idx);
    if (ret < 0)
        return ret;
    idx += (word32)ret;
    /* integer - g */
    ret = SetASNIntMP(&key->g, -1, output + idx);
    if (ret < 0)
        return ret;
    idx += (word32)ret;
    *outSz = idx;

    return (int)idx;
#else
    ASNSetData dataASN[dhParamASN_Length];
    int ret = 0;
    int sz = 0;

    WOLFSSL_ENTER("wc_DhParamsToDer");

    if (key == NULL || outSz == NULL) {
        ret = BAD_FUNC_ARG;
    }

    if (ret == 0) {
        XMEMSET(dataASN, 0, sizeof(dataASN));
        /* Set mp_int containing p and g. */
        SetASN_MP(&dataASN[DHPARAMASN_IDX_PRIME], &key->p);
        SetASN_MP(&dataASN[DHPARAMASN_IDX_BASE], &key->g);
        /* privateValueLength not encoded. */
        dataASN[DHPARAMASN_IDX_PRIVLEN].noOut = 1;

        /* Calculate the size of the DH parameters. */
        ret = SizeASN_Items(dhParamASN, dataASN, dhParamASN_Length, &sz);
    }
    if ((ret == 0) && (output == NULL)) {
        *outSz = (word32)sz;
        ret = WC_NO_ERR_TRACE(LENGTH_ONLY_E);
    }
    /* Check buffer is big enough for encoding. */
    if ((ret == 0) && (*outSz < (word32)sz)) {
        ret = BUFFER_E;
    }
    if (ret == 0) {
        /* Encode the DH parameters into buffer. */
        SetASN_Items(dhParamASN, dataASN, dhParamASN_Length, output);
        /* Set the actual encoding size. */
        *outSz = (word32)sz;
        /* Return count of bytes written. */
        ret = sz;
    }

    return ret;
#endif
}

#endif /* WOLFSSL_DH_EXTRA */

/* Decode DH parameters.
 *
 * PKCS #3, 9 - DHParameter.
 * (Also in: RFC 2786, 3)
 *
 * @param [in]      input     Buffer holding BER encoded data.
 * @param [in, out] inOutIdx  On in, start of RSA public key.
 *                            On out, start of ASN.1 item after RSA public key.
 * @param [in]      inSz      Number of bytes in buffer.
 * @param [in, out] p         Buffer to hold prime.
 * @param [out]     pInOutSz  On in, size of buffer to hold prime in bytes.
 *                            On out, size of prime in bytes.
 * @param [in, out] g         Buffer to hold base.
 * @param [out]     gInOutSz  On in, size of buffer to hold base in bytes.
 *                            On out, size of base in bytes.
 * @return  0 on success.
 * @return  ASN_PARSE_E when BER encoded data does not match ASN.1 items or
 *          is invalid.
 * @return  BUFFER_E when data in buffer is too small.
 * @return  ASN_EXPECT_0_E when the INTEGER has the MSB set.
 */
int wc_DhParamsLoad(const byte* input, word32 inSz, byte* p, word32* pInOutSz,
                 byte* g, word32* gInOutSz)
{
#ifndef WOLFSSL_ASN_TEMPLATE
    word32 idx = 0;
    int    ret;
    int    length;

    if (GetSequence(input, &idx, &length, inSz) <= 0)
        return ASN_PARSE_E;

    ret = GetASNInt(input, &idx, &length, inSz);
    if (ret != 0)
        return ret;

    if (length <= (int)*pInOutSz) {
        XMEMCPY(p, &input[idx], (size_t)length);
        *pInOutSz = (word32)length;
    }
    else {
        return BUFFER_E;
    }
    idx += (word32)length;

    ret = GetASNInt(input, &idx, &length, inSz);
    if (ret != 0)
        return ret;

    if (length <= (int)*gInOutSz) {
        XMEMCPY(g, &input[idx], (size_t)length);
        *gInOutSz = (word32)length;
    }
    else {
        return BUFFER_E;
    }

    return 0;
#else
    DECL_ASNGETDATA(dataASN, dhParamASN_Length);
    word32 idx = 0;
    int ret = 0;

    /* Make sure pointers are valid before use. */
    if ((input == NULL) || (p == NULL) || (pInOutSz == NULL) || (g == NULL) ||
            (gInOutSz == NULL)) {
        ret = BAD_FUNC_ARG;
    }

    CALLOC_ASNGETDATA(dataASN, dhParamASN_Length, ret, NULL);

    if (ret == 0) {
        /* Set the buffers to copy p and g into. */
        GetASN_Buffer(&dataASN[DHPARAMASN_IDX_PRIME], p, pInOutSz);
        GetASN_Buffer(&dataASN[DHPARAMASN_IDX_BASE], g, gInOutSz);
        /* Decode the DH Parameters. */
        ret = GetASN_Items(dhParamASN, dataASN, dhParamASN_Length, 1, input,
                           &idx, inSz);
    }

    FREE_ASNGETDATA(dataASN, NULL);
    return ret;
#endif /* WOLFSSL_ASN_TEMPLATE */
}
#endif /* !NO_DH */


#ifndef NO_DSA

static mp_int* GetDsaInt(DsaKey* key, int idx)
{
    if (idx == 0)
        return &key->p;
    if (idx == 1)
        return &key->q;
    if (idx == 2)
        return &key->g;
    if (idx == 3)
        return &key->y;
    if (idx == 4)
        return &key->x;

    return NULL;
}

#ifdef WOLFSSL_ASN_TEMPLATE
/* ASN.1 template for DSA public and private keys.
 * Public key: seq, p, q, g, y
 * Private key: seq, version, p, q, g, y, x
 * RFC 3279, 2.3.2 - DSA in SubjectPublicKeyInfo
 */
static const ASNItem dsaKeyASN[] = {
/* SEQ */    { 0, ASN_SEQUENCE, 1, 1, 0 },
/* VER */        { 1, ASN_INTEGER, 0, 0, 0 },
/* P   */        { 1, ASN_INTEGER, 0, 0, 0 },
/* Q   */        { 1, ASN_INTEGER, 0, 0, 0 },
/* G   */        { 1, ASN_INTEGER, 0, 0, 0 },
/* Y   */        { 1, ASN_INTEGER, 0, 0, 0 },
/* X   */        { 1, ASN_INTEGER, 0, 0, 0 },
};
enum {
    DSAKEYASN_IDX_SEQ = 0,
    DSAKEYASN_IDX_VER,
    DSAKEYASN_IDX_P,
    DSAKEYASN_IDX_Q,
    DSAKEYASN_IDX_G,
    DSAKEYASN_IDX_Y,
    DSAKEYASN_IDX_X
};

/* Number of items in ASN.1 template for DSA private key. */
#define dsaKeyASN_Length (sizeof(dsaKeyASN) / sizeof(ASNItem))
/* Number of items in ASN.1 template for DSA public key. */
#define dsaPublicKeyASN_Length ((sizeof(dsaKeyASN) / sizeof(ASNItem)) - 2)

/* ASN.1 template for PublicKeyInfo with DSA.
 * X.509: RFC 5280, 4.1 - SubjectPublicKeyInfo
 * RFC 3279, 2.3.2 - DSA in SubjectPublicKeyInfo
 */
static const ASNItem dsaPubKeyASN[] = {
/* SEQ             */ { 0, ASN_SEQUENCE, 1, 1, 0 },
/* ALGOID_SEQ      */     { 1, ASN_SEQUENCE, 1, 1, 0 },
/* ALGOID_OID      */         { 2, ASN_OBJECT_ID, 0, 0, 0 },
/* ALGOID_PARAMS   */         { 2, ASN_SEQUENCE, 1, 1, 0 },
                                                   /* p */
/* ALGOID_PARAMS_P */             { 3, ASN_INTEGER, 0, 0, 0 },
                                                   /* q */
/* ALGOID_PARAMS_Q */             { 3, ASN_INTEGER, 0, 0, 0 },
                                                   /* g */
/* ALGOID_PARAMS_G */             { 3, ASN_INTEGER, 0, 0, 0 },
/* PUBKEY_STR      */     { 1, ASN_BIT_STRING, 0, 1, 1 },
                                               /* y */
/* PUBKEY_Y        */         { 2, ASN_INTEGER, 0, 0, 0 },
};
enum {
    DSAPUBKEYASN_IDX_SEQ = 0,
    DSAPUBKEYASN_IDX_ALGOID_SEQ,
    DSAPUBKEYASN_IDX_ALGOID_OID,
    DSAPUBKEYASN_IDX_ALGOID_PARAMS,
    DSAPUBKEYASN_IDX_ALGOID_PARAMS_P,
    DSAPUBKEYASN_IDX_ALGOID_PARAMS_Q,
    DSAPUBKEYASN_IDX_ALGOID_PARAMS_G,
    DSAPUBKEYASN_IDX_PUBKEY_STR,
    DSAPUBKEYASN_IDX_PUBKEY_Y
};

/* Number of items in ASN.1 template for PublicKeyInfo with DSA. */
#define dsaPubKeyASN_Length (sizeof(dsaPubKeyASN) / sizeof(ASNItem))
#endif /* WOLFSSL_ASN_TEMPLATE */

/* Decode DSA public key.
 *
 * X.509: RFC 5280, 4.1 - SubjectPublicKeyInfo
 * RFC 3279, 2.3.2 - DSA in SubjectPublicKeyInfo
 *
 * @param [in]      input     Buffer holding BER encoded data.
 * @param [in, out] inOutIdx  On in, start of DSA public key.
 *                            On out, start of ASN.1 item after DSA public key.
 * @param [in, out] key       DSA key object.
 * @param [in]      inSz      Number of bytes in buffer.
 * @return  0 on success.
 * @return  ASN_PARSE_E when BER encoded data does not match ASN.1 items or
 *          is invalid.
 * @return  BUFFER_E when data in buffer is too small.
 * @return  ASN_OBJECT_ID_E when the expected OBJECT_ID tag is not found.
 * @return  ASN_EXPECT_0_E when the INTEGER has the MSB set or NULL has a
 *          non-zero length.
 * @return  ASN_BITSTR_E when the expected BIT_STRING tag is not found.
 * @return  ASN_UNKNOWN_OID_E when the OID cannot be verified.
 */
int wc_DsaPublicKeyDecode(const byte* input, word32* inOutIdx, DsaKey* key,
                          word32 inSz)
{
#ifndef WOLFSSL_ASN_TEMPLATE
    int    length;
    int    ret = 0;
    word32 oid;
    word32 maxIdx;

    if (input == NULL || inOutIdx == NULL || key == NULL)
        return BAD_FUNC_ARG;

    if (GetSequence(input, inOutIdx, &length, inSz) < 0)
        return ASN_PARSE_E;

    maxIdx = (word32)(*inOutIdx + (word32)length);
    if (GetInt(&key->p,  input, inOutIdx, maxIdx) < 0 ||
        GetInt(&key->q,  input, inOutIdx, maxIdx) < 0 ||
        GetInt(&key->g,  input, inOutIdx, maxIdx) < 0 ||
        GetInt(&key->y,  input, inOutIdx, maxIdx) < 0 )
        ret = ASN_DH_KEY_E;

    if (ret != 0) {
        if (GetSequence(input, inOutIdx, &length, inSz) < 0)
            return ASN_PARSE_E;

        ret = GetObjectId(input, inOutIdx, &oid, oidIgnoreType, inSz);
        if (ret != 0)
            return ret;

        if (GetSequence(input, inOutIdx, &length, inSz) < 0)
            return ASN_PARSE_E;

        if (GetInt(&key->p,  input, inOutIdx, inSz) < 0 ||
            GetInt(&key->q,  input, inOutIdx, inSz) < 0 ||
            GetInt(&key->g,  input, inOutIdx, inSz) < 0)
            return ASN_DH_KEY_E;

        if (CheckBitString(input, inOutIdx, &length, inSz, 0, NULL) < 0)
            return ASN_PARSE_E;

        if (GetInt(&key->y,  input, inOutIdx, inSz) < 0 )
            return ASN_DH_KEY_E;

        ret = 0;
    }

    key->type = DSA_PUBLIC;
    return ret;
#else
    /* dsaPubKeyASN is longer than dsaPublicKeyASN. */
    DECL_ASNGETDATA(dataASN, dsaPubKeyASN_Length);
    int ret = 0;

    /* Validated parameters. */
    if ((input == NULL) || (inOutIdx == NULL) || (key == NULL)) {
        ret = BAD_FUNC_ARG;
    }

    ALLOC_ASNGETDATA(dataASN, dsaPubKeyASN_Length, ret, key->heap);

    if (ret == 0) {
        int i;

        /* Clear dynamic data items. */
        XMEMSET(dataASN, 0, sizeof(ASNGetData) * dsaPublicKeyASN_Length);
        /* seq
         *   p, q, g, y
         * Start DSA ints from DSAKEYASN_IDX_VER instead of DSAKEYASN_IDX_P */
        for (i = 0; i < DSA_INTS - 1; i++)
            GetASN_MP(&dataASN[(int)DSAKEYASN_IDX_VER + i], GetDsaInt(key, i));
        /* Parse as simple form. */
        ret = GetASN_Items(dsaKeyASN, dataASN, dsaPublicKeyASN_Length, 0, input,
                           inOutIdx, inSz);
        if (ret != 0) {
            /* Clear dynamic data items. */
            XMEMSET(dataASN, 0, sizeof(ASNGetData) * dsaPubKeyASN_Length);
            /* Set DSA OID to expect. */
            GetASN_ExpBuffer(&dataASN[DSAPUBKEYASN_IDX_ALGOID_OID],
                    keyDsaOid, sizeof(keyDsaOid));
            /* p, q, g */
            for (i = 0; i < DSA_INTS - 2; i++)
                GetASN_MP(&dataASN[(int)DSAPUBKEYASN_IDX_ALGOID_PARAMS_P + i],
                        GetDsaInt(key, i));
            /* y */
            GetASN_MP(&dataASN[DSAPUBKEYASN_IDX_PUBKEY_Y], GetDsaInt(key, i));
            /* Parse as SubjectPublicKeyInfo. */
            ret = GetASN_Items(dsaPubKeyASN, dataASN, dsaPubKeyASN_Length, 1,
                input, inOutIdx, inSz);
        }
    }

    if (ret == 0) {
        /* Data parsed - set type of key parsed. */
        key->type = DSA_PUBLIC;
    }

    FREE_ASNGETDATA(dataASN, key->heap);
    return ret;
#endif
}

int wc_DsaParamsDecode(const byte* input, word32* inOutIdx, DsaKey* key,
                        word32 inSz)
{
    int    length;
    word32 maxIdx;

    if (input == NULL || inOutIdx == NULL || key == NULL)
        return BAD_FUNC_ARG;

    if (GetSequence(input, inOutIdx, &length, inSz) < 0)
        return ASN_PARSE_E;

    maxIdx = (word32)(*inOutIdx + (word32)length);
    if (GetInt(&key->p, input, inOutIdx, maxIdx) < 0 ||
        GetInt(&key->q, input, inOutIdx, maxIdx) < 0 ||
        GetInt(&key->g, input, inOutIdx, maxIdx) < 0)
        return ASN_DH_KEY_E;

    return 0;
}


#ifdef WOLFSSL_ASN_TEMPLATE
/* ASN.1 template for a DSA key holding private key in an OCTET_STRING. */
static const ASNItem dsaKeyOctASN[] = {
/*  SEQ      */ { 0, ASN_SEQUENCE, 1, 1, 0 },
                /* p */
/*  P        */     { 1, ASN_INTEGER, 0, 0, 0 },
                /* q */
/*  Q        */     { 1, ASN_INTEGER, 0, 0, 0 },
                /* g */
/*  G        */     { 1, ASN_INTEGER, 0, 0, 0 },
                /* Private key */
/*  PKEY_STR */     { 1, ASN_OCTET_STRING, 0, 1, 0 },
                    /* x */
/*  X        */         { 2, ASN_INTEGER, 0, 0, 0 },
};
enum {
    DSAKEYOCTASN_IDX_SEQ = 0,
    DSAKEYOCTASN_IDX_P,
    DSAKEYOCTASN_IDX_Q,
    DSAKEYOCTASN_IDX_G,
    DSAKEYOCTASN_IDX_PKEY_STR,
    DSAKEYOCTASN_IDX_X
};

/* Number of items in ASN.1 template for a DSA key (OCTET_STRING version). */
#define dsaKeyOctASN_Length (sizeof(dsaKeyOctASN) / sizeof(ASNItem))
#endif

/* Decode DSA private key.
 *
 * @param [in]      input     Buffer holding BER encoded data.
 * @param [in, out] inOutIdx  On in, start of DSA public key.
 *                            On out, start of ASN.1 item after DSA public key.
 * @param [in, out] key       DSA key object.
 * @param [in]      inSz      Number of bytes in buffer.
 * @return  0 on success.
 * @return  ASN_PARSE_E when BER encoded data does not match ASN.1 items or
 *          is invalid.
 * @return  BUFFER_E when data in buffer is too small.
 * @return  ASN_EXPECT_0_E when the INTEGER has the MSB set or NULL has a
 *          non-zero length.
 */
int wc_DsaPrivateKeyDecode(const byte* input, word32* inOutIdx, DsaKey* key,
                           word32 inSz)
{
#ifndef WOLFSSL_ASN_TEMPLATE
    int length, version, ret = 0, temp = 0;
    word32 algId = 0;

    /* Sanity checks on input */
    if (input == NULL || inOutIdx == NULL || key == NULL) {
        return BAD_FUNC_ARG;
    }

    /* if has pkcs8 header skip it */
    if (ToTraditionalInline_ex(input, inOutIdx, inSz, &algId) < 0) {
        /* ignore error, did not have pkcs8 header */
    }

    if (GetSequence(input, inOutIdx, &length, inSz) < 0)
        return ASN_PARSE_E;

    temp = (int)*inOutIdx;

    /* Default case expects a certificate with OctetString but no version ID */
    ret = GetInt(&key->p, input, inOutIdx, inSz);
    if (ret < 0) {
        mp_clear(&key->p);
        ret = ASN_PARSE_E;
    }
    else {
        ret = GetInt(&key->q, input, inOutIdx, inSz);
        if (ret < 0) {
            mp_clear(&key->p);
            mp_clear(&key->q);
            ret = ASN_PARSE_E;
        }
        else {
            ret = GetInt(&key->g, input, inOutIdx, inSz);
            if (ret < 0) {
                mp_clear(&key->p);
                mp_clear(&key->q);
                mp_clear(&key->g);
                ret = ASN_PARSE_E;
            }
            else {
                ret = GetOctetString(input, inOutIdx, &length, inSz);
                if (ret < 0) {
                    mp_clear(&key->p);
                    mp_clear(&key->q);
                    mp_clear(&key->g);
                    ret = ASN_PARSE_E;
                }
                else {
                    ret = GetInt(&key->y, input, inOutIdx, inSz);
                    if (ret < 0) {
                        mp_clear(&key->p);
                        mp_clear(&key->q);
                        mp_clear(&key->g);
                        mp_clear(&key->y);
                        ret = ASN_PARSE_E;
                    }
                }
            }
        }
    }
    /* An alternate pass if default certificate fails parsing */
    if (ret == WC_NO_ERR_TRACE(ASN_PARSE_E)) {
        *inOutIdx = (word32)temp;
        if (GetMyVersion(input, inOutIdx, &version, inSz) < 0)
            return ASN_PARSE_E;

        if (GetInt(&key->p,  input, inOutIdx, inSz) < 0 ||
            GetInt(&key->q,  input, inOutIdx, inSz) < 0 ||
            GetInt(&key->g,  input, inOutIdx, inSz) < 0 ||
            GetInt(&key->y,  input, inOutIdx, inSz) < 0 ||
            GetInt(&key->x,  input, inOutIdx, inSz) < 0 )
            return ASN_DH_KEY_E;
    }

    key->type = DSA_PRIVATE;
    return 0;
#else
    /* dsaKeyASN is longer than dsaKeyOctASN. */
    DECL_ASNGETDATA(dataASN, dsaKeyASN_Length);
    int ret = 0;
    byte version = 0;

    /* Sanity checks on input */
    if ((input == NULL) || (inOutIdx == NULL) || (key == NULL)) {
        ret = BAD_FUNC_ARG;
    }

    CALLOC_ASNGETDATA(dataASN, dsaKeyASN_Length, ret, key->heap);

    if (ret == 0) {
        int i;

        /* Try dsaKeyOctASN */
        /* Initialize key data and set mp_ints for params */
        for (i = 0; i < DSA_INTS - 2; i++) {
            GetASN_MP(&dataASN[(int)DSAKEYOCTASN_IDX_P + i], GetDsaInt(key, i));
        }
        /* and priv */
        GetASN_MP(&dataASN[DSAKEYOCTASN_IDX_X], GetDsaInt(key, i));
        /* Try simple form. */
        ret = GetASN_Items(dsaKeyOctASN, dataASN, dsaKeyOctASN_Length, 1, input,
                           inOutIdx, inSz);

        if (ret != 0) {
            /* Try dsaKeyASN */
            XMEMSET(dataASN, 0, sizeof(*dataASN) * dsaKeyASN_Length);
            GetASN_Int8Bit(&dataASN[DSAKEYASN_IDX_VER], &version);
            for (i = 0; i < DSA_INTS; i++) {
                mp_int* n = GetDsaInt(key, i);
                mp_clear(n);
                GetASN_MP(&dataASN[(int)DSAKEYASN_IDX_P + i], n);
            }

            /* Try simple OCTET_STRING form. */
            ret = GetASN_Items(dsaKeyASN, dataASN, dsaKeyASN_Length, 1, input,
                               inOutIdx, inSz);
        }
    }

    if (ret == 0) {
        /* Set the contents to be a private key. */
        key->type = DSA_PRIVATE;
    }

    FREE_ASNGETDATA(dataASN, key->heap);
    return ret;
#endif
}

#ifndef WOLFSSL_ASN_TEMPLATE
/* Release Tmp DSA resources */
static WC_INLINE void FreeTmpDsas(byte** tmps, void* heap, int ints)
{
    int i;

    for (i = 0; i < ints; i++)
        XFREE(tmps[i], heap, DYNAMIC_TYPE_DSA);

    (void)heap;
}
#endif /* !WOLFSSL_ASN_TEMPLATE */

#if !defined(HAVE_SELFTEST) && (defined(WOLFSSL_KEY_GEN) || \
        defined(WOLFSSL_CERT_GEN))
/* Encode a DSA public key into buffer.
 *
 * @param [out] output       Buffer to hold encoded data.
 * @param [in]  key          DSA key object.
 * @param [out] outLen       Length of buffer.
 * @param [out] with_header  Whether to encode in SubjectPublicKeyInfo block.
 * @return  Size of encoded data in bytes on success.
 * @return  BAD_FUNC_ARG when output or key is NULL, or buffer size is less
 *          than a minimal size (5 bytes), or buffer size is smaller than
 *          encoding size.
 * @return  MEMORY_E when dynamic memory allocation fails.
 */
int wc_SetDsaPublicKey(byte* output, DsaKey* key, int outLen, int with_header)
{
#ifndef WOLFSSL_ASN_TEMPLATE
    /* p, g, q = DSA params, y = public exponent */
#ifdef WOLFSSL_SMALL_STACK
    byte* p = NULL;
    byte* g = NULL;
    byte* q = NULL;
    byte* y = NULL;
#else
    byte p[MAX_DSA_INT_SZ];
    byte g[MAX_DSA_INT_SZ];
    byte q[MAX_DSA_INT_SZ];
    byte y[MAX_DSA_INT_SZ];
#endif
    byte innerSeq[MAX_SEQ_SZ];
    byte outerSeq[MAX_SEQ_SZ];
    byte bitString[1 + MAX_LENGTH_SZ + 1];
    int pSz, gSz, qSz, ySz;
    word32 idx, innerSeqSz, outerSeqSz, bitStringSz = 0;
    WOLFSSL_ENTER("wc_SetDsaPublicKey");

    if (output == NULL || key == NULL || outLen < MAX_SEQ_SZ) {
        return BAD_FUNC_ARG;
    }

    /* p */
    WC_ALLOC_VAR_EX(p, byte, MAX_DSA_INT_SZ, key->heap,
        DYNAMIC_TYPE_TMP_BUFFER, return MEMORY_E);
    if ((pSz = SetASNIntMP(&key->p, MAX_DSA_INT_SZ, p)) < 0) {
        WOLFSSL_MSG("SetASNIntMP Error with p");
        WC_FREE_VAR_EX(p, key->heap, DYNAMIC_TYPE_TMP_BUFFER);
        return pSz;
    }

    /* q */
    WC_ALLOC_VAR_EX(q, byte, MAX_DSA_INT_SZ, key->heap,
        DYNAMIC_TYPE_TMP_BUFFER, return MEMORY_E);
    if ((qSz = SetASNIntMP(&key->q, MAX_DSA_INT_SZ, q)) < 0) {
        WOLFSSL_MSG("SetASNIntMP Error with q");
        WC_FREE_VAR_EX(p, key->heap, DYNAMIC_TYPE_TMP_BUFFER);
        WC_FREE_VAR_EX(q, key->heap, DYNAMIC_TYPE_TMP_BUFFER);
        return qSz;
    }

    /* g */
    WC_ALLOC_VAR_EX(g, byte, MAX_DSA_INT_SZ, key->heap,
        DYNAMIC_TYPE_TMP_BUFFER, return MEMORY_E);
    if ((gSz = SetASNIntMP(&key->g, MAX_DSA_INT_SZ, g)) < 0) {
        WOLFSSL_MSG("SetASNIntMP Error with g");
        WC_FREE_VAR_EX(p, key->heap, DYNAMIC_TYPE_TMP_BUFFER);
        WC_FREE_VAR_EX(q, key->heap, DYNAMIC_TYPE_TMP_BUFFER);
        WC_FREE_VAR_EX(g, key->heap, DYNAMIC_TYPE_TMP_BUFFER);
        return gSz;
    }

    /* y */
    WC_ALLOC_VAR_EX(y, byte, MAX_DSA_INT_SZ, key->heap,
        DYNAMIC_TYPE_TMP_BUFFER, return MEMORY_E);
    if ((ySz = SetASNIntMP(&key->y, MAX_DSA_INT_SZ, y)) < 0) {
        WOLFSSL_MSG("SetASNIntMP Error with y");
        WC_FREE_VAR_EX(p, key->heap, DYNAMIC_TYPE_TMP_BUFFER);
        WC_FREE_VAR_EX(q, key->heap, DYNAMIC_TYPE_TMP_BUFFER);
        WC_FREE_VAR_EX(g, key->heap, DYNAMIC_TYPE_TMP_BUFFER);
        WC_FREE_VAR_EX(y, key->heap, DYNAMIC_TYPE_TMP_BUFFER);
        return ySz;
    }

    if (with_header) {
        word32 algoSz;
#ifdef WOLFSSL_SMALL_STACK
        byte* algo = NULL;

        algo = (byte*)XMALLOC(MAX_ALGO_SZ, key->heap, DYNAMIC_TYPE_TMP_BUFFER);
        if (algo == NULL) {
            XFREE(p,    key->heap, DYNAMIC_TYPE_TMP_BUFFER);
            XFREE(q,    key->heap, DYNAMIC_TYPE_TMP_BUFFER);
            XFREE(g,    key->heap, DYNAMIC_TYPE_TMP_BUFFER);
            XFREE(y,    key->heap, DYNAMIC_TYPE_TMP_BUFFER);
            return MEMORY_E;
        }
#else
        byte algo[MAX_ALGO_SZ];
#endif
        innerSeqSz  = SetSequence((word32)(pSz + qSz + gSz), innerSeq);
        algoSz = SetAlgoID(DSAk, algo, oidKeyType, 0);
        bitStringSz  = SetBitString((word32)ySz, 0, bitString);
        outerSeqSz = SetSequence(algoSz + innerSeqSz +
                                 (word32)(pSz + qSz + gSz), outerSeq);

        idx = SetSequence(algoSz + innerSeqSz + (word32)(pSz + qSz + gSz) +
                          bitStringSz + (word32)ySz + outerSeqSz, output);

        /* check output size */
        if ((idx + algoSz + bitStringSz + innerSeqSz +
             (word32)(pSz + qSz + gSz + ySz)) > (word32)outLen)
        {
                WC_FREE_VAR_EX(p, key->heap, DYNAMIC_TYPE_TMP_BUFFER);
                WC_FREE_VAR_EX(q, key->heap, DYNAMIC_TYPE_TMP_BUFFER);
                WC_FREE_VAR_EX(g, key->heap, DYNAMIC_TYPE_TMP_BUFFER);
                WC_FREE_VAR_EX(y, key->heap, DYNAMIC_TYPE_TMP_BUFFER);
                WC_FREE_VAR_EX(algo, key->heap, DYNAMIC_TYPE_TMP_BUFFER);
            WOLFSSL_MSG("Error, output size smaller than outlen");
            return BUFFER_E;
        }

        /* outerSeq */
        XMEMCPY(output + idx, outerSeq, outerSeqSz);
        idx += outerSeqSz;
        /* algo */
        XMEMCPY(output + idx, algo, algoSz);
        idx += algoSz;
        WC_FREE_VAR_EX(algo, key->heap, DYNAMIC_TYPE_TMP_BUFFER);
    } else {
        innerSeqSz  = SetSequence((word32)(pSz + qSz + gSz + ySz), innerSeq);

        /* check output size */
        if ((innerSeqSz + (word32)(pSz + qSz + gSz + ySz)) > (word32)outLen) {
            WC_FREE_VAR_EX(p, key->heap, DYNAMIC_TYPE_TMP_BUFFER);
            WC_FREE_VAR_EX(q, key->heap, DYNAMIC_TYPE_TMP_BUFFER);
            WC_FREE_VAR_EX(g, key->heap, DYNAMIC_TYPE_TMP_BUFFER);
            WC_FREE_VAR_EX(y, key->heap, DYNAMIC_TYPE_TMP_BUFFER);
            WOLFSSL_MSG("Error, output size smaller than outlen");
            return BUFFER_E;
        }

        idx = 0;
    }

    /* innerSeq */
    XMEMCPY(output + idx, innerSeq, innerSeqSz);
    idx += innerSeqSz;
    /* p */
    XMEMCPY(output + idx, p, (size_t)pSz);
    idx += (word32)pSz;
    /* q */
    XMEMCPY(output + idx, q, (size_t)qSz);
    idx += (word32)qSz;
    /* g */
    XMEMCPY(output + idx, g, (size_t)gSz);
    idx += (word32)gSz;
    /* bit string */
    if (bitStringSz > 0) {
        XMEMCPY(output + idx, bitString, bitStringSz);
        idx += bitStringSz;
    }
    /* y */
    XMEMCPY(output + idx, y, (size_t)ySz);
    idx += (word32)ySz;

    WC_FREE_VAR_EX(p, key->heap, DYNAMIC_TYPE_TMP_BUFFER);
    WC_FREE_VAR_EX(q, key->heap, DYNAMIC_TYPE_TMP_BUFFER);
    WC_FREE_VAR_EX(g, key->heap, DYNAMIC_TYPE_TMP_BUFFER);
    WC_FREE_VAR_EX(y, key->heap, DYNAMIC_TYPE_TMP_BUFFER);
    return (int)idx;
#else
    DECL_ASNSETDATA(dataASN, dsaPubKeyASN_Length);
    int ret = 0;
    int i;
    int sz = 0;
    const ASNItem *data = NULL;
    int count = 0;

    WOLFSSL_ENTER("wc_SetDsaPublicKey");

    if ((output == NULL) || (key == NULL) || (outLen < MAX_SEQ_SZ)) {
        ret = BAD_FUNC_ARG;
    }

    CALLOC_ASNSETDATA(dataASN, dsaPubKeyASN_Length, ret, key->heap);

    if (ret == 0) {
        if (with_header) {
            /* Using dsaPubKeyASN */
            data = dsaPubKeyASN;
            count = dsaPubKeyASN_Length;
            /* Set the algorithm OID to write out. */
            SetASN_OID(&dataASN[DSAPUBKEYASN_IDX_ALGOID_OID], DSAk, oidKeyType);
            /* Set the mp_ints to encode - parameters and public value. */
            for (i = 0; i < DSA_INTS - 2; i++) {
                SetASN_MP(&dataASN[(int)DSAPUBKEYASN_IDX_ALGOID_PARAMS_P + i],
                        GetDsaInt(key, i));
            }
            SetASN_MP(&dataASN[DSAPUBKEYASN_IDX_PUBKEY_Y], GetDsaInt(key, i));
        }
        else {
            /* Using dsaKeyASN */
            data = dsaKeyASN;
            count = dsaPublicKeyASN_Length;
            /* Set the mp_ints to encode - parameters and public value. */
            for (i = 0; i < DSA_INTS - 1; i++) {
                /* Move all DSA ints up one slot (ignore VERSION so now
                 * it means P) */
                SetASN_MP(&dataASN[(int)DSAKEYASN_IDX_VER + i],
                        GetDsaInt(key, i));
            }
        }
        ret = SizeASN_Items(data, dataASN, count, &sz);
    }
    /* Check buffer is big enough for encoding. */
    if ((ret == 0) && (sz > (int)outLen)) {
        ret = BAD_FUNC_ARG;
    }
    /* Encode the DSA public key into output buffer. */
    if (ret == 0) {
        ret = SetASN_Items(data, dataASN, count, output);
    }

    FREE_ASNSETDATA(dataASN, key->heap);
    return ret;
#endif /* WOLFSSL_ASN_TEMPLATE */
}

/* Encode a DSA public key into buffer.
 *
 * @param [out] output       Buffer to hold encoded data.
 * @param [in]  key          DSA key object.
 * @param [out] outLen       Length of buffer.
 * @param [out] with_header  Whether to encode in SubjectPublicKeyInfo block.
 * @return  Size of encoded data in bytes on success.
 * @return  BAD_FUNC_ARG when output or key is NULL, or buffer size is less
 *          than a minimal size (5 bytes), or buffer size is smaller than
 *          encoding size.
 * @return  MEMORY_E when dynamic memory allocation fails.
 */
int wc_DsaKeyToPublicDer(DsaKey* key, byte* output, word32 inLen)
{
    return wc_SetDsaPublicKey(output, key, (int)inLen, 1);
}
#endif /* !HAVE_SELFTEST && (WOLFSSL_KEY_GEN || WOLFSSL_CERT_GEN) */

static int DsaKeyIntsToDer(DsaKey* key, byte* output, word32* inLen,
                           int ints, int includeVersion)
{
#ifndef WOLFSSL_ASN_TEMPLATE
    word32 seqSz = 0, verSz = 0, intTotalLen = 0, outLen, j;
    word32 sizes[DSA_INTS];
    int    i, ret = 0;

    byte  seq[MAX_SEQ_SZ];
    byte  ver[MAX_VERSION_SZ];
    byte* tmps[DSA_INTS];

    if (ints > DSA_INTS || inLen == NULL)
        return BAD_FUNC_ARG;

    XMEMSET(sizes, 0, sizeof(sizes));
    for (i = 0; i < ints; i++)
        tmps[i] = NULL;

    /* write all big ints from key to DER tmps */
    for (i = 0; i < ints; i++) {
        int mpSz;
        mp_int* keyInt = GetDsaInt(key, i);
        word32 rawLen = (word32)mp_unsigned_bin_size(keyInt) + 1;

        tmps[i] = (byte*)XMALLOC(rawLen + MAX_SEQ_SZ, key->heap,
                                                              DYNAMIC_TYPE_DSA);
        if (tmps[i] == NULL) {
            ret = MEMORY_E;
            break;
        }

        mpSz = SetASNIntMP(keyInt, -1, tmps[i]);
        if (mpSz < 0) {
            ret = mpSz;
            break;
        }
        sizes[i] = (word32)mpSz;
        intTotalLen += (word32)mpSz;
    }

    if (ret != 0) {
        FreeTmpDsas(tmps, key->heap, ints);
        return ret;
    }

    /* make headers */
    if (includeVersion)
        verSz = (word32)SetMyVersion(0, ver, FALSE);
    seqSz = SetSequence(verSz + intTotalLen, seq);

    outLen = seqSz + verSz + intTotalLen;
    *inLen = outLen;
    if (output == NULL) {
        FreeTmpDsas(tmps, key->heap, ints);
        return WC_NO_ERR_TRACE(LENGTH_ONLY_E);
    }
    if (outLen > *inLen) {
        FreeTmpDsas(tmps, key->heap, ints);
        return BAD_FUNC_ARG;
    }

    /* write to output */
    XMEMCPY(output, seq, seqSz);
    j = seqSz;
    if (includeVersion) {
        XMEMCPY(output + j, ver, verSz);
        j += verSz;
    }

    for (i = 0; i < ints; i++) {
        XMEMCPY(output + j, tmps[i], sizes[i]);
        j += sizes[i];
    }
    FreeTmpDsas(tmps, key->heap, ints);

    return (int)outLen;
#else
    DECL_ASNSETDATA(dataASN, dsaKeyASN_Length);
    int ret = 0;
    int sz = 0;

    (void)ints;

    if ((key == NULL) || (inLen == NULL)) {
        ret = BAD_FUNC_ARG;
    }
    if ((ret == 0) && (ints > DSA_INTS)) {
        ret = BAD_FUNC_ARG;
    }

    CALLOC_ASNSETDATA(dataASN, dsaKeyASN_Length, ret, key->heap);

    if (ret == 0) {
        int i;

        if (includeVersion) {
            /* Set the version. */
            SetASN_Int8Bit(&dataASN[DSAKEYASN_IDX_VER], 0);
        }
        else {
            dataASN[DSAKEYASN_IDX_VER].noOut = 1;
        }
        dataASN[DSAKEYASN_IDX_Y].noOut = mp_iszero(&key->y);
        dataASN[DSAKEYASN_IDX_X].noOut = mp_iszero(&key->x);
        /* Set the mp_ints to encode - params, public and private value. */
        for (i = 0; i < DSA_INTS; i++) {
            if (i < ints)
                SetASN_MP(&dataASN[(int)DSAKEYASN_IDX_P + i], GetDsaInt(key, i));
            else
                dataASN[(int)DSAKEYASN_IDX_P + i].noOut = 1;
        }
        /* Calculate size of the encoding. */
        ret = SizeASN_Items(dsaKeyASN, dataASN, dsaKeyASN_Length, &sz);
    }
    if ((ret == 0) && (output == NULL)) {
        *inLen = (word32)sz;
        ret = WC_NO_ERR_TRACE(LENGTH_ONLY_E);
    }
    /* Check buffer is big enough for encoding. */
    if ((ret == 0) && (sz > (int)*inLen)) {
        ret = BAD_FUNC_ARG;
    }
    if (ret == 0) {
        /* Encode the DSA private key into output buffer. */
        SetASN_Items(dsaKeyASN, dataASN, dsaKeyASN_Length, output);
        /* Return the size of the encoding. */
        ret = sz;
    }

    FREE_ASNSETDATA(dataASN, key->heap);
    return ret;
#endif /* WOLFSSL_ASN_TEMPLATE */
}

/* Encode a DSA private key into buffer.
 *
 * @param [in]  key          DSA key object.
 * @param [out] output       Buffer to hold encoded data.
 * @param [out] inLen        Length of buffer.
 * @return  Size of encoded data in bytes on success.
 * @return  BAD_FUNC_ARG when key or output is NULL, or key is not a private key
 *          or, buffer size is smaller than encoding size.
 * @return  MEMORY_E when dynamic memory allocation fails.
 */
int wc_DsaKeyToDer(DsaKey* key, byte* output, word32 inLen)
{
    if (!key || !output)
        return BAD_FUNC_ARG;

    if (key->type != DSA_PRIVATE)
        return BAD_FUNC_ARG;

    return DsaKeyIntsToDer(key, output, &inLen, DSA_INTS, 1);
}

/* Convert DsaKey parameters to DER format, write to output (inLen),
   return bytes written. Version is excluded to be compatible with
   OpenSSL d2i_DSAparams */
int wc_DsaKeyToParamsDer(DsaKey* key, byte* output, word32 inLen)
{
    if (!key || !output)
        return BAD_FUNC_ARG;

    return DsaKeyIntsToDer(key, output, &inLen, DSA_PARAM_INTS, 0);
}

/* This version of the function allows output to be NULL. In that case, the
   DsaKeyIntsToDer will return WC_NO_ERR_TRACE(LENGTH_ONLY_E) and the required
   output buffer size will be pointed to by inLen. */
int wc_DsaKeyToParamsDer_ex(DsaKey* key, byte* output, word32* inLen)
{
    if (!key || !inLen)
        return BAD_FUNC_ARG;

    return DsaKeyIntsToDer(key, output, inLen, DSA_PARAM_INTS, 0);
}

#endif /* NO_DSA */

#ifndef NO_CERTS
/* Initialize decoded certificate object with buffer of DER encoding.
 *
 * @param [in, out] cert    Decoded certificate object.
 * @param [in]      source  Buffer containing DER encoded certificate.
 * @param [in]      inSz    Size of DER data in buffer in bytes.
 * @param [in]      heap    Dynamic memory hint.
 */
void InitDecodedCert(DecodedCert* cert,
                     const byte* source, word32 inSz, void* heap)
{
    InitDecodedCert_ex(cert, source, inSz, heap, INVALID_DEVID);
}


/* Initialize decoded certificate object with buffer of DER encoding.
 *
 * @param [in, out] cert    Decoded certificate object.
 * @param [in]      source  Buffer containing DER encoded certificate.
 * @param [in]      inSz    Size of DER data in buffer in bytes.
 * @param [in]      heap    Dynamic memory hint.
 * @param [in]      devId   Crypto callback ID to use.
 */
void InitDecodedCert_ex(DecodedCert* cert,
                     const byte* source, word32 inSz, void* heap, int devId)
{
    if (cert != NULL) {
        XMEMSET(cert, 0, sizeof(DecodedCert));

        cert->subjectCNEnc    = CTC_UTF8;
        cert->issuer[0]       = '\0';
        cert->subject[0]      = '\0';
        cert->source          = source;  /* don't own */
        cert->maxIdx          = inSz;    /* can't go over this index */
        cert->heap            = heap;
        cert->maxPathLen      = WOLFSSL_MAX_PATH_LEN;
    #if defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_CERT_EXT)
        #ifdef WOLFSSL_CERT_NAME_ALL
        cert->subjectNEnc     = CTC_UTF8;
        cert->subjectIEnc     = CTC_UTF8;
        cert->subjectDNQEnc   = CTC_UTF8;
        cert->subjectGNEnc    = CTC_UTF8;
        #endif
        cert->subjectSNEnc    = CTC_UTF8;
        cert->subjectCEnc     = CTC_PRINTABLE;
        cert->subjectLEnc     = CTC_UTF8;
        cert->subjectSTEnc    = CTC_UTF8;
        cert->subjectOEnc     = CTC_UTF8;
        cert->subjectOUEnc    = CTC_UTF8;
    #ifdef WOLFSSL_HAVE_ISSUER_NAMES
        cert->issuerSNEnc    = CTC_UTF8;
        cert->issuerCEnc     = CTC_PRINTABLE;
        cert->issuerLEnc     = CTC_UTF8;
        cert->issuerSTEnc    = CTC_UTF8;
        cert->issuerOEnc     = CTC_UTF8;
        cert->issuerOUEnc    = CTC_UTF8;
    #endif /* WOLFSSL_HAVE_ISSUER_NAMES */
    #endif /* WOLFSSL_CERT_GEN || WOLFSSL_CERT_EXT */

        InitSignatureCtx(&cert->sigCtx, heap, devId);
    }
}

void wc_InitDecodedCert(DecodedCert* cert, const byte* source, word32 inSz,
                        void* heap)
{
    InitDecodedCert(cert, source, inSz, heap);
}

/* Free the alternative names object.
 *
 * Frees each linked list items and its name.
 *
 * @param [in, out] altNames  Alternative names.
 * @param [in]      heap      Dynamic memory hint.
 */
void FreeAltNames(DNS_entry* altNames, void* heap)
{
    (void)heap;

    while (altNames) {
        DNS_entry* tmp = altNames->next;

        XFREE(altNames->name, heap, DYNAMIC_TYPE_ALTNAME);
    #ifdef WOLFSSL_IP_ALT_NAME
        XFREE(altNames->ipString, heap, DYNAMIC_TYPE_ALTNAME);
    #endif
    #ifdef WOLFSSL_RID_ALT_NAME
        XFREE(altNames->ridString, heap, DYNAMIC_TYPE_ALTNAME);
    #endif
        XFREE(altNames,       heap, DYNAMIC_TYPE_ALTNAME);
        altNames = tmp;
    }
}

/* malloc and initialize a new alt name structure */
DNS_entry* AltNameNew(void* heap)
{
    DNS_entry* ret;
    ret = (DNS_entry*)XMALLOC(sizeof(DNS_entry), heap, DYNAMIC_TYPE_ALTNAME);
    if (ret != NULL) {
        XMEMSET(ret, 0, sizeof(DNS_entry));
    }
    (void)heap;
    return ret;
}

DNS_entry* AltNameDup(DNS_entry* from, void* heap)
{
    DNS_entry* ret;

    ret = AltNameNew(heap);
    if (ret == NULL) {
        WOLFSSL_MSG("\tOut of Memory");
        return NULL;
    }

    ret->type = from->type;
    ret->len = from->len;


    ret->name = CopyString(from->name, from->len, heap, DYNAMIC_TYPE_ALTNAME);
#ifdef WOLFSSL_IP_ALT_NAME
    ret->ipString = CopyString(from->ipString, 0, heap, DYNAMIC_TYPE_ALTNAME);
#endif
#ifdef WOLFSSL_RID_ALT_NAME
    ret->ridString = CopyString(from->ridString, 0, heap, DYNAMIC_TYPE_ALTNAME);
#endif
    if (ret->name == NULL
#ifdef WOLFSSL_IP_ALT_NAME
            || (from->ipString != NULL && ret->ipString == NULL)
#endif
#ifdef WOLFSSL_RID_ALT_NAME
            || (from->ridString != NULL && ret->ridString == NULL)
#endif
            ) {
        WOLFSSL_MSG("\tOut of Memory");
        FreeAltNames(ret, heap);
        return NULL;
    }

#ifdef WOLFSSL_FPKI
    ret->oidSum = from->oidSum;
#endif

    return ret;
}


#ifndef IGNORE_NAME_CONSTRAINTS

/* Free the subtree names object.
 *
 * Frees each linked list items and its name.
 *
 * @param [in, out] names  Subtree names.
 * @param [in]      heap   Dynamic memory hint.
 */
void FreeNameSubtrees(Base_entry* names, void* heap)
{
    (void)heap;

    while (names) {
        Base_entry* tmp = names->next;

        XFREE(names->name, heap, DYNAMIC_TYPE_ALTNAME);
        XFREE(names,       heap, DYNAMIC_TYPE_ALTNAME);
        names = tmp;
    }
}

#endif /* IGNORE_NAME_CONSTRAINTS */

/* Free the decoded cert object's dynamic data.
 *
 * @param [in, out] cert  Decoded certificate object.
 */
void FreeDecodedCert(DecodedCert* cert)
{
    if (cert == NULL)
        return;
    if (cert->subjectCNStored == 1) {
        XFREE(cert->subjectCN, cert->heap, DYNAMIC_TYPE_SUBJECT_CN);
    }
    if (cert->pubKeyStored == 1) {
        XFREE((void*)cert->publicKey, cert->heap, DYNAMIC_TYPE_PUBLIC_KEY);
    }
    if (cert->weOwnAltNames && cert->altNames)
        FreeAltNames(cert->altNames, cert->heap);
#ifndef IGNORE_NAME_CONSTRAINTS
    if (cert->altEmailNames)
        FreeAltNames(cert->altEmailNames, cert->heap);
    if (cert->altDirNames)
        FreeAltNames(cert->altDirNames, cert->heap);
    if (cert->permittedNames)
        FreeNameSubtrees(cert->permittedNames, cert->heap);
    if (cert->excludedNames)
        FreeNameSubtrees(cert->excludedNames, cert->heap);
#endif /* IGNORE_NAME_CONSTRAINTS */
#ifdef WOLFSSL_SEP
    XFREE(cert->deviceType, cert->heap, DYNAMIC_TYPE_X509_EXT);
    XFREE(cert->hwType, cert->heap, DYNAMIC_TYPE_X509_EXT);
    XFREE(cert->hwSerialNum, cert->heap, DYNAMIC_TYPE_X509_EXT);
#endif /* WOLFSSL_SEP */
#ifdef WOLFSSL_X509_NAME_AVAILABLE
    if (cert->issuerName != NULL)
        wolfSSL_X509_NAME_free((WOLFSSL_X509_NAME*)cert->issuerName);
    if (cert->subjectName != NULL)
        wolfSSL_X509_NAME_free((WOLFSSL_X509_NAME*)cert->subjectName);
#endif /* WOLFSSL_X509_NAME_AVAILABLE */
#if defined(WOLFSSL_RENESAS_TSIP_TLS) || defined(WOLFSSL_RENESAS_FSPSM_TLS)
    XFREE(cert->sce_tsip_encRsaKeyIdx, cert->heap, DYNAMIC_TYPE_RSA);
#endif
    FreeSignatureCtx(&cert->sigCtx);
}

void wc_FreeDecodedCert(DecodedCert* cert)
{
    FreeDecodedCert(cert);
}

#ifndef WOLFSSL_ASN_TEMPLATE
static int GetCertHeader(DecodedCert* cert)
{
    int ret = 0, len;

    if (GetSequence(cert->source, &cert->srcIdx, &len, cert->maxIdx) < 0)
        return ASN_PARSE_E;

    /* Reset the max index for the size indicated in the outer wrapper. */
    cert->maxIdx = (word32)len + cert->srcIdx;
    cert->certBegin = cert->srcIdx;

    if (GetSequence(cert->source, &cert->srcIdx, &len, cert->maxIdx) < 0)
        return ASN_PARSE_E;

    cert->sigIndex = (word32)len + cert->srcIdx;
    if (cert->sigIndex > cert->maxIdx)
        return ASN_PARSE_E;

    if (GetExplicitVersion(cert->source, &cert->srcIdx, &cert->version,
                                                            cert->sigIndex) < 0)
        return ASN_PARSE_E;

    ret = wc_GetSerialNumber(cert->source, &cert->srcIdx, cert->serial,
        &cert->serialSz, cert->sigIndex);
    if (ret < 0) {
        return ret;
    }

    return ret;
}
#endif

#if defined(HAVE_ED25519) || defined(HAVE_ED448) || defined(HAVE_FALCON) || \
    defined(HAVE_DILITHIUM) || defined(HAVE_SPHINCS)
/* Store the key data under the BIT_STRING in dynamically allocated data.
 *
 * @param [in, out] cert    Certificate object.
 * @param [in]      source  Buffer containing encoded key.
 * @param [in, out] srcIdx  On in, start of key data.
 *                          On out, start of element after key data.
 * @param [in]      maxIdx  Maximum index of certificate data.
 */
static int StoreKey(DecodedCert* cert, const byte* source, word32* srcIdx,
                    word32 maxIdx)
{
    int ret;
    int length;
    byte* publicKey;

    ret = CheckBitString(source, srcIdx, &length, maxIdx, 1, NULL);
    if (ret == 0) {
    #ifdef HAVE_OCSP
        ret = CalcHashId_ex(source + *srcIdx, (word32)length,
            cert->subjectKeyHash, HashIdAlg(cert->signatureOID));
    }
    if (ret == 0) {
    #endif
        publicKey = (byte*)XMALLOC((size_t)length, cert->heap,
                                   DYNAMIC_TYPE_PUBLIC_KEY);
        if (publicKey == NULL) {
            ret = MEMORY_E;
        }
        else {
            XMEMCPY(publicKey, &source[*srcIdx], (size_t)length);
            cert->publicKey = publicKey;
            cert->pubKeyStored = 1;
            cert->pubKeySize   = (word32)length;

            *srcIdx += (word32)length;
        }
    }

    return ret;
}
#endif /* HAVE_ED25519 || HAVE_ED448 */
#endif

#if defined(HAVE_ECC) && defined(HAVE_ECC_KEY_EXPORT)

static int SetCurve(ecc_key* key, byte* output, size_t outSz)
{
#ifdef HAVE_OID_ENCODING
    int ret;
#endif
    int idx;
    word32 oidSz = 0;

    /* validate key */
    if (key == NULL || key->dp == NULL) {
        return BAD_FUNC_ARG;
    }

#ifdef HAVE_OID_ENCODING
    ret = EncodeObjectId(key->dp->oid, key->dp->oidSz, NULL, &oidSz);
    if (ret != 0) {
        return ret;
    }
#else
    oidSz = key->dp->oidSz;
#endif

    idx = SetObjectId((int)oidSz, output);

    /* length only */
    if (output == NULL) {
        return idx + (int)oidSz;
    }

    /* verify output buffer has room */
    if (oidSz > outSz)
        return BUFFER_E;

#ifdef HAVE_OID_ENCODING
    ret = EncodeObjectId(key->dp->oid, key->dp->oidSz, output+idx, &oidSz);
    if (ret != 0) {
        return ret;
    }
#else
    XMEMCPY(output+idx, key->dp->oid, oidSz);
#endif
    idx += (int)oidSz;

    return idx;
}

#endif /* HAVE_ECC && HAVE_ECC_KEY_EXPORT */

#ifdef HAVE_ECC
#ifdef WOLFSSL_ASN_TEMPLATE
/* ASN.1 template for ECC public key (SubjectPublicKeyInfo).
 * RFC 5480, 2 - Subject Public Key Information Fields
 *           2.1.1 - Unrestricted Algorithm Identifier and Parameters
 * X9.62 ECC point format.
 * See ASN.1 template 'eccSpecifiedASN' for specifiedCurve.
 */
static const ASNItem eccPublicKeyASN[] = {
/* SEQ            */ { 0, ASN_SEQUENCE, 1, 1, 0 },
                                             /* AlgorithmIdentifier */
/* ALGOID_SEQ     */     { 1, ASN_SEQUENCE, 1, 1, 0 },
                                                 /* algorithm */
/* ALGOID_OID     */         { 2, ASN_OBJECT_ID, 0, 0, 0 },
                                                 /* namedCurve */
/* ALGOID_CURVEID */         { 2, ASN_OBJECT_ID, 0, 0, 2 },
                                                 /* specifiedCurve - explicit parameters */
/* ALGOID_PARAMS  */         { 2, ASN_SEQUENCE, 1, 0, 2 },
                                             /* Public Key */
/* PUBKEY         */     { 1, ASN_BIT_STRING, 0, 0, 0 },
};
enum {
    ECCPUBLICKEYASN_IDX_SEQ = 0,
    ECCPUBLICKEYASN_IDX_ALGOID_SEQ,
    ECCPUBLICKEYASN_IDX_ALGOID_OID,
    ECCPUBLICKEYASN_IDX_ALGOID_CURVEID,
    ECCPUBLICKEYASN_IDX_ALGOID_PARAMS,
    ECCPUBLICKEYASN_IDX_PUBKEY
};

/* Number of items in ASN.1 template for ECC public key. */
#define eccPublicKeyASN_Length (sizeof(eccPublicKeyASN) / sizeof(ASNItem))
#endif /* WOLFSSL_ASN_TEMPLATE */
#endif /* HAVE_ECC */

#if defined(HAVE_ECC) && defined(HAVE_ECC_KEY_EXPORT)

/* Encode public ECC key in DER format.
 *
 * RFC 5480, 2 - Subject Public Key Information Fields
 *           2.1.1 - Unrestricted Algorithm Identifier and Parameters
 * X9.62 ECC point format.
 * SEC 1 Ver. 2.0, C.2 - Syntax for Elliptic Curve Domain Parameters
 *
 * @param [out] output       Buffer to put encoded data in.
 * @param [in]  key          ECC key object.
 * @param [in]  outLen       Size of buffer in bytes.
 * @param [in]  with_header  Whether to use SubjectPublicKeyInfo format.
 * @return  Size of encoded data in bytes on success.
 * @return  BAD_FUNC_ARG when key or key's parameters is NULL.
 * @return  MEMORY_E when dynamic memory allocation failed.
 */
static int SetEccPublicKey(byte* output, ecc_key* key, int outLen,
                           int with_header, int comp)
{
#ifndef WOLFSSL_ASN_TEMPLATE
    int ret;
    word32 idx = 0, curveSz, algoSz, pubSz, bitStringSz;
    byte bitString[1 + MAX_LENGTH_SZ + 1]; /* 6 */
    byte algo[MAX_ALGO_SZ];  /* 20 */

    /* public size */
    pubSz = key->dp ? (word32)key->dp->size : MAX_ECC_BYTES;
    if (comp)
        pubSz = 1 + pubSz;
    else
        pubSz = 1 + 2 * pubSz;

    /* check for buffer overflow */
    if (output != NULL && pubSz > (word32)outLen) {
        return BUFFER_E;
    }

    /* headers */
    if (with_header) {
        ret = SetCurve(key, NULL, 0);
        if (ret <= 0) {
            return ret;
        }
        curveSz = (word32)ret;
        ret = 0;

        /* calculate size */
        algoSz  = SetAlgoID(ECDSAk, algo, oidKeyType, (int)curveSz);
        bitStringSz = SetBitString(pubSz, 0, bitString);
        idx = SetSequence(pubSz + curveSz + bitStringSz + algoSz, NULL);

        /* check for buffer overflow */
        if (output != NULL &&
                curveSz + algoSz + bitStringSz + idx + pubSz > (word32)outLen) {
            return BUFFER_E;
        }

        idx = SetSequence(pubSz + curveSz + bitStringSz + algoSz,
            output);
        /* algo */
        if (output)
            XMEMCPY(output + idx, algo, algoSz);
        idx += algoSz;
        /* curve */
        if (output)
            (void)SetCurve(key, output + idx, curveSz);
        idx += curveSz;
        /* bit string */
        if (output)
            XMEMCPY(output + idx, bitString, bitStringSz);
        idx += bitStringSz;
    }

    /* pub */
    if (output) {
        PRIVATE_KEY_UNLOCK();
        ret = wc_ecc_export_x963_ex(key, output + idx, &pubSz, comp);
        PRIVATE_KEY_LOCK();
        if (ret != 0) {
            return ret;
        }
    }
    idx += pubSz;

    return (int)idx;
#else
    word32 pubSz = 0;
    int sz = 0;
    int ret = 0;
    int curveIdSz = 0;
    byte* curveOid = NULL;

    /* Check key validity. */
    if ((key == NULL) || (key->dp == NULL)) {
        ret = BAD_FUNC_ARG;
    }

    if (ret == 0) {
        /* Calculate the size of the encoded public point. */
        PRIVATE_KEY_UNLOCK();
    #if defined(HAVE_COMP_KEY) && defined(HAVE_FIPS) && FIPS_VERSION3_LT(6,0,0)
        /* in earlier versions of FIPS the get length functionality is not
         * available with compressed keys */
        pubSz = key->dp ? key->dp->size : MAX_ECC_BYTES;
        if (comp)
            pubSz = 1 + pubSz;
        else
            pubSz = 1 + 2 * pubSz;
        ret = WC_NO_ERR_TRACE(LENGTH_ONLY_E);
    #else
        ret = wc_ecc_export_x963_ex(key, NULL, &pubSz, comp);
    #endif
        PRIVATE_KEY_LOCK();
        /* LENGTH_ONLY_E on success. */
        if (ret == WC_NO_ERR_TRACE(LENGTH_ONLY_E)) {
            ret = 0;
        }
    }
    if ((ret == 0) && with_header) {
        /* Including SubjectPublicKeyInfo header. */
        DECL_ASNSETDATA(dataASN, eccPublicKeyASN_Length);

        CALLOC_ASNSETDATA(dataASN, eccPublicKeyASN_Length, ret, key->heap);

        /* Get the length of the named curve OID to put into the encoding. */
        curveIdSz = SetCurve(key, NULL, 0);
        if (curveIdSz < 0) {
            ret = curveIdSz;
        }

        if (ret == 0) {
            /* Set the key type OID. */
            SetASN_OID(&dataASN[ECCPUBLICKEYASN_IDX_ALGOID_OID], ECDSAk,
                    oidKeyType);
            /* Set the curve OID. */
            SetASN_ReplaceBuffer(&dataASN[ECCPUBLICKEYASN_IDX_ALGOID_CURVEID],
                NULL, (word32)curveIdSz);
            /* Don't try to write out explicit parameters. */
            dataASN[ECCPUBLICKEYASN_IDX_ALGOID_PARAMS].noOut = 1;
            /* Set size of public point to ensure space is made for it. */
            SetASN_Buffer(&dataASN[ECCPUBLICKEYASN_IDX_PUBKEY], NULL, pubSz);
            /* Calculate size of ECC public key. */
            ret = SizeASN_Items(eccPublicKeyASN, dataASN,
                                eccPublicKeyASN_Length, &sz);
        }

        /* Check buffer, if passed in, is big enough for encoded data. */
        if ((ret == 0) && (output != NULL) && (sz > outLen)) {
            ret = BUFFER_E;
        }
        if ((ret == 0) && (output != NULL)) {
            /* Encode ECC public key. */
            SetASN_Items(eccPublicKeyASN, dataASN, eccPublicKeyASN_Length,
                         output);
            /* Skip to where public point is to be encoded. */
            output += sz - (int)pubSz;
            /* Cache the location to place the name curve OID. */
            curveOid = (byte*)
                dataASN[ECCPUBLICKEYASN_IDX_ALGOID_CURVEID].data.buffer.data;
        }

        FREE_ASNSETDATA(dataASN, key->heap);
    }
    else if ((ret == 0) && (output != NULL) && (pubSz > (word32)outLen)) {
        ret = BUFFER_E;
    }
    else {
        /* Total size is the public point size. */
        sz = (int)pubSz;
    }

    if ((ret == 0) && (output != NULL)) {
        /* Put named curve OID data into encoding. */
        curveIdSz = SetCurve(key, curveOid, (size_t)curveIdSz);
        if (curveIdSz < 0) {
            ret = curveIdSz;
        }
    }
    if ((ret == 0) && (output != NULL)) {
        /* Encode public point. */
        PRIVATE_KEY_UNLOCK();
        ret = wc_ecc_export_x963_ex(key, output, &pubSz, comp);
        PRIVATE_KEY_LOCK();
    }
    if (ret == 0) {
        /* Return the size of the encoding. */
        ret = sz;
    }

    return ret;
#endif
}


/* Encode the public part of an ECC key in a DER.
 *
 * Pass NULL for output to get the size of the encoding.
 *
 * @param [in]  key            ECC key object.
 * @param [out] output         Buffer to hold DER encoding.
 * @param [in]  inLen          Size of buffer in bytes.
 * @param [in]  with_AlgCurve  Whether to use SubjectPublicKeyInfo format.
 * @return  Size of encoded data in bytes on success.
 * @return  BAD_FUNC_ARG when key or key's parameters is NULL.
 * @return  MEMORY_E when dynamic memory allocation failed.
 */
WOLFSSL_ABI
int wc_EccPublicKeyToDer(ecc_key* key, byte* output, word32 inLen,
                                                              int with_AlgCurve)
{
    return SetEccPublicKey(output, key, (int)inLen, with_AlgCurve, 0);
}

int wc_EccPublicKeyToDer_ex(ecc_key* key, byte* output, word32 inLen,
                                                    int with_AlgCurve, int comp)
{
    return SetEccPublicKey(output, key, (int)inLen, with_AlgCurve, comp);
}

int wc_EccPublicKeyDerSize(ecc_key* key, int with_AlgCurve)
{
    return SetEccPublicKey(NULL, key, 0, with_AlgCurve, 0);
}

#endif /* HAVE_ECC && HAVE_ECC_KEY_EXPORT */

#ifdef WOLFSSL_ASN_TEMPLATE
#if defined(WC_ENABLE_ASYM_KEY_EXPORT) || defined(WC_ENABLE_ASYM_KEY_IMPORT)
/* ASN.1 template for the SubjectPublicKeyInfo of a general asymmetric key.
 * Used with Ed448/Ed25519, Curve448/Curve25519, SPHINCS+, falcon, dilithium,
 * etc.
 *
 * X.509: RFC 5280, 4.1 - SubjectPublicKeyInfo
 * RFC 8410, 4 - Subject Public Key Fields
 */
static const ASNItem publicKeyASN[] = {
            /* SubjectPublicKeyInfo */
/* SEQ        */ { 0, ASN_SEQUENCE, 1, 1, 0 },
                                     /* AlgorithmIdentifier */
/* ALGOID_SEQ */     { 1, ASN_SEQUENCE, 1, 1, 0 },
                                         /* Ed25519/Ed448 OID, etc. */
/* ALGOID_OID */         { 2, ASN_OBJECT_ID, 0, 0, 1 },
                                     /* Public key stream */
/* PUBKEY     */     { 1, ASN_BIT_STRING, 0, 0, 0 },
};
enum {
    PUBKEYASN_IDX_SEQ = 0,
    PUBKEYASN_IDX_ALGOID_SEQ,
    PUBKEYASN_IDX_ALGOID_OID,
    PUBKEYASN_IDX_PUBKEY
};

/* Number of items in ASN.1 template for public key SubjectPublicKeyInfo. */
#define publicKeyASN_Length (sizeof(publicKeyASN) / sizeof(ASNItem))
#endif /* WC_ENABLE_ASYM_KEY_EXPORT || WC_ENABLE_ASYM_KEY_IMPORT */
#endif /* WOLFSSL_ASN_TEMPLATE */

#ifdef WC_ENABLE_ASYM_KEY_EXPORT

/* Build ASN.1 formatted public key based on RFC 5280 and RFC 8410
 *
 * Pass NULL for output to get the size of the encoding.
 *
 * @param [in]  pubKey       public key buffer
 * @param [in]  pubKeyLen    public key buffer length
 * @param [out] output       Buffer to put encoded data in (optional)
 * @param [in]  outLen       Size of buffer in bytes
 * @param [in]  keyType      is "enum Key_Sum" like ED25519k
 * @param [in]  withHeader   Whether to include SubjectPublicKeyInfo around key.
 * @return  Size of encoded data in bytes on success
 * @return  BAD_FUNC_ARG when key is NULL.
 * @return  MEMORY_E when dynamic memory allocation failed.
 */
int SetAsymKeyDerPublic(const byte* pubKey, word32 pubKeyLen,
    byte* output, word32 outLen, int keyType, int withHeader)
{
    int ret = 0;
#ifndef WOLFSSL_ASN_TEMPLATE
    word32 idx = 0;
    word32 seqDataSz = 0;
    word32 sz;
#else
    int sz = 0;
    DECL_ASNSETDATA(dataASN, publicKeyASN_Length);
#endif

    /* validate parameters */
    if (pubKey == NULL){
        return BAD_FUNC_ARG;
    }
    if (output != NULL && outLen == 0) {
        return BUFFER_E;
    }

#ifndef WOLFSSL_ASN_TEMPLATE
    /* calculate size */
    if (withHeader) {
        word32 algoSz      = SetAlgoID(keyType, NULL, oidKeyType, 0);
        word32 bitStringSz = SetBitString(pubKeyLen, 0, NULL);

        seqDataSz = algoSz + bitStringSz + pubKeyLen;
        sz = SetSequence(seqDataSz, NULL) + seqDataSz;
    }
    else {
        sz = pubKeyLen;
    }

    /* checkout output size */
    if (output != NULL && sz > outLen) {
        ret = BUFFER_E;
    }

    /* headers */
    if (ret == 0 && output != NULL && withHeader) {
        /* sequence */
        idx = SetSequence(seqDataSz, output);
        /* algo */
        idx += SetAlgoID(keyType, output + idx, oidKeyType, 0);
        /* bit string */
        idx += SetBitString(pubKeyLen, 0, output + idx);
    }

    if (ret == 0 && output != NULL) {
        /* pub */
        XMEMCPY(output + idx, pubKey, pubKeyLen);
        idx += pubKeyLen;

        sz = idx;
    }

    if (ret == 0) {
        ret = (int)sz;
    }
#else
    if (withHeader) {
        CALLOC_ASNSETDATA(dataASN, publicKeyASN_Length, ret, NULL);

        if (ret == 0) {
            /* Set the OID. */
            SetASN_OID(&dataASN[PUBKEYASN_IDX_ALGOID_OID], (word32)keyType,
                    oidKeyType);
            /* Leave space for public point. */
            SetASN_Buffer(&dataASN[PUBKEYASN_IDX_PUBKEY], NULL, pubKeyLen);
            /* Calculate size of public key encoding. */
            ret = SizeASN_Items(publicKeyASN, dataASN, publicKeyASN_Length,
                                &sz);
        }
        if ((ret == 0) && (output != NULL) && (sz > (int)outLen)) {
            ret = BUFFER_E;
        }
        if ((ret == 0) && (output != NULL)) {
            /* Encode public key. */
            SetASN_Items(publicKeyASN, dataASN, publicKeyASN_Length, output);
            /* Set location to encode public point. */
            output = (byte*)dataASN[PUBKEYASN_IDX_PUBKEY].data.buffer.data;
        }

        FREE_ASNSETDATA(dataASN, NULL);
    }
    else if ((output != NULL) && (pubKeyLen > outLen)) {
        ret = BUFFER_E;
    }
    else if (ret == 0) {
        sz = (int)pubKeyLen;
    }

    if ((ret == 0) && (output != NULL)) {
        /* Put public key into space provided. */
        XMEMCPY(output, pubKey, pubKeyLen);
    }
    if (ret == 0) {
        ret = sz;
    }
#endif /* WOLFSSL_ASN_TEMPLATE */
    return ret;
}
#endif /* WC_ENABLE_ASYM_KEY_EXPORT */

#if defined(HAVE_ED25519) && defined(HAVE_ED25519_KEY_EXPORT)
/* Encode the public part of an Ed25519 key in DER.
 *
 * Pass NULL for output to get the size of the encoding.
 *
 * @param [in]  key       Ed25519 key object.
 * @param [out] output    Buffer to put encoded data in.
 * @param [in]  outLen    Size of buffer in bytes.
 * @param [in]  withAlg   Whether to use SubjectPublicKeyInfo format.
 * @return  Size of encoded data in bytes on success.
 * @return  BAD_FUNC_ARG when key is NULL.
 * @return  MEMORY_E when dynamic memory allocation failed.
 */
int wc_Ed25519PublicKeyToDer(const ed25519_key* key, byte* output, word32 inLen,
                             int withAlg)
{
    int    ret;
    byte   pubKey[ED25519_PUB_KEY_SIZE];
    word32 pubKeyLen = (word32)sizeof(pubKey);

    if (key == NULL) {
        return BAD_FUNC_ARG;
    }

    #if defined(HAVE_FIPS) && FIPS_VERSION3_LT(7,0,0)
    ret = wc_ed25519_export_public((ed25519_key *)key, pubKey, &pubKeyLen);
    #else
    ret = wc_ed25519_export_public(key, pubKey, &pubKeyLen);
    #endif
    if (ret == 0) {
        ret = SetAsymKeyDerPublic(pubKey, pubKeyLen, output, inLen,
            ED25519k, withAlg);
    }
    return ret;
}
#endif /* HAVE_ED25519 && HAVE_ED25519_KEY_EXPORT */

#if defined(HAVE_ED448) && defined(HAVE_ED448_KEY_EXPORT)
/* Encode the public part of an Ed448 key in DER.
 *
 * Pass NULL for output to get the size of the encoding.
 *
 * @param [in]  key       Ed448 key object.
 * @param [out] output    Buffer to put encoded data in.
 * @param [in]  outLen    Size of buffer in bytes.
 * @param [in]  withAlg   Whether to use SubjectPublicKeyInfo format.
 * @return  Size of encoded data in bytes on success.
 * @return  BAD_FUNC_ARG when key is NULL.
 * @return  MEMORY_E when dynamic memory allocation failed.
 */
int wc_Ed448PublicKeyToDer(ed448_key* key, byte* output, word32 inLen,
                           int withAlg)
{
    int    ret;
    byte   pubKey[ED448_PUB_KEY_SIZE];
    word32 pubKeyLen = (word32)sizeof(pubKey);

    if (key == NULL) {
        return BAD_FUNC_ARG;
    }

    ret = wc_ed448_export_public(key, pubKey, &pubKeyLen);
    if (ret == 0) {
        ret = SetAsymKeyDerPublic(pubKey, pubKeyLen, output, inLen,
            ED448k, withAlg);
    }
    return ret;
}
#endif /* HAVE_ED448 && HAVE_ED448_KEY_EXPORT */
#if !defined(NO_RSA) && !defined(NO_CERTS)
#ifdef WOLFSSL_ASN_TEMPLATE
/* ASN.1 template for header before RSA key in certificate. */
static const ASNItem rsaCertKeyASN[] = {
/* STR */ { 0, ASN_BIT_STRING, 0, 1, 0 },
/* SEQ */     { 1, ASN_SEQUENCE, 1, 0, 0 },
};
enum {
    RSACERTKEYASN_IDX_STR = 0,
    RSACERTKEYASN_IDX_SEQ
};

/* Number of items in ASN.1 template for header before RSA key in cert. */
#define rsaCertKeyASN_Length (sizeof(rsaCertKeyASN) / sizeof(ASNItem))
#endif

/* Store RSA key pointer and length in certificate object.
 *
 * @param [in, out] cert    Certificate object.
 * @param [in]      source  Buffer containing encoded key.
 * @param [in, out] srcIdx  On in, start of RSA key data.
 *                          On out, start of element after RSA key data.
 * @param [in]      maxIdx  Maximum index of key data.
 * @return  0 on success.
 * @return  ASN_PARSE_E when BER encoded data does not match ASN.1 items or
 *          is invalid.
 * @return  BUFFER_E when data in buffer is too small.
 * @return  ASN_BITSTR_E when the expected BIT_STRING tag is not found.
 * @return  ASN_EXPECT_0_E when the INTEGER has the MSB set or NULL has a
 *          non-zero length.
 */
static int StoreRsaKey(DecodedCert* cert, const byte* source, word32* srcIdx,
                       word32 maxIdx)
{
#ifndef WOLFSSL_ASN_TEMPLATE
    int    length;
    int    pubLen;
    word32 pubIdx;

    if (CheckBitString(source, srcIdx, &pubLen, maxIdx, 1, NULL) != 0)
        return ASN_PARSE_E;
    pubIdx = *srcIdx;

    if (GetSequence(source, srcIdx, &length, pubIdx + (word32)pubLen) < 0)
        return ASN_PARSE_E;

#if defined(WOLFSSL_RENESAS_TSIP_TLS) || defined(WOLFSSL_RENESAS_FSPSM_TLS)
    cert->sigCtx.CertAtt.pubkey_n_start =
            cert->sigCtx.CertAtt.pubkey_e_start = pubIdx;
#endif
    cert->pubKeySize = (word32)pubLen;
    cert->publicKey = source + pubIdx;
#ifdef WOLFSSL_MAXQ10XX_TLS
    cert->publicKeyIndex = pubIdx;
#endif
    *srcIdx += (word32)length;

#ifdef HAVE_OCSP
    return CalcHashId_ex(cert->publicKey, cert->pubKeySize,
        cert->subjectKeyHash, HashIdAlg(cert->signatureOID));
#else
    return 0;
#endif
#else
    ASNGetData dataASN[rsaCertKeyASN_Length];
    int ret;

    /* No dynamic data. */
    XMEMSET(dataASN, 0, sizeof(dataASN));
    /* Decode the header before the key data. */
    ret = GetASN_Items(rsaCertKeyASN, dataASN, rsaCertKeyASN_Length, 1, source,
                       srcIdx, maxIdx);
    if (ret == 0) {
        /* Store the pointer and length in certificate object starting at
         * SEQUENCE. */
        GetASN_GetConstRef(&dataASN[RSACERTKEYASN_IDX_STR],
                &cert->publicKey, &cert->pubKeySize);

#ifdef WOLFSSL_MAXQ10XX_TLS
    cert->publicKeyIndex = dataASN[RSACERTKEYASN_IDX_SEQ].offset;
#endif

    #if defined(WOLFSSL_RENESAS_TSIP_TLS) || defined(WOLFSSL_RENESAS_FSPSM_TLS)
        /* Start of SEQUENCE. */
        cert->sigCtx.CertAtt.pubkey_n_start =
            cert->sigCtx.CertAtt.pubkey_e_start = dataASN[RSACERTKEYASN_IDX_SEQ].offset;
    #endif
    #ifdef HAVE_OCSP
        /* Calculate the hash of the public key for OCSP. */
        ret = CalcHashId_ex(cert->publicKey, cert->pubKeySize,
                         cert->subjectKeyHash, HashIdAlg(cert->signatureOID));
    #endif
    }

    return ret;
#endif /* WOLFSSL_ASN_TEMPLATE */
}
#endif /* !NO_RSA && !NO_CERTS */

#if defined(HAVE_ECC) && !defined(NO_CERTS)

#ifdef WOLFSSL_ASN_TEMPLATE
/* ASN.1 template for header before ECC key in certificate. */
static const ASNItem eccCertKeyASN[] = {
/* OID        */     { 1, ASN_OBJECT_ID, 0, 0, 2 },
                            /* Algo parameters */
/* PARAMS     */     { 1, ASN_SEQUENCE, 1, 0, 2 },
                            /* Subject public key */
/* SUBJPUBKEY */ { 0, ASN_BIT_STRING, 0, 0, 0 },
};
enum {
    ECCCERTKEYASN_IDX_OID = 0,
    ECCCERTKEYASN_IDX_PARAMS,
    ECCCERTKEYASN_IDX_SUBJPUBKEY
};

/* Number of items in ASN.1 template for header before ECC key in cert. */
#define eccCertKeyASN_Length (sizeof(eccCertKeyASN) / sizeof(ASNItem))

#ifdef WOLFSSL_CUSTOM_CURVES
static int EccSpecifiedECDomainDecode(const byte* input, word32 inSz,
                                      ecc_key* key, void* heap, int* curveSz);
#endif /* WOLFSSL_CUSTOM_CURVES */
#endif /* WOLFSSL_ASN_TEMPLATE */

/* Store public ECC key in certificate object.
 *
 * Parse parameters and store public key data.
 *
 * @param [in, out] cert       Certificate object.
 * @param [in]      source     Buffer containing encoded key.
 * @param [in, out] srcIdx     On in, start of ECC key data.
 *                             On out, start of element after ECC key data.
 * @param [in]      maxIdx     Maximum index of key data.
 * @param [in]      pubKey     Buffer holding encoded public key.
 * @param [in]      pubKeyLen  Length of encoded public key in bytes.
 * @return  0 on success.
 * @return  BAD_FUNC_ARG when pubKey is NULL.
 * @return  ASN_PARSE_E when BER encoded data does not match ASN.1 items or
 *          is invalid.
 * @return  BUFFER_E when data in buffer is too small.
 * @return  ASN_UNKNOWN_OID_E when the OID cannot be verified.
 * @return  ASN_BITSTR_E when the expected BIT_STRING tag is not found.
 * @return  ASN_EXPECT_0_E when the INTEGER has the MSB set or NULL has a
 *          non-zero length.
 * @return  ASN_OBJECT_ID_E when the expected OBJECT_ID tag is not found.
 */
static int StoreEccKey(DecodedCert* cert, const byte* source, word32* srcIdx,
                       word32 maxIdx, const byte* pubKey, word32 pubKeyLen)
{
#ifndef WOLFSSL_ASN_TEMPLATE
    int ret;
    word32 localIdx;
    byte* publicKey;
    byte  tag;
    int length;

    if (pubKey == NULL) {
        return BAD_FUNC_ARG;
    }

    localIdx = *srcIdx;
    if (GetASNTag(source, &localIdx, &tag, maxIdx) < 0)
        return ASN_PARSE_E;

    if (tag != (ASN_SEQUENCE | ASN_CONSTRUCTED)) {
        if (GetObjectId(source, srcIdx, &cert->pkCurveOID, oidCurveType,
                                                                    maxIdx) < 0)
            return ASN_PARSE_E;

        if ((ret = CheckCurve(cert->pkCurveOID)) < 0)
            return ECC_CURVE_OID_E;

    #if defined(WOLFSSL_RENESAS_FSPSM_TLS) || defined(WOLFSSL_RENESAS_TSIP_TLS)
        cert->sigCtx.CertAtt.curve_id = ret;
    #else
        (void)ret;
    #endif
        /* key header */
        ret = CheckBitString(source, srcIdx, &length, maxIdx, 1, NULL);
        if (ret != 0)
            return ret;
    #if defined(WOLFSSL_RENESAS_FSPSM_TLS) || defined(WOLFSSL_RENESAS_TSIP_TLS)
        cert->sigCtx.CertAtt.pubkey_n_start =
                cert->sigCtx.CertAtt.pubkey_e_start = (*srcIdx + 1);
        cert->sigCtx.CertAtt.pubkey_n_len = ((length - 1) >> 1);
        cert->sigCtx.CertAtt.pubkey_e_start +=
                cert->sigCtx.CertAtt.pubkey_n_len;
        cert->sigCtx.CertAtt.pubkey_e_len   =
                cert->sigCtx.CertAtt.pubkey_n_len;
    #endif
    #ifdef WOLFSSL_MAXQ10XX_TLS
        cert->publicKeyIndex = *srcIdx + 1;
    #endif

    #ifdef HAVE_OCSP
        ret = CalcHashId_ex(source + *srcIdx, (word32)length,
            cert->subjectKeyHash, HashIdAlg(cert->signatureOID));
        if (ret != 0)
            return ret;
    #endif
        *srcIdx += (word32)length;
    }

    publicKey = (byte*)XMALLOC(pubKeyLen, cert->heap, DYNAMIC_TYPE_PUBLIC_KEY);
    if (publicKey == NULL)
        return MEMORY_E;
    XMEMCPY(publicKey, pubKey, pubKeyLen);
    cert->publicKey = publicKey;
    cert->pubKeyStored = 1;
    cert->pubKeySize   = pubKeyLen;

    return 0;
#else
    int ret = 0;
    DECL_ASNGETDATA(dataASN, eccCertKeyASN_Length);
    byte* publicKey;

    /* Validate parameters. */
    if (pubKey == NULL) {
        ret = BAD_FUNC_ARG;
    }

    /* Clear dynamic data and check OID is a curve. */
    CALLOC_ASNGETDATA(dataASN, eccCertKeyASN_Length, ret, cert->heap);
    if (ret == 0) {
        GetASN_OID(&dataASN[ECCCERTKEYASN_IDX_OID], oidCurveType);
        /* Parse ECC public key header. */
        ret = GetASN_Items(eccCertKeyASN, dataASN, eccCertKeyASN_Length, 1,
                source, srcIdx, maxIdx);
    }
    if (ret == 0) {
        if (dataASN[ECCCERTKEYASN_IDX_OID].tag != 0) {
            /* Store curve OID. */
            cert->pkCurveOID = dataASN[ECCCERTKEYASN_IDX_OID].data.oid.sum;
        }
        else {
    #ifdef WOLFSSL_CUSTOM_CURVES
            /* Parse explicit parameters. */
            ret = EccSpecifiedECDomainDecode(
                    dataASN[ECCCERTKEYASN_IDX_PARAMS].data.ref.data,
                    dataASN[ECCCERTKEYASN_IDX_PARAMS].data.ref.length, NULL,
                    NULL, &cert->pkCurveSize);
    #else
            /* Explicit parameters not supported in build configuration. */
            ret = ASN_PARSE_E;
    #endif
        }

    #ifdef WOLFSSL_MAXQ10XX_TLS
        cert->publicKeyIndex =
            GetASNItem_DataIdx(dataASN[ECCCERTKEYASN_IDX_SUBJPUBKEY], source)
            + 1;
    #endif

    #ifdef HAVE_OCSP
        if (ret == 0) {
            /* Calculate the hash of the subject public key for OCSP. */
            ret = CalcHashId_ex(
                    dataASN[ECCCERTKEYASN_IDX_SUBJPUBKEY].data.ref.data,
                    dataASN[ECCCERTKEYASN_IDX_SUBJPUBKEY].data.ref.length,
                    cert->subjectKeyHash, HashIdAlg(cert->signatureOID));
        }
    }
    if (ret == 0) {
    #endif
        /* Store public key data length. */
        cert->pubKeySize = pubKeyLen;
        /* Must allocated space for key.
         * Don't memcpy into constant pointer so use temp. */
        publicKey = (byte*)XMALLOC(cert->pubKeySize, cert->heap,
                                   DYNAMIC_TYPE_PUBLIC_KEY);
        if (publicKey == NULL) {
            ret = MEMORY_E;
        }
        else {
            /* Copy in whole public key and store pointer. */
            XMEMCPY(publicKey, pubKey, cert->pubKeySize);
            cert->publicKey = publicKey;
            /* Indicate publicKey needs to be freed. */
            cert->pubKeyStored = 1;
        }
    }
    FREE_ASNGETDATA(dataASN, cert->heap);

    return ret;
#endif /* WOLFSSL_ASN_TEMPLATE */
}
#endif /* HAVE_ECC && !NO_CERTS */

#ifndef NO_CERTS
#if !defined(NO_DSA)
#ifdef WOLFSSL_ASN_TEMPLATE
/* ASN.1 template for DSA key in certificate.
 * X.509: RFC 5280, 4.1 - SubjectPublicKeyInfo
 * RFC 3279, 2.3.2 - DSA in SubjectPublicKeyInfo
 */
static const ASNItem dsaCertKeyASN[] = {
/*  0 */        { 1, ASN_SEQUENCE, 1, 1, 0 },
/*  1 */            { 2, ASN_INTEGER, 0, 0, 0 },
/*  2 */            { 2, ASN_INTEGER, 0, 0, 0 },
/*  3 */            { 2, ASN_INTEGER, 0, 0, 0 },
/*  4 */    { 0, ASN_BIT_STRING, 0, 1, 0 },
/*  5 */        { 1, ASN_INTEGER, 0, 0, 0 },
};

/* Number of items in ASN.1 template for DSA key in certificate. */
#define dsaCertKeyASN_Length (sizeof(dsaCertKeyASN) / sizeof(ASNItem))
#endif /* WOLFSSL_ASN_TEMPLATE */

/* Parse DSA parameters to ensure valid.
 *
 * @param [in]      source  Buffer containing encoded key.
 * @param [in, out] srcIdx  On in, start of DSA key data.
 *                          On out, start of element after DSA key data.
 * @param [in]      maxIdx  Maximum index of key data.
 * @param [in]      heap    Dynamic memory hint.
 * @return  0 on success.
 * @return  ASN_PARSE_E when BER encoded data does not match ASN.1 items or
 *          is invalid.
 * @return  BUFFER_E when data in buffer is too small.
 * @return  ASN_BITSTR_E when the expected BIT_STRING tag is not found.
 * @return  ASN_EXPECT_0_E when the INTEGER has the MSB set or NULL has a
 *          non-zero length.
 */
static int ParseDsaKey(const byte* source, word32* srcIdx, word32 maxIdx,
                       void* heap)
{
#ifndef WOLFSSL_ASN_TEMPLATE
    int ret;
    int length;

    (void)heap;

    ret = GetSequence(source, srcIdx, &length, maxIdx);
    if (ret < 0)
        return ret;

    ret = SkipInt(source, srcIdx, maxIdx);
    if (ret != 0)
        return ret;
    ret = SkipInt(source, srcIdx, maxIdx);
    if (ret != 0)
        return ret;
    ret = SkipInt(source, srcIdx, maxIdx);
    if (ret != 0)
        return ret;

    ret = CheckBitString(source, srcIdx, &length, maxIdx, 1, NULL);
    if (ret != 0)
        return ret;

    ret = GetASNInt(source, srcIdx, &length, maxIdx);
    if (ret != 0)
        return ASN_PARSE_E;

    *srcIdx += (word32)length;

    return 0;
#else
    DECL_ASNGETDATA(dataASN, dsaCertKeyASN_Length);
    int ret = 0;

    (void)heap;

    CALLOC_ASNGETDATA(dataASN, dsaCertKeyASN_Length, ret, heap);
    if (ret == 0) {
        /* Parse the DSA key data to ensure valid. */
        ret = GetASN_Items(dsaCertKeyASN, dataASN, dsaCertKeyASN_Length, 1,
                           source, srcIdx, maxIdx);
    }

    FREE_ASNGETDATA(dataASN, heap);
    return ret;
#endif /* WOLFSSL_ASN_TEMPLATE */
}
#endif /* !NO_DSA */

/* Decode the SubjectPublicKeyInfo block in a certificate.
 *
 * Stores the public key in fields of the certificate object.
 * Validates the BER/DER items and does not store in a key object.
 *
 * @param [in, out] cert      Decoded certificate object.
 * @param [in]      source    BER/DER encoded SubjectPublicKeyInfo block.
 * @param [in, out] inOutIdx  On in, start of public key.
 *                            On out, start of ASN.1 item after public key.
 * @param [in]      maxIdx    Maximum index of key data.
 * @return  0 on success.
 * @return  ASN_PARSE_E when BER encoded data does not match ASN.1 items or
 *          is invalid.
 * @return  BUFFER_E when data in buffer is too small.
 */
static int GetCertKey(DecodedCert* cert, const byte* source, word32* inOutIdx,
                      word32 maxIdx)
{
    word32 srcIdx = *inOutIdx;
#if defined(HAVE_ECC) || !defined(NO_DSA)
    int pubLen;
#endif
#if defined(HAVE_ECC) || !defined(NO_DSA)
    int pubIdx = (int)srcIdx;
#endif
    int ret = 0;
    int length;

    /* Validate parameters. */
    if (source == NULL) {
        return ASN_PARSE_E;
    }

#ifndef WOLFSSL_ASN_TEMPLATE
    if (GetSequence(source, &srcIdx, &length, maxIdx) < 0)
#else
    /* Get SEQUENCE and expect all data to be accounted for. */
    if (GetASN_Sequence(source, &srcIdx, &length, maxIdx, 1) != 0)
#endif
    {
        return ASN_PARSE_E;
    }

#if defined(HAVE_ECC) || !defined(NO_DSA)
    pubLen = (int)srcIdx - pubIdx + length;
#endif
    maxIdx = srcIdx + (word32)length;

    /* Decode the algorithm identifier for the key. */
    if (GetAlgoId(source, &srcIdx, &cert->keyOID, oidKeyType, maxIdx) < 0) {
        return ASN_PARSE_E;
    }

    (void)length;

    /* Parse each type of public key. */
    switch (cert->keyOID) {
#ifndef NO_RSA
    #ifdef WC_RSA_PSS
        case RSAPSSk:
            if (srcIdx != maxIdx &&
                          source[srcIdx] == (ASN_SEQUENCE | ASN_CONSTRUCTED)) {
                word32 seqIdx = srcIdx;
                int seqLen;
                /* Not set when -1. */
                enum wc_HashType hash = WC_HASH_TYPE_NONE;
                int mgf = -1;
                int saltLen = 0;
                /* Defaults for sig algorithm parameters. */
                enum wc_HashType sigHash = WC_HASH_TYPE_SHA;
                int sigMgf = WC_MGF1SHA1;
                int sigSaltLen = 20;

                if (GetSequence(source, &srcIdx, &seqLen, maxIdx) < 0) {
                    return ASN_PARSE_E;
                }
                /* Get the pubic key parameters. */
                ret = DecodeRsaPssParams(source + seqIdx,
                    (word32)seqLen + srcIdx - seqIdx, &hash, &mgf, &saltLen);
                if (ret != 0) {
                    return ASN_PARSE_E;
                }
                /* Get the signature parameters. */
                ret = DecodeRsaPssParams(source + cert->sigParamsIndex,
                    cert->sigParamsLength, &sigHash, &sigMgf, &sigSaltLen);
                if (ret != 0) {
                    return ASN_PARSE_E;
                }
                /* Validated signature params match public key params. */
                if (hash != WC_HASH_TYPE_NONE && hash != sigHash) {
                    WOLFSSL_MSG("RSA PSS: hash not matching signature hash");
                    return ASN_PARSE_E;
                }
                if (mgf != -1 && mgf != sigMgf) {
                    WOLFSSL_MSG("RSA PSS: MGF not matching signature MGF");
                    return ASN_PARSE_E;
                }
                if (saltLen > sigSaltLen) {
                    WOLFSSL_MSG("RSA PSS: sig salt length too small");
                    return ASN_PARSE_E;
                }
                srcIdx += (word32)seqLen;
            }
            FALL_THROUGH;
    #endif /* WC_RSA_PSS */
        case RSAk:
            ret = StoreRsaKey(cert, source, &srcIdx, maxIdx);
            break;
#endif /* NO_RSA */
    #ifdef HAVE_ECC
    #if defined(WOLFSSL_SM2) && defined(WOLFSSL_SM3)
        case SM2k:
    #endif
        case ECDSAk:
            ret = StoreEccKey(cert, source, &srcIdx, maxIdx, source + pubIdx,
                              (word32)pubLen);
            break;
    #endif /* HAVE_ECC */
    #ifdef HAVE_ED25519
        case ED25519k:
            cert->pkCurveOID = ED25519k;
            ret = StoreKey(cert, source, &srcIdx, maxIdx);
            break;
    #endif /* HAVE_ED25519 */
    #ifdef HAVE_ED448
        case ED448k:
            cert->pkCurveOID = ED448k;
            ret = StoreKey(cert, source, &srcIdx, maxIdx);
            break;
    #endif /* HAVE_ED448 */
    #ifdef HAVE_FALCON
        case FALCON_LEVEL1k:
            cert->pkCurveOID = FALCON_LEVEL1k;
            ret = StoreKey(cert, source, &srcIdx, maxIdx);
            break;
        case FALCON_LEVEL5k:
            cert->pkCurveOID = FALCON_LEVEL5k;
            ret = StoreKey(cert, source, &srcIdx, maxIdx);
            break;
    #endif /* HAVE_FALCON */
    #ifdef HAVE_DILITHIUM
        #ifdef WOLFSSL_DILITHIUM_FIPS204_DRAFT
        case DILITHIUM_LEVEL2k:
        case DILITHIUM_LEVEL3k:
        case DILITHIUM_LEVEL5k:
        #endif
        case ML_DSA_LEVEL2k:
        case ML_DSA_LEVEL3k:
        case ML_DSA_LEVEL5k:
            cert->pkCurveOID = cert->keyOID;
            ret = StoreKey(cert, source, &srcIdx, maxIdx);
            break;
    #endif /* HAVE_DILITHIUM */
    #ifdef HAVE_SPHINCS
        case SPHINCS_FAST_LEVEL1k:
            cert->pkCurveOID = SPHINCS_FAST_LEVEL1k;
            ret = StoreKey(cert, source, &srcIdx, maxIdx);
            break;
        case SPHINCS_FAST_LEVEL3k:
            cert->pkCurveOID = SPHINCS_FAST_LEVEL3k;
            ret = StoreKey(cert, source, &srcIdx, maxIdx);
            break;
        case SPHINCS_FAST_LEVEL5k:
            cert->pkCurveOID = SPHINCS_FAST_LEVEL5k;
            ret = StoreKey(cert, source, &srcIdx, maxIdx);
            break;
        case SPHINCS_SMALL_LEVEL1k:
            cert->pkCurveOID = SPHINCS_SMALL_LEVEL1k;
            ret = StoreKey(cert, source, &srcIdx, maxIdx);
            break;
        case SPHINCS_SMALL_LEVEL3k:
            cert->pkCurveOID = SPHINCS_SMALL_LEVEL3k;
            ret = StoreKey(cert, source, &srcIdx, maxIdx);
            break;
        case SPHINCS_SMALL_LEVEL5k:
            cert->pkCurveOID = SPHINCS_SMALL_LEVEL5k;
            ret = StoreKey(cert, source, &srcIdx, maxIdx);
            break;
    #endif /* HAVE_SPHINCS */
    #ifndef NO_DSA
        case DSAk:
            cert->publicKey = source + pubIdx;
            cert->pubKeySize = (word32)pubLen;
            ret = ParseDsaKey(source, &srcIdx, maxIdx, cert->heap);
            break;
    #endif /* NO_DSA */
        default:
            WOLFSSL_MSG("Unknown or not compiled in key OID");
            WOLFSSL_ERROR_VERBOSE(ASN_UNKNOWN_OID_E);
            ret = ASN_UNKNOWN_OID_E;
    }

    /* Return index after public key. */
    *inOutIdx = srcIdx;

    /* Return error code. */
    return ret;
}
#endif

/* Return the hash algorithm to use with the signature algorithm.
 *
 * @param [in] oidSum  Signature id.
 * @return  Hash algorithm id.
 */
int HashIdAlg(word32 oidSum)
{
    (void)oidSum;

#if defined(WOLFSSL_SM2) && defined(WOLFSSL_SM3)
    if (oidSum == CTC_SM3wSM2) {
        return WC_SM3;
    }
    if (oidSum == SM2k) {
        return WC_SM3;
    }
#endif
#if defined(NO_SHA) || (!defined(NO_SHA256) && defined(WC_ASN_HASH_SHA256))
    return WC_SHA256;
#else
    return WC_SHA;
#endif
}

/* Calculate hash of the id using the SHA-1 or SHA-256.
 *
 * @param [in]  data  Data to hash.
 * @param [in]  len   Length of data to hash.
 * @param [out] hash  Buffer to hold hash.
 * @return  0 on success.
 * @return  MEMORY_E when dynamic memory allocation fails.
 */
int CalcHashId(const byte* data, word32 len, byte* hash)
{
    /* Use default hash algorithm. */
    return CalcHashId_ex(data, len, hash,
#if defined(NO_SHA) || (!defined(NO_SHA256) && defined(WC_ASN_HASH_SHA256))
        WC_SHA256
#else
        WC_SHA
#endif
        );
}

/* Calculate hash of the id using the SHA-1 or SHA-256.
 *
 * @param [in]  data  Data to hash.
 * @param [in]  len   Length of data to hash.
 * @param [out] hash  Buffer to hold hash.
 * @return  0 on success.
 * @return  MEMORY_E when dynamic memory allocation fails.
 */
int CalcHashId_ex(const byte* data, word32 len, byte* hash, int hashAlg)
{
    int ret;

#if defined(WOLFSSL_SM2) && defined(WOLFSSL_SM3)
    if (hashAlg == WC_SM3) {
        ret = wc_Sm3Hash(data, len, hash);
    }
    else
#endif
#if defined(NO_SHA) || (!defined(NO_SHA256) && defined(WC_ASN_HASH_SHA256))
    if (hashAlg == WC_SHA256) {
        ret = wc_Sha256Hash(data, len, hash);
    }
    else
#elif !defined(NO_SHA)
    if (hashAlg == WC_SHA) {
    #if defined(WOLFSSL_SM2) && defined(WOLFSSL_SM3)
        XMEMSET(hash + WC_SHA_DIGEST_SIZE, 0, KEYID_SIZE - WC_SHA_DIGEST_SIZE);
    #endif
        ret = wc_ShaHash(data, len, hash);
    }
    else
#else
    (void)data;
    (void)len;
    (void)hash;
#endif
    {
        ret = NOT_COMPILED_IN;
    }

    return ret;
}

#ifndef NO_CERTS
/* Get the hash of the id using the SHA-1 or SHA-256.
 *
 * If the id is not the length of the hash, then hash it.
 *
 * @param [in]  id    Id to get hash for.
 * @param [in]  len   Length of id in bytes.
 * @param [out] hash  Buffer to hold hash.
 * @return  0 on success.
 * @return  MEMORY_E when dynamic memory allocation fails.
 */
int GetHashId(const byte* id, int length, byte* hash, int hashAlg)
{
    int ret;

#if defined(WOLFSSL_SM2) && defined(WOLFSSL_SM3)
    if (length == wc_HashGetDigestSize(wc_HashTypeConvert(hashAlg)))
#else
    if (length == KEYID_SIZE)
#endif
    {
    #if defined(WOLFSSL_SM2) && defined(WOLFSSL_SM3)
        XMEMSET(hash + length, 0, KEYID_SIZE - length);
    #endif
        XMEMCPY(hash, id, (size_t)length);
        ret = 0;
    }
    else {
        ret = CalcHashId_ex(id, (word32)length, hash, hashAlg);
    }

    return ret;
}
#endif /* !NO_CERTS */

#ifdef WOLFSSL_ASN_TEMPLATE
/* Id for email address. */
#define ASN_EMAIL     0x100
/* Id for domain component. */
#define ASN_DC        0x102
/* Id for jurisdiction country. */
#define ASN_JURIS_C   0x203
/* Id for jurisdiction state. */
#define ASN_JURIS_ST  0x202

/* Set the string for a name component into the subject name. */
#define SetCertNameSubject(cert, id, val) \
    *((char**)(((byte *)(cert)) + certNameSubject[(id) - 3].data)) = (val)
/* Set the string length for a name component into the subject name. */
#define SetCertNameSubjectLen(cert, id, val) \
    *((int*)(((byte *)(cert)) + certNameSubject[(id) - 3].len)) = (int)(val)
/* Set the encoding for a name component into the subject name. */
#define SetCertNameSubjectEnc(cert, id, val) \
    *((byte*)(((byte *)(cert)) + certNameSubject[(id) - 3].enc)) = (val)

/* Get the string of a name component from the subject name. */
#ifdef WOLFSSL_NAMES_STATIC
    #define GetCertNameSubjectStr(id) \
        ((certNameSubject[(id) - 3].strLen) ? \
         (certNameSubject[(id) - 3].str) : \
         NULL)
#else
    #define GetCertNameSubjectStr(id) \
        (certNameSubject[(id) - 3].str)
#endif
/* Get the string length of a name component from the subject name. */
#define GetCertNameSubjectStrLen(id) \
    (certNameSubject[(id) - 3].strLen)
/* Get the NID of a name component from the subject name. */
#define GetCertNameSubjectNID(id) \
    (certNameSubject[(id) - 3].nid)

#define ValidCertNameSubject(id) \
    (((id) - 3) >= 0 && ((id) - 3) < certNameSubjectSz && \
            (certNameSubject[(id) - 3].strLen > 0))

/* Set the string for a name component into the issuer name. */
#define SetCertNameIssuer(cert, id, val) \
    *((char**)(((byte *)(cert)) + certNameSubject[(id) - 3].dataI)) = (val)
/* Set the string length for a name component into the issuer name. */
#define SetCertNameIssuerLen(cert, id, val) \
    *((int*)(((byte *)(cert)) + certNameSubject[(id) - 3].lenI)) = (int)(val)
/* Set the encoding for a name component into the issuer name. */
#define SetCertNameIssuerEnc(cert, id, val) \
    *((byte*)(((byte *)(cert)) + certNameSubject[(id) - 3].encI)) = (val)


/* Mapping of certificate name component to useful information. */
typedef struct CertNameData {
    /* Type string of name component. */
#ifdef WOLFSSL_NAMES_STATIC
    const char str[20]; /* large enough for largest string in certNameSubject[]
                         * below
                         */
    #define EMPTY_STR { 0 }
#else
    const char* str;
    #define EMPTY_STR NULL
#endif
    /* Length of type string of name component. */
    byte        strLen;
#if defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_CERT_EXT)
    /* Offset of data in subject name component. */
    size_t      data;
    /* Offset of length in subject name component. */
    size_t      len;
    /* Offset of encoding in subject name component. */
    size_t      enc;
#ifdef WOLFSSL_HAVE_ISSUER_NAMES
    /* Offset of data in subject name component. */
    size_t      dataI;
    /* Offset of length in subject name component. */
    size_t      lenI;
    /* Offset of encoding in subject name component. */
    size_t      encI;
#endif
#endif
#ifdef WOLFSSL_X509_NAME_AVAILABLE
    /* NID of type for subject name component. */
    int         nid;
#endif
} CertNameData;

/* List of data for common name components. */
static const CertNameData certNameSubject[] = {
    /* Common Name */
    {
        "/CN=", 4,
#if defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_CERT_EXT)
        WC_OFFSETOF(DecodedCert, subjectCN),
        WC_OFFSETOF(DecodedCert, subjectCNLen),
        WC_OFFSETOF(DecodedCert, subjectCNEnc),
#ifdef WOLFSSL_HAVE_ISSUER_NAMES
        WC_OFFSETOF(DecodedCert, issuerCN),
        WC_OFFSETOF(DecodedCert, issuerCNLen),
        WC_OFFSETOF(DecodedCert, issuerCNEnc),
#endif
#endif
#ifdef WOLFSSL_X509_NAME_AVAILABLE
        WC_NID_commonName
#endif
    },
    /* Surname */
    {
        "/SN=", 4,
#if defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_CERT_EXT)
        WC_OFFSETOF(DecodedCert, subjectSN),
        WC_OFFSETOF(DecodedCert, subjectSNLen),
        WC_OFFSETOF(DecodedCert, subjectSNEnc),
#ifdef WOLFSSL_HAVE_ISSUER_NAMES
        WC_OFFSETOF(DecodedCert, issuerSN),
        WC_OFFSETOF(DecodedCert, issuerSNLen),
        WC_OFFSETOF(DecodedCert, issuerSNEnc),
#endif
#endif
#ifdef WOLFSSL_X509_NAME_AVAILABLE
        WC_NID_surname
#endif
    },
    /* Serial Number */
    {
        "/serialNumber=", 14,
#if defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_CERT_EXT)
        WC_OFFSETOF(DecodedCert, subjectSND),
        WC_OFFSETOF(DecodedCert, subjectSNDLen),
        WC_OFFSETOF(DecodedCert, subjectSNDEnc),
#ifdef WOLFSSL_HAVE_ISSUER_NAMES
        WC_OFFSETOF(DecodedCert, issuerSND),
        WC_OFFSETOF(DecodedCert, issuerSNDLen),
        WC_OFFSETOF(DecodedCert, issuerSNDEnc),
#endif
#endif
#ifdef WOLFSSL_X509_NAME_AVAILABLE
        WC_NID_serialNumber
#endif
    },
    /* Country Name */
    {
        "/C=", 3,
#if defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_CERT_EXT)
        WC_OFFSETOF(DecodedCert, subjectC),
        WC_OFFSETOF(DecodedCert, subjectCLen),
        WC_OFFSETOF(DecodedCert, subjectCEnc),
#ifdef WOLFSSL_HAVE_ISSUER_NAMES
        WC_OFFSETOF(DecodedCert, issuerC),
        WC_OFFSETOF(DecodedCert, issuerCLen),
        WC_OFFSETOF(DecodedCert, issuerCEnc),
#endif
#endif
#ifdef WOLFSSL_X509_NAME_AVAILABLE
        WC_NID_countryName
#endif
    },
    /* Locality Name */
    {
        "/L=", 3,
#if defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_CERT_EXT)
        WC_OFFSETOF(DecodedCert, subjectL),
        WC_OFFSETOF(DecodedCert, subjectLLen),
        WC_OFFSETOF(DecodedCert, subjectLEnc),
#ifdef WOLFSSL_HAVE_ISSUER_NAMES
        WC_OFFSETOF(DecodedCert, issuerL),
        WC_OFFSETOF(DecodedCert, issuerLLen),
        WC_OFFSETOF(DecodedCert, issuerLEnc),
#endif
#endif
#ifdef WOLFSSL_X509_NAME_AVAILABLE
        WC_NID_localityName
#endif
    },
    /* State Name */
    {
        "/ST=", 4,
#if defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_CERT_EXT)
        WC_OFFSETOF(DecodedCert, subjectST),
        WC_OFFSETOF(DecodedCert, subjectSTLen),
        WC_OFFSETOF(DecodedCert, subjectSTEnc),
#ifdef WOLFSSL_HAVE_ISSUER_NAMES
        WC_OFFSETOF(DecodedCert, issuerST),
        WC_OFFSETOF(DecodedCert, issuerSTLen),
        WC_OFFSETOF(DecodedCert, issuerSTEnc),
#endif
#endif
#ifdef WOLFSSL_X509_NAME_AVAILABLE
        WC_NID_stateOrProvinceName
#endif
    },
    /* Street Address */
    {
        "/street=", 8,
#if defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_CERT_EXT)
        WC_OFFSETOF(DecodedCert, subjectStreet),
        WC_OFFSETOF(DecodedCert, subjectStreetLen),
        WC_OFFSETOF(DecodedCert, subjectStreetEnc),
#ifdef WOLFSSL_HAVE_ISSUER_NAMES
        0,
        0,
        0,
#endif
#endif
#ifdef WOLFSSL_X509_NAME_AVAILABLE
        WC_NID_streetAddress
#endif
    },
    /* Organization Name */
    {
        "/O=", 3,
#if defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_CERT_EXT)
        WC_OFFSETOF(DecodedCert, subjectO),
        WC_OFFSETOF(DecodedCert, subjectOLen),
        WC_OFFSETOF(DecodedCert, subjectOEnc),
#ifdef WOLFSSL_HAVE_ISSUER_NAMES
        WC_OFFSETOF(DecodedCert, issuerO),
        WC_OFFSETOF(DecodedCert, issuerOLen),
        WC_OFFSETOF(DecodedCert, issuerOEnc),
#endif
#endif
#ifdef WOLFSSL_X509_NAME_AVAILABLE
        WC_NID_organizationName
#endif
    },
    /* Organization Unit Name */
    {
        "/OU=", 4,
#if defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_CERT_EXT)
        WC_OFFSETOF(DecodedCert, subjectOU),
        WC_OFFSETOF(DecodedCert, subjectOULen),
        WC_OFFSETOF(DecodedCert, subjectOUEnc),
#ifdef WOLFSSL_HAVE_ISSUER_NAMES
        WC_OFFSETOF(DecodedCert, issuerOU),
        WC_OFFSETOF(DecodedCert, issuerOULen),
        WC_OFFSETOF(DecodedCert, issuerOUEnc),
#endif
#endif
#ifdef WOLFSSL_X509_NAME_AVAILABLE
        WC_NID_organizationalUnitName
#endif
    },
    /* Title */
    {
        EMPTY_STR, 0,
#if defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_CERT_EXT)
        0,
        0,
        0,
#ifdef WOLFSSL_HAVE_ISSUER_NAMES
        0,
        0,
        0,
#endif
#endif
#ifdef WOLFSSL_X509_NAME_AVAILABLE
        0,
#endif
    },
    /* Undefined */
    {
        EMPTY_STR, 0,
#if defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_CERT_EXT)
        0,
        0,
        0,
#ifdef WOLFSSL_HAVE_ISSUER_NAMES
        0,
        0,
        0,
#endif
#endif
#ifdef WOLFSSL_X509_NAME_AVAILABLE
        0,
#endif
    },
    /* Undefined */
    {
        EMPTY_STR, 0,
#if defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_CERT_EXT)
        0,
        0,
        0,
#ifdef WOLFSSL_HAVE_ISSUER_NAMES
        0,
        0,
        0,
#endif
#endif
#ifdef WOLFSSL_X509_NAME_AVAILABLE
        0,
#endif
    },
    /* Business Category */
    {
        "/businessCategory=", 18,
#if defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_CERT_EXT)
        WC_OFFSETOF(DecodedCert, subjectBC),
        WC_OFFSETOF(DecodedCert, subjectBCLen),
        WC_OFFSETOF(DecodedCert, subjectBCEnc),
#ifdef WOLFSSL_HAVE_ISSUER_NAMES
        0,
        0,
        0,
#endif
#endif
#ifdef WOLFSSL_X509_NAME_AVAILABLE
        WC_NID_businessCategory
#endif
    },
    /* Undefined */
    {
        EMPTY_STR, 0,
#if defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_CERT_EXT)
        0,
        0,
        0,
#ifdef WOLFSSL_HAVE_ISSUER_NAMES
        0,
        0,
        0,
#endif
#endif
#ifdef WOLFSSL_X509_NAME_AVAILABLE
        0,
#endif
    },
    /* Postal Code */
    {
        "/postalCode=", 12,
#if defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_CERT_EXT)
        WC_OFFSETOF(DecodedCert, subjectPC),
        WC_OFFSETOF(DecodedCert, subjectPCLen),
        WC_OFFSETOF(DecodedCert, subjectPCEnc),
#ifdef WOLFSSL_HAVE_ISSUER_NAMES
        0,
        0,
        0,
#endif
#endif
#ifdef WOLFSSL_X509_NAME_AVAILABLE
        WC_NID_postalCode
#endif
    },
    /* User Id */
    {
        "/userid=", 8,
#if defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_CERT_EXT)
        WC_OFFSETOF(DecodedCert, subjectUID),
        WC_OFFSETOF(DecodedCert, subjectUIDLen),
        WC_OFFSETOF(DecodedCert, subjectUIDEnc),
#ifdef WOLFSSL_HAVE_ISSUER_NAMES
        0,
        0,
        0,
#endif
#endif
#ifdef WOLFSSL_X509_NAME_AVAILABLE
        WC_NID_userId
#endif
    },
#ifdef WOLFSSL_CERT_NAME_ALL
    /* Name, id 41 */
    {
        "/N=", 3,
    #if defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_CERT_EXT)
        WC_OFFSETOF(DecodedCert, subjectN),
        WC_OFFSETOF(DecodedCert, subjectNLen),
        WC_OFFSETOF(DecodedCert, subjectNEnc),
#ifdef WOLFSSL_HAVE_ISSUER_NAMES
        0,
        0,
        0,
#endif
    #endif
    #ifdef WOLFSSL_X509_NAME_AVAILABLE
        WC_NID_name
    #endif
    },
    /* Given Name, id 42 */
    {
        "/GN=", 4,
    #if defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_CERT_EXT)
        WC_OFFSETOF(DecodedCert, subjectGN),
        WC_OFFSETOF(DecodedCert, subjectGNLen),
        WC_OFFSETOF(DecodedCert, subjectGNEnc),
#ifdef WOLFSSL_HAVE_ISSUER_NAMES
        0,
        0,
        0,
#endif
    #endif
    #ifdef WOLFSSL_X509_NAME_AVAILABLE
        WC_NID_givenName
    #endif
    },
    /* initials, id 43 */
    {
        "/initials=", 10,
    #if defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_CERT_EXT)
        WC_OFFSETOF(DecodedCert, subjectI),
        WC_OFFSETOF(DecodedCert, subjectILen),
        WC_OFFSETOF(DecodedCert, subjectIEnc),
#ifdef WOLFSSL_HAVE_ISSUER_NAMES
        0,
        0,
        0,
#endif
    #endif
    #ifdef WOLFSSL_X509_NAME_AVAILABLE
        WC_NID_initials
    #endif
    },
    /* DN Qualifier Name, id 46 */
    {
        "/dnQualifier=", 13,
    #if defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_CERT_EXT)
        WC_OFFSETOF(DecodedCert, subjectDNQ),
        WC_OFFSETOF(DecodedCert, subjectDNQLen),
        WC_OFFSETOF(DecodedCert, subjectDNQEnc),
#ifdef WOLFSSL_HAVE_ISSUER_NAMES
        0,
        0,
        0,
#endif
    #endif
    #ifdef WOLFSSL_X509_NAME_AVAILABLE
        WC_NID_dnQualifier
    #endif
    },
#endif /* WOLFSSL_CERT_NAME_ALL */
};

static const int certNameSubjectSz =
        (int) (sizeof(certNameSubject) / sizeof(CertNameData));


/* ASN.1 template for an RDN.
 * X.509: RFC 5280, 4.1.2.4 - RelativeDistinguishedName
 */
static const ASNItem rdnASN[] = {
/* SET       */ { 1, ASN_SET, 1, 1, 0 },
                           /* AttributeTypeAndValue */
/* ATTR_SEQ  */     { 2, ASN_SEQUENCE, 1, 1, 0 },
                                   /* AttributeType */
/* ATTR_TYPE */         { 3, ASN_OBJECT_ID, 0, 0, 0 },
                           /* AttributeValue: Choice of tags - rdnChoice. */
/* ATTR_VAL  */         { 3, 0, 0, 0, 0 },
};
enum {
    RDNASN_IDX_SET = 0,
    RDNASN_IDX_ATTR_SEQ,
    RDNASN_IDX_ATTR_TYPE,
    RDNASN_IDX_ATTR_VAL
};

/* Number of items in ASN.1 template for an RDN. */
#define rdnASN_Length (sizeof(rdnASN) / sizeof(ASNItem))

/* Supported types of encodings (tags) for RDN strings.
 * X.509: RFC 5280, 4.1.2.4 - DirectoryString
 * (IA5 String not listed in RFC but required for alternative types)
 */
static const byte rdnChoice[] = {
    ASN_PRINTABLE_STRING, ASN_IA5_STRING, ASN_UTF8STRING, ASN_T61STRING,
    ASN_UNIVERSALSTRING, ASN_BMPSTRING, 0
};
#endif

#ifdef WOLFSSL_IP_ALT_NAME
/* used to set the human readable string for the IP address with a ASN_IP_TYPE
 * DNS entry
 * return 0 on success
 */
static int GenerateDNSEntryIPString(DNS_entry* entry, void* heap)
{
    int ret = 0;
    size_t nameSz = 0;
    char tmpName[WOLFSSL_MAX_IPSTR];
    unsigned char* ip;

    if (entry == NULL || entry->type != ASN_IP_TYPE) {
        return BAD_FUNC_ARG;
    }

    if (entry->len != WOLFSSL_IP4_ADDR_LEN &&
            entry->len != WOLFSSL_IP6_ADDR_LEN) {
        WOLFSSL_MSG("Unexpected IP size");
        return BAD_FUNC_ARG;
    }
    ip = (unsigned char*)entry->name;

    XMEMSET(tmpName, 0, sizeof(tmpName));

    /* store IP addresses as a string */
    if (entry->len == WOLFSSL_IP4_ADDR_LEN) {
        if (XSNPRINTF(tmpName, sizeof(tmpName), "%u.%u.%u.%u", 0xFFU & ip[0],
                      0xFFU & ip[1], 0xFFU & ip[2], 0xFFU & ip[3])
            >= (int)sizeof(tmpName))
        {
            WOLFSSL_MSG("IP buffer overrun");
            return BUFFER_E;
        }
    }

    if (entry->len == WOLFSSL_IP6_ADDR_LEN) {
        size_t i;
        for (i = 0; i < 8; i++) {
            if (XSNPRINTF(tmpName + i * 5, sizeof(tmpName) - i * 5,
                    "%02X%02X%s", 0xFF & ip[2 * i], 0xFF & ip[2 * i + 1],
                    (i < 7) ? ":" : "")
                >= (int)sizeof(tmpName))
            {
                WOLFSSL_MSG("IPv6 buffer overrun");
                return BUFFER_E;
            }
        }
    }

    nameSz = XSTRLEN(tmpName);
    entry->ipString = (char*)XMALLOC(nameSz + 1, heap,
        DYNAMIC_TYPE_ALTNAME);
    if (entry->ipString == NULL) {
        ret = MEMORY_E;
    }

    if (ret == 0) {
        XMEMCPY(entry->ipString, tmpName, nameSz);
        entry->ipString[nameSz] = '\0';
    }

    (void)heap;

    return ret;
}
#endif /* WOLFSSL_IP_ALT_NAME */

#ifdef WOLFSSL_RID_ALT_NAME
/* used to set the human readable string for the registeredID with an
 * ASN_RID_TYPE DNS entry
 * return 0 on success
 */
static int GenerateDNSEntryRIDString(DNS_entry* entry, void* heap)
{
    int i, j, ret   = 0;
    word16 nameSz   = 0;
#if !defined(WOLFCRYPT_ONLY) && defined(OPENSSL_EXTRA)
    int nid         = 0;
#endif
    int tmpSize     = MAX_OID_SZ;
    word32 oid      = 0;
    word32 idx      = 0;
    word16 tmpName[MAX_OID_SZ];
    char   oidName[MAX_OID_SZ];
    char*  finalName = NULL;

    if (entry == NULL || entry->type != ASN_RID_TYPE) {
        return BAD_FUNC_ARG;
    }

    if (entry->len <= 0) {
        return BAD_FUNC_ARG;
    }

    XMEMSET(&oidName, 0, MAX_OID_SZ);

    ret = GetOID((const byte*)entry->name, &idx, &oid, oidIgnoreType,
                 entry->len);
    if (ret == 0) {
    #if !defined(WOLFCRYPT_ONLY) && defined(OPENSSL_EXTRA)
        if ((nid = oid2nid(oid, oidCsrAttrType)) > 0) {
            /* OID has known string value */
            finalName = (char*)wolfSSL_OBJ_nid2ln(nid);
        }
        else
    #endif
        {
            /* Decode OBJECT_ID into dotted form array. */
            ret = DecodeObjectId((const byte*)(entry->name),(word32)entry->len,
                    tmpName, (word32*)&tmpSize);

            if (ret == 0) {
                j = 0;
                /* Append each number of dotted form. */
                for (i = 0; i < tmpSize; i++) {
                    if (j >= MAX_OID_SZ) {
                        return BUFFER_E;
                    }

                    if (i < tmpSize - 1) {
                        ret = XSNPRINTF(oidName + j, (word32)(MAX_OID_SZ - j),
                            "%d.", tmpName[i]);
                    }
                    else {
                        ret = XSNPRINTF(oidName + j, (word32)(MAX_OID_SZ - j),
                            "%d", tmpName[i]);
                    }

                    if (ret >= 0) {
                        j += ret;
                    }
                    else {
                        return BUFFER_E;
                    }
                }
                ret = 0;
                finalName = oidName;
            }
        }
    }

    if (ret == 0) {
        nameSz = (word16)XSTRLEN((const char*)finalName);
        if (nameSz > MAX_OID_SZ) {
            return BUFFER_E;
        }

        entry->ridString = (char*)XMALLOC((word32)(nameSz + 1), heap,
                DYNAMIC_TYPE_ALTNAME);

        if (entry->ridString == NULL) {
            ret = MEMORY_E;
        }

        if (ret == 0) {
            XMEMCPY(entry->ridString, finalName, (word32)(nameSz + 1));
        }
    }

    return ret;
}
#endif /* WOLFSSL_RID_ALT_NAME */

#ifdef WOLFSSL_ASN_TEMPLATE

#if defined(WOLFSSL_CERT_GEN) || !defined(NO_CERTS)

/* Adds a DNS entry to a list of DNS entries
 *
 * @param [in, out] lst      Linked list of DNS name entries.
 * @param [in]      entry    Entry to add to the list
 * @return  0 on success.
 */
static int AddDNSEntryToList(DNS_entry** lst, DNS_entry* entry)
{
#if defined(OPENSSL_EXTRA) && !defined(WOLFSSL_ALT_NAMES_NO_REV)
    entry->next = NULL;
    if (*lst == NULL) {
        /* First on list */
        *lst = entry;
    }
    else {
        DNS_entry* temp = *lst;

        /* Find end */
        for (; (temp->next != NULL); temp = temp->next);

        /* Add to end */
        temp->next = entry;
    }
#else
    /* Prepend entry to linked list. */
    entry->next = *lst;
    *lst = entry;
#endif

    return 0;
}


/* Allocate a DNS entry and set the fields.
 *
 * @param [in]      heap     Heap hint.
 * @param [in]      str      DNS name string.
 * @param [in]      strLen   Length of DNS name string.
 * @param [in]      type     Type of DNS name string.
 * @param [in, out] entries  Linked list of DNS name entries.
 * @return  0 on success.
 * @return  MEMORY_E when dynamic memory allocation fails.
 */
static int SetDNSEntry(void* heap, const char* str, int strLen,
                       int type, DNS_entry** entries)
{
    DNS_entry* dnsEntry;
    int ret = 0;

    /* TODO: consider one malloc. */
    /* Allocate DNS Entry object. */
    dnsEntry = AltNameNew(heap);
    if (dnsEntry == NULL) {
        ret = MEMORY_E;
    }
    if (ret == 0) {
        /* Allocate DNS Entry name - length of string plus 1 for NUL. */
        dnsEntry->name = (char*)XMALLOC((size_t)strLen + 1, heap,
                                                          DYNAMIC_TYPE_ALTNAME);
        if (dnsEntry->name == NULL) {
            ret = MEMORY_E;
        }
    }
    if (ret == 0) {
        /* Set tag type, name length, name and NUL terminate name. */
        dnsEntry->type = type;
        dnsEntry->len = strLen;
        XMEMCPY(dnsEntry->name, str, (size_t)strLen);
        dnsEntry->name[strLen] = '\0';

#ifdef WOLFSSL_RID_ALT_NAME
        /* store registeredID as a string */
        if (type == ASN_RID_TYPE)
            ret = GenerateDNSEntryRIDString(dnsEntry, heap);
#endif
    }
#ifdef WOLFSSL_IP_ALT_NAME
    /* store IP addresses as a string */
    if (ret == 0 && type == ASN_IP_TYPE)
        ret = GenerateDNSEntryIPString(dnsEntry, heap);
#endif
    if (ret == 0) {
        ret = AddDNSEntryToList(entries, dnsEntry);
    }

    /* failure cleanup */
    if (ret != 0 && dnsEntry != NULL) {
        XFREE(dnsEntry->name, heap, DYNAMIC_TYPE_ALTNAME);
        XFREE(dnsEntry, heap, DYNAMIC_TYPE_ALTNAME);
    }

    return ret;
}
#endif

/* Set the details of a subject name component into a certificate.
 *
 * @param [in, out] cert    Certificate object.
 * @param [in]      id      Id of component.
 * @param [in]      str     String for component.
 * @param [in]      strLen  Length of string.
 * @param [in]      tag     BER tag representing encoding of string.
 * @return  0 on success, negative values on failure.
 */
static int SetSubject(DecodedCert* cert, int id, byte* str, int strLen,
                      byte tag)
{
    int ret = 0;

    /* Put string and encoding into certificate. */
    if (id == ASN_COMMON_NAME) {
        cert->subjectCN = (char *)str;
        cert->subjectCNLen = (int)strLen;
        cert->subjectCNEnc = (char)tag;
    }
#if defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_CERT_EXT)
    else if (id > ASN_COMMON_NAME && id <= ASN_USER_ID) {
        /* Use table and offsets to put data into appropriate fields. */
        SetCertNameSubject(cert, id, (char*)str);
        SetCertNameSubjectLen(cert, id, strLen);
        SetCertNameSubjectEnc(cert, id, tag);
    }
#endif
#if !defined(IGNORE_NAME_CONSTRAINTS) || \
     defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_CERT_EXT)
    else if (id == ASN_EMAIL) {
        cert->subjectEmail = (char*)str;
        cert->subjectEmailLen = strLen;
    }
#endif
#ifdef WOLFSSL_CERT_EXT
    /* TODO: consider mapping id to an index and using SetCertNameSubect*(). */
    else if (id == ASN_JURIS_C) {
        cert->subjectJC = (char*)str;
        cert->subjectJCLen = strLen;
        cert->subjectJCEnc = (char)tag;
    }
    else if (id == ASN_JURIS_ST) {
        cert->subjectJS = (char*)str;
        cert->subjectJSLen = strLen;
        cert->subjectJSEnc = (char)tag;
    }
#endif

    return ret;
}

#if (defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_CERT_EXT)) && \
    defined(WOLFSSL_HAVE_ISSUER_NAMES)
/* Set the details of an issuer name component into a certificate.
 *
 * @param [in, out] cert    Certificate object.
 * @param [in]      id      Id of component.
 * @param [in]      str     String for component.
 * @param [in]      strLen  Length of string.
 * @param [in]      tag     BER tag representing encoding of string.
 * @return  0 on success, negative values on failure.
 */
static int SetIssuer(DecodedCert* cert, int id, byte* str, int strLen,
                      byte tag)
{
    int ret = 0;

    /* Put string and encoding into certificate. */
    if (id == ASN_COMMON_NAME) {
        cert->issuerCN = (char *)str;
        cert->issuerCNLen = (int)strLen;
        cert->issuerCNEnc = (char)tag;
    }
    else if (id > ASN_COMMON_NAME && id <= ASN_USER_ID) {
        /* Use table and offsets to put data into appropriate fields. */
        SetCertNameIssuer(cert, id, (char*)str);
        SetCertNameIssuerLen(cert, id, strLen);
        SetCertNameIssuerEnc(cert, id, tag);
    }
    else if (id == ASN_EMAIL) {
        cert->issuerEmail = (char*)str;
        cert->issuerEmailLen = strLen;
    }

    return ret;
}
#endif

/* Get a RelativeDistinguishedName from the encoding and put in certificate.
 *
 * @param [in, out] cert       Certificate object.
 * @param [in, out] full       Full name string. ([/<type>=<value>]*)
 * @param [in, out] idx        Index int full name to place next component.
 * @param [in, out] nid        NID of component type.
 * @param [in]      isSubject  Whether this data is for a subject name.
 * @param [in]      dataASN    Decoded data of RDN. Expected rdnASN type.
 * @return  0 on success.
 * @return  MEMORY_E when dynamic memory allocation fails.
 * @return  ASN_PARSE_E when type not supported.
 */
static int GetRDN(DecodedCert* cert, char* full, word32* idx, int* nid,
                  int isSubject, ASNGetData* dataASN)
{
    int         ret = 0;
    const char* typeStr = NULL;
    byte        typeStrLen = 0;
    byte*       oid;
    word32      oidSz;
    int         id = 0;

    (void)nid;

    /* Get name type OID from data items. */
    GetASN_OIDData(&dataASN[RDNASN_IDX_ATTR_TYPE], &oid, &oidSz);

    /* v1 name types */
    if ((oidSz == 3) && (oid[0] == 0x55) && (oid[1] == 0x04)) {
        id = oid[2];
        /* Check range of supported ids in table. */
        if (ValidCertNameSubject(id)) {
            /* Get the type string, length and NID from table. */
            typeStr = GetCertNameSubjectStr(id);
            typeStrLen = GetCertNameSubjectStrLen(id);
        #ifdef WOLFSSL_X509_NAME_AVAILABLE
            *nid = GetCertNameSubjectNID(id);
        #endif
        }
    }
    else if (oidSz == sizeof(attrEmailOid) && XMEMCMP(oid, attrEmailOid, oidSz) == 0) {
        /* Set the email id, type string, length and NID. */
        id = ASN_EMAIL;
        typeStr =  WOLFSSL_EMAIL_ADDR;
        typeStrLen = sizeof(WOLFSSL_EMAIL_ADDR) - 1;
    #ifdef WOLFSSL_X509_NAME_AVAILABLE
        *nid = WC_NID_emailAddress;
    #endif
    }
    else if (oidSz == sizeof(uidOid) && XMEMCMP(oid, uidOid, oidSz) == 0) {
        /* Set the user id, type string, length and NID. */
        id = ASN_USER_ID;
        typeStr = WOLFSSL_USER_ID;
        typeStrLen = sizeof(WOLFSSL_USER_ID) - 1;
    #ifdef WOLFSSL_X509_NAME_AVAILABLE
        *nid = WC_NID_userId;
    #endif
    }
    else if (oidSz == sizeof(dcOid) && XMEMCMP(oid, dcOid, oidSz) == 0) {
        /* Set the domain component, type string, length and NID. */
        id = ASN_DC;
        typeStr = WOLFSSL_DOMAIN_COMPONENT;
        typeStrLen = sizeof(WOLFSSL_DOMAIN_COMPONENT) - 1;
    #ifdef WOLFSSL_X509_NAME_AVAILABLE
        *nid = WC_NID_domainComponent;
    #endif
    }
    else if (oidSz == sizeof(rfc822Mlbx) && XMEMCMP(oid, rfc822Mlbx, oidSz) == 0) {
        /* Set the RFC822 mailbox, type string, length and NID. */
        id = ASN_RFC822_MAILBOX;
        typeStr = WOLFSSL_RFC822_MAILBOX;
        typeStrLen = sizeof(WOLFSSL_RFC822_MAILBOX) - 1;
    #ifdef WOLFSSL_X509_NAME_AVAILABLE
        *nid = WC_NID_rfc822Mailbox;
    #endif
    }
    else if (oidSz == sizeof(fvrtDrk) && XMEMCMP(oid, fvrtDrk, oidSz) == 0) {
        /* Set the favourite drink, type string, length and NID. */
        id = ASN_FAVOURITE_DRINK;
        typeStr = WOLFSSL_FAVOURITE_DRINK;
        typeStrLen = sizeof(WOLFSSL_FAVOURITE_DRINK) - 1;
    #ifdef WOLFSSL_X509_NAME_AVAILABLE
        *nid = WC_NID_favouriteDrink;
    #endif
    }
#ifdef WOLFSSL_CERT_REQ
    else if (oidSz == sizeof(attrPkcs9ContentTypeOid) &&
             XMEMCMP(oid, attrPkcs9ContentTypeOid, oidSz) == 0) {
        /* Set the pkcs9_contentType, type string, length and NID. */
        id = ASN_CONTENT_TYPE;
        typeStr = WOLFSSL_CONTENT_TYPE;
        typeStrLen = sizeof(WOLFSSL_CONTENT_TYPE) - 1;
    #ifdef WOLFSSL_X509_NAME_AVAILABLE
        *nid = WC_NID_pkcs9_contentType;
    #endif
    }
#endif
    /* Other OIDs that start with the same values. */
    else if (oidSz == sizeof(dcOid) && XMEMCMP(oid, dcOid, oidSz-1) == 0) {
        WOLFSSL_MSG("Unknown pilot attribute type");
        WOLFSSL_ERROR_VERBOSE(ASN_PARSE_E);
        ret = ASN_PARSE_E;
    }
    else if (oidSz == ASN_JOI_PREFIX_SZ + 1 &&
                         XMEMCMP(oid, ASN_JOI_PREFIX, ASN_JOI_PREFIX_SZ) == 0) {
        /* Set the jurisdiction id. */
        id = 0x200 + oid[ASN_JOI_PREFIX_SZ];

        /* Set the jurisdiction type string, length and NID if known. */
        if (oid[ASN_JOI_PREFIX_SZ] == ASN_JOI_C) {
            typeStr = WOLFSSL_JOI_C;
            typeStrLen = sizeof(WOLFSSL_JOI_C) - 1;
        #ifdef WOLFSSL_X509_NAME_AVAILABLE
            *nid = WC_NID_jurisdictionCountryName;
        #endif /* WOLFSSL_X509_NAME_AVAILABLE */
        }
        else if (oid[ASN_JOI_PREFIX_SZ] == ASN_JOI_ST) {
            typeStr = WOLFSSL_JOI_ST;
            typeStrLen = sizeof(WOLFSSL_JOI_ST) - 1;
        #ifdef WOLFSSL_X509_NAME_AVAILABLE
            *nid = WC_NID_jurisdictionStateOrProvinceName;
        #endif /* WOLFSSL_X509_NAME_AVAILABLE */
        }
        else {
            WOLFSSL_MSG("Unknown Jurisdiction, skipping");
        }
    }

    if ((ret == 0) && (typeStr != NULL)) {
        /* OID type to store for subject name and add to full string. */
        byte*  str;
        word32 strLen;
        byte   tag = dataASN[RDNASN_IDX_ATTR_VAL].tag;

        /* Get the string reference and length. */
        GetASN_GetRef(&dataASN[RDNASN_IDX_ATTR_VAL], &str, &strLen);

        if (isSubject) {
            /* Store subject field components. */
            ret = SetSubject(cert, id, str, (int)strLen, tag);
        }
    #if (defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_CERT_EXT)) && \
        defined(WOLFSSL_HAVE_ISSUER_NAMES)
        /* Put issuer common name string and encoding into certificate. */
        else {
            ret = SetIssuer(cert, id, str, (int)strLen, tag);
        }
    #endif
        if (ret == 0) {
            /* Check there is space for this in the full name string and
             * terminating NUL character. */
            if ((typeStrLen + strLen) < (word32)(WC_ASN_NAME_MAX - *idx))
            {
                /* Add RDN to full string. */
                XMEMCPY(&full[*idx], typeStr, typeStrLen);
                *idx += typeStrLen;
                XMEMCPY(&full[*idx], str, strLen);
                *idx += strLen;
            }
            else {
                WOLFSSL_MSG("ASN Name too big, skipping");
            }
        }
    }

    return ret;
}
#endif /* WOLFSSL_ASN_TEMPLATE */

/* Get a certificate name into the certificate object.
 *
 * @param [in, out] cert      Decoded certificate object.
 * @param [out]     full      Buffer to hold full name as a string.
 * @param [out]     hash      Buffer to hold hash of name.
 * @param [in]      nameType  ASN_ISSUER or ASN_SUBJECT.
 * @param [in]      input     Buffer holding certificate name.
 * @param [in, out] inOutIdx  On in, start of certificate name.
 *                            On out, start of ASN.1 item after cert name.
 * @param [in]      maxIdx    Index of next item after certificate name.
 * @return  0 on success.
 * @return  ASN_PARSE_E when BER encoded data does not match ASN.1 items or
 *          is invalid.
 * @return  BUFFER_E when data in buffer is too small.
 * @return  ASN_OBJECT_ID_E when the expected OBJECT_ID tag is not found.
 * @return  ASN_UNKNOWN_OID_E when the OID cannot be verified.
 * @return  MEMORY_E when dynamic memory allocation fails.
 */
static int GetCertName(DecodedCert* cert, char* full, byte* hash, int nameType,
                       const byte* input, word32* inOutIdx, word32 maxIdx)
{
#ifndef WOLFSSL_ASN_TEMPLATE
    int    length;  /* length of all distinguished names */
    int    dummy;
    int    ret;
    word32 idx;
    word32 srcIdx = *inOutIdx;
#if (defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)) && \
    !defined(WOLFCRYPT_ONLY)
    WOLFSSL_X509_NAME* dName = NULL;
#endif

    WOLFSSL_MSG("Getting Cert Name");

    /* For OCSP, RFC2560 section 4.1.1 states the issuer hash should be
     * calculated over the entire DER encoding of the Name field, including
     * the tag and length. */
    if (CalcHashId_ex(input + *inOutIdx, maxIdx - *inOutIdx, hash,
            HashIdAlg(cert->signatureOID)) != 0) {
        return ASN_PARSE_E;
    }

#if (defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)) && \
    !defined(WOLFCRYPT_ONLY)
    dName = wolfSSL_X509_NAME_new_ex(cert->heap);
    if (dName == NULL) {
        return MEMORY_E;
    }
#endif /* OPENSSL_EXTRA */

    if (GetSequence(input, &srcIdx, &length, maxIdx) < 0) {
#if (defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)) && \
            !defined(WOLFCRYPT_ONLY)
        wolfSSL_X509_NAME_free(dName);
#endif /* OPENSSL_EXTRA */
        return ASN_PARSE_E;
    }

#if defined(HAVE_PKCS7) || defined(WOLFSSL_CERT_EXT)
    /* store pointer to raw issuer */
    if (nameType == ASN_ISSUER) {
        cert->issuerRaw = &input[srcIdx];
        cert->issuerRawLen = length;
    }
#endif
#if !defined(IGNORE_NAME_CONSTRAINTS) || defined(WOLFSSL_CERT_EXT)
    if (nameType == ASN_SUBJECT) {
        cert->subjectRaw = &input[srcIdx];
        cert->subjectRawLen = length;
    }
#endif

    length += (int)srcIdx;
    idx = 0;

    while (srcIdx < (word32)length) {
        byte        b       = 0;
        byte        joint[3];
        byte        tooBig  = FALSE;
        int         oidSz;
        const char* copy    = NULL;
        int         copyLen = 0;
        int         strLen  = 0;
        byte        id      = 0;
    #if (defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)) \
                && !defined(WOLFCRYPT_ONLY)
         int        nid = WC_NID_undef;
         int        enc;
    #endif /* OPENSSL_EXTRA */

        if (GetSet(input, &srcIdx, &dummy, maxIdx) < 0) {
            WOLFSSL_MSG("Cert name lacks set header, trying sequence");
        }

        if (GetSequence(input, &srcIdx, &dummy, maxIdx) <= 0) {
        #if (defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)) && \
            !defined(WOLFCRYPT_ONLY)
            wolfSSL_X509_NAME_free(dName);
        #endif /* OPENSSL_EXTRA */
            return ASN_PARSE_E;
        }

        ret = GetASNObjectId(input, &srcIdx, &oidSz, maxIdx);
        if (ret != 0) {
        #if (defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)) && \
            !defined(WOLFCRYPT_ONLY)
            wolfSSL_X509_NAME_free(dName);
        #endif /* OPENSSL_EXTRA */
            return ret;
        }

        /* make sure there is room for joint */
        if ((srcIdx + sizeof(joint)) > (word32)maxIdx) {
        #if (defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)) && \
            !defined(WOLFCRYPT_ONLY)
            wolfSSL_X509_NAME_free(dName);
        #endif /* OPENSSL_EXTRA */
            return ASN_PARSE_E;
        }

        XMEMCPY(joint, &input[srcIdx], sizeof(joint));

        /* v1 name types */
        if (joint[0] == 0x55 && joint[1] == 0x04) {
            srcIdx += 3;
            id = joint[2];
            if (GetHeader(input, &b, &srcIdx, &strLen, maxIdx, 1) < 0) {
            #if (defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)) && \
            !defined(WOLFCRYPT_ONLY)
                wolfSSL_X509_NAME_free(dName);
            #endif /* OPENSSL_EXTRA */
                return ASN_PARSE_E;
            }

        #ifndef WOLFSSL_NO_ASN_STRICT
            /* RFC 5280 section 4.1.2.4 lists a DirectoryString as being
             * 1..MAX in length */
            if (strLen < 1) {
                WOLFSSL_MSG("Non conforming DirectoryString of length 0 was"
                            " found");
                WOLFSSL_MSG("Use WOLFSSL_NO_ASN_STRICT if wanting to allow"
                            " empty DirectoryString's");
            #if (defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)) && \
            !defined(WOLFCRYPT_ONLY)
                wolfSSL_X509_NAME_free(dName);
            #endif /* OPENSSL_EXTRA */
                return ASN_PARSE_E;
            }
        #endif

            if (id == ASN_COMMON_NAME) {
                if (nameType == ASN_SUBJECT) {
                    cert->subjectCN = (char *)&input[srcIdx];
                    cert->subjectCNLen = strLen;
                    cert->subjectCNEnc = (char)b;
                }
            #if (defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_CERT_EXT)) && \
                defined(WOLFSSL_HAVE_ISSUER_NAMES)
                else if (nameType == ASN_ISSUER) {
                    cert->issuerCN = (char*)&input[srcIdx];
                    cert->issuerCNLen = strLen;
                    cert->issuerCNEnc = (char)b;
                }
            #endif

                copy = WOLFSSL_COMMON_NAME;
                copyLen = sizeof(WOLFSSL_COMMON_NAME) - 1;
            #if (defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)) \
                && !defined(WOLFCRYPT_ONLY)
                nid = WC_NID_commonName;
            #endif /* OPENSSL_EXTRA */
            }
        #ifdef WOLFSSL_CERT_NAME_ALL
            else if (id == ASN_NAME) {
                copy = WOLFSSL_NAME;
                copyLen = sizeof(WOLFSSL_NAME) - 1;
                #if defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_CERT_EXT)
                    if (nameType == ASN_SUBJECT) {
                        cert->subjectN = (char*)&input[srcIdx];
                        cert->subjectNLen = strLen;
                        cert->subjectNEnc = b;
                    }
                #endif /* WOLFSSL_CERT_GEN || WOLFSSL_CERT_EXT */
                #if (defined(OPENSSL_EXTRA) || \
                        defined(OPENSSL_EXTRA_X509_SMALL)) \
                        && !defined(WOLFCRYPT_ONLY)
                    nid = WC_NID_name;
                #endif /* OPENSSL_EXTRA */
            }
            else if (id == ASN_INITIALS) {
                copy = WOLFSSL_INITIALS;
                copyLen = sizeof(WOLFSSL_INITIALS) - 1;
                #if defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_CERT_EXT)
                    if (nameType == ASN_SUBJECT) {
                        cert->subjectI = (char*)&input[srcIdx];
                        cert->subjectILen = strLen;
                        cert->subjectIEnc = b;
                    }
                #endif /* WOLFSSL_CERT_GEN || WOLFSSL_CERT_EXT */
                #if (defined(OPENSSL_EXTRA) || \
                        defined(OPENSSL_EXTRA_X509_SMALL)) \
                        && !defined(WOLFCRYPT_ONLY)
                    nid = WC_NID_initials;
                #endif /* OPENSSL_EXTRA */
            }
            else if (id == ASN_GIVEN_NAME) {
                copy = WOLFSSL_GIVEN_NAME;
                copyLen = sizeof(WOLFSSL_GIVEN_NAME) - 1;
                #if defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_CERT_EXT)
                    if (nameType == ASN_SUBJECT) {
                        cert->subjectGN = (char*)&input[srcIdx];
                        cert->subjectGNLen = strLen;
                        cert->subjectGNEnc = b;
                    }
                #endif /* WOLFSSL_CERT_GEN || WOLFSSL_CERT_EXT */
                #if (defined(OPENSSL_EXTRA) || \
                        defined(OPENSSL_EXTRA_X509_SMALL)) \
                        && !defined(WOLFCRYPT_ONLY)
                    nid = WC_NID_givenName;
                #endif /* OPENSSL_EXTRA */
            }
            else if (id == ASN_DNQUALIFIER) {
                copy = WOLFSSL_DNQUALIFIER;
                copyLen = sizeof(WOLFSSL_DNQUALIFIER) - 1;
                #if defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_CERT_EXT)
                    if (nameType == ASN_SUBJECT) {
                        cert->subjectDNQ = (char*)&input[srcIdx];
                        cert->subjectDNQLen = strLen;
                        cert->subjectDNQEnc = b;
                    }
                #endif /* WOLFSSL_CERT_GEN || WOLFSSL_CERT_EXT */
                #if (defined(OPENSSL_EXTRA) || \
                        defined(OPENSSL_EXTRA_X509_SMALL)) \
                        && !defined(WOLFCRYPT_ONLY)
                    nid = WC_NID_dnQualifier;
                #endif /* OPENSSL_EXTRA */
            }
        #endif /* WOLFSSL_CERT_NAME_ALL */
            else if (id == ASN_SUR_NAME) {
                copy = WOLFSSL_SUR_NAME;
                copyLen = sizeof(WOLFSSL_SUR_NAME) - 1;
                #if defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_CERT_EXT)
                    if (nameType == ASN_SUBJECT) {
                        cert->subjectSN = (char*)&input[srcIdx];
                        cert->subjectSNLen = strLen;
                        cert->subjectSNEnc = (char)b;
                    }
                #if defined(WOLFSSL_HAVE_ISSUER_NAMES)
                    else if (nameType == ASN_ISSUER) {
                        cert->issuerSN = (char*)&input[srcIdx];
                        cert->issuerSNLen = strLen;
                        cert->issuerSNEnc = (char)b;
                    }
                #endif /* WOLFSSL_HAVE_ISSUER_NAMES */
                #endif /* WOLFSSL_CERT_GEN || WOLFSSL_CERT_EXT */
                #if (defined(OPENSSL_EXTRA) || \
                        defined(OPENSSL_EXTRA_X509_SMALL)) \
                        && !defined(WOLFCRYPT_ONLY)
                    nid = WC_NID_surname;
                #endif /* OPENSSL_EXTRA */
            }
            else if (id == ASN_COUNTRY_NAME) {
                copy = WOLFSSL_COUNTRY_NAME;
                copyLen = sizeof(WOLFSSL_COUNTRY_NAME) - 1;
                #if defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_CERT_EXT)
                    if (nameType == ASN_SUBJECT) {
                        cert->subjectC = (char*)&input[srcIdx];
                        cert->subjectCLen = strLen;
                        cert->subjectCEnc = (char)b;
                    }
                #if defined(WOLFSSL_HAVE_ISSUER_NAMES)
                    else if (nameType == ASN_ISSUER) {
                        cert->issuerC = (char*)&input[srcIdx];
                        cert->issuerCLen = strLen;
                        cert->issuerCEnc = (char)b;
                    }
                #endif /* WOLFSSL_HAVE_ISSUER_NAMES */
                #endif /* WOLFSSL_CERT_GEN || WOLFSSL_CERT_EXT */
                #if (defined(OPENSSL_EXTRA) || \
                        defined(OPENSSL_EXTRA_X509_SMALL)) \
                        && !defined(WOLFCRYPT_ONLY)
                    nid = WC_NID_countryName;
                #endif /* OPENSSL_EXTRA */
            }
            else if (id == ASN_LOCALITY_NAME) {
                copy = WOLFSSL_LOCALITY_NAME;
                copyLen = sizeof(WOLFSSL_LOCALITY_NAME) - 1;
                #if defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_CERT_EXT)
                    if (nameType == ASN_SUBJECT) {
                        cert->subjectL = (char*)&input[srcIdx];
                        cert->subjectLLen = strLen;
                        cert->subjectLEnc = (char)b;
                    }
                    #if defined(WOLFSSL_HAVE_ISSUER_NAMES)
                    else if (nameType == ASN_ISSUER) {
                        cert->issuerL = (char*)&input[srcIdx];
                        cert->issuerLLen = strLen;
                        cert->issuerLEnc = (char)b;
                    }
                    #endif /* WOLFSSL_HAVE_ISSUER_NAMES */
                #endif /* WOLFSSL_CERT_GEN || WOLFSSL_CERT_EXT */
                #if (defined(OPENSSL_EXTRA) || \
                        defined(OPENSSL_EXTRA_X509_SMALL)) \
                        && !defined(WOLFCRYPT_ONLY)
                    nid = WC_NID_localityName;
                #endif /* OPENSSL_EXTRA */
            }
            else if (id == ASN_STATE_NAME) {
                copy = WOLFSSL_STATE_NAME;
                copyLen = sizeof(WOLFSSL_STATE_NAME) - 1;
                #if defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_CERT_EXT)
                    if (nameType == ASN_SUBJECT) {
                        cert->subjectST = (char*)&input[srcIdx];
                        cert->subjectSTLen = strLen;
                        cert->subjectSTEnc = (char)b;
                    }
                #if defined(WOLFSSL_HAVE_ISSUER_NAMES)
                    else if (nameType == ASN_ISSUER) {
                        cert->issuerST = (char*)&input[srcIdx];
                        cert->issuerSTLen = strLen;
                        cert->issuerSTEnc = (char)b;
                    }
                #endif /* WOLFSSL_HAVE_ISSUER_NAMES */
                #endif /* WOLFSSL_CERT_GEN || WOLFSSL_CERT_EXT*/
                #if (defined(OPENSSL_EXTRA) || \
                        defined(OPENSSL_EXTRA_X509_SMALL)) \
                        && !defined(WOLFCRYPT_ONLY)
                    nid = WC_NID_stateOrProvinceName;
                #endif /* OPENSSL_EXTRA */
            }
            else if (id == ASN_ORG_NAME) {
                copy = WOLFSSL_ORG_NAME;
                copyLen = sizeof(WOLFSSL_ORG_NAME) - 1;
                #if defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_CERT_EXT)
                    if (nameType == ASN_SUBJECT) {
                        cert->subjectO = (char*)&input[srcIdx];
                        cert->subjectOLen = strLen;
                        cert->subjectOEnc = (char)b;
                    }
                #if defined(WOLFSSL_HAVE_ISSUER_NAMES)
                    else if (nameType == ASN_ISSUER) {
                        cert->issuerO = (char*)&input[srcIdx];
                        cert->issuerOLen = strLen;
                        cert->issuerOEnc = (char)b;
                    }
                #endif /* WOLFSSL_HAVE_ISSUER_NAMES */
                #endif /* WOLFSSL_CERT_GEN || WOLFSSL_CERT_EXT */
                #if (defined(OPENSSL_EXTRA) || \
                        defined(OPENSSL_EXTRA_X509_SMALL)) \
                        && !defined(WOLFCRYPT_ONLY)
                    nid = WC_NID_organizationName;
                #endif /* OPENSSL_EXTRA */
            }
            else if (id == ASN_ORGUNIT_NAME) {
                copy = WOLFSSL_ORGUNIT_NAME;
                copyLen = sizeof(WOLFSSL_ORGUNIT_NAME) - 1;
                #if defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_CERT_EXT)
                    if (nameType == ASN_SUBJECT) {
                        cert->subjectOU = (char*)&input[srcIdx];
                        cert->subjectOULen = strLen;
                        cert->subjectOUEnc = (char)b;
                    }
                #if defined(WOLFSSL_HAVE_ISSUER_NAMES)
                    else if (nameType == ASN_ISSUER) {
                        cert->issuerOU = (char*)&input[srcIdx];
                        cert->issuerOULen = strLen;
                        cert->issuerOUEnc = (char)b;
                    }
                #endif /* WOLFSSL_HAVE_ISSUER_NAMES */
                #endif /* WOLFSSL_CERT_GEN || WOLFSSL_CERT_EXT */
                #if (defined(OPENSSL_EXTRA) || \
                        defined(OPENSSL_EXTRA_X509_SMALL)) \
                        && !defined(WOLFCRYPT_ONLY)
                    nid = WC_NID_organizationalUnitName;
                #endif /* OPENSSL_EXTRA */
            }
            else if (id == ASN_SERIAL_NUMBER) {
                copy = WOLFSSL_SERIAL_NUMBER;
                copyLen = sizeof(WOLFSSL_SERIAL_NUMBER) - 1;
                #if defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_CERT_EXT)
                    if (nameType == ASN_SUBJECT) {
                        cert->subjectSND = (char*)&input[srcIdx];
                        cert->subjectSNDLen = strLen;
                        cert->subjectSNDEnc = (char)b;
                    }
                #if defined(WOLFSSL_HAVE_ISSUER_NAMES)
                    else if (nameType == ASN_ISSUER) {
                        cert->issuerSND = (char*)&input[srcIdx];
                        cert->issuerSNDLen = strLen;
                        cert->issuerSNDEnc = (char)b;
                    }
                #endif /* WOLFSSL_HAVE_ISSUER_NAMES */
                #endif /* WOLFSSL_CERT_GEN || WOLFSSL_CERT_EXT */
                #if (defined(OPENSSL_EXTRA) || \
                        defined(OPENSSL_EXTRA_X509_SMALL)) \
                        && !defined(WOLFCRYPT_ONLY)
                    nid = WC_NID_serialNumber;
                #endif /* OPENSSL_EXTRA */
            }
            else if (id == ASN_USER_ID) {
                copy = WOLFSSL_USER_ID;
                copyLen = sizeof(WOLFSSL_USER_ID) - 1;
                #if defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_CERT_EXT)
                    if (nameType == ASN_SUBJECT) {
                        cert->subjectUID = (char*)&input[srcIdx];
                        cert->subjectUIDLen = strLen;
                        cert->subjectUIDEnc = (char)b;
                    }
                #endif /* WOLFSSL_CERT_GEN || WOLFSSL_CERT_EXT */
                #if (defined(OPENSSL_EXTRA) || \
                        defined(OPENSSL_EXTRA_X509_SMALL)) \
                        && !defined(WOLFCRYPT_ONLY)
                    nid = WC_NID_userId;
                #endif /* OPENSSL_EXTRA */
            }
        #ifdef WOLFSSL_CERT_EXT
            else if (id == ASN_STREET_ADDR) {
                copy = WOLFSSL_STREET_ADDR_NAME;
                copyLen = sizeof(WOLFSSL_STREET_ADDR_NAME) - 1;
                #if defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_CERT_EXT)
                    if (nameType == ASN_SUBJECT) {
                        cert->subjectStreet = (char*)&input[srcIdx];
                        cert->subjectStreetLen = strLen;
                        cert->subjectStreetEnc = (char)b;
                    }
                #endif /* WOLFSSL_CERT_GEN || WOLFSSL_CERT_EXT */
                #if (defined(OPENSSL_EXTRA) || \
                        defined(OPENSSL_EXTRA_X509_SMALL)) \
                        && !defined(WOLFCRYPT_ONLY)
                    nid = WC_NID_streetAddress;
                #endif /* OPENSSL_EXTRA */
            }
            else if (id == ASN_BUS_CAT) {
                copy = WOLFSSL_BUS_CAT;
                copyLen = sizeof(WOLFSSL_BUS_CAT) - 1;
            #if defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_CERT_EXT)
                if (nameType == ASN_SUBJECT) {
                    cert->subjectBC = (char*)&input[srcIdx];
                    cert->subjectBCLen = strLen;
                    cert->subjectBCEnc = (char)b;
                }
            #endif /* WOLFSSL_CERT_GEN || WOLFSSL_CERT_EXT */
            #if (defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)) \
                        && !defined(WOLFCRYPT_ONLY)
                nid = WC_NID_businessCategory;
            #endif /* OPENSSL_EXTRA */
            }
            else if (id == ASN_POSTAL_CODE) {
                copy = WOLFSSL_POSTAL_NAME;
                copyLen = sizeof(WOLFSSL_POSTAL_NAME) - 1;
                #if defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_CERT_EXT)
                    if (nameType == ASN_SUBJECT) {
                        cert->subjectPC = (char*)&input[srcIdx];
                        cert->subjectPCLen = strLen;
                        cert->subjectPCEnc = (char)b;
                    }
                #endif /* WOLFSSL_CERT_GEN || WOLFSSL_CERT_EXT*/
                #if (defined(OPENSSL_EXTRA) || \
                        defined(OPENSSL_EXTRA_X509_SMALL)) \
                        && !defined(WOLFCRYPT_ONLY)
                    nid = WC_NID_postalCode;
                #endif /* OPENSSL_EXTRA */
            }
        #endif /* WOLFSSL_CERT_EXT */
        }
    #ifdef WOLFSSL_CERT_EXT
        else if ((srcIdx + ASN_JOI_PREFIX_SZ + 2 <= (word32)maxIdx) &&
                 (0 == XMEMCMP(&input[srcIdx], ASN_JOI_PREFIX,
                               ASN_JOI_PREFIX_SZ)) &&
                 ((input[srcIdx+ASN_JOI_PREFIX_SZ] == ASN_JOI_C) ||
                  (input[srcIdx+ASN_JOI_PREFIX_SZ] == ASN_JOI_ST)))
        {
            srcIdx += ASN_JOI_PREFIX_SZ;
            id = input[srcIdx++];
            b = input[srcIdx++]; /* encoding */

            if (GetLength(input, &srcIdx, &strLen,
                          maxIdx) < 0) {
            #if (defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)) && \
            !defined(WOLFCRYPT_ONLY)
                wolfSSL_X509_NAME_free(dName);
            #endif /* OPENSSL_EXTRA */
                return ASN_PARSE_E;
            }

            /* Check for jurisdiction of incorporation country name */
            if (id == ASN_JOI_C) {
                copy = WOLFSSL_JOI_C;
                copyLen = sizeof(WOLFSSL_JOI_C) - 1;
                #if defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_CERT_EXT)
                    if (nameType == ASN_SUBJECT) {
                        cert->subjectJC = (char*)&input[srcIdx];
                        cert->subjectJCLen = strLen;
                        cert->subjectJCEnc = (char)b;
                    }
                #endif /* WOLFSSL_CERT_GEN || WOLFSSL_CERT_EXT */
                #if (defined(OPENSSL_EXTRA) || \
                        defined(OPENSSL_EXTRA_X509_SMALL)) \
                        && !defined(WOLFCRYPT_ONLY)
                    nid = WC_NID_jurisdictionCountryName;
                #endif /* OPENSSL_EXTRA */
            }

            /* Check for jurisdiction of incorporation state name */
            else if (id == ASN_JOI_ST) {
                copy = WOLFSSL_JOI_ST;
                copyLen = sizeof(WOLFSSL_JOI_ST) - 1;
                #if defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_CERT_EXT)
                    if (nameType == ASN_SUBJECT) {
                        cert->subjectJS = (char*)&input[srcIdx];
                        cert->subjectJSLen = strLen;
                        cert->subjectJSEnc = (char)b;
                    }
                #endif /* WOLFSSL_CERT_GEN || WOLFSSL_CERT_EXT */
                #if (defined(OPENSSL_EXTRA) || \
                        defined(OPENSSL_EXTRA_X509_SMALL)) \
                        && !defined(WOLFCRYPT_ONLY)
                    nid = WC_NID_jurisdictionStateOrProvinceName;
                #endif /* OPENSSL_EXTRA */
            }

            if ((strLen + copyLen) > (int)(WC_ASN_NAME_MAX - idx)) {
                WOLFSSL_MSG("ASN Name too big, skipping");
                tooBig = TRUE;
            }
        }
    #endif /* WOLFSSL_CERT_EXT */
        else {
            /* skip */
            byte email = FALSE;
            byte pilot = FALSE;

            if (joint[0] == 0x2a && joint[1] == 0x86) {  /* email id hdr 42.134.* */
                id = ASN_EMAIL_NAME;
                email = TRUE;
            }

            if (joint[0] == 0x9  && joint[1] == 0x92) { /* uid id hdr 9.146.* */
                /* last value of OID is the type of pilot attribute */
                id    = input[srcIdx + (word32)oidSz - 1];
                if (id == 0x01)
                    id = ASN_USER_ID;
                pilot = TRUE;
            }

            srcIdx += (word32)oidSz + 1;

            if (GetLength(input, &srcIdx, &strLen, maxIdx) < 0) {
            #if (defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)) && \
            !defined(WOLFCRYPT_ONLY)
                wolfSSL_X509_NAME_free(dName);
            #endif /* OPENSSL_EXTRA */
                return ASN_PARSE_E;
            }

            if (strLen > (int)(WC_ASN_NAME_MAX - idx)) {
                WOLFSSL_MSG("ASN name too big, skipping");
                tooBig = TRUE;
            }

            if (email) {
                copyLen = sizeof(WOLFSSL_EMAIL_ADDR) - 1;
                if ((copyLen + strLen) > (int)(WC_ASN_NAME_MAX - idx)) {
                    WOLFSSL_MSG("ASN name too big, skipping");
                    tooBig = TRUE;
                }
                else {
                    copy = WOLFSSL_EMAIL_ADDR;
                }

                #if !defined(IGNORE_NAME_CONSTRAINTS) || \
                     defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_CERT_EXT)
                    if (nameType == ASN_SUBJECT) {
                        cert->subjectEmail = (char*)&input[srcIdx];
                        cert->subjectEmailLen = strLen;
                    }
                #if defined(WOLFSSL_HAVE_ISSUER_NAMES) && \
                    (defined(WOLFSSL_CERT_GEN) || defined(WOLFSSL_CERT_EXT))
                    else if (nameType == ASN_ISSUER) {
                        cert->issuerEmail = (char*)&input[srcIdx];
                        cert->issuerEmailLen = strLen;
                    }
                #endif /* WOLFSSL_HAVE_ISSUER_NAMES */
                #endif /* WOLFSSL_CERT_GEN || WOLFSSL_CERT_EXT */
                #if (defined(OPENSSL_EXTRA) || \
                        defined(OPENSSL_EXTRA_X509_SMALL)) \
                        && !defined(WOLFCRYPT_ONLY)
                    nid = WC_NID_emailAddress;
                #endif /* OPENSSL_EXTRA */
            }

            if (pilot) {
                switch (id) {
                    case ASN_USER_ID:
                        copy = WOLFSSL_USER_ID;
                        copyLen = sizeof(WOLFSSL_USER_ID) - 1;
                    #if (defined(OPENSSL_EXTRA) || \
                        defined(OPENSSL_EXTRA_X509_SMALL)) \
                        && !defined(WOLFCRYPT_ONLY)
                        nid = WC_NID_userId;
                    #endif /* OPENSSL_EXTRA */
                        break;
                    case ASN_DOMAIN_COMPONENT:
                        copy = WOLFSSL_DOMAIN_COMPONENT;
                        copyLen = sizeof(WOLFSSL_DOMAIN_COMPONENT) - 1;
                    #if (defined(OPENSSL_EXTRA) || \
                        defined(OPENSSL_EXTRA_X509_SMALL)) \
                        && !defined(WOLFCRYPT_ONLY)
                        nid = WC_NID_domainComponent;
                    #endif /* OPENSSL_EXTRA */
                        break;
                    case ASN_RFC822_MAILBOX:
                        copy = WOLFSSL_RFC822_MAILBOX;
                        copyLen = sizeof(WOLFSSL_RFC822_MAILBOX) - 1;
                    #if (defined(OPENSSL_EXTRA) || \
                        defined(OPENSSL_EXTRA_X509_SMALL)) \
                        && !defined(WOLFCRYPT_ONLY)
                        nid = WC_NID_rfc822Mailbox;
                    #endif /* OPENSSL_EXTRA */
                        break;
                    case ASN_FAVOURITE_DRINK:
                        copy = WOLFSSL_FAVOURITE_DRINK;
                        copyLen = sizeof(WOLFSSL_FAVOURITE_DRINK) - 1;
                    #if (defined(OPENSSL_EXTRA) || \
                        defined(OPENSSL_EXTRA_X509_SMALL)) \
                        && !defined(WOLFCRYPT_ONLY)
                        nid = WC_NID_favouriteDrink;
                    #endif /* OPENSSL_EXTRA */
                        break;
                    case ASN_CONTENT_TYPE:
                        copy = WOLFSSL_CONTENT_TYPE;
                        copyLen = sizeof(WOLFSSL_CONTENT_TYPE) - 1;
                    #if (defined(OPENSSL_EXTRA) || \
                        defined(OPENSSL_EXTRA_X509_SMALL)) \
                        && !defined(WOLFCRYPT_ONLY)
                        nid = WC_NID_pkcs9_contentType;
                    #endif /* OPENSSL_EXTRA */
                        break;
                    default:
                        WOLFSSL_MSG("Unknown pilot attribute type");
                    #if (defined(OPENSSL_EXTRA) || \
                                defined(OPENSSL_EXTRA_X509_SMALL)) && \
                                !defined(WOLFCRYPT_ONLY)
                        wolfSSL_X509_NAME_free(dName);
                    #endif /* OPENSSL_EXTRA */
                        return ASN_PARSE_E;
                }
            }
        }
        if ((copyLen + strLen) > (int)(WC_ASN_NAME_MAX - idx))
        {
            WOLFSSL_MSG("ASN Name too big, skipping");
            tooBig = TRUE;
        }
        if ((copy != NULL) && !tooBig) {
            XMEMCPY(&full[idx], copy, (size_t)copyLen);
            idx += (word32)copyLen;
            XMEMCPY(&full[idx], &input[srcIdx], (size_t)strLen);
            idx += (word32)strLen;
        }
        #if (defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)) && \
            !defined(WOLFCRYPT_ONLY)
        switch (b) {
            case CTC_UTF8:
                enc = WOLFSSL_MBSTRING_UTF8;
                break;
            case CTC_PRINTABLE:
                enc = WOLFSSL_V_ASN1_PRINTABLESTRING;
                break;
            default:
                WOLFSSL_MSG("Unknown encoding type, using UTF8 by default");
                enc = WOLFSSL_MBSTRING_UTF8;
        }

        if (nid != WC_NID_undef) {
            if (wolfSSL_X509_NAME_add_entry_by_NID(dName, nid, enc,
                            &input[srcIdx], strLen, -1, -1) !=
                            WOLFSSL_SUCCESS) {
                wolfSSL_X509_NAME_free(dName);
                return ASN_PARSE_E;
            }
        }
        #endif /* OPENSSL_EXTRA */
        srcIdx += (word32)strLen;
    }
    full[idx++] = 0;

#if (defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)) && \
            !defined(WOLFCRYPT_ONLY)
    if (nameType == ASN_ISSUER) {
#if (defined(OPENSSL_ALL) || defined(WOLFSSL_NGINX) || defined(HAVE_LIGHTY)) &&\
    (defined(HAVE_PKCS7) || defined(WOLFSSL_CERT_EXT))
        dName->rawLen = min(cert->issuerRawLen, WC_ASN_NAME_MAX);
        XMEMCPY(dName->raw, cert->issuerRaw, dName->rawLen);
#endif
        cert->issuerName = dName;
    }
    else {
#if defined(OPENSSL_ALL) || defined(WOLFSSL_NGINX)
        dName->rawLen = min(cert->subjectRawLen, WC_ASN_NAME_MAX);
        XMEMCPY(dName->raw, cert->subjectRaw, dName->rawLen);
#endif
        cert->subjectName = dName;
    }
#endif

    *inOutIdx = srcIdx;

    return 0;
#else
    DECL_ASNGETDATA(dataASN, rdnASN_Length);
    int    ret = 0;
    word32 idx = 0;
    int    len = 0;
    word32 srcIdx = *inOutIdx;
#ifdef WOLFSSL_X509_NAME_AVAILABLE
    WOLFSSL_X509_NAME* dName = NULL;
#endif /* WOLFSSL_X509_NAME_AVAILABLE */

    WOLFSSL_MSG("Getting Cert Name");

    /* For OCSP, RFC2560 section 4.1.1 states the issuer hash should be
     * calculated over the entire DER encoding of the Name field, including
     * the tag and length. */
    if (CalcHashId_ex(input + srcIdx, maxIdx - srcIdx, hash,
            HashIdAlg(cert->signatureOID)) != 0) {
        ret = ASN_PARSE_E;
    }

    CALLOC_ASNGETDATA(dataASN, rdnASN_Length, ret, cert->heap);

#ifdef WOLFSSL_X509_NAME_AVAILABLE
    if (ret == 0) {
        /* Create an X509_NAME to hold data for OpenSSL compatibility APIs. */
        dName = wolfSSL_X509_NAME_new_ex(cert->heap);
        if (dName == NULL) {
            ret = MEMORY_E;
        }
    }
#endif /* WOLFSSL_X509_NAME_AVAILABLE */

    if (ret == 0) {
        /* Expecting a SEQUENCE using up all data. */
        ret = GetASN_Sequence(input, &srcIdx, &len, maxIdx, 1);
    }
    if (ret == 0) {
    #if defined(HAVE_PKCS7) || defined(WOLFSSL_CERT_EXT)
        /* Store pointer and length to raw issuer. */
        if (nameType == ASN_ISSUER) {
            cert->issuerRaw = &input[srcIdx];
            cert->issuerRawLen = len;
        }
    #endif
    #if !defined(IGNORE_NAME_CONSTRAINTS) || defined(WOLFSSL_CERT_EXT)
        /* Store pointer and length to raw subject. */
        if (nameType == ASN_SUBJECT) {
            cert->subjectRaw = &input[srcIdx];
            cert->subjectRawLen = len;
        }
    #endif

        /* Process all RDNs in name. */
        while ((ret == 0) && (srcIdx < maxIdx)) {
            int nid = 0;

            /* Initialize for data and setup RDN choice. */
            GetASN_Choice(&dataASN[RDNASN_IDX_ATTR_VAL], rdnChoice);
            /* Ignore type OID as too many to store in table. */
            GetASN_OID(&dataASN[RDNASN_IDX_ATTR_TYPE], oidIgnoreType);
            /* Parse RDN. */
            ret = GetASN_Items(rdnASN, dataASN, rdnASN_Length, 1, input,
                               &srcIdx, maxIdx);
            if (ret == 0) {
                /* Put RDN data into certificate. */
                ret = GetRDN(cert, full, &idx, &nid, nameType == ASN_SUBJECT,
                             dataASN);
            }
        #ifdef WOLFSSL_X509_NAME_AVAILABLE
            /* TODO: push this back up to ssl.c
             * (do parsing for WOLFSSL_X509_NAME on demand) */
            if (ret == 0) {
                int enc;
                byte*  str;
                word32 strLen;
                byte   tag = dataASN[RDNASN_IDX_ATTR_VAL].tag;

                /* Get string reference. */
                GetASN_GetRef(&dataASN[RDNASN_IDX_ATTR_VAL], &str, &strLen);

            #ifndef WOLFSSL_NO_ASN_STRICT
                /* RFC 5280 section 4.1.2.4 lists a DirectoryString as being
                 * 1..MAX in length */
                if (ret == 0 && strLen < 1) {
                    WOLFSSL_MSG("Non conforming DirectoryString of length 0 was"
                                " found");
                    WOLFSSL_MSG("Use WOLFSSL_NO_ASN_STRICT if wanting to allow"
                                " empty DirectoryString's");
                    ret = ASN_PARSE_E;
                }
            #endif

                /* Convert BER tag to a OpenSSL type. */
                switch (tag) {
                    case CTC_UTF8:
                        enc = WOLFSSL_MBSTRING_UTF8;
                        break;
                    case CTC_PRINTABLE:
                        enc = WOLFSSL_V_ASN1_PRINTABLESTRING;
                        break;
                    default:
                        WOLFSSL_MSG("Unknown encoding type, default UTF8");
                        enc = WOLFSSL_MBSTRING_UTF8;
                }
                if (nid != 0) {
                    /* Add an entry to the X509_NAME. */
                    if (wolfSSL_X509_NAME_add_entry_by_NID(dName, nid, enc, str,
                            (int)strLen, -1, -1) != WOLFSSL_SUCCESS) {
                        ret = ASN_PARSE_E;
                    }
                }
            }
        #endif
        }
    }
    if (ret == 0) {
        /* Terminate string. */
        full[idx] = 0;
        /* Return index into encoding after name. */
        *inOutIdx = srcIdx;

#ifdef WOLFSSL_X509_NAME_AVAILABLE
        /* Store X509_NAME in certificate. */
        if (nameType == ASN_ISSUER) {
        #if (defined(OPENSSL_ALL) || defined(WOLFSSL_NGINX) || \
             defined(HAVE_LIGHTY)) && \
            (defined(HAVE_PKCS7) || defined(WOLFSSL_CERT_EXT))
            dName->rawLen = (int)min((word32)cert->issuerRawLen,
                WC_ASN_NAME_MAX);
            XMEMCPY(dName->raw, cert->issuerRaw, (size_t)dName->rawLen);
        #endif
            cert->issuerName = dName;
        }
        else {
        #if defined(OPENSSL_ALL) || defined(WOLFSSL_NGINX)
            dName->rawLen = (int)min((word32)cert->subjectRawLen,
                WC_ASN_NAME_MAX);
            XMEMCPY(dName->raw, cert->subjectRaw, (size_t)dName->rawLen);
        #endif
            cert->subjectName = dName;
        }
    }
    else {
        /* Dispose of unused X509_NAME. */
        wolfSSL_X509_NAME_free(dName);
#endif
    }

    FREE_ASNGETDATA(dataASN, cert->heap);
    return ret;
#endif /* WOLFSSL_ASN_TEMPLATE */
}

#ifdef WOLFSSL_ASN_TEMPLATE
/* ASN.1 template for certificate name. */
static const ASNItem certNameASN[] = {
/* OID  */ { 0, ASN_OBJECT_ID, 0, 0, 1 },
/* NAME */ { 0, ASN_SEQUENCE, 1, 0, 0 },
};
enum {
    CERTNAMEASN_IDX_OID = 0,
    CERTNAMEASN_IDX_NAME
};

/* Number of items in ASN.1 template for certificate name. */
#define certNameASN_Length (sizeof(certNameASN) / sizeof(ASNItem))
#endif

/* Get a certificate name into the certificate object.
 *
 * Either the issuer or subject name.
 *
 * @param [in, out] cert      Decoded certificate object.
 * @param [in]      nameType  Type being decoded: ASN_ISSUER or ASN_SUBJECT.
 * @param [in]      maxIdx    Index of next item after certificate name.
 * @return  0 on success.
 * @return  ASN_PARSE_E when BER encoded data does not match ASN.1 items or
 *          is invalid.
 * @return  BUFFER_E when data in buffer is too small.
 * @return  ASN_OBJECT_ID_E when the expected OBJECT_ID tag is not found.
 * @return  ASN_UNKNOWN_OID_E when the OID cannot be verified.
 * @return  MEMORY_E when dynamic memory allocation fails.
 */
int GetName(DecodedCert* cert, int nameType, int maxIdx)
{
#ifndef WOLFSSL_ASN_TEMPLATE
    char*  full;
    byte*  hash;
    int    length;
    word32 localIdx;
    byte   tag;

    WOLFSSL_MSG("Getting Name");

    if (nameType == ASN_ISSUER) {
        full = cert->issuer;
        hash = cert->issuerHash;
    }
    else {
        full = cert->subject;
        hash = cert->subjectHash;
    }

    if (cert->srcIdx >= (word32)maxIdx) {
        return BUFFER_E;
    }

    localIdx = cert->srcIdx;
    if (GetASNTag(cert->source, &localIdx, &tag, (word32)maxIdx) < 0) {
        return ASN_PARSE_E;
    }

    if (tag == ASN_OBJECT_ID) {
        WOLFSSL_MSG("Trying optional prefix...");

        if (SkipObjectId(cert->source, &cert->srcIdx, (word32)maxIdx) < 0)
            return ASN_PARSE_E;
        WOLFSSL_MSG("Got optional prefix");
    }

    localIdx = cert->srcIdx;
    if (GetASNTag(cert->source, &localIdx, &tag, (word32)maxIdx) < 0) {
        return ASN_PARSE_E;
    }
    localIdx = cert->srcIdx + 1;
    if (GetLength(cert->source, &localIdx, &length, (word32)maxIdx) < 0) {
        return ASN_PARSE_E;
    }
    length += (int)(localIdx - cert->srcIdx);

    return GetCertName(cert, full, hash, nameType, cert->source, &cert->srcIdx,
                       cert->srcIdx + (word32)length);
#else
    ASNGetData dataASN[certNameASN_Length];
    word32 idx = cert->srcIdx;
    int    ret = 0;

    WOLFSSL_MSG("Getting Name");

    XMEMSET(dataASN, 0, sizeof(dataASN));
    /* Initialize for data and don't check optional prefix OID. */
    GetASN_OID(&dataASN[CERTNAMEASN_IDX_OID], oidIgnoreType);
    ret = GetASN_Items(certNameASN, dataASN, certNameASN_Length, 0,
                       cert->source, &idx, (word32)maxIdx);
    if (ret == 0) {
        char* full;
        byte* hash;

        /* Store offset of SEQUENCE that is start of name. */
        cert->srcIdx = dataASN[CERTNAMEASN_IDX_NAME].offset;

        /* Get fields to fill in based on name type. */
        if (nameType == ASN_ISSUER) {
            full = cert->issuer;
            hash = cert->issuerHash;
        }
        else {
            full = cert->subject;
            hash = cert->subjectHash;
        }

        /* Parse certificate name. */
        ret = GetCertName(cert, full, hash, nameType, cert->source,
                          &cert->srcIdx, idx);
    }

    return ret;
#endif
}

#ifndef NO_ASN_TIME

/* two byte date/time, add to value */
static WC_INLINE int GetTime(int* value, const byte* date, int* idx)
{
    int i = *idx;

    if (date[i] < 0x30 || date[i] > 0x39 || date[i+1] < 0x30 ||
                                                             date[i+1] > 0x39) {
        return ASN_PARSE_E;
    }

    *value += (int)btoi(date[i++]) * 10;
    *value += (int)btoi(date[i++]);

    *idx = i;

    return 0;
}

#ifdef WOLFSSL_LINUXKM
static WC_INLINE int GetTime_Long(long* value, const byte* date, int* idx)
{
    int i = *idx;

    if (date[i] < 0x30 || date[i] > 0x39 || date[i+1] < 0x30 ||
                                                             date[i+1] > 0x39) {
        return ASN_PARSE_E;
    }

    *value += (long)btoi(date[i++]) * 10;
    *value += (long)btoi(date[i++]);

    *idx = i;

    return 0;
}
#endif

/* Extract certTime from date string parameter.
 * Reminder: idx is incremented in each call to GetTime()
 * Return 0 on failure, 1 for success.  */
int ExtractDate(const unsigned char* date, unsigned char format,
                struct tm* certTime, int* idx)
{
    XMEMSET(certTime, 0, sizeof(struct tm));

    /* Get the first two bytes of the year (century) */
    if (format == ASN_UTC_TIME) {
        if (btoi(date[*idx]) >= 5)
            certTime->tm_year = 1900;
        else
            certTime->tm_year = 2000;
    }
    else {
        /* format == GENERALIZED_TIME */
#ifdef WOLFSSL_LINUXKM
        if (GetTime_Long(&certTime->tm_year, date, idx) != 0) return 0;
#else
        if (GetTime(&certTime->tm_year, date, idx) != 0) return 0;
#endif
        certTime->tm_year *= 100;
    }

#ifdef AVR
    /* Extract the time from the struct tm and adjust tm_year, tm_mon */
    /* AVR libc stores these as uint8_t instead of int */
    /* AVR time_t also offsets from midnight 1 Jan 2000 */
    int tm_year = certTime->tm_year - 2000;
    int tm_mon  = certTime->tm_mon - 1;
    int tm_mday = certTime->tm_mday;
    int tm_hour = certTime->tm_hour;
    int tm_min  = certTime->tm_min;
    int tm_sec  = certTime->tm_sec;

    if (GetTime(&tm_year, date, idx) != 0) return 0;
    if (GetTime(&tm_mon , date, idx) != 0) return 0;
    if (GetTime(&tm_mday, date, idx) != 0) return 0;
    if (GetTime(&tm_hour, date, idx) != 0) return 0;
    if (GetTime(&tm_min , date, idx) != 0) return 0;
    if (GetTime(&tm_sec , date, idx) != 0) return 0;

    /* Re-populate certTime with computed values */
    certTime->tm_year = tm_year;
    certTime->tm_mon  = tm_mon;
    certTime->tm_mday = tm_mday;
    certTime->tm_hour = tm_hour;
    certTime->tm_min  = tm_min;
    certTime->tm_sec  = tm_sec;
#else /* !AVR */
    /* Get the next two bytes of the year. */
    #ifdef WOLFSSL_LINUXKM
        if (GetTime_Long(&certTime->tm_year, date, idx) != 0) return 0;
    #else
        if (GetTime(&certTime->tm_year, date, idx) != 0) return 0;
    #endif
    certTime->tm_year -= 1900;

    /* The next fields are expected in specific order in [date] string: */
    if (GetTime(&certTime->tm_mon , date, idx) != 0) return 0;
    certTime->tm_mon  -= 1;
    if (GetTime(&certTime->tm_mday, date, idx) != 0) return 0;
    if (GetTime(&certTime->tm_hour, date, idx) != 0) return 0;
    if (GetTime(&certTime->tm_min , date, idx) != 0) return 0;
    if (GetTime(&certTime->tm_sec , date, idx) != 0) return 0;

#endif /* !AVR */

    return 1;
}


#ifdef WOLFSSL_ASN_TIME_STRING
int GetTimeString(byte* date, int format, char* buf, int len)
{
    struct tm t;
    int idx = 0;

    if (!ExtractDate(date, (unsigned char)format, &t, &idx)) {
        return 0;
    }

    if (date[idx] != 'Z') {
        WOLFSSL_MSG("UTCtime, not Zulu") ;
        return 0;
    }

    /* place month in buffer */
    buf[0] = '\0';
    switch(t.tm_mon) {
        case 0:  XSTRNCAT(buf, "Jan ", 5); break;
        case 1:  XSTRNCAT(buf, "Feb ", 5); break;
        case 2:  XSTRNCAT(buf, "Mar ", 5); break;
        case 3:  XSTRNCAT(buf, "Apr ", 5); break;
        case 4:  XSTRNCAT(buf, "May ", 5); break;
        case 5:  XSTRNCAT(buf, "Jun ", 5); break;
        case 6:  XSTRNCAT(buf, "Jul ", 5); break;
        case 7:  XSTRNCAT(buf, "Aug ", 5); break;
        case 8:  XSTRNCAT(buf, "Sep ", 5); break;
        case 9:  XSTRNCAT(buf, "Oct ", 5); break;
        case 10: XSTRNCAT(buf, "Nov ", 5); break;
        case 11: XSTRNCAT(buf, "Dec ", 5); break;
        default:
            return 0;

    }
    idx = 4; /* use idx now for char buffer */

    if (XSNPRINTF(buf + idx, (size_t)(len - idx), "%2d %02d:%02d:%02d %d GMT",
              t.tm_mday, t.tm_hour, t.tm_min, t.tm_sec, (int)t.tm_year + 1900)
        >= len - idx)
    {
        WOLFSSL_MSG("buffer overrun in GetTimeString");
        return 0;
    }

    return 1;
}
#endif /* WOLFSSL_ASN_TIME_STRING */

/* Check time struct for valid values. Returns 0 for success */
static int ValidateGmtime(struct tm* inTime)
{
    int ret = 1;
    if ((inTime != NULL) &&
        (inTime->tm_sec >= 0) && (inTime->tm_sec <= 61) &&
        (inTime->tm_min >= 0) && (inTime->tm_min <= 59) &&
        (inTime->tm_hour >= 0) && (inTime->tm_hour <= 23) &&
        (inTime->tm_mday >= 1) && (inTime->tm_mday <= 31) &&
        (inTime->tm_mon >= 0) && (inTime->tm_mon <= 11) &&
        (inTime->tm_wday >= 0) && (inTime->tm_wday <= 6) &&
        (inTime->tm_yday >= 0) && (inTime->tm_yday <= 365)) {
        ret = 0;
    }

    return ret;
}

#if !defined(NO_ASN_TIME) && !defined(USER_TIME) && \
    !defined(TIME_OVERRIDES) && (defined(OPENSSL_EXTRA) || defined(HAVE_PKCS7))
/* Set current time string, either UTC or GeneralizedTime.
 * (void*) currTime should be a pointer to time_t, output is placed in buf.
 *
 * Return time string length placed in buf on success, negative on error */
int GetAsnTimeString(void* currTime, byte* buf, word32 len)
{
    byte* data_ptr  = buf;
    byte  uf_time[ASN_GENERALIZED_TIME_SIZE];
    int data_len = 0;

    WOLFSSL_ENTER("GetAsnTimeString");

    if (buf == NULL || len == 0)
        return BAD_FUNC_ARG;

    XMEMSET(uf_time, 0, sizeof(uf_time));
    /* GetFormattedTime returns length with null terminator */
    data_len = GetFormattedTime(currTime, uf_time, (word32)sizeof(uf_time));
    if (data_len <= 0) {
        return ASN_TIME_E;
    }
    /* ensure room to add 2 bytes (ASN type and length) before proceeding */
    else if (len < (word32)data_len + 2) {
        return BUFFER_E;
    }

    if (data_len == ASN_UTC_TIME_SIZE-1) {
        /* increment data_len for ASN length byte after adding the data_ptr */
        *data_ptr = (byte)ASN_UTC_TIME; data_ptr++; data_len++;
        /* -1 below excludes null terminator */
        *data_ptr = (byte)ASN_UTC_TIME_SIZE - 1; data_ptr++; data_len++;
        XMEMCPY(data_ptr, (byte *)uf_time, ASN_UTC_TIME_SIZE - 1);
        data_ptr += ASN_UTC_TIME_SIZE - 1;
    }
    else if (data_len == ASN_GENERALIZED_TIME_SIZE-1) {
        /* increment data_len for ASN length byte after adding the data_ptr */
        *data_ptr = (byte)ASN_GENERALIZED_TIME; data_ptr++; data_len++;
        /* -1 below excludes null terminator */
        *data_ptr = (byte)ASN_GENERALIZED_TIME_SIZE - 1; data_ptr++; data_len++;
        XMEMCPY(data_ptr, (byte*)uf_time, ASN_GENERALIZED_TIME_SIZE - 1);
        data_ptr += ASN_GENERALIZED_TIME_SIZE - 1;
    }
    else {
        WOLFSSL_MSG("Invalid time size returned");
        return ASN_TIME_E;
    }
    /* append null terminator */
    *data_ptr = 0;

    /* return length without null terminator */
    return data_len;
}

/* return just the time string as either UTC or Generalized Time*/
int GetFormattedTime(void* currTime, byte* buf, word32 len)
{
    struct tm* ts      = NULL;
    struct tm* tmpTime = NULL;
    int year, mon, day, hour, mini, sec;
    int ret;
#if defined(NEED_TMP_TIME)
    struct tm tmpTimeStorage;
    tmpTime = &tmpTimeStorage;
#endif
    /* Needed in case XGMTIME does not use the tmpTime argument. */
    (void)tmpTime;

    WOLFSSL_ENTER("GetFormattedTime");

    if (buf == NULL || len == 0)
        return BAD_FUNC_ARG;

    ts = (struct tm *)XGMTIME((time_t*)currTime, tmpTime);
    if (ValidateGmtime(ts)) {
        WOLFSSL_MSG("failed to get time data.");
        return ASN_TIME_E;
    }

    /* Note ASN_UTC_TIME_SIZE and ASN_GENERALIZED_TIME_SIZE include space for
     * the null terminator. ASN encoded values leave off the terminator. */

    if (ts->tm_year >= 50 && ts->tm_year < 150) {
        /* UTC Time */
        if (ts->tm_year >= 50 && ts->tm_year < 100) {
            year = ts->tm_year;
        }
        else {
            year = ts->tm_year - 100;
        }
        mon  = ts->tm_mon + 1;
        day  = ts->tm_mday;
        hour = ts->tm_hour;
        mini = ts->tm_min;
        sec  = ts->tm_sec;
        if (len < ASN_UTC_TIME_SIZE) {
            WOLFSSL_MSG("buffer for GetFormattedTime is too short.");
            return BUFFER_E;
        }
        ret = XSNPRINTF((char*)buf, len,
                        "%02d%02d%02d%02d%02d%02dZ", year, mon, day,
                        hour, mini, sec);
    }
    else {
        /* GeneralizedTime */
        year = ts->tm_year + 1900;
        mon  = ts->tm_mon + 1;
        day  = ts->tm_mday;
        hour = ts->tm_hour;
        mini = ts->tm_min;
        sec  = ts->tm_sec;
        if (len < ASN_GENERALIZED_TIME_SIZE) {
            WOLFSSL_MSG("buffer for GetFormattedTime is too short.");
            return BUFFER_E;
        }
        ret = XSNPRINTF((char*)buf, len,
                        "%4d%02d%02d%02d%02d%02dZ", year, mon, day,
                        hour, mini, sec);
    }

    return ret;
}

#endif /* !NO_ASN_TIME && !USER_TIME && !TIME_OVERRIDES &&
        * (OPENSSL_EXTRA || HAVE_PKCS7) */

#if defined(USE_WOLF_VALIDDATE)

/* to the second */
int DateGreaterThan(const struct tm* a, const struct tm* b)
{
    if (a->tm_year > b->tm_year)
        return 1;

    if (a->tm_year == b->tm_year && a->tm_mon > b->tm_mon)
        return 1;

    if (a->tm_year == b->tm_year && a->tm_mon == b->tm_mon &&
           a->tm_mday > b->tm_mday)
        return 1;

    if (a->tm_year == b->tm_year && a->tm_mon == b->tm_mon &&
        a->tm_mday == b->tm_mday && a->tm_hour > b->tm_hour)
        return 1;

    if (a->tm_year == b->tm_year && a->tm_mon == b->tm_mon &&
        a->tm_mday == b->tm_mday && a->tm_hour == b->tm_hour &&
        a->tm_min > b->tm_min)
        return 1;

    if (a->tm_year == b->tm_year && a->tm_mon == b->tm_mon &&
        a->tm_mday == b->tm_mday && a->tm_hour == b->tm_hour &&
        a->tm_min  == b->tm_min  && a->tm_sec > b->tm_sec)
        return 1;

    return 0; /* false */
}


static WC_INLINE int DateLessThan(const struct tm* a, const struct tm* b)
{
    return DateGreaterThan(b,a);
}

/* like atoi but only use first byte */
/* Make sure before and after dates are valid */
/* date = ASN.1 raw */
/* format = ASN_UTC_TIME or ASN_GENERALIZED_TIME */
/* dateType = ASN_AFTER or ASN_BEFORE */
int wc_ValidateDate(const byte* date, byte format, int dateType)
{
    time_t ltime;
    struct tm  certTime;
    struct tm* localTime;
    struct tm* tmpTime;
    int    i = 0;
    int    timeDiff = 0;
    int    diffHH = 0, diffMM = 0;

#if defined(NEED_TMP_TIME)
    struct tm tmpTimeStorage;
    tmpTime = &tmpTimeStorage;
#else
    tmpTime = NULL;
#endif
    (void)tmpTime;

    ltime = wc_Time(0);
#ifndef NO_TIME_SIGNEDNESS_CHECK
    if (sizeof(ltime) == sizeof(word32) && (sword32)ltime < 0){
        /* A negative response here could be due to a 32-bit time_t
         * where the year is 2038 or later. */
        WOLFSSL_MSG("wc_Time failed to return a valid value");
        return 0;
    }
#endif

#ifdef WOLFSSL_BEFORE_DATE_CLOCK_SKEW
    if (dateType == ASN_BEFORE) {
        WOLFSSL_MSG("Skewing local time for before date check");
        ltime += WOLFSSL_BEFORE_DATE_CLOCK_SKEW;
    }
#endif

#ifdef WOLFSSL_AFTER_DATE_CLOCK_SKEW
    if (dateType == ASN_AFTER) {
        WOLFSSL_MSG("Skewing local time for after date check");
        ltime -= WOLFSSL_AFTER_DATE_CLOCK_SKEW;
    }
#endif

    if (!ExtractDate(date, format, &certTime, &i)) {
        WOLFSSL_MSG("Error extracting the date");
        return 0;
    }

    if ((date[i] == '+') || (date[i] == '-')) {
        int diffSign;

        WOLFSSL_MSG("Using time differential, not Zulu") ;
        diffSign = date[i++] == '+' ? 1 : -1 ;
        if (GetTime(&diffHH, date, &i) != 0)
            return 0;
        if (GetTime(&diffMM, date, &i) != 0)
            return 0;
        timeDiff = diffSign * (diffHH*60 + diffMM) * 60 ;
    } else if (date[i] != 'Z') {
        WOLFSSL_MSG("UTCtime, neither Zulu or time differential") ;
        return 0;
    }

    ltime -= (time_t)timeDiff;
    localTime = XGMTIME(&ltime, tmpTime);

    if (ValidateGmtime(localTime)) {
        WOLFSSL_MSG("XGMTIME failed");
        return 0;
    }

    if (dateType == ASN_BEFORE) {
        if (DateLessThan(localTime, &certTime)) {
            WOLFSSL_MSG("Date BEFORE check failed");
            return 0;
        }
    }
    else {  /* dateType == ASN_AFTER */
        if (DateGreaterThan(localTime, &certTime)) {
            WOLFSSL_MSG("Date AFTER check failed");
            return 0;
        }
    }

    return 1;
}
#endif /* USE_WOLF_VALIDDATE */

int wc_GetTime(void* timePtr, word32 timeSize)
{
    time_t* ltime = (time_t*)timePtr;

    if (timePtr == NULL) {
        return BAD_FUNC_ARG;
    }

    if ((word32)sizeof(time_t) > timeSize) {
        return BUFFER_E;
    }

    *ltime = wc_Time(0);

    return 0;
}

#ifdef TIME_OVERRIDES
    #ifndef HAVE_TIME_T_TYPE
        typedef long time_t;
    #endif
    extern time_t XTIME(time_t* t);
#endif

static wc_time_cb timeFunc = NULL;

int wc_SetTimeCb(wc_time_cb f)
{
    timeFunc = f;
    return 0;
}

time_t wc_Time(time_t* t)
{
    if (timeFunc != NULL) {
        return timeFunc(t);
    }
    return XTIME(t);
}

#endif /* !NO_ASN_TIME */


#ifdef WOLFSSL_ASN_TEMPLATE
/* TODO: use a CHOICE instead of two items? */
/* ASN.1 template for a date - either UTC or Generalized Time. */
static const ASNItem dateASN[] = {
/* UTC */ { 0, ASN_UTC_TIME, 0, 0, 2 },
/* GT  */ { 0, ASN_GENERALIZED_TIME, 0, 0, 2 },
};
enum {
    DATEASN_IDX_UTC = 0,
    DATEASN_IDX_GT
};

/* Number of items in ASN.1 template for a date. */
#define dateASN_Length (sizeof(dateASN) / sizeof(ASNItem))
#endif /* WOLFSSL_ASN_TEMPLATE */

/* Get date buffer, format and length. Returns 0=success or error */
/* Decode a DateInfo - choice of UTC TIME or GENERALIZED TIME.
 *
 * @param [in]      source   Buffer containing encoded date.
 * @param [in, out] idx      On in, the index of the date.
 *                           On out, index after date.
 * @param [out]     pDate    Pointer into buffer of data bytes.
 * @param [out]     pFormat  Format of date - BER/DER tag.
 * @param [out]     pLength  Length of date bytes.
 * @param [in]      maxIdx   Index of next item after date.
 * @return  0 on success.
 * @return  BAD_FUNC_ARG when source or idx is NULL.
 * @return  ASN_PARSE_E when BER encoded data does not match ASN.1 items or
 *          is invalid.
 * @return  BUFFER_E when data in buffer is too small.
 */
static int GetDateInfo(const byte* source, word32* idx, const byte** pDate,
                        byte* pFormat, int* pLength, word32 maxIdx)
{
#ifndef WOLFSSL_ASN_TEMPLATE
    int length;
    byte format;

    if (source == NULL || idx == NULL)
        return BAD_FUNC_ARG;

    /* get ASN format header */
    if (*idx+1 > maxIdx)
        return BUFFER_E;
    format = source[*idx];
    *idx += 1;
    if (format != ASN_UTC_TIME && format != ASN_GENERALIZED_TIME) {
        WOLFSSL_ERROR_VERBOSE(ASN_TIME_E);
        return ASN_TIME_E;
    }

    /* get length */
    if (GetLength(source, idx, &length, maxIdx) < 0)
        return ASN_PARSE_E;
    if (length > MAX_DATE_SIZE || length < MIN_DATE_SIZE)
        return ASN_DATE_SZ_E;

    /* return format, date and length */
    if (pFormat)
        *pFormat = format;
    if (pDate)
        *pDate = &source[*idx];
    if (pLength)
        *pLength = length;

    *idx += (word32)length;

    return 0;
#else
    ASNGetData dataASN[dateASN_Length];
    int ret = 0;

    if ((source == NULL) || (idx == NULL)) {
        ret = BAD_FUNC_ARG;
    }
    if (ret == 0) {
        /* Initialize data. */
        XMEMSET(dataASN, 0, sizeof(dataASN));
        /* Parse date. */
        ret = GetASN_Items(dateASN, dataASN, dateASN_Length, 0, source, idx,
                           maxIdx);
    }
    if (ret == 0) {
        /* Determine which tag was seen. */
        int i = (dataASN[DATEASN_IDX_UTC].tag != 0) ? DATEASN_IDX_UTC
                                                    : DATEASN_IDX_GT;
        /* Return data from seen item. */
        if (pFormat != NULL) {
            *pFormat = dataASN[i].tag;
        }
        if (pDate != NULL) {
            *pDate = dataASN[i].data.ref.data;
        }
        if (pLength != NULL) {
            *pLength = (int)dataASN[i].data.ref.length;
        }
    }

    return ret;
#endif
}

#if !defined(NO_CERTS) && !defined(WOLFSSL_ASN_TEMPLATE)
static int GetDate(DecodedCert* cert, int dateType, int verify, int maxIdx)
{
    int    ret, length;
    const byte *datePtr = NULL;
    byte   date[MAX_DATE_SIZE];
    byte   format;
    word32 startIdx = 0;

    if (dateType == ASN_BEFORE)
        cert->beforeDate = &cert->source[cert->srcIdx];
    else
        cert->afterDate = &cert->source[cert->srcIdx];
    startIdx = cert->srcIdx;

    ret = GetDateInfo(cert->source, &cert->srcIdx, &datePtr, &format,
                      &length, (word32)maxIdx);
    if (ret < 0)
        return ret;

    XMEMSET(date, 0, MAX_DATE_SIZE);
    XMEMCPY(date, datePtr, (size_t)length);

    if (dateType == ASN_BEFORE)
        cert->beforeDateLen = (int)(cert->srcIdx - startIdx);
    else
        cert->afterDateLen  = (int)(cert->srcIdx - startIdx);

#ifndef NO_ASN_TIME_CHECK
    if (verify != NO_VERIFY && verify != VERIFY_SKIP_DATE &&
            (! AsnSkipDateCheck) &&
            !XVALIDATE_DATE(date, format, dateType)) {
        if (dateType == ASN_BEFORE) {
            WOLFSSL_ERROR_VERBOSE(ASN_BEFORE_DATE_E);
            return ASN_BEFORE_DATE_E;
        }
        else {
            WOLFSSL_ERROR_VERBOSE(ASN_AFTER_DATE_E);
            return ASN_AFTER_DATE_E;
        }
    }
#else
    (void)verify;
#endif

    return 0;
}

static int GetValidity(DecodedCert* cert, int verify, int maxIdx)
{
    int length;
    int badDate = 0;

    if (GetSequence(cert->source, &cert->srcIdx, &length, (word32)maxIdx) < 0)
        return ASN_PARSE_E;

    maxIdx = (int)cert->srcIdx + length;

    if (GetDate(cert, ASN_BEFORE, verify, maxIdx) < 0)
        badDate = ASN_BEFORE_DATE_E; /* continue parsing */

    if (GetDate(cert, ASN_AFTER, verify, maxIdx) < 0)
        return ASN_AFTER_DATE_E;

    if (badDate != 0)
        return badDate;

    return 0;
}
#endif /* !NO_CERTS && !WOLFSSL_ASN_TEMPLATE */


int wc_GetDateInfo(const byte* certDate, int certDateSz, const byte** date,
    byte* format, int* length)
{
    int ret;
    word32 idx = 0;

    ret = GetDateInfo(certDate, &idx, date, format, length, (word32)certDateSz);

    return ret;
}

#ifndef NO_ASN_TIME
int wc_GetDateAsCalendarTime(const byte* date, int length, byte format,
    struct tm* timearg)
{
    int idx = 0;
    (void)length;
    if (!ExtractDate(date, format, timearg, &idx))
        return ASN_TIME_E;
    return 0;
}

#if defined(WOLFSSL_CERT_GEN) && defined(WOLFSSL_ALT_NAMES)
int wc_GetCertDates(Cert* cert, struct tm* before, struct tm* after)
{
    int ret = 0;
    const byte* date;
    byte format;
    int length;

    if (cert == NULL)
        return BAD_FUNC_ARG;

    if (before && cert->beforeDateSz > 0) {
        ret = wc_GetDateInfo(cert->beforeDate, cert->beforeDateSz, &date,
                             &format, &length);
        if (ret == 0)
            ret = wc_GetDateAsCalendarTime(date, length, format, before);
    }
    if (after && cert->afterDateSz > 0) {
        ret = wc_GetDateInfo(cert->afterDate, cert->afterDateSz, &date,
                             &format, &length);
        if (ret == 0)
            ret = wc_GetDateAsCalendarTime(date, length, format, after);
    }

    return ret;
}
#endif /* WOLFSSL_CERT_GEN && WOLFSSL_ALT_NAMES */
#endif /* !NO_ASN_TIME */

#if !defined(WOLFSSL_ASN_TEMPLATE) && !defined(NO_CERTS)
static int GetSigAlg(DecodedCert* cert, word32* sigOid, word32 maxIdx)
{
    int length;
    word32 endSeqIdx;

    if (GetSequence(cert->source, &cert->srcIdx, &length, maxIdx) < 0)
        return ASN_PARSE_E;
    endSeqIdx = cert->srcIdx + (word32)length;

    if (GetObjectId(cert->source, &cert->srcIdx, sigOid, oidSigType,
                    maxIdx) < 0) {
        return ASN_OBJECT_ID_E;
    }

    if (cert->srcIdx != endSeqIdx) {
#ifdef WC_RSA_PSS
        if (*sigOid == CTC_RSASSAPSS) {
            cert->sigParamsIndex = cert->srcIdx;
            cert->sigParamsLength = endSeqIdx - cert->srcIdx;
        }
        else
#endif
        /* Only allowed a ASN NULL header with zero length. */
        if  (endSeqIdx - cert->srcIdx != 2)
            return ASN_PARSE_E;
        else {
            byte tag;
            if (GetASNTag(cert->source, &cert->srcIdx, &tag, endSeqIdx) != 0)
                return ASN_PARSE_E;
            if (tag != ASN_TAG_NULL)
                return ASN_PARSE_E;
        }
    }

    cert->srcIdx = endSeqIdx;

    return 0;
}
#endif

#ifndef NO_CERTS
#ifdef WOLFSSL_ASN_TEMPLATE
/* TODO: move code around to not require this. */
static int DecodeCertInternal(DecodedCert* cert, int verify, int* criticalExt,
                              int* badDateRet, int stopAtPubKey,
                              int stopAfterPubKey);
#endif

/* Assumes the target is a Raw-Public-Key certificate and parsed up to the
 * public key. Returns CRYPTOCB_UNAVAILABLE if it determines that the cert is
 * different from the Paw-Public-Key cert. In that case, cert->srcIdx is not
 * consumed so as succeeding parse function can take over.
 * In case that the target is Raw-Public-Key cert and contains a public key,
 * returns 0  and consumes cert->srcIdx so as a public key retrieval function
 * can follow.
 */
#if defined(HAVE_RPK)
int TryDecodeRPKToKey(DecodedCert* cert)
{
    int ret = 0, len;
    word32 tmpIdx;
    word32 oid;

    WOLFSSL_ENTER("TryDecodeRPKToKey");

    if (cert == NULL)
        return BAD_FUNC_ARG;

    tmpIdx = cert->srcIdx;

    /* both X509 cert and RPK cert should start with a Sequence tag */
    if (ret == 0) {
        if (GetSequence(cert->source, &tmpIdx, &len, cert->maxIdx) < 0)
            ret = ASN_PARSE_E;
    }
    /* TBSCertificate of X509 or AlgorithmIdentifier of RPK cert */
    if (ret == 0) {
        if (GetSequence(cert->source, &tmpIdx, &len, cert->maxIdx) < 0)
            ret = ASN_PARSE_E;
    }
    /* OBJ ID should be next in RPK cert */
    if (ret == 0) {
        if (GetObjectId(cert->source, &tmpIdx, &oid, oidKeyType, cert->maxIdx)
                                                                            < 0)
            ret = CRYPTOCB_UNAVAILABLE;
    }
    /* consume cert->srcIdx */
    if (ret == 0) {
        WOLFSSL_MSG("Looks like RPK certificate");
        cert->srcIdx = tmpIdx;
    }
    WOLFSSL_LEAVE("TryDecodeRPKToKey", ret);
    return ret;
}
#endif /* HAVE_RPK */

/* Parse the certificate up to the X.509 public key.
 *
 * If cert data is invalid then badDate get set to error value.
 *
 * @param [in, out] cert     Decoded certificate object.
 * @param [in]      verify   Whether to verify dates.
 * @param [out]     badDate  Error code when verify dates.
 * @return  0 on success.
 * @return  BAD_FUNC_ARG when cert or badDate is NULL.
 * @return  ASN_TIME_E when date BER tag is nor UTC or GENERALIZED time.
 * @return  ASN_DATE_SZ_E when time data is not supported.
 * @return  ASN_PARSE_E when BER encoded data does not match ASN.1 items or
 *          is invalid.
 * @return  BUFFER_E when data in buffer is too small.
 * @return  ASN_OBJECT_ID_E when the expected OBJECT_ID tag is not found.
 * @return  ASN_EXPECT_0_E when the INTEGER has the MSB set.
 */
int wc_GetPubX509(DecodedCert* cert, int verify, int* badDate)
{
#ifndef WOLFSSL_ASN_TEMPLATE
    int ret;

    if (cert == NULL || badDate == NULL)
        return BAD_FUNC_ARG;

    *badDate = 0;
    if ( (ret = GetCertHeader(cert)) < 0)
        return ret;

    WOLFSSL_MSG("Got Cert Header");

#ifdef WOLFSSL_CERT_REQ
    if (!cert->isCSR) {
#endif
        /* Using the sigIndex as the upper bound because that's where the
         * actual certificate data ends. */
        if ((ret = GetSigAlg(cert, &cert->signatureOID, cert->sigIndex)) < 0)
            return ret;

        WOLFSSL_MSG("Got Algo ID");

        if ( (ret = GetName(cert, ASN_ISSUER, (int)cert->sigIndex)) < 0)
            return ret;

        if ( (ret = GetValidity(cert, verify, (int)cert->sigIndex)) < 0)
            *badDate = ret;
#ifdef WOLFSSL_CERT_REQ
    }
#endif

    if ( (ret = GetName(cert, ASN_SUBJECT, (int)cert->sigIndex)) < 0)
        return ret;

    WOLFSSL_MSG("Got Subject Name");
    return ret;
#else
    /* Use common decode routine and stop at public key. */
    int ret;

    *badDate = 0;

    ret = DecodeCertInternal(cert, verify, NULL, badDate, 1, 0);
    if (ret >= 0) {
        /* Store current index: public key. */
        cert->srcIdx = (word32)ret;
    }
    return ret;
#endif /* WOLFSSL_ASN_TEMPLATE */
}

/* Parse the certificate up to and including X.509 public key.
 *
 * @param [in, out] cert     Decoded certificate object.
 * @param [in]      verify   Whether to verify dates.
 * @return  0 on success.
 * @return  ASN_TIME_E when date BER tag is nor UTC or GENERALIZED time.
 * @return  ASN_DATE_SZ_E when time data is not supported.
 * @return  ASN_BEFORE_DATE_E when ASN_BEFORE date is invalid.
 * @return  ASN_AFTER_DATE_E when ASN_AFTER date is invalid.
 * @return  ASN_PARSE_E when BER encoded data does not match ASN.1 items or
 *          is invalid.
 * @return  BUFFER_E when data in buffer is too small.
 * @return  ASN_OBJECT_ID_E when the expected OBJECT_ID tag is not found.
 * @return  ASN_BITSTR_E when the expected BIT_STRING tag is not found.
 * @return  ASN_EXPECT_0_E when the INTEGER has the MSB set.
 */
int DecodeToKey(DecodedCert* cert, int verify)
{
#ifndef WOLFSSL_ASN_TEMPLATE
    int badDate = 0;
    int ret;

#if defined(HAVE_RPK)

    /* Raw Public Key certificate has only a SubjectPublicKeyInfo structure
     * as its contents. So try to call GetCertKey to get public key from it.
     * If it fails, the cert should be a X509 cert and proceed to process as
     * x509 cert. */
    ret = GetCertKey(cert, cert->source, &cert->srcIdx, cert->maxIdx);
    if (ret == 0) {
        WOLFSSL_MSG("Raw Public Key certificate found and parsed");
        cert->isRPK = 1;
        return ret;
    }
#endif /* HAVE_RPK */

    if ( (ret = wc_GetPubX509(cert, verify, &badDate)) < 0)
        return ret;

    /* Determine if self signed */
#ifdef WOLFSSL_CERT_REQ
    if (cert->isCSR)
        cert->selfSigned = 1;
    else
#endif
    {
        cert->selfSigned = XMEMCMP(cert->issuerHash, cert->subjectHash,
            KEYID_SIZE) == 0 ? 1 : 0;
    }

    ret = GetCertKey(cert, cert->source, &cert->srcIdx, cert->maxIdx);
    if (ret != 0)
        return ret;

    WOLFSSL_MSG("Got Key");

    if (badDate != 0)
        return badDate;

    return ret;
#else
    int ret;
    int badDate = 0;

#ifdef WOLFSSL_DUAL_ALG_CERTS
    /* Call internal version and decode completely to also handle extensions.
     * This is required to parse a potential alternative public key in the
     * SubjectAlternativeKey extension. */
    ret = DecodeCertInternal(cert, verify, NULL, &badDate, 0, 0);
#else
    /* Call internal version and stop after public key. */
    ret = DecodeCertInternal(cert, verify, NULL, &badDate, 0, 1);
#endif
    /* Always return date errors. */
    if (ret == 0) {
        ret = badDate;
    }
    return ret;
#endif /* WOLFSSL_ASN_TEMPLATE */
}

#if !defined(WOLFSSL_ASN_TEMPLATE)
static int GetSignature(DecodedCert* cert)
{
    int length;
    int ret;

    ret = CheckBitString(cert->source, &cert->srcIdx, &length, cert->maxIdx, 1,
                         NULL);
    if (ret != 0)
        return ret;

    cert->sigLength = (word32)length;
    cert->signature = &cert->source[cert->srcIdx];
    cert->srcIdx += cert->sigLength;

    if (cert->srcIdx != cert->maxIdx)
        return ASN_PARSE_E;

    return 0;
}
#endif /* !WOLFSSL_ASN_TEMPLATE */
#endif /* !NO_CERTS */

#ifndef WOLFSSL_ASN_TEMPLATE
static word32 SetOctetString8Bit(word32 len, byte* output)
{
    output[0] = ASN_OCTET_STRING;
    output[1] = (byte)len;
    return 2;
}
static word32 SetDigest(const byte* digest, word32 digSz, byte* output)
{
    word32 idx = SetOctetString8Bit(digSz, output);
    XMEMCPY(&output[idx], digest, digSz);

    return idx + digSz;
}
#endif


/* Encode a length for DER.
 *
 * @param [in]  length  Value to encode.
 * @param [out] output  Buffer to encode into.
 * @return  Number of bytes encoded.
 */
word32 SetLength(word32 length, byte* output)
{
    /* Start encoding at start of buffer. */
    word32 i = 0;

    if (length < ASN_LONG_LENGTH) {
        /* Only one byte needed to encode. */
        if (output) {
            /* Write out length value. */
            output[i] = (byte)length;
        }
        /* Skip over length. */
        i++;
    }
    else {
        /* Calculate the number of bytes required to encode value. */
        byte j = (byte)BytePrecision(length);

        if (output) {
            /* Encode count byte. */
            output[i] = (byte)(j | ASN_LONG_LENGTH);
        }
        /* Skip over count byte. */
        i++;

        /* Encode value as a big-endian byte array. */
        for (; j > 0; --j) {
            if (output) {
                /* Encode next most-significant byte. */
                output[i] = (byte)(length >> ((j - 1) * WOLFSSL_BIT_SIZE));
            }
            /* Skip over byte. */
            i++;
        }
    }

    /* Return number of bytes in encoded length. */
    return i;
}

word32 SetLengthEx(word32 length, byte* output, byte isIndef)
{
    if (isIndef) {
        if (output != NULL) {
            output[0] = ASN_INDEF_LENGTH;
        }
        return 1;
    }
    else {
        return SetLength(length, output);
    }
}
/* Encode a DER header - type/tag and length.
 *
 * @param [in]  tag     DER tag of ASN.1 item.
 * @param [in]  len     Length of data in ASN.1 item.
 * @param [out] output  Buffer to encode into.
 * @return  Number of bytes encoded.
 */
word32 SetHeader(byte tag, word32 len, byte* output, byte isIndef)
{
    if (output) {
        /* Encode tag first. */
        output[0] = tag;
    }
    /* Encode the length. */
    return SetLengthEx(len, output ? output + ASN_TAG_SZ : NULL, isIndef) +
        ASN_TAG_SZ;
}

/* Encode a SEQUENCE header in DER.
 *
 * @param [in]  len     Length of data in SEQUENCE.
 * @param [out] output  Buffer to encode into.
 * @return  Number of bytes encoded.
 */
word32 SetSequence(word32 len, byte* output)
{
    return SetHeader(ASN_SEQUENCE | ASN_CONSTRUCTED, len, output, 0);
}

word32 SetSequenceEx(word32 len, byte* output, byte isIndef)
{
    return SetHeader(ASN_SEQUENCE | ASN_CONSTRUCTED, len, output, isIndef);
}

/* Encode an OCTET STRING header in DER.
 *
 * @param [in]  len     Length of data in OCTET STRING.
 * @param [out] output  Buffer to encode into.
 * @return  Number of bytes encoded.
 */
word32 SetOctetString(word32 len, byte* output)
{
    return SetHeader(ASN_OCTET_STRING, len, output, 0);
}

word32 SetOctetStringEx(word32 len, byte* output, byte indef)
{
    if (indef)
        return SetHeader(ASN_OCTET_STRING | ASN_CONSTRUCTED, len, output, indef);
    return SetOctetString(len, output);
}

/* Encode a SET header in DER.
 *
 * @param [in]  len     Length of data in SET.
 * @param [out] output  Buffer to encode into.
 * @return  Number of bytes encoded.
 */
word32 SetSet(word32 len, byte* output)
{
    return SetHeader(ASN_SET | ASN_CONSTRUCTED, len, output, 0);
}

/* Encode an implicit context specific header in DER.
 *
 * Implicit means that it is constructed only if the included ASN.1 item is.
 *
 * @param [in]  tag     Tag for the implicit ASN.1 item.
 * @param [in]  number  Context specific number.
 * @param [in]  len     Length of data in SET.
 * @param [out] output  Buffer to encode into.
 * @return  Number of bytes encoded.
 */
word32 SetImplicit(byte tag, byte number, word32 len, byte* output, byte isIndef)
{
    byte useIndef = 0;

    if ((tag == ASN_OCTET_STRING) && isIndef) {
        tag = ASN_CONSTRUCTED | ASN_CONTEXT_SPECIFIC | number;
    }
    else {
        tag = (byte)(((tag == ASN_SEQUENCE || tag == ASN_SET) ?
            ASN_CONSTRUCTED : 0) | ASN_CONTEXT_SPECIFIC | number);
    }

    if (isIndef && (tag & ASN_CONSTRUCTED)) {
        useIndef = 1;
    }

    return SetHeader(tag, len, output, useIndef);
}

/* Encode an explicit context specific header in DER.
 *
 * Explicit means that there will be an ASN.1 item underneath.
 *
 * @param [in]  number  Context specific number.
 * @param [in]  len     Length of data in SET.
 * @param [out] output  Buffer to encode into.
 * @return  Number of bytes encoded.
 */
word32 SetExplicit(byte number, word32 len, byte* output, byte isIndef)
{
    return SetHeader((byte)(ASN_CONTEXT_SPECIFIC | ASN_CONSTRUCTED | number),
                     len, output, isIndef);
}

#if defined(OPENSSL_EXTRA)
/* Encode an Othername into DER.
 *
 * @param [in]  name    Pointer to the WOLFSSL_ASN1_OTHERNAME to be encoded.
 * @param [out] output  Buffer to encode into. If NULL, don't encode.
 * @return  Number of bytes encoded or WOLFSSL_FAILURE if name parameter is bad.
 */
word32 SetOthername(void *name, byte *output)
{
    WOLFSSL_ASN1_OTHERNAME *nm = (WOLFSSL_ASN1_OTHERNAME *)name;
    char *nameStr = NULL;
    word32 nameSz = 0;

    if ((nm == NULL) || (nm->value == NULL)) {
        WOLFSSL_MSG("otherName value is NULL");
        return WOLFSSL_FAILURE;
    }

    nameStr = nm->value->value.utf8string->data;
    nameSz = (word32)nm->value->value.utf8string->length;

    if (output == NULL) {
        return nm->type_id->objSz +
            SetHeader(ASN_CONSTRUCTED | ASN_CONTEXT_SPECIFIC, nameSz + 2, NULL, 0) +
            SetHeader(CTC_UTF8, nameSz, NULL, 0) + nameSz;
    }
    else {
        const byte *output_start = output;
        /* otherName OID */
        XMEMCPY(output, nm->type_id->obj, nm->type_id->objSz);
        output += nm->type_id->objSz;

        output += SetHeader(ASN_CONSTRUCTED | ASN_CONTEXT_SPECIFIC, nameSz + 2,
                            output, 0);

        /* work around false positive from -fstack-protector */
        PRAGMA_GCC_DIAG_PUSH
        PRAGMA_GCC("GCC diagnostic ignored \"-Wstringop-overflow\"")

        output += SetHeader(CTC_UTF8, nameSz, output, 0);

        PRAGMA_GCC_DIAG_POP

        XMEMCPY(output, nameStr, nameSz);

        output += nameSz;
        return (word32)(output - output_start);
    }
}
#endif /* OPENSSL_EXTRA */


#ifdef HAVE_ECC
/* Determines whether the signature algorithm is using ECDSA.
 *
 * @param [in] algoOID  Signature algorithm identifier.
 * @return  1 when algorithm is using ECDSA.
 * @return  0 otherwise.
 */
static WC_INLINE int IsSigAlgoECDSA(word32 algoOID)
{
    /* ECDSA sigAlgo must not have ASN1 NULL parameters */
    if (algoOID == CTC_SHAwECDSA || algoOID == CTC_SHA256wECDSA ||
        algoOID == CTC_SHA384wECDSA || algoOID == CTC_SHA512wECDSA) {
        return 1;
    }

    return 0;
}
#endif

/* Determines if OID is for an EC signing algorithm including ECDSA and EdDSA
 * and post-quantum algorithms.
 *
 * @param [in] algoOID  Algorithm OID.
 * @return  1 when is EC signing algorithm.
 * @return  0 otherwise.
 */
static WC_INLINE int IsSigAlgoECC(word32 algoOID)
{
    (void)algoOID;

    return (0
        #ifdef HAVE_ECC
              || IsSigAlgoECDSA(algoOID)
        #endif
        #ifdef WOLFSSL_SM2
              || (algoOID == SM2k)
        #endif
        #ifdef HAVE_ED25519
              || (algoOID == ED25519k)
        #endif
        #ifdef HAVE_CURVE25519
              || (algoOID == X25519k)
        #endif
        #ifdef HAVE_ED448
              || (algoOID == ED448k)
        #endif
        #ifdef HAVE_CURVE448
              || (algoOID == X448k)
        #endif
        #ifdef HAVE_FACON
              || (algoOID == FALCON_LEVEL1k)
              || (algoOID == FALCON_LEVEL5k)
        #endif
        #ifdef HAVE_DILITHIUM
            #ifdef WOLFSSL_DILITHIUM_FIPS204_DRAFT
              || (algoOID == DILITHIUM_LEVEL2k)
              || (algoOID == DILITHIUM_LEVEL3k)
              || (algoOID == DILITHIUM_LEVEL5k)
            #endif
              || (algoOID == ML_DSA_LEVEL2k)
              || (algoOID == ML_DSA_LEVEL3k)
              || (algoOID == ML_DSA_LEVEL5k)
        #endif
        #ifdef HAVE_SPHINCS
              || (algoOID == SPHINCS_FAST_LEVEL1k)
              || (algoOID == SPHINCS_FAST_LEVEL3k)
              || (algoOID == SPHINCS_FAST_LEVEL5k)
              || (algoOID == SPHINCS_SMALL_LEVEL1k)
              || (algoOID == SPHINCS_SMALL_LEVEL3k)
              || (algoOID == SPHINCS_SMALL_LEVEL5k)
        #endif
    );
}

/* Encode an algorithm identifier.
 *
 * [algoOID, type] is unique.
 *
 * @param [in]  algoOID   Algorithm identifier.
 * @param [out] output    Buffer to hold encoding.
 * @param [in]  type      Type of OID being encoded.
 * @param [in]  curveSz   Add extra space for curve data.
 * @return  Encoded data size on success.
 * @return  0 when dynamic memory allocation fails.
 */
static word32 SetAlgoIDImpl(int algoOID, byte* output, int type, int curveSz,
                            byte absentParams)
{
#ifndef WOLFSSL_ASN_TEMPLATE
    word32 tagSz, idSz, seqSz, algoSz = 0;
    const  byte* algoName = 0;
    byte   ID_Length[1 + MAX_LENGTH_SZ];
    byte   seqArray[MAX_SEQ_SZ + 1];  /* add object_id to end */
    word32    length = 0;

    tagSz = ((type == oidHashType ||
             (type == oidSigType && !IsSigAlgoECC((word32)algoOID)) ||
             (type == oidKeyType && algoOID == RSAk)) &&
                (absentParams == FALSE)) ? 2U : 0U;
    algoName = OidFromId((word32)algoOID, (word32)type, &algoSz);
    if (algoName == NULL) {
        WOLFSSL_MSG("Unknown Algorithm");
        return 0;
    }

    idSz  = (word32)SetObjectId((int)algoSz, ID_Length);
    seqSz = SetSequence(idSz + algoSz + tagSz + (word32)curveSz, seqArray);

    /* Copy only algo to output for DSA keys */
    if (algoOID == DSAk && output) {
        XMEMCPY(output, ID_Length, idSz);
        XMEMCPY(output + idSz, algoName, algoSz);
        if (tagSz == 2)
            SetASNNull(&output[seqSz + idSz + algoSz]);
    }
    else if (output) {
        XMEMCPY(output, seqArray, seqSz);
        XMEMCPY(output + seqSz, ID_Length, idSz);
        XMEMCPY(output + seqSz + idSz, algoName, algoSz);
        if (tagSz == 2)
            SetASNNull(&output[seqSz + idSz + algoSz]);
    }

    if (algoOID == DSAk)
        length = idSz + algoSz + tagSz;
    else
        length = seqSz + idSz + algoSz + tagSz;

    return length;
#else
    DECL_ASNSETDATA(dataASN, algoIdASN_Length);
    int ret = 0;
    const byte* algoName = 0;
    word32 algoSz = 0;

    CALLOC_ASNSETDATA(dataASN, algoIdASN_Length, ret, NULL);

    algoName = OidFromId((word32)algoOID, (word32)type, &algoSz);
    if (algoName == NULL) {
        WOLFSSL_MSG("Unknown Algorithm");
    }
    else {
        int sz;
        int o = 0;

        /* Set the OID and OID type to encode. */
        SetASN_OID(&dataASN[ALGOIDASN_IDX_OID], (word32)algoOID, (word32)type);
        /* Hashes, signatures not ECC and keys not RSA output NULL tag. */
        if (!(type == oidHashType ||
                 (type == oidSigType && !IsSigAlgoECC((word32)algoOID)) ||
                 (type == oidKeyType && algoOID == RSAk))) {
            /* Don't put out NULL DER item. */
            dataASN[ALGOIDASN_IDX_NULL].noOut = 1;
        }
        /* Override for absent (not NULL) params */
        if (TRUE == absentParams) {
            dataASN[ALGOIDASN_IDX_NULL].noOut = 1;
        }
        if (algoOID == DSAk) {
            /* Don't include SEQUENCE for DSA keys. */
            o = 1;
        }
        else if (curveSz > 0) {
            /* Don't put out NULL DER item. */
            dataASN[ALGOIDASN_IDX_NULL].noOut = 0;
            /* Include space for extra data of length curveSz.
             * Subtract 1 for sequence and 1 for length encoding. */
            SetASN_Buffer(&dataASN[ALGOIDASN_IDX_NULL], NULL,
                          (word32)curveSz - 2);
        }

        /* Calculate size of encoding. */
        ret = SizeASN_Items(algoIdASN + o, dataASN + o,
                            (int)algoIdASN_Length - (int)o, &sz);
        if (ret == 0 && output != NULL) {
            /* Encode into buffer. */
            SetASN_Items(algoIdASN + o, dataASN + o,
                         (int)algoIdASN_Length - (int)o, output);
            if (curveSz > 0) {
                /* Return size excluding curve data. */
                sz = (int)(dataASN[o].offset -
                           dataASN[ALGOIDASN_IDX_NULL].offset);
            }
        }

        if (ret == 0) {
            /* Return encoded size. */
            ret = sz;
        }
        else {
            /* Unsigned return type so 0 indicates error. */
            ret = 0;
        }
    }

    FREE_ASNSETDATA(dataASN, NULL);
    return (word32)ret;
#endif /* WOLFSSL_ASN_TEMPLATE */
}

/* Encode an algorithm identifier.
 *
 * [algoOID, type] is unique.
 *
 * @param [in]  algoOID   Algorithm identifier.
 * @param [out] output    Buffer to hold encoding.
 * @param [in]  type      Type of OID being encoded.
 * @param [in]  curveSz   Add extra space for curve data.
 * @return  Encoded data size on success.
 * @return  0 when dynamic memory allocation fails.
 */
word32 SetAlgoID(int algoOID, byte* output, int type, int curveSz)
{
    return SetAlgoIDImpl(algoOID, output, type, curveSz, FALSE);
}

word32 SetAlgoIDEx(int algoOID, byte* output, int type, int curveSz,
                   byte absentParams)
{
    return SetAlgoIDImpl(algoOID, output, type, curveSz, absentParams);
}

#ifdef WOLFSSL_ASN_TEMPLATE
/* Always encode PKCS#1 v1.5 RSA signature and compare to encoded data. */
/* ASN.1 template for DigestInfo for a PKCS#1 v1.5 RSA signature.
 * PKCS#1 v2.2: RFC 8017, A.2.4 - DigestInfo
 */
static const ASNItem digestInfoASN[] = {
/* SEQ          */ { 0, ASN_SEQUENCE, 1, 1, 0 },
                                         /* digestAlgorithm */
/* DIGALGO_SEQ  */     { 1, ASN_SEQUENCE, 1, 1, 0 },
/* DIGALGO_OID  */         { 2, ASN_OBJECT_ID, 0, 0, 0 },
/* DIGALGO_NULL */         { 2, ASN_TAG_NULL, 0, 0, 0 },
                                         /* digest */
/* DIGEST       */     { 1, ASN_OCTET_STRING, 0, 0, 0 }
};
enum {
    DIGESTINFOASN_IDX_SEQ = 0,
    DIGESTINFOASN_IDX_DIGALGO_SEQ,
    DIGESTINFOASN_IDX_DIGALGO_OID,
    DIGESTINFOASN_IDX_DIGALGO_NULL,
    DIGESTINFOASN_IDX_DIGEST
};

/* Number of items in ASN.1 template for DigestInfo for RSA. */
#define digestInfoASN_Length (sizeof(digestInfoASN) / sizeof(ASNItem))
#endif

/* Encode signature.
 *
 * @param [out] out     Buffer to hold encoding.
 * @param [in]  digest  Buffer holding digest.
 * @param [in]  digSz   Length of digest in bytes.
 * @return  Encoded data size on success.
 * @return  0 when dynamic memory allocation fails.
 */
word32 wc_EncodeSignature(byte* out, const byte* digest, word32 digSz,
                          int hashOID)
{
#ifndef WOLFSSL_ASN_TEMPLATE
    byte digArray[MAX_ENCODED_DIG_SZ];
    byte algoArray[MAX_ALGO_SZ];
    byte seqArray[MAX_SEQ_SZ];
    word32 encDigSz, algoSz, seqSz;

    encDigSz = SetDigest(digest, digSz, digArray);
    algoSz   = SetAlgoID(hashOID, algoArray, oidHashType, 0);
    seqSz    = SetSequence(encDigSz + algoSz, seqArray);

    XMEMCPY(out, seqArray, seqSz);
    XMEMCPY(out + seqSz, algoArray, algoSz);
    XMEMCPY(out + seqSz + algoSz, digArray, encDigSz);

    return encDigSz + algoSz + seqSz;
#else
    DECL_ASNSETDATA(dataASN, digestInfoASN_Length);
    int ret = 0;
    int sz = 0;
    unsigned char dgst[WC_MAX_DIGEST_SIZE];

    CALLOC_ASNSETDATA(dataASN, digestInfoASN_Length, ret, NULL);

    if (ret == 0) {
        /* Set hash OID and type. */
        SetASN_OID(&dataASN[DIGESTINFOASN_IDX_DIGALGO_OID], (word32)hashOID,
                   oidHashType);
        /* Set digest. */
        if (digest == out) {
            XMEMCPY(dgst, digest, digSz);
            digest = dgst;
        }
        SetASN_Buffer(&dataASN[DIGESTINFOASN_IDX_DIGEST], digest, digSz);

        /* Calculate size of encoding. */
        ret = SizeASN_Items(digestInfoASN, dataASN, digestInfoASN_Length, &sz);
    }
    if (ret == 0) {
        /* Encode PKCS#1 v1.5 RSA signature. */
        SetASN_Items(digestInfoASN, dataASN, digestInfoASN_Length, out);
        ret = sz;
    }
    else {
        /* Unsigned return type so 0 indicates error. */
        ret = 0;
    }

    FREE_ASNSETDATA(dataASN, NULL);
    return (word32)ret;
#endif
}


#ifndef NO_CERTS

int wc_GetCTC_HashOID(int type)
{
    int ret;
    enum wc_HashType hType;

    hType = wc_HashTypeConvert(type);
    ret = wc_HashGetOID(hType);
    if (ret < 0) {
        ret = 0; /* backwards compatibility */
    }

    return ret;
}

/* Initialize a signature context object.
 *
 * Object used for signing and verifying a certificate signature.
 *
 * @param [in, out] sigCtx  Signature context object.
 * @param [in]      heap    Dynamic memory hint.
 * @param [in]      devId   Hardware device identifier.
 */
void InitSignatureCtx(SignatureCtx* sigCtx, void* heap, int devId)
{
    if (sigCtx) {
        XMEMSET(sigCtx, 0, sizeof(SignatureCtx));
        sigCtx->devId = devId;
        sigCtx->heap = heap;
    }
}

/* Free dynamic data in a signature context object.
 *
 * @param [in, out] sigCtx  Signature context object.
 */
void FreeSignatureCtx(SignatureCtx* sigCtx)
{
    if (sigCtx == NULL)
        return;

#ifndef WOLFSSL_NO_MALLOC
    XFREE(sigCtx->digest, sigCtx->heap, DYNAMIC_TYPE_DIGEST);
    sigCtx->digest = NULL;
#if !defined(NO_RSA) || !defined(NO_DSA)
    XFREE(sigCtx->sigCpy, sigCtx->heap, DYNAMIC_TYPE_SIGNATURE);
    sigCtx->sigCpy = NULL;
#endif
#endif

#ifndef NO_ASN_CRYPT
#ifndef WOLFSSL_NO_MALLOC
    if (sigCtx->key.ptr)
#endif
    {
        switch (sigCtx->keyOID) {
        #ifndef NO_RSA
            #ifdef WC_RSA_PSS
            case RSAPSSk:
            #endif
            case RSAk:
                wc_FreeRsaKey(sigCtx->key.rsa);
            #ifndef WOLFSSL_NO_MALLOC
                XFREE(sigCtx->key.rsa, sigCtx->heap, DYNAMIC_TYPE_RSA);
                sigCtx->key.rsa = NULL;
            #endif
                break;
        #endif /* !NO_RSA */
        #ifndef NO_DSA
            case DSAk:
                wc_FreeDsaKey(sigCtx->key.dsa);
            #ifndef WOLFSSL_NO_MALLOC
                XFREE(sigCtx->key.dsa, sigCtx->heap, DYNAMIC_TYPE_DSA);
                sigCtx->key.dsa = NULL;
            #endif
                break;
        #endif
        #ifdef HAVE_ECC
            case ECDSAk:
        #ifdef WOLFSSL_SM2
            case SM2k:
        #endif
            #if defined(WC_ECC_NONBLOCK) && defined(WOLFSSL_ASYNC_CRYPT_SW) && \
                defined(WC_ASYNC_ENABLE_ECC)
                if (sigCtx->key.ecc->nb_ctx != NULL) {
                    XFREE(sigCtx->key.ecc->nb_ctx, sigCtx->heap,
                          DYNAMIC_TYPE_TMP_BUFFER);
                }
            #endif /* WC_ECC_NONBLOCK && WOLFSSL_ASYNC_CRYPT_SW &&
                      WC_ASYNC_ENABLE_ECC */
                wc_ecc_free(sigCtx->key.ecc);
            #ifndef WOLFSSL_NO_MALLOC
                XFREE(sigCtx->key.ecc, sigCtx->heap, DYNAMIC_TYPE_ECC);
                sigCtx->key.ecc = NULL;
            #endif
                break;
        #endif /* HAVE_ECC */
        #ifdef HAVE_ED25519
            case ED25519k:
                wc_ed25519_free(sigCtx->key.ed25519);
            #ifndef WOLFSSL_NO_MALLOC
                XFREE(sigCtx->key.ed25519, sigCtx->heap, DYNAMIC_TYPE_ED25519);
                sigCtx->key.ed25519 = NULL;
            #endif
                break;
        #endif /* HAVE_ED25519 */
        #ifdef HAVE_ED448
            case ED448k:
                wc_ed448_free(sigCtx->key.ed448);
            #ifndef WOLFSSL_NO_MALLOC
                XFREE(sigCtx->key.ed448, sigCtx->heap, DYNAMIC_TYPE_ED448);
                sigCtx->key.ed448 = NULL;
            #endif
                break;
        #endif /* HAVE_ED448 */
        #if defined(HAVE_FALCON)
            case FALCON_LEVEL1k:
            case FALCON_LEVEL5k:
                wc_falcon_free(sigCtx->key.falcon);
            #ifndef WOLFSSL_NO_MALLOC
                XFREE(sigCtx->key.falcon, sigCtx->heap, DYNAMIC_TYPE_FALCON);
                sigCtx->key.falcon = NULL;
            #endif
                break;
        #endif /* HAVE_FALCON */
        #if defined(HAVE_DILITHIUM)
            #ifdef WOLFSSL_DILITHIUM_FIPS204_DRAFT
            case DILITHIUM_LEVEL2k:
            case DILITHIUM_LEVEL3k:
            case DILITHIUM_LEVEL5k:
            #endif
            case ML_DSA_LEVEL2k:
            case ML_DSA_LEVEL3k:
            case ML_DSA_LEVEL5k:
                wc_dilithium_free(sigCtx->key.dilithium);
            #ifndef WOLFSSL_NO_MALLOC
                XFREE(sigCtx->key.dilithium, sigCtx->heap,
                    DYNAMIC_TYPE_DILITHIUM);
                sigCtx->key.dilithium = NULL;
            #endif
                break;
        #endif /* HAVE_DILITHIUM */
        #if defined(HAVE_SPHINCS)
            case SPHINCS_FAST_LEVEL1k:
            case SPHINCS_FAST_LEVEL3k:
            case SPHINCS_FAST_LEVEL5k:
            case SPHINCS_SMALL_LEVEL1k:
            case SPHINCS_SMALL_LEVEL3k:
            case SPHINCS_SMALL_LEVEL5k:
                wc_sphincs_free(sigCtx->key.sphincs);
            #ifndef WOLFSSL_NO_MALLOC
                XFREE(sigCtx->key.sphincs, sigCtx->heap, DYNAMIC_TYPE_SPHINCS);
                sigCtx->key.sphincs = NULL;
            #endif
                break;
        #endif /* HAVE_SPHINCS */
            default:
                break;
        } /* switch (keyOID) */
    #ifndef WOLFSSL_NO_MALLOC
        sigCtx->key.ptr = NULL;
    #endif
    }
#endif /* !NO_ASN_CRYPT */

    /* reset state, we are done */
    sigCtx->state = SIG_STATE_BEGIN;
}

#if !defined(NO_ASN_CRYPT) && !defined(NO_HASH_WRAPPER)
static int HashForSignature(const byte* buf, word32 bufSz, word32 sigOID,
                            byte* digest, int* typeH, int* digestSz, int verify,
                            void* heap, int devId)
{
    int ret = 0;

    switch (sigOID) {
    #if defined(WOLFSSL_MD2)
        case CTC_MD2wRSA:
            if (!verify) {
                ret = HASH_TYPE_E;
                WOLFSSL_MSG("MD2 not supported for signing");
            }
            else if ((ret = wc_Md2Hash(buf, bufSz, digest)) == 0) {
                *typeH    = MD2h;
                *digestSz = WC_MD2_DIGEST_SIZE;
            }
        break;
    #endif
    #ifndef NO_MD5
        case CTC_MD5wRSA:
            if ((ret = wc_Md5Hash_ex(buf, bufSz, digest, heap, devId)) == 0) {
                *typeH    = MD5h;
                *digestSz = WC_MD5_DIGEST_SIZE;
            }
            break;
    #endif
    #ifndef NO_SHA
        case CTC_SHAwRSA:
        case CTC_SHAwDSA:
        case CTC_SHAwECDSA:
            if ((ret = wc_ShaHash_ex(buf, bufSz, digest, heap, devId)) == 0) {
                *typeH    = SHAh;
                *digestSz = WC_SHA_DIGEST_SIZE;
            }
            break;
    #endif
    #ifdef WOLFSSL_SHA224
        case CTC_SHA224wRSA:
        case CTC_SHA224wECDSA:
            if ((ret = wc_Sha224Hash_ex(buf, bufSz, digest, heap, devId)) == 0)
            {
                *typeH    = SHA224h;
                *digestSz = WC_SHA224_DIGEST_SIZE;
            }
            break;
    #endif
    #ifndef NO_SHA256
        case CTC_SHA256wRSA:
        case CTC_SHA256wECDSA:
        case CTC_SHA256wDSA:
            if ((ret = wc_Sha256Hash_ex(buf, bufSz, digest, heap, devId)) == 0)
            {
                *typeH    = SHA256h;
                *digestSz = WC_SHA256_DIGEST_SIZE;
            }
            break;
    #endif
    #ifdef WOLFSSL_SHA384
        case CTC_SHA384wRSA:
        case CTC_SHA384wECDSA:
            if ((ret = wc_Sha384Hash_ex(buf, bufSz, digest, heap, devId)) == 0)
            {
                *typeH    = SHA384h;
                *digestSz = WC_SHA384_DIGEST_SIZE;
            }
            break;
    #endif
    #ifdef WOLFSSL_SHA512
        case CTC_SHA512wRSA:
        case CTC_SHA512wECDSA:
            if ((ret = wc_Sha512Hash_ex(buf, bufSz, digest, heap, devId)) == 0)
            {
                *typeH    = SHA512h;
                *digestSz = WC_SHA512_DIGEST_SIZE;
            }
            break;
    #endif
    #ifdef WOLFSSL_SHA3
    #ifndef WOLFSSL_NOSHA3_224
        case CTC_SHA3_224wRSA:
        case CTC_SHA3_224wECDSA:
            if ((ret = wc_Sha3_224Hash_ex(buf, bufSz, digest, heap, devId))
                    == 0) {
                *typeH    = SHA3_224h;
                *digestSz = WC_SHA3_224_DIGEST_SIZE;
            }
            break;
    #endif
    #ifndef WOLFSSL_NOSHA3_256
        case CTC_SHA3_256wRSA:
        case CTC_SHA3_256wECDSA:
            if ((ret = wc_Sha3_256Hash_ex(buf, bufSz, digest, heap, devId))
                    == 0) {
                *typeH    = SHA3_256h;
                *digestSz = WC_SHA3_256_DIGEST_SIZE;
            }
            break;
    #endif
    #ifndef WOLFSSL_NOSHA3_384
        case CTC_SHA3_384wRSA:
        case CTC_SHA3_384wECDSA:
            if ((ret = wc_Sha3_384Hash_ex(buf, bufSz, digest, heap, devId))
                    == 0) {
                *typeH    = SHA3_384h;
                *digestSz = WC_SHA3_384_DIGEST_SIZE;
            }
            break;
    #endif
    #ifndef WOLFSSL_NOSHA3_512
        case CTC_SHA3_512wRSA:
        case CTC_SHA3_512wECDSA:
            if ((ret = wc_Sha3_512Hash_ex(buf, bufSz, digest, heap, devId))
                    == 0) {
                *typeH    = SHA3_512h;
                *digestSz = WC_SHA3_512_DIGEST_SIZE;
            }
            break;
    #endif
    #endif
    #if defined(WOLFSSL_SM2) & defined(WOLFSSL_SM3)
        case CTC_SM3wSM2:
            if ((ret = wc_Sm3Hash_ex(buf, bufSz, digest, heap, devId)) == 0) {
                *typeH    = SM3h;
                *digestSz = WC_SM3_DIGEST_SIZE;
            }
            break;
    #endif
    #ifdef HAVE_ED25519
        case CTC_ED25519:
            /* Hashes done in signing operation.
             * Two dependent hashes with prefixes performed.
             */
            break;
    #endif
    #ifdef HAVE_ED448
        case CTC_ED448:
            /* Hashes done in signing operation.
             * Two dependent hashes with prefixes performed.
             */
            break;
    #endif
    #ifdef HAVE_FALCON
        case CTC_FALCON_LEVEL1:
        case CTC_FALCON_LEVEL5:
            /* Hashes done in signing operation. */
            break;
    #endif
    #ifdef HAVE_DILITHIUM
        #ifdef WOLFSSL_DILITHIUM_FIPS204_DRAFT
        case CTC_DILITHIUM_LEVEL2:
        case CTC_DILITHIUM_LEVEL3:
        case CTC_DILITHIUM_LEVEL5:
        #endif
        case CTC_ML_DSA_LEVEL2:
        case CTC_ML_DSA_LEVEL3:
        case CTC_ML_DSA_LEVEL5:
            /* Hashes done in signing operation. */
            break;
    #endif
    #ifdef HAVE_SPHINCS
        case CTC_SPHINCS_FAST_LEVEL1:
        case CTC_SPHINCS_FAST_LEVEL3:
        case CTC_SPHINCS_FAST_LEVEL5:
        case CTC_SPHINCS_SMALL_LEVEL1:
        case CTC_SPHINCS_SMALL_LEVEL3:
        case CTC_SPHINCS_SMALL_LEVEL5:
            /* Hashes done in signing operation. */
            break;
    #endif

        default:
            ret = HASH_TYPE_E;
            WOLFSSL_MSG("Hash for Signature has unsupported type");
    }

    (void)buf;
    (void)bufSz;
    (void)sigOID;
    (void)digest;
    (void)digestSz;
    (void)typeH;
    (void)verify;

    return ret;
}
#endif /* !NO_ASN_CRYPT && !NO_HASH_WRAPPER */

#if !defined(NO_DSA) && !defined(HAVE_SELFTEST)
/* Try to parse as ASN.1 bitstring */
static int DecodeDsaAsn1Sig(const byte* sig, word32 sigSz, byte* sigCpy,
    void* heap)
{
    int ret = 0;
    int rSz = 0, sSz = 0, mpinit = 0;
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    mp_int* r = NULL;
    mp_int* s = NULL;
#else
    mp_int r[1];
    mp_int s[1];
#endif

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    r = (mp_int*)XMALLOC(sizeof(*r), heap, DYNAMIC_TYPE_TMP_BUFFER);
    s = (mp_int*)XMALLOC(sizeof(*s), heap, DYNAMIC_TYPE_TMP_BUFFER);
    if (r == NULL || s == NULL) {
        ret = MEMORY_E;
    }
#endif
    if (ret == 0) {
        ret = mp_init_multi(r, s, NULL, NULL, NULL, NULL);
    }
    if (ret == 0) {
        mpinit = 1;

        if (DecodeECC_DSA_Sig(sig, sigSz, r, s) != 0) {
            WOLFSSL_MSG("DSA sig decode ASN.1 failed!");
            ret = ASN_SIG_CONFIRM_E;
        }
    }
    if (ret == 0) {
        rSz = mp_unsigned_bin_size(r);
        sSz = mp_unsigned_bin_size(s);
        if (rSz + sSz > (int)sigSz) {
            WOLFSSL_MSG("DSA sig size invalid");
            ret = ASN_SIG_CONFIRM_E;
        }
    }
    if (ret == 0) {
        if (mp_to_unsigned_bin(r, sigCpy) != MP_OKAY ||
            mp_to_unsigned_bin(s, sigCpy + rSz) != MP_OKAY) {
            WOLFSSL_MSG("DSA sig to unsigned bin failed!");
            ret = ASN_SIG_CONFIRM_E;
        }
    }

    if (mpinit) {
        mp_free(r);
        mp_free(s);
    }
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    XFREE(r, heap, DYNAMIC_TYPE_TMP_BUFFER);
    XFREE(s, heap, DYNAMIC_TYPE_TMP_BUFFER);
#endif
    (void)heap;
    return ret;
}
#endif

/* Return codes: 0=Success, Negative (see error-crypt.h), ASN_SIG_CONFIRM_E */
int ConfirmSignature(SignatureCtx* sigCtx,
    const byte* buf, word32 bufSz,
    const byte* key, word32 keySz, word32 keyOID,
    const byte* sig, word32 sigSz, word32 sigOID,
    const byte* sigParams, word32 sigParamsSz,
    byte* rsaKeyIdx)
{
    int ret = WC_NO_ERR_TRACE(ASN_SIG_CONFIRM_E); /* default to failure */

#if defined(WOLFSSL_RENESAS_TSIP_TLS) || defined(WOLFSSL_RENESAS_FSPSM_TLS)
    CertAttribute* certatt = NULL;
#endif

    if (sigCtx == NULL || buf == NULL || bufSz == 0 || key == NULL ||
        keySz == 0 || sig == NULL || sigSz == 0) {
        return BAD_FUNC_ARG;
    }

    (void)key;
    (void)keySz;
    (void)sig;
    (void)sigSz;
    (void)sigParams;
    (void)sigParamsSz;

    WOLFSSL_ENTER("ConfirmSignature");

#if !defined(WOLFSSL_RENESAS_TSIP_TLS) && !defined(WOLFSSL_RENESAS_FSPSM_TLS)
    (void)rsaKeyIdx;
#else
    #if !defined(NO_RSA) || defined(HAVE_ECC)
    certatt = (CertAttribute*)&sigCtx->CertAtt;
    #endif
    if (certatt) {
        certatt->keyIndex = rsaKeyIdx;
        certatt->cert = buf;
        certatt->certSz = bufSz;
    }
#endif

#ifndef NO_ASN_CRYPT
    switch (sigCtx->state) {
        case SIG_STATE_BEGIN:
        {
            sigCtx->keyOID = keyOID; /* must set early for cleanup */

#ifndef WOLFSSL_NO_MALLOC
            sigCtx->digest = (byte*)XMALLOC(WC_MAX_DIGEST_SIZE, sigCtx->heap,
                                                    DYNAMIC_TYPE_DIGEST);
            if (sigCtx->digest == NULL) {
                ERROR_OUT(MEMORY_E, exit_cs);
            }
#endif

        #if !defined(NO_RSA) && defined(WC_RSA_PSS)
            /* RSA PSS Defaults */
            sigCtx->hash = WC_HASH_TYPE_SHA;
            sigCtx->mgf = WC_MGF1SHA1;
            sigCtx->saltLen = 20;
        #endif

            sigCtx->state = SIG_STATE_HASH;
        } /* SIG_STATE_BEGIN */
        FALL_THROUGH;

        case SIG_STATE_HASH:
        {
        #if !defined(NO_RSA) && defined(WC_RSA_PSS)
            if (sigOID == RSAPSSk) {
                word32 fakeSigOID = 0;
                ret = DecodeRsaPssParams(sigParams, sigParamsSz, &sigCtx->hash,
                    &sigCtx->mgf, &sigCtx->saltLen);
                if (ret != 0) {
                    goto exit_cs;
                }
                ret = RsaPssHashOidToSigOid(sigCtx->hash, &fakeSigOID);
                if (ret != 0) {
                    goto exit_cs;
                }
                /* Decode parameters. */
                ret = HashForSignature(buf, bufSz, fakeSigOID, sigCtx->digest,
                    &sigCtx->typeH, &sigCtx->digestSz, 1, sigCtx->heap,
                    sigCtx->devId);
                if (ret != 0) {
                    goto exit_cs;
                }
            }
            else
        #endif
        #if defined(WOLFSSL_SM2) && defined(WOLFSSL_SM3)
            if (sigOID == CTC_SM3wSM2) {
                ; /* SM2 hash requires public key. Done later. */
            }
            else
        #endif
            {
                ret = HashForSignature(buf, bufSz, sigOID, sigCtx->digest,
                                       &sigCtx->typeH, &sigCtx->digestSz, 1,
                                       sigCtx->heap, sigCtx->devId);
                if (ret != 0) {
                    goto exit_cs;
                }
            }

            sigCtx->state = SIG_STATE_KEY;
        } /* SIG_STATE_HASH */
        FALL_THROUGH;

        case SIG_STATE_KEY:
        {
            switch (keyOID) {
            #ifndef NO_RSA
                #ifdef WC_RSA_PSS
                case RSAPSSk:
                #endif
                case RSAk:
                {
                    word32 idx = 0;

                #ifndef WOLFSSL_NO_MALLOC
                    sigCtx->key.rsa = (RsaKey*)XMALLOC(sizeof(RsaKey),
                                                sigCtx->heap, DYNAMIC_TYPE_RSA);
                    if (sigCtx->key.rsa == NULL) {
                        ERROR_OUT(MEMORY_E, exit_cs);
                    }
                #endif
                    if ((ret = wc_InitRsaKey_ex(sigCtx->key.rsa, sigCtx->heap,
                                                        sigCtx->devId)) != 0) {
                        goto exit_cs;
                    }
                #ifndef WOLFSSL_NO_MALLOC
                    sigCtx->sigCpy = (byte*)XMALLOC(sigSz, sigCtx->heap,
                                                        DYNAMIC_TYPE_SIGNATURE);
                    if (sigCtx->sigCpy == NULL) {
                        ERROR_OUT(MEMORY_E, exit_cs);
                    }
                #endif
                    if (sigSz > MAX_ENCODED_SIG_SZ) {
                        WOLFSSL_MSG("Verify Signature is too big");
                        ERROR_OUT(BUFFER_E, exit_cs);
                    }
                    if ((ret = wc_RsaPublicKeyDecode(key, &idx, sigCtx->key.rsa,
                                                                 keySz)) != 0) {
                        WOLFSSL_MSG("ASN Key decode error RSA");
                        WOLFSSL_ERROR_VERBOSE(ret);
                        goto exit_cs;
                    }
                    XMEMCPY(sigCtx->sigCpy, sig, sigSz);
                    sigCtx->out = NULL;

                #ifdef WOLFSSL_ASYNC_CRYPT
                    sigCtx->asyncDev = &sigCtx->key.rsa->asyncDev;
                #endif
                    break;
                }
            #endif /* !NO_RSA */
            #if !defined(NO_DSA) && !defined(HAVE_SELFTEST)
                case DSAk:
                {
                    word32 idx = 0;

                    if (sigSz < DSA_MIN_SIG_SIZE) {
                        WOLFSSL_MSG("Verify Signature is too small");
                        ERROR_OUT(BUFFER_E, exit_cs);
                    }
                #ifndef WOLFSSL_NO_MALLOC
                    sigCtx->key.dsa = (DsaKey*)XMALLOC(sizeof(DsaKey),
                                                sigCtx->heap, DYNAMIC_TYPE_DSA);
                    if (sigCtx->key.dsa == NULL) {
                        ERROR_OUT(MEMORY_E, exit_cs);
                    }
                #endif
                    if ((ret = wc_InitDsaKey_h(sigCtx->key.dsa, sigCtx->heap)) != 0) {
                        WOLFSSL_MSG("wc_InitDsaKey_h error");
                        goto exit_cs;
                    }
                #ifndef WOLFSSL_NO_MALLOC
                    sigCtx->sigCpy = (byte*)XMALLOC(sigSz,
                                         sigCtx->heap, DYNAMIC_TYPE_SIGNATURE);
                    if (sigCtx->sigCpy == NULL) {
                        ERROR_OUT(MEMORY_E, exit_cs);
                    }
                #endif
                    if ((ret = wc_DsaPublicKeyDecode(key, &idx, sigCtx->key.dsa,
                                                                 keySz)) != 0) {
                        WOLFSSL_MSG("ASN Key decode error DSA");
                        WOLFSSL_ERROR_VERBOSE(ret);
                        goto exit_cs;
                    }
                    if (sigSz != DSA_160_SIG_SIZE &&
                        sigSz != DSA_256_SIG_SIZE) {
                        ret = DecodeDsaAsn1Sig(sig, sigSz, sigCtx->sigCpy,
                            sigCtx->heap);
                    }
                    else {
                        XMEMCPY(sigCtx->sigCpy, sig, sigSz);
                    }
                    break;
                }
            #endif /* !NO_DSA && !HAVE_SELFTEST */
            #ifdef HAVE_ECC
            #if defined(WOLFSSL_SM2) && defined(WOLFSSL_SM3)
                case SM2k:
            #endif
                case ECDSAk:
                {
                    word32 idx = 0;
            #if defined(WC_ECC_NONBLOCK) && defined(WOLFSSL_ASYNC_CRYPT_SW) && \
                defined(WC_ASYNC_ENABLE_ECC)
                    ecc_nb_ctx_t* nbCtx;
            #endif /* WC_ECC_NONBLOCK && WOLFSSL_ASYNC_CRYPT_SW &&
                      WC_ASYNC_ENABLE_ECC */

                    sigCtx->verify = 0;
            #ifndef WOLFSSL_NO_MALLOC
                    sigCtx->key.ecc = (ecc_key*)XMALLOC(sizeof(ecc_key),
                                                sigCtx->heap, DYNAMIC_TYPE_ECC);
                    if (sigCtx->key.ecc == NULL) {
                        ERROR_OUT(MEMORY_E, exit_cs);
                    }
            #endif
                    if ((ret = wc_ecc_init_ex(sigCtx->key.ecc, sigCtx->heap,
                                                          sigCtx->devId)) < 0) {
                        goto exit_cs;
                    }
            #if defined(WC_ECC_NONBLOCK) && defined(WOLFSSL_ASYNC_CRYPT_SW) && \
                defined(WC_ASYNC_ENABLE_ECC)
                    nbCtx = (ecc_nb_ctx_t*)XMALLOC(sizeof(ecc_nb_ctx_t),
                                sigCtx->heap, DYNAMIC_TYPE_TMP_BUFFER);
                    if (nbCtx == NULL) {
                        ERROR_OUT(MEMORY_E, exit_cs);
                    }

                    ret = wc_ecc_set_nonblock(sigCtx->key.ecc, nbCtx);
                    if (ret != 0) {
                        goto exit_cs;
                    }

            #endif /* WC_ECC_NONBLOCK && WOLFSSL_ASYNC_CRYPT_SW &&
                      WC_ASYNC_ENABLE_ECC */
                    ret = wc_EccPublicKeyDecode(key, &idx, sigCtx->key.ecc,
                                                                         keySz);
                    if (ret < 0) {
                        WOLFSSL_MSG("ASN Key import error ECC");
                        WOLFSSL_ERROR_VERBOSE(ret);
                        goto exit_cs;
                    }
                #ifdef WOLFSSL_ASYNC_CRYPT
                    sigCtx->asyncDev = &sigCtx->key.ecc->asyncDev;
                #endif
                    break;
                }
            #endif /* HAVE_ECC */
            #if defined(HAVE_ED25519) && defined(HAVE_ED25519_KEY_IMPORT)
                case ED25519k:
                {
                    sigCtx->verify = 0;
                #ifndef WOLFSSL_NO_MALLOC
                    sigCtx->key.ed25519 = (ed25519_key*)XMALLOC(
                                              sizeof(ed25519_key), sigCtx->heap,
                                              DYNAMIC_TYPE_ED25519);
                    if (sigCtx->key.ed25519 == NULL) {
                        ERROR_OUT(MEMORY_E, exit_cs);
                    }
                #endif
                    if ((ret = wc_ed25519_init_ex(sigCtx->key.ed25519,
                                            sigCtx->heap, sigCtx->devId)) < 0) {
                        goto exit_cs;
                    }
                    if ((ret = wc_ed25519_import_public(key, keySz,
                                                    sigCtx->key.ed25519)) < 0) {
                        WOLFSSL_MSG("ASN Key import error ED25519");
                        WOLFSSL_ERROR_VERBOSE(ret);
                        goto exit_cs;
                    }
                #ifdef WOLFSSL_ASYNC_CRYPT
                    sigCtx->asyncDev = &sigCtx->key.ed25519->asyncDev;
                #endif
                    break;
                }
            #endif
            #if defined(HAVE_ED448) && defined(HAVE_ED448_KEY_IMPORT)
                case ED448k:
                {
                    sigCtx->verify = 0;
                #ifndef WOLFSSL_NO_MALLOC
                    sigCtx->key.ed448 = (ed448_key*)XMALLOC(
                                                sizeof(ed448_key), sigCtx->heap,
                                                DYNAMIC_TYPE_ED448);
                    if (sigCtx->key.ed448 == NULL) {
                        ERROR_OUT(MEMORY_E, exit_cs);
                    }
                #endif
                    if ((ret = wc_ed448_init(sigCtx->key.ed448)) < 0) {
                        goto exit_cs;
                    }
                    if ((ret = wc_ed448_import_public(key, keySz,
                                                      sigCtx->key.ed448)) < 0) {
                        WOLFSSL_MSG("ASN Key import error ED448");
                        WOLFSSL_ERROR_VERBOSE(ret);
                        goto exit_cs;
                    }
                #ifdef WOLFSSL_ASYNC_CRYPT
                    sigCtx->asyncDev = &sigCtx->key.ed448->asyncDev;
                #endif
                    break;
                }
            #endif
            #if defined(HAVE_FALCON)
                case FALCON_LEVEL1k:
                {
                    word32 idx = 0;
                    sigCtx->verify = 0;
                #ifndef WOLFSSL_NO_MALLOC
                    sigCtx->key.falcon =
                        (falcon_key*)XMALLOC(sizeof(falcon_key),
                                             sigCtx->heap,
                                             DYNAMIC_TYPE_FALCON);
                    if (sigCtx->key.falcon == NULL) {
                        ERROR_OUT(MEMORY_E, exit_cs);
                    }
                #endif
                    if ((ret = wc_falcon_init_ex(sigCtx->key.falcon,
                                            sigCtx->heap, sigCtx->devId)) < 0) {
                        goto exit_cs;
                    }
                    if ((ret = wc_falcon_set_level(sigCtx->key.falcon, 1))
                        < 0) {
                        goto exit_cs;
                    }
                    if ((ret = wc_Falcon_PublicKeyDecode(key, &idx,
                        sigCtx->key.falcon, keySz)) < 0) {
                        WOLFSSL_MSG("ASN Key import error Falcon Level 1");
                        WOLFSSL_ERROR_VERBOSE(ret);
                        goto exit_cs;
                    }
                    break;
                }
                case FALCON_LEVEL5k:
                {
                    word32 idx = 0;
                    sigCtx->verify = 0;
                #ifndef WOLFSSL_NO_MALLOC
                    sigCtx->key.falcon =
                        (falcon_key*)XMALLOC(sizeof(falcon_key),
                                             sigCtx->heap,
                                             DYNAMIC_TYPE_FALCON);
                    if (sigCtx->key.falcon == NULL) {
                        ERROR_OUT(MEMORY_E, exit_cs);
                    }
                #endif
                    if ((ret = wc_falcon_init_ex(sigCtx->key.falcon,
                                            sigCtx->heap, sigCtx->devId)) < 0) {
                        goto exit_cs;
                    }
                    if ((ret = wc_falcon_set_level(sigCtx->key.falcon, 5))
                        < 0) {
                        goto exit_cs;
                    }
                    if ((ret = wc_Falcon_PublicKeyDecode(key, &idx,
                        sigCtx->key.falcon, keySz)) < 0) {
                        WOLFSSL_MSG("ASN Key import error Falcon Level 5");
                        WOLFSSL_ERROR_VERBOSE(ret);
                        goto exit_cs;
                    }
                    break;
                }
            #endif /* HAVE_FALCON */
            #if defined(HAVE_DILITHIUM) && \
                !defined(WOLFSSL_DILITHIUM_NO_VERIFY) && \
                !defined(WOLFSSL_DILITHIUM_NO_ASN1)
                #ifdef WOLFSSL_DILITHIUM_FIPS204_DRAFT
                case DILITHIUM_LEVEL2k:
                case DILITHIUM_LEVEL3k:
                case DILITHIUM_LEVEL5k:
                #endif
                case ML_DSA_LEVEL2k:
                case ML_DSA_LEVEL3k:
                case ML_DSA_LEVEL5k:
                {
                    word32 idx = 0;
                    int level;
                    if (keyOID == ML_DSA_LEVEL2k) {
                        level = WC_ML_DSA_44;
                    }
                    else if (keyOID == ML_DSA_LEVEL3k) {
                        level = WC_ML_DSA_65;
                    }
                    else if (keyOID == ML_DSA_LEVEL5k) {
                        level = WC_ML_DSA_87;
                    }
                #ifdef WOLFSSL_DILITHIUM_FIPS204_DRAFT
                    else if (keyOID == DILITHIUM_LEVEL2k) {
                        level = WC_ML_DSA_44_DRAFT;
                    }
                    else if (keyOID == DILITHIUM_LEVEL3k) {
                        level = WC_ML_DSA_65_DRAFT;
                    }
                    else if (keyOID == DILITHIUM_LEVEL5k) {
                        level = WC_ML_DSA_87_DRAFT;
                    }
                #endif
                    else {
                        WOLFSSL_MSG("Invalid Dilithium key OID");
                        goto exit_cs;
                    }
                    sigCtx->verify = 0;
                #ifndef WOLFSSL_NO_MALLOC
                    sigCtx->key.dilithium = (dilithium_key*)XMALLOC(
                        sizeof(dilithium_key), sigCtx->heap,
                        DYNAMIC_TYPE_DILITHIUM);
                    if (sigCtx->key.dilithium == NULL) {
                        ERROR_OUT(MEMORY_E, exit_cs);
                    }
                #endif
                    if ((ret = wc_dilithium_init_ex(sigCtx->key.dilithium,
                            sigCtx->heap, sigCtx->devId)) < 0) {
                        goto exit_cs;
                    }
                    if ((ret = wc_dilithium_set_level(sigCtx->key.dilithium,
                            level)) < 0) {
                        goto exit_cs;
                    }
                    if ((ret = wc_Dilithium_PublicKeyDecode(key, &idx,
                        sigCtx->key.dilithium, keySz)) < 0) {
                        WOLFSSL_MSG("ASN Key import error Dilithium");
                        goto exit_cs;
                    }
                    break;
                }
            #endif /* HAVE_DILITHIUM */
            #if defined(HAVE_SPHINCS)
                case SPHINCS_FAST_LEVEL1k:
                {
                    word32 idx = 0;
                    sigCtx->verify = 0;
                #ifndef WOLFSSL_NO_MALLOC
                    sigCtx->key.sphincs =
                        (sphincs_key*)XMALLOC(sizeof(sphincs_key),
                                             sigCtx->heap,
                                             DYNAMIC_TYPE_SPHINCS);
                    if (sigCtx->key.sphincs == NULL) {
                        ERROR_OUT(MEMORY_E, exit_cs);
                    }
                #endif

                    if ((ret = wc_sphincs_init(sigCtx->key.sphincs)) < 0) {
                        goto exit_cs;
                    }
                    if ((ret = wc_sphincs_set_level_and_optim(
                                   sigCtx->key.sphincs, 1, FAST_VARIANT))
                        < 0) {
                        goto exit_cs;
                    }
                    if ((ret = wc_Sphincs_PublicKeyDecode(key, &idx,
                        sigCtx->key.sphincs, keySz)) < 0) {
                        WOLFSSL_MSG("ASN Key import err: Sphincs-fast Level1");
                        goto exit_cs;
                    }
                    break;
                }
                case SPHINCS_FAST_LEVEL3k:
                {
                    word32 idx = 0;
                    sigCtx->verify = 0;
                #ifndef WOLFSSL_NO_MALLOC
                    sigCtx->key.sphincs =
                        (sphincs_key*)XMALLOC(sizeof(sphincs_key),
                                             sigCtx->heap,
                                             DYNAMIC_TYPE_SPHINCS);
                    if (sigCtx->key.sphincs == NULL) {
                        ERROR_OUT(MEMORY_E, exit_cs);
                    }
                #endif
                    if ((ret = wc_sphincs_init(sigCtx->key.sphincs)) < 0) {
                        goto exit_cs;
                    }
                    if ((ret = wc_sphincs_set_level_and_optim(
                                   sigCtx->key.sphincs, 3, FAST_VARIANT))
                        < 0) {
                        goto exit_cs;
                    }
                    if ((ret = wc_Sphincs_PublicKeyDecode(key, &idx,
                        sigCtx->key.sphincs, keySz)) < 0) {
                        WOLFSSL_MSG("ASN Key import err: Sphincs-fast Level3");
                        goto exit_cs;
                    }
                    break;
                }
                case SPHINCS_FAST_LEVEL5k:
                {
                    word32 idx = 0;
                    sigCtx->verify = 0;
                #ifndef WOLFSSL_NO_MALLOC
                    sigCtx->key.sphincs =
                        (sphincs_key*)XMALLOC(sizeof(sphincs_key),
                                             sigCtx->heap,
                                             DYNAMIC_TYPE_SPHINCS);
                    if (sigCtx->key.sphincs == NULL) {
                        ERROR_OUT(MEMORY_E, exit_cs);
                    }
                #endif
                    if ((ret = wc_sphincs_init(sigCtx->key.sphincs)) < 0) {
                        goto exit_cs;
                    }
                    if ((ret = wc_sphincs_set_level_and_optim(
                                   sigCtx->key.sphincs, 5, FAST_VARIANT))
                        < 0) {
                        goto exit_cs;
                    }
                    if ((ret = wc_Sphincs_PublicKeyDecode(key, &idx,
                        sigCtx->key.sphincs, keySz)) < 0) {
                        WOLFSSL_MSG("ASN Key import err: Sphincs-fast Level5");
                        goto exit_cs;
                    }
                    break;
                }
                case SPHINCS_SMALL_LEVEL1k:
                {
                    word32 idx = 0;
                    sigCtx->verify = 0;
                #ifndef WOLFSSL_NO_MALLOC
                    sigCtx->key.sphincs =
                        (sphincs_key*)XMALLOC(sizeof(sphincs_key),
                                             sigCtx->heap,
                                             DYNAMIC_TYPE_SPHINCS);
                    if (sigCtx->key.sphincs == NULL) {
                        ERROR_OUT(MEMORY_E, exit_cs);
                    }
                #endif
                    if ((ret = wc_sphincs_init(sigCtx->key.sphincs)) < 0) {
                        goto exit_cs;
                    }
                    if ((ret = wc_sphincs_set_level_and_optim(
                                   sigCtx->key.sphincs, 1, SMALL_VARIANT))
                        < 0) {
                        goto exit_cs;
                    }
                    if ((ret = wc_Sphincs_PublicKeyDecode(key, &idx,
                        sigCtx->key.sphincs, keySz)) < 0) {
                        WOLFSSL_MSG("ASN Key import err: Sphincs-fast Level1");
                        goto exit_cs;
                    }
                    break;
                }
                case SPHINCS_SMALL_LEVEL3k:
                {
                    word32 idx = 0;
                    sigCtx->verify = 0;
                #ifndef WOLFSSL_NO_MALLOC
                    sigCtx->key.sphincs =
                        (sphincs_key*)XMALLOC(sizeof(sphincs_key),
                                             sigCtx->heap,
                                             DYNAMIC_TYPE_SPHINCS);
                    if (sigCtx->key.sphincs == NULL) {
                        ERROR_OUT(MEMORY_E, exit_cs);
                    }
                #endif
                    if ((ret = wc_sphincs_init(sigCtx->key.sphincs)) < 0) {
                        goto exit_cs;
                    }
                    if ((ret = wc_sphincs_set_level_and_optim(
                                   sigCtx->key.sphincs, 3, SMALL_VARIANT))
                        < 0) {
                        goto exit_cs;
                    }
                    if ((ret = wc_Sphincs_PublicKeyDecode(key, &idx,
                        sigCtx->key.sphincs, keySz)) < 0) {
                        WOLFSSL_MSG("ASN Key import err: Sphincs-fast Level3");
                        goto exit_cs;
                    }
                    break;
                }
                case SPHINCS_SMALL_LEVEL5k:
                {
                    word32 idx = 0;
                    sigCtx->verify = 0;
                #ifndef WOLFSSL_NO_MALLOC
                    sigCtx->key.sphincs =
                        (sphincs_key*)XMALLOC(sizeof(sphincs_key),
                                             sigCtx->heap,
                                             DYNAMIC_TYPE_SPHINCS);
                    if (sigCtx->key.sphincs == NULL) {
                        ERROR_OUT(MEMORY_E, exit_cs);
                    }
                #endif
                    if ((ret = wc_sphincs_init(sigCtx->key.sphincs)) < 0) {
                        goto exit_cs;
                    }
                    if ((ret = wc_sphincs_set_level_and_optim(
                                   sigCtx->key.sphincs, 5, SMALL_VARIANT))
                        < 0) {
                        goto exit_cs;
                    }
                    if ((ret = wc_Sphincs_PublicKeyDecode(key, &idx,
                        sigCtx->key.sphincs, keySz)) < 0) {
                        WOLFSSL_MSG("ASN Key import err: Sphincs-fast Level5");
                        goto exit_cs;
                    }
                    break;
                }
            #endif /* HAVE_SPHINCS */
                default:
                    WOLFSSL_MSG("Verify Key type unknown");
                    ret = ASN_UNKNOWN_OID_E;
                    WOLFSSL_ERROR_VERBOSE(ret);
                    break;
            } /* switch (keyOID) */

            if (ret != 0) {
                goto exit_cs;
            }

            sigCtx->state = SIG_STATE_DO;

        #ifdef WOLFSSL_ASYNC_CRYPT
            if (sigCtx->devId != INVALID_DEVID && sigCtx->asyncDev && sigCtx->asyncCtx) {
                /* make sure event is initialized */
                WOLF_EVENT* event = &sigCtx->asyncDev->event;
                ret = wolfAsync_EventInit(event, WOLF_EVENT_TYPE_ASYNC_WOLFSSL,
                    sigCtx->asyncCtx, WC_ASYNC_FLAG_CALL_AGAIN);
            }
        #endif
        } /* SIG_STATE_KEY */
        FALL_THROUGH;

        case SIG_STATE_DO:
        {
            switch (keyOID) {
            #ifndef NO_RSA
                case RSAk:
                #ifdef WC_RSA_PSS
                case RSAPSSk:
                if (sigOID == RSAPSSk) {
                    /* TODO: pkCbRsaPss - RSA PSS callback. */
                    ret = wc_RsaPSS_VerifyInline_ex(sigCtx->sigCpy, sigSz,
                        &sigCtx->out, sigCtx->hash, sigCtx->mgf,
                        sigCtx->saltLen, sigCtx->key.rsa);
                }
                else
                #endif
                {
                #if defined(HAVE_PK_CALLBACKS)
                    if (sigCtx->pkCbRsa) {
                        ret = sigCtx->pkCbRsa(
                                sigCtx->sigCpy, sigSz, &sigCtx->out,
                                key, keySz,
                                sigCtx->pkCtxRsa);
                    }
                #if !defined(WOLFSSL_RENESAS_FSPSM_TLS) && \
                    !defined(WOLFSSL_RENESAS_TSIP_TLS)
                    else
                #else
                    if (!sigCtx->pkCbRsa ||
                        ret == WC_NO_ERR_TRACE(CRYPTOCB_UNAVAILABLE))
                #endif /* WOLFSSL_RENESAS_FSPSM_TLS */
                #endif /* HAVE_PK_CALLBACKS */
                    {
                        ret = wc_RsaSSL_VerifyInline(sigCtx->sigCpy, sigSz,
                                                 &sigCtx->out, sigCtx->key.rsa);
                    }
                }
                break;
            #endif /* !NO_RSA */
            #if !defined(NO_DSA) && !defined(HAVE_SELFTEST)
                case DSAk:
                {
                    ret = wc_DsaVerify(sigCtx->digest, sigCtx->sigCpy,
                            sigCtx->key.dsa, &sigCtx->verify);
                    break;
                }
            #endif /* !NO_DSA && !HAVE_SELFTEST */
            #if defined(WOLFSSL_SM2) && defined(WOLFSSL_SM3)
                case SM2k:
                {
                    /* OpenSSL creates signature without CERT_SIG_ID. */
                    ret = wc_ecc_sm2_create_digest(CERT_SIG_ID, 0, buf, bufSz,
                        WC_HASH_TYPE_SM3, sigCtx->digest, WC_SM3_DIGEST_SIZE,
                        sigCtx->key.ecc);
                    if (ret == 0) {
                        sigCtx->typeH    = SM3h;
                        sigCtx->digestSz = WC_SM3_DIGEST_SIZE;
                    }
                    else {
                        WOLFSSL_MSG("SM2wSM3 create digest failed");
                        WOLFSSL_ERROR_VERBOSE(ret);
                        goto exit_cs;
                    }
                    ret = wc_ecc_sm2_verify_hash(sig, sigSz, sigCtx->digest,
                        sigCtx->digestSz, &sigCtx->verify, sigCtx->key.ecc);
                    break;
                }
            #endif
            #if defined(HAVE_ECC) && defined(HAVE_ECC_VERIFY)
                case ECDSAk:
                {
                #if defined(WOLFSSL_SM2) && defined(WOLFSSL_SM3)
                    if (sigOID == CTC_SM3wSM2) {
                        ret = wc_ecc_sm2_create_digest(CERT_SIG_ID,
                            CERT_SIG_ID_SZ, buf, bufSz, WC_HASH_TYPE_SM3,
                            sigCtx->digest, WC_SM3_DIGEST_SIZE,
                            sigCtx->key.ecc);
                        if (ret == 0) {
                            sigCtx->typeH    = SM3h;
                            sigCtx->digestSz = WC_SM3_DIGEST_SIZE;
                        }
                        else {
                            WOLFSSL_MSG("SM2wSM3 create digest failed");
                            WOLFSSL_ERROR_VERBOSE(ret);
                            goto exit_cs;
                        }
                        ret = wc_ecc_sm2_verify_hash(sig, sigSz, sigCtx->digest,
                            sigCtx->digestSz, &sigCtx->verify, sigCtx->key.ecc);
                    }
                    else
                #endif
                #if defined(HAVE_PK_CALLBACKS)
                    if (sigCtx->pkCbEcc) {
                        ret = sigCtx->pkCbEcc(
                                sig, sigSz,
                                sigCtx->digest, (unsigned int)sigCtx->digestSz,
                                key, keySz, &sigCtx->verify,
                                sigCtx->pkCtxEcc);
                    }
                #if !defined(WOLFSSL_RENESAS_FSPSM_TLS) && \
                    !defined(WOLFSSL_RENESAS_TSIP_TLS)
                    else
                #else
                    if (!sigCtx->pkCbEcc ||
                        ret == WC_NO_ERR_TRACE(CRYPTOCB_UNAVAILABLE))
                #endif /* WOLFSSL_RENESAS_FSPSM_TLS */
                #endif /* HAVE_PK_CALLBACKS */
                    {
                        ret = wc_ecc_verify_hash(sig, sigSz, sigCtx->digest,
                            (word32)sigCtx->digestSz, &sigCtx->verify,
                            sigCtx->key.ecc);
                    }
                    break;
                }
            #endif /* HAVE_ECC */
            #if defined(HAVE_ED25519) && defined(HAVE_ED25519_VERIFY)
                case ED25519k:
                {
                    ret = wc_ed25519_verify_msg(sig, sigSz, buf, bufSz,
                                          &sigCtx->verify, sigCtx->key.ed25519);
                    break;
                }
            #endif
            #if defined(HAVE_ED448) && defined(HAVE_ED448_VERIFY)
                case ED448k:
                {
                    ret = wc_ed448_verify_msg(sig, sigSz, buf, bufSz,
                                             &sigCtx->verify, sigCtx->key.ed448,
                                             NULL, 0);
                    break;
                }
            #endif
            #if defined(HAVE_FALCON)
                case FALCON_LEVEL1k:
                case FALCON_LEVEL5k:
                {
                    ret = wc_falcon_verify_msg(sig, sigSz, buf, bufSz,
                                               &sigCtx->verify,
                                               sigCtx->key.falcon);
                    break;
                }
            #endif /* HAVE_FALCON */
            #if defined(HAVE_DILITHIUM) && !defined(WOLFSSL_DILITHIUM_NO_VERIFY)
                #ifdef WOLFSSL_DILITHIUM_FIPS204_DRAFT
                case DILITHIUM_LEVEL2k:
                case DILITHIUM_LEVEL3k:
                case DILITHIUM_LEVEL5k:
                {
                    ret = wc_dilithium_verify_msg(sig, sigSz, buf, bufSz,
                                               &sigCtx->verify,
                                               sigCtx->key.dilithium);
                    break;
                }
                #endif
                case ML_DSA_LEVEL2k:
                case ML_DSA_LEVEL3k:
                case ML_DSA_LEVEL5k:
                {
                    ret = wc_dilithium_verify_ctx_msg(sig, sigSz, NULL, 0, buf,
                        bufSz, &sigCtx->verify, sigCtx->key.dilithium);
                    break;
                }
            #endif /* HAVE_DILITHIUM */
            #if defined(HAVE_SPHINCS)
                case SPHINCS_FAST_LEVEL1k:
                case SPHINCS_FAST_LEVEL3k:
                case SPHINCS_FAST_LEVEL5k:
                case SPHINCS_SMALL_LEVEL1k:
                case SPHINCS_SMALL_LEVEL3k:
                case SPHINCS_SMALL_LEVEL5k:
                {
                    ret = wc_sphincs_verify_msg(sig, sigSz, buf, bufSz,
                                                &sigCtx->verify,
                                                sigCtx->key.sphincs);
                    break;
                }
            #endif /* HAVE_SPHINCS */
                default:
                    break;
            }  /* switch (keyOID) */

        #ifdef WOLFSSL_ASYNC_CRYPT
            if (ret == WC_NO_ERR_TRACE(WC_PENDING_E)) {
                goto exit_cs;
            }
        #endif

            if (ret < 0) {
                /* treat all errors as ASN_SIG_CONFIRM_E */
                ret = ASN_SIG_CONFIRM_E;
                WOLFSSL_ERROR_VERBOSE(ret);
                goto exit_cs;
            }

            sigCtx->state = SIG_STATE_CHECK;
        } /* SIG_STATE_DO */
        FALL_THROUGH;

        case SIG_STATE_CHECK:
        {
            switch (keyOID) {
            #ifndef NO_RSA
                case RSAk:
                #ifdef WC_RSA_PSS
                case RSAPSSk:
                if (sigOID == RSAPSSk) {
                #if (defined(HAVE_SELFTEST) && \
                     (!defined(HAVE_SELFTEST_VERSION) || \
                      (HAVE_SELFTEST_VERSION < 2))) || \
                    (defined(HAVE_FIPS) && defined(HAVE_FIPS_VERSION) && \
                     (HAVE_FIPS_VERSION < 2))
                    ret = wc_RsaPSS_CheckPadding_ex(sigCtx->digest,
                        sigCtx->digestSz, sigCtx->out, ret, sigCtx->hash,
                        sigCtx->saltLen);
                #elif (defined(HAVE_SELFTEST) && \
                       (HAVE_SELFTEST_VERSION == 2)) || \
                      (defined(HAVE_FIPS) && defined(HAVE_FIPS_VERSION) && \
                       (HAVE_FIPS_VERSION == 2))
                    ret = wc_RsaPSS_CheckPadding_ex(sigCtx->digest,
                        sigCtx->digestSz, sigCtx->out, ret, sigCtx->hash,
                        sigCtx->saltLen, 0);
                #else
                    ret = wc_RsaPSS_CheckPadding_ex2(sigCtx->digest,
                        (word32)sigCtx->digestSz, sigCtx->out, (word32)ret, sigCtx->hash,
                        sigCtx->saltLen, wc_RsaEncryptSize(sigCtx->key.rsa) * 8,
                        sigCtx->heap);
                #endif
                    break;
                }
                else
                #endif
                {
                    int encodedSigSz, verifySz;
                #if defined(WOLFSSL_RENESAS_TSIP_TLS) || \
                                            defined(WOLFSSL_RENESAS_FSPSM_TLS)
                    if (sigCtx->CertAtt.verifyByTSIP_SCE == 1) break;
                #endif
                #if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
                    byte* encodedSig = (byte*)XMALLOC(MAX_ENCODED_SIG_SZ,
                                        sigCtx->heap, DYNAMIC_TYPE_TMP_BUFFER);
                    if (encodedSig == NULL) {
                        ERROR_OUT(MEMORY_E, exit_cs);
                    }
                #else
                    byte encodedSig[MAX_ENCODED_SIG_SZ];
                #endif

                    verifySz = ret;

                    /* make sure we're right justified */
                    encodedSigSz = (int)wc_EncodeSignature(encodedSig,
                            sigCtx->digest, (word32)sigCtx->digestSz,
                            sigCtx->typeH);
                    if (encodedSigSz == verifySz && sigCtx->out != NULL &&
                        XMEMCMP(sigCtx->out, encodedSig,
                            (size_t)encodedSigSz) == 0) {
                        ret = 0;
                    }
                    else {
                        WOLFSSL_MSG("RSA SSL verify match encode error");
                        ret = ASN_SIG_CONFIRM_E;
                        WOLFSSL_ERROR_VERBOSE(ret);
                    }

                #if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
                    XFREE(encodedSig, sigCtx->heap, DYNAMIC_TYPE_TMP_BUFFER);
                #endif
                    break;
                }
            #endif /* NO_RSA */
            #if !defined(NO_DSA) && !defined(HAVE_SELFTEST)
                case DSAk:
                {
                    if (sigCtx->verify == 1) {
                        ret = 0;
                    }
                    else {
                        WOLFSSL_MSG("DSA Verify didn't match");
                        ret = ASN_SIG_CONFIRM_E;
                        WOLFSSL_ERROR_VERBOSE(ret);
                    }
                    break;
                }
            #endif /* !NO_DSA && !HAVE_SELFTEST */
            #ifdef HAVE_ECC
            #if defined(WOLFSSL_SM2) && defined(WOLFSSL_SM3)
                case SM2k:
            #endif
                case ECDSAk:
                {
                    if (sigCtx->verify == 1) {
                        ret = 0;
                    }
                    else {
                        WOLFSSL_MSG("ECC Verify didn't match");
                        ret = ASN_SIG_CONFIRM_E;
                        WOLFSSL_ERROR_VERBOSE(ret);
                    }
                    break;
                }
            #endif /* HAVE_ECC */
            #ifdef HAVE_ED25519
                case ED25519k:
                {
                    if (sigCtx->verify == 1) {
                        ret = 0;
                    }
                    else {
                        WOLFSSL_MSG("ED25519 Verify didn't match");
                        ret = ASN_SIG_CONFIRM_E;
                        WOLFSSL_ERROR_VERBOSE(ret);
                    }
                    break;
                }
            #endif /* HAVE_ED25519 */
            #ifdef HAVE_ED448
                case ED448k:
                {
                    if (sigCtx->verify == 1) {
                        ret = 0;
                    }
                    else {
                        WOLFSSL_MSG("ED448 Verify didn't match");
                        ret = ASN_SIG_CONFIRM_E;
                        WOLFSSL_ERROR_VERBOSE(ret);
                    }
                    break;
                }
            #endif /* HAVE_ED448 */
            #ifdef HAVE_FALCON
                case FALCON_LEVEL1k:
                {
                    if (sigCtx->verify == 1) {
                        ret = 0;
                    }
                    else {
                        WOLFSSL_MSG("FALCON_LEVEL1 Verify didn't match");
                        ret = ASN_SIG_CONFIRM_E;
                        WOLFSSL_ERROR_VERBOSE(ret);
                    }
                    break;
                }
                case FALCON_LEVEL5k:
                {
                    if (sigCtx->verify == 1) {
                        ret = 0;
                    }
                    else {
                        WOLFSSL_MSG("FALCON_LEVEL5 Verify didn't match");
                        ret = ASN_SIG_CONFIRM_E;
                        WOLFSSL_ERROR_VERBOSE(ret);
                    }
                    break;
                }
            #endif /* HAVE_FALCON */
            #ifdef HAVE_DILITHIUM
                #ifdef WOLFSSL_DILITHIUM_FIPS204_DRAFT
                case DILITHIUM_LEVEL2k:
                case DILITHIUM_LEVEL3k:
                case DILITHIUM_LEVEL5k:
                #endif /* WOLFSSL_DILITHIUM_FIPS204_DRAFT */
                case ML_DSA_LEVEL2k:
                case ML_DSA_LEVEL3k:
                case ML_DSA_LEVEL5k:
                    if (sigCtx->verify == 1) {
                        ret = 0;
                    }
                    else {
                        WOLFSSL_MSG("DILITHIUM Verify didn't match");
                        ret = ASN_SIG_CONFIRM_E;
                    }
                    break;
            #endif /* HAVE_DILITHIUM */
            #ifdef HAVE_SPHINCS
                case SPHINCS_FAST_LEVEL1k:
                {
                    if (sigCtx->verify == 1) {
                        ret = 0;
                    }
                    else {
                        WOLFSSL_MSG("SPHINCS_FAST_LEVEL1 Verify didn't match");
                        ret = ASN_SIG_CONFIRM_E;
                    }
                    break;
                }
                case SPHINCS_FAST_LEVEL3k:
                {
                    if (sigCtx->verify == 1) {
                        ret = 0;
                    }
                    else {
                        WOLFSSL_MSG("SPHINCS_FAST_LEVEL3 Verify didn't match");
                        ret = ASN_SIG_CONFIRM_E;
                    }
                    break;
                }
                case SPHINCS_FAST_LEVEL5k:
                {
                    if (sigCtx->verify == 1) {
                        ret = 0;
                    }
                    else {
                        WOLFSSL_MSG("SPHINCS_FAST_LEVEL5 Verify didn't match");
                        ret = ASN_SIG_CONFIRM_E;
                    }
                    break;
                }
                case SPHINCS_SMALL_LEVEL1k:
                {
                    if (sigCtx->verify == 1) {
                        ret = 0;
                    }
                    else {
                        WOLFSSL_MSG("SPHINCS_SMALL_LEVEL1 Verify didn't match");
                        ret = ASN_SIG_CONFIRM_E;
                    }
                    break;
                }
                case SPHINCS_SMALL_LEVEL3k:
                {
                    if (sigCtx->verify == 1) {
                        ret = 0;
                    }
                    else {
                        WOLFSSL_MSG("SPHINCS_SMALL_LEVEL3 Verify didn't match");
                        ret = ASN_SIG_CONFIRM_E;
                    }
                    break;
                }
                case SPHINCS_SMALL_LEVEL5k:
                {
                    if (sigCtx->verify == 1) {
                        ret = 0;
                    }
                    else {
                        WOLFSSL_MSG("SPHINCS_SMALL_LEVEL5 Verify didn't match");
                        ret = ASN_SIG_CONFIRM_E;
                    }
                    break;
                }
            #endif /* HAVE_SPHINCS */
                default:
                    break;
            }  /* switch (keyOID) */

            break;
        } /* SIG_STATE_CHECK */

        default:
            break;
    } /* switch (sigCtx->state) */

exit_cs:

#else
    /* For NO_ASN_CRYPT return "not compiled in" */
    ret = NOT_COMPILED_IN;
#endif /* !NO_ASN_CRYPT */

    (void)keyOID;
    (void)sigOID;

    WOLFSSL_LEAVE("ConfirmSignature", ret);

#ifdef WOLFSSL_ASYNC_CRYPT
    if (ret == WC_NO_ERR_TRACE(WC_PENDING_E))
        return ret;
#endif

    FreeSignatureCtx(sigCtx);

    return ret;
}

#ifndef IGNORE_NAME_CONSTRAINTS

static int MatchBaseName(int type, const char* name, int nameSz,
                         const char* base, int baseSz)
{
    if (base == NULL || baseSz <= 0 || name == NULL || nameSz <= 0 ||
            name[0] == '.' || nameSz < baseSz ||
            (type != ASN_RFC822_TYPE && type != ASN_DNS_TYPE &&
             type != ASN_DIR_TYPE)) {
        return 0;
    }

    if (type == ASN_DIR_TYPE)
        return XMEMCMP(name, base, (size_t)baseSz) == 0;

    /* If an email type, handle special cases where the base is only
     * a domain, or is an email address itself. */
    if (type == ASN_RFC822_TYPE) {
        const char* p = NULL;
        int count = 0;

        if (base[0] != '.') {
            p = base;
            count = 0;

            /* find the '@' in the base */
            while (*p != '@' && count < baseSz) {
                count++;
                p++;
            }

            /* No '@' in base, reset p to NULL */
            if (count >= baseSz)
                p = NULL;
        }

        if (p == NULL) {
            /* Base isn't an email address, it is a domain name,
             * wind the name forward one character past its '@'. */
            p = name;
            count = 0;
            while (*p != '@' && count < baseSz) {
                count++;
                p++;
            }

            if (count < baseSz && *p == '@') {
                name = p + 1;
                nameSz -= count + 1;
            }
        }
    }

    /* RFC 5280 section 4.2.1.10
     * "...Any DNS name that can be constructed by simply adding zero or more
     *  labels to the left-hand side of the name satisfies the name constraint."
     * i.e www.host.example.com works for host.example.com name constraint and
     * host1.example.com does not. */
    if (type == ASN_DNS_TYPE || (type == ASN_RFC822_TYPE && base[0] == '.')) {
        int szAdjust = nameSz - baseSz;
        name += szAdjust;
        nameSz -= szAdjust;
    }

    while (nameSz > 0) {
        if (XTOLOWER((unsigned char)*name) !=
                                               XTOLOWER((unsigned char)*base))
            return 0;
        name++;
        base++;
        nameSz--;
    }

    return 1;
}


/* Search through the list to find if the name is permitted.
 * name     The DNS name to search for
 * dnsList  The list to search through
 * nameType Type of DNS name to currently searching
 * return 1 if found in list or if not needed
 * return 0 if not found in the list but is needed
 */
static int PermittedListOk(DNS_entry* name, Base_entry* dnsList, byte nameType)
{
    Base_entry* current = dnsList;
    int match = 0;
    int need  = 0;
    int ret   = 1; /* is ok unless needed and no match found */

    while (current != NULL) {
        if (current->type == nameType) {
            need = 1; /* restriction on permitted names is set for this type */
            if (name->len >= current->nameSz &&
                MatchBaseName(nameType, name->name, name->len,
                              current->name, current->nameSz)) {
                match = 1; /* found the current name in the permitted list*/
                break;
            }
        }
        current = current->next;
    }

    /* check if permitted name restriction was set and no matching name found */
    if (need && !match)
        ret = 0;

    return ret;
}


/* Search through the list to find if the name is excluded.
 * name     The DNS name to search for
 * dnsList  The list to search through
 * nameType Type of DNS name to currently searching
 * return 1 if found in list and 0 if not found in the list
 */
static int IsInExcludedList(DNS_entry* name, Base_entry* dnsList, byte nameType)
{
    int ret = 0; /* default of not found in the list */
    Base_entry* current = dnsList;

    while (current != NULL) {
        if (current->type == nameType) {
            if (name->len >= current->nameSz &&
                MatchBaseName(nameType, name->name, name->len,
                              current->name, current->nameSz)) {
                ret = 1;
                break;
            }
        }
        current = current->next;
    }

    return ret;
}


static int ConfirmNameConstraints(Signer* signer, DecodedCert* cert)
{
    const byte nameTypes[] = {ASN_RFC822_TYPE, ASN_DNS_TYPE, ASN_DIR_TYPE};
    int i;

    if (signer == NULL || cert == NULL)
        return 0;

    if (signer->excludedNames == NULL && signer->permittedNames == NULL)
        return 1;

    for (i=0; i < (int)sizeof(nameTypes); i++) {
        byte nameType = nameTypes[i];
        DNS_entry* name = NULL;
        DNS_entry  subjectDnsName; /* temporary node used for subject name */

        XMEMSET(&subjectDnsName, 0, sizeof(DNS_entry));
        switch (nameType) {
            case ASN_DNS_TYPE:
                /* Should it also consider CN in subject? It could use
                 * subjectDnsName too */
                name = cert->altNames;
                break;
            case ASN_RFC822_TYPE:
                /* Shouldn't it validate E= in subject as well? */
                name = cert->altEmailNames;

                /* Add subject email for checking. */
                if (cert->subjectEmail != NULL) {
                    /* RFC 5280 section 4.2.1.10
                     * "When constraints are imposed on the rfc822Name name
                     * form, but the certificate does not include a subject
                     * alternative name, the rfc822Name constraint MUST be
                     * applied to the attribute of type emailAddress in the
                     * subject distinguished name" */
                    subjectDnsName.next = NULL;
                    subjectDnsName.type = ASN_RFC822_TYPE;
                    subjectDnsName.len  = cert->subjectEmailLen;
                    subjectDnsName.name = (char *)cert->subjectEmail;
                }
                break;
            case ASN_DIR_TYPE:
            #ifndef WOLFSSL_NO_ASN_STRICT
                name = cert->altDirNames;
            #endif

                /* RFC 5280 section 4.2.1.10
                    "Restrictions of the form directoryName MUST be
                    applied to the subject field .... and to any names
                    of type directoryName in the subjectAltName
                    extension"
                */
                if (cert->subjectRaw != NULL) {
                    subjectDnsName.next = NULL;
                    subjectDnsName.type = ASN_DIR_TYPE;
                    subjectDnsName.len = cert->subjectRawLen;
                    subjectDnsName.name = (char *)cert->subjectRaw;
                }
                break;
            default:
                /* Other types of names are ignored for now.
                 * Shouldn't it be rejected if it there is a altNamesByType[nameType]
                 * and signer->extNameConstraintCrit is set? */
                return 0;
        }

        while (name != NULL) {
            if (IsInExcludedList(name, signer->excludedNames, nameType) == 1) {
                WOLFSSL_MSG("Excluded name was found!");
                return 0;
            }

            /* Check against the permitted list */
            if (PermittedListOk(name, signer->permittedNames, nameType) != 1) {
                WOLFSSL_MSG("Permitted name was not found!");
                return 0;
            }

            name = name->next;
        }

        /* handle comparing against subject name too */
        if (subjectDnsName.len > 0 && subjectDnsName.name != NULL) {
            if (IsInExcludedList(&subjectDnsName, signer->excludedNames,
                        nameType) == 1) {
                WOLFSSL_MSG("Excluded name was found!");
                return 0;
            }

            /* Check against the permitted list */
            if (PermittedListOk(&subjectDnsName, signer->permittedNames,
                        nameType) != 1) {
                WOLFSSL_MSG("Permitted name was not found!");
                return 0;
            }
        }
    }

    return 1;
}

#endif /* IGNORE_NAME_CONSTRAINTS */

#ifndef WOLFSSL_ASN_TEMPLATE
static void AddAltName(DecodedCert* cert, DNS_entry* dnsEntry)
{
#if (defined(WOLFSSL_ASN_ALL) || defined(OPENSSL_EXTRA)) && \
    !defined(WOLFSSL_ALT_NAMES_NO_REV)
    /* logic to add alt name to end of list */
    dnsEntry->next = NULL;
    if (cert->altNames == NULL) {
        /* First on list */
        cert->altNames = dnsEntry;
    }
    else {
        DNS_entry* temp = cert->altNames;

        /* Find end */
        for (; (temp->next != NULL); temp = temp->next);

        /* Add to end */
        temp->next = dnsEntry;
    }
#else
    dnsEntry->next = cert->altNames;
    cert->altNames = dnsEntry;
#endif
}
#endif

#ifdef WOLFSSL_ASN_TEMPLATE
#if defined(WOLFSSL_SEP) || defined(WOLFSSL_FPKI)
/* ASN.1 template for OtherName of an X.509 certificate.
 * X.509: RFC 5280, 4.2.1.6 - OtherName (without implicit outer SEQUENCE).
 * HW Name: RFC 4108, 5 - Hardware Module Name
 * Only support HW Name where the type is a HW serial number.
 *
 * Other Names handled for FPKI (Federal PKI) use:
 * UPN (Universal Principal Name), a non-standard Other Name
 *  (RFC3280 sec 4.2.1.7). Often used with FIPS 201 smartcard login.
 * FASC-N (Federal Agency Smart Credential Number), defined in the document
 *  fpki-x509-cert-policy-common.pdf. Used for a smart card ID.
 */
static const ASNItem otherNameASN[] = {
/* TYPEID   */ { 0, ASN_OBJECT_ID, 0, 0, 0 },
/* VALUE    */ { 0, ASN_CONTEXT_SPECIFIC | ASN_OTHERNAME_VALUE, 1, 1, 0 },
/* UPN      */     { 1, ASN_UTF8STRING, 0, 0, 2 },
/* FASC-N   */     { 1, ASN_OCTET_STRING, 0, 0, 2 },
/* HWN_SEQ  */     { 1, ASN_SEQUENCE, 1, 0, 2 },
/* HWN_TYPE */         { 2, ASN_OBJECT_ID, 0, 0, 0 },
/* HWN_NUM  */         { 2, ASN_OCTET_STRING, 0, 0, 0 }
};
enum {
    OTHERNAMEASN_IDX_TYPEID = 0,
    OTHERNAMEASN_IDX_VALUE,
    OTHERNAMEASN_IDX_UPN,
    OTHERNAMEASN_IDX_FASCN,
    OTHERNAMEASN_IDX_HWN_SEQ,
    OTHERNAMEASN_IDX_HWN_TYPE,
    OTHERNAMEASN_IDX_HWN_NUM
};

/* Number of items in ASN.1 template for OtherName of an X.509 certificate. */
#define otherNameASN_Length (sizeof(otherNameASN) / sizeof(ASNItem))

#ifdef WOLFSSL_SEP
static int DecodeSEP(ASNGetData* dataASN, DecodedCert* cert)
{
    int ret = 0;
    word32 oidLen, serialLen;

    oidLen = dataASN[OTHERNAMEASN_IDX_HWN_TYPE].data.oid.length;
    serialLen = dataASN[OTHERNAMEASN_IDX_HWN_NUM].data.ref.length;

    /* Allocate space for HW type OID. */
    cert->hwType = (byte*)XMALLOC(oidLen, cert->heap,
                                  DYNAMIC_TYPE_X509_EXT);
    if (cert->hwType == NULL)
        ret = MEMORY_E;

    if (ret == 0) {
        /* Copy, into cert HW type OID */
        XMEMCPY(cert->hwType,
                dataASN[OTHERNAMEASN_IDX_HWN_TYPE].data.oid.data, oidLen);
        cert->hwTypeSz = (int)oidLen;
        /* TODO: check this is the HW serial number OID - no test data. */

        /* Allocate space for HW serial number, +1 for null terminator. */
        cert->hwSerialNum = (byte*)XMALLOC(serialLen + 1, cert->heap,
                                           DYNAMIC_TYPE_X509_EXT);
        if (cert->hwSerialNum == NULL) {
            WOLFSSL_MSG("\tOut of Memory");
            ret = MEMORY_E;
        }
    }
    if (ret == 0) {
        /* Copy into cert HW serial number. */
        XMEMCPY(cert->hwSerialNum,
                dataASN[OTHERNAMEASN_IDX_HWN_NUM].data.ref.data, serialLen);
        cert->hwSerialNum[serialLen] = '\0';
        cert->hwSerialNumSz = (int)serialLen;
    }
    return ret;
}
#endif /* WOLFSSL_SEP */

static int DecodeOtherHelper(ASNGetData* dataASN, DecodedCert* cert, int oid)
{
    DNS_entry* entry = NULL;
    int ret = 0;
    word32 bufLen   = 0;
    const char* buf = NULL;

    switch (oid) {
#ifdef WOLFSSL_FPKI
        case FASCN_OID:
            bufLen = dataASN[OTHERNAMEASN_IDX_FASCN].data.ref.length;
            buf    = (const char*)dataASN[OTHERNAMEASN_IDX_FASCN].data.ref.data;
            break;
#endif /* WOLFSSL_FPKI */
        case UPN_OID:
            bufLen = dataASN[OTHERNAMEASN_IDX_UPN].data.ref.length;
            buf    = (const char*)dataASN[OTHERNAMEASN_IDX_UPN].data.ref.data;
            break;
        default:
            WOLFSSL_ERROR_VERBOSE(ASN_UNKNOWN_OID_E);
            ret = ASN_UNKNOWN_OID_E;
            break;
    }

    if (ret == 0) {
        ret = SetDNSEntry(cert->heap, buf, (int)bufLen, ASN_OTHER_TYPE, &entry);
        if (ret == 0) {
        #ifdef WOLFSSL_FPKI
            entry->oidSum = oid;
        #endif
            AddDNSEntryToList(&cert->altNames, entry);
        }
    }
    return ret;
}

/* Decode data with OtherName format from after implicit SEQUENCE.
 *
 * @param [in, out] cert      Certificate object.
 * @param [in]      input     Buffer containing encoded OtherName.
 * @param [in, out] inOutIdx  On in, the index of the start of the OtherName.
 *                            On out, index after OtherName.
 * @param [in]      maxIdx    Maximum index of data in buffer.
 * @return  0 on success.
 * @return  MEMORY_E on dynamic memory allocation failure.
 * @return  ASN_PARSE_E when BER encoded data does not match ASN.1 items or
 *          is invalid.
 * @return  ASN_PARSE_E when OID does is not HW Name.
 * @return  ASN_UNKNOWN_OID_E when the OID cannot be verified.
 * @return  BUFFER_E when data in buffer is too small.
 */
static int DecodeOtherName(DecodedCert* cert, const byte* input,
                           word32* inOutIdx, int len)
{
    DECL_ASNGETDATA(dataASN, otherNameASN_Length);
    int ret = 0;
    word32 maxIdx = *inOutIdx + (word32)len;
    const char* name = (const char*)input + *inOutIdx;

    CALLOC_ASNGETDATA(dataASN, otherNameASN_Length, ret, cert->heap);

    if (ret == 0) {
        /* Check the first OID is a recognized Alt Cert Name type. */
        GetASN_OID(&dataASN[OTHERNAMEASN_IDX_TYPEID], oidCertAltNameType);
        /* Parse OtherName. */
        ret = GetASN_Items(otherNameASN, dataASN, otherNameASN_Length, 1, input,
                           inOutIdx, maxIdx);
    }
    if (ret == 0) {
        /* Ensure expected OID. */
        switch (dataASN[OTHERNAMEASN_IDX_TYPEID].data.oid.sum) {
        #ifdef WOLFSSL_SEP
            case HW_NAME_OID:
                /* Only support HW serial number. */
                GetASN_OID(&dataASN[OTHERNAMEASN_IDX_HWN_TYPE], oidIgnoreType);
                ret = DecodeSEP(dataASN, cert);
                break;
        #endif /* WOLFSSL_SEP */
        #ifdef WOLFSSL_FPKI
            case FASCN_OID:
        #endif /* WOLFSSL_FPKI */
            case UPN_OID:
                ret = DecodeOtherHelper(dataASN, cert,
                           (int)dataASN[OTHERNAMEASN_IDX_TYPEID].data.oid.sum);
                break;
            default:
                WOLFSSL_MSG("\tadding unsupported OID");
                ret = SetDNSEntry(cert->heap, name, len, ASN_OTHER_TYPE,
                        &cert->altNames);
                break;
        }
    }

    FREE_ASNGETDATA(dataASN, cert->heap);
    return ret;
}
#endif /* WOLFSSL_SEP || WOLFSSL_FPKI */

/* Decode a GeneralName.
 *
 * @param [in]      input     Buffer containing encoded OtherName.
 * @param [in, out] inOutIdx  On in, the index of the start of the OtherName.
 *                            On out, index after OtherName.
 * @param [in]      len       Length of data in buffer.
 * @param [in]      cert      Decoded certificate object.
 * @return  0 on success.
 * @return  ASN_PARSE_E when BER encoded data does not match ASN.1 items or
 *          is invalid.
 * @return  BUFFER_E when data in buffer is too small.
 * @return  ASN_UNKNOWN_OID_E when the OID cannot be verified.
 * @return  MEMORY_E when dynamic memory allocation fails.
 */
static int DecodeGeneralName(const byte* input, word32* inOutIdx, byte tag,
                             int len, DecodedCert* cert)
{
    int ret = 0;
    word32 idx = *inOutIdx;

    /* GeneralName choice: dnsName */
    if (tag == (ASN_CONTEXT_SPECIFIC | ASN_DNS_TYPE)) {
        ret = SetDNSEntry(cert->heap, (const char*)(input + idx), len,
                ASN_DNS_TYPE, &cert->altNames);
        if (ret == 0) {
            idx += (word32)len;
        }
    }
#ifndef IGNORE_NAME_CONSTRAINTS
    /* GeneralName choice: directoryName */
    else if (tag == (ASN_CONTEXT_SPECIFIC | ASN_CONSTRUCTED | ASN_DIR_TYPE)) {
        int strLen;
        word32 idxDir = idx;

        /* Expecting a SEQUENCE using up all data. */
        if (GetASN_Sequence(input, &idxDir, &strLen, idx + (word32)len, 1) < 0)
        {
            WOLFSSL_MSG("\tfail: seq length");
            return ASN_PARSE_E;
        }

        ret = SetDNSEntry(cert->heap, (const char*)(input + idxDir), strLen,
                ASN_DIR_TYPE, &cert->altDirNames);
        if (ret == 0) {
            idx += (word32)len;
        }
    }
    /* GeneralName choice: rfc822Name */
    else if (tag == (ASN_CONTEXT_SPECIFIC | ASN_RFC822_TYPE)) {
        ret = SetDNSEntry(cert->heap, (const char*)(input + idx), len,
                ASN_RFC822_TYPE, &cert->altEmailNames);
        if (ret == 0) {
            idx += (word32)len;
        }
    }
    /* GeneralName choice: uniformResourceIdentifier */
    else if (tag == (ASN_CONTEXT_SPECIFIC | ASN_URI_TYPE)) {
        WOLFSSL_MSG("\tPutting URI into list but not using");

    #if !defined(WOLFSSL_NO_ASN_STRICT) && !defined(WOLFSSL_FPKI)
        /* Verify RFC 5280 Sec 4.2.1.6 rule:
            "The name MUST NOT be a relative URI"
            As per RFC 3986 Sec 4.3, an absolute URI is only required to contain
            a scheme and hier-part.  So the only strict requirement is a ':'
            being present after the scheme.  If a '/' is present as part of the
            hier-part, it must come after the ':' (see RFC 3986 Sec 3). */
        {
            int i;

            /* skip past scheme (i.e http,ftp,...) finding first ':' char */
            for (i = 0; i < len; i++) {
                if (input[idx + (word32)i] == ':') {
                    break;
                }
                if (input[idx + (word32)i] == '/') {
                    i = len; /* error, found relative path since '/' was
                              * encountered before ':'. Returning error
                              * value in next if statement. */
                }
            }

            /* test hier-part is empty */
            if (i == 0 || i == len) {
                WOLFSSL_MSG("\tEmpty or malformed URI");
                WOLFSSL_ERROR_VERBOSE(ASN_ALT_NAME_E);
                return ASN_ALT_NAME_E;
            }

            /* test if scheme is missing  */
            if (input[idx + (word32)i] != ':') {
                WOLFSSL_MSG("\tAlt Name must be absolute URI");
                WOLFSSL_ERROR_VERBOSE(ASN_ALT_NAME_E);
                return ASN_ALT_NAME_E;
            }
        }
    #endif

        ret = SetDNSEntry(cert->heap, (const char*)(input + idx), len,
                ASN_URI_TYPE, &cert->altNames);
        if (ret == 0) {
            idx += (word32)len;
        }
    }
    #ifdef WOLFSSL_IP_ALT_NAME
    /* GeneralName choice: iPAddress */
    else if (tag == (ASN_CONTEXT_SPECIFIC | ASN_IP_TYPE)) {
        ret = SetDNSEntry(cert->heap, (const char*)(input + idx), len,
                ASN_IP_TYPE, &cert->altNames);
        if (ret == 0) {
            idx += (word32)len;
        }
    }
    #endif /* WOLFSSL_IP_ALT_NAME */
    #ifdef WOLFSSL_RID_ALT_NAME
    /* GeneralName choice: registeredID */
    else if (tag == (ASN_CONTEXT_SPECIFIC | ASN_RID_TYPE)) {
        ret = SetDNSEntry(cert->heap, (const char*)(input + idx), len,
                ASN_RID_TYPE, &cert->altNames);
        if (ret == 0) {
            idx += (word32)len;
        }
    }
    #endif /* WOLFSSL_RID_ALT_NAME */
#endif /* IGNORE_NAME_CONSTRAINTS */
#if defined(WOLFSSL_SEP) || defined(WOLFSSL_FPKI)
    /* GeneralName choice: otherName */
    else if (tag == (ASN_CONTEXT_SPECIFIC | ASN_CONSTRUCTED | ASN_OTHER_TYPE)) {
        /* TODO: test data for code path */
        ret = DecodeOtherName(cert, input, &idx, len);
    }
#endif
    /* GeneralName choice: dNSName, x400Address, ediPartyName */
    else {
        WOLFSSL_MSG("\tUnsupported name type, skipping");
        idx += (word32)len;
    }

    if (ret == 0) {
        /* Return index of next encoded byte. */
        *inOutIdx = idx;
    }
    return ret;
}

/* ASN.1 choices for GeneralName.
 * X.509: RFC 5280, 4.2.1.6 - GeneralName.
 */
static const byte generalNameChoice[] = {
    ASN_CONTEXT_SPECIFIC | ASN_CONSTRUCTED | 0,
    ASN_CONTEXT_SPECIFIC                   | 1,
    ASN_CONTEXT_SPECIFIC                   | 2,
    ASN_CONTEXT_SPECIFIC                   | 3,
    ASN_CONTEXT_SPECIFIC | ASN_CONSTRUCTED | 4,
    ASN_CONTEXT_SPECIFIC | ASN_CONSTRUCTED | 5,
    ASN_CONTEXT_SPECIFIC                   | 6,
    ASN_CONTEXT_SPECIFIC                   | 7,
    ASN_CONTEXT_SPECIFIC                   | 8,
    0
};

/* ASN.1 template for GeneralName.
 * X.509: RFC 5280, 4.2.1.6 - GeneralName.
 */
static const ASNItem altNameASN[] = {
    { 0, ASN_CONTEXT_SPECIFIC | 0, 0, 1, 0 }
};
enum {
    ALTNAMEASN_IDX_GN = 0
};

/* Number of items in ASN.1 template for GeneralName. */
#define altNameASN_Length (sizeof(altNameASN) / sizeof(ASNItem))
#endif /* WOLFSSL_ASN_TEMPLATE */

#if defined(WOLFSSL_SEP) && !defined(WOLFSSL_ASN_TEMPLATE)
/* return 0 on success */
static int DecodeSepHwAltName(DecodedCert* cert, const byte* input,
    word32* idxIn, word32 sz)
{
    word32 idx = *idxIn;
    int  strLen;
    int  ret;
    byte tag;

    /* Certificates issued with this OID in the subject alt name are for
     * verifying signatures created on a module.
     * RFC 4108 Section 5. */
    if (cert->hwType != NULL) {
        WOLFSSL_MSG("\tAlready seen Hardware Module Name");
        return ASN_PARSE_E;
    }

    if (GetASNTag(input, &idx, &tag, sz) < 0) {
        return ASN_PARSE_E;
    }

    if (tag != (ASN_CONTEXT_SPECIFIC | ASN_CONSTRUCTED)) {
        WOLFSSL_MSG("\twrong type");
        return ASN_PARSE_E;
    }

    if (GetLength(input, &idx, &strLen, sz) < 0) {
        WOLFSSL_MSG("\tfail: str len");
        return ASN_PARSE_E;
    }

    if (GetSequence(input, &idx, &strLen, sz) < 0) {
        WOLFSSL_MSG("\tBad Sequence");
        return ASN_PARSE_E;
    }

    ret = GetASNObjectId(input, &idx, &strLen, sz);
    if (ret != 0) {
        WOLFSSL_MSG("\tbad OID");
        return ret;
    }

    cert->hwType = (byte*)XMALLOC((size_t)strLen, cert->heap,
                                  DYNAMIC_TYPE_X509_EXT);
    if (cert->hwType == NULL) {
        WOLFSSL_MSG("\tOut of Memory");
        return MEMORY_E;
    }

    XMEMCPY(cert->hwType, &input[idx], (size_t)strLen);
    cert->hwTypeSz = strLen;
    idx += (word32)strLen;

    ret = GetOctetString(input, &idx, &strLen, sz);
    if (ret < 0) {
        XFREE(cert->hwType, cert->heap, DYNAMIC_TYPE_X509_EXT);
        cert->hwType = NULL;
        return ret;
    }

    cert->hwSerialNum = (byte*)XMALLOC((size_t)strLen + 1, cert->heap,
                                       DYNAMIC_TYPE_X509_EXT);
    if (cert->hwSerialNum == NULL) {
        WOLFSSL_MSG("\tOut of Memory");
        XFREE(cert->hwType, cert->heap, DYNAMIC_TYPE_X509_EXT);
        cert->hwType = NULL;
        return MEMORY_E;
    }

    XMEMCPY(cert->hwSerialNum, &input[idx], (size_t)strLen);
    cert->hwSerialNum[strLen] = '\0';
    cert->hwSerialNumSz = strLen;
    idx += (word32)strLen;

    *idxIn = idx;
    return 0;
}
#endif /* WOLFSSL_SEP */

#if !defined(WOLFSSL_ASN_TEMPLATE)
/* return 0 on success */
static int DecodeConstructedOtherName(DecodedCert* cert, const byte* input,
        word32* idx, word32 sz, int oid)
{
    int ret    = 0;
    int strLen = 0;
    byte tag;
    DNS_entry* dnsEntry = NULL;

    if (GetASNTag(input, idx, &tag, sz) < 0) {
        ret = ASN_PARSE_E;
    }

    if (ret == 0 && (tag != (ASN_CONTEXT_SPECIFIC | ASN_CONSTRUCTED))) {
        ret = ASN_PARSE_E;
    }

    if (ret == 0 && (GetLength(input, idx, &strLen, sz) < 0)) {
        ret = ASN_PARSE_E;
    }

    if (ret == 0) {
        dnsEntry = AltNameNew(cert->heap);
        if (dnsEntry == NULL) {
            WOLFSSL_MSG("\tOut of Memory");
            return MEMORY_E;
        }

        switch (oid) {
        #ifdef WOLFSSL_FPKI
            case FASCN_OID:
                ret = GetOctetString(input, idx, &strLen, sz);
                if (ret > 0) {
                    ret = 0;
                }
                break;
        #endif /* WOLFSSL_FPKI */
            case UPN_OID:
                if (GetASNTag(input, idx, &tag, sz) < 0) {
                    ret = ASN_PARSE_E;
                }

                if (ret == 0 &&
                        tag != ASN_PRINTABLE_STRING && tag != ASN_UTF8STRING &&
                                    tag != ASN_IA5_STRING) {
                    WOLFSSL_MSG("Was expecting a string for UPN");
                    ret = ASN_PARSE_E;
                }

                if (ret == 0 && (GetLength(input, idx, &strLen, sz) < 0)) {
                    WOLFSSL_MSG("Was expecting a string for UPN");
                    ret = ASN_PARSE_E;
                }
                break;

            default:
                WOLFSSL_MSG("Unknown constructed other name, skipping");
                XFREE(dnsEntry, cert->heap, DYNAMIC_TYPE_ALTNAME);
                dnsEntry = NULL;
        }
    }

    if (ret == 0 && dnsEntry != NULL) {
        dnsEntry->type = ASN_OTHER_TYPE;
        dnsEntry->len = strLen;
        dnsEntry->name = (char*)XMALLOC((size_t)strLen + 1, cert->heap,
            DYNAMIC_TYPE_ALTNAME);
    #ifdef WOLFSSL_FPKI
        dnsEntry->oidSum = oid;
    #endif /* WOLFSSL_FPKI */
        if (dnsEntry->name == NULL) {
            WOLFSSL_MSG("\tOut of Memory");
            ret = MEMORY_E;
        }
        else {
            XMEMCPY(dnsEntry->name, &input[*idx], (size_t)strLen);
            dnsEntry->name[strLen] = '\0';
            AddAltName(cert, dnsEntry);
        }
    }

    if (ret == 0) {
        *idx += (word32)strLen;
    }
    else {
        XFREE(dnsEntry, cert->heap, DYNAMIC_TYPE_ALTNAME);
    }

    return ret;
}
#endif

/* Decode subject alternative names extension.
 *
 * RFC 5280 4.2.1.6.  Subject Alternative Name
 *
 * @param [in]      input  Buffer holding encoded data.
 * @param [in]      sz     Size of encoded data in bytes.
 * @param [in, out] cert   Decoded certificate object.
 * @return  0 on success.
 * @return  ASN_PARSE_E when BER encoded data does not match ASN.1 items or
 *          is invalid.
 * @return  BUFFER_E when data in buffer is too small.
 * @return  ASN_UNKNOWN_OID_E when the OID cannot be verified.
 * @return  MEMORY_E when dynamic memory allocation fails.
 */
static int DecodeAltNames(const byte* input, word32 sz, DecodedCert* cert)
{
#ifndef WOLFSSL_ASN_TEMPLATE
    word32 idx = 0;
    int length = 0;
    word32 numNames = 0;

    WOLFSSL_ENTER("DecodeAltNames");

    if (GetSequence(input, &idx, &length, sz) < 0) {
        WOLFSSL_MSG("\tBad Sequence");
        return ASN_PARSE_E;
    }

    if (length == 0) {
        /* RFC 5280 4.2.1.6.  Subject Alternative Name
           If the subjectAltName extension is present, the sequence MUST
           contain at least one entry. */
        WOLFSSL_ERROR_VERBOSE(ASN_PARSE_E);
        return ASN_PARSE_E;
    }

#ifdef OPENSSL_ALL
    cert->extSubjAltNameSrc = input;
    cert->extSubjAltNameSz = sz;
#endif

    cert->weOwnAltNames = 1;

    while (length > 0) {
        byte current_byte;

        /* Verify idx can't overflow input buffer */
        if (idx >= (word32)sz) {
            WOLFSSL_MSG("\tBad Index");
            return BUFFER_E;
        }

        numNames++;
        if (numNames > WOLFSSL_MAX_ALT_NAMES) {
            WOLFSSL_MSG("\tToo many subject alternative names");
            return ASN_ALT_NAME_E;
        }

        current_byte = input[idx++];
        length--;

        /* Save DNS Type names in the altNames list. */
        /* Save Other Type names in the cert's OidMap */
        if (current_byte == (ASN_CONTEXT_SPECIFIC | ASN_DNS_TYPE)) {
            DNS_entry* dnsEntry;
            int strLen;
            word32 lenStartIdx = idx;

            if (GetLength(input, &idx, &strLen, sz) < 0) {
                WOLFSSL_MSG("\tfail: str length");
                return ASN_PARSE_E;
            }
            length -= (int)(idx - lenStartIdx);

            dnsEntry = AltNameNew(cert->heap);
            if (dnsEntry == NULL) {
                WOLFSSL_MSG("\tOut of Memory");
                return MEMORY_E;
            }

            dnsEntry->type = ASN_DNS_TYPE;
            dnsEntry->name = (char*)XMALLOC((size_t)strLen + 1, cert->heap,
                                         DYNAMIC_TYPE_ALTNAME);
            if (dnsEntry->name == NULL) {
                WOLFSSL_MSG("\tOut of Memory");
                XFREE(dnsEntry, cert->heap, DYNAMIC_TYPE_ALTNAME);
                return MEMORY_E;
            }
            dnsEntry->len = strLen;
            XMEMCPY(dnsEntry->name, &input[idx], (size_t)strLen);
            dnsEntry->name[strLen] = '\0';

            AddAltName(cert, dnsEntry);

            length -= strLen;
            idx    += (word32)strLen;
        }
    #ifndef IGNORE_NAME_CONSTRAINTS
        else if (current_byte ==
                (ASN_CONTEXT_SPECIFIC | ASN_CONSTRUCTED | ASN_DIR_TYPE)) {
            DNS_entry* dirEntry;
            int strLen;
            word32 lenStartIdx = idx;

            if (GetLength(input, &idx, &strLen, sz) < 0) {
                WOLFSSL_MSG("\tfail: str length");
                return ASN_PARSE_E;
            }

            if (GetSequence(input, &idx, &strLen, sz) < 0) {
                WOLFSSL_MSG("\tfail: seq length");
                return ASN_PARSE_E;
            }
            length -= (int)(idx - lenStartIdx);

            dirEntry = AltNameNew(cert->heap);
            if (dirEntry == NULL) {
                WOLFSSL_MSG("\tOut of Memory");
                return MEMORY_E;
            }

            dirEntry->type = ASN_DIR_TYPE;
            dirEntry->name = (char*)XMALLOC((size_t)strLen + 1, cert->heap,
                                         DYNAMIC_TYPE_ALTNAME);
            if (dirEntry->name == NULL) {
                WOLFSSL_MSG("\tOut of Memory");
                XFREE(dirEntry, cert->heap, DYNAMIC_TYPE_ALTNAME);
                return MEMORY_E;
            }
            dirEntry->len = strLen;
            XMEMCPY(dirEntry->name, &input[idx], (size_t)strLen);
            dirEntry->name[strLen] = '\0';
            dirEntry->next = cert->altDirNames;
            cert->altDirNames = dirEntry;

            length -= strLen;
            idx    += (word32)strLen;
        }
        else if (current_byte == (ASN_CONTEXT_SPECIFIC | ASN_RFC822_TYPE)) {
            DNS_entry* emailEntry;
            int strLen;
            word32 lenStartIdx = idx;

            if (GetLength(input, &idx, &strLen, sz) < 0) {
                WOLFSSL_MSG("\tfail: str length");
                return ASN_PARSE_E;
            }
            length -= (int)(idx - lenStartIdx);

            emailEntry = AltNameNew(cert->heap);
            if (emailEntry == NULL) {
                WOLFSSL_MSG("\tOut of Memory");
                return MEMORY_E;
            }

            emailEntry->type = ASN_RFC822_TYPE;
            emailEntry->name = (char*)XMALLOC((size_t)strLen + 1, cert->heap,
                                         DYNAMIC_TYPE_ALTNAME);
            if (emailEntry->name == NULL) {
                WOLFSSL_MSG("\tOut of Memory");
                XFREE(emailEntry, cert->heap, DYNAMIC_TYPE_ALTNAME);
                return MEMORY_E;
            }
            emailEntry->len = strLen;
            XMEMCPY(emailEntry->name, &input[idx], (size_t)strLen);
            emailEntry->name[strLen] = '\0';

            emailEntry->next = cert->altEmailNames;
            cert->altEmailNames = emailEntry;

            length -= strLen;
            idx    += (word32)strLen;
        }
        else if (current_byte == (ASN_CONTEXT_SPECIFIC | ASN_URI_TYPE)) {
            DNS_entry* uriEntry;
            int strLen;
            word32 lenStartIdx = idx;

            WOLFSSL_MSG("\tPutting URI into list but not using");
            if (GetLength(input, &idx, &strLen, sz) < 0) {
                WOLFSSL_MSG("\tfail: str length");
                return ASN_PARSE_E;
            }
            length -= (int)(idx - lenStartIdx);

            /* check that strLen at index is not past input buffer */
            if ((word32)strLen + idx > sz) {
                return BUFFER_E;
            }

        #if !defined(WOLFSSL_NO_ASN_STRICT) && !defined(WOLFSSL_FPKI)
            /* Verify RFC 5280 Sec 4.2.1.6 rule:
                "The name MUST NOT be a relative URI"
                As per RFC 3986 Sec 4.3, an absolute URI is only required to contain
                a scheme and hier-part.  So the only strict requirement is a ':'
                being present after the scheme.  If a '/' is present as part of the
                hier-part, it must come after the ':' (see RFC 3986 Sec 3). */

            {
                word32 i;

                /* skip past scheme (i.e http,ftp,...) finding first ':' char */
                for (i = 0; i < (word32)strLen; i++) {
                    if (input[idx + i] == ':') {
                        break;
                    }
                    if (input[idx + i] == '/') {
                        WOLFSSL_MSG("\tAlt Name must be absolute URI");
                        WOLFSSL_ERROR_VERBOSE(ASN_ALT_NAME_E);
                        return ASN_ALT_NAME_E;
                    }
                }

                /* test hier-part is empty */
                if (i == 0 || i == (word32)strLen) {
                    WOLFSSL_MSG("\tEmpty or malformed URI");
                    WOLFSSL_ERROR_VERBOSE(ASN_ALT_NAME_E);
                    return ASN_ALT_NAME_E;
                }

                /* test if scheme is missing */
                if (input[idx + i] != ':') {
                    WOLFSSL_MSG("\tAlt Name must be absolute URI");
                    WOLFSSL_ERROR_VERBOSE(ASN_ALT_NAME_E);
                    return ASN_ALT_NAME_E;
                }
            }
        #endif

            uriEntry = AltNameNew(cert->heap);
            if (uriEntry == NULL) {
                WOLFSSL_MSG("\tOut of Memory");
                return MEMORY_E;
            }

            uriEntry->type = ASN_URI_TYPE;
            uriEntry->name = (char*)XMALLOC((size_t)strLen + 1, cert->heap,
                                         DYNAMIC_TYPE_ALTNAME);
            if (uriEntry->name == NULL) {
                WOLFSSL_MSG("\tOut of Memory");
                XFREE(uriEntry, cert->heap, DYNAMIC_TYPE_ALTNAME);
                return MEMORY_E;
            }
            uriEntry->len = strLen;
            XMEMCPY(uriEntry->name, &input[idx], (size_t)strLen);
            uriEntry->name[strLen] = '\0';

            AddAltName(cert, uriEntry);

            length -= strLen;
            idx    += (word32)strLen;
        }
#ifdef WOLFSSL_IP_ALT_NAME
        else if (current_byte == (ASN_CONTEXT_SPECIFIC | ASN_IP_TYPE)) {
            DNS_entry* ipAddr;
            int strLen;
            word32 lenStartIdx = idx;
            WOLFSSL_MSG("Decoding Subject Alt. Name: IP Address");

            if (GetLength(input, &idx, &strLen, sz) < 0) {
                WOLFSSL_MSG("\tfail: str length");
                return ASN_PARSE_E;
            }
            length -= (idx - lenStartIdx);
            /* check that strLen at index is not past input buffer */
            if (strLen + idx > sz) {
                return BUFFER_E;
            }

            ipAddr = AltNameNew(cert->heap);
            if (ipAddr == NULL) {
                WOLFSSL_MSG("\tOut of Memory");
                return MEMORY_E;
            }

            ipAddr->type = ASN_IP_TYPE;
            ipAddr->name = (char*)XMALLOC((size_t)strLen + 1, cert->heap,
                                         DYNAMIC_TYPE_ALTNAME);
            if (ipAddr->name == NULL) {
                WOLFSSL_MSG("\tOut of Memory");
                XFREE(ipAddr, cert->heap, DYNAMIC_TYPE_ALTNAME);
                return MEMORY_E;
            }
            ipAddr->len = strLen;
            XMEMCPY(ipAddr->name, &input[idx], strLen);
            ipAddr->name[strLen] = '\0';

            if (GenerateDNSEntryIPString(ipAddr, cert->heap) != 0) {
                WOLFSSL_MSG("\tOut of Memory for IP string");
                XFREE(ipAddr->name, cert->heap, DYNAMIC_TYPE_ALTNAME);
                XFREE(ipAddr, cert->heap, DYNAMIC_TYPE_ALTNAME);
                return MEMORY_E;
            }
            AddAltName(cert, ipAddr);

            length -= strLen;
            idx    += (word32)strLen;
        }
#endif /* WOLFSSL_IP_ALT_NAME */
#ifdef WOLFSSL_RID_ALT_NAME
        else if (current_byte == (ASN_CONTEXT_SPECIFIC | ASN_RID_TYPE)) {
            DNS_entry* rid;
            int strLen;
            word32 lenStartIdx = idx;
            WOLFSSL_MSG("Decoding Subject Alt. Name: Registered Id");

            if (GetLength(input, &idx, &strLen, sz) < 0) {
                WOLFSSL_MSG("\tfail: str length");
                return ASN_PARSE_E;
            }
            length -= (idx - lenStartIdx);
            /* check that strLen at index is not past input buffer */
            if (strLen + idx > sz) {
                return BUFFER_E;
            }

            rid = AltNameNew(cert->heap);
            if (rid == NULL) {
                WOLFSSL_MSG("\tOut of Memory");
                return MEMORY_E;
            }

            rid->type = ASN_RID_TYPE;
            rid->name = (char*)XMALLOC((size_t)strLen + 1, cert->heap,
                                         DYNAMIC_TYPE_ALTNAME);
            if (rid->name == NULL) {
                WOLFSSL_MSG("\tOut of Memory");
                XFREE(rid, cert->heap, DYNAMIC_TYPE_ALTNAME);
                return MEMORY_E;
            }
            rid->len = strLen;
            XMEMCPY(rid->name, &input[idx], strLen);
            rid->name[strLen] = '\0';

            if (GenerateDNSEntryRIDString(rid, cert->heap) != 0) {
                WOLFSSL_MSG("\tOut of Memory for registered Id string");
                XFREE(rid->name, cert->heap, DYNAMIC_TYPE_ALTNAME);
                XFREE(rid, cert->heap, DYNAMIC_TYPE_ALTNAME);
                return MEMORY_E;
            }

            AddAltName(cert, rid);

            length -= strLen;
            idx    += (word32)strLen;
        }
#endif /* WOLFSSL_RID_ALT_NAME */
#endif /* IGNORE_NAME_CONSTRAINTS */
        else if (current_byte ==
                (ASN_CONTEXT_SPECIFIC | ASN_CONSTRUCTED | ASN_OTHER_TYPE)) {
            int strLen;
            word32 lenStartIdx = idx;
            word32 oid = 0;
            int    ret = 0;

            if (GetLength(input, &idx, &strLen, sz) < 0) {
                WOLFSSL_MSG("\tfail: other name length");
                return ASN_PARSE_E;
            }
            /* Consume the rest of this sequence. */
            length -= (int)(((word32)strLen + idx - lenStartIdx));

            if (GetObjectId(input, &idx, &oid, oidCertAltNameType, sz) < 0) {
                WOLFSSL_MSG("\tbad OID");
                return ASN_PARSE_E;
            }

            /* handle parsing other type alt names */
            switch (oid) {
            #ifdef WOLFSSL_SEP
                case HW_NAME_OID:
                    ret = DecodeSepHwAltName(cert, input, &idx, sz);
                    if (ret != 0)
                        return ret;
                    break;
            #endif /* WOLFSSL_SEP */
            #ifdef WOLFSSL_FPKI
                case FASCN_OID:
                case UPN_OID:
                    ret = DecodeConstructedOtherName(cert, input, &idx, sz,
                            oid);
                    if (ret != 0)
                        return ret;
                    break;
            #endif /* WOLFSSL_FPKI */

                default:
                    WOLFSSL_MSG("\tUnsupported other name type, skipping");
                    if (GetLength(input, &idx, &strLen, sz) < 0) {
                        /* check to skip constructed other names too */
                        if (DecodeConstructedOtherName(cert, input, &idx, sz,
                                    (int)oid) != 0) {
                            WOLFSSL_MSG("\tfail: unsupported other name length");
                            return ASN_PARSE_E;
                        }
                    }
                    else {
                        idx += (word32)strLen;
                    }
            }
            (void)ret;
        }
        else {
            int strLen;
            word32 lenStartIdx = idx;

            WOLFSSL_MSG("\tUnsupported name type, skipping");

            if (GetLength(input, &idx, &strLen, sz) < 0) {
                WOLFSSL_MSG("\tfail: unsupported name length");
                return ASN_PARSE_E;
            }
            length -= (int)((word32)strLen + idx - lenStartIdx);
            idx += (word32)strLen;
        }
    }

    return 0;
#else
    word32 idx = 0;
    int length = 0;
    int ret = 0;
    word32 numNames = 0;

    WOLFSSL_ENTER("DecodeAltNames");

    /* Get SEQUENCE and expect all data to be accounted for. */
    if (GetASN_Sequence(input, &idx, &length, sz, 1) != 0) {
        WOLFSSL_MSG("\tBad Sequence");
        ret = ASN_PARSE_E;
    }

    if ((ret == 0) && (length == 0)) {
        /* RFC 5280 4.2.1.6.  Subject Alternative Name
           If the subjectAltName extension is present, the sequence MUST
           contain at least one entry. */
        WOLFSSL_ERROR_VERBOSE(ASN_PARSE_E);
        ret = ASN_PARSE_E;
    }
    if (ret == 0) {
    #ifdef OPENSSL_ALL
        cert->extSubjAltNameSrc = input;
        cert->extSubjAltNameSz = sz;
    #endif

        cert->weOwnAltNames = 1;

        if ((word32)length + idx != sz) {
            ret = ASN_PARSE_E;
        }
    }

    while ((ret == 0) && (idx < sz)) {
        ASNGetData dataASN[altNameASN_Length];

        numNames++;
        if (numNames > WOLFSSL_MAX_ALT_NAMES) {
            WOLFSSL_MSG("\tToo many subject alternative names");
            ret = ASN_ALT_NAME_E;
            break;
        }

        /* Clear dynamic data items. */
        XMEMSET(dataASN, 0, sizeof(dataASN));
        /* Parse GeneralName with the choices supported. */
        GetASN_Choice(&dataASN[ALTNAMEASN_IDX_GN], generalNameChoice);
        /* Decode a GeneralName choice. */
        ret = GetASN_Items(altNameASN, dataASN, altNameASN_Length, 0, input,
                           &idx, sz);
        if (ret == 0) {
            ret = DecodeGeneralName(input, &idx, dataASN[ALTNAMEASN_IDX_GN].tag,
                                  (int)dataASN[ALTNAMEASN_IDX_GN].length, cert);
        }
    }

    return ret;
#endif
}

#ifdef WOLFSSL_ASN_TEMPLATE
/* ASN.1 template for BasicConstraints.
 * X.509: RFC 5280, 4.2.1.9 - BasicConstraints.
 */
static const ASNItem basicConsASN[] = {
/* SEQ  */ { 0, ASN_SEQUENCE, 1, 1, 0 },
/* CA   */     { 1, ASN_BOOLEAN, 0, 0, 1 },
/* PLEN */     { 1, ASN_INTEGER, 0, 0, 1 }
};
enum {
    BASICCONSASN_IDX_SEQ = 0,
    BASICCONSASN_IDX_CA,
    BASICCONSASN_IDX_PLEN
};

/* Number of items in ASN.1 template for BasicContraints. */
#define basicConsASN_Length (sizeof(basicConsASN) / sizeof(ASNItem))
#endif

/* Decode basic constraints extension
 *
 * X.509: RFC 5280, 4.2.1.9 - BasicConstraints.
 *
 * @param [in]      input          Buffer holding data.
 * @param [in]      sz             Size of data in buffer.
 * @param [out]     isCa           Whether it is a CA.
 * @param [out]     pathLength     CA path length.
 * @param [out]     pathLengthSet  Whether pathLength is valid on return.
 * @return  0 on success.
 * @return  MEMORY_E on dynamic memory allocation failure.
 * @return  ASN_PARSE_E when CA boolean is present and false (default is false).
 * @return  ASN_PARSE_E when CA boolean is not present unless
 *          WOLFSSL_X509_BASICCONS_INT is defined. Only a CA extension.
 * @return  ASN_PARSE_E when path length more than 7 bits.
 * @return  ASN_PARSE_E when BER encoded data does not match ASN.1 items or
 *          is invalid.
 * @return  BUFFER_E when data in buffer is too small.
 * @return  ASN_EXPECT_0_E when the INTEGER has the MSB set or NULL has a
 *          non-zero length.
 */
int DecodeBasicCaConstraint(const byte* input, int sz, byte *isCa,
                            word16 *pathLength, byte *pathLengthSet)
{
#ifndef WOLFSSL_ASN_TEMPLATE
    word32 idx = 0;
    int length = 0;
    int ret;

    WOLFSSL_ENTER("DecodeBasicCaConstraint");

    if (GetSequence(input, &idx, &length, (word32)sz) < 0) {
        WOLFSSL_MSG("\tfail: bad SEQUENCE");
        return ASN_PARSE_E;
    }

    if (length == 0)
        return 0;

    /* If the basic ca constraint is false, this extension may be named, but
     * left empty. So, if the length is 0, just return. */

    ret = GetBoolean(input, &idx, (word32)sz);

    /* Removed logic for WOLFSSL_X509_BASICCONS_INT which was mistreating the
     * pathlen value as if it were the CA Boolean value 7/2/2021 - KH.
     * When CA Boolean not asserted use the default value "False" */
    if (ret < 0) {
        WOLFSSL_MSG("\tfail: constraint not valid BOOLEAN, set default FALSE");
        ret = 0;
    }

    *isCa = ret ? 1 : 0;

    /* If there isn't any more data, return. */
    if (idx >= (word32)sz) {
        return 0;
    }

    ret = GetInteger16Bit(input, &idx, (word32)sz);
    if (ret < 0)
        return ret;
    else if (ret > WOLFSSL_MAX_PATH_LEN) {
        WOLFSSL_ERROR_VERBOSE(ASN_PATHLEN_SIZE_E);
        return ASN_PATHLEN_SIZE_E;
    }

    *pathLength = (word16)ret;
    *pathLengthSet = 1;

    return 0;
#else
    DECL_ASNGETDATA(dataASN, basicConsASN_Length);
    int ret = 0;
    word32 idx = 0;
    byte innerIsCA = 0;

    WOLFSSL_ENTER("DecodeBasicCaConstraint");

    CALLOC_ASNGETDATA(dataASN, basicConsASN_Length, ret, NULL);

    if (ret == 0) {
        /* Get the CA boolean and path length when present. */
        GetASN_Boolean(&dataASN[BASICCONSASN_IDX_CA], &innerIsCA);
        GetASN_Int16Bit(&dataASN[BASICCONSASN_IDX_PLEN], pathLength);

        ret = GetASN_Items(basicConsASN, dataASN, basicConsASN_Length, 1, input,
                           &idx, (word32)sz);
    }

    /* Empty SEQUENCE is OK - nothing to store. */
    if ((ret == 0) && (dataASN[BASICCONSASN_IDX_SEQ].length != 0)) {
        /* Bad encoding when CA Boolean is false
         * (default when not present). */
#if !defined(ASN_TEMPLATE_SKIP_ISCA_CHECK) && \
    !defined(WOLFSSL_ALLOW_ENCODING_CA_FALSE)
        if ((dataASN[BASICCONSASN_IDX_CA].length != 0) && (!innerIsCA)) {
            WOLFSSL_ERROR_VERBOSE(ASN_PARSE_E);
            ret = ASN_PARSE_E;
        }
#endif
        /* Path length must be a 7-bit value. */
        if ((ret == 0) && (*pathLength >= (1 << 7))) {
            WOLFSSL_ERROR_VERBOSE(ASN_PARSE_E);
            ret = ASN_PARSE_E;
        }
        if ((ret == 0) && *pathLength > WOLFSSL_MAX_PATH_LEN) {
            WOLFSSL_ERROR_VERBOSE(ASN_PATHLEN_SIZE_E);
            ret = ASN_PATHLEN_SIZE_E;
        }
        /* Store CA boolean and whether a path length was seen. */
        if (ret == 0) {
            /* isCA in certificate is a 1 bit of a byte. */
            *isCa = innerIsCA ? 1 : 0;
            *pathLengthSet = (dataASN[BASICCONSASN_IDX_PLEN].length > 0);
        }
    }

    FREE_ASNGETDATA(dataASN, NULL);
    return ret;
#endif

}

/* Decode basic constraints extension in a certificate.
 *
 * X.509: RFC 5280, 4.2.1.9 - BasicConstraints.
 *
 * @param [in]      input  Buffer holding data.
 * @param [in]      sz     Size of data in buffer.
 * @param [in, out] cert   Certificate object.
 * @return  0 on success.
 * @return  MEMORY_E on dynamic memory allocation failure.
 * @return  ASN_PARSE_E when CA boolean is present and false (default is false).
 * @return  ASN_PARSE_E when CA boolean is not present unless
 *          WOLFSSL_X509_BASICCONS_INT is defined. Only a CA extension.
 * @return  ASN_PARSE_E when path length more than 7 bits.
 * @return  ASN_PARSE_E when BER encoded data does not match ASN.1 items or
 *          is invalid.
 * @return  BUFFER_E when data in buffer is too small.
 * @return  ASN_EXPECT_0_E when the INTEGER has the MSB set or NULL has a
 *          non-zero length.
 */
static int DecodeBasicCaConstraintInternal(const byte* input, int sz,
                                           DecodedCert* cert)
{
    int ret;
    byte isCa = 0;
    word16 pathLength = 0;
    byte pathLengthSet = 0;
    ret = DecodeBasicCaConstraint(input, sz, &isCa, &pathLength,
                                  &pathLengthSet);
    if (ret != 0)
        return ret;

    cert->isCA = isCa ? 1 : 0;
    cert->pathLengthSet = pathLengthSet ? 1 : 0;
    if (pathLengthSet) {
        cert->pathLength = pathLength;
    }

    return 0;
}


static int DecodePolicyConstraints(const byte* input, int sz, DecodedCert* cert)
{
    word32 idx = 0;
    int length = 0;
    int skipLength = 0;
    int ret;
    byte tag;

    WOLFSSL_ENTER("DecodePolicyConstraints");

    if (GetSequence(input, &idx, &length, (word32)sz) < 0) {
        WOLFSSL_MSG("\tfail: bad SEQUENCE");
        return ASN_PARSE_E;
    }

    if (length == 0)
        return ASN_PARSE_E;

    if (GetASNTag(input, &idx, &tag, (word32)sz) < 0) {
        WOLFSSL_MSG("\tfail: bad TAG");
        return ASN_PARSE_E;
    }

    if (tag == (ASN_CONTEXT_SPECIFIC | 0)) {
        /* requireExplicitPolicy */
        cert->extPolicyConstRxpSet = 1;
    }
    else if (tag == (ASN_CONTEXT_SPECIFIC | 1)) {
        /* inhibitPolicyMapping */
        cert->extPolicyConstIpmSet = 1;
    }
    else {
        WOLFSSL_MSG("\tfail: invalid TAG");
        return ASN_PARSE_E;
    }

    ret = GetLength(input, &idx, &skipLength, (word32)sz);
    if (ret < 0) {
        WOLFSSL_MSG("\tfail: invalid length");
        return ret;
    }
    if (skipLength > 1) {
        WOLFSSL_MSG("\tfail: skip value too big");
        return BUFFER_E;
    }
    if (idx >= (word32)sz) {
        WOLFSSL_MSG("\tfail: no policy const skip to read");
        return BUFFER_E;
    }
    cert->policyConstSkip = input[idx];

    return 0;
}


/* Context-Specific value for: DistributionPoint.distributionPoint
 * From RFC5280 SS4.2.1.13, Distribution Point */
#define DISTRIBUTION_POINT  (ASN_CONTEXT_SPECIFIC | 0)
/* Context-Specific value for: DistributionPoint.DistributionPointName.fullName
 *  From RFC3280 SS4.2.1.13, Distribution Point Name */
#define CRLDP_FULL_NAME     (ASN_CONTEXT_SPECIFIC | 0)
/* Context-Specific value for choice: GeneralName.uniformResourceIdentifier
 * From RFC3280 SS4.2.1.7, GeneralName */
#define GENERALNAME_URI     (ASN_CONTEXT_SPECIFIC | 6)

#ifdef WOLFSSL_ASN_TEMPLATE
/* ASN.1 template for CRL distribution points.
 * X.509: RFC 5280, 4.2.1.13 - CRL Distribution Points.
 */
static const ASNItem crlDistASN[] = {
/* SEQ                */ { 0, ASN_SEQUENCE, 1, 1, 0 },
/* DP_SEQ             */     { 1, ASN_SEQUENCE, 1, 1, 0 },
                                                /* Distribution point name */
/* DP_DISTPOINT       */         { 2, DISTRIBUTION_POINT, 1, 1, 1 },
                                                    /* fullName */
/* DP_DISTPOINT_FN    */             { 3, CRLDP_FULL_NAME, 1, 1, 2 },
/* DP_DISTPOINT_FN_GN */                 { 4, GENERALNAME_URI, 0, 0, 0 },
                                                    /* nameRelativeToCRLIssuer */
/* DP_DISTPOINT_RN    */             { 3, ASN_CONTEXT_SPECIFIC | 1, 1, 0, 2 },
                                                /* reasons: IMPLICIT BIT STRING */
/* DP_REASONS         */         { 2, ASN_CONTEXT_SPECIFIC | 1, 0, 0, 1 },
                                                /* cRLIssuer */
/* DP_CRLISSUER       */         { 2, ASN_CONTEXT_SPECIFIC | 2, 1, 0, 1 },
};
enum {
    CRLDISTASN_IDX_SEQ = 0,
    CRLDISTASN_IDX_DP_SEQ,
    CRLDISTASN_IDX_DP_DISTPOINT,
    CRLDISTASN_IDX_DP_DISTPOINT_FN,
    CRLDISTASN_IDX_DP_DISTPOINT_FN_GN,
    CRLDISTASN_IDX_DP_DISTPOINT_RN, /* Relative name */
    CRLDISTASN_IDX_DP_REASONS,
    CRLDISTASN_IDX_DP_CRLISSUER
};

/* Number of items in ASN.1 template for CRL distribution points. */
#define crlDistASN_Length (sizeof(crlDistASN) / sizeof(ASNItem))
#endif

/* Decode CRL distribution point extension in a certificate.
 *
 * X.509: RFC 5280, 4.2.1.13 - CRL Distribution Points.
 *
 * @param [in]      input  Buffer holding data.
 * @param [in]      sz     Size of data in buffer.
 * @param [in, out] cert   Certificate object.
 * @return  0 on success.
 * @return  MEMORY_E on dynamic memory allocation failure.
 * @return  ASN_PARSE_E when invalid bits of reason are set.
 * @return  ASN_PARSE_E when BITSTRING value is more than 2 bytes.
 * @return  ASN_PARSE_E when unused bits of BITSTRING is invalid.
 * @return  ASN_PARSE_E when BER encoded data does not match ASN.1 items or
 *          is invalid.
 * @return  BUFFER_E when data in buffer is too small.
 */
static int DecodeCrlDist(const byte* input, word32 sz, DecodedCert* cert)
{
#ifndef WOLFSSL_ASN_TEMPLATE
    word32 idx = 0, localIdx;
    int length = 0;
    byte tag   = 0;

    WOLFSSL_ENTER("DecodeCrlDist");

    cert->extCrlInfoRaw = input;
    cert->extCrlInfoRawSz = (int)sz;

    /* Unwrap the list of Distribution Points*/
    if (GetSequence(input, &idx, &length, sz) < 0)
        return ASN_PARSE_E;

    /* Unwrap a single Distribution Point */
    if (GetSequence(input, &idx, &length, sz) < 0)
        return ASN_PARSE_E;

    /* The Distribution Point has three explicit optional members
     *  First check for a DistributionPointName
     */
    localIdx = idx;
    if (GetASNTag(input, &localIdx, &tag, sz) == 0 &&
            tag == (ASN_CONSTRUCTED | DISTRIBUTION_POINT))
    {
        idx++;
        if (GetLength(input, &idx, &length, sz) < 0)
            return ASN_PARSE_E;

        localIdx = idx;
        if (GetASNTag(input, &localIdx, &tag, sz) == 0 &&
                tag == (ASN_CONSTRUCTED | CRLDP_FULL_NAME))
        {
            idx++;
            if (GetLength(input, &idx, &length, sz) < 0)
                return ASN_PARSE_E;

            localIdx = idx;
            if (GetASNTag(input, &localIdx, &tag, sz) == 0 &&
                    tag == GENERALNAME_URI)
            {
                idx++;
                if (GetLength(input, &idx, &length, sz) < 0)
                    return ASN_PARSE_E;

                cert->extCrlInfoSz = length;
                cert->extCrlInfo = input + idx;
                idx += (word32)length;
            }
            else
                /* This isn't a URI, skip it. */
                idx += (word32)length;
        }
        else {
            /* This isn't a FULLNAME, skip it. */
            idx += (word32)length;
        }
    }

    /* Check for reasonFlags */
    localIdx = idx;
    if (idx < (word32)sz &&
        GetASNTag(input, &localIdx, &tag, sz) == 0 &&
        tag == (ASN_CONSTRUCTED | ASN_CONTEXT_SPECIFIC | 1))
    {
        idx++;
        if (GetLength(input, &idx, &length, sz) < 0)
            return ASN_PARSE_E;
        idx += (word32)length;
    }

    /* Check for cRLIssuer */
    localIdx = idx;
    if (idx < (word32)sz &&
        GetASNTag(input, &localIdx, &tag, sz) == 0 &&
        tag == (ASN_CONSTRUCTED | ASN_CONTEXT_SPECIFIC | 2))
    {
        idx++;
        if (GetLength(input, &idx, &length, sz) < 0)
            return ASN_PARSE_E;
        idx += (word32)length;
    }

    if (idx < (word32)sz)
    {
        WOLFSSL_MSG("\tThere are more CRL Distribution Point records, "
                   "but we only use the first one.");
    }

    return 0;
#else
    DECL_ASNGETDATA(dataASN, crlDistASN_Length);
    word32 idx = 0;
    int ret = 0;
#ifdef CRLDP_VALIDATE_DATA
    word16 reason;
#endif

    WOLFSSL_ENTER("DecodeCrlDist");

    CALLOC_ASNGETDATA(dataASN, crlDistASN_Length, ret, cert->heap);

    cert->extCrlInfoRaw = input;
    cert->extCrlInfoRawSz = (int)sz;

    if  (ret == 0) {
        /* Get the GeneralName choice */
        GetASN_Choice(&dataASN[CRLDISTASN_IDX_DP_DISTPOINT_FN_GN], generalNameChoice);
        /* Parse CRL distribution point. */
        ret = GetASN_Items(crlDistASN, dataASN, crlDistASN_Length, 0, input,
                           &idx, sz);
    }
    if (ret == 0) {
        /* If the choice was a URI, store it in certificate. */
        if (dataASN[CRLDISTASN_IDX_DP_DISTPOINT_FN_GN].tag == GENERALNAME_URI) {
            word32 sz32;
            GetASN_GetConstRef(&dataASN[CRLDISTASN_IDX_DP_DISTPOINT_FN_GN],
                    &cert->extCrlInfo, &sz32);
            cert->extCrlInfoSz = (int)sz32;
        }

    #ifdef CRLDP_VALIDATE_DATA
        if (dataASN[CRLDISTASN_IDX_DP_REASONS].data.ref.data != NULL) {
             /* TODO: test case */
             /* Validate ReasonFlags. */
             ret = GetASN_BitString_Int16Bit(&dataASN[CRLDISTASN_IDX_DP_REASONS],
                     &reason);
             /* First bit (LSB) unused and eight other bits defined. */
             if ((ret == 0) && ((reason >> 9) || (reason & 0x01))) {
                WOLFSSL_ERROR_VERBOSE(ASN_PARSE_E);
                ret = ASN_PARSE_E;
             }
        }
    #endif
    }

    /* Only parsing the first one. */
    if (ret == 0 && idx < (word32)sz) {
        WOLFSSL_MSG("\tThere are more CRL Distribution Point records, "
                    "but we only use the first one.");
    }
    /* TODO: validate other points. */

    FREE_ASNGETDATA(dataASN, cert->heap);
    return ret;
#endif /* WOLFSSL_ASN_TEMPLATE */
}

#ifdef WOLFSSL_ASN_TEMPLATE
/* ASN.1 template for the access description.
 * X.509: RFC 5280, 4.2.2.1 - Authority Information Access.
 */
static const ASNItem accessDescASN[] = {
/* SEQ  */ { 0, ASN_SEQUENCE, 1, 1, 0 },
                                 /* accessMethod */
/* METH */     { 1, ASN_OBJECT_ID, 0, 0, 0 },
                                 /* accessLocation: GeneralName */
/* LOC  */     { 1, ASN_CONTEXT_SPECIFIC | 0, 0, 0, 0 },
};
enum {
    ACCESSDESCASN_IDX_SEQ = 0,
    ACCESSDESCASN_IDX_METH,
    ACCESSDESCASN_IDX_LOC
};

/* Number of items in ASN.1 template for the access description. */
#define accessDescASN_Length (sizeof(accessDescASN) / sizeof(ASNItem))
#endif

/* Decode authority information access extension in a certificate.
 *
 * X.509: RFC 5280, 4.2.2.1 - Authority Information Access.
 *
 * @param [in]      input  Buffer holding data.
 * @param [in]      sz     Size of data in buffer.
 * @param [in, out] cert   Certificate object.
 * @return  0 on success.
 * @return  MEMORY_E on dynamic memory allocation failure.
 * @return  ASN_PARSE_E when BER encoded data does not match ASN.1 items or
 *          is invalid.
 * @return  BUFFER_E when data in buffer is too small.
 * @return  ASN_OBJECT_ID_E when the expected OBJECT_ID tag is not found.
 * @return  ASN_UNKNOWN_OID_E when the OID cannot be verified.
 */
static int DecodeAuthInfo(const byte* input, word32 sz, DecodedCert* cert)
{
#ifndef WOLFSSL_ASN_TEMPLATE
    word32 idx = 0;
    int length = 0;
    byte b = 0;
    word32 oid;

    WOLFSSL_ENTER("DecodeAuthInfo");

    /* Unwrap the list of AIAs */
    if (GetSequence(input, &idx, &length, sz) < 0)
        return ASN_PARSE_E;

    while ((idx < (word32)sz)) {
        /* Unwrap a single AIA */
        if (GetSequence(input, &idx, &length, sz) < 0)
            return ASN_PARSE_E;

        oid = 0;
        if (GetObjectId(input, &idx, &oid, oidCertAuthInfoType, sz) < 0) {
            return ASN_PARSE_E;
        }

        /* Only supporting URIs right now. */
        if (GetASNTag(input, &idx, &b, sz) < 0)
            return ASN_PARSE_E;

        if (GetLength(input, &idx, &length, sz) < 0)
            return ASN_PARSE_E;

        /* Set ocsp entry */
        if (b == GENERALNAME_URI && oid == AIA_OCSP_OID &&
                cert->extAuthInfo == NULL) {
            cert->extAuthInfoSz = length;
            cert->extAuthInfo = input + idx;
        }
    #ifdef WOLFSSL_ASN_CA_ISSUER
        /* Set CaIssuers entry */
        else if ((b == GENERALNAME_URI) && oid == AIA_CA_ISSUER_OID &&
                cert->extAuthInfoCaIssuer == NULL)
        {
            cert->extAuthInfoCaIssuerSz = length;
            cert->extAuthInfoCaIssuer = input + idx;
        }
    #endif
        idx += (word32)length;
    }

    return 0;
#else
    word32 idx = 0;
    int length = 0;
    int ret    = 0;

    WOLFSSL_ENTER("DecodeAuthInfo");

    /* Unwrap the list of AIAs */
    if (GetASN_Sequence(input, &idx, &length, sz, 1) < 0) {
        ret = ASN_PARSE_E;
    }

    while ((ret == 0) && (idx < (word32)sz)) {
        ASNGetData dataASN[accessDescASN_Length];

        /* Clear dynamic data and retrieve OID and name. */
        XMEMSET(dataASN, 0, sizeof(dataASN));
        GetASN_OID(&dataASN[ACCESSDESCASN_IDX_METH], oidCertAuthInfoType);
        GetASN_Choice(&dataASN[ACCESSDESCASN_IDX_LOC], generalNameChoice);
        /* Parse AccessDescription. */
        ret = GetASN_Items(accessDescASN, dataASN, accessDescASN_Length, 0,
                           input, &idx, sz);
        if (ret == 0) {
            word32 sz32;

            /* Check we have OCSP and URI. */
            if ((dataASN[ACCESSDESCASN_IDX_METH].data.oid.sum == AIA_OCSP_OID) &&
                    (dataASN[ACCESSDESCASN_IDX_LOC].tag == GENERALNAME_URI) &&
                    (cert->extAuthInfo == NULL)) {
                /* Store URI for OCSP lookup. */
                GetASN_GetConstRef(&dataASN[ACCESSDESCASN_IDX_LOC],
                        &cert->extAuthInfo, &sz32);
                cert->extAuthInfoSz = (int)sz32;
            }
        #ifdef WOLFSSL_ASN_CA_ISSUER
            /* Check we have CA Issuer and URI. */
            else if ((dataASN[ACCESSDESCASN_IDX_METH].data.oid.sum ==
                        AIA_CA_ISSUER_OID) &&
                    (dataASN[ACCESSDESCASN_IDX_LOC].tag == GENERALNAME_URI) &&
                    (cert->extAuthInfoCaIssuer == NULL)) {
                /* Set CaIssuers entry */
                GetASN_GetConstRef(&dataASN[ACCESSDESCASN_IDX_LOC],
                        &cert->extAuthInfoCaIssuer, &sz32);
                cert->extAuthInfoCaIssuerSz = (int)sz32;
            }
        #endif
            /* Otherwise skip. */
        }
    }

    return ret;
#endif
}


#ifdef WOLFSSL_ASN_TEMPLATE
/* ASN.1 template for AuthorityKeyIdentifier.
 * X.509: RFC 5280, 4.2.1.1 - Authority Key Identifier.
 */
static const ASNItem authKeyIdASN[] = {
/* SEQ    */    { 0, ASN_SEQUENCE, 1, 1, 0 },
                                     /* keyIdentifier */
/* KEYID  */        { 1, ASN_CONTEXT_SPECIFIC | ASN_AUTHKEYID_KEYID, 0, 0, 1 },
                                     /* authorityCertIssuer */
/* ISSUER */        { 1, ASN_CONTEXT_SPECIFIC | ASN_AUTHKEYID_ISSUER, 1, 0, 1 },
                                     /* authorityCertSerialNumber */
/* SERIAL */        { 1, ASN_CONTEXT_SPECIFIC | ASN_AUTHKEYID_SERIAL, 0, 0, 1 },
};
enum {
    AUTHKEYIDASN_IDX_SEQ = 0,
    AUTHKEYIDASN_IDX_KEYID,
    AUTHKEYIDASN_IDX_ISSUER,
    AUTHKEYIDASN_IDX_SERIAL
};

/* Number of items in ASN.1 template for AuthorityKeyIdentifier. */
#define authKeyIdASN_Length (sizeof(authKeyIdASN) / sizeof(ASNItem))
#endif

/* Decode authority key identifier extension.
 *
 * X.509: RFC 5280, 4.2.1.1 - Authority Key Identifier.
 *
 * @param [in]   input                   Buffer holding data.
 * @param [in]   sz                      Size of data in buffer.
 * @param [out]  extAuthKeyId            Beginning of the ID. NULL if not
 *                                       present.
 * @param [out]  extAuthKeyIdSz          Size of data in extAuthKeyId. 0 if not
 *                                       present.
 * @param [out]  extAuthKeyIdIssuer      Beginning of the Issuer. NULL if not
 *                                       present.
 * @param [out]  extAuthKeyIdIssuerSz    Size of data in extAuthKeyIdIssuer. 0
 *                                       if not present.
 * @param [out]  extAuthKeyIdIssuerSN    Beginning of the Issuer Serial. NULL
 *                                       if not present.
 * @param [out]  extAuthKeyIdIssuerSNSz  Size of data in extAuthKeyIdIssuerSN.
 *                                       0 if not present.
 * @return  0 on success.
 * @return  MEMORY_E on dynamic memory allocation failure.
 * @return  ASN_PARSE_E when BER encoded data does not match ASN.1 items or
 *          is invalid.
 * @return  BUFFER_E when data in buffer is too small.
 */
int DecodeAuthKeyId(const byte* input, word32 sz, const byte **extAuthKeyId,
        word32 *extAuthKeyIdSz, const byte **extAuthKeyIdIssuer,
        word32 *extAuthKeyIdIssuerSz, const byte **extAuthKeyIdIssuerSN,
        word32 *extAuthKeyIdIssuerSNSz)
{
#ifndef WOLFSSL_ASN_TEMPLATE
    word32 idx = 0;
    int length = 0;
    byte tag;

    WOLFSSL_ENTER("DecodeAuthKeyId");

    if (extAuthKeyId)
        *extAuthKeyId = NULL;
    if (extAuthKeyIdSz)
        *extAuthKeyIdSz = 0;

    if (extAuthKeyIdIssuer)
        *extAuthKeyIdIssuer = NULL;
    if (extAuthKeyIdIssuerSz)
        *extAuthKeyIdIssuerSz = 0;

    if (extAuthKeyIdIssuerSN)
        *extAuthKeyIdIssuerSN = NULL;
    if (extAuthKeyIdIssuerSNSz)
        *extAuthKeyIdIssuerSNSz = 0;

    if (GetSequence(input, &idx, &length, sz) < 0) {
        WOLFSSL_MSG("\tfail: should be a SEQUENCE");
        return ASN_PARSE_E;
    }

    if (GetASNTag(input, &idx, &tag, sz) < 0) {
        return ASN_PARSE_E;
    }

    if (tag != (ASN_CONTEXT_SPECIFIC | 0)) {
        WOLFSSL_MSG("\tinfo: OPTIONAL item 0, not available");
        return 0;
    }

    if (GetLength(input, &idx, &length, sz) <= 0) {
        WOLFSSL_MSG("\tfail: extension data length");
        return ASN_PARSE_E;
    }

    if (extAuthKeyIdSz)
        *extAuthKeyIdSz = length;
    if (extAuthKeyId)
        *extAuthKeyId = &input[idx];
    return 0;

#else
    DECL_ASNGETDATA(dataASN, authKeyIdASN_Length);
    int ret = 0;

    WOLFSSL_ENTER("DecodeAuthKeyId");

    if (extAuthKeyId)
        *extAuthKeyId = NULL;
    if (extAuthKeyIdSz)
        *extAuthKeyIdSz = 0;

    if (extAuthKeyIdIssuer)
        *extAuthKeyIdIssuer = NULL;
    if (extAuthKeyIdIssuerSz)
        *extAuthKeyIdIssuerSz = 0;

    if (extAuthKeyIdIssuerSN)
        *extAuthKeyIdIssuerSN = NULL;
    if (extAuthKeyIdIssuerSNSz)
        *extAuthKeyIdIssuerSNSz = 0;

    CALLOC_ASNGETDATA(dataASN, authKeyIdASN_Length, ret, NULL);

    if (ret == 0) {
        /* Parse an authority key identifier. */
        word32 idx = 0;
        ret = GetASN_Items(authKeyIdASN, dataASN, authKeyIdASN_Length, 1, input,
                           &idx, sz);
    }
    /* Each field is optional */
    if (ret == 0 && extAuthKeyId && extAuthKeyIdSz &&
            dataASN[AUTHKEYIDASN_IDX_KEYID].data.ref.data != NULL) {
        GetASN_GetConstRef(&dataASN[AUTHKEYIDASN_IDX_KEYID],
                extAuthKeyId, extAuthKeyIdSz);
    }
#ifdef WOLFSSL_AKID_NAME
    if (ret == 0 && dataASN[AUTHKEYIDASN_IDX_ISSUER].data.ref.data != NULL) {
        /* We only support using one (first) name. Parse the name to perform
         * a sanity check. */
        word32 idx = 0;
        ASNGetData nameASN[altNameASN_Length];
        XMEMSET(nameASN, 0, sizeof(nameASN));
        /* Parse GeneralName with the choices supported. */
        GetASN_Choice(&nameASN[ALTNAMEASN_IDX_GN], generalNameChoice);
        /* Decode a GeneralName choice. */
        ret = GetASN_Items(altNameASN, nameASN, altNameASN_Length, 0,
                dataASN[AUTHKEYIDASN_IDX_ISSUER].data.ref.data, &idx,
                dataASN[AUTHKEYIDASN_IDX_ISSUER].data.ref.length);

        if (ret == 0 && extAuthKeyIdIssuer && extAuthKeyIdIssuerSz) {
            GetASN_GetConstRef(&nameASN[ALTNAMEASN_IDX_GN],
                    extAuthKeyIdIssuer, extAuthKeyIdIssuerSz);
        }
    }
    if (ret == 0 && extAuthKeyIdIssuerSN && extAuthKeyIdIssuerSNSz &&
            dataASN[AUTHKEYIDASN_IDX_SERIAL].data.ref.data != NULL) {
        GetASN_GetConstRef(&dataASN[AUTHKEYIDASN_IDX_SERIAL],
                extAuthKeyIdIssuerSN, extAuthKeyIdIssuerSNSz);
    }
    if (ret == 0 && extAuthKeyIdIssuerSz && extAuthKeyIdIssuerSNSz) {
        if ((*extAuthKeyIdIssuerSz > 0) ^
                (*extAuthKeyIdIssuerSNSz > 0)) {
            WOLFSSL_MSG("authorityCertIssuer and authorityCertSerialNumber MUST"
                       " both be present or both be absent");
        }
    }
#endif /* WOLFSSL_AKID_NAME */

    FREE_ASNGETDATA(dataASN, NULL);
    return ret;
#endif /* WOLFSSL_ASN_TEMPLATE */
}

/* Decode authority key identifier extension in a certificate.
 *
 * X.509: RFC 5280, 4.2.1.1 - Authority Key Identifier.
 *
 * @param [in]      input  Buffer holding data.
 * @param [in]      sz     Size of data in buffer.
 * @param [in, out] cert   Certificate object.
 * @return  0 on success.
 * @return  MEMORY_E on dynamic memory allocation failure.
 * @return  ASN_PARSE_E when BER encoded data does not match ASN.1 items or
 *          is invalid.
 * @return  BUFFER_E when data in buffer is too small.
 */
static int DecodeAuthKeyIdInternal(const byte* input, word32 sz,
                                   DecodedCert* cert)
{
    int ret;
    const byte *extAuthKeyId = NULL;
    word32 extAuthKeyIdSz = 0;
    const byte *extAuthKeyIdIssuer = NULL;
    word32 extAuthKeyIdIssuerSz = 0;
    const byte *extAuthKeyIdIssuerSN = NULL;
    word32 extAuthKeyIdIssuerSNSz = 0;

    ret = DecodeAuthKeyId(input, sz, &extAuthKeyId, &extAuthKeyIdSz,
            &extAuthKeyIdIssuer, &extAuthKeyIdIssuerSz, &extAuthKeyIdIssuerSN,
            &extAuthKeyIdIssuerSNSz);

    if (ret != 0)
        return ret;

#ifndef WOLFSSL_ASN_TEMPLATE

    if (extAuthKeyIdSz == 0)
    {
        cert->extAuthKeyIdSet = 0;
        return 0;
    }

    cert->extAuthKeyIdSz = extAuthKeyIdSz;

#if defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)
#ifdef WOLFSSL_AKID_NAME
    cert->extRawAuthKeyIdSrc = input;
    cert->extRawAuthKeyIdSz = sz;
#endif
    cert->extAuthKeyIdSrc = extAuthKeyId;
#endif /* OPENSSL_EXTRA */

    return GetHashId(extAuthKeyId, extAuthKeyIdSz, cert->extAuthKeyId,
        HashIdAlg(cert->signatureOID));
#else

    /* Each field is optional */
    if (extAuthKeyIdSz > 0) {
#ifdef OPENSSL_EXTRA
        cert->extAuthKeyIdSrc = extAuthKeyId;
        cert->extAuthKeyIdSz = extAuthKeyIdSz;
#endif /* OPENSSL_EXTRA */
        /* Get the hash or hash of the hash if wrong size. */
        ret = GetHashId(extAuthKeyId, (int)extAuthKeyIdSz, cert->extAuthKeyId,
                        HashIdAlg(cert->signatureOID));
    }
#ifdef WOLFSSL_AKID_NAME
    if (ret == 0 && extAuthKeyIdIssuerSz > 0) {
        cert->extAuthKeyIdIssuer = extAuthKeyIdIssuer;
        cert->extAuthKeyIdIssuerSz = extAuthKeyIdIssuerSz;
    }
    if (ret == 0 && extAuthKeyIdIssuerSNSz > 0) {
        cert->extAuthKeyIdIssuerSN = extAuthKeyIdIssuerSN;
        cert->extAuthKeyIdIssuerSNSz = extAuthKeyIdIssuerSNSz;
    }
#endif /* WOLFSSL_AKID_NAME */
    if (ret == 0) {
#if defined(OPENSSL_EXTRA) && defined(WOLFSSL_AKID_NAME)
        /* Store the raw authority key id. */
        cert->extRawAuthKeyIdSrc = input;
        cert->extRawAuthKeyIdSz = sz;
#endif /* OPENSSL_EXTRA */
    }

    return ret;
#endif /* WOLFSSL_ASN_TEMPLATE */
}

/* Decode subject key id extension.
 *
 * X.509: RFC 5280, 4.2.1.2 - Subject Key Identifier.
 *
 * @param [in]   input          Buffer holding data.
 * @param [in]   sz             Size of data in buffer.
 * @param [out]  extSubjKeyId   Beginning of the ID.
 * @param [out]  extSubjKeyIdSz Size of data in extSubjKeyId.
 * @return  0 on success.
 * @return  ASN_PARSE_E when the OCTET_STRING tag is not found or length is
 *          invalid.
 * @return  MEMORY_E on dynamic memory allocation failure.
 */
int DecodeSubjKeyId(const byte* input, word32 sz, const byte **extSubjKeyId,
                    word32 *extSubjKeyIdSz)
{
    word32 idx = 0;
    int length = 0;
    int ret = 0;

    WOLFSSL_ENTER("DecodeSubjKeyId");

    ret = GetOctetString(input, &idx, &length, sz);
    if (ret < 0)
        return ret;

    *extSubjKeyIdSz = (word32)length;
    *extSubjKeyId = &input[idx];
    return 0;
}

/* Decode subject key id extension in a certificate.
 *
 * X.509: RFC 5280, 4.2.1.2 - Subject Key Identifier.
 *
 * @param [in]      input  Buffer holding data.
 * @param [in]      sz     Size of data in buffer.
 * @param [in, out] cert   Certificate object.
 * @return  0 on success.
 * @return  ASN_PARSE_E when the OCTET_STRING tag is not found or length is
 *          invalid.
 * @return  MEMORY_E on dynamic memory allocation failure.
 */
static int DecodeSubjKeyIdInternal(const byte* input, word32 sz,
                                   DecodedCert* cert)
{
    int ret = 0;
    const byte *extSubjKeyId = NULL;
    word32 extSubjKeyIdSz = 0;

    ret = DecodeSubjKeyId(input, sz, &extSubjKeyId, &extSubjKeyIdSz);
    if (ret != 0)
        return ret;

    cert->extSubjKeyIdSz = extSubjKeyIdSz;

#if defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)
    cert->extSubjKeyIdSrc = extSubjKeyId;
#endif /* OPENSSL_EXTRA */

    /* Get the hash or hash of the hash if wrong size. */
    ret = GetHashId(extSubjKeyId, (int)extSubjKeyIdSz, cert->extSubjKeyId,
        HashIdAlg(cert->signatureOID));

    return ret;
}

#ifdef WOLFSSL_ASN_TEMPLATE
/* ASN.1 template for KeyUsage.
 * X.509: RFC 5280, 4.2.1.3 - Key Usage.
 */
static const ASNItem keyUsageASN[] = {
/* STR */ { 0, ASN_BIT_STRING, 0, 0, 0 },
};
enum {
    KEYUSAGEASN_IDX_STR = 0
};

/* Number of items in ASN.1 template for KeyUsage. */
#define keyUsageASN_Length (sizeof(keyUsageASN) / sizeof(ASNItem))
#endif

/* Decode key usage extension in a certificate.
 *
 * X.509: RFC 5280, 4.2.1.3 - Key Usage.
 *
 * @param [in]      input        Buffer holding data.
 * @param [in]      sz           Size of data in buffer.
 * @param [out]     extKeyUsage  Key usage bitfield.
 * @return  0 on success.
 * @return  ASN_BITSTR_E when the expected BIT_STRING tag is not found.
 * @return  ASN_PARSE_E when BER encoded data does not match ASN.1 items or
 *          is invalid.
 * @return  MEMORY_E on dynamic memory allocation failure.
 */
int DecodeKeyUsage(const byte* input, word32 sz, word16 *extKeyUsage)
{
#ifndef WOLFSSL_ASN_TEMPLATE
    word32 idx = 0;
    int length;
    int ret;
    WOLFSSL_ENTER("DecodeKeyUsage");

    ret = CheckBitString(input, &idx, &length, sz, 0, NULL);
    if (ret != 0)
        return ret;

    if (length == 0 || length > 2)
        return ASN_PARSE_E;

    *extKeyUsage = (word16)(input[idx]);
    if (length == 2)
        *extKeyUsage |= (word16)(input[idx+1] << 8);

    return 0;
#else
    ASNGetData dataASN[keyUsageASN_Length];
    word32 idx = 0;
    byte keyUsage[2];
    word32 keyUsageSz = sizeof(keyUsage);
    int ret;
    WOLFSSL_ENTER("DecodeKeyUsage");

    /* Clear dynamic data and set where to store extended key usage. */
    XMEMSET(dataASN, 0, sizeof(dataASN));
    XMEMSET(keyUsage, 0, sizeof(keyUsage));
    GetASN_Buffer(&dataASN[KEYUSAGEASN_IDX_STR], keyUsage, &keyUsageSz);
    /* Parse key usage. */
    ret = GetASN_Items(keyUsageASN, dataASN, keyUsageASN_Length, 0, input,
                        &idx, sz);
    if (ret == 0) {
        /* Decode the bit string number as LE */
        *extKeyUsage = (word16)(keyUsage[0]);
        if (keyUsageSz == 2)
            *extKeyUsage |= (word16)(keyUsage[1] << 8);
    }
    return ret;
#endif /* WOLFSSL_ASN_TEMPLATE */
}

/* Decode key usage extension in a certificate.
 *
 * X.509: RFC 5280, 4.2.1.3 - Key Usage.
 *
 * @param [in]      input  Buffer holding data.
 * @param [in]      sz     Size of data in buffer.
 * @param [in, out] cert   Certificate object.
 * @return  0 on success.
 * @return  ASN_BITSTR_E when the expected BIT_STRING tag is not found.
 * @return  ASN_PARSE_E when BER encoded data does not match ASN.1 items or
 *          is invalid.
 * @return  MEMORY_E on dynamic memory allocation failure.
 */
static int DecodeKeyUsageInternal(const byte* input, word32 sz,
                                  DecodedCert* cert)
{
    return DecodeKeyUsage(input, sz, &cert->extKeyUsage);
}

#ifdef WOLFSSL_ASN_TEMPLATE
/* ASN.1 template for KeyPurposeId.
 * X.509: RFC 5280, 4.2.1.12 - Extended Key Usage.
 */
static const ASNItem keyPurposeIdASN[] = {
/* OID */ { 0, ASN_OBJECT_ID, 0, 0, 0 },
};
enum {
    KEYPURPOSEIDASN_IDX_OID = 0
};

/* Number of items in ASN.1 template for KeyPurposeId. */
#define keyPurposeIdASN_Length (sizeof(keyPurposeIdASN) / sizeof(ASNItem))
#endif

/* Decode extended key usage extension.
 *
 * X.509: RFC 5280, 4.2.1.12 - Extended Key Usage.
 *
 * @param [in]  input  Buffer holding data.
 * @param [in]  sz     Size of data in buffer.
 * @param [out] extExtKeyUsageSrc   Beginning of the OIDs.
 * @param [out] extExtKeyUsageSz    Size of data in extExtKeyUsageSrc.
 * @param [out] extExtKeyUsageCount Number of usages read.
 * @param [out] extExtKeyUsage      Usages read.
 * @param [out] extExtKeyUsageSsh   SSH usages read.
 * @return  0 on success.
 * @return  ASN_BITSTR_E when the expected BIT_STRING tag is not found.
 * @return  ASN_PARSE_E when BER encoded data does not match ASN.1 items or
 *          is invalid.
 * @return  MEMORY_E on dynamic memory allocation failure.
 */
int DecodeExtKeyUsage(const byte* input, word32 sz,
        const byte **extExtKeyUsageSrc, word32 *extExtKeyUsageSz,
        word32 *extExtKeyUsageCount, byte *extExtKeyUsage,
        byte *extExtKeyUsageSsh)
{
#ifndef WOLFSSL_ASN_TEMPLATE
    word32 idx = 0, oid;
    int length, ret;

    WOLFSSL_ENTER("DecodeExtKeyUsage");

    (void) extExtKeyUsageSrc;
    (void) extExtKeyUsageSz;
    (void) extExtKeyUsageCount;
    (void) extExtKeyUsageSsh;

#if defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)
    *extExtKeyUsageSrc = NULL;
    *extExtKeyUsageSz = 0;
    *extExtKeyUsageCount = 0;
#endif
    *extExtKeyUsage = 0;
#ifdef WOLFSSL_WOLFSSH
    *extExtKeyUsageSsh = 0;
#endif

    if (GetSequence(input, &idx, &length, sz) < 0) {
        WOLFSSL_MSG("\tfail: should be a SEQUENCE");
        return ASN_PARSE_E;
    }

#if defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)
    *extExtKeyUsageSrc = input + idx;
    *extExtKeyUsageSz = length;
#endif

    while (idx < (word32)sz) {
        ret = GetObjectId(input, &idx, &oid, oidCertKeyUseType, sz);
        if (ret == WC_NO_ERR_TRACE(ASN_UNKNOWN_OID_E))
            continue;
        else if (ret < 0)
            return ret;

        switch (oid) {
            case EKU_ANY_OID:
                *extExtKeyUsage |= EXTKEYUSE_ANY;
                break;
            case EKU_SERVER_AUTH_OID:
                *extExtKeyUsage |= EXTKEYUSE_SERVER_AUTH;
                break;
            case EKU_CLIENT_AUTH_OID:
                *extExtKeyUsage |= EXTKEYUSE_CLIENT_AUTH;
                break;
            case EKU_CODESIGNING_OID:
                *extExtKeyUsage |= EXTKEYUSE_CODESIGN;
                break;
            case EKU_EMAILPROTECT_OID:
                *extExtKeyUsage |= EXTKEYUSE_EMAILPROT;
                break;
            case EKU_TIMESTAMP_OID:
                *extExtKeyUsage |= EXTKEYUSE_TIMESTAMP;
                break;
            case EKU_OCSP_SIGN_OID:
                *extExtKeyUsage |= EXTKEYUSE_OCSP_SIGN;
                break;
            #ifdef WOLFSSL_WOLFSSH
            case EKU_SSH_CLIENT_AUTH_OID:
                *extExtKeyUsageSsh |= EXTKEYUSE_SSH_CLIENT_AUTH;
                break;
            case EKU_SSH_MSCL_OID:
                *extExtKeyUsageSsh |= EXTKEYUSE_SSH_MSCL;
                break;
            case EKU_SSH_KP_CLIENT_AUTH_OID:
                *extExtKeyUsageSsh |= EXTKEYUSE_SSH_KP_CLIENT_AUTH;
                break;
            #endif /* WOLFSSL_WOLFSSH */
            default:
                break;
        }

    #if defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)
        (*extExtKeyUsageCount)++;
    #endif
    }

    return 0;
#else
    word32 idx = 0;
    int length;
    int ret = 0;

    WOLFSSL_ENTER("DecodeExtKeyUsage");

    (void) extExtKeyUsageSrc;
    (void) extExtKeyUsageSz;
    (void) extExtKeyUsageCount;
    (void) extExtKeyUsageSsh;

#if defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)
    *extExtKeyUsageSrc = NULL;
    *extExtKeyUsageSz = 0;
    *extExtKeyUsageCount = 0;
#endif
    *extExtKeyUsage = 0;
#ifdef WOLFSSL_WOLFSSH
    *extExtKeyUsageSsh = 0;
#endif

    /* Strip SEQUENCE OF and expect to account for all the data. */
    if (GetASN_Sequence(input, &idx, &length, sz, 1) < 0) {
        WOLFSSL_MSG("\tfail: should be a SEQUENCE");
        ret = ASN_PARSE_E;
    }

    if (ret == 0) {
    #if defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)
        /* Keep reference for WOLFSSL_X509. */
        *extExtKeyUsageSrc = input + idx;
        *extExtKeyUsageSz = (word32)length;
    #endif
    }

    /* Check all OIDs. */
    while ((ret == 0) && (idx < (word32)sz)) {
        ASNGetData dataASN[keyPurposeIdASN_Length];

        /* Clear dynamic data items and set OID type expected. */
        XMEMSET(dataASN, 0, sizeof(dataASN));
        GetASN_OID(&dataASN[KEYPURPOSEIDASN_IDX_OID], oidIgnoreType);
        /* Decode KeyPurposeId. */
        ret = GetASN_Items(keyPurposeIdASN, dataASN, keyPurposeIdASN_Length, 0,
                           input, &idx, sz);
        /* Skip unknown OIDs. */
        if (ret == WC_NO_ERR_TRACE(ASN_UNKNOWN_OID_E)) {
            ret = 0;
        }
        else if (ret == 0) {
            /* Store the bit for the OID. */
            switch (dataASN[KEYPURPOSEIDASN_IDX_OID].data.oid.sum) {
                case EKU_ANY_OID:
                    *extExtKeyUsage |= EXTKEYUSE_ANY;
                    break;
                case EKU_SERVER_AUTH_OID:
                    *extExtKeyUsage |= EXTKEYUSE_SERVER_AUTH;
                    break;
                case EKU_CLIENT_AUTH_OID:
                    *extExtKeyUsage |= EXTKEYUSE_CLIENT_AUTH;
                    break;
                case EKU_CODESIGNING_OID:
                    *extExtKeyUsage |= EXTKEYUSE_CODESIGN;
                    break;
                case EKU_EMAILPROTECT_OID:
                    *extExtKeyUsage |= EXTKEYUSE_EMAILPROT;
                    break;
                case EKU_TIMESTAMP_OID:
                    *extExtKeyUsage |= EXTKEYUSE_TIMESTAMP;
                    break;
                case EKU_OCSP_SIGN_OID:
                    *extExtKeyUsage |= EXTKEYUSE_OCSP_SIGN;
                    break;
            }

        #if defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)
            /* Keep count for WOLFSSL_X509. */
            (*extExtKeyUsageCount)++;
        #endif
        }
    }

    return ret;
#endif /* WOLFSSL_ASN_TEMPLATE */
}

/* Decode extended key usage extension in a certificate.
 *
 * X.509: RFC 5280, 4.2.1.12 - Extended Key Usage.
 *
 * @param [in]      input  Buffer holding data.
 * @param [in]      sz     Size of data in buffer.
 * @param [in, out] cert   Certificate object.
 * @return  0 on success.
 * @return  ASN_BITSTR_E when the expected BIT_STRING tag is not found.
 * @return  ASN_PARSE_E when BER encoded data does not match ASN.1 items or
 *          is invalid.
 * @return  MEMORY_E on dynamic memory allocation failure.
 */
static int DecodeExtKeyUsageInternal(const byte* input, word32 sz,
                                     DecodedCert* cert)
{
    int ret = 0;


    ret = DecodeExtKeyUsage(input, sz,
#if defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)
            &cert->extExtKeyUsageSrc, &cert->extExtKeyUsageSz,
            &cert->extExtKeyUsageCount,
#else
            NULL, NULL, NULL,
#endif
            &cert->extExtKeyUsage,
#ifdef WOLFSSL_WOLFSSH
            &cert->extExtKeyUsageSsh
#else
            NULL
#endif
            );

    if (ret != 0)
        return ret;

    return 0;
}

#ifndef IGNORE_NETSCAPE_CERT_TYPE

static int DecodeNsCertType(const byte* input, int sz, DecodedCert* cert)
{
    word32 idx = 0;
    int len = 0;

    WOLFSSL_ENTER("DecodeNsCertType");

    if (CheckBitString(input, &idx, &len, (word32)sz, 0, NULL) < 0)
        return ASN_PARSE_E;

    /* Don't need to worry about unused bits as CheckBitString makes sure
     * they're zero. */
    if (idx < (word32)sz)
        cert->nsCertType = input[idx];
    else
        return ASN_PARSE_E;

    return 0;
}
#endif


#ifndef IGNORE_NAME_CONSTRAINTS
#ifdef WOLFSSL_ASN_TEMPLATE
/* ASN.1 template for GeneralSubtree.
 * X.509: RFC 5280, 4.2.1.10 - Name Constraints.
 */
static const ASNItem subTreeASN[] = {
/* SEQ  */ { 0, ASN_SEQUENCE, 1, 1, 0 },
                              /* base     GeneralName */
/* BASE */     { 1, ASN_CONTEXT_SPECIFIC | 0, 0, 0, 0 },
                              /* minimum  BaseDistance DEFAULT 0*/
/* MIN  */     { 1, ASN_CONTEXT_SPECIFIC | ASN_SUBTREE_MIN, 0, 0, 1 },
                              /* maximum  BaseDistance OPTIONAL  */
/* MAX  */     { 1, ASN_CONTEXT_SPECIFIC | ASN_SUBTREE_MAX, 0, 0, 1 },
};
enum {
    SUBTREEASN_IDX_SEQ = 0,
    SUBTREEASN_IDX_BASE,
    SUBTREEASN_IDX_MIN,
    SUBTREEASN_IDX_MAX
};

/* Number of items in ASN.1 template for GeneralSubtree. */
#define subTreeASN_Length (sizeof(subTreeASN) / sizeof(ASNItem))
#endif

#ifdef WOLFSSL_ASN_TEMPLATE
/* Decode the Subtree's GeneralName.
 *
 * @param [in]      input  Buffer holding data.
 * @param [in]      sz     Size of data in buffer.
 * @param [in]      tag    BER tag on GeneralName.
 * @param [in, out] head   Linked list of subtree names.
 * @param [in]      heap   Dynamic memory hint.
 * @return  0 on success.
 * @return  MEMORY_E when dynamic memory allocation fails.
 * @return  ASN_PARSE_E when SEQUENCE is not found as expected.
 */
static int DecodeSubtreeGeneralName(const byte* input, word32 sz, byte tag,
                                    Base_entry** head, void* heap)
{
    Base_entry* entry = NULL;
    word32 nameIdx = 0;
    word32 len = sz;
    int strLen;
    int ret = 0;

    (void)heap;

    /* if constructed has leading sequence */
    if ((tag & ASN_CONSTRUCTED) == ASN_CONSTRUCTED) {
        ret = GetASN_Sequence(input, &nameIdx, &strLen, sz, 0);
        if (ret < 0) {
            ret = ASN_PARSE_E;
        }
        else {
            len = (word32)strLen;
            ret = 0;
        }
    }
    if (ret == 0) {
        /* TODO: consider one malloc. */
        /* Allocate Base Entry object. */
        entry = (Base_entry*)XMALLOC(sizeof(Base_entry), heap,
                                     DYNAMIC_TYPE_ALTNAME);
        if (entry == NULL) {
            ret = MEMORY_E;
        }
    }
    if (ret == 0) {
        /* Allocate name. */
        entry->name = (char*)XMALLOC(len + 1, heap, DYNAMIC_TYPE_ALTNAME);
        if (entry->name == NULL) {
            XFREE(entry, heap, DYNAMIC_TYPE_ALTNAME);
            ret = MEMORY_E;
        }
    }
    if (ret == 0) {
        /* Store name, size and tag in object. */
        XMEMCPY(entry->name, &input[nameIdx], len);
        entry->name[len] = '\0';
        entry->nameSz = (int)len;
        entry->type = tag & ASN_TYPE_MASK;

        /* Put entry at front of linked list. */
        entry->next = *head;
        *head = entry;
    }

    return ret;
}
#endif

/* Decode a subtree of a name constraints in a certificate.
 *
 * X.509: RFC 5280, 4.2.1.10 - Name Constraints.
 *
 * @param [in]      input  Buffer holding data.
 * @param [in]      sz     Size of data in buffer.
 * @param [in, out] head   Linked list of subtree names.
 * @param [in]      limit  If > 0, limit on number of tree
 *                         entries to  process, exceeding
 *                         is an error.
 * @param [in]      heap   Dynamic memory hint.
 * @return  0 on success.
 * @return  MEMORY_E when dynamic memory allocation fails.
 * @return  ASN_PARSE_E when SEQUENCE is not found as expected.
 */
static int DecodeSubtree(const byte* input, word32 sz, Base_entry** head,
                         word32 limit, void* heap)
{
#ifndef WOLFSSL_ASN_TEMPLATE
    word32 idx = 0;
    int ret = 0;
    word32 cnt = 0;

    (void)heap;

    while (idx < (word32)sz) {
        int seqLength, strLength;
        word32 nameIdx;
        byte b, bType;

        if (limit > 0) {
            cnt++;
            if (cnt > limit) {
                WOLFSSL_MSG("too many name constraints");
                return ASN_NAME_INVALID_E;
            }
        }

        if (GetSequence(input, &idx, &seqLength, sz) < 0) {
            WOLFSSL_MSG("\tfail: should be a SEQUENCE");
            return ASN_PARSE_E;
        }

        if (idx >= (word32)sz) {
            WOLFSSL_MSG("\tfail: expecting tag");
            return ASN_PARSE_E;
        }

        nameIdx = idx;
        b = input[nameIdx++];

        if (GetLength(input, &nameIdx, &strLength, sz) <= 0) {
            WOLFSSL_MSG("\tinvalid length");
            return ASN_PARSE_E;
        }

        /* Get type, LSB 4-bits */
        bType = (byte)(b & ASN_TYPE_MASK);

        if (bType == ASN_DNS_TYPE || bType == ASN_RFC822_TYPE ||
                                                        bType == ASN_DIR_TYPE) {
            Base_entry* entry;

            /* if constructed has leading sequence */
            if (b & ASN_CONSTRUCTED) {
                if (GetSequence(input, &nameIdx, &strLength, sz) < 0) {
                    WOLFSSL_MSG("\tfail: constructed be a SEQUENCE");
                    return ASN_PARSE_E;
                }
            }

            entry = (Base_entry*)XMALLOC(sizeof(Base_entry), heap,
                                                          DYNAMIC_TYPE_ALTNAME);
            if (entry == NULL) {
                WOLFSSL_MSG("allocate error");
                return MEMORY_E;
            }

            entry->name = (char*)XMALLOC((size_t)strLength+1, heap,
                DYNAMIC_TYPE_ALTNAME);
            if (entry->name == NULL) {
                WOLFSSL_MSG("allocate error");
                XFREE(entry, heap, DYNAMIC_TYPE_ALTNAME);
                return MEMORY_E;
            }

            XMEMCPY(entry->name, &input[nameIdx], (size_t)strLength);
            entry->name[strLength] = '\0';
            entry->nameSz = strLength;
            entry->type = bType;

            entry->next = *head;
            *head = entry;
        }

        idx += (word32)seqLength;
    }

    return ret;
#else
    DECL_ASNGETDATA(dataASN, subTreeASN_Length);
    word32 idx = 0;
    int ret = 0;
    word32 cnt = 0;

    (void)heap;

    ALLOC_ASNGETDATA(dataASN, subTreeASN_Length, ret, heap);

    /* Process all subtrees. */
    while ((ret == 0) && (idx < (word32)sz)) {
        byte minVal = 0;
        byte maxVal = 0;
        if (limit > 0) {
            cnt++;
            if (cnt > limit) {
                WOLFSSL_MSG("too many name constraints");
                ret = ASN_NAME_INVALID_E;
                break;
            }
        }

        /* Clear dynamic data and set choice for GeneralName and location to
         * store minimum and maximum.
         */
        XMEMSET(dataASN, 0, sizeof(*dataASN) * subTreeASN_Length);
        GetASN_Choice(&dataASN[SUBTREEASN_IDX_BASE], generalNameChoice);
        GetASN_Int8Bit(&dataASN[SUBTREEASN_IDX_MIN], &minVal);
        GetASN_Int8Bit(&dataASN[SUBTREEASN_IDX_MAX], &maxVal);
        /* Parse GeneralSubtree. */
        ret = GetASN_Items(subTreeASN, dataASN, subTreeASN_Length, 0, input,
                           &idx, sz);
        if (ret == 0) {
            byte t = dataASN[SUBTREEASN_IDX_BASE].tag;

            /* Check GeneralName tag is one of the types we can handle. */
            if (t == (ASN_CONTEXT_SPECIFIC | ASN_DNS_TYPE) ||
                t == (ASN_CONTEXT_SPECIFIC | ASN_RFC822_TYPE) ||
                t == (ASN_CONTEXT_SPECIFIC | ASN_CONSTRUCTED | ASN_DIR_TYPE)) {
                /* Parse the general name and store a new entry. */
                ret = DecodeSubtreeGeneralName(input +
                    GetASNItem_DataIdx(dataASN[SUBTREEASN_IDX_BASE], input),
                    dataASN[SUBTREEASN_IDX_BASE].length, t, head, heap);
            }
            /* Skip entry. */
        }
    }

    FREE_ASNGETDATA(dataASN, heap);
    return ret;
#endif
}

#ifdef WOLFSSL_ASN_TEMPLATE
/* ASN.1 template for NameConstraints.
 * X.509: RFC 5280, 4.2.1.10 - Name Constraints.
 */
static const ASNItem nameConstraintsASN[] = {
/* SEQ     */ { 0, ASN_SEQUENCE, 1, 1, 0 },
                                         /* permittedSubtrees */
/* PERMIT  */     { 1, ASN_CONTEXT_SPECIFIC | 0, 1, 0, 1 },
                                         /* excludededSubtrees */
/* EXCLUDE */     { 1, ASN_CONTEXT_SPECIFIC | 1, 1, 0, 1 },
};
enum {
    NAMECONSTRAINTSASN_IDX_SEQ = 0,
    NAMECONSTRAINTSASN_IDX_PERMIT,
    NAMECONSTRAINTSASN_IDX_EXCLUDE
};

/* Number of items in ASN.1 template for NameConstraints. */
#define nameConstraintsASN_Length (sizeof(nameConstraintsASN) / sizeof(ASNItem))
#endif

/* Decode name constraints extension in a certificate.
 *
 * X.509: RFC 5280, 4.2.1.10 - Name Constraints.
 *
 * @param [in]      input  Buffer holding data.
 * @param [in]      sz     Size of data in buffer.
 * @param [in, out] cert   Certificate object.
 * @return  0 on success.
 * @return  ASN_PARSE_E when BER encoded data does not match ASN.1 items or
 *          is invalid.
 * @return  MEMORY_E on dynamic memory allocation failure.
 */
static int DecodeNameConstraints(const byte* input, word32 sz,
    DecodedCert* cert)
{
#ifndef WOLFSSL_ASN_TEMPLATE
    word32 idx = 0;
    int length = 0;

    WOLFSSL_ENTER("DecodeNameConstraints");

    if (GetSequence(input, &idx, &length, sz) < 0) {
        WOLFSSL_MSG("\tfail: should be a SEQUENCE");
        return ASN_PARSE_E;
    }

    while (idx < (word32)sz) {
        byte b = input[idx++];
        Base_entry** subtree = NULL;

        if (GetLength(input, &idx, &length, sz) <= 0) {
            WOLFSSL_MSG("\tinvalid length");
            return ASN_PARSE_E;
        }

        if (b == (ASN_CONTEXT_SPECIFIC | ASN_CONSTRUCTED | 0))
            subtree = &cert->permittedNames;
        else if (b == (ASN_CONTEXT_SPECIFIC | ASN_CONSTRUCTED | 1))
            subtree = &cert->excludedNames;
        else {
            WOLFSSL_MSG("\tinvalid subtree");
            return ASN_PARSE_E;
        }

        if (DecodeSubtree(input + idx, (word32)length, subtree,
                WOLFSSL_MAX_NAME_CONSTRAINTS, cert->heap) < 0) {
            WOLFSSL_MSG("\terror parsing subtree");
            return ASN_PARSE_E;
        }

        idx += (word32)length;
    }

    return 0;
#else
    DECL_ASNGETDATA(dataASN, nameConstraintsASN_Length);
    word32 idx = 0;
    int    ret = 0;

    CALLOC_ASNGETDATA(dataASN, nameConstraintsASN_Length, ret, cert->heap);

    if (ret == 0) {
        /* Parse NameConstraints. */
        ret = GetASN_Items(nameConstraintsASN, dataASN,
                           nameConstraintsASN_Length, 1, input, &idx, sz);
    }
    if (ret == 0) {
        /* If there was a permittedSubtrees then parse it. */
        if (dataASN[NAMECONSTRAINTSASN_IDX_PERMIT].data.ref.data != NULL) {
            ret = DecodeSubtree(
                    dataASN[NAMECONSTRAINTSASN_IDX_PERMIT].data.ref.data,
                    dataASN[NAMECONSTRAINTSASN_IDX_PERMIT].data.ref.length,
                    &cert->permittedNames, WOLFSSL_MAX_NAME_CONSTRAINTS,
                    cert->heap);
        }
    }
    if (ret == 0) {
        /* If there was a excludedSubtrees then parse it. */
        if (dataASN[NAMECONSTRAINTSASN_IDX_EXCLUDE].data.ref.data != NULL) {
            ret = DecodeSubtree(
                    dataASN[NAMECONSTRAINTSASN_IDX_EXCLUDE].data.ref.data,
                    dataASN[NAMECONSTRAINTSASN_IDX_EXCLUDE].data.ref.length,
                    &cert->excludedNames, WOLFSSL_MAX_NAME_CONSTRAINTS,
                    cert->heap);
        }
    }

    FREE_ASNGETDATA(dataASN, cert->heap);

    return ret;
#endif /* WOLFSSL_ASN_TEMPLATE */
}
#endif /* IGNORE_NAME_CONSTRAINTS */

#if defined(WOLFSSL_CERT_EXT) || \
    defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)

/* Decode ITU-T X.690 OID format to a string representation
 * return string length */
int DecodePolicyOID(char *out, word32 outSz, const byte *in, word32 inSz)
{
    word32 val, inIdx = 0, outIdx = 0;
    int w = 0;

    if (out == NULL || in == NULL || outSz < 4 || inSz < 2)
        return BAD_FUNC_ARG;

    /* The first byte expands into b/40 dot b%40. */
    val = in[inIdx++];

    w = XSNPRINTF(out, outSz, "%u.%u", val / 40, val % 40);
    if (w < 0) {
        w = BUFFER_E;
        goto exit;
    }
    outIdx += (word32)w;
    val = 0;

    while (inIdx < inSz && outIdx < outSz) {
        /* extract the next OID digit from in to val */
        /* first bit is used to set if value is coded on 1 or multiple bytes */
        if (in[inIdx] & 0x80) {
            val += in[inIdx] & 0x7F;
            val *= 128;
        }
        else {
            /* write val as text into out */
            val += in[inIdx];
            w = XSNPRINTF(out + outIdx, outSz - outIdx, ".%u", val);
            if (w < 0 || (word32)w > outSz - outIdx) {
                w = BUFFER_E;
                goto exit;
            }
            outIdx += (word32)w;
            val = 0;
        }
        inIdx++;
    }
    if (outIdx == outSz)
        outIdx--;
    out[outIdx] = 0;

    w = (int)outIdx;

exit:
    return w;
}
#endif /* WOLFSSL_CERT_EXT || OPENSSL_EXTRA || OPENSSL_EXTRA_X509_SMALL */

#if defined(WOLFSSL_SEP) || defined(WOLFSSL_CERT_EXT)
#ifdef WOLFSSL_ASN_TEMPLATE
    /* ASN.1 template for PolicyInformation.
     * X.509: RFC 5280, 4.2.1.4 - Certificate Policies.
     */
    static const ASNItem policyInfoASN[] = {
    /* SEQ   */ { 0, ASN_SEQUENCE, 1, 1, 0 },
                                      /* policyIdentifier */
    /* ID    */     { 1, ASN_OBJECT_ID, 0, 0, 0 },
                                      /* policyQualifiers */
    /* QUALI */     { 1, ASN_SEQUENCE, 1, 0, 1 },
    };
    enum {
        POLICYINFOASN_IDX_SEQ = 0,
        POLICYINFOASN_IDX_ID,
        POLICYINFOASN_IDX_QUALI
    };

    /* Number of items in ASN.1 template for PolicyInformation. */
    #define policyInfoASN_Length (sizeof(policyInfoASN) / sizeof(ASNItem))
#endif

/* Reference: https://tools.ietf.org/html/rfc5280#section-4.2.1.4 */
static int DecodeCertPolicy(const byte* input, word32 sz, DecodedCert* cert)
{
#ifndef WOLFSSL_ASN_TEMPLATE
    word32 idx = 0;
    word32 oldIdx;
    int policy_length = 0;
    int ret;
    int total_length = 0;
#if defined(WOLFSSL_CERT_EXT) && !defined(WOLFSSL_DUP_CERTPOL)
    int i;
#endif

    WOLFSSL_ENTER("DecodeCertPolicy");

    /* Check if cert is null before dereferencing below */
    if (cert == NULL)
        return BAD_FUNC_ARG;

#if defined(WOLFSSL_CERT_EXT)
        cert->extCertPoliciesNb = 0;
#endif

    if (GetSequence(input, &idx, &total_length, sz) < 0) {
        WOLFSSL_MSG("\tGet CertPolicy total seq failed");
        return ASN_PARSE_E;
    }

    /* Validate total length */
    if (total_length > (int)(sz - idx)) {
        WOLFSSL_MSG("\tCertPolicy length mismatch");
        return ASN_PARSE_E;
    }

    /* Unwrap certificatePolicies */
    do {
        int length = 0;

        if (GetSequence(input, &idx, &policy_length, sz) < 0) {
            WOLFSSL_MSG("\tGet CertPolicy seq failed");
            return ASN_PARSE_E;
        }

        oldIdx = idx;
        ret = GetASNObjectId(input, &idx, &length, sz);
        if (ret != 0)
            return ret;
        policy_length -= (int)(idx - oldIdx);

        if (length > 0) {
            /* Verify length won't overrun buffer */
            if (length > (int)(sz - idx)) {
                WOLFSSL_MSG("\tCertPolicy length exceeds input buffer");
                return ASN_PARSE_E;
            }

    #ifdef WOLFSSL_SEP
            if (cert->deviceType == NULL) {
                cert->deviceType = (byte*)XMALLOC((size_t)length, cert->heap,
                                                        DYNAMIC_TYPE_X509_EXT);
                if (cert->deviceType == NULL) {
                    WOLFSSL_MSG("\tCouldn't alloc memory for deviceType");
                    return MEMORY_E;
                }
                cert->deviceTypeSz = length;
                XMEMCPY(cert->deviceType, input + idx, (size_t)length);
            }
    #endif

    #ifdef WOLFSSL_CERT_EXT
            /* decode cert policy */
            if (DecodePolicyOID(cert->extCertPolicies[
                                cert->extCertPoliciesNb], MAX_CERTPOL_SZ,
                                input + idx, length) <= 0) {
                WOLFSSL_MSG("\tCouldn't decode CertPolicy");
                WOLFSSL_ERROR_VERBOSE(ASN_PARSE_E);
                return ASN_PARSE_E;
            }
        #ifndef WOLFSSL_DUP_CERTPOL
            /* From RFC 5280 section 4.2.1.4 "A certificate policy OID MUST
             * NOT appear more than once in a certificate policies
             * extension". This is a sanity check for duplicates.
             * extCertPolicies should only have OID values, additional
             * qualifiers need to be stored in a separate array. */
            for (i = 0; i < cert->extCertPoliciesNb; i++) {
                if (XMEMCMP(cert->extCertPolicies[i],
                            cert->extCertPolicies[cert->extCertPoliciesNb],
                            MAX_CERTPOL_SZ) == 0) {
                    WOLFSSL_MSG("Duplicate policy OIDs not allowed");
                    WOLFSSL_MSG("Use WOLFSSL_DUP_CERTPOL if wanted");
                    WOLFSSL_ERROR_VERBOSE(CERTPOLICIES_E);
                    return CERTPOLICIES_E;
                }
            }
        #endif /* !WOLFSSL_DUP_CERTPOL */
            cert->extCertPoliciesNb++;
    #endif
        }
        idx += (word32)policy_length;
    } while((int)idx < total_length
    #ifdef WOLFSSL_CERT_EXT
        && cert->extCertPoliciesNb < MAX_CERTPOL_NB
    #endif
    );

    WOLFSSL_LEAVE("DecodeCertPolicy", 0);
    return 0;
#else /* WOLFSSL_ASN_TEMPLATE */
    word32 idx = 0;
    int ret = 0;
    int total_length = 0;
#if defined(WOLFSSL_CERT_EXT) && !defined(WOLFSSL_DUP_CERTPOL)
    int i;
#endif

    WOLFSSL_ENTER("DecodeCertPolicy");

    /* Check if cert is null before dereferencing below */
    if (cert == NULL) {
        ret = BAD_FUNC_ARG;
    }

    if (ret == 0) {
    #if defined(WOLFSSL_CERT_EXT)
        cert->extCertPoliciesNb = 0;
    #endif

        /* Strip SEQUENCE OF and check using all data. */
        if (GetASN_Sequence(input, &idx, &total_length, (word32)sz, 1) < 0)
        {
            ret = ASN_PARSE_E;
        }
    }

    /* Unwrap certificatePolicies */
    while ((ret == 0) && ((int)idx < total_length)
    #if defined(WOLFSSL_CERT_EXT)
        && (cert->extCertPoliciesNb < MAX_CERTPOL_NB)
    #endif
            ) {
        ASNGetData dataASN[policyInfoASN_Length];
        byte* data = NULL;
        word32 length = 0;

        /* Clear dynamic data and check OID is a cert policy type. */
        XMEMSET(dataASN, 0, sizeof(dataASN));
        GetASN_OID(&dataASN[POLICYINFOASN_IDX_ID], oidCertPolicyType);
        ret = GetASN_Items(policyInfoASN, dataASN, policyInfoASN_Length, 1,
                            input, &idx, (word32)sz);
        if (ret == 0) {
            /* Get the OID. */
            GetASN_OIDData(&dataASN[POLICYINFOASN_IDX_ID], &data, &length);
            if (length == 0) {
                ret = ASN_PARSE_E;
            }
        }
    #ifdef WOLFSSL_SEP
        /* Store OID in device type. */
        if (ret == 0 && cert->deviceType == NULL) {
            cert->deviceType = (byte*)XMALLOC(length, cert->heap,
                                                DYNAMIC_TYPE_X509_EXT);
            if (cert->deviceType != NULL) {
                /* Store device type data and length. */
                cert->deviceTypeSz = (int)length;
                XMEMCPY(cert->deviceType, data, length);
            }
            else {
                WOLFSSL_MSG("\tCouldn't alloc memory for deviceType");
                ret = MEMORY_E;
            }
        }
    #endif /* WOLFSSL_SEP */

    #ifdef WOLFSSL_CERT_EXT
        if (ret == 0) {
            /* Decode cert policy. */
            if (DecodePolicyOID(
                    cert->extCertPolicies[cert->extCertPoliciesNb],
                    MAX_CERTPOL_SZ, data, length) <= 0) {
                WOLFSSL_MSG("\tCouldn't decode CertPolicy");
                WOLFSSL_ERROR_VERBOSE(ASN_PARSE_E);
                ret = ASN_PARSE_E;
            }
        }
        #ifndef WOLFSSL_DUP_CERTPOL
        /* From RFC 5280 section 4.2.1.4 "A certificate policy OID MUST
         * NOT appear more than once in a certificate policies
         * extension". This is a sanity check for duplicates.
         * extCertPolicies should only have OID values, additional
         * qualifiers need to be stored in a separate array. */
        for (i = 0; (ret == 0) && (i < cert->extCertPoliciesNb); i++) {
            if (XMEMCMP(cert->extCertPolicies[i],
                        cert->extCertPolicies[cert->extCertPoliciesNb],
                        MAX_CERTPOL_SZ) == 0) {
                WOLFSSL_MSG("Duplicate policy OIDs not allowed");
                WOLFSSL_MSG("Use WOLFSSL_DUP_CERTPOL if wanted");
                WOLFSSL_ERROR_VERBOSE(CERTPOLICIES_E);
                ret = CERTPOLICIES_E;
            }
        }
        #endif /* !WOLFSSL_DUP_CERTPOL */
        if (ret == 0) {
            /* Keep count of policies seen. */
            cert->extCertPoliciesNb++;
        }
    #endif /* WOLFSSL_CERT_EXT */
    }

    WOLFSSL_LEAVE("DecodeCertPolicy", 0);
    return ret;
#endif /* WOLFSSL_ASN_TEMPLATE */
}
#endif /* WOLFSSL_SEP || WOLFSSL_CERT_EXT  */

#ifdef WOLFSSL_SUBJ_DIR_ATTR
#ifdef WOLFSSL_ASN_TEMPLATE
/* ASN.1 template for subject dir attribute.
 * X.509: RFC 5280, 4.2.1.8 - Subject Directory Attributes.
 */
static const ASNItem subjDirAttrASN[] = {
/* SEQ  */     { 1, ASN_SEQUENCE, 1, 1, 0 },
/* OID  */          { 2, ASN_OBJECT_ID, 0, 0, 0 },
/* PLEN */          { 2, ASN_SET, 1, 0, 0 },
};
enum {
    SUBJDIRATTRASN_IDX_SEQ = 0,
    SUBJDIRATTRASN_IDX_OID,
    SUBJDIRATTRASN_IDX_SET
};

/* Number of items in ASN.1 template for BasicConstraints. */
#define subjDirAttrASN_Length (sizeof(subjDirAttrASN) / sizeof(ASNItem))
#endif
/* Decode subject directory attributes extension in a certificate.
 *
 * X.509: RFC 5280, 4.2.1.8 - Subject Directory Attributes.
 *
 * @param [in]      input  Buffer holding data.
 * @param [in]      sz     Size of data in buffer.
 * @param [in, out] cert   Certificate object.
 * @return  0 on success.
 * @return  ASN_PARSE_E when BER encoded data does not match ASN.1 items or
 *          is invalid.
 */
static int DecodeSubjDirAttr(const byte* input, word32 sz, DecodedCert* cert)
{
#ifndef WOLFSSL_ASN_TEMPLATE
    word32 idx = 0;
    int length = 0;
    int ret = 0;

    WOLFSSL_ENTER("DecodeSubjDirAttr");

#ifdef OPENSSL_ALL
    cert->extSubjDirAttrSrc = input;
    cert->extSubjDirAttrSz = sz;
#endif /* OPENSSL_ALL */

    /* Unwrap the list of Attributes */
    if (GetSequence(input, &idx, &length, sz) < 0)
        return ASN_PARSE_E;

    if (length == 0) {
        /* RFC 5280 4.2.1.8.  Subject Directory Attributes
           If the subjectDirectoryAttributes extension is present, the
           sequence MUST contain at least one entry. */
        WOLFSSL_ERROR_VERBOSE(ASN_PARSE_E);
        return ASN_PARSE_E;
    }

    /* length is the length of the list contents */
    while (idx < (word32)sz) {
        word32 oid;

        if (GetSequence(input, &idx, &length, sz) < 0)
            return ASN_PARSE_E;

        if (GetObjectId(input, &idx, &oid, oidSubjDirAttrType, sz) < 0)
            return ASN_PARSE_E;

        if (GetSet(input, &idx, &length, sz) < 0)
            return ASN_PARSE_E;

        /* There may be more than one countryOfCitizenship, but save the
         * first one for now. */
        if (oid == SDA_COC_OID) {
            byte tag;

            if (GetHeader(input, &tag, &idx, &length, sz, 1) < 0)
                return ASN_PARSE_E;

            if (length != COUNTRY_CODE_LEN)
                return ASN_PARSE_E;

            if (tag == ASN_PRINTABLE_STRING) {
                XMEMCPY(cert->countryOfCitizenship,
                        input + idx, COUNTRY_CODE_LEN);
                cert->countryOfCitizenship[COUNTRY_CODE_LEN] = 0;
            }
        }
        idx += length;
    }

    return ret;
#else
    DECL_ASNGETDATA(dataASN, subjDirAttrASN_Length);
    int ret = 0;
    word32 idx = 0;
    int length;

    WOLFSSL_ENTER("DecodeSubjDirAttr");

#ifdef OPENSSL_ALL
    cert->extSubjDirAttrSrc = input;
    cert->extSubjDirAttrSz = sz;
#endif /* OPENSSL_ALL */

    CALLOC_ASNGETDATA(dataASN, subjDirAttrASN_Length, ret, cert->heap);

    /* Strip outer SEQUENCE. */
    if ((ret == 0) && (GetSequence(input, &idx, &length, sz) < 0)) {
        ret = ASN_PARSE_E;
    }
    /* Handle each inner SEQUENCE. */
    while ((ret == 0) && (idx < (word32)sz)) {
        ret = GetASN_Items(subjDirAttrASN, dataASN, subjDirAttrASN_Length, 1,
            input, &idx, sz);

        /* There may be more than one countryOfCitizenship, but save the
         * first one for now. */
        if ((ret == 0) &&
                (dataASN[SUBJDIRATTRASN_IDX_OID].data.oid.sum == SDA_COC_OID)) {
            int cuLen;
            word32 setIdx = 0;
            byte* setData;
            word32 setLen;

            GetASN_GetRef(&dataASN[SUBJDIRATTRASN_IDX_SET], &setData, &setLen);
            if (GetASNHeader(setData, ASN_PRINTABLE_STRING, &setIdx, &cuLen,
                    setLen) < 0) {
                ret = ASN_PARSE_E;
            }
            if ((ret == 0) && (cuLen != COUNTRY_CODE_LEN)) {
                ret = ASN_PARSE_E;
            }
            if (ret == 0) {
                XMEMCPY(cert->countryOfCitizenship, setData + setIdx,
                    (size_t)cuLen);
                cert->countryOfCitizenship[COUNTRY_CODE_LEN] = 0;
            }
        }
    }
    FREE_ASNGETDATA(dataASN, cert->heap);
    return ret;
#endif /* WOLFSSL_ASN_TEMPLATE */
}
#endif /* WOLFSSL_SUBJ_DIR_ATTR */

#ifdef WOLFSSL_SUBJ_INFO_ACC
/* Decode subject information access extension in a certificate.
 *
 * X.509: RFC 5280, 4.2.2.2 - Subject Information Access.
 *
 * @param [in]      input  Buffer holding data.
 * @param [in]      sz     Size of data in buffer.
 * @param [in, out] cert   Certificate object.
 * @return  0 on success.
 * @return  ASN_BITSTR_E when the expected BIT_STRING tag is not found.
 * @return  ASN_PARSE_E when BER encoded data does not match ASN.1 items or
 *          is invalid.
 * @return  MEMORY_E on dynamic memory allocation failure.
 */
static int DecodeSubjInfoAcc(const byte* input, word32 sz, DecodedCert* cert)
{
    word32 idx = 0;
    int length = 0;
    int ret = 0;

    WOLFSSL_ENTER("DecodeSubjInfoAcc");

#ifdef OPENSSL_ALL
    cert->extSubjAltNameSrc = input;
    cert->extSubjAltNameSz = sz;
#endif /* OPENSSL_ALL */

    /* Unwrap SubjectInfoAccessSyntax, the list of AccessDescriptions */
    if (GetSequence(input, &idx, &length, sz) < 0)
        return ASN_PARSE_E;

    if (length == 0) {
        /* RFC 5280 4.2.2.2.  Subject Information Access
           If the subjectInformationAccess extension is present, the
           sequence MUST contain at least one entry. */
        WOLFSSL_ERROR_VERBOSE(ASN_PARSE_E);
        return ASN_PARSE_E;
    }

    /* Per fpkx-x509-cert-profile-common... section 5.3.
     * [The] subjectInfoAccess extension must contain at least one
     * instance of the id-ad-caRepository access method containing a
     * publicly accessible HTTP URI which returns as certs-only
     * CMS.
     */

    while (idx < (word32)sz) {
        word32 oid = 0;
        byte b;

        /* Unwrap an AccessDescription */
        if (GetSequence(input, &idx, &length, sz) < 0)
            return ASN_PARSE_E;

        /* Get the accessMethod */
        if (GetObjectId(input, &idx, &oid, oidCertAuthInfoType, sz) < 0)
            return ASN_PARSE_E;

        /* Only supporting URIs right now. */
        if (GetASNTag(input, &idx, &b, sz) < 0)
            return ASN_PARSE_E;

        if (GetLength(input, &idx, &length, sz) < 0)
            return ASN_PARSE_E;

        /* Set caRepo entry */
        if (b == GENERALNAME_URI && oid == AIA_CA_REPO_OID) {
            cert->extSubjInfoAccCaRepoSz = (word32)length;
            cert->extSubjInfoAccCaRepo = input + idx;
            break;
        }
        idx += (word32)length;
    }

    if (cert->extSubjInfoAccCaRepo == NULL ||
            cert->extSubjInfoAccCaRepoSz == 0) {
        WOLFSSL_MSG("SubjectInfoAccess missing an URL.");
        ret = ASN_PARSE_E;
    }

    WOLFSSL_LEAVE("DecodeSubjInfoAcc", ret);
    return ret;
}
#endif /* WOLFSSL_SUBJ_INFO_ACC */

#ifdef WOLFSSL_DUAL_ALG_CERTS
/* The subject alternative public key is an extension that holds the same thing
 * as a subject public key. */
static const ASNItem subjAltPubKeyInfoASN[] = {
                           /* subjectPublicKeyInfo SubjectPublicKeyInfo */
/* ALT_SPUBKEYINFO_SEQ          */      { 0, ASN_SEQUENCE, 1, 1, 0 },
                           /* algorithm          AlgorithmIdentifier */
                           /* AlgorithmIdentifier ::= SEQUENCE */
/* ALT_SPUBKEYINFO_ALGO_SEQ     */         { 1, ASN_SEQUENCE, 1, 1, 0 },
                          /* Algorithm    OBJECT IDENTIFIER */
/* ALT_SPUBKEYINFO_ALGO_OID     */             { 2, ASN_OBJECT_ID, 0, 0, 0 },
                           /* parameters   ANY defined by algorithm OPTIONAL */
/* ALT_SPUBKEYINFO_ALGO_NULL    */             { 2, ASN_TAG_NULL, 0, 0, 1 },
/* ALT_SPUBKEYINFO_ALGO_CURVEID */             { 2, ASN_OBJECT_ID, 0, 0, 1 },
#ifdef WC_RSA_PSS
/* ALT_SPUBKEYINFO_ALGO_P_SEQ   */             { 2, ASN_SEQUENCE, 1, 0, 1 },
#endif
                           /* subjectPublicKey   BIT STRING */
/* ALT_SPUBKEYINFO_PUBKEY       */          { 1, ASN_BIT_STRING, 0, 0, 0 }
};

#define subjAltPubKeyInfoASN_Length (sizeof(subjAltPubKeyInfoASN) / \
                                     sizeof(ASNItem))

enum {
    ALT_SPUBKEYINFO_SEQ = 0,
    ALT_SPUBKEYINFO_ALGO_SEQ,
    ALT_SPUBKEYINFO_ALGO_OID,
    ALT_SPUBKEYINFO_ALGO_NULL,
    ALT_SPUBKEYINFO_ALGO_CURVEID,
#ifdef WC_RSA_PSS
    ALT_SPUBKEYINFO_ALGO_P_SEQ,
#endif
    ALT_SPUBKEYINFO_PUBKEY
};

static int DecodeSubjAltPubKeyInfo(const byte* input, int sz, DecodedCert* cert)
{
    int ret = 0;
    word32 idx = 0;
    DECL_ASNGETDATA(dataASN, subjAltPubKeyInfoASN_Length);

    WOLFSSL_ENTER("DecodeSubjAltPubKeyInfo");

    if (ret == 0) {
        CALLOC_ASNGETDATA(dataASN, subjAltPubKeyInfoASN_Length, ret,
                          cert->heap);
        (void)cert;
    }

    if (ret == 0) {
        GetASN_OID(&dataASN[ALT_SPUBKEYINFO_ALGO_OID], oidKeyType);
        GetASN_OID(&dataASN[ALT_SPUBKEYINFO_ALGO_CURVEID], oidCurveType);

        ret = GetASN_Items(subjAltPubKeyInfoASN, dataASN,
                           subjAltPubKeyInfoASN_Length, 1, input, &idx,
                           (word32)sz);
    }

    if (ret == 0) {
        /* dataASN[ALT_SPUBKEYINFO_SEQ].data.u8 */
        cert->sapkiDer = (byte *)input;
        /* dataASN[ALT_SPUBKEYINFO_SEQ].length */
        cert->sapkiLen = sz;
        cert->sapkiOID = dataASN[ALT_SPUBKEYINFO_ALGO_OID].data.oid.sum;
    }

    FREE_ASNGETDATA(dataASN, cert->heap);
    WOLFSSL_LEAVE("DecodeSubjAltPubKeyInfo", ret);
    return ret;
}

/* The alternative signature algorithm extension holds the same thing as a
 * as a signature algorithm identifier. */
static const ASNItem altSigAlgASN[] = {
                          /* AltSigAlg            AlgorithmIdentifier */
                          /* AlgorithmIdentifier ::= SEQUENCE */
/* ALTSIG_ALGOID_SEQ                */ { 0, ASN_SEQUENCE, 1, 1, 0 },
                          /* Algorithm    OBJECT IDENTIFIER */
/* ALTSIG_ALGOID_OID                */     { 1, ASN_OBJECT_ID, 0, 0, 0 },
                          /* parameters   ANY defined by algorithm OPTIONAL */
/* ALTSIG_ALGOID_PARAMS_NULL        */     { 1, ASN_TAG_NULL, 0, 0, 1 },
#ifdef WC_RSA_PSS
/* ALTSIG_ALGOID_PARAMS             */     { 1, ASN_SEQUENCE, 1, 0, 1 },
#endif
};

#define altSigAlgASN_Length (sizeof(altSigAlgASN) / sizeof(ASNItem))

enum {
    ALTSIG_ALGOID_SEQ = 0,
    ALTSIG_ALGOID_OID,
    ALTSIG_ALGOID_PARAMS_NULL,
#ifdef WC_RSA_PSS
    ALTSIG_ALGOID_PARAMS,
#endif
};

static int DecodeAltSigAlg(const byte* input, int sz, DecodedCert* cert)
{
    int ret = 0;
    word32 idx = 0;
    DECL_ASNGETDATA(dataASN, altSigAlgASN_Length);

    WOLFSSL_ENTER("DecodeAltSigAlg");

    if (ret == 0) {
        CALLOC_ASNGETDATA(dataASN, altSigAlgASN_Length, ret, cert->heap);
        (void)cert;
    }

    /* We do this to make sure the format of the extension is correct. */
    if (ret == 0) {
        GetASN_OID(&dataASN[ALTSIG_ALGOID_OID], oidSigType);

        ret = GetASN_Items(altSigAlgASN, dataASN,
                           altSigAlgASN_Length, 1, input, &idx,
                           (word32)sz);
    }

    if (ret == 0) {
        cert->altSigAlgDer = (byte *)input;
        cert->altSigAlgLen = sz;
        cert->altSigAlgOID = dataASN[ALTSIG_ALGOID_OID].data.oid.sum;
    }

    FREE_ASNGETDATA(dataASN, cert->heap);
    WOLFSSL_LEAVE("DecodeAltSigAlg", ret);
    return ret;
}

/* The alternative signature value extension holds an ASN.1 bitstring just
 * like a traditional signature in the certificate. */
static int DecodeAltSigVal(const byte* input, int sz, DecodedCert* cert)
{
    int ret = 0;
    word32 idx = 0;
    int len = 0;

    (void)cert;

    WOLFSSL_ENTER("DecodeAltSigVal");

    if (ret == 0) {
        ret = CheckBitString(input, &idx, &len, sz, 1, NULL);
    }

    if (ret == 0) {
        cert->altSigValDer = (byte *)input + idx;
        cert->altSigValLen = len;
    }

    WOLFSSL_LEAVE("DecodeAltSigVal", ret);
    return ret;
}
#endif /* WOLFSSL_DUAL_ALG_CERTS */

/* Macro to check if bit is set, if not sets and return success.
    Otherwise returns failure */
/* Macro required here because bit-field operation */
#ifndef WOLFSSL_NO_ASN_STRICT
    #define VERIFY_AND_SET_OID(bit) \
        if ((bit) == 0) \
            (bit) = 1; \
        else \
            return ASN_OBJECT_ID_E;
#else
    /* With no strict defined, the verify is skipped */
#define VERIFY_AND_SET_OID(bit) bit = 1;
#endif

/* Parse extension type specific data based on OID sum.
 *
 * Supported extensions:
 *   Basic Constraints - BASIC_CA_OID
 *   CRL Distribution Points - CRL_DIST_OID
 *   Authority Information Access - AUTH_INFO_OID
 *   Subject Alternative Name - ALT_NAMES_OID
 *   Authority Key Identifier - AUTH_KEY_OID
 *   Subject Key Identifier - SUBJ_KEY_OID
 *   Certificate Policies - CERT_POLICY_OID (conditional parsing)
 *   Key Usage - KEY_USAGE_OID
 *   Extended Key Usage - EXT_KEY_USAGE_OID
 *   Name Constraints - NAME_CONS_OID
 *   Inhibit anyPolicy - INHIBIT_ANY_OID
 *   Netscape Certificate Type - NETSCAPE_CT_OID (able to be excluded)
 *   OCSP no check - OCSP_NOCHECK_OID (when compiling OCSP)
 *   Subject Directory Attributes - SUBJ_DIR_ATTR_OID
 *   Subject Information Access - SUBJ_INFO_ACC_OID
 * Unsupported extensions from RFC 5280:
 *   4.2.1.5 - Policy mappings
 *   4.2.1.7 - Issuer Alternative Name
 *   4.2.1.11 - Policy Constraints
 *   4.2.1.15 - Freshest CRL
 *
 * @param [in]      input     Buffer containing extension type specific data.
 * @param [in]      length    Length of data.
 * @param [in]      oid       OID sum for extension.
 * @param [in]      critical  Whether extension is critical.
 * @param [in, out] cert      Certificate object.
 * @return  0 on success.
 * @return  ASN_PARSE_E when BER encoding is invalid.
 * @return  MEMORY_E on dynamic memory allocation failure.
 * @return  Other negative values on error.
 */
int DecodeExtensionType(const byte* input, word32 length, word32 oid,
                        byte critical, DecodedCert* cert, int *isUnknownExt)
{
    int ret = 0;
    word32 idx = 0;

    if (isUnknownExt != NULL)
        *isUnknownExt = 0;

    switch (oid) {
        /* Basic Constraints. */
        case BASIC_CA_OID:
            VERIFY_AND_SET_OID(cert->extBasicConstSet);
            cert->extBasicConstCrit = critical ? 1 : 0;
            if (DecodeBasicCaConstraintInternal(input, (int)length, cert) < 0) {
                ret = ASN_PARSE_E;
            }
            break;

        /* CRL Distribution point. */
        case CRL_DIST_OID:
            VERIFY_AND_SET_OID(cert->extCRLdistSet);
            cert->extCRLdistCrit = critical ? 1 : 0;
            if (DecodeCrlDist(input, length, cert) < 0) {
                ret = ASN_PARSE_E;
            }
            break;

        /* Authority information access. */
        case AUTH_INFO_OID:
            VERIFY_AND_SET_OID(cert->extAuthInfoSet);
            cert->extAuthInfoCrit = critical ? 1 : 0;
        #ifndef WOLFSSL_ALLOW_CRIT_AIA
            /* This check is added due to RFC 5280 section 4.2.2.1
            * stating that conforming CA's must mark this extension
            * as non-critical. When parsing extensions check that
            * certificate was made in compliance with this. */
            if (critical) {
                WOLFSSL_MSG("Critical Authority Information Access is not"
                            "allowed");
                WOLFSSL_MSG("Use macro WOLFSSL_ALLOW_CRIT_AIA if wanted");
                ret = ASN_CRIT_EXT_E;
            }
        #endif
            if ((ret == 0) && (DecodeAuthInfo(input, length, cert) < 0)) {
                ret = ASN_PARSE_E;
            }
            break;

        /* Subject alternative name. */
        case ALT_NAMES_OID:
            VERIFY_AND_SET_OID(cert->extSubjAltNameSet);
            cert->extSubjAltNameCrit = critical ? 1 : 0;
            ret = DecodeAltNames(input, length, cert);
            break;

        /* Authority Key Identifier. */
        case AUTH_KEY_OID:
            VERIFY_AND_SET_OID(cert->extAuthKeyIdSet);
            cert->extAuthKeyIdCrit = critical ? 1 : 0;
        #ifndef WOLFSSL_ALLOW_CRIT_AKID
            /* This check is added due to RFC 5280 section 4.2.1.1
             * stating that conforming CA's must mark this extension
             * as non-critical. When parsing extensions check that
             * certificate was made in compliance with this. */
            if (critical) {
                WOLFSSL_MSG("Critical Auth Key ID is not allowed");
                WOLFSSL_MSG("Use macro WOLFSSL_ALLOW_CRIT_AKID if wanted");
                ret = ASN_CRIT_EXT_E;
            }
        #endif
            if ((ret == 0) &&
                (DecodeAuthKeyIdInternal(input, length, cert) < 0)) {
                ret = ASN_PARSE_E;
            }
            break;

        /* Subject Key Identifier. */
        case SUBJ_KEY_OID:
            VERIFY_AND_SET_OID(cert->extSubjKeyIdSet);
            cert->extSubjKeyIdCrit = critical ? 1 : 0;
        #ifndef WOLFSSL_ALLOW_CRIT_SKID
            /* This check is added due to RFC 5280 section 4.2.1.2
             * stating that conforming CA's must mark this extension
             * as non-critical. When parsing extensions check that
             * certificate was made in compliance with this. */
            if (critical) {
                WOLFSSL_MSG("Critical Subject Key ID is not allowed");
                WOLFSSL_MSG("Use macro WOLFSSL_ALLOW_CRIT_SKID if wanted");
                ret = ASN_CRIT_EXT_E;
            }
        #endif

            if ((ret == 0) &&
                (DecodeSubjKeyIdInternal(input, length, cert) < 0)) {
                ret = ASN_PARSE_E;
            }
            break;

        /* Certificate policies. */
        case CERT_POLICY_OID:
        #ifdef WOLFSSL_SEP
            VERIFY_AND_SET_OID(cert->extCertPolicySet);
            cert->extCertPolicyCrit = critical ? 1 : 0;
        #endif
        #if defined(WOLFSSL_SEP) || defined(WOLFSSL_CERT_EXT)
            if (DecodeCertPolicy(input, length, cert) < 0) {
                ret = ASN_PARSE_E;
            }
        #else
            WOLFSSL_MSG("Certificate Policy extension not supported.");
            #ifndef WOLFSSL_NO_ASN_STRICT
            if (critical) {
                WOLFSSL_ERROR_VERBOSE(ASN_CRIT_EXT_E);
                ret = ASN_CRIT_EXT_E;
            }
            #endif
        #endif
            break;

        /* Key usage. */
        case KEY_USAGE_OID:
            VERIFY_AND_SET_OID(cert->extKeyUsageSet);
            cert->extKeyUsageCrit = critical ? 1 : 0;
            if (DecodeKeyUsageInternal(input, length, cert) < 0) {
                ret = ASN_PARSE_E;
            }
            break;

        /* Extended key usage. */
        case EXT_KEY_USAGE_OID:
            VERIFY_AND_SET_OID(cert->extExtKeyUsageSet);
            cert->extExtKeyUsageCrit = critical ? 1 : 0;
            if (DecodeExtKeyUsageInternal(input, length, cert) < 0) {
                ret = ASN_PARSE_E;
            }
            break;

        #ifndef IGNORE_NAME_CONSTRAINTS
        /* Name constraints. */
        case NAME_CONS_OID:
        #ifndef WOLFSSL_NO_ASN_STRICT
            /* Verify RFC 5280 Sec 4.2.1.10 rule:
                "The name constraints extension,
                which MUST be used only in a CA certificate" */
            if (!cert->isCA) {
                WOLFSSL_MSG("Name constraints allowed only for CA certs");
                WOLFSSL_ERROR_VERBOSE(ASN_NAME_INVALID_E);
                ret = ASN_NAME_INVALID_E;
            }
        #endif
            VERIFY_AND_SET_OID(cert->extNameConstraintSet);
            cert->extNameConstraintCrit = critical ? 1 : 0;
            if (DecodeNameConstraints(input, length, cert) < 0) {
                ret = ASN_PARSE_E;
            }
            break;
        #endif /* IGNORE_NAME_CONSTRAINTS */

        /* Inhibit anyPolicy. */
        case INHIBIT_ANY_OID:
            VERIFY_AND_SET_OID(cert->inhibitAnyOidSet);
            WOLFSSL_MSG("Inhibit anyPolicy extension not supported yet.");
            break;

   #ifndef IGNORE_NETSCAPE_CERT_TYPE
        /* Netscape's certificate type. */
        case NETSCAPE_CT_OID:
            if (DecodeNsCertType(input, (int)length, cert) < 0)
                ret = ASN_PARSE_E;
            break;
    #endif
    #ifdef HAVE_OCSP
        /* OCSP no check. */
        case OCSP_NOCHECK_OID:
            VERIFY_AND_SET_OID(cert->ocspNoCheckSet);
            ret = GetASNNull(input, &idx, length);
            if (ret != 0) {
                ret = ASN_PARSE_E;
            }
            break;
    #endif
        case POLICY_CONST_OID:
            VERIFY_AND_SET_OID(cert->extPolicyConstSet);
            cert->extPolicyConstCrit = critical ? 1 : 0;
            if (DecodePolicyConstraints(&input[idx], (int)length, cert) < 0)
                return ASN_PARSE_E;
            break;
    #ifdef WOLFSSL_SUBJ_DIR_ATTR
        case SUBJ_DIR_ATTR_OID:
            VERIFY_AND_SET_OID(cert->extSubjDirAttrSet);
            if (DecodeSubjDirAttr(&input[idx], length, cert) < 0)
                return ASN_PARSE_E;
            break;
    #endif
    #ifdef WOLFSSL_SUBJ_INFO_ACC
        case SUBJ_INFO_ACC_OID:
            VERIFY_AND_SET_OID(cert->extSubjInfoAccSet);
            if (DecodeSubjInfoAcc(&input[idx], length, cert) < 0)
                return ASN_PARSE_E;
            break;
    #endif
    #ifdef WOLFSSL_DUAL_ALG_CERTS
        case SUBJ_ALT_PUB_KEY_INFO_OID:
            VERIFY_AND_SET_OID(cert->extSapkiSet);
            cert->extSapkiCrit = critical ? 1 : 0;
            if (DecodeSubjAltPubKeyInfo(&input[idx], length, cert) < 0)
                return ASN_PARSE_E;
            break;
        case ALT_SIG_ALG_OID:
            VERIFY_AND_SET_OID(cert->extAltSigAlgSet);
            cert->extAltSigAlgCrit = critical ? 1 : 0;
            if (DecodeAltSigAlg(&input[idx], length, cert) < 0)
                return ASN_PARSE_E;
            break;
        case ALT_SIG_VAL_OID:
            VERIFY_AND_SET_OID(cert->extAltSigValSet);
            cert->extAltSigValCrit = critical ? 1 : 0;
            if (DecodeAltSigVal(&input[idx], length, cert) < 0)
                return ASN_PARSE_E;
            break;
    #endif /* WOLFSSL_DUAL_ALG_CERTS */
        default:
            if (isUnknownExt != NULL)
                *isUnknownExt = 1;
        #ifndef WOLFSSL_NO_ASN_STRICT
            /* While it is a failure to not support critical extensions,
             * still parse the certificate ignoring the unsupported
             * extension to allow caller to accept it with the verify
             * callback. */
            if (critical) {
                WOLFSSL_ERROR_VERBOSE(ASN_CRIT_EXT_E);
                ret = ASN_CRIT_EXT_E;
            }
        #endif
            break;
    }

    return ret;
}

#ifdef WOLFSSL_ASN_TEMPLATE
/* ASN.1 template for extensions.
 * X.509: RFC 5280, 4.1 - Basic Certificate Fields.
 */
static const ASNItem certExtHdrASN[] = {
/* EXTTAG */ { 0, ASN_CONTEXT_SPECIFIC | 3, 1, 1, 0 },
/* EXTSEQ */     { 1, ASN_SEQUENCE, 1, 1, 0 },
};
enum {
    CERTEXTHDRASN_IDX_EXTTAG = 0,
    CERTEXTHDRASN_IDX_EXTSEQ
};

/* Number of items in ASN.1 template for extensions. */
#define certExtHdrASN_Length (sizeof(certExtHdrASN) / sizeof(ASNItem))

/* ASN.1 template for Extension.
 * X.509: RFC 5280, 4.1 - Basic Certificate Fields.
 */
static const ASNItem certExtASN[] = {
/* SEQ  */ { 0, ASN_SEQUENCE, 1, 1, 0 },
                              /* Extension object id */
/* OID  */     { 1, ASN_OBJECT_ID, 0, 0, 0 },
                              /* critical - when true, must be parseable. */
/* CRIT */     { 1, ASN_BOOLEAN, 0, 0, 1 },
                              /* Data for extension - leave index at start of data. */
/* VAL  */     { 1, ASN_OCTET_STRING, 0, 1, 0 },
};
enum {
    CERTEXTASN_IDX_SEQ = 0,
    CERTEXTASN_IDX_OID,
    CERTEXTASN_IDX_CRIT,
    CERTEXTASN_IDX_VAL
};

/* Number of items in ASN.1 template for Extension. */
#define certExtASN_Length (sizeof(certExtASN) / sizeof(ASNItem))
#endif

#ifdef WC_ASN_UNKNOWN_EXT_CB
int wc_SetUnknownExtCallback(DecodedCert* cert,
                             wc_UnknownExtCallback cb) {
    if (cert == NULL) {
        return BAD_FUNC_ARG;
    }

    cert->unknownExtCallback = cb;
    return 0;
}

int wc_SetUnknownExtCallbackEx(DecodedCert* cert,
                               wc_UnknownExtCallbackEx cb, void *ctx) {
    if (cert == NULL) {
        return BAD_FUNC_ARG;
    }

    cert->unknownExtCallbackEx = cb;
    cert->unknownExtCallbackExCtx = ctx;
    return 0;
}
#endif /* WC_ASN_UNKNOWN_EXT_CB */

/*
 *  Processing the Certificate Extensions. This does not modify the current
 *  index. It is works starting with the recorded extensions pointer.
 */
static int DecodeCertExtensions(DecodedCert* cert)
{
#ifndef WOLFSSL_ASN_TEMPLATE
    int ret = 0;
    word32 idx = 0;
    word32 sz = (word32)cert->extensionsSz;
    const byte* input = cert->extensions;
    int length;
    word32 oid;
    byte critical = 0;
    byte criticalFail = 0;
    byte tag = 0;

    WOLFSSL_ENTER("DecodeCertExtensions");

    if (input == NULL || sz == 0)
        return BAD_FUNC_ARG;

#ifdef WOLFSSL_CERT_REQ
    if (!cert->isCSR)
#endif
    { /* Not included in CSR */
        if (GetASNTag(input, &idx, &tag, sz) < 0) {
            return ASN_PARSE_E;
        }

        if (tag != ASN_EXTENSIONS) {
            WOLFSSL_MSG("\tfail: should be an EXTENSIONS");
            return ASN_PARSE_E;
        }

        if (GetLength(input, &idx, &length, sz) < 0) {
            WOLFSSL_MSG("\tfail: invalid length");
            return ASN_PARSE_E;
        }
    }

    if (GetSequence(input, &idx, &length, sz) < 0) {
        WOLFSSL_MSG("\tfail: should be a SEQUENCE (1)");
        return ASN_PARSE_E;
    }

    while (idx < (word32)sz) {
        word32 localIdx;

        if (GetSequence(input, &idx, &length, sz) < 0) {
            WOLFSSL_MSG("\tfail: should be a SEQUENCE");
            return ASN_PARSE_E;
        }

        oid = 0;
        if ((ret = GetObjectId(input, &idx, &oid, oidCertExtType, sz)) < 0) {
            WOLFSSL_MSG("\tfail: OBJECT ID");
            return ret;
        }

        /* check for critical flag */
        critical = 0;
        if ((idx + 1) > (word32)sz) {
            WOLFSSL_MSG("\tfail: malformed buffer");
            return BUFFER_E;
        }

        localIdx = idx;
        if (GetASNTag(input, &localIdx, &tag, sz) == 0) {
            if (tag == ASN_BOOLEAN) {
                ret = GetBoolean(input, &idx, sz);
                if (ret < 0) {
                    WOLFSSL_MSG("\tfail: critical boolean");
                    return ret;
                }

                critical = (byte)ret;
            }
        }

        /* process the extension based on the OID */
        ret = GetOctetString(input, &idx, &length, sz);
        if (ret < 0) {
            WOLFSSL_MSG("\tfail: bad OCTET STRING");
            return ret;
        }

        ret = DecodeExtensionType(input + idx, (word32)length, oid, critical,
            cert, NULL);
        if (ret == WC_NO_ERR_TRACE(ASN_CRIT_EXT_E)) {
            ret = 0;
            criticalFail = 1;
        }
        if (ret < 0)
            goto end;
        idx += (word32)length;
    }

    ret = criticalFail ? ASN_CRIT_EXT_E : 0;
end:
    return ret;
#else
    DECL_ASNGETDATA(dataASN, certExtASN_Length);
    ASNGetData dataExtsASN[certExtHdrASN_Length];
    int ret = 0;
    const byte* input = cert->extensions;
    int sz = cert->extensionsSz;
    word32 idx = 0;
    int criticalRet = 0;
    int offset = 0;

    WOLFSSL_ENTER("DecodeCertExtensions");

    if (input == NULL || sz == 0)
        ret = BAD_FUNC_ARG;

    ALLOC_ASNGETDATA(dataASN, certExtASN_Length, ret, cert->heap);

#ifdef WOLFSSL_CERT_REQ
    if (cert->isCSR) {
        offset = CERTEXTHDRASN_IDX_EXTSEQ;
    }
#endif
    if (ret == 0) {
        /* Clear dynamic data. */
        XMEMSET(dataExtsASN, 0, sizeof(dataExtsASN));
        /* Parse extensions header. */
        ret = GetASN_Items(certExtHdrASN + offset, dataExtsASN + offset,
                           (int)(certExtHdrASN_Length - (size_t)offset), 0,
                           input, &idx, (word32)sz);
    }
    /* Parse each extension. */
    while ((ret == 0) && (idx < (word32)sz)) {
        byte critical = 0;
        int isUnknownExt = 0;

        /* Clear dynamic data. */
        XMEMSET(dataASN, 0, sizeof(*dataASN) * certExtASN_Length);
        /* Ensure OID is an extension type. */
        GetASN_OID(&dataASN[CERTEXTASN_IDX_OID], oidCertExtType);
        /* Set criticality variable. */
        GetASN_Int8Bit(&dataASN[CERTEXTASN_IDX_CRIT], &critical);
        /* Parse extension wrapper. */
        ret = GetASN_Items(certExtASN, dataASN, certExtASN_Length, 0, input,
                           &idx, (word32)sz);
        if (ret == 0) {
            word32 oid = dataASN[CERTEXTASN_IDX_OID].data.oid.sum;
            word32 length = dataASN[CERTEXTASN_IDX_VAL].length;

            /* Decode the extension by type. */
            ret = DecodeExtensionType(input + idx, length, oid, critical, cert,
                                      &isUnknownExt);
#ifdef WC_ASN_UNKNOWN_EXT_CB
            if (isUnknownExt && (cert->unknownExtCallback != NULL ||
                                 cert->unknownExtCallbackEx != NULL)) {
                word16 decOid[MAX_OID_SZ];
                word32 decOidSz = sizeof(decOid);
                ret = DecodeObjectId(
                          dataASN[CERTEXTASN_IDX_OID].data.oid.data,
                          dataASN[CERTEXTASN_IDX_OID].data.oid.length,
                          decOid, &decOidSz);
                if (ret != 0) {
                    /* Should never get here as the extension was successfully
                     * decoded earlier. Something might be corrupted. */
                    WOLFSSL_MSG("DecodeObjectId() failed. Corruption?");
                    WOLFSSL_ERROR(ret);
                }

                if ((ret == 0) && (cert->unknownExtCallback != NULL)) {
                    ret = cert->unknownExtCallback(decOid, decOidSz, critical,
                              dataASN[CERTEXTASN_IDX_VAL].data.buffer.data,
                              dataASN[CERTEXTASN_IDX_VAL].length);
                }

                if ((ret == 0) && (cert->unknownExtCallbackEx != NULL)) {
                    ret = cert->unknownExtCallbackEx(decOid, decOidSz, critical,
                              dataASN[CERTEXTASN_IDX_VAL].data.buffer.data,
                              dataASN[CERTEXTASN_IDX_VAL].length,
                              cert->unknownExtCallbackExCtx);
                }
            }
#else
            (void)isUnknownExt;
#endif

            /* Move index on to next extension. */
            idx += length;
        }
        /* Don't fail criticality until all other extensions have been checked.
         */
        if (ret == WC_NO_ERR_TRACE(ASN_CRIT_EXT_E)) {
            criticalRet = ASN_CRIT_EXT_E;
            ret = 0;
        }
    }

    if (ret == 0) {
        /* Use criticality return. */
        ret = criticalRet;
    }

    FREE_ASNGETDATA(dataASN, cert->heap);
    return ret;
#endif
}

#ifdef WOLFSSL_ASN_TEMPLATE

#if defined(HAVE_RPK)
/* ASN template for a Raw Public Key certificate defined RFC7250. */
static const ASNItem RPKCertASN[] = {
/* SubjectPublicKeyInfo ::= SEQUENCE */ { 0, ASN_SEQUENCE, 1, 1, 0 },
    /* algorithm    AlgorithmIdentifier */
    /* AlgorithmIdentifier ::= SEQUENCE */   { 1, ASN_SEQUENCE, 1, 1, 0 },
        /* Algorithm  OBJECT IDENTIFIER */
        /* TBS_SPUBKEYINFO_ALGO_OID     */       { 2, ASN_OBJECT_ID, 0, 0, 0 },
        /* parameters   ANY defined by algorithm OPTIONAL */
        /* TBS_SPUBKEYINFO_ALGO_NULL     */      { 2, ASN_TAG_NULL, 0, 0, 1 },
        /* TBS_SPUBKEYINFO_ALGO_CURVEID  */      { 2, ASN_OBJECT_ID, 0, 0, 1 },
#ifdef WC_RSA_PSS
        /* TBS_SPUBKEYINFO_ALGO_P_SEQ    */      { 2, ASN_SEQUENCE, 1, 0, 1 },
#endif
        /* subjectPublicKey   BIT STRING */
        /* TBS_SPUBKEYINFO_PUBKEY        */   { 1, ASN_BIT_STRING, 0, 0, 0 },
};
/* Number of items in ASN template for a RawPublicKey certificate. */
#define RPKCertASN_Length (sizeof(RPKCertASN) / sizeof(ASNItem))

enum {
    RPKCERTASN_IDX_SPUBKEYINFO_SEQ = 0,
    RPKCERTASN_IDX_SPUBKEYINFO_ALGO_SEQ,
    RPKCERTASN_IDX_SPUBKEYINFO_ALGO_OID,
    RPKCERTASN_IDX_SPUBKEYINFO_ALGO_NULL,
    RPKCERTASN_IDX_SPUBKEYINFO_ALGO_CURVEID,
#ifdef WC_RSA_PSS
    RPKCERTASN_IDX_SPUBKEYINFO_ALGO_P_SEQ,
#endif
    RPKCERTASN_IDX_SPUBKEYINFO_PUBKEY
};

#endif /* HAVE_RPK */

/* ASN template for an X509 certificate.
 * X.509: RFC 5280, 4.1 - Basic Certificate Fields.
 */
static const ASNItem x509CertASN[] = {
        /* Certificate ::= SEQUENCE */
/* SEQ                           */    { 0, ASN_SEQUENCE, 1, 1, 0 },
                                                   /* tbsCertificate       TBSCertificate */
                                                   /* TBSCertificate ::= SEQUENCE */
/* TBS_SEQ                       */        { 1, ASN_SEQUENCE, 1, 1, 0 },
                                                   /* version         [0]  EXPLICIT Version DEFAULT v1 */
/* TBS_VER                       */            { 2, ASN_CONTEXT_SPECIFIC | ASN_X509_CERT_VERSION, 1, 1, 1 },
                                                   /* Version ::= INTEGER { v1(0), v2(1), v3(2) */
/* TBS_VER_INT                   */                { 3, ASN_INTEGER, 0, 0, 0 },
                                                   /* serialNumber         CertificateSerialNumber */
                                                   /* CertificateSerialNumber ::= INTEGER */
/* TBS_SERIAL                    */            { 2, ASN_INTEGER, 0, 0, 0 },
                                                   /* signature            AlgorithmIdentifier */
                                                   /* AlgorithmIdentifier ::= SEQUENCE */
/* TBS_ALGOID_SEQ                */            { 2, ASN_SEQUENCE, 1, 1, 0 },
                                                   /* Algorithm    OBJECT IDENTIFIER */
/* TBS_ALGOID_OID                */                { 3, ASN_OBJECT_ID, 0, 0, 0 },
                                                   /* parameters   ANY defined by algorithm OPTIONAL */
/* TBS_ALGOID_PARAMS_NULL        */                { 3, ASN_TAG_NULL, 0, 0, 2 },
#ifdef WC_RSA_PSS
/* TBS_ALGOID_PARAMS             */                { 3, ASN_SEQUENCE, 1, 0, 2 },
#endif
                                                   /* issuer               Name */
/* TBS_ISSUER_SEQ                */            { 2, ASN_SEQUENCE, 1, 0, 0 },
                                                   /* validity             Validity */
                                                   /* Validity ::= SEQUENCE */
/* TBS_VALIDITY_SEQ              */            { 2, ASN_SEQUENCE, 1, 1, 0 },
                                                   /* notBefore   Time */
                                                   /* Time :: CHOICE { UTCTime, GeneralizedTime } */
/* TBS_VALIDITY_NOTB_UTC         */                { 3, ASN_UTC_TIME, 0, 0, 2 },
/* TBS_VALIDITY_NOTB_GT          */                { 3, ASN_GENERALIZED_TIME, 0, 0, 2 },
                                                   /* notAfter   Time */
                                                   /* Time :: CHOICE { UTCTime, GeneralizedTime } */
/* TBS_VALIDITY_NOTA_UTC         */                { 3, ASN_UTC_TIME, 0, 0, 3 },
/* TBS_VALIDITY_NOTA_GT          */                { 3, ASN_GENERALIZED_TIME, 0, 0, 3 },
                                                   /* subject              Name */
/* TBS_SUBJECT_SEQ               */            { 2, ASN_SEQUENCE, 1, 0, 0 },
                                                   /* subjectPublicKeyInfo SubjectPublicKeyInfo */
/* TBS_SPUBKEYINFO_SEQ           */            { 2, ASN_SEQUENCE, 1, 1, 0 },
                                                   /* algorithm          AlgorithmIdentifier */
                                                   /* AlgorithmIdentifier ::= SEQUENCE */
/* TBS_SPUBKEYINFO_ALGO_SEQ      */                { 3, ASN_SEQUENCE, 1, 1, 0 },
                                                   /* Algorithm    OBJECT IDENTIFIER */
/* TBS_SPUBKEYINFO_ALGO_OID      */                    { 4, ASN_OBJECT_ID, 0, 0, 0 },
                                                   /* parameters   ANY defined by algorithm OPTIONAL */
/* TBS_SPUBKEYINFO_ALGO_NULL     */                    { 4, ASN_TAG_NULL, 0, 0, 2 },
/* TBS_SPUBKEYINFO_ALGO_CURVEID  */                    { 4, ASN_OBJECT_ID, 0, 0, 2 },
#ifdef WC_RSA_PSS
/* TBS_SPUBKEYINFO_ALGO_P_SEQ    */                    { 4, ASN_SEQUENCE, 1, 0, 2 },
#endif
                                                   /* subjectPublicKey   BIT STRING */
/* TBS_SPUBKEYINFO_PUBKEY        */                { 3, ASN_BIT_STRING, 0, 0, 0 },
                                                   /* issuerUniqueID       UniqueIdentfier OPTIONAL */
/* TBS_ISSUERUID                 */            { 2, ASN_CONTEXT_SPECIFIC | 1, 0, 0, 1 },
                                                   /* subjectUniqueID      UniqueIdentfier OPTIONAL */
/* TBS_SUBJECTUID                */            { 2, ASN_CONTEXT_SPECIFIC | 2, 0, 0, 1 },
                                                   /* extensions           Extensions OPTIONAL */
/* TBS_EXT                       */            { 2, ASN_CONTEXT_SPECIFIC | 3, 1, 1, 1 },
/* TBS_EXT_SEQ                   */                { 3, ASN_SEQUENCE, 1, 0, 0 },
                                                   /* signatureAlgorithm   AlgorithmIdentifier */
                                                   /* AlgorithmIdentifier ::= SEQUENCE */
/* SIGALGO_SEQ                   */        { 1, ASN_SEQUENCE, 1, 1, 0 },
                                                   /* Algorithm    OBJECT IDENTIFIER */
/* SIGALGO_OID                   */            { 2, ASN_OBJECT_ID, 0, 0, 0 },
                                                   /* parameters   ANY defined by algorithm OPTIONAL */
/* SIGALGO_PARAMS_NULL           */            { 2, ASN_TAG_NULL, 0, 0, 2 },
#ifdef WC_RSA_PSS
/* SIGALGO_PARAMS                */            { 2, ASN_SEQUENCE, 1, 0, 2 },
#endif
                                                   /* signature            BIT STRING */
/* SIGNATURE                     */        { 1, ASN_BIT_STRING, 0, 0, 0 },
};
enum {
    X509CERTASN_IDX_SEQ = 0,
    X509CERTASN_IDX_TBS_SEQ,
    X509CERTASN_IDX_TBS_VER,
    X509CERTASN_IDX_TBS_VER_INT,
    X509CERTASN_IDX_TBS_SERIAL,
    X509CERTASN_IDX_TBS_ALGOID_SEQ,
    X509CERTASN_IDX_TBS_ALGOID_OID,
    X509CERTASN_IDX_TBS_ALGOID_PARAMS_NULL,
#ifdef WC_RSA_PSS
    X509CERTASN_IDX_TBS_ALGOID_PARAMS,
#endif
    X509CERTASN_IDX_TBS_ISSUER_SEQ,
    X509CERTASN_IDX_TBS_VALIDITY_SEQ,
    X509CERTASN_IDX_TBS_VALIDITY_NOTB_UTC,
    X509CERTASN_IDX_TBS_VALIDITY_NOTB_GT,
    X509CERTASN_IDX_TBS_VALIDITY_NOTA_UTC,
    X509CERTASN_IDX_TBS_VALIDITY_NOTA_GT,
    X509CERTASN_IDX_TBS_SUBJECT_SEQ,
    X509CERTASN_IDX_TBS_SPUBKEYINFO_SEQ,
    X509CERTASN_IDX_TBS_SPUBKEYINFO_ALGO_SEQ,
    X509CERTASN_IDX_TBS_SPUBKEYINFO_ALGO_OID,
    X509CERTASN_IDX_TBS_SPUBKEYINFO_ALGO_NULL,
    X509CERTASN_IDX_TBS_SPUBKEYINFO_ALGO_CURVEID,
#ifdef WC_RSA_PSS
    X509CERTASN_IDX_TBS_SPUBKEYINFO_ALGO_P_SEQ,
#endif
    X509CERTASN_IDX_TBS_SPUBKEYINFO_PUBKEY,
    X509CERTASN_IDX_TBS_ISSUERUID,
    X509CERTASN_IDX_TBS_SUBJECTUID,
    X509CERTASN_IDX_TBS_EXT,
    X509CERTASN_IDX_TBS_EXT_SEQ,
    X509CERTASN_IDX_SIGALGO_SEQ,
    X509CERTASN_IDX_SIGALGO_OID,
    X509CERTASN_IDX_SIGALGO_PARAMS_NULL,
#ifdef WC_RSA_PSS
    X509CERTASN_IDX_SIGALGO_PARAMS,
#endif
    X509CERTASN_IDX_SIGNATURE,
    WOLF_ENUM_DUMMY_LAST_ELEMENT(X509CERTASN_IDX)
};

/* Number of items in ASN template for an X509 certificate. */
#define x509CertASN_Length (sizeof(x509CertASN) / sizeof(ASNItem))

/* Check the data data.
 *
 * @param [in] dataASN   ASN template dynamic data item.
 * @param [in] dataType  ASN_BEFORE or ASN_AFTER date.
 * @return  0 on success.
 * @return  ASN_TIME_E when BER tag is nor UTC or GENERALIZED time.
 * @return  ASN_DATE_SZ_E when time data is not supported.
 * @return  ASN_BEFORE_DATE_E when ASN_BEFORE date is invalid.
 * @return  ASN_AFTER_DATE_E when ASN_AFTER date is invalid.
 */
static int CheckDate(ASNGetData *dataASN, int dateType)
{
    int ret = 0;

    /* Check BER tag is valid. */
    if ((dataASN->tag != ASN_UTC_TIME) &&
            (dataASN->tag != ASN_GENERALIZED_TIME)) {
        ret = ASN_TIME_E;
    }
    /* Check date length is valid. */
    if ((ret == 0) && ((dataASN->length > MAX_DATE_SIZE) ||
                       (dataASN->length < MIN_DATE_SIZE))) {
        ret = ASN_DATE_SZ_E;
    }

#ifndef NO_ASN_TIME_CHECK
    /* Check date is a valid string and ASN_BEFORE or ASN_AFTER now. */
    if ((ret == 0) && (! AsnSkipDateCheck)) {
        if (!XVALIDATE_DATE(dataASN->data.ref.data, dataASN->tag, dateType)) {
            if (dateType == ASN_BEFORE) {
                ret = ASN_BEFORE_DATE_E;
            }
            else if (dateType == ASN_AFTER) {
                ret = ASN_AFTER_DATE_E;
            }
            else {
                ret = ASN_TIME_E;
            }
        }
    }
#endif
    (void)dateType;

    return ret;
}

/* Decode a certificate. Internal/non-public API.
 *
 * @param [in]  cert             Certificate object.
 * @param [in]  verify           Whether to verify dates before and after now.
 * @param [out] criticalExt      Critical extension return code.
 * @param [out] badDateRet       Bad date return code.
 * @param [in]  stopAtPubKey     Stop parsing before subjectPublicKeyInfo.
 * @param [in]  stopAfterPubKey  Stop parsing after subjectPublicKeyInfo.
 * @return  0 on success if of the stop arguments is not set, otherwise set to
 *          the corresponding byte offset at which the parsing stopped.
 * @return  ASN_CRIT_EXT_E when a critical extension was not recognized.
 * @return  ASN_TIME_E when date BER tag is nor UTC or GENERALIZED time.
 * @return  ASN_DATE_SZ_E when time data is not supported.
 * @return  ASN_BEFORE_DATE_E when ASN_BEFORE date is invalid.
 * @return  ASN_AFTER_DATE_E when ASN_AFTER date is invalid.
 * @return  ASN_PARSE_E when BER encoded data does not match ASN.1 items or
 *          is invalid.
 * @return  BUFFER_E when data in buffer is too small.
 * @return  ASN_OBJECT_ID_E when the expected OBJECT_ID tag is not found.
 * @return  ASN_BITSTR_E when the expected BIT_STRING tag is not found.
 * @return  ASN_EXPECT_0_E when the INTEGER has the MSB set or NULL has a
 *          non-zero length.
 * @return  ASN_UNKNOWN_OID_E when the OID cannot be verified.
 */
static int DecodeCertInternal(DecodedCert* cert, int verify, int* criticalExt,
                              int* badDateRet, int stopAtPubKey,
                              int stopAfterPubKey)
{
    DECL_ASNGETDATA(dataASN, x509CertASN_Length);
    int ret = 0;
    int badDate = 0;
    byte version = 0;
    word32 idx;
    word32 serialSz = 0;
    const unsigned char* issuer = NULL;
    word32 issuerSz = 0;
    const unsigned char* subject = NULL;
    word32 subjectSz = 0;
    word32 pubKeyOffset = 0;
    word32 pubKeyEnd = 0;
    int done = 0;

#if defined(HAVE_RPK)
    /* try to parse the cert as Raw Public Key cert */
    DECL_ASNGETDATA(RPKdataASN, RPKCertASN_Length);
    CALLOC_ASNGETDATA(RPKdataASN, RPKCertASN_Length, ret, cert->heap);
    GetASN_OID(&RPKdataASN[RPKCERTASN_IDX_SPUBKEYINFO_ALGO_OID],
                                                                oidKeyType);
    GetASN_OID(&RPKdataASN[RPKCERTASN_IDX_SPUBKEYINFO_ALGO_CURVEID],
                                                                oidCurveType);
    ret = GetASN_Items(RPKCertASN, RPKdataASN, RPKCertASN_Length, 1,
                           cert->source, &cert->srcIdx, cert->maxIdx);

    if (ret == 0) {
        if (( RPKdataASN[RPKCERTASN_IDX_SPUBKEYINFO_ALGO_NULL].length &&
              RPKdataASN[RPKCERTASN_IDX_SPUBKEYINFO_ALGO_CURVEID].length)
#ifdef WC_RSA_PSS
         || ( RPKdataASN[RPKCERTASN_IDX_SPUBKEYINFO_ALGO_P_SEQ].length &&
            ( RPKdataASN[RPKCERTASN_IDX_SPUBKEYINFO_ALGO_NULL].length ||
              RPKdataASN[RPKCERTASN_IDX_SPUBKEYINFO_ALGO_CURVEID].length))
#endif
        ) {
            WOLFSSL_MSG("Multiple RPK algorithm parameters set.");
            ret = ASN_PARSE_E;
        }
    }
    if (ret == 0) {
        cert->keyOID =
                RPKdataASN[RPKCERTASN_IDX_SPUBKEYINFO_ALGO_OID].data.oid.sum;

        /* Parse the public key. */
        pubKeyOffset = RPKdataASN[RPKCERTASN_IDX_SPUBKEYINFO_SEQ].offset;
        pubKeyEnd = cert->maxIdx;
        ret = GetCertKey(cert, cert->source, &pubKeyOffset, pubKeyEnd);
        if (ret == 0) {
            WOLFSSL_MSG("Raw Public Key certificate found and parsed");
            cert->isRPK = 1;
        }
    }
    /* Dispose of memory before allocating for extension decoding. */
    FREE_ASNGETDATA(RPKdataASN, cert->heap);

    if (ret == 0) {
        return ret;
    }
    else {
        ret = 0;    /* proceed to the original x509 parsing */
    }
#endif /* HAVE_RPK */

    CALLOC_ASNGETDATA(dataASN, x509CertASN_Length, ret, cert->heap);

    if (ret == 0) {
        version = 0;
        serialSz = EXTERNAL_SERIAL_SIZE;

        /* Get the version and put the serial number into the buffer. */
        GetASN_Int8Bit(&dataASN[X509CERTASN_IDX_TBS_VER_INT], &version);
        GetASN_Buffer(&dataASN[X509CERTASN_IDX_TBS_SERIAL], cert->serial,
                &serialSz);
        /* Check OID types for signature, algorithm, ECC curve and sigAlg. */
        GetASN_OID(&dataASN[X509CERTASN_IDX_TBS_ALGOID_OID], oidSigType);
        GetASN_OID(&dataASN[X509CERTASN_IDX_TBS_SPUBKEYINFO_ALGO_OID],
                oidKeyType);
        GetASN_OID(&dataASN[X509CERTASN_IDX_TBS_SPUBKEYINFO_ALGO_CURVEID],
                oidCurveType);
        GetASN_OID(&dataASN[X509CERTASN_IDX_SIGALGO_OID], oidSigType);
        /* Parse the X509 certificate. */
        ret = GetASN_Items(x509CertASN, dataASN, x509CertASN_Length, 1,
                           cert->source, &cert->srcIdx, cert->maxIdx);
#ifdef WOLFSSL_CLANG_TIDY
        /* work around clang-tidy false positive re cert->source. */
        if ((ret == 0) && (cert->source == NULL)) {
            ret = ASN_PARSE_E;
        }
#endif
    }
    /* Check version is valid/supported - can't be negative. */
    if ((ret == 0) && (version > MAX_X509_VERSION)) {
        WOLFSSL_MSG("Unexpected certificate version");
        WOLFSSL_ERROR_VERBOSE(ASN_PARSE_E);
        ret = ASN_PARSE_E;
    }
    if (ret == 0) {
        int i;

        pubKeyOffset = dataASN[X509CERTASN_IDX_TBS_SPUBKEYINFO_SEQ].offset;
        /* Set fields extracted from data. */
        cert->version = version;
        cert->serialSz = (int)serialSz;

    #if !defined(WOLFSSL_NO_ASN_STRICT) && !defined(WOLFSSL_PYTHON) && \
        !defined(WOLFSSL_ASN_ALLOW_0_SERIAL)
        /* RFC 5280 section 4.1.2.2 states that non-conforming CAs may issue
         * a negative or zero serial number and should be handled gracefully.
         * Since it is a non-conforming CA that issues a serial of 0 then we
         * treat it as an error here. */
        if (cert->serialSz == 1 && cert->serial[0] == 0) {
            WOLFSSL_MSG("Error serial number of 0, use WOLFSSL_NO_ASN_STRICT "
                "if wanted");
            ret = ASN_PARSE_E;
        }
    #endif
        if (cert->serialSz == 0) {
            WOLFSSL_MSG("Error serial size is zero. Should be at least one "
                        "even with no serial number.");
            ret = ASN_PARSE_E;
        }

        cert->signatureOID = dataASN[X509CERTASN_IDX_TBS_ALGOID_OID].data.oid.sum;
        cert->keyOID = dataASN[X509CERTASN_IDX_TBS_SPUBKEYINFO_ALGO_OID].data.oid.sum;
        cert->certBegin = dataASN[X509CERTASN_IDX_TBS_SEQ].offset;

        /* No bad date error - don't always care. */
        badDate = 0;
        /* Find the item with the ASN_BEFORE date and check it. */
        i = (dataASN[X509CERTASN_IDX_TBS_VALIDITY_NOTB_UTC].tag != 0)
                ? X509CERTASN_IDX_TBS_VALIDITY_NOTB_UTC
                : X509CERTASN_IDX_TBS_VALIDITY_NOTB_GT;
        if ((CheckDate(&dataASN[i], ASN_BEFORE) < 0) && (verify != NO_VERIFY) &&
                (verify != VERIFY_SKIP_DATE) && (! AsnSkipDateCheck)) {
            badDate = ASN_BEFORE_DATE_E;
        }
        /* Store reference to ASN_BEFORE date. */
        cert->beforeDate = GetASNItem_Addr(dataASN[i], cert->source);
        cert->beforeDateLen = (int)GetASNItem_Length(dataASN[i], cert->source);

        /* Find the item with the ASN_AFTER date and check it. */
        i = (dataASN[X509CERTASN_IDX_TBS_VALIDITY_NOTA_UTC].tag != 0)
                ? X509CERTASN_IDX_TBS_VALIDITY_NOTA_UTC
                : X509CERTASN_IDX_TBS_VALIDITY_NOTA_GT;
        if ((CheckDate(&dataASN[i], ASN_AFTER) < 0) && (verify != NO_VERIFY) &&
                (verify != VERIFY_SKIP_DATE) && (! AsnSkipDateCheck)) {
            badDate = ASN_AFTER_DATE_E;
        }
        /* Store reference to ASN_AFTER date. */
        cert->afterDate = GetASNItem_Addr(dataASN[i], cert->source);
        cert->afterDateLen = (int)GetASNItem_Length(dataASN[i], cert->source);

        /* Get the issuer name. */
        issuer = cert->source + dataASN[X509CERTASN_IDX_TBS_ISSUER_SEQ].offset;
        issuerSz = dataASN[X509CERTASN_IDX_TBS_VALIDITY_SEQ].offset -
            dataASN[X509CERTASN_IDX_TBS_ISSUER_SEQ].offset;

        /* Get the subject name. */
        subject = cert->source +
            dataASN[X509CERTASN_IDX_TBS_SUBJECT_SEQ].offset;
        subjectSz = dataASN[X509CERTASN_IDX_TBS_SPUBKEYINFO_SEQ].offset -
            dataASN[X509CERTASN_IDX_TBS_SUBJECT_SEQ].offset;
    }
    if ((ret == 0) && stopAtPubKey) {
        /* Return any bad date error through badDateRet and return offset of
         * subjectPublicKeyInfo.
         */
        if (badDateRet != NULL) {
            *badDateRet = badDate;
        }
        done = 1;
    }

    if ((ret == 0) && (!done)) {
        /* Store the signature information. */
        cert->sigIndex = dataASN[X509CERTASN_IDX_SIGALGO_SEQ].offset;
        GetASN_GetConstRef(&dataASN[X509CERTASN_IDX_SIGNATURE],
                &cert->signature, &cert->sigLength);
        /* Make sure 'signature' and 'signatureAlgorithm' are the same. */
        if (dataASN[X509CERTASN_IDX_SIGALGO_OID].data.oid.sum
                != cert->signatureOID) {
            WOLFSSL_ERROR_VERBOSE(ASN_SIG_OID_E);
            ret = ASN_SIG_OID_E;
        }
        /* Parameters not allowed after ECDSA or EdDSA algorithm OID. */
        else if (IsSigAlgoECC(cert->signatureOID)) {
        #ifndef WOLFSSL_ECC_SIGALG_PARAMS_NULL_ALLOWED
            if (dataASN[X509CERTASN_IDX_SIGALGO_PARAMS_NULL].tag != 0) {
                WOLFSSL_ERROR_VERBOSE(ASN_PARSE_E);
                ret = ASN_PARSE_E;
            }
        #endif
        #ifdef WC_RSA_PSS
            if (dataASN[X509CERTASN_IDX_SIGALGO_PARAMS].tag != 0) {
                WOLFSSL_ERROR_VERBOSE(ASN_PARSE_E);
                ret = ASN_PARSE_E;
            }
        #endif
        }
    #ifdef WC_RSA_PSS
        /* Check parameters starting with a SEQUENCE. */
        else if (dataASN[X509CERTASN_IDX_SIGALGO_PARAMS].tag != 0) {
            word32 oid = dataASN[X509CERTASN_IDX_SIGALGO_OID].data.oid.sum;
            word32 sigAlgParamsSz = 0;

            /* Parameters only with RSA PSS. */
            if (oid != CTC_RSASSAPSS) {
                WOLFSSL_ERROR_VERBOSE(ASN_PARSE_E);
                ret = ASN_PARSE_E;
            }
            if (ret == 0) {
                const byte* tbsParams;
                word32 tbsParamsSz;
                const byte* sigAlgParams;

                /* Check RSA PSS parameters are the same. */
                tbsParams =
                    GetASNItem_Addr(dataASN[X509CERTASN_IDX_TBS_ALGOID_PARAMS],
                        cert->source);
                tbsParamsSz =
                    GetASNItem_Length(dataASN[X509CERTASN_IDX_TBS_ALGOID_PARAMS],
                        cert->source);
                sigAlgParams =
                    GetASNItem_Addr(dataASN[X509CERTASN_IDX_SIGALGO_PARAMS],
                        cert->source);
                sigAlgParamsSz =
                    GetASNItem_Length(dataASN[X509CERTASN_IDX_SIGALGO_PARAMS],
                        cert->source);
                if ((tbsParamsSz != sigAlgParamsSz) ||
                        (XMEMCMP(tbsParams, sigAlgParams, tbsParamsSz) != 0)) {
                    WOLFSSL_ERROR_VERBOSE(ASN_PARSE_E);
                    ret = ASN_PARSE_E;
                }
            }
            if (ret == 0) {
                /* Store parameters for use in signature verification. */
                cert->sigParamsIndex =
                    dataASN[X509CERTASN_IDX_SIGALGO_PARAMS].offset;
                cert->sigParamsLength = sigAlgParamsSz;
            }
        }
    #endif
    }
    if ((ret == 0) && (!done)) {
        pubKeyEnd = dataASN[X509CERTASN_IDX_TBS_ISSUERUID].offset;
        if (stopAfterPubKey) {
            /* Return any bad date error through badDateRed and return offset
             * after subjectPublicKeyInfo.
             */
            if (badDateRet != NULL) {
                *badDateRet = badDate;
            }
            done = 1;
        }
    }
    if ((ret == 0) && (!done) &&
            (dataASN[X509CERTASN_IDX_TBS_EXT_SEQ].data.ref.data != NULL)) {
    #ifndef ALLOW_V1_EXTENSIONS
        /* Certificate extensions were only defined in version 2. */
        if (cert->version < 2) {
            WOLFSSL_MSG("\tv1 and v2 certs not allowed extensions");
            WOLFSSL_ERROR_VERBOSE(ASN_VERSION_E);
            ret = ASN_VERSION_E;
        }
    #endif
        if (ret == 0) {
            /* Save references to extension data. */
            cert->extensions    = GetASNItem_Addr(
                    dataASN[X509CERTASN_IDX_TBS_EXT], cert->source);
            cert->extensionsSz  = (int)GetASNItem_Length(
                    dataASN[X509CERTASN_IDX_TBS_EXT], cert->source);
            cert->extensionsIdx = dataASN[X509CERTASN_IDX_TBS_EXT].offset;
            /* Advance past extensions. */
            cert->srcIdx = dataASN[X509CERTASN_IDX_SIGALGO_SEQ].offset;
        }
    }

    /* Dispose of memory before allocating for extension decoding. */
    FREE_ASNGETDATA(dataASN, cert->heap);

    if ((ret == 0) && (issuer != NULL)) {
        idx = 0;
        /* Put issuer into cert and calculate hash. */
        ret = GetCertName(cert, cert->issuer, cert->issuerHash, ASN_ISSUER, issuer,
            &idx, issuerSz);
    }
    if ((ret == 0) && (subject != NULL)) {
        idx = 0;
        /* Put subject into cert and calculate hash. */
        ret = GetCertName(cert, cert->subject, cert->subjectHash, ASN_SUBJECT,
            subject, &idx, subjectSz);
    }
    if (ret == 0) {
        /* Determine if self signed by comparing issuer and subject hashes. */
    #ifdef WOLFSSL_CERT_REQ
        if (cert->isCSR) {
            cert->selfSigned = 1;
        }
        else
    #endif
        {
            cert->selfSigned = (XMEMCMP(cert->issuerHash, cert->subjectHash,
                                        KEYID_SIZE) == 0);
        }
        if (stopAtPubKey) {
            ret = (int)pubKeyOffset;
        }
    }

    if ((ret == 0) && (!stopAtPubKey)) {
        /* Parse the public key. */
        idx = pubKeyOffset;
        ret = GetCertKey(cert, cert->source, &idx, pubKeyEnd);
    }
    if ((ret == 0) && (!stopAtPubKey) && (!stopAfterPubKey) &&
            (cert->extensions != NULL)) {
        /* Decode the extension data starting at [3]. */
        ret = DecodeCertExtensions(cert);
        if (criticalExt != NULL) {
            if (ret == WC_NO_ERR_TRACE(ASN_CRIT_EXT_E)) {
                /* Return critical extension not recognized. */
                *criticalExt = ret;
                ret = 0;
            }
            else {
                /* No critical extension error. */
                *criticalExt = 0;
            }
        }
    }

    if ((ret == 0) && (!done) && (badDate != 0)) {
        /* Parsed whole certificate fine but return any date errors. */
        ret = badDate;
    }

    return ret;
}

/* Decode BER/DER data into certificate object.
 *
 * BER/DER data information held in source, srcIdx and maxIdx fields of
 * certificate object.
 *
 * @param [in] cert         Decoded certificate object.
 * @param [in] verify       Whether to find CA and verify certificate.
 * @param [in] criticalExt  Any error for critical extensions not recognized.
 * @return  0 on success.
 * @return  ASN_CRIT_EXT_E when a critical extension was not recognized.
 * @return  ASN_TIME_E when date BER tag is nor UTC or GENERALIZED time.
 * @return  ASN_DATE_SZ_E when time data is not supported.
 * @return  ASN_BEFORE_DATE_E when ASN_BEFORE date is invalid.
 * @return  ASN_AFTER_DATE_E when ASN_AFTER date is invalid.
 * @return  ASN_PARSE_E when BER encoded data does not match ASN.1 items or
 *          is invalid.
 * @return  BUFFER_E when data in buffer is too small.
 * @return  ASN_OBJECT_ID_E when the expected OBJECT_ID tag is not found.
 * @return  ASN_BITSTR_E when the expected BIT_STRING tag is not found.
 * @return  ASN_EXPECT_0_E when the INTEGER has the MSB set or NULL has a
 *          non-zero length.
 * @return  ASN_UNKNOWN_OID_E when the OID cannot be verified.
 */
int DecodeCert(DecodedCert* cert, int verify, int* criticalExt)
{
    return DecodeCertInternal(cert, verify, criticalExt, NULL, 0, 0);
}

#ifdef WOLFSSL_CERT_REQ
/* ASN.1 template for certificate request Attribute.
 * PKCS #10: RFC 2986, 4.1 - CertificationRequestInfo
 */
static const ASNItem reqAttrASN[] = {
/* SEQ  */ { 0, ASN_SEQUENCE, 1, 1, 0 },
                              /* type */
/* TYPE */     { 1, ASN_OBJECT_ID, 0, 0, 0 },
                              /* values */
/* VALS */     { 1, ASN_SET, 1, 0, 0 },
};
enum {
    REQATTRASN_IDX_SEQ = 0,
    REQATTRASN_IDX_TYPE,
    REQATTRASN_IDX_VALS
};

/* Number of items in ASN.1 template for certificate request Attribute. */
#define reqAttrASN_Length (sizeof(reqAttrASN) / sizeof(ASNItem))

/* ASN.1 template for a string choice. */
static const ASNItem strAttrASN[] = {
    { 0, 0, 0, 0, 0 },
};
enum {
    STRATTRASN_IDX_STR = 0
};

/* Number of items in ASN.1 template for a string choice. */
#define strAttrASN_Length (sizeof(strAttrASN) / sizeof(ASNItem))

/* ASN.1 choices for types for a string in an attribute. */
static const byte strAttrChoice[] = {
    ASN_PRINTABLE_STRING, ASN_IA5_STRING, ASN_UTF8STRING, 0
};

/* Decode a certificate request attribute's value.
 *
 * @param [in]  cert         Certificate request object.
 * @param [out] criticalExt  Critical extension return code.
 * @param [in]  oid          OID describing which attribute was found.
 * @param [in]  aIdx         Index into certificate source to start parsing.
 * @param [in]  input        Attribute value data.
 * @param [in]  maxIdx       Maximum index to parse to.
 * @return  0 on success.
 * @return  ASN_PARSE_E when BER encoded data does not match ASN.1 items or
 *          is invalid.
 */
static int DecodeCertReqAttrValue(DecodedCert* cert, int* criticalExt,
    word32 oid, word32 aIdx, const byte* input, word32 maxIdx)
{
    int ret = 0;
    word32 idx = 0;
    ASNGetData strDataASN[strAttrASN_Length];

    switch (oid) {
        case PKCS9_CONTENT_TYPE_OID:
            /* Clear dynamic data and specify choices acceptable. */
            XMEMSET(strDataASN, 0, sizeof(strDataASN));
            GetASN_Choice(&strDataASN[STRATTRASN_IDX_STR], strAttrChoice);
            /* Parse a string. */
            ret = GetASN_Items(strAttrASN, strDataASN, strAttrASN_Length,
                               1, input, &idx, maxIdx);
            if (ret == 0) {
                /* Store references to password data. */
                cert->contentType =
                        (char*)strDataASN[STRATTRASN_IDX_STR].data.ref.data;
                cert->contentTypeLen =
                        (int)strDataASN[STRATTRASN_IDX_STR].data.ref.length;
            }
            break;

        /* A password by which the entity may request certificate revocation.
         * PKCS#9: RFC 2985, 5.4.1 - Challenge password
         */
        case CHALLENGE_PASSWORD_OID:
            /* Clear dynamic data and specify choices acceptable. */
            XMEMSET(strDataASN, 0, sizeof(strDataASN));
            GetASN_Choice(&strDataASN[STRATTRASN_IDX_STR], strAttrChoice);
            /* Parse a string. */
            ret = GetASN_Items(strAttrASN, strDataASN, strAttrASN_Length,
                               1, input, &idx, maxIdx);
            if (ret == 0) {
                /* Store references to password data. */
                cert->cPwd =
                        (char*)strDataASN[STRATTRASN_IDX_STR].data.ref.data;
                cert->cPwdLen = (int)strDataASN[STRATTRASN_IDX_STR].
                    data.ref.length;
            }
            break;

        /* Requested serial number to issue with.
         * PKCS#9: RFC 2985, 5.2.10 - Serial Number
         * (References: ISO/IEC 9594-6:1997)
         */
        case SERIAL_NUMBER_OID:
            /* Clear dynamic data and specify choices acceptable. */
            XMEMSET(strDataASN, 0, sizeof(strDataASN));
            GetASN_Choice(&strDataASN[STRATTRASN_IDX_STR], strAttrChoice);
            /* Parse a string. */
            ret = GetASN_Items(strAttrASN, strDataASN, strAttrASN_Length,
                               1, input, &idx, maxIdx);
            if (ret == 0) {
                /* Store references to serial number. */
                cert->sNum =
                        (char*)strDataASN[STRATTRASN_IDX_STR].data.ref.data;
                cert->sNumLen = (int)strDataASN[STRATTRASN_IDX_STR].
                    data.ref.length;
                /* Store serial number if small enough. */
                if (cert->sNumLen <= EXTERNAL_SERIAL_SIZE) {
                    XMEMCPY(cert->serial, cert->sNum, (size_t)cert->sNumLen);
                    cert->serialSz = cert->sNumLen;
                }
            }
            break;

        case UNSTRUCTURED_NAME_OID:
            /* Clear dynamic data and specify choices acceptable. */
            XMEMSET(strDataASN, 0, sizeof(strDataASN));
            GetASN_Choice(&strDataASN[STRATTRASN_IDX_STR], strAttrChoice);
            /* Parse a string. */
            ret = GetASN_Items(strAttrASN, strDataASN, strAttrASN_Length,
                               1, input, &idx, maxIdx);
            if (ret == 0) {
                /* Store references to unstructured name. */
                cert->unstructuredName =
                        (char*)strDataASN[STRATTRASN_IDX_STR].data.ref.data;
                cert->unstructuredNameLen = (int)strDataASN[STRATTRASN_IDX_STR].
                    data.ref.length;
            }
            break;

        /* Certificate extensions to be included in generated certificate.
         * PKCS#9: RFC 2985, 5.4.2 - Extension request
         */
        case EXTENSION_REQUEST_OID:
            /* Store references to all extensions. */
            cert->extensions    = input;
            cert->extensionsSz  = (int)maxIdx;
            cert->extensionsIdx = aIdx;

            /* Decode and validate extensions. */
            ret = DecodeCertExtensions(cert);
            if (ret == WC_NO_ERR_TRACE(ASN_CRIT_EXT_E)) {
                /* Return critical extension not recognized. */
                *criticalExt = ret;
                ret = 0;
            }
            else {
                /* No critical extension error. */
                *criticalExt = 0;
            }
            break;

        default:
            ret = ASN_PARSE_E;
            break;
    }

    return ret;
}

/* Decode attributes of a BER encoded certificate request.
 *
 * RFC 2986 - PKCS #10: Certification Request Syntax Specification Version 1.7
 *
 * Outer sequence has been removed.
 *
 * @param [in]  cert         Certificate request object.
 * @param [out] criticalExt  Critical extension return code.
 * @param [in]  idx          Index into certificate source to start parsing.
 * @param [in]  maxIdx       Maximum index to parse to.
 * @return  0 on success.
 * @return  ASN_CRIT_EXT_E when a critical extension was not recognized.
 * @return  ASN_PARSE_E when BER encoded data does not match ASN.1 items or
 *          is invalid.
 * @return  BUFFER_E when data in buffer is too small.
 * @return  ASN_OBJECT_ID_E when the expected OBJECT_ID tag is not found.
 * @return  ASN_EXPECT_0_E when the INTEGER has the MSB set or NULL has a
 *          non-zero length.
 * @return  ASN_UNKNOWN_OID_E when the OID cannot be verified.
 */
static int DecodeCertReqAttributes(DecodedCert* cert, int* criticalExt,
                                   word32 idx, word32 maxIdx)
{
    DECL_ASNGETDATA(dataASN, reqAttrASN_Length);
    int ret = 0;

    WOLFSSL_ENTER("DecodeCertReqAttributes");

    ALLOC_ASNGETDATA(dataASN, reqAttrASN_Length, ret, cert->heap);

    /* Parse each attribute until all data used up. */
    while ((ret == 0) && (idx < maxIdx)) {
        /* Clear dynamic data. */
        XMEMSET(dataASN, 0, sizeof(ASNGetData) * reqAttrASN_Length);
        GetASN_OID(&dataASN[REQATTRASN_IDX_TYPE], oidIgnoreType);

        /* Parse an attribute. */
        ret = GetASN_Items(reqAttrASN, dataASN, reqAttrASN_Length, 0,
                           cert->source, &idx, maxIdx);
        /* idx is now at end of attribute data. */
        if (ret == 0) {
            ret = DecodeCertReqAttrValue(cert, criticalExt,
                dataASN[REQATTRASN_IDX_TYPE].data.oid.sum,
                GetASNItem_DataIdx(dataASN[REQATTRASN_IDX_VALS], cert->source),
                dataASN[REQATTRASN_IDX_VALS].data.ref.data,
                dataASN[REQATTRASN_IDX_VALS].data.ref.length);
        }
    }

    FREE_ASNGETDATA(dataASN, cert->heap);
    return ret;
}

/* ASN.1 template for a certificate request.
 * PKCS#10: RFC 2986, 4.1 - CertificationRequestInfo
 * PKCS#10: RFC 2986, 4.2 - CertificationRequest
 */
static const ASNItem certReqASN[] = {
            /* CertificationRequest */
/* SEQ                              */ { 0, ASN_SEQUENCE, 1, 1, 0 },
                                                          /* CertificationRequestInfo */
/* INFO_SEQ                         */     { 1, ASN_SEQUENCE, 1, 1, 0 },
                                                              /* version              INTEGER { v1(0), v2(1), v3(2) */
/* INFO_VER                         */         { 2, ASN_INTEGER, 0, 0, 0 },
                                                              /* subject              Name */
/* INFO_SUBJ_SEQ                    */         { 2, ASN_SEQUENCE, 1, 0, 0 },
                                                              /* subjectPublicKeyInfo SubjectPublicKeyInfo */
/* INFO_SPUBKEYINFO_SEQ             */         { 2, ASN_SEQUENCE, 1, 1, 0 },
                                                                  /* algorithm          AlgorithmIdentifier */
/* INFO_SPUBKEYINFO_ALGOID_SEQ      */             { 3, ASN_SEQUENCE, 1, 1, 0 },
                                                                      /* Algorithm    OBJECT IDENTIFIER */
/* INFO_SPUBKEYINFO_ALGOID_OID      */                 { 4, ASN_OBJECT_ID, 0, 0, 0 },
                                                                      /* parameters   ANY defined by algorithm OPTIONAL */
/* INFO_SPUBKEYINFO_ALGOID_NULL     */                 { 4, ASN_TAG_NULL, 0, 0, 1 },
/* INFO_SPUBKEYINFO_ALGOID_CURVEID  */                 { 4, ASN_OBJECT_ID, 0, 0, 1 },
/* INFO_SPUBKEYINFO_ALGOID_PARAMS   */                 { 4, ASN_SEQUENCE, 1, 0, 1 },
                                                                  /* subjectPublicKey   BIT STRING */
/* INFO_SPUBKEYINFO_PUBKEY          */             { 3, ASN_BIT_STRING, 0, 0, 0 },
                                                              /* attributes       [0] Attributes */
/* INFO_ATTRS                       */         { 2, ASN_CONTEXT_SPECIFIC | 0, 1, 0, 1 },
                                                          /* signatureAlgorithm   AlgorithmIdentifier */
/* INFO_SIGALGO_SEQ                 */     { 1, ASN_SEQUENCE, 1, 1, 0 },
                                                              /* Algorithm    OBJECT IDENTIFIER */
/* INFO_SIGALGO_OID                 */         { 2, ASN_OBJECT_ID, 0, 0, 0 },
                                                              /* parameters   ANY defined by algorithm OPTIONAL */
/* INFO_SIGALGO_NULL                */         { 2, ASN_TAG_NULL, 0, 0, 1 },
                                                          /* signature            BIT STRING */
/* INFO_SIGNATURE                   */     { 1, ASN_BIT_STRING, 0, 0, 0 },
};
enum {
    CERTREQASN_IDX_SEQ = 0,
    CERTREQASN_IDX_INFO_SEQ,
    CERTREQASN_IDX_INFO_VER,
    CERTREQASN_IDX_INFO_SUBJ_SEQ,
    CERTREQASN_IDX_INFO_SPUBKEYINFO_SEQ,
    CERTREQASN_IDX_INFO_SPUBKEYINFO_ALGOID_SEQ,
    CERTREQASN_IDX_INFO_SPUBKEYINFO_ALGOID_OID,
    CERTREQASN_IDX_INFO_SPUBKEYINFO_ALGOID_NULL,
    CERTREQASN_IDX_INFO_SPUBKEYINFO_ALGOID_CURVEID,
    CERTREQASN_IDX_INFO_SPUBKEYINFO_ALGOID_PARAMS,
    CERTREQASN_IDX_INFO_SPUBKEYINFO_PUBKEY,
    CERTREQASN_IDX_INFO_ATTRS,
    CERTREQASN_IDX_INFO_SIGALGO_SEQ,
    CERTREQASN_IDX_INFO_SIGALGO_OID,
    CERTREQASN_IDX_INFO_SIGALGO_NULL,
    CERTREQASN_IDX_INFO_SIGNATURE
};

/* Number of items in ASN.1 template for a certificate request. */
#define certReqASN_Length (sizeof(certReqASN) / sizeof(ASNItem))

/* Parse BER encoded certificate request.
 *
 * RFC 2986 - PKCS #10: Certification Request Syntax Specification Version 1.7
 *
 * @param [in]  cert         Certificate request object.
 * @param [out] criticalExt  Critical extension return code.
 * @return  0 on success.
 * @return  ASN_CRIT_EXT_E when a critical extension was not recognized.
 * @return  ASN_PARSE_E when BER encoded data does not match ASN.1 items or
 *          is invalid.
 * @return  BUFFER_E when data in buffer is too small.
 * @return  ASN_OBJECT_ID_E when the expected OBJECT_ID tag is not found.
 * @return  ASN_EXPECT_0_E when the INTEGER has the MSB set or NULL has a
 *          non-zero length.
 * @return  ASN_UNKNOWN_OID_E when the OID cannot be verified.
 * @return  MEMORY_E on dynamic memory allocation failure.
 */
static int DecodeCertReq(DecodedCert* cert, int* criticalExt)
{
    DECL_ASNGETDATA(dataASN, certReqASN_Length);
    int ret = 0;
    byte version = 0;
    word32 idx;

    CALLOC_ASNGETDATA(dataASN, certReqASN_Length, ret, cert->heap);

    if (ret == 0) {
        /* Default version is 0. */
        version = 0;

        /* Set version var and OID types to expect. */
        GetASN_Int8Bit(&dataASN[CERTREQASN_IDX_INFO_VER], &version);
        GetASN_OID(&dataASN[CERTREQASN_IDX_INFO_SPUBKEYINFO_ALGOID_OID],
                oidKeyType);
        GetASN_OID(&dataASN[CERTREQASN_IDX_INFO_SPUBKEYINFO_ALGOID_CURVEID],
                oidCurveType);
        GetASN_OID(&dataASN[CERTREQASN_IDX_INFO_SIGALGO_OID], oidSigType);
        /* Parse a certificate request. */
        ret = GetASN_Items(certReqASN, dataASN, certReqASN_Length, 1,
                           cert->source, &cert->srcIdx, cert->maxIdx);
    }
    /* Check version is valid/supported - can't be negative. */
    if ((ret == 0) && (version > MAX_X509_VERSION)) {
        WOLFSSL_MSG("Unexpected certificate request version");
        ret = ASN_PARSE_E;
    }
    if (ret == 0) {
        /* Set fields of certificate request. */
        cert->version = version;
        cert->signatureOID =
              dataASN[CERTREQASN_IDX_INFO_SIGALGO_OID].data.oid.sum;
        cert->keyOID =
              dataASN[CERTREQASN_IDX_INFO_SPUBKEYINFO_ALGOID_OID].data.oid.sum;
        cert->certBegin = dataASN[CERTREQASN_IDX_INFO_SEQ].offset;

        /* Parse the subject name. */
        idx = dataASN[CERTREQASN_IDX_INFO_SUBJ_SEQ].offset;
        ret = GetCertName(cert, cert->subject, cert->subjectHash, ASN_SUBJECT,
                          cert->source, &idx,
                          dataASN[CERTREQASN_IDX_INFO_SPUBKEYINFO_SEQ].offset);
    }
    if (ret == 0) {
        /* Parse the certificate request Attributes. */
        ret = DecodeCertReqAttributes(cert, criticalExt,
                GetASNItem_DataIdx(dataASN[CERTREQASN_IDX_INFO_ATTRS],
                        cert->source),
                dataASN[CERTREQASN_IDX_INFO_SIGALGO_SEQ].offset);
    }
    if (ret == 0) {
        /* Parse the certificate request's key. */
        idx = dataASN[CERTREQASN_IDX_INFO_SPUBKEYINFO_SEQ].offset;
        ret = GetCertKey(cert, cert->source, &idx,
                dataASN[CERTREQASN_IDX_INFO_ATTRS].offset);
    }
    if (ret == 0) {
        /* Store references to signature. */
        cert->sigIndex = dataASN[CERTREQASN_IDX_INFO_SIGALGO_SEQ].offset;
        GetASN_GetConstRef(&dataASN[CERTREQASN_IDX_INFO_SIGNATURE],
                &cert->signature, &cert->sigLength);
    }

    FREE_ASNGETDATA(dataASN, cert->heap);
    return ret;
}

#endif /* WOLFSSL_CERT_REQ */

#endif /* WOLFSSL_ASN_TEMPLATE */

int ParseCert(DecodedCert* cert, int type, int verify, void* cm)
{
    int   ret;
#if (!defined(WOLFSSL_NO_MALLOC) && !defined(NO_WOLFSSL_CM_VERIFY)) || \
    defined(WOLFSSL_DYN_CERT)
    char* ptr;
#endif

    ret = ParseCertRelative(cert, type, verify, cm, NULL);
    if (ret < 0)
        return ret;

#if (!defined(WOLFSSL_NO_MALLOC) && !defined(NO_WOLFSSL_CM_VERIFY)) || \
    defined(WOLFSSL_DYN_CERT)
    /* cert->subjectCN not stored as copy of WOLFSSL_NO_MALLOC defined */
    if (cert->subjectCNLen > 0) {
        ptr = (char*)XMALLOC((size_t)cert->subjectCNLen + 1, cert->heap,
                              DYNAMIC_TYPE_SUBJECT_CN);
        if (ptr == NULL)
            return MEMORY_E;
        XMEMCPY(ptr, cert->subjectCN, (size_t)cert->subjectCNLen);
        ptr[cert->subjectCNLen] = '\0';
        cert->subjectCN = ptr;
        cert->subjectCNStored = 1;
    }
#endif

#if (!defined(WOLFSSL_NO_MALLOC) && !defined(NO_WOLFSSL_CM_VERIFY)) || \
    defined(WOLFSSL_DYN_CERT)
    /* cert->publicKey not stored as copy if WOLFSSL_NO_MALLOC defined */
    if ((cert->keyOID == RSAk
    #ifdef WC_RSA_PSS
         || cert->keyOID == RSAPSSk
    #endif
         ) && cert->publicKey != NULL && cert->pubKeySize > 0) {
        ptr = (char*)XMALLOC(cert->pubKeySize, cert->heap,
                              DYNAMIC_TYPE_PUBLIC_KEY);
        if (ptr == NULL)
            return MEMORY_E;
        XMEMCPY(ptr, cert->publicKey, cert->pubKeySize);
        cert->publicKey = (byte *)ptr;
        cert->pubKeyStored = 1;
    }
#endif

    return ret;
}

int wc_ParseCert(DecodedCert* cert, int type, int verify, void* cm)
{
    return ParseCert(cert, type, verify, cm);
}

#ifdef WOLFCRYPT_ONLY

/* dummy functions, not using wolfSSL so don't need actual ones */
Signer* GetCA(void* signers, byte* hash);
Signer* GetCA(void* signers, byte* hash)
{
    (void)hash;

    return (Signer*)signers;
}

#ifndef NO_SKID
Signer* GetCAByName(void* signers, byte* hash);
Signer* GetCAByName(void* signers, byte* hash)
{
    (void)hash;

    return (Signer*)signers;
}
#endif /* NO_SKID */

#ifdef WOLFSSL_AKID_NAME
Signer* GetCAByAKID(void* vp, const byte* issuer, word32 issuerSz,
        const byte* serial, word32 serialSz);
Signer* GetCAByAKID(void* vp, const byte* issuer, word32 issuerSz,
        const byte* serial, word32 serialSz)
{
    (void)issuer;
    (void)issuerSz;
    (void)serial;
    (void)serialSz;

    return (Signer*)vp;
}
#endif

#endif /* WOLFCRYPT_ONLY */

#if defined(WOLFSSL_NO_TRUSTED_CERTS_VERIFY) && !defined(NO_SKID)
static Signer* GetCABySubjectAndPubKey(DecodedCert* cert, void* cm)
{
    Signer* ca = NULL;
    if (cert->extSubjKeyIdSet)
        ca = GetCA(cm, cert->extSubjKeyId);
    if (ca == NULL)
        ca = GetCAByName(cm, cert->subjectHash);
    if (ca) {
        if ((ca->pubKeySize == cert->pubKeySize) &&
               (XMEMCMP(ca->publicKey, cert->publicKey, ca->pubKeySize) == 0)) {
            return ca;
        }
    }
    return NULL;
}
#endif

#if defined(WOLFSSL_SMALL_CERT_VERIFY) || defined(OPENSSL_EXTRA)
#ifdef WOLFSSL_ASN_TEMPLATE
/* Get the Hash of the Authority Key Identifier from the list of extensions.
 *
 * @param [in]  input   Input data.
 * @param [in]  maxIdx  Maximum index for data.
 * @param [in]  sigOID  Signature OID for determining hash algorithm.
 * @param [out] hash    Hash of AKI.
 * @param [out] set     Whether the hash buffer was set.
 * @param [in]  heap    Dynamic memory allocation hint.
 * @return  0 on success.
 * @return  ASN_PARSE_E when BER encoded data does not match ASN.1 items or
 *          is invalid.
 * @return  MEMORY_E on dynamic memory allocation failure.
 */
static int GetAKIHash(const byte* input, word32 maxIdx, word32 sigOID,
                      byte* hash, int* set, void* heap)
{
    /* AKI and Certificate Extension ASN.1 templates are the same length. */
    DECL_ASNGETDATA(dataASN, certExtASN_Length);
    int ret = 0;
    word32 idx = 0;
    word32 extEndIdx;
    byte* extData;
    word32 extDataSz;
    byte critical;

    ALLOC_ASNGETDATA(dataASN, certExtASN_Length, ret, heap);
    (void)heap;

    extEndIdx = idx + maxIdx;

    /* Step through each extension looking for AKI. */
    while ((ret == 0) && (idx < extEndIdx)) {
        /* Clear dynamic data and check for certificate extension type OIDs. */
        XMEMSET(dataASN, 0, sizeof(*dataASN) * certExtASN_Length);
        GetASN_OID(&dataASN[CERTEXTASN_IDX_OID], oidCertExtType);
        /* Set criticality variable. */
        GetASN_Int8Bit(&dataASN[CERTEXTASN_IDX_CRIT], &critical);
        /* Parse an extension. */
        ret = GetASN_Items(certExtASN, dataASN, certExtASN_Length, 0, input,
                &idx, extEndIdx);
        if (ret == 0) {
            /* Get reference to extension data and move index on past this
             * extension. */
            GetASN_GetRef(&dataASN[CERTEXTASN_IDX_VAL], &extData, &extDataSz);
            idx += extDataSz;

            /* Check whether we have the AKI extension. */
            if (dataASN[CERTEXTASN_IDX_OID].data.oid.sum == AUTH_KEY_OID) {
                /* Clear dynamic data. */
                XMEMSET(dataASN, 0, sizeof(*dataASN) * authKeyIdASN_Length);
                /* Start parsing extension data from the start. */
                idx = 0;
                /* Parse AKI extension data. */
                ret = GetASN_Items(authKeyIdASN, dataASN, authKeyIdASN_Length,
                        1, extData, &idx, extDataSz);
                if ((ret == 0) &&
                        (dataASN[AUTHKEYIDASN_IDX_KEYID].data.ref.data
                                != NULL)) {
                    /* We parsed successfully and have data. */
                    *set = 1;
                    /* Get the hash or hash of the hash if wrong size. */
                    ret = GetHashId(
                        dataASN[AUTHKEYIDASN_IDX_KEYID].data.ref.data,
                        (int)dataASN[AUTHKEYIDASN_IDX_KEYID].data.ref.length,
                        hash, HashIdAlg(sigOID));
                }
                break;
            }
        }
    }

    FREE_ASNGETDATA(dataASN, heap);
    return ret;
}
#endif

/* Only quick step through the certificate to find fields that are then used
 * in certificate signature verification.
 * Must use the signature OID from the signed part of the certificate.
 * Works also on certificate signing requests.
 *
 * This is only for minimizing dynamic memory usage during TLS certificate
 * chain processing.
 * Doesn't support:
 *   OCSP Only: alt lookup using subject and pub key w/o sig check
 */
static int CheckCertSignature_ex(const byte* cert, word32 certSz, void* heap,
        void* cm, const byte* pubKey, word32 pubKeySz, int pubKeyOID, int req)
{
#ifndef WOLFSSL_ASN_TEMPLATE
#if !defined(WOLFSSL_SMALL_STACK) || defined(WOLFSSL_NO_MALLOC)
    SignatureCtx  sigCtx[1];
#else
    SignatureCtx* sigCtx;
#endif
    byte          hash[KEYID_SIZE];
    Signer*       ca = NULL;
    word32        idx = 0;
    int           len;
    word32        tbsCertIdx = 0;
    word32        sigIndex   = 0;
    word32        signatureOID = 0;
    word32        oid = 0;
    word32        issuerIdx = 0;
    word32        issuerSz  = 0;
#ifndef NO_SKID
    int           extLen = 0;
    word32        extIdx = 0;
    word32        extEndIdx = 0;
    int           extAuthKeyIdSet = 0;
#endif
    int           ret = 0;
    word32        localIdx;
    byte          tag;
    const byte*   sigParams = NULL;
    word32        sigParamsSz = 0;


    if (cert == NULL) {
        return BAD_FUNC_ARG;
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    sigCtx = (SignatureCtx*)XMALLOC(sizeof(*sigCtx), heap, DYNAMIC_TYPE_SIGNATURE);
    if (sigCtx == NULL)
        return MEMORY_E;
#endif

    InitSignatureCtx(sigCtx, heap, INVALID_DEVID);

    /* Certificate SEQUENCE */
    if (GetSequence(cert, &idx, &len, certSz) < 0)
        ret = ASN_PARSE_E;
    if (ret == 0) {
        tbsCertIdx = idx;

        /* TBSCertificate SEQUENCE */
        if (GetSequence(cert, &idx, &len, certSz) < 0)
            ret = ASN_PARSE_E;
    }
    if (ret == 0) {
        sigIndex = len + idx;

        if ((idx + 1) > certSz)
            ret = BUFFER_E;
    }
    if (ret == 0) {
        /* version - optional */
        localIdx = idx;
        if (GetASNTag(cert, &localIdx, &tag, certSz) == 0) {
            if (tag == (ASN_CONTEXT_SPECIFIC | ASN_CONSTRUCTED)) {
                idx++;
                if (GetLength(cert, &idx, &len, certSz) < 0)
                    ret = ASN_PARSE_E;
                idx += len;
            }
        }
    }

    if (ret == 0) {
        /* serialNumber */
        if (GetASNHeader(cert, ASN_INTEGER, &idx, &len, certSz) < 0)
            ret = ASN_PARSE_E;
    }
    if (ret == 0) {
        idx += len;

        /* signature */
        if (!req) {
            if (GetAlgoId(cert, &idx, &signatureOID, oidSigType, certSz) < 0)
                ret = ASN_PARSE_E;
        #ifdef WC_RSA_PSS
            else if (signatureOID == CTC_RSASSAPSS) {
                int start = idx;
                sigParams = cert + idx;
                if (GetSequence(cert, &idx, &len, certSz) < 0)
                    ret = ASN_PARSE_E;
                if (ret == 0) {
                    idx += len;
                    sigParamsSz = idx - start;
                }
            }
        #endif
        }
    }

    if (ret == 0) {
        issuerIdx = idx;
        /* issuer for cert or subject for csr */
        if (GetSequence(cert, &idx, &len, certSz) < 0)
            ret = ASN_PARSE_E;
    }
    if (ret == 0) {
        issuerSz = len + idx - issuerIdx;
    }
#ifndef NO_SKID
    if (!req && ret == 0) {
        idx += len;

        /* validity */
        if (GetSequence(cert, &idx, &len, certSz) < 0)
            ret = ASN_PARSE_E;
    }
    if (!req && ret == 0) {
        idx += len;

        /* subject */
        if (GetSequence(cert, &idx, &len, certSz) < 0)
            ret = ASN_PARSE_E;
    }
    if (ret == 0) {
        idx += len;

        /* subjectPublicKeyInfo */
        if (GetSequence(cert, &idx, &len, certSz) < 0)
            ret = ASN_PARSE_E;
    }
    if (req && ret == 0) {
        idx += len;

        /* attributes */
        if (GetASNHeader_ex(cert,
                ASN_CONTEXT_SPECIFIC | ASN_CONSTRUCTED, &idx,
                &len, certSz, 1) < 0)
            ret = ASN_PARSE_E;
    }
    if (!req) {
        if (ret == 0) {
            idx += len;

            if ((idx + 1) > certSz)
                ret = BUFFER_E;
        }
        if (ret == 0) {
            /* issuerUniqueID - optional */
            localIdx = idx;
            if (GetASNTag(cert, &localIdx, &tag, certSz) == 0) {
                if (tag == (ASN_CONTEXT_SPECIFIC | ASN_CONSTRUCTED | 1)) {
                    idx++;
                    if (GetLength(cert, &idx, &len, certSz) < 0)
                        ret = ASN_PARSE_E;
                    idx += len;
                }
            }
        }
        if (ret == 0) {
            if ((idx + 1) > certSz)
                ret = BUFFER_E;
        }
        if (ret == 0) {
            /* subjectUniqueID - optional */
            localIdx = idx;
            if (GetASNTag(cert, &localIdx, &tag, certSz) == 0) {
                if (tag == (ASN_CONTEXT_SPECIFIC | ASN_CONSTRUCTED | 2)) {
                    idx++;
                    if (GetLength(cert, &idx, &len, certSz) < 0)
                        ret = ASN_PARSE_E;
                    idx += len;
                }
            }
        }

        if (ret == 0) {
            if ((idx + 1) > certSz)
                ret = BUFFER_E;
        }
        /* extensions - optional */
        localIdx = idx;
        if (ret == 0 && GetASNTag(cert, &localIdx, &tag, certSz) == 0 &&
                tag == (ASN_CONTEXT_SPECIFIC | ASN_CONSTRUCTED | 3)) {
            idx++;
            if (GetLength(cert, &idx, &extLen, certSz) < 0)
                ret = ASN_PARSE_E;
            if (ret == 0) {
                if (GetSequence(cert, &idx, &extLen, certSz) < 0)
                    ret = ASN_PARSE_E;
            }
            if (ret == 0) {
                extEndIdx = idx + extLen;

                /* Check each extension for the ones we want. */
                while (ret == 0 && idx < extEndIdx) {
                    if (GetSequence(cert, &idx, &len, certSz) < 0)
                        ret = ASN_PARSE_E;
                    if (ret == 0) {
                        extIdx = idx;
                        if (GetObjectId(cert, &extIdx, &oid, oidCertExtType,
                                                                  certSz) < 0) {
                            ret = ASN_PARSE_E;
                        }

                        if (ret == 0) {
                            if ((extIdx + 1) > certSz)
                                ret = BUFFER_E;
                        }
                    }

                    if (ret == 0) {
                        localIdx = extIdx;
                        if (GetASNTag(cert, &localIdx, &tag, certSz) == 0 &&
                                tag == ASN_BOOLEAN) {
                            if (GetBoolean(cert, &extIdx, certSz) < 0)
                                ret = ASN_PARSE_E;
                        }
                    }
                    if (ret == 0) {
                        if (GetOctetString(cert, &extIdx, &extLen, certSz) < 0)
                            ret = ASN_PARSE_E;
                    }

                    if (ret == 0) {
                        switch (oid) {
                        case AUTH_KEY_OID:
                            if (GetSequence(cert, &extIdx, &extLen, certSz) < 0)
                                ret = ASN_PARSE_E;

                            if (ret == 0 && (extIdx + 1) >= certSz)
                                ret = BUFFER_E;

                            if (ret == 0 &&
                                    GetASNTag(cert, &extIdx, &tag, certSz) == 0 &&
                                    tag == (ASN_CONTEXT_SPECIFIC | 0)) {
                                if (GetLength(cert, &extIdx, &extLen, certSz) <= 0)
                                    ret = ASN_PARSE_E;
                                if (ret == 0) {
                                    extAuthKeyIdSet = 1;
                                    /* Get the hash or hash of the hash if wrong
                                     * size. */
                                    ret = GetHashId(cert + extIdx, extLen,
                                        hash, HashIdAlg(signatureOID));
                                }
                            }
                            break;

                        default:
                            break;
                        }
                    }
                    idx += len;
                }
            }
        }
    }
    else if (ret == 0) {
        idx += len;
    }

    if (ret == 0 && pubKey == NULL) {
        if (extAuthKeyIdSet)
            ca = GetCA(cm, hash);
        if (ca == NULL) {
            ret = CalcHashId_ex(cert + issuerIdx, issuerSz, hash,
                HashIdAlg(signatureOID));
            if (ret == 0)
                ca = GetCAByName(cm, hash);
        }
    }
#else
    if (ret == 0 && pubKey == NULL) {
        ret = CalcHashId_ex(cert + issuerIdx, issuerSz, hash,
            HashIdAlg(signatureOID));
        if (ret == 0)
            ca = GetCA(cm, hash);
    }
#endif /* !NO_SKID */
    if (ca == NULL && pubKey == NULL)
        ret = ASN_NO_SIGNER_E;

    if (ret == 0) {
        idx = sigIndex;
        /* signatureAlgorithm */
        if (GetAlgoId(cert, &idx, &oid, oidSigType, certSz) < 0)
            ret = ASN_PARSE_E;
    #ifdef WC_RSA_PSS
        else if (signatureOID == CTC_RSASSAPSS) {
            word32 sz = idx;
            const byte* params = cert + idx;
            if (GetSequence(cert, &idx, &len, certSz) < 0)
                ret = ASN_PARSE_E;
            if (ret == 0) {
                idx += len;
                sz = idx - sz;

                if (req) {
                    if ((sz != sigParamsSz) ||
                                        (XMEMCMP(sigParams, params, sz) != 0)) {
                        ret = ASN_PARSE_E;
                    }
                }
                else {
                    sigParams = params;
                    sigParamsSz = sz;
                }
            }
        }
    #endif
        /* In CSR signature data is not present in body */
        if (req)
            signatureOID = oid;
    }
    if (ret == 0) {
        if (oid != signatureOID)
            ret = ASN_SIG_OID_E;
    }
    if (ret == 0) {
        /* signatureValue */
        if (CheckBitString(cert, &idx, &len, certSz, 1, NULL) < 0)
            ret = ASN_PARSE_E;
    }

    if (ret == 0) {
        if (pubKey != NULL) {
            ret = ConfirmSignature(sigCtx, cert + tbsCertIdx,
                sigIndex - tbsCertIdx, pubKey, pubKeySz, pubKeyOID,
                cert + idx, len, signatureOID, sigParams, sigParamsSz, NULL);
        }
        else {
            ret = ConfirmSignature(sigCtx, cert + tbsCertIdx,
                sigIndex - tbsCertIdx, ca->publicKey, ca->pubKeySize,
                ca->keyOID, cert + idx, len, signatureOID, sigParams,
                sigParamsSz, NULL);
        }
        if (ret != 0) {
            WOLFSSL_ERROR_VERBOSE(ret);
            WOLFSSL_MSG("Confirm signature failed");
        }
    }

    FreeSignatureCtx(sigCtx);
    WC_FREE_VAR_EX(sigCtx, heap, DYNAMIC_TYPE_SIGNATURE);
    return ret;
#else /* WOLFSSL_ASN_TEMPLATE */
    /* X509 ASN.1 template longer than Certificate Request template. */
    DECL_ASNGETDATA(dataASN, x509CertASN_Length);
    WC_DECLARE_VAR(sigCtx, SignatureCtx, 1, 0);
    byte hash[KEYID_SIZE];
    Signer* ca = NULL;
    int ret = 0;
    word32 idx = 0;
#ifndef NO_SKID
    int extAuthKeyIdSet = 0;
#endif
    const byte* tbs = NULL;
    word32 tbsSz = 0;
#ifdef WC_RSA_PSS
    const byte* tbsParams = NULL;
    word32 tbsParamsSz = 0;
#endif
    const byte* sig = NULL;
    word32 sigSz = 0;
    word32 sigOID = 0;
    const byte* sigParams = NULL;
    word32 sigParamsSz = 0;
    const byte* caName = NULL;
    word32 caNameLen = 0;
#ifndef NO_SKID
    const byte* akiData = NULL;
    word32 akiLen = 0;
#endif

    (void)req;
    (void)heap;

    if (cert == NULL) {
        ret = BAD_FUNC_ARG;
    }

    ALLOC_ASNGETDATA(dataASN, x509CertASN_Length, ret, heap);

    if ((ret == 0) && (!req)) {
        /* Clear dynamic data for certificate items. */
        XMEMSET(dataASN, 0, sizeof(ASNGetData) * x509CertASN_Length);
        /* Set OID types expected for signature and public key. */
        GetASN_OID(&dataASN[X509CERTASN_IDX_TBS_ALGOID_OID], oidSigType);
        GetASN_OID(&dataASN[X509CERTASN_IDX_TBS_SPUBKEYINFO_ALGO_OID],
                oidKeyType);
        GetASN_OID(&dataASN[X509CERTASN_IDX_TBS_SPUBKEYINFO_ALGO_CURVEID],
                oidCurveType);
        GetASN_OID(&dataASN[X509CERTASN_IDX_SIGALGO_OID], oidSigType);
        /* Parse certificate. */
        ret = GetASN_Items(x509CertASN, dataASN, x509CertASN_Length, 1, cert,
                           &idx, certSz);

        /* Check signature OIDs match. */
        if ((ret == 0) && dataASN[X509CERTASN_IDX_TBS_ALGOID_OID].data.oid.sum
                != dataASN[X509CERTASN_IDX_SIGALGO_OID].data.oid.sum) {
            ret = ASN_SIG_OID_E;
        }
        /* Store the data for verification in the certificate. */
        if (ret == 0) {
            tbs = GetASNItem_Addr(dataASN[X509CERTASN_IDX_TBS_SEQ], cert);
            tbsSz = GetASNItem_Length(dataASN[X509CERTASN_IDX_TBS_SEQ], cert);
            caName = GetASNItem_Addr(dataASN[X509CERTASN_IDX_TBS_ISSUER_SEQ],
                    cert);
            caNameLen = GetASNItem_Length(dataASN[X509CERTASN_IDX_TBS_ISSUER_SEQ],
                    cert);
            sigOID = dataASN[X509CERTASN_IDX_SIGALGO_OID].data.oid.sum;
        #ifdef WC_RSA_PSS
            if (dataASN[X509CERTASN_IDX_TBS_ALGOID_PARAMS].tag != 0) {
                tbsParams =
                    GetASNItem_Addr(dataASN[X509CERTASN_IDX_TBS_ALGOID_PARAMS],
                        cert);
                tbsParamsSz =
                    GetASNItem_Length(dataASN[X509CERTASN_IDX_TBS_ALGOID_PARAMS],
                        cert);
            }
            if (dataASN[X509CERTASN_IDX_SIGALGO_PARAMS].tag != 0) {
                sigParams =
                    GetASNItem_Addr(dataASN[X509CERTASN_IDX_SIGALGO_PARAMS],
                        cert);
                sigParamsSz =
                    GetASNItem_Length(dataASN[X509CERTASN_IDX_SIGALGO_PARAMS],
                        cert);
            }
        #endif
            GetASN_GetConstRef(&dataASN[X509CERTASN_IDX_SIGNATURE], &sig, &sigSz);
        #ifdef WC_RSA_PSS
            if (tbsParamsSz != sigParamsSz) {
                ret = ASN_PARSE_E;
            }
            else if ((tbsParamsSz > 0) && (sigOID != CTC_RSASSAPSS)) {
                ret = ASN_PARSE_E;
            }
            else if ((tbsParamsSz > 0) &&
                     (XMEMCMP(tbsParams, sigParams, tbsParamsSz) != 0)) {
                ret = ASN_PARSE_E;
            }
        #endif
        }
    }
    else if (ret == 0) {
#ifndef WOLFSSL_CERT_REQ
        ret = NOT_COMPILED_IN;
#else
        /* Clear dynamic data for certificate request items. */
        XMEMSET(dataASN, 0, sizeof(ASNGetData) * certReqASN_Length);
        /* Set OID types expected for signature and public key. */
        GetASN_OID(&dataASN[CERTREQASN_IDX_INFO_SPUBKEYINFO_ALGOID_OID],
                oidKeyType);
        GetASN_OID(&dataASN[CERTREQASN_IDX_INFO_SPUBKEYINFO_ALGOID_CURVEID],
                oidCurveType);
        GetASN_OID(&dataASN[CERTREQASN_IDX_INFO_SIGALGO_OID], oidSigType);
        /* Parse certificate request. */
        ret = GetASN_Items(certReqASN, dataASN, certReqASN_Length, 1, cert,
                           &idx, certSz);
        if (ret == 0) {
            /* Store the data for verification in the certificate. */
            tbs = GetASNItem_Addr(dataASN[CERTREQASN_IDX_INFO_SEQ], cert);
            tbsSz = GetASNItem_Length(dataASN[CERTREQASN_IDX_INFO_SEQ], cert);
            caName = GetASNItem_Addr(
                    dataASN[CERTREQASN_IDX_INFO_SUBJ_SEQ], cert);
            caNameLen = GetASNItem_Length(
                    dataASN[CERTREQASN_IDX_INFO_SUBJ_SEQ], cert);
            sigOID = dataASN[CERTREQASN_IDX_INFO_SIGALGO_OID].data.oid.sum;
        #ifdef WC_RSA_PSS
            sigParams = GetASNItem_Addr(dataASN[X509CERTASN_IDX_SIGALGO_PARAMS],
                cert);
            sigParamsSz =
                GetASNItem_Length(dataASN[X509CERTASN_IDX_SIGALGO_PARAMS],
                    cert);
        #endif
            GetASN_GetConstRef(&dataASN[CERTREQASN_IDX_INFO_SIGNATURE], &sig,
                    &sigSz);
        }
#endif
    }

#ifndef NO_SKID
    if ((ret == 0) && (pubKey == NULL) && !req) {
        akiData = dataASN[X509CERTASN_IDX_TBS_EXT_SEQ].data.ref.data;
        akiLen = dataASN[X509CERTASN_IDX_TBS_EXT_SEQ].data.ref.length;
    }
#endif

    FREE_ASNGETDATA(dataASN, heap);

    /* If no public passed, then find the CA. */
    if ((ret == 0) && (pubKey == NULL)) {
#ifndef NO_SKID
        /* Find the AKI extension in list of extensions and get hash. */
        if ((!req) && (akiData != NULL)) {
            /* TODO: test case */
            ret = GetAKIHash(akiData, akiLen, sigOID, hash, &extAuthKeyIdSet,
                heap);
        }

        /* Get the CA by hash one was found. */
        if (extAuthKeyIdSet) {
            ca = GetCA(cm, hash);
        }
        if (ca == NULL)
#endif
        {
            /* Try hash of issuer name. */
            ret = CalcHashId_ex(caName, caNameLen, hash, HashIdAlg(sigOID));
            if (ret == 0) {
                ca = GetCAByName(cm, hash);
            }
        }

        if (ca != NULL) {
            /* Extract public key information. */
            pubKey = ca->publicKey;
            pubKeySz = ca->pubKeySize;
            pubKeyOID = (int)ca->keyOID;
        }
        else {
            /* No public key to verify with. */
            ret = ASN_NO_SIGNER_E;
        }
    }

    if (ret == 0) {
        WC_ALLOC_VAR_EX(sigCtx, SignatureCtx, 1, heap, DYNAMIC_TYPE_SIGNATURE,
            ret=MEMORY_E);
        if (WC_VAR_OK(sigCtx))
        {
            InitSignatureCtx(sigCtx, heap, INVALID_DEVID);

            /* Check signature. */
            ret = ConfirmSignature(sigCtx, tbs, tbsSz, pubKey, pubKeySz,
                (word32)pubKeyOID, sig, sigSz, sigOID, sigParams, sigParamsSz,
                NULL);
            if (ret != 0) {
                WOLFSSL_MSG("Confirm signature failed");
            }

            FreeSignatureCtx(sigCtx);
            WC_FREE_VAR_EX(sigCtx, heap, DYNAMIC_TYPE_SIGNATURE);
        }
    }

    return ret;
#endif /* WOLFSSL_ASN_TEMPLATE */
}

/* Call CheckCertSignature_ex using a public key buffer for verification */
int CheckCertSignaturePubKey(const byte* cert, word32 certSz, void* heap,
        const byte* pubKey, word32 pubKeySz, int pubKeyOID)
{
    return CheckCertSignature_ex(cert, certSz, heap, NULL,
            pubKey, pubKeySz, pubKeyOID, 0);
}

/* Call CheckCertSignature_ex using a public key and oid */
int wc_CheckCertSigPubKey(const byte* cert, word32 certSz, void* heap,
        const byte* pubKey, word32 pubKeySz, int pubKeyOID)
{
        return CheckCertSignaturePubKey(cert, certSz, heap, pubKey, pubKeySz,
                                        pubKeyOID);
}

#ifdef WOLFSSL_CERT_REQ
int CheckCSRSignaturePubKey(const byte* cert, word32 certSz, void* heap,
        const byte* pubKey, word32 pubKeySz, int pubKeyOID)
{
    return CheckCertSignature_ex(cert, certSz, heap, NULL,
            pubKey, pubKeySz, pubKeyOID, 1);
}
#endif /* WOLFSSL_CERT_REQ */

/* Call CheckCertSignature_ex using a certificate manager (cm) */
int wc_CheckCertSignature(const byte* cert, word32 certSz, void* heap, void* cm)
{
    return CheckCertSignature_ex(cert, certSz, heap, cm, NULL, 0, 0, 0);
}
#endif /* WOLFSSL_SMALL_CERT_VERIFY || OPENSSL_EXTRA */

#if (defined(HAVE_ED25519) && defined(HAVE_ED25519_KEY_IMPORT) || \
    (defined(HAVE_ED448) && defined(HAVE_ED448_KEY_IMPORT)))
/* ASN.1 DER decode instruction. */
typedef struct DecodeInstr {
    /* Tag expected. */
    byte tag;
    /* Operation to perform: step in or go over */
    WC_BITFIELD op:1;
    /* ASN.1 item is optional. */
    WC_BITFIELD optional:1;
} DecodeInstr;

/* Step into ASN.1 item. */
#define DECODE_INSTR_IN    0
/* Step over ASN.1 item. */
#define DECODE_INSTR_OVER  1

/* Get the public key data from the DER encoded X.509 certificate.
 *
 * Assumes data has previously been parsed for complete validity.
 *
 * @param [in]  cert      DER encoded X.509 certificate data.
 * @param [in]  certSz    Length of DER encoding.
 * @param [out] pubKey    Public key data. (From the BIT_STRING.)
 * @param [out] pubKeySz  Length of public key data in bytes.
 * @return  0 on success.
 * @return  BAD_FUNC_ARG when cert, pubKey or pubKeySz is NULL.
 * @return  ASN_PARSE_E when certificate encoding is invalid.
 */
int wc_CertGetPubKey(const byte* cert, word32 certSz,
    const unsigned char** pubKey, word32* pubKeySz)
{
    int ret = 0;
    int l = 0;
    word32 o = 0;
    int i;
    static DecodeInstr ops[] = {
        /* Outer SEQ */
        { ASN_SEQUENCE | ASN_CONSTRUCTED, DECODE_INSTR_IN  , 0 },
        /* TBSCertificate: SEQ */
        { ASN_SEQUENCE | ASN_CONSTRUCTED, DECODE_INSTR_IN  , 0 },
        /* Version: [0] */
        { ASN_CONTEXT_SPECIFIC | ASN_CONSTRUCTED | ASN_X509_CERT_VERSION,
          DECODE_INSTR_OVER, 1 },
        /* CertificateSerialNumber: INT  */
        { ASN_INTEGER,                    DECODE_INSTR_OVER, 0 },
        /* AlgorithmIdentifier: SEQ */
        { ASN_SEQUENCE | ASN_CONSTRUCTED, DECODE_INSTR_OVER, 0 },
        /* issuer: SEQ */
        { ASN_SEQUENCE | ASN_CONSTRUCTED, DECODE_INSTR_OVER, 0 },
        /* Validity: SEQ */
        { ASN_SEQUENCE | ASN_CONSTRUCTED, DECODE_INSTR_OVER, 0 },
        /* subject: SEQ */
        { ASN_SEQUENCE | ASN_CONSTRUCTED, DECODE_INSTR_OVER, 0 },
        /* subjectPublicKeyInfo SEQ */
        { ASN_SEQUENCE | ASN_CONSTRUCTED, DECODE_INSTR_IN  , 0 },
        /* AlgorithmIdentifier: SEQ */
        { ASN_SEQUENCE | ASN_CONSTRUCTED, DECODE_INSTR_OVER, 0 },
        /* PublicKey: BIT_STRING  */
        { ASN_BIT_STRING,                 DECODE_INSTR_IN  , 0 },
    };

    /* Validate parameters. */
    if ((cert == NULL) || (pubKey == NULL) || (pubKeySz == NULL)) {
        ret = BAD_FUNC_ARG;
    }

    /* Process each instruction to take us to public key data. */
    for (i = 0; (ret == 0) && (i < (int)(sizeof(ops) / sizeof(*ops))); i++) {
        DecodeInstr op = ops[i];

        /* Check the current ASN.1 item has the expected tag. */
        if (cert[o] != op.tag) {
            /* If not optional then error, otherwise skip op. */
            if (!op.optional) {
                ret = ASN_PARSE_E;
            }
        }
        else {
            /* Move past tag. */
            o++;
            /* Get the length of ASN.1 item and move past length encoding. */
            if (GetLength(cert, &o, &l, certSz) < 0) {
                ret = ASN_PARSE_E;
            }
            /* Skip data if required. */
            else if (op.op == DECODE_INSTR_OVER) {
                o += (word32)l;
            }
        }
    }

    if (ret == 0) {
        /* Return the public key data and length.
         * Skip first byte of BIT_STRING data: unused bits. */
        *pubKey = cert + o + 1;
        *pubKeySz = (word32)(l - 1);
    }

    return ret;
}
#endif

/*
 * @brief Export the SubjectPublicKeyInfo from an X.509 certificate
 *
 * This function extracts the SubjectPublicKeyInfo (SPKI) section from an X.509
 * certificate in DER format. The SPKI contains the public key algorithm and
 * the public key itself.
 *
 * @param certDer      [in]  Pointer to the DER encoded certificate
 * @param certSz       [in]  Size of the DER encoded certificate
 * @param pubKeyDer    [out] Buffer to hold the extracted SPKI (can be NULL to
 *                           get size)
 * @param pubKeyDerSz  [in,out] On input, size of pubKeyDer buffer
 *                              On output, actual size of the SPKI
 *
 * @return 0 on success, negative on error
 * @return BAD_FUNC_ARG if certDer is NULL, certSz is 0, or pubKeyDerSz is NULL
 * @return BUFFER_E if the provided buffer is too small
 */
int wc_GetSubjectPubKeyInfoDerFromCert(const byte* certDer,
                                                   word32 certDerSz,
                                                   byte* pubKeyDer,
                                                   word32* pubKeyDerSz)
{
    WC_DECLARE_VAR(cert, DecodedCert, 1, 0);
    int         ret;
    word32      startIdx;
    word32      idx;
    word32      length;
    int         badDate;

    if (certDer == NULL || certDerSz == 0 || pubKeyDerSz == NULL) {
        return BAD_FUNC_ARG;
    }

    WC_ALLOC_VAR_EX(cert, DecodedCert, 1, NULL, DYNAMIC_TYPE_TMP_BUFFER,
        return MEMORY_E);

    length = 0;
    badDate = 0;

    wc_InitDecodedCert(cert, certDer, certDerSz, NULL);

    /* Parse up to the SubjectPublicKeyInfo */
    ret = wc_GetPubX509(cert, 0, &badDate);
    if (ret >= 0) {
        /* Save the starting index of SubjectPublicKeyInfo */
        startIdx = cert->srcIdx;

        /* Get the length of the SubjectPublicKeyInfo sequence */
        idx = startIdx;
        ret = GetSequence(certDer, &idx, (int*)&length, certDerSz);
        if (ret >= 0) {
            /* Calculate total length including sequence header */
            length += (idx - startIdx);

            /* Copy the SubjectPublicKeyInfo if buffer provided */
            if (pubKeyDer != NULL) {
                if (*pubKeyDerSz < (word32)length) {
                    ret = BUFFER_E;
                }
                else {
                    XMEMCPY(pubKeyDer, &certDer[startIdx], length);
                }
            }
        }
    }

    if (ret >= 0) {
        ret = 0;
    }

    *pubKeyDerSz = length;
    wc_FreeDecodedCert(cert);

    WC_FREE_VAR_EX(cert, NULL, DYNAMIC_TYPE_TMP_BUFFER);

    return ret;
}


#ifdef HAVE_OCSP
Signer* findSignerByKeyHash(Signer *list, byte *hash)
{
    Signer *s;
    for (s = list; s != NULL; s = s->next) {
        if (XMEMCMP(s->subjectKeyHash, hash, KEYID_SIZE) == 0) {
            return s;
        }
    }
    return NULL;
}
#endif /* WOLFSSL_OCSP */

Signer* findSignerByName(Signer *list, byte *hash)
{
    Signer *s;
    for (s = list; s != NULL; s = s->next) {
        if (XMEMCMP(s->subjectNameHash, hash, SIGNER_DIGEST_SIZE) == 0) {
            return s;
        }
    }
    return NULL;
}

int ParseCertRelative(DecodedCert* cert, int type, int verify, void* cm,
                      Signer *extraCAList)
{
    int    ret = 0;
#ifndef WOLFSSL_ASN_TEMPLATE
    word32 confirmOID = 0;
#ifdef WOLFSSL_CERT_REQ
    int    len = 0;
#endif
#endif
#if defined(WOLFSSL_RENESAS_TSIP_TLS) || defined(WOLFSSL_RENESAS_FSPSM_TLS)
    int    idx = 0;
#endif
    byte*  sce_tsip_encRsaKeyIdx;
    (void)extraCAList;

    if (cert == NULL) {
        return BAD_FUNC_ARG;
    }

#ifdef WOLFSSL_CERT_REQ
    if (type == CERTREQ_TYPE)
        cert->isCSR = 1;
#endif

    if (cert->sigCtx.state == SIG_STATE_BEGIN) {
#ifndef WOLFSSL_ASN_TEMPLATE
        cert->badDate = 0;
        cert->criticalExt = 0;
        if ((ret = DecodeToKey(cert, verify)) < 0) {
            if (ret == WC_NO_ERR_TRACE(ASN_BEFORE_DATE_E) ||
                ret == WC_NO_ERR_TRACE(ASN_AFTER_DATE_E)) {
                cert->badDate = ret;
                if ((verify == VERIFY_SKIP_DATE) || AsnSkipDateCheck)
                    ret = 0;
            }
            else
                return ret;
        }

        WOLFSSL_MSG("Parsed Past Key");
#if defined(HAVE_RPK)
        if (cert->isRPK) {
            return ret;
        }
#endif /* HAVE_RPK */

#ifdef WOLFSSL_CERT_REQ
        /* Read attributes */
        if (cert->isCSR) {
            if (GetASNHeader_ex(cert->source,
                    ASN_CONTEXT_SPECIFIC | ASN_CONSTRUCTED, &cert->srcIdx,
                    &len, cert->maxIdx, 1) < 0) {
                WOLFSSL_MSG("GetASNHeader_ex error");
                return ASN_PARSE_E;
            }

            if (len) {
                word32 attrMaxIdx = cert->srcIdx + (word32)len;
                word32 oid;
                byte   tag;

                if (attrMaxIdx > cert->maxIdx) {
                    WOLFSSL_MSG("Attribute length greater than CSR length");
                    return ASN_PARSE_E;
                }

                while (cert->srcIdx < attrMaxIdx) {
                    /* Attributes have the structure:
                     * SEQ -> OID -> SET -> ATTRIBUTE */
                    if (GetSequence(cert->source, &cert->srcIdx, &len,
                            attrMaxIdx) < 0) {
                        WOLFSSL_MSG("attr GetSequence error");
                        return ASN_PARSE_E;
                    }
                    if (GetObjectId(cert->source, &cert->srcIdx, &oid,
                            oidCsrAttrType, attrMaxIdx) < 0) {
                        WOLFSSL_MSG("attr GetObjectId error");
                        return ASN_PARSE_E;
                    }
                    if (GetSet(cert->source, &cert->srcIdx, &len,
                            attrMaxIdx) < 0) {
                        WOLFSSL_MSG("attr GetSet error");
                        return ASN_PARSE_E;
                    }
                    switch (oid) {
                    case PKCS9_CONTENT_TYPE_OID:
                        if (GetHeader(cert->source, &tag,
                                &cert->srcIdx, &len, attrMaxIdx, 1) < 0) {
                            WOLFSSL_MSG("attr GetHeader error");
                            return ASN_PARSE_E;
                        }
                        if (tag != ASN_PRINTABLE_STRING && tag != ASN_UTF8STRING &&
                                tag != ASN_IA5_STRING) {
                            WOLFSSL_MSG("Unsupported attribute value format");
                            return ASN_PARSE_E;
                        }
                        cert->contentType = (char*)cert->source + cert->srcIdx;
                        cert->contentTypeLen = len;
                        cert->srcIdx += (word32)len;
                        break;
                    case CHALLENGE_PASSWORD_OID:
                        if (GetHeader(cert->source, &tag,
                                &cert->srcIdx, &len, attrMaxIdx, 1) < 0) {
                            WOLFSSL_MSG("attr GetHeader error");
                            return ASN_PARSE_E;
                        }
                        if (tag != ASN_PRINTABLE_STRING && tag != ASN_UTF8STRING &&
                                tag != ASN_IA5_STRING) {
                            WOLFSSL_MSG("Unsupported attribute value format");
                            return ASN_PARSE_E;
                        }
                        cert->cPwd = (char*)cert->source + cert->srcIdx;
                        cert->cPwdLen = len;
                        cert->srcIdx += (word32)len;
                        break;
                    case SERIAL_NUMBER_OID:
                        if (GetHeader(cert->source, &tag,
                                &cert->srcIdx, &len, attrMaxIdx, 1) < 0) {
                            WOLFSSL_MSG("attr GetHeader error");
                            return ASN_PARSE_E;
                        }
                        if (tag != ASN_PRINTABLE_STRING && tag != ASN_UTF8STRING &&
                                tag != ASN_IA5_STRING) {
                            WOLFSSL_MSG("Unsupported attribute value format");
                            return ASN_PARSE_E;
                        }
                        cert->sNum = (char*)cert->source + cert->srcIdx;
                        cert->sNumLen = len;
                        cert->srcIdx += (word32)len;
                        if (cert->sNumLen <= EXTERNAL_SERIAL_SIZE) {
                            XMEMCPY(cert->serial, cert->sNum,
                                    (size_t)cert->sNumLen);
                            cert->serialSz = cert->sNumLen;
                        }
                        break;
                    case DNQUALIFIER_OID:
                        if (GetHeader(cert->source, &tag,
                                &cert->srcIdx, &len, attrMaxIdx, 1) < 0) {
                            WOLFSSL_MSG("attr GetHeader error");
                            return ASN_PARSE_E;
                        }
                        cert->dnQualifier = (char*)cert->source + cert->srcIdx;
                        cert->dnQualifierLen = len;
                        cert->srcIdx += (word32)len;
                        break;
                    case INITIALS_OID:
                        if (GetHeader(cert->source, &tag,
                                &cert->srcIdx, &len, attrMaxIdx, 1) < 0) {
                            WOLFSSL_MSG("attr GetHeader error");
                            return ASN_PARSE_E;
                        }
                        cert->initials = (char*)cert->source + cert->srcIdx;
                        cert->initialsLen = len;
                        cert->srcIdx += (word32)len;
                        break;
                    case SURNAME_OID:
                        if (GetHeader(cert->source, &tag,
                                &cert->srcIdx, &len, attrMaxIdx, 1) < 0) {
                            WOLFSSL_MSG("attr GetHeader error");
                            return ASN_PARSE_E;
                        }
                        cert->surname = (char*)cert->source + cert->srcIdx;
                        cert->surnameLen = len;
                        cert->srcIdx += (word32)len;
                        break;
                    case GIVEN_NAME_OID:
                        if (GetHeader(cert->source, &tag,
                                &cert->srcIdx, &len, attrMaxIdx, 1) < 0) {
                            WOLFSSL_MSG("attr GetHeader error");
                            return ASN_PARSE_E;
                        }
                        cert->givenName = (char*)cert->source + cert->srcIdx;
                        cert->givenNameLen = len;
                        cert->srcIdx += (word32)len;
                        break;
                    case UNSTRUCTURED_NAME_OID:
                        if (GetHeader(cert->source, &tag,
                                &cert->srcIdx, &len, attrMaxIdx, 1) < 0) {
                            WOLFSSL_MSG("attr GetHeader error");
                            return ASN_PARSE_E;
                        }
                        cert->unstructuredName =
                            (char*)cert->source + cert->srcIdx;
                        cert->unstructuredNameLen = len;
                        cert->srcIdx += (word32)len;
                        break;
                    case EXTENSION_REQUEST_OID:
                        /* save extensions */
                        cert->extensions    = &cert->source[cert->srcIdx];
                        cert->extensionsSz  = len;
                        cert->extensionsIdx = cert->srcIdx; /* for potential later use */

                        if ((ret = DecodeCertExtensions(cert)) < 0) {
                            if (ret == WC_NO_ERR_TRACE(ASN_CRIT_EXT_E)) {
                                cert->criticalExt = ret;
                            }
                            else {
                                return ret;
                            }
                        }
                        cert->srcIdx += (word32)len;
                        break;
                    default:
                        WOLFSSL_MSG("Unsupported attribute type");
                        WOLFSSL_ERROR_VERBOSE(ASN_PARSE_E);
                        return ASN_PARSE_E;
                    }
                }
            }
        }
#endif

        if (cert->srcIdx < cert->sigIndex) {
        #ifndef ALLOW_V1_EXTENSIONS
            if (cert->version < 2) {
                WOLFSSL_MSG("\tv1 and v2 certs not allowed extensions");
                WOLFSSL_ERROR_VERBOSE(ASN_VERSION_E);
                return ASN_VERSION_E;
            }
        #endif

            /* save extensions */
            cert->extensions    = &cert->source[cert->srcIdx];
            cert->extensionsSz  = (int)(cert->sigIndex - cert->srcIdx);
            cert->extensionsIdx = cert->srcIdx;   /* for potential later use */

            if ((ret = DecodeCertExtensions(cert)) < 0) {
                if (ret == WC_NO_ERR_TRACE(ASN_CRIT_EXT_E))
                    cert->criticalExt = ret;
                else
                    return ret;
            }

        #ifdef HAVE_OCSP
            if (verify == VERIFY_OCSP_CERT) {
                /* trust for the lifetime of the responder's cert*/
                if (cert->ocspNoCheckSet)
                    verify = VERIFY;
                else
                    verify = VERIFY_OCSP;
            }
        #endif
            /* advance past extensions */
            cert->srcIdx = cert->sigIndex;
        }

        if ((ret = GetSigAlg(cert,
#ifdef WOLFSSL_CERT_REQ
                !cert->isCSR ? &confirmOID : &cert->signatureOID,
#else
                &confirmOID,
#endif
                cert->maxIdx)) < 0) {
            return ret;
        }

        if ((ret = GetSignature(cert)) < 0) {
            return ret;
        }

        if (confirmOID != cert->signatureOID
#ifdef WOLFSSL_CERT_REQ
                && !cert->isCSR
#endif
                ) {
            WOLFSSL_ERROR_VERBOSE(ASN_SIG_OID_E);
            return ASN_SIG_OID_E;
        }
#else
#ifdef WOLFSSL_CERT_REQ
        if (cert->isCSR) {
            ret = DecodeCertReq(cert, &cert->criticalExt);
            if (ret < 0) {
                return ret;
            }
        }
        else
#endif
        {
            ret = DecodeCert(cert, verify, &cert->criticalExt);
            if (ret == WC_NO_ERR_TRACE(ASN_BEFORE_DATE_E) ||
                ret == WC_NO_ERR_TRACE(ASN_AFTER_DATE_E)) {
                cert->badDate = ret;
                if ((verify == VERIFY_SKIP_DATE) || AsnSkipDateCheck)
                    ret = 0;
            }
            else if (ret < 0) {
                WOLFSSL_ERROR_VERBOSE(ret);
                return ret;
            }
#if defined(HAVE_RPK)
            if (cert->isRPK) {
                return ret;
            }
#endif /* HAVE_RPK */
        }
#endif

    #ifndef ALLOW_INVALID_CERTSIGN
        /* https://datatracker.ietf.org/doc/html/rfc5280#section-4.2.1.9
         *   If the cA boolean is not asserted, then the keyCertSign bit in the
         *   key usage extension MUST NOT be asserted. */
        if (!cert->isCA && cert->extKeyUsageSet &&
                (cert->extKeyUsage & KEYUSE_KEY_CERT_SIGN) != 0
        #ifdef ALLOW_SELFSIGNED_INVALID_CERTSIGN
                && !cert->selfSigned
        #endif
        ) {
            WOLFSSL_ERROR_VERBOSE(KEYUSAGE_E);
            return KEYUSAGE_E;
        }
    #endif

    #ifndef NO_SKID
        if (cert->extSubjKeyIdSet == 0 && cert->publicKey != NULL &&
                                                         cert->pubKeySize > 0) {
            if (cert->signatureOID == CTC_SM3wSM2) {
                /* TODO: GmSSL creates IDs this way but whole public key info
                 * block should be hashed. */
                ret = CalcHashId_ex(cert->publicKey + cert->pubKeySize - 65, 65,
                    cert->extSubjKeyId, HashIdAlg(cert->signatureOID));
            }
            else {
                ret = CalcHashId_ex(cert->publicKey, cert->pubKeySize,
                    cert->extSubjKeyId, HashIdAlg(cert->signatureOID));
            }
            if (ret != 0) {
                WOLFSSL_ERROR_VERBOSE(ret);
                return ret;
            }
        }
    #endif /* !NO_SKID */

        if (!cert->selfSigned || (verify != NO_VERIFY && type != CA_TYPE &&
                                                   type != TRUSTED_PEER_TYPE)) {
            cert->ca = NULL;
#ifdef HAVE_CERTIFICATE_STATUS_REQUEST_V2
        if (extraCAList != NULL) {
            cert->ca = findSignerByName(extraCAList, cert->issuerHash);
        }
#endif
    #ifndef NO_SKID
            if (cert->ca == NULL && cert->extAuthKeyIdSet) {
                cert->ca = GetCA(cm, cert->extAuthKeyId);
        #ifdef WOLFSSL_AKID_NAME
                if (cert->ca == NULL) {
                    cert->ca = GetCAByAKID(cm, cert->extAuthKeyIdIssuer,
                        cert->extAuthKeyIdIssuerSz, cert->extAuthKeyIdIssuerSN,
                        cert->extAuthKeyIdIssuerSNSz);
                }
        #endif
            }
            if (cert->ca == NULL && cert->extSubjKeyIdSet
                                 && verify != VERIFY_OCSP) {
                cert->ca = GetCA(cm, cert->extSubjKeyId);
            }
            if (cert->ca != NULL && XMEMCMP(cert->issuerHash,
                    cert->ca->subjectNameHash, KEYID_SIZE) != 0) {
                cert->ca = NULL;
            }
            if (cert->ca == NULL) {
                cert->ca = GetCAByName(cm, cert->issuerHash);
                /* If AKID is available then this CA doesn't have the public
                 * key required */
                if (cert->ca && cert->extAuthKeyIdSet) {
                    WOLFSSL_MSG("CA SKID doesn't match AKID");
                    cert->ca = NULL;
                }
            }

            /* OCSP Only: alt lookup using subject and pub key w/o sig check */
        #ifdef WOLFSSL_NO_TRUSTED_CERTS_VERIFY
            if (cert->ca == NULL && verify == VERIFY_OCSP) {
                cert->ca = GetCABySubjectAndPubKey(cert, cm);
                if (cert->ca) {
                    ret = 0; /* success */
                    goto exit_pcr;
                }
            }
        #endif /* WOLFSSL_NO_TRUSTED_CERTS_VERIFY */
    #else
            cert->ca = GetCA(cm, cert->issuerHash);
    #endif /* !NO_SKID */

            if (cert->ca) {
                WOLFSSL_MSG("CA found");
            }
        }

        /* Set to WOLFSSL_MAX_PATH_LEN by default in InitDecodedCert_ex */
        if (cert->pathLengthSet)
            cert->maxPathLen = cert->pathLength;

        if (!cert->selfSigned) {
            /* Need to perform a pathlen check on anything that will be used
             * to sign certificates later on. Otherwise, pathLen doesn't
             * mean anything.
             * Nothing to check if we don't have the issuer of this cert. */
            if (type != CERT_TYPE && cert->isCA && cert->extKeyUsageSet &&
                (cert->extKeyUsage & KEYUSE_KEY_CERT_SIGN) != 0 && cert->ca) {
                if (cert->ca->maxPathLen == 0) {
                    /* This cert CAN NOT be used as an intermediate cert. The
                     * issuer does not allow it. */
                    cert->maxPathLen = 0;
                    if (verify != NO_VERIFY) {
                        WOLFSSL_MSG("\tNon-entity cert, maxPathLen is 0");
                        WOLFSSL_MSG("\tmaxPathLen status: ERROR");
                        WOLFSSL_ERROR_VERBOSE(ASN_PATHLEN_INV_E);
                        return ASN_PATHLEN_INV_E;
                    }
                }
                else {
                    cert->maxPathLen = (byte)min(cert->ca->maxPathLen - 1U,
                                           cert->maxPathLen);
                }
            }
        }

        #ifdef HAVE_OCSP
        if (type != CA_TYPE &&
                                                type != TRUSTED_PEER_TYPE) {
            /* Need the CA's public key hash for OCSP */
            if (cert->ca) {
                XMEMCPY(cert->issuerKeyHash, cert->ca->subjectKeyHash,
                    KEYID_SIZE);
            }
            else if (cert->selfSigned) {
                XMEMCPY(cert->issuerKeyHash, cert->subjectKeyHash,
                    KEYID_SIZE);
            }
        }
        #endif /* HAVE_OCSP */
    }
#if defined(WOLFSSL_RENESAS_TSIP_TLS) || defined(WOLFSSL_RENESAS_FSPSM_TLS)
    /* prepare for TSIP TLS cert verification API use */
    if (cert->keyOID == RSAk) {
        /* to call TSIP API, it needs keys position info in bytes */
        if ((ret = RsaPublicKeyDecodeRawIndex(cert->publicKey, (word32*)&idx,
                                   cert->pubKeySize,
                                   &cert->sigCtx.CertAtt.pubkey_n_start,
                                   &cert->sigCtx.CertAtt.pubkey_n_len,
                                   &cert->sigCtx.CertAtt.pubkey_e_start,
                                   &cert->sigCtx.CertAtt.pubkey_e_len)) != 0) {
            WOLFSSL_MSG("Decoding index from cert failed.");
            return ret;
        }
        cert->sigCtx.CertAtt.certBegin = cert->certBegin;
    }
    else if (cert->keyOID == ECDSAk) {
        cert->sigCtx.CertAtt.certBegin = cert->certBegin;
    }
    /* check if we can use TSIP for cert verification */
    /* if the ca is verified as tsip root ca.         */
    /* TSIP can only handle 2048 bits(256 byte) key.  */
    if (cert->ca && Renesas_cmn_checkCA(cert->ca->cm_idx) != 0 &&
        (cert->sigCtx.CertAtt.pubkey_n_len == 256 ||
         cert->sigCtx.CertAtt.curve_id == ECC_SECP256R1)) {

        /* assign memory to encrypted tsip Rsa key index */
        if (!cert->sce_tsip_encRsaKeyIdx)
            cert->sce_tsip_encRsaKeyIdx =
                            (byte*)XMALLOC(TSIP_TLS_ENCPUBKEY_SZ_BY_CERTVRFY,
                             cert->heap, DYNAMIC_TYPE_RSA);
        if (cert->sce_tsip_encRsaKeyIdx == NULL)
                return MEMORY_E;
    }
    else {
        if (cert->ca) {
            /* TSIP isn't usable */
            if (Renesas_cmn_checkCA(cert->ca->cm_idx) == 0)
                WOLFSSL_MSG("SCE-TSIP isn't usable because the ca isn't verified "
                            "by TSIP.");
            else if (cert->sigCtx.CertAtt.pubkey_n_len != 256)
                WOLFSSL_MSG("SCE-TSIP isn't usable because the ca isn't signed by "
                            "RSA 2048.");
            else
                WOLFSSL_MSG("SCE-TSIP isn't usable");
        }
        cert->sce_tsip_encRsaKeyIdx = NULL;
    }

    sce_tsip_encRsaKeyIdx = cert->sce_tsip_encRsaKeyIdx;

#else
    sce_tsip_encRsaKeyIdx = NULL;
#endif

    if (verify != NO_VERIFY && type != CA_TYPE && type != TRUSTED_PEER_TYPE) {
        if (cert->ca) {
            if (verify == VERIFY || verify == VERIFY_OCSP ||
                                                 verify == VERIFY_SKIP_DATE) {
                word32 keyOID = cert->ca->keyOID;
            #if defined(WOLFSSL_SM2) && defined(WOLFSSL_SM3)
                if (cert->selfSigned && (cert->signatureOID == CTC_SM3wSM2)) {
                    keyOID = SM2k;
                }
            #endif
                /* try to confirm/verify signature */
                if ((ret = ConfirmSignature(&cert->sigCtx,
                        cert->source + cert->certBegin,
                        cert->sigIndex - cert->certBegin,
                        cert->ca->publicKey, cert->ca->pubKeySize,
                        keyOID, cert->signature, cert->sigLength,
                        cert->signatureOID,
                    #ifdef WC_RSA_PSS
                        cert->source + cert->sigParamsIndex,
                        cert->sigParamsLength,
                    #else
                        NULL, 0,
                    #endif
                        sce_tsip_encRsaKeyIdx)) != 0) {
                    if (ret != WC_NO_ERR_TRACE(WC_PENDING_E)) {
                        WOLFSSL_MSG("Confirm signature failed");
                    }
                    WOLFSSL_ERROR_VERBOSE(ret);
                    return ret;
                }

            #ifdef WOLFSSL_DUAL_ALG_CERTS
                if ((ret == 0) && cert->extAltSigAlgSet &&
                    cert->extAltSigValSet) {
                #ifndef WOLFSSL_SMALL_STACK
                    byte der[WC_MAX_CERT_VERIFY_SZ];
                #else
                    byte *der = (byte*)XMALLOC(WC_MAX_CERT_VERIFY_SZ, cert->heap,
                                            DYNAMIC_TYPE_DCERT);
                    if (der == NULL) {
                        ret = MEMORY_E;
                    } else
                #endif /* ! WOLFSSL_SMALL_STACK */
                    {
                        ret = wc_GeneratePreTBS(cert, der, WC_MAX_CERT_VERIFY_SZ);

                        if (ret > 0) {
                            ret = ConfirmSignature(&cert->sigCtx, der, ret,
                                    cert->ca->sapkiDer, cert->ca->sapkiLen,
                                    cert->ca->sapkiOID, cert->altSigValDer,
                                    cert->altSigValLen, cert->altSigAlgOID,
                                    NULL, 0, NULL);
                        }
                        WC_FREE_VAR_EX(der, cert->heap, DYNAMIC_TYPE_DCERT);

                        if (ret != 0) {
                            WOLFSSL_MSG("Confirm alternative signature failed");
                            WOLFSSL_ERROR_VERBOSE(ret);
                            return ret;
                        }
                        else {
                            WOLFSSL_MSG("Alt signature has been verified!");
                        }
                    }
                }
            #endif /* WOLFSSL_DUAL_ALG_CERTS */
            }
        #ifndef IGNORE_NAME_CONSTRAINTS
            if (verify == VERIFY || verify == VERIFY_OCSP ||
                        verify == VERIFY_NAME || verify == VERIFY_SKIP_DATE) {
                /* check that this cert's name is permitted by the signer's
                 * name constraints */
                if (!ConfirmNameConstraints(cert->ca, cert)) {
                    WOLFSSL_MSG("Confirm name constraint failed");
                    WOLFSSL_ERROR_VERBOSE(ASN_NAME_INVALID_E);
                    return ASN_NAME_INVALID_E;
                }
            }
        #endif /* IGNORE_NAME_CONSTRAINTS */
        } /* cert->ca */
#ifdef WOLFSSL_CERT_REQ
        else if (type == CERTREQ_TYPE) {
            /* try to confirm/verify signature */
            if ((ret = ConfirmSignature(&cert->sigCtx,
                    cert->source + cert->certBegin,
                    cert->sigIndex - cert->certBegin,
                    cert->publicKey, cert->pubKeySize,
                    cert->keyOID, cert->signature,
                    cert->sigLength, cert->signatureOID,
                #ifdef WC_RSA_PSS
                    cert->source + cert->sigParamsIndex, cert->sigParamsLength,
                #else
                    NULL, 0,
                #endif
                    sce_tsip_encRsaKeyIdx)) != 0) {
                if (ret != WC_NO_ERR_TRACE(WC_PENDING_E)) {
                    WOLFSSL_MSG("Confirm signature failed");
                }
                WOLFSSL_ERROR_VERBOSE(ret);
                return ret;
            }

        #ifdef WOLFSSL_DUAL_ALG_CERTS
            if ((ret == 0) && cert->extAltSigAlgSet &&
                cert->extAltSigValSet) {
            #ifndef WOLFSSL_SMALL_STACK
                byte der[WC_MAX_CERT_VERIFY_SZ];
            #else
                byte *der = (byte*)XMALLOC(WC_MAX_CERT_VERIFY_SZ, cert->heap,
                                        DYNAMIC_TYPE_DCERT);
                if (der == NULL) {
                    ret = MEMORY_E;
                } else
            #endif /* ! WOLFSSL_SMALL_STACK */
                {
                    ret = wc_GeneratePreTBS(cert, der, WC_MAX_CERT_VERIFY_SZ);

                    if (ret > 0) {
                        ret = ConfirmSignature(&cert->sigCtx, der, ret,
                                cert->sapkiDer, cert->sapkiLen,
                                cert->sapkiOID, cert->altSigValDer,
                                cert->altSigValLen, cert->altSigAlgOID,
                                NULL, 0, NULL);
                    }
                    WC_FREE_VAR_EX(der, cert->heap, DYNAMIC_TYPE_DCERT);

                    if (ret != 0) {
                        WOLFSSL_MSG("Confirm alternative signature failed");
                        WOLFSSL_ERROR_VERBOSE(ret);
                        return ret;
                    }
                    else {
                        WOLFSSL_MSG("Alt signature has been verified!");
                    }
                }
            }
        #endif /* WOLFSSL_DUAL_ALG_CERTS */
        }
#endif
        else {
            /* no signer */
            WOLFSSL_MSG_CERT_LOG("No CA signer to verify with");
            /* If you end up here with error -188,
             * consider using WOLFSSL_ALT_CERT_CHAINS. */
#if defined(OPENSSL_ALL) || defined(WOLFSSL_QT)
            /* ret needs to be self-signer error for openssl compatibility */
            if (cert->selfSigned) {
                WOLFSSL_ERROR_VERBOSE(ASN_SELF_SIGNED_E);
                return ASN_SELF_SIGNED_E;
            }
            else
#endif
            {
                WOLFSSL_ERROR_VERBOSE(ASN_NO_SIGNER_E);
                WOLFSSL_MSG_CERT("Consider using WOLFSSL_ALT_CERT_CHAINS.");
                return ASN_NO_SIGNER_E;
            }
        }
    } /* verify != NO_VERIFY && type != CA_TYPE && type != TRUSTED_PEER_TYPE */

#if defined(WOLFSSL_NO_TRUSTED_CERTS_VERIFY) && !defined(NO_SKID)
exit_pcr:
#endif

    if (cert->badDate != 0) {
        if (verify != VERIFY_SKIP_DATE) {
            return cert->badDate;
        }
        WOLFSSL_MSG_CERT_LOG("Date error: Verify option is skipping");
    }

    if (cert->criticalExt != 0)
        return cert->criticalExt;

    return ret;
}

int FillSigner(Signer* signer, DecodedCert* cert, int type, DerBuffer *der)
{
    int ret = 0;

    if (signer == NULL || cert == NULL)
        return BAD_FUNC_ARG;

#ifdef WOLFSSL_DUAL_ALG_CERTS
    if (ret == 0 && signer != NULL) {
        if (cert->extSapkiSet && cert->sapkiLen > 0) {
            /* Allocated space for alternative public key. */
            signer->sapkiDer = (byte*)XMALLOC(cert->sapkiLen, cert->heap,
                                              DYNAMIC_TYPE_PUBLIC_KEY);
            if (signer->sapkiDer == NULL) {
                ret = MEMORY_E;
            }
            else {
                XMEMCPY(signer->sapkiDer, cert->sapkiDer, cert->sapkiLen);
                signer->sapkiLen = cert->sapkiLen;
                signer->sapkiOID = cert->sapkiOID;
            }
        }
    }
#endif /* WOLFSSL_DUAL_ALG_CERTS */

#if defined(WOLFSSL_AKID_NAME) || defined(HAVE_CRL)
    if (ret == 0 && signer != NULL)
        ret = CalcHashId(cert->serial, (word32)cert->serialSz,
                         signer->serialHash);
#endif
    if (ret == 0 && signer != NULL) {
    #ifdef WOLFSSL_SIGNER_DER_CERT
        ret = AllocDer(&signer->derCert, der->length, der->type, NULL);
    }
    if (ret == 0 && signer != NULL) {
        XMEMCPY(signer->derCert->buffer, der->buffer, der->length);
    #else
    (void)der;
    #endif
        signer->keyOID         = cert->keyOID;
        if (cert->pubKeyStored) {
            signer->publicKey      = cert->publicKey;
            signer->pubKeySize     = cert->pubKeySize;
        }

        if (cert->subjectCNStored) {
            signer->nameLen        = cert->subjectCNLen;
            signer->name           = cert->subjectCN;
        }
        signer->maxPathLen     = cert->maxPathLen;
        signer->selfSigned     = cert->selfSigned;
    #ifndef IGNORE_NAME_CONSTRAINTS
        signer->permittedNames = cert->permittedNames;
        signer->excludedNames  = cert->excludedNames;
    #endif
    #ifndef NO_SKID
        XMEMCPY(signer->subjectKeyIdHash, cert->extSubjKeyId,
                SIGNER_DIGEST_SIZE);
    #endif
        XMEMCPY(signer->subjectNameHash, cert->subjectHash,
                SIGNER_DIGEST_SIZE);
    #if defined(HAVE_OCSP) || defined(HAVE_CRL)
        XMEMCPY(signer->issuerNameHash, cert->issuerHash,
                SIGNER_DIGEST_SIZE);
    #endif
    #ifdef HAVE_OCSP
        XMEMCPY(signer->subjectKeyHash, cert->subjectKeyHash,
                KEYID_SIZE);
    #endif
        signer->keyUsage = cert->extKeyUsageSet ? cert->extKeyUsage
                                                : 0xFFFF;
        signer->next    = NULL; /* If Key Usage not set, all uses valid. */
        cert->publicKey = 0;    /* in case lock fails don't free here.   */
        cert->subjectCN = 0;
    #ifndef IGNORE_NAME_CONSTRAINTS
        cert->permittedNames = NULL;
        cert->excludedNames = NULL;
    #endif
        signer->type = (byte)type;
    }
    return ret;
}

/* Create and init an new signer */
Signer* MakeSigner(void* heap)
{
    Signer* signer = (Signer*) XMALLOC(sizeof(Signer), heap,
                                       DYNAMIC_TYPE_SIGNER);
    if (signer) {
        XMEMSET(signer, 0, sizeof(Signer));
    }
    (void)heap;

    return signer;
}


/* Free an individual signer.
 *
 * Used by Certificate Manager.
 *
 * @param [in, out] signer  On in, signer object.
 *                          On out, pointer is no longer valid.
 * @param [in]      heap    Dynamic memory hint.
 */
void FreeSigner(Signer* signer, void* heap)
{
    (void)signer;
    (void)heap;
    XFREE(signer->name, heap, DYNAMIC_TYPE_SUBJECT_CN);
    XFREE((void*)signer->publicKey, heap, DYNAMIC_TYPE_PUBLIC_KEY);
#ifdef WOLFSSL_DUAL_ALG_CERTS
    XFREE(signer->sapkiDer, heap, DYNAMIC_TYPE_PUBLIC_KEY);
#endif
#ifndef IGNORE_NAME_CONSTRAINTS
    if (signer->permittedNames)
        FreeNameSubtrees(signer->permittedNames, heap);
    if (signer->excludedNames)
        FreeNameSubtrees(signer->excludedNames, heap);
#endif
#ifdef WOLFSSL_SIGNER_DER_CERT
    FreeDer(&signer->derCert);
#endif
    XFREE(signer, heap, DYNAMIC_TYPE_SIGNER);
}


/* Free the whole singer table with number of rows.
 *
 * Each table entry is a linked list of signers.
 * Used by Certificate Manager.
 *
 * @param [in, out] table   Array of signer objects.
 * @param [in]      rows    Number of entries in table.
 * @param [in]      heap    Dynamic memory hint.
 */
void FreeSignerTable(Signer** table, int rows, void* heap)
{
    int i;

    for (i = 0; i < rows; i++) {
        Signer* signer = table[i];
        while (signer) {
            Signer* next = signer->next;
            FreeSigner(signer, heap);
            signer = next;
        }
        table[i] = NULL;
    }
}

void FreeSignerTableType(Signer** table, int rows, byte type, void* heap)
{
    int i;

    for (i = 0; i < rows; i++) {
        Signer* signer = table[i];
        Signer** next = &table[i];

        while (signer) {
            if (signer->type == type) {
                *next = signer->next;
                FreeSigner(signer, heap);
                signer = *next;
            }
            else {
                next = &signer->next;
                signer = signer->next;
            }
        }
    }
}

#ifdef WOLFSSL_TRUST_PEER_CERT
/* Free an individual trusted peer cert.
 *
 * @param [in, out] tp    Trusted peer certificate object.
 * @param [in]      heap  Dynamic memory hint.
 */
void FreeTrustedPeer(TrustedPeerCert* tp, void* heap)
{
    if (tp == NULL) {
        return;
    }

    XFREE(tp->name, heap, DYNAMIC_TYPE_SUBJECT_CN);

    XFREE(tp->sig, heap, DYNAMIC_TYPE_SIGNATURE);
#ifndef IGNORE_NAME_CONSTRAINTS
    if (tp->permittedNames)
        FreeNameSubtrees(tp->permittedNames, heap);
    if (tp->excludedNames)
        FreeNameSubtrees(tp->excludedNames, heap);
#endif
    XFREE(tp, heap, DYNAMIC_TYPE_CERT);

    (void)heap;
}

/* Free the whole Trusted Peer linked list.
 *
 * Each table entry is a linked list of trusted peer certificates.
 * Used by Certificate Manager.
 *
 * @param [in, out] table   Array of trusted peer certificate objects.
 * @param [in]      rows    Number of entries in table.
 * @param [in]      heap    Dynamic memory hint.
 */
void FreeTrustedPeerTable(TrustedPeerCert** table, int rows, void* heap)
{
    int i;

    for (i = 0; i < rows; i++) {
        TrustedPeerCert* tp = table[i];
        while (tp) {
            TrustedPeerCert* next = tp->next;
            FreeTrustedPeer(tp, heap);
            tp = next;
        }
        table[i] = NULL;
    }
}
#endif /* WOLFSSL_TRUST_PEER_CERT */

#if !defined(WOLFSSL_ASN_TEMPLATE) || defined(HAVE_PKCS7)
int SetSerialNumber(const byte* sn, word32 snSz, byte* output,
    word32 outputSz, int maxSnSz)
{
    int i;
    int snSzInt = (int)snSz;

    if (sn == NULL || output == NULL || snSzInt < 0)
        return BAD_FUNC_ARG;

    /* remove leading zeros */
    while (snSzInt > 0 && sn[0] == 0) {
        snSzInt--;
        sn++;
    }
    /* RFC 5280 - 4.1.2.2:
     *   Serial numbers must be a positive value (and not zero) */
    if (snSzInt == 0) {
        WOLFSSL_ERROR_VERBOSE(BAD_FUNC_ARG);
        return BAD_FUNC_ARG;
    }

    if (sn[0] & 0x80)
        maxSnSz--;
    /* truncate if input is too long */
    if (snSzInt > maxSnSz)
        snSzInt = maxSnSz;

    i = SetASNInt(snSzInt, sn[0], NULL);
    /* truncate if input is too long */
    if (snSzInt > (int)outputSz - i)
        snSzInt = (int)outputSz - i;
    /* sanity check number of bytes to copy */
    if (snSzInt <= 0) {
        return BUFFER_E;
    }

    /* write out ASN.1 Integer */
    (void)SetASNInt(snSzInt, sn[0], output);
    XMEMCPY(output + i, sn, (size_t)snSzInt);

    /* compute final length */
    i += snSzInt;

    return i;
}
#endif /* !WOLFSSL_ASN_TEMPLATE */

#endif /* !NO_CERTS */

#if defined(WOLFSSL_ASN_TEMPLATE) || defined(HAVE_PKCS12) || \
    (defined(HAVE_ECC_KEY_EXPORT) && !defined(NO_ASN_CRYPT)) || \
    (!defined(NO_RSA) && defined(WOLFSSL_KEY_GEN))
int SetMyVersion(word32 version, byte* output, int header)
{
    int i = 0;

    if (output == NULL)
        return BAD_FUNC_ARG;

    if (header) {
        output[i++] = ASN_CONTEXT_SPECIFIC | ASN_CONSTRUCTED;
        output[i++] = 3;
    }
    output[i++] = ASN_INTEGER;
    output[i++] = 0x01;
    output[i++] = (byte)version;

    return i;
}
#endif

#ifndef WOLFSSL_ASN_TEMPLATE
int wc_GetSerialNumber(const byte* input, word32* inOutIdx,
    byte* serial, int* serialSz, word32 maxIdx)
{
    int result = 0;
    int ret;

    WOLFSSL_ENTER("wc_GetSerialNumber");

    if (serial == NULL || input == NULL || serialSz == NULL) {
        return BAD_FUNC_ARG;
    }

    /* First byte is ASN type */
    if ((*inOutIdx+1) > maxIdx) {
        WOLFSSL_MSG("Bad idx first");
        return BUFFER_E;
    }

    ret = GetASNInt(input, inOutIdx, serialSz, maxIdx);
    if (ret != 0)
        return ret;

    if (*serialSz > EXTERNAL_SERIAL_SIZE || *serialSz <= 0) {
        WOLFSSL_MSG("Serial size bad");
        WOLFSSL_ERROR_VERBOSE(ASN_PARSE_E);
        return ASN_PARSE_E;
    }

    /* return serial */
    XMEMCPY(serial, &input[*inOutIdx], (size_t)*serialSz);
    *inOutIdx += (word32)*serialSz;

    return result;
}
#endif

#ifndef NO_CERTS

/* TODO: consider moving PEM code out to a different file. */

int AllocDer(DerBuffer** pDer, word32 length, int type, void* heap)
{
    int ret = WC_NO_ERR_TRACE(BAD_FUNC_ARG);
    if (pDer) {
        int dynType = 0;
        DerBuffer* der;

        /* Determine dynamic type */
        switch (type) {
            case CA_TYPE:   dynType = DYNAMIC_TYPE_CA;   break;
            case CHAIN_CERT_TYPE:
            case CERT_TYPE: dynType = DYNAMIC_TYPE_CERT; break;
            case CRL_TYPE:  dynType = DYNAMIC_TYPE_CRL;  break;
            case DSA_TYPE:  dynType = DYNAMIC_TYPE_DSA;  break;
            case ECC_TYPE:  dynType = DYNAMIC_TYPE_ECC;  break;
            case RSA_TYPE:  dynType = DYNAMIC_TYPE_RSA;  break;
            default:        dynType = DYNAMIC_TYPE_KEY;  break;
        }

        /* Setup new buffer */
        *pDer = (DerBuffer*)XMALLOC(sizeof(DerBuffer) + length, heap, dynType);
        if (*pDer == NULL) {
            return MEMORY_E;
        }
        XMEMSET(*pDer, 0, sizeof(DerBuffer) + length);

        der = *pDer;
        der->type = type;
        der->dynType = dynType; /* Cache this for FreeDer */
        der->heap = heap;
        der->buffer = (byte*)der + sizeof(DerBuffer);
        der->length = length;
        ret = 0; /* Success */
    } else {
        ret = BAD_FUNC_ARG;
    }
    return ret;
}

int AllocCopyDer(DerBuffer** pDer, const unsigned char* buff, word32 length,
    int type, void* heap)
{
    int ret = AllocDer(pDer, length, type, heap);
    if (ret == 0) {
        XMEMCPY((*pDer)->buffer, buff, length);
    }

    return ret;
}

void FreeDer(DerBuffer** pDer)
{
    if (pDer && *pDer) {
        DerBuffer* der = (DerBuffer*)*pDer;

        /* ForceZero private keys */
        if (((der->type == PRIVATEKEY_TYPE) ||
             (der->type == ALT_PRIVATEKEY_TYPE)) && der->buffer != NULL) {
            ForceZero(der->buffer, der->length);
        }
        der->buffer = NULL;
        der->length = 0;
        XFREE(der, der->heap, der->dynType);

        *pDer = NULL;
    }
}

int wc_AllocDer(DerBuffer** pDer, word32 length, int type, void* heap)
{
    return AllocDer(pDer, length, type, heap);
}
void wc_FreeDer(DerBuffer** pDer)
{
    FreeDer(pDer);
}


#if defined(WOLFSSL_PEM_TO_DER) || defined(WOLFSSL_DER_TO_PEM)

/* Note: If items added make sure MAX_X509_HEADER_SZ is
    updated to reflect maximum length and pem_struct_min_sz
    to reflect minimum size */
static wcchar BEGIN_CERT           = "-----BEGIN CERTIFICATE-----";
static wcchar END_CERT             = "-----END CERTIFICATE-----";
#ifdef WOLFSSL_CERT_REQ
    static wcchar BEGIN_CERT_REQ   = "-----BEGIN CERTIFICATE REQUEST-----";
    static wcchar END_CERT_REQ     = "-----END CERTIFICATE REQUEST-----";
#endif
#if defined(WOLFSSL_ACERT)
    static wcchar BEGIN_ACERT      = "-----BEGIN ATTRIBUTE CERTIFICATE-----";
    static wcchar END_ACERT        = "-----END ATTRIBUTE CERTIFICATE-----";
#endif /* WOLFSSL_ACERT */
#ifndef NO_DH
    static wcchar BEGIN_DH_PARAM   = "-----BEGIN DH PARAMETERS-----";
    static wcchar END_DH_PARAM     = "-----END DH PARAMETERS-----";
    static wcchar BEGIN_X942_PARAM = "-----BEGIN X9.42 DH PARAMETERS-----";
    static wcchar END_X942_PARAM   = "-----END X9.42 DH PARAMETERS-----";
#endif
#ifndef NO_DSA
    static wcchar BEGIN_DSA_PARAM  = "-----BEGIN DSA PARAMETERS-----";
    static wcchar END_DSA_PARAM    = "-----END DSA PARAMETERS-----";
#endif
static wcchar BEGIN_X509_CRL       = "-----BEGIN X509 CRL-----";
static wcchar END_X509_CRL         = "-----END X509 CRL-----";
static wcchar BEGIN_TRUSTED_CERT   = "-----BEGIN TRUSTED CERTIFICATE-----";
static wcchar END_TRUSTED_CERT     = "-----END TRUSTED CERTIFICATE-----";
static wcchar BEGIN_RSA_PRIV       = "-----BEGIN RSA PRIVATE KEY-----";
static wcchar END_RSA_PRIV         = "-----END RSA PRIVATE KEY-----";
static wcchar BEGIN_RSA_PUB        = "-----BEGIN RSA PUBLIC KEY-----";
static wcchar END_RSA_PUB          = "-----END RSA PUBLIC KEY-----";
static wcchar BEGIN_PRIV_KEY       = "-----BEGIN PRIVATE KEY-----";
static wcchar END_PRIV_KEY         = "-----END PRIVATE KEY-----";
static wcchar BEGIN_ENC_PRIV_KEY   = "-----BEGIN ENCRYPTED PRIVATE KEY-----";
static wcchar END_ENC_PRIV_KEY     = "-----END ENCRYPTED PRIVATE KEY-----";
#ifdef HAVE_ECC
    static wcchar BEGIN_EC_PRIV    = "-----BEGIN EC PRIVATE KEY-----";
    static wcchar END_EC_PRIV      = "-----END EC PRIVATE KEY-----";
#ifdef OPENSSL_EXTRA
    static wcchar BEGIN_EC_PARAM   = "-----BEGIN EC PARAMETERS-----";
    static wcchar END_EC_PARAM     = "-----END EC PARAMETERS-----";
#endif
#endif
#ifdef HAVE_PKCS7
static wcchar BEGIN_PKCS7          = "-----BEGIN PKCS7-----";
static wcchar END_PKCS7            = "-----END PKCS7-----";
#endif
#if (defined(HAVE_ECC) || !defined(NO_DSA)) && defined(WOLFSSL_PEM_TO_DER)
    static wcchar BEGIN_DSA_PRIV   = "-----BEGIN DSA PRIVATE KEY-----";
    static wcchar END_DSA_PRIV     = "-----END DSA PRIVATE KEY-----";
#endif
#if defined(OPENSSL_EXTRA) && defined(WOLFSSL_PEM_TO_DER)
    static wcchar BEGIN_PRIV_KEY_PREFIX = "-----BEGIN";
    static wcchar PRIV_KEY_SUFFIX = "PRIVATE KEY-----";
    static wcchar END_PRIV_KEY_PREFIX   = "-----END";
#endif
static wcchar BEGIN_PUB_KEY        = "-----BEGIN PUBLIC KEY-----";
static wcchar END_PUB_KEY          = "-----END PUBLIC KEY-----";
#if defined(HAVE_ED25519) || defined(HAVE_ED448)
    static wcchar BEGIN_EDDSA_PRIV = "-----BEGIN EDDSA PRIVATE KEY-----";
    static wcchar END_EDDSA_PRIV   = "-----END EDDSA PRIVATE KEY-----";
#endif
#if defined(HAVE_FALCON)
    static wcchar BEGIN_FALCON_LEVEL1_PRIV  = "-----BEGIN FALCON_LEVEL1 PRIVATE KEY-----";
    static wcchar END_FALCON_LEVEL1_PRIV    = "-----END FALCON_LEVEL1 PRIVATE KEY-----";
    static wcchar BEGIN_FALCON_LEVEL5_PRIV  = "-----BEGIN FALCON_LEVEL5 PRIVATE KEY-----";
    static wcchar END_FALCON_LEVEL5_PRIV    = "-----END FALCON_LEVEL5 PRIVATE KEY-----";
#endif /* HAVE_FALCON */
#if defined(HAVE_DILITHIUM)
    #ifdef WOLFSSL_DILITHIUM_FIPS204_DRAFT
    static wcchar BEGIN_DILITHIUM_LEVEL2_PRIV = "-----BEGIN DILITHIUM_LEVEL2 PRIVATE KEY-----";
    static wcchar END_DILITHIUM_LEVEL2_PRIV   = "-----END DILITHIUM_LEVEL2 PRIVATE KEY-----";
    static wcchar BEGIN_DILITHIUM_LEVEL3_PRIV = "-----BEGIN DILITHIUM_LEVEL3 PRIVATE KEY-----";
    static wcchar END_DILITHIUM_LEVEL3_PRIV   = "-----END DILITHIUM_LEVEL3 PRIVATE KEY-----";
    static wcchar BEGIN_DILITHIUM_LEVEL5_PRIV = "-----BEGIN DILITHIUM_LEVEL5 PRIVATE KEY-----";
    static wcchar END_DILITHIUM_LEVEL5_PRIV   = "-----END DILITHIUM_LEVEL5 PRIVATE KEY-----";
    #endif
    static wcchar BEGIN_ML_DSA_LEVEL2_PRIV = "-----BEGIN ML_DSA_LEVEL2 PRIVATE KEY-----";
    static wcchar END_ML_DSA_LEVEL2_PRIV   = "-----END ML_DSA_LEVEL2 PRIVATE KEY-----";
    static wcchar BEGIN_ML_DSA_LEVEL3_PRIV = "-----BEGIN ML_DSA_LEVEL3 PRIVATE KEY-----";
    static wcchar END_ML_DSA_LEVEL3_PRIV   = "-----END ML_DSA_LEVEL3 PRIVATE KEY-----";
    static wcchar BEGIN_ML_DSA_LEVEL5_PRIV = "-----BEGIN ML_DSA_LEVEL5 PRIVATE KEY-----";
    static wcchar END_ML_DSA_LEVEL5_PRIV   = "-----END ML_DSA_LEVEL5 PRIVATE KEY-----";
#endif /* HAVE_DILITHIUM */
#if defined(HAVE_SPHINCS)
    static wcchar BEGIN_SPHINCS_FAST_LEVEL1_PRIV = "-----BEGIN SPHINCS_FAST_LEVEL1 PRIVATE KEY-----";
    static wcchar END_SPHINCS_FAST_LEVEL1_PRIV   = "-----END SPHINCS_FAST_LEVEL1 PRIVATE KEY-----";
    static wcchar BEGIN_SPHINCS_FAST_LEVEL3_PRIV = "-----BEGIN SPHINCS_FAST_LEVEL3 PRIVATE KEY-----";
    static wcchar END_SPHINCS_FAST_LEVEL3_PRIV   = "-----END SPHINCS_FAST_LEVEL3 PRIVATE KEY-----";
    static wcchar BEGIN_SPHINCS_FAST_LEVEL5_PRIV = "-----BEGIN SPHINCS_FAST_LEVEL5 PRIVATE KEY-----";
    static wcchar END_SPHINCS_FAST_LEVEL5_PRIV   = "-----END SPHINCS_FAST_LEVEL5 PRIVATE KEY-----";

    static wcchar BEGIN_SPHINCS_SMALL_LEVEL1_PRIV = "-----BEGIN SPHINCS_SMALL_LEVEL1 PRIVATE KEY-----";
    static wcchar END_SPHINCS_SMALL_LEVEL1_PRIV   = "-----END SPHINCS_SMALL_LEVEL1 PRIVATE KEY-----";
    static wcchar BEGIN_SPHINCS_SMALL_LEVEL3_PRIV = "-----BEGIN SPHINCS_SMALL_LEVEL3 PRIVATE KEY-----";
    static wcchar END_SPHINCS_SMALL_LEVEL3_PRIV   = "-----END SPHINCS_SMALL_LEVEL3 PRIVATE KEY-----";
    static wcchar BEGIN_SPHINCS_SMALL_LEVEL5_PRIV = "-----BEGIN SPHINCS_SMALL_LEVEL5 PRIVATE KEY-----";
    static wcchar END_SPHINCS_SMALL_LEVEL5_PRIV   = "-----END SPHINCS_SMALL_LEVEL5 PRIVATE KEY-----";
#endif /* HAVE_SPHINCS */

const int pem_struct_min_sz = XSTR_SIZEOF("-----BEGIN X509 CRL-----"
                                             "-----END X509 CRL-----");

#ifdef WOLFSSL_PEM_TO_DER
static WC_INLINE const char* SkipEndOfLineChars(const char* line,
                                                const char* endOfLine)
{
    /* eat end of line characters */
    while (line < endOfLine &&
              (line[0] == '\r' || line[0] == '\n')) {
        line++;
    }
    return line;
}
#endif

int wc_PemGetHeaderFooter(int type, const char** header, const char** footer)
{
    int ret = WC_NO_ERR_TRACE(BAD_FUNC_ARG);

    switch (type) {
        case CA_TYPE:       /* same as below */
        case TRUSTED_PEER_TYPE:
        case CHAIN_CERT_TYPE:
        case CERT_TYPE:
            if (header) *header = BEGIN_CERT;
            if (footer) *footer = END_CERT;
            ret = 0;
            break;

        case CRL_TYPE:
            if (header) *header = BEGIN_X509_CRL;
            if (footer) *footer = END_X509_CRL;
            ret = 0;
            break;
    #ifndef NO_DH
        case DH_PARAM_TYPE:
            if (header) *header = BEGIN_DH_PARAM;
            if (footer) *footer = END_DH_PARAM;
            ret = 0;
            break;
        case X942_PARAM_TYPE:
            if (header) *header = BEGIN_X942_PARAM;
            if (footer) *footer = END_X942_PARAM;
            ret = 0;
            break;
    #endif
    #ifndef NO_DSA
        case DSA_PARAM_TYPE:
            if (header) *header = BEGIN_DSA_PARAM;
            if (footer) *footer = END_DSA_PARAM;
            ret = 0;
            break;
    #endif
    #ifdef WOLFSSL_CERT_REQ
        case CERTREQ_TYPE:
            if (header) *header = BEGIN_CERT_REQ;
            if (footer) *footer = END_CERT_REQ;
            ret = 0;
            break;
    #endif
    #ifdef HAVE_PKCS7
        case PKCS7_TYPE:
            if (header) *header = BEGIN_PKCS7;
            if (footer) *footer = END_PKCS7;
            ret = 0;
            break;
    #endif
    #if defined(WOLFSSL_ACERT)
        case ACERT_TYPE:
            if (header) *header = BEGIN_ACERT;
            if (footer) *footer = END_ACERT;
            ret = 0;
            break;
    #endif /* WOLFSSL_ACERT */
    #ifndef NO_DSA
        case DSA_TYPE:
        case DSA_PRIVATEKEY_TYPE:
            if (header) *header = BEGIN_DSA_PRIV;
            if (footer) *footer = END_DSA_PRIV;
            ret = 0;
            break;
    #endif
    #ifdef HAVE_ECC
        case ECC_TYPE:
        case ECC_PRIVATEKEY_TYPE:
            if (header) *header = BEGIN_EC_PRIV;
            if (footer) *footer = END_EC_PRIV;
            ret = 0;
            break;
    #ifdef OPENSSL_EXTRA
        case ECC_PARAM_TYPE:
            if (header) *header = BEGIN_EC_PARAM;
            if (footer) *footer = END_EC_PARAM;
            ret = 0;
            break;
    #endif
    #endif
        case RSA_TYPE:
        case PRIVATEKEY_TYPE:
    #ifdef WOLFSSL_DUAL_ALG_CERTS
        case ALT_PRIVATEKEY_TYPE:
    #endif
            if (header) *header = BEGIN_RSA_PRIV;
            if (footer) *footer = END_RSA_PRIV;
            ret = 0;
            break;
    #ifdef HAVE_ED25519
        case ED25519_TYPE:
    #endif
    #ifdef HAVE_ED448
        case ED448_TYPE:
    #endif
    #if defined(HAVE_ED25519) || defined(HAVE_ED448)
        case EDDSA_PRIVATEKEY_TYPE:
            if (header) *header = BEGIN_EDDSA_PRIV;
            if (footer) *footer = END_EDDSA_PRIV;
            ret = 0;
            break;
    #endif
#ifdef HAVE_FALCON
        case FALCON_LEVEL1_TYPE:
            if (header) *header = BEGIN_FALCON_LEVEL1_PRIV;
            if (footer) *footer = END_FALCON_LEVEL1_PRIV;
            ret = 0;
            break;
        case FALCON_LEVEL5_TYPE:
            if (header) *header = BEGIN_FALCON_LEVEL5_PRIV;
            if (footer) *footer = END_FALCON_LEVEL5_PRIV;
            ret = 0;
            break;
#endif /* HAVE_FALCON */
#ifdef HAVE_DILITHIUM
    #ifdef WOLFSSL_DILITHIUM_FIPS204_DRAFT
        case DILITHIUM_LEVEL2_TYPE:
            if (header) *header = BEGIN_DILITHIUM_LEVEL2_PRIV;
            if (footer) *footer = END_DILITHIUM_LEVEL2_PRIV;
            ret = 0;
            break;
        case DILITHIUM_LEVEL3_TYPE:
            if (header) *header = BEGIN_DILITHIUM_LEVEL3_PRIV;
            if (footer) *footer = END_DILITHIUM_LEVEL3_PRIV;
            ret = 0;
            break;
        case DILITHIUM_LEVEL5_TYPE:
            if (header) *header = BEGIN_DILITHIUM_LEVEL5_PRIV;
            if (footer) *footer = END_DILITHIUM_LEVEL5_PRIV;
            ret = 0;
            break;
    #endif
        case ML_DSA_LEVEL2_TYPE:
            if (header) *header = BEGIN_ML_DSA_LEVEL2_PRIV;
            if (footer) *footer = END_ML_DSA_LEVEL2_PRIV;
            ret = 0;
            break;
        case ML_DSA_LEVEL3_TYPE:
            if (header) *header = BEGIN_ML_DSA_LEVEL3_PRIV;
            if (footer) *footer = END_ML_DSA_LEVEL3_PRIV;
            ret = 0;
            break;
        case ML_DSA_LEVEL5_TYPE:
            if (header) *header = BEGIN_ML_DSA_LEVEL5_PRIV;
            if (footer) *footer = END_ML_DSA_LEVEL5_PRIV;
            ret = 0;
            break;
#endif /* HAVE_DILITHIUM */
#ifdef HAVE_SPHINCS
        case SPHINCS_FAST_LEVEL1_TYPE:
            if (header) *header = BEGIN_SPHINCS_FAST_LEVEL1_PRIV;
            if (footer) *footer = END_SPHINCS_FAST_LEVEL1_PRIV;
            ret = 0;
            break;
        case SPHINCS_FAST_LEVEL3_TYPE:
            if (header) *header = BEGIN_SPHINCS_FAST_LEVEL3_PRIV;
            if (footer) *footer = END_SPHINCS_FAST_LEVEL3_PRIV;
            ret = 0;
            break;
        case SPHINCS_FAST_LEVEL5_TYPE:
            if (header) *header = BEGIN_SPHINCS_FAST_LEVEL5_PRIV;
            if (footer) *footer = END_SPHINCS_FAST_LEVEL5_PRIV;
            ret = 0;
            break;
        case SPHINCS_SMALL_LEVEL1_TYPE:
            if (header) *header = BEGIN_SPHINCS_SMALL_LEVEL1_PRIV;
            if (footer) *footer = END_SPHINCS_SMALL_LEVEL1_PRIV;
            ret = 0;
            break;
        case SPHINCS_SMALL_LEVEL3_TYPE:
            if (header) *header = BEGIN_SPHINCS_SMALL_LEVEL3_PRIV;
            if (footer) *footer = END_SPHINCS_SMALL_LEVEL3_PRIV;
            ret = 0;
            break;
        case SPHINCS_SMALL_LEVEL5_TYPE:
            if (header) *header = BEGIN_SPHINCS_SMALL_LEVEL5_PRIV;
            if (footer) *footer = END_SPHINCS_SMALL_LEVEL5_PRIV;
            ret = 0;
            break;
#endif /* HAVE_SPHINCS */
        case PUBLICKEY_TYPE:
        case ECC_PUBLICKEY_TYPE:
            if (header) *header = BEGIN_PUB_KEY;
            if (footer) *footer = END_PUB_KEY;
            ret = 0;
            break;
        case RSA_PUBLICKEY_TYPE:
            if (header) *header = BEGIN_RSA_PUB;
            if (footer) *footer = END_RSA_PUB;
            ret = 0;
            break;
    #ifndef NO_DH
        case DH_PRIVATEKEY_TYPE:
    #endif
        case PKCS8_PRIVATEKEY_TYPE:
            if (header) *header = BEGIN_PRIV_KEY;
            if (footer) *footer = END_PRIV_KEY;
            ret = 0;
            break;
        case PKCS8_ENC_PRIVATEKEY_TYPE:
            if (header) *header = BEGIN_ENC_PRIV_KEY;
            if (footer) *footer = END_ENC_PRIV_KEY;
            ret = 0;
            break;
        case TRUSTED_CERT_TYPE:
            if (header) *header = BEGIN_TRUSTED_CERT;
            if (footer) *footer = END_TRUSTED_CERT;
            ret = 0;
            break;
        default:
            ret = BAD_FUNC_ARG;
            break;
    }
    return ret;
}

#ifdef WOLFSSL_ENCRYPTED_KEYS

static wcchar kProcTypeHeader = "Proc-Type";
static wcchar kDecInfoHeader = "DEK-Info";

#ifdef WOLFSSL_PEM_TO_DER
#ifndef NO_DES3
    static wcchar kEncTypeDes = "DES-CBC";
    static wcchar kEncTypeDes3 = "DES-EDE3-CBC";
#endif
#if !defined(NO_AES) && defined(HAVE_AES_CBC) && defined(WOLFSSL_AES_128)
    static wcchar kEncTypeAesCbc128 = "AES-128-CBC";
#endif
#if !defined(NO_AES) && defined(HAVE_AES_CBC) && defined(WOLFSSL_AES_192)
    static wcchar kEncTypeAesCbc192 = "AES-192-CBC";
#endif
#if !defined(NO_AES) && defined(HAVE_AES_CBC) && defined(WOLFSSL_AES_256)
    static wcchar kEncTypeAesCbc256 = "AES-256-CBC";
#endif

int wc_EncryptedInfoGet(EncryptedInfo* info, const char* cipherInfo)
{
    int ret = 0;

    if (info == NULL || cipherInfo == NULL)
        return BAD_FUNC_ARG;

    /* determine cipher information */
#ifndef NO_DES3
    if (XSTRCMP(cipherInfo, kEncTypeDes) == 0) {
        info->cipherType = WC_CIPHER_DES;
        info->keySz = DES_KEY_SIZE;
/* DES_IV_SIZE is incorrectly 16 in FIPS v2. It should be 8, same as the
 * block size. */
#if defined(HAVE_FIPS) && defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION == 2)
        if (info->ivSz == 0) info->ivSz  = DES_BLOCK_SIZE;
#else
        if (info->ivSz == 0) info->ivSz  = DES_IV_SIZE;
#endif
    }
    else if (XSTRCMP(cipherInfo, kEncTypeDes3) == 0) {
        info->cipherType = WC_CIPHER_DES3;
        info->keySz = DES3_KEY_SIZE;
#if defined(HAVE_FIPS) && defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION == 2)
        if (info->ivSz == 0) info->ivSz  = DES_BLOCK_SIZE;
#else
        if (info->ivSz == 0) info->ivSz  = DES_IV_SIZE;
#endif
    }
    else
#endif /* !NO_DES3 */
#if !defined(NO_AES) && defined(HAVE_AES_CBC) && defined(WOLFSSL_AES_128)
    if (XSTRCMP(cipherInfo, kEncTypeAesCbc128) == 0) {
        info->cipherType = WC_CIPHER_AES_CBC;
        info->keySz = AES_128_KEY_SIZE;
        if (info->ivSz == 0) info->ivSz  = AES_IV_SIZE;
    }
    else
#endif
#if !defined(NO_AES) && defined(HAVE_AES_CBC) && defined(WOLFSSL_AES_192)
    if (XSTRCMP(cipherInfo, kEncTypeAesCbc192) == 0) {
        info->cipherType = WC_CIPHER_AES_CBC;
        info->keySz = AES_192_KEY_SIZE;
        if (info->ivSz == 0) info->ivSz  = AES_IV_SIZE;
    }
    else
#endif
#if !defined(NO_AES) && defined(HAVE_AES_CBC) && defined(WOLFSSL_AES_256)
    if (XSTRCMP(cipherInfo, kEncTypeAesCbc256) == 0) {
        info->cipherType = WC_CIPHER_AES_CBC;
        info->keySz = AES_256_KEY_SIZE;
        if (info->ivSz == 0) info->ivSz  = AES_IV_SIZE;
    }
    else
#endif
    {
        ret = NOT_COMPILED_IN;
    }
    return ret;
}

int wc_EncryptedInfoParse(EncryptedInfo* info, const char** pBuffer,
                          size_t bufSz)
{
    int         err = 0;
    const char* bufferStart;
    const char* bufferEnd;
    char*       line;

    if (info == NULL || pBuffer == NULL || bufSz == 0)
        return BAD_FUNC_ARG;

    bufferStart = *pBuffer;
    bufferEnd = bufferStart + bufSz;

    /* find encrypted info marker */
    line = XSTRNSTR(bufferStart, kProcTypeHeader,
                    min((word32)bufSz, PEM_LINE_LEN));
    if (line != NULL) {
        word32      lineSz;
        char*       finish;
        char*       start;
        word32      startSz;
        const char* newline = NULL;

        if (line >= bufferEnd) {
            return BUFFER_E;
        }

        lineSz = (word32)(bufferEnd - line);

        /* find DEC-Info marker */
        start = XSTRNSTR(line, kDecInfoHeader, min(lineSz, PEM_LINE_LEN));

        if (start == NULL)
            return BUFFER_E;

        /* skip dec-info and ": " */
        start += XSTRLEN(kDecInfoHeader);
        if (start >= bufferEnd)
            return BUFFER_E;

        if (start[0] == ':') {
            start++;
            if (start >= bufferEnd)
                return BUFFER_E;
        }
        if (start[0] == ' ')
            start++;

        startSz = (word32)(bufferEnd - start);
        finish = XSTRNSTR(start, ",", min(startSz, PEM_LINE_LEN));

        if ((start != NULL) && (finish != NULL) && (start < finish)) {
            word32 finishSz;

            if (finish >= bufferEnd) {
                return BUFFER_E;
            }

            finishSz = (word32)(bufferEnd - finish);
            newline = XSTRNSTR(finish, "\r", min(finishSz, PEM_LINE_LEN));

            /* get cipher name */
            if (NAME_SZ <= (finish - start)) /* buffer size of info->name */
                return BUFFER_E;
            if (XMEMCPY(info->name, start, (size_t)(finish - start)) == NULL)
                return BUFFER_E;
            info->name[finish - start] = '\0'; /* null term */

            /* populate info */
            err = wc_EncryptedInfoGet(info, info->name);
            if (err != 0)
                return err;

            /* get IV */
            if (finishSz < info->ivSz + 1)
                return BUFFER_E;

            if (newline == NULL) {
                newline = XSTRNSTR(finish, "\n", min(finishSz,
                                                     PEM_LINE_LEN));
            }
            if ((newline != NULL) && (newline > finish)) {
                finish++;
                info->ivSz = (word32)(newline - finish);
                if (info->ivSz > IV_SZ)
                    return BUFFER_E;
                if (XMEMCPY(info->iv, finish, info->ivSz) == NULL)
                    return BUFFER_E;
                info->set = 1;
            }
            else
                return BUFFER_E;
        }
        else
            return BUFFER_E;

        /* eat end of line characters */
        newline = SkipEndOfLineChars(newline, bufferEnd);

        /* return new headerEnd */

        *pBuffer = newline;
    }

    return err;
}
#endif /* WOLFSSL_PEM_TO_DER */

#ifdef WOLFSSL_DER_TO_PEM
static int wc_EncryptedInfoAppend(char* dest, int destSz, char* cipherInfo)
{
    if (cipherInfo != NULL) {
        int cipherInfoStrLen = (int)XSTRLEN((char*)cipherInfo);

        if (cipherInfoStrLen > HEADER_ENCRYPTED_KEY_SIZE - (9+14+10+3))
            cipherInfoStrLen = HEADER_ENCRYPTED_KEY_SIZE - (9+14+10+3);

        if (destSz - (int)XSTRLEN(dest) >= cipherInfoStrLen + (9+14+8+2+2+1)) {
            /* strncat's src length needs to include the NULL */
            XSTRNCAT(dest, kProcTypeHeader, 10);
            XSTRNCAT(dest, ": 4,ENCRYPTED\n", 15);
            XSTRNCAT(dest, kDecInfoHeader, 9);
            XSTRNCAT(dest, ": ", 3);
            XSTRNCAT(dest, cipherInfo, (size_t)destSz - XSTRLEN(dest) - 1);
            XSTRNCAT(dest, "\n\n", 4);
        }
    }
    return 0;
}
#endif /* WOLFSSL_DER_TO_PEM */
#endif /* WOLFSSL_ENCRYPTED_KEYS */

#ifdef WOLFSSL_DER_TO_PEM

/* Used for compatibility API */
WOLFSSL_ABI
int wc_DerToPem(const byte* der, word32 derSz,
                byte* output, word32 outSz, int type)
{
    return wc_DerToPemEx(der, derSz, output, outSz, NULL, type);
}

/* convert der buffer to pem into output, can't do inplace, der and output
   need to be different */
int wc_DerToPemEx(const byte* der, word32 derSz, byte* output, word32 outSz,
             byte *cipher_info, int type)
{
    const char* headerStr = NULL;
    const char* footerStr = NULL;
#ifdef WOLFSSL_SMALL_STACK
    char* header = NULL;
    char* footer = NULL;
#else
    char header[MAX_X509_HEADER_SZ + HEADER_ENCRYPTED_KEY_SIZE];
    char footer[MAX_X509_HEADER_SZ];
#endif
    size_t headerLen = MAX_X509_HEADER_SZ + HEADER_ENCRYPTED_KEY_SIZE;
    size_t footerLen = MAX_X509_HEADER_SZ;
    int i;
    int err;
    int outLen;   /* return length or error */

    (void)cipher_info;

    if (der == output)      /* no in place conversion */
        return BAD_FUNC_ARG;

    err = wc_PemGetHeaderFooter(type, &headerStr, &footerStr);
    if (err != 0)
        return err;

#ifdef WOLFSSL_SMALL_STACK
    header = (char*)XMALLOC(headerLen, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    if (header == NULL)
        return MEMORY_E;

    footer = (char*)XMALLOC(footerLen, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    if (footer == NULL) {
        XFREE(header, NULL, DYNAMIC_TYPE_TMP_BUFFER);
        return MEMORY_E;
    }
#endif

    /* build header and footer based on type */
    XSTRNCPY(header, headerStr, headerLen - 1);
    header[headerLen - 2] = 0;
    XSTRNCPY(footer, footerStr, footerLen - 1);
    footer[footerLen - 2] = 0;

    /* add new line to end */
    XSTRNCAT(header, "\n", 2);
    XSTRNCAT(footer, "\n", 2);

#ifdef WOLFSSL_ENCRYPTED_KEYS
    err = wc_EncryptedInfoAppend(header, (int)headerLen, (char*)cipher_info);
    if (err != 0) {
        WC_FREE_VAR_EX(header, NULL, DYNAMIC_TYPE_TMP_BUFFER);
        WC_FREE_VAR_EX(footer, NULL, DYNAMIC_TYPE_TMP_BUFFER);
        return err;
    }
#endif

    headerLen = XSTRLEN(header);
    footerLen = XSTRLEN(footer);

    /* if null output and 0 size passed in then return size needed */
    if (!output && outSz == 0) {
        WC_FREE_VAR_EX(header, NULL, DYNAMIC_TYPE_TMP_BUFFER);
        WC_FREE_VAR_EX(footer, NULL, DYNAMIC_TYPE_TMP_BUFFER);
        outLen = 0;
        if ((err = Base64_Encode(der, derSz, NULL, (word32*)&outLen))
                != WC_NO_ERR_TRACE(LENGTH_ONLY_E)) {
            WOLFSSL_ERROR_VERBOSE(err);
            return err;
        }
        return (int)headerLen + (int)footerLen + outLen;
    }

    if (!der || !output) {
        WC_FREE_VAR_EX(header, NULL, DYNAMIC_TYPE_TMP_BUFFER);
        WC_FREE_VAR_EX(footer, NULL, DYNAMIC_TYPE_TMP_BUFFER);
        return BAD_FUNC_ARG;
    }

    /* don't even try if outSz too short */
    if (outSz < (word32)headerLen + (word32)footerLen + derSz) {
        WC_FREE_VAR_EX(header, NULL, DYNAMIC_TYPE_TMP_BUFFER);
        WC_FREE_VAR_EX(footer, NULL, DYNAMIC_TYPE_TMP_BUFFER);
        return BAD_FUNC_ARG;
    }

    /* header */
    XMEMCPY(output, header, (size_t)headerLen);
    i = (int)headerLen;

    WC_FREE_VAR_EX(header, NULL, DYNAMIC_TYPE_TMP_BUFFER);

    /* body */
    outLen = (int)outSz - (int)(headerLen + footerLen);  /* input to Base64_Encode */
    if ( (err = Base64_Encode(der, derSz, output + i, (word32*)&outLen)) < 0) {
        WC_FREE_VAR_EX(footer, NULL, DYNAMIC_TYPE_TMP_BUFFER);
        WOLFSSL_ERROR_VERBOSE(err);
        return err;
    }
    i += outLen;

    /* footer */
    if ( (i + (int)footerLen) > (int)outSz) {
        WC_FREE_VAR_EX(footer, NULL, DYNAMIC_TYPE_TMP_BUFFER);
        return BAD_FUNC_ARG;
    }
    XMEMCPY(output + i, footer, (size_t)footerLen);

    WC_FREE_VAR_EX(footer, NULL, DYNAMIC_TYPE_TMP_BUFFER);

    return outLen + (int)headerLen + (int)footerLen;
}

#endif /* WOLFSSL_DER_TO_PEM */

#ifdef WOLFSSL_PEM_TO_DER

/* Remove PEM header/footer, convert to ASN1, store any encrypted data
   info->consumed tracks of PEM bytes consumed in case multiple parts */
int PemToDer(const unsigned char* buff, long longSz, int type,
              DerBuffer** pDer, void* heap, EncryptedInfo* info, int* keyFormat)
{
    const char* header      = NULL;
    const char* footer      = NULL;
    const char* headerEnd   = NULL;
    const char* footerEnd   = NULL;
    const char* consumedEnd = NULL;
    const char* bufferEnd   = (const char*)(buff + longSz);
    long        neededSz;
    int         ret         = 0;
    word32      sz          = (word32)longSz;
    int         encrypted_key = 0;
    DerBuffer*  der;
    word32      algId = 0;
    word32      idx;
#ifdef OPENSSL_EXTRA
    char        beginBuf[PEM_LINE_LEN + 1]; /* add 1 for null terminator */
    char        endBuf[PEM_LINE_LEN + 1];   /* add 1 for null terminator */
#endif
#ifdef WOLFSSL_ENCRYPTED_KEYS
    int hashType = WC_HASH_TYPE_NONE;
#if !defined(NO_MD5)
    hashType = WC_MD5;
#elif !defined(NO_SHA)
    hashType = WC_SHA;
#endif
#endif

    WOLFSSL_ENTER("PemToDer");

    /* get PEM header and footer based on type */
    ret = wc_PemGetHeaderFooter(type, &header, &footer);
    if (ret != 0)
        return ret;

    /* map header if not found for type */
    for (;;) {
        headerEnd = XSTRNSTR((char*)buff, header, sz);
        if (headerEnd) {
            break;
        }

        if (type == PRIVATEKEY_TYPE
#ifdef WOLFSSL_DUAL_ALG_CERTS
            || type == ALT_PRIVATEKEY_TYPE
#endif
           ) {
            if (header == BEGIN_RSA_PRIV) {
                header = BEGIN_PRIV_KEY;
                footer = END_PRIV_KEY;
            }
            else if (header == BEGIN_PRIV_KEY) {
                header = BEGIN_ENC_PRIV_KEY;
                footer = END_ENC_PRIV_KEY;
            }
#ifdef HAVE_ECC
            else if (header == BEGIN_ENC_PRIV_KEY) {
                header = BEGIN_EC_PRIV;
                footer = END_EC_PRIV;
            }
            else if (header == BEGIN_EC_PRIV) {
                header = BEGIN_DSA_PRIV;
                footer = END_DSA_PRIV;
            }
#endif
#if defined(HAVE_ED25519) || defined(HAVE_ED448)
    #ifdef HAVE_ECC
            else if (header == BEGIN_DSA_PRIV) {
    #else
            else if (header == BEGIN_ENC_PRIV_KEY) {
    #endif
                header = BEGIN_EDDSA_PRIV;
                footer = END_EDDSA_PRIV;
            }
#endif
            else {
            #ifdef WOLF_PRIVATE_KEY_ID
                /* allow loading a public key for use with crypto or PK callbacks */
                type = PUBLICKEY_TYPE;
                header = BEGIN_PUB_KEY;
                footer = END_PUB_KEY;
            #else
                break;
            #endif
            }
        }
        else if (type == PUBLICKEY_TYPE) {
            if (header == BEGIN_PUB_KEY) {
                header = BEGIN_RSA_PUB;
                footer = END_RSA_PUB;
            }
            else {
                break;
            }
        }
#if defined(HAVE_ECC) && defined(OPENSSL_EXTRA)
        else if (type == ECC_PARAM_TYPE) {
            if (header == BEGIN_EC_PARAM) {
                header = BEGIN_EC_PARAM;
                footer = END_EC_PARAM;
            }
            else {
                break;
            }
        }
#endif
#ifdef HAVE_CRL
        else if ((type == CRL_TYPE) && (header != BEGIN_X509_CRL)) {
            header =  BEGIN_X509_CRL;
            footer = END_X509_CRL;
        }
#endif
        else {
            break;
        }
    }

    if (!headerEnd) {
#ifdef OPENSSL_EXTRA
        if (type == PRIVATEKEY_TYPE
#ifdef WOLFSSL_DUAL_ALG_CERTS
            || type == ALT_PRIVATEKEY_TYPE
#endif
           ) {
            /* see if there is a -----BEGIN * PRIVATE KEY----- header */
            headerEnd = XSTRNSTR((char*)buff, PRIV_KEY_SUFFIX, sz);
            if (headerEnd) {
                const char* beginEnd;
                unsigned int endLen;

                beginEnd = headerEnd + XSTR_SIZEOF(PRIV_KEY_SUFFIX);
                if (beginEnd >= (char*)buff + sz) {
                    return BUFFER_E;
                }

                /* back up to BEGIN_PRIV_KEY_PREFIX */
                while (headerEnd > (char*)buff &&
                        XSTRNCMP(headerEnd, BEGIN_PRIV_KEY_PREFIX,
                                XSTR_SIZEOF(BEGIN_PRIV_KEY_PREFIX)) != 0 &&
                        *headerEnd != '\n') {
                    headerEnd--;
                }
                if (headerEnd <= (char*)buff ||
                        XSTRNCMP(headerEnd, BEGIN_PRIV_KEY_PREFIX,
                        XSTR_SIZEOF(BEGIN_PRIV_KEY_PREFIX)) != 0 ||
                        beginEnd - headerEnd > PEM_LINE_LEN) {
                    WOLFSSL_MSG("Couldn't find PEM header");
                    WOLFSSL_ERROR(ASN_NO_PEM_HEADER);
                    return ASN_NO_PEM_HEADER;
                }

                /* headerEnd now points to beginning of header */
                XMEMCPY(beginBuf, headerEnd, (size_t)(beginEnd - headerEnd));
                beginBuf[beginEnd - headerEnd] = '\0';
                /* look for matching footer */
                footer = XSTRNSTR(beginEnd,
                                beginBuf + XSTR_SIZEOF(BEGIN_PRIV_KEY_PREFIX),
                                (unsigned int)((char*)buff + sz - beginEnd));
                if (!footer) {
                    WOLFSSL_MSG("Couldn't find PEM footer");
                    WOLFSSL_ERROR(ASN_NO_PEM_HEADER);
                    return ASN_NO_PEM_HEADER;
                }

                footer -= XSTR_SIZEOF(END_PRIV_KEY_PREFIX);
                if (footer > (char*)buff + sz - XSTR_SIZEOF(END_PRIV_KEY_PREFIX)
                        || XSTRNCMP(footer, END_PRIV_KEY_PREFIX,
                            XSTR_SIZEOF(END_PRIV_KEY_PREFIX)) != 0) {
                    WOLFSSL_MSG("Unexpected footer for PEM");
                    return BUFFER_E;
                }

                endLen = (unsigned int)((size_t)(beginEnd - headerEnd) -
                            (XSTR_SIZEOF(BEGIN_PRIV_KEY_PREFIX) -
                                    XSTR_SIZEOF(END_PRIV_KEY_PREFIX)));
                XMEMCPY(endBuf, footer, (size_t)endLen);
                endBuf[endLen] = '\0';

                header = beginBuf;
                footer = endBuf;
                headerEnd = beginEnd;
            }
        }

        if (!headerEnd) {
            WOLFSSL_MSG("Couldn't find PEM header");
            WOLFSSL_ERROR(ASN_NO_PEM_HEADER);
            return ASN_NO_PEM_HEADER;
        }
#else
        WOLFSSL_MSG("Couldn't find PEM header");
        return ASN_NO_PEM_HEADER;
#endif
    } else {
        headerEnd += XSTRLEN(header);
    }

    /* eat end of line characters */
    headerEnd = SkipEndOfLineChars(headerEnd, bufferEnd);

    if (keyFormat) {
        /* keyFormat is Key_Sum enum */
        if (type == PRIVATEKEY_TYPE
        #ifdef WOLFSSL_DUAL_ALG_CERTS
            || type == ALT_PRIVATEKEY_TYPE
        #endif
           ) {
        #ifndef NO_RSA
            if (header == BEGIN_RSA_PRIV)
                *keyFormat = RSAk;
        #endif
        #ifdef HAVE_ECC
            if (header == BEGIN_EC_PRIV)
                *keyFormat = ECDSAk;
        #endif
        #ifndef NO_DSA
            if (header == BEGIN_DSA_PRIV)
                *keyFormat = DSAk;
        #endif
        }
    #ifdef WOLF_PRIVATE_KEY_ID
        else if (type == PUBLICKEY_TYPE) {
        #ifndef NO_RSA
            if (header == BEGIN_RSA_PUB)
                *keyFormat = RSAk;
        #endif
        }
    #endif
    }

#ifdef WOLFSSL_ENCRYPTED_KEYS
    if (info) {
        ret = wc_EncryptedInfoParse(info, &headerEnd,
                                    (size_t)(bufferEnd - headerEnd));
        if (ret < 0)
            return ret;
        if (info->set)
            encrypted_key = 1;
    }
#endif /* WOLFSSL_ENCRYPTED_KEYS */

    /* find footer */
    footerEnd = XSTRNSTR(headerEnd, footer, (unsigned int)((char*)buff +
        sz - headerEnd));
    if (!footerEnd) {
        if (info)
            info->consumed = longSz; /* No more certs if no footer */
        return BUFFER_E;
    }

    consumedEnd = footerEnd + XSTRLEN(footer);

    if (consumedEnd < bufferEnd) { /* handle no end of line on last line */
        /* eat end of line characters */
        consumedEnd = SkipEndOfLineChars(consumedEnd, bufferEnd);
        /* skip possible null term */
        if (consumedEnd < bufferEnd && consumedEnd[0] == '\0')
            consumedEnd++;
    }

    if (info)
        info->consumed = (long)(consumedEnd - (const char*)buff);

    /* set up der buffer */
    neededSz = (long)(footerEnd - headerEnd);
    if (neededSz > (long)sz || neededSz <= 0)
        return BUFFER_E;

    ret = AllocDer(pDer, (word32)neededSz, type, heap);
    if (ret < 0) {
        return ret;
    }
    der = *pDer;

    switch (type) {
    case PUBLICKEY_TYPE:
    case ECC_PUBLICKEY_TYPE:
    case RSA_PUBLICKEY_TYPE:
    case CERT_TYPE:
    case TRUSTED_CERT_TYPE:
    case CRL_TYPE:
        if (Base64_Decode_nonCT((byte*)headerEnd, (word32)neededSz,
                          der->buffer, &der->length) < 0)
        {
            WOLFSSL_ERROR(BUFFER_E);
            return BUFFER_E;
        }
        break;
    default:
        if (Base64_Decode((byte*)headerEnd, (word32)neededSz,
                          der->buffer, &der->length) < 0) {
            WOLFSSL_ERROR(BUFFER_E);
            return BUFFER_E;
        }
        break;
    }

    if ((header == BEGIN_PRIV_KEY
#ifdef OPENSSL_EXTRA
         || header == beginBuf
#endif
#ifdef HAVE_ECC
         || header == BEGIN_EC_PRIV
#endif
        ) && !encrypted_key)
    {
        /* detect pkcs8 key and get alg type */
        /* keep PKCS8 header */
        idx = 0;
        ret = ToTraditionalInline_ex(der->buffer, &idx, der->length, &algId);
        if (ret > 0) {
            if (keyFormat)
                *keyFormat = (int)algId;
        }
        else {
            /* ignore failure here and assume key is not pkcs8 wrapped */
        }
        return 0;
    }

#ifdef WOLFSSL_ENCRYPTED_KEYS
    if (encrypted_key || header == BEGIN_ENC_PRIV_KEY) {
        int   passwordSz = NAME_SZ;
        WC_DECLARE_VAR(password, char, NAME_SZ, 0);

        if (!info || !info->passwd_cb) {
            WOLFSSL_MSG("No password callback set");
            WOLFSSL_ERROR_VERBOSE(NO_PASSWORD);
            return NO_PASSWORD;
        }

    #ifdef WOLFSSL_SMALL_STACK
        password = (char*)XMALLOC((size_t)passwordSz, heap, DYNAMIC_TYPE_STRING);
        if (password == NULL) {
            return MEMORY_E;
        }
    #endif

        /* get password */
        ret = info->passwd_cb(password, passwordSz, PEM_PASS_READ,
            info->passwd_userdata);
        if (ret >= 0) {
            passwordSz = ret;
        #ifdef WOLFSSL_CHECK_MEM_ZERO
            wc_MemZero_Add("PEM password", password, passwordSz);
        #endif

            /* convert and adjust length */
            if (header == BEGIN_ENC_PRIV_KEY) {
            #ifndef NO_PWDBASED
                ret = wc_DecryptPKCS8Key(der->buffer, der->length,
                    password, passwordSz);
                if (ret > 0) {
                    /* update length by decrypted content */
                    der->length = (word32)ret;
                    idx = 0;
                    /* detect pkcs8 key and get alg type */
                    /* keep PKCS8 header */
                    ret = ToTraditionalInline_ex(der->buffer, &idx, der->length,
                        &algId);
                    if (ret >= 0) {
                        if (keyFormat)
                            *keyFormat = (int)algId;
                        ret = 0;
                    }
                }
            #else
                WOLFSSL_ERROR_VERBOSE(NOT_COMPILED_IN);
                ret = NOT_COMPILED_IN;
            #endif
            }
            /* decrypt the key */
            else {
                if (passwordSz == 0) {
                    /* The key is encrypted but does not have a password */
                    WOLFSSL_MSG("No password for encrypted key");
                    WOLFSSL_ERROR_VERBOSE(NO_PASSWORD);
                    ret = NO_PASSWORD;
                }
                else {
                #if ((defined(WOLFSSL_ENCRYPTED_KEYS) && !defined(NO_DES3)) || \
                         (!defined(NO_AES) && defined(HAVE_AES_CBC) && \
                          defined(HAVE_AES_DECRYPT))) && \
                        !defined(NO_WOLFSSL_SKIP_TRAILING_PAD)
                    int     padVal = 0;
                #endif

                    ret = wc_BufferKeyDecrypt(info, der->buffer, der->length,
                        (byte*)password, passwordSz, hashType);

#ifndef NO_WOLFSSL_SKIP_TRAILING_PAD
                #ifndef NO_DES3
                    if (info->cipherType == WC_CIPHER_DES3) {
                        /* Assuming there is padding:
                         *      (der->length > 0 && der->length > DES_BLOCK_SIZE &&
                         *       (der->length % DES_BLOCK_SIZE) != 0)
                         * and assuming the last value signifies the number of
                         * padded bytes IE if last value is 0x08 then there are
                         * 8 bytes of padding:
                         *      padVal = der->buffer[der->length-1];
                         * then strip this padding before proceeding:
                         * der->length -= padVal;
                         */
                        if (der->length > DES_BLOCK_SIZE &&
                            (der->length % DES_BLOCK_SIZE) != 0) {
                            padVal = der->buffer[der->length-1];
                            if (padVal < DES_BLOCK_SIZE) {
                                der->length -= (word32)padVal;
                            }
                        }
                    }
                #endif /* !NO_DES3 */
                #if !defined(NO_AES) && defined(HAVE_AES_CBC) && \
                    defined(HAVE_AES_DECRYPT)
                    if (info->cipherType == WC_CIPHER_AES_CBC) {
                        if (der->length > WC_AES_BLOCK_SIZE) {
                            padVal = der->buffer[der->length-1];
                            if (padVal <= WC_AES_BLOCK_SIZE) {
                                der->length -= (word32)padVal;
                            }
                        }
                    }
                #endif
#endif /* !NO_WOLFSSL_SKIP_TRAILING_PAD */
                }
            }
#ifdef OPENSSL_EXTRA
            if (ret) {
                WOLFSSL_PEMerr(0, WOLFSSL_PEM_R_BAD_DECRYPT_E);
            }
#endif
            ForceZero(password, (word32)passwordSz);
        }
#ifdef OPENSSL_EXTRA
        else {
            WOLFSSL_PEMerr(0, WOLFSSL_PEM_R_BAD_PASSWORD_READ_E);
        }
#endif

    #ifdef WOLFSSL_SMALL_STACK
        XFREE(password, heap, DYNAMIC_TYPE_STRING);
    #elif defined(WOLFSSL_CHECK_MEM_ZERO)
        wc_MemZero_Check(password, NAME_SZ);
    #endif
    }
#endif /* WOLFSSL_ENCRYPTED_KEYS */

    return ret;
}

int wc_PemToDer(const unsigned char* buff, long longSz, int type,
              DerBuffer** pDer, void* heap, EncryptedInfo* info, int* keyFormat)
{
    int ret = PemToDer(buff, longSz, type, pDer, heap, info, keyFormat);
#if defined(HAVE_PKCS8) || defined(HAVE_PKCS12)
    if (ret == 0 && type == PRIVATEKEY_TYPE) {
        DerBuffer* der = *pDer;
        /* if a PKCS8 key header exists remove it */
        ret = ToTraditional(der->buffer, der->length);
        if (ret > 0) {
            der->length = (word32)ret;
        }
        ret = 0; /* ignore error removing PKCS8 header */
    }
#endif
    return ret;
}

#ifdef WOLFSSL_ENCRYPTED_KEYS
/* our KeyPemToDer password callback, password in userData */
static int KeyPemToDerPassCb(char* passwd, int sz, int rw, void* userdata)
{
    (void)rw;

    if (userdata == NULL)
        return 0;

    XSTRNCPY(passwd, (char*)userdata, (size_t)sz);
    return (int)min((word32)sz, (word32)XSTRLEN((char*)userdata));
}
#endif

/* Return bytes written to buff or < 0 for error */
int wc_KeyPemToDer(const unsigned char* pem, int pemSz,
                        unsigned char* buff, int buffSz, const char* pass)
{
    int ret;
    DerBuffer* der = NULL;
    WC_DECLARE_VAR(info, EncryptedInfo, 1, 0);

    WOLFSSL_ENTER("wc_KeyPemToDer");

    if (pem == NULL || (buff != NULL && buffSz <= 0)) {
        WOLFSSL_MSG("Bad pem der args");
        return BAD_FUNC_ARG;
    }

    WC_ALLOC_VAR_EX(info, EncryptedInfo, 1, NULL, DYNAMIC_TYPE_ENCRYPTEDINFO,
        return MEMORY_E);

    XMEMSET(info, 0, sizeof(EncryptedInfo));
#ifdef WOLFSSL_ENCRYPTED_KEYS
    info->passwd_cb = KeyPemToDerPassCb;
    info->passwd_userdata = (void*)pass;
#else
    (void)pass;
#endif

    ret = PemToDer(pem, pemSz, PRIVATEKEY_TYPE, &der, NULL, info, NULL);

    WC_FREE_VAR_EX(info, NULL, DYNAMIC_TYPE_ENCRYPTEDINFO);

    if (ret < 0 || der == NULL) {
        WOLFSSL_MSG("Bad Pem To Der");
    }
    else if (buff == NULL) {
        WOLFSSL_MSG("Return needed der buff length");
        ret = (int)der->length;
    }
    else if (der->length <= (word32)buffSz) {
        XMEMCPY(buff, der->buffer, der->length);
        ret = (int)der->length;
    }
    else {
        WOLFSSL_MSG("Bad der length");
        ret = BAD_FUNC_ARG;
    }

    FreeDer(&der);
    return ret;
}


/* Return bytes written to buff or < 0 for error */
int wc_CertPemToDer(const unsigned char* pem, int pemSz,
                        unsigned char* buff, int buffSz, int type)
{
    int ret;
    DerBuffer* der = NULL;

    WOLFSSL_ENTER("wc_CertPemToDer");

    if (pem == NULL || buff == NULL || buffSz <= 0) {
        WOLFSSL_MSG("Bad pem der args");
        return BAD_FUNC_ARG;
    }

    if (type != CERT_TYPE && type != CHAIN_CERT_TYPE && type != CA_TYPE &&
            type != CERTREQ_TYPE && type != PKCS7_TYPE) {
        WOLFSSL_MSG("Bad cert type");
        return BAD_FUNC_ARG;
    }


    ret = PemToDer(pem, pemSz, type, &der, NULL, NULL, NULL);
    if (ret < 0 || der == NULL) {
        WOLFSSL_MSG("Bad Pem To Der");
    }
    else {
        if (der->length <= (word32)buffSz) {
            XMEMCPY(buff, der->buffer, der->length);
            ret = (int)der->length;
        }
        else {
            WOLFSSL_MSG("Bad der length");
            ret = BAD_FUNC_ARG;
        }
    }

    FreeDer(&der);
    return ret;
}

#endif /* WOLFSSL_PEM_TO_DER */
#endif /* WOLFSSL_PEM_TO_DER || WOLFSSL_DER_TO_PEM */


#ifdef WOLFSSL_PEM_TO_DER
#if defined(WOLFSSL_CERT_EXT) || defined(WOLFSSL_PUB_PEM_TO_DER)
/* Return bytes written to buff, needed buff size if buff is NULL, or less than
   zero for error */
int wc_PubKeyPemToDer(const unsigned char* pem, int pemSz,
                           unsigned char* buff, int buffSz)
{
    int ret;
    DerBuffer* der = NULL;

    WOLFSSL_ENTER("wc_PubKeyPemToDer");

    if (pem == NULL || (buff != NULL && buffSz <= 0)) {
        WOLFSSL_MSG("Bad pem der args");
        return BAD_FUNC_ARG;
    }

    ret = PemToDer(pem, pemSz, PUBLICKEY_TYPE, &der, NULL, NULL, NULL);
    if (ret < 0 || der == NULL) {
        WOLFSSL_MSG("Bad Pem To Der");
    }
    else if (buff == NULL) {
        WOLFSSL_MSG("Return needed der buff length");
        ret = (int)der->length;
    }
    else if (der->length <= (word32)buffSz) {
        XMEMCPY(buff, der->buffer, der->length);
        ret = (int)der->length;
    }
    else {
        WOLFSSL_MSG("Bad der length");
        ret = BAD_FUNC_ARG;
    }

    FreeDer(&der);
    return ret;
}
#endif /* WOLFSSL_CERT_EXT || WOLFSSL_PUB_PEM_TO_DER */
#endif /* WOLFSSL_PEM_TO_DER */

#if !defined(NO_FILESYSTEM) && defined(WOLFSSL_PEM_TO_DER)

#ifdef WOLFSSL_CERT_GEN
int wc_PemCertToDer_ex(const char* fileName, DerBuffer** der)
{
#ifndef WOLFSSL_SMALL_STACK
    byte   staticBuffer[FILE_BUFFER_SIZE];
#endif
    byte*  fileBuf = NULL;
    int    ret     = 0;
    XFILE  file    = XBADFILE;
    int    dynamic = 0;
    long   sz      = 0;

    WOLFSSL_ENTER("wc_PemCertToDer");

    if (fileName == NULL) {
        ret = BAD_FUNC_ARG;
    }
    else {
        file = XFOPEN(fileName, "rb");
        if (file == XBADFILE) {
            ret = IO_FAILED_E;
        }
    }

    if (ret == 0) {
        if (XFSEEK(file, 0, XSEEK_END) != 0) {
            ret = IO_FAILED_E;
        }
    }
    if (ret == 0) {
        sz = XFTELL(file);
        if (sz <= 0) {
            ret = IO_FAILED_E;
        }
    }
    if (ret == 0) {
        if (XFSEEK(file, 0, XSEEK_SET) != 0) {
            ret = IO_FAILED_E;
        }
    }
    if (ret == 0) {
#ifndef WOLFSSL_SMALL_STACK
        if (sz <= (long)sizeof(staticBuffer))
            fileBuf = staticBuffer;
        else
#endif
        {
            fileBuf = (byte*)XMALLOC((size_t)sz, NULL, DYNAMIC_TYPE_FILE);
            if (fileBuf == NULL)
                ret = MEMORY_E;
            else
                dynamic = 1;
        }
    }
    if (ret == 0) {
        if ((size_t)XFREAD(fileBuf, 1, (size_t)sz, file) != (size_t)sz) {
            ret = IO_FAILED_E;
        }
        else {
            ret = PemToDer(fileBuf, sz, CA_TYPE, der,  0, NULL,NULL);
        }
    }

    if (file != XBADFILE)
        XFCLOSE(file);
    if (dynamic)
        XFREE(fileBuf, NULL, DYNAMIC_TYPE_FILE);

    return ret;
}
/* load pem cert from file into der buffer, return der size or error */
int wc_PemCertToDer(const char* fileName, unsigned char* derBuf, int derSz)
{
    int ret;
    DerBuffer* converted = NULL;
    ret = wc_PemCertToDer_ex(fileName, &converted);
    if (ret == 0) {
        if (converted->length < (word32)derSz) {
            XMEMCPY(derBuf, converted->buffer, converted->length);
            ret = (int)converted->length;
        }
        else
            ret = BUFFER_E;

        FreeDer(&converted);
    }
    return ret;
}
#endif /* WOLFSSL_CERT_GEN */

#if defined(WOLFSSL_CERT_EXT) || defined(WOLFSSL_PUB_PEM_TO_DER)
/* load pem public key from file into der buffer, return der size or error */
int wc_PemPubKeyToDer_ex(const char* fileName, DerBuffer** der)
{
#ifndef WOLFSSL_SMALL_STACK
    byte   staticBuffer[FILE_BUFFER_SIZE];
#endif
    byte*  fileBuf = NULL;
    int    dynamic = 0;
    int    ret     = 0;
    long   sz      = 0;
    XFILE  file    = XBADFILE;

    WOLFSSL_ENTER("wc_PemPubKeyToDer");

    if (fileName == NULL) {
        ret = BAD_FUNC_ARG;
    }
    else {
        file = XFOPEN(fileName, "rb");
        if (file == XBADFILE) {
            ret = IO_FAILED_E;
        }
    }

    if (ret == 0) {
        if (XFSEEK(file, 0, XSEEK_END) != 0) {
            ret = IO_FAILED_E;
        }
    }
    if (ret == 0) {
        sz = XFTELL(file);
        if (sz <= 0) {
            ret = IO_FAILED_E;
        }
    }
    if (ret == 0) {
        if (XFSEEK(file, 0, XSEEK_SET) != 0) {
            ret = IO_FAILED_E;
        }
    }
    if (ret == 0) {
#ifndef WOLFSSL_SMALL_STACK
        if (sz <= (long)sizeof(staticBuffer))
            fileBuf = staticBuffer;
        else
#endif
        {
            fileBuf = (byte*)XMALLOC((size_t)sz, NULL, DYNAMIC_TYPE_FILE);
            if (fileBuf == NULL)
                ret = MEMORY_E;
            else
                dynamic = 1;
        }
    }
    if (ret == 0) {
        if ((size_t)XFREAD(fileBuf, 1, (size_t)sz, file) != (size_t)sz) {
            ret = BUFFER_E;
        }
        else {
            ret = PemToDer(fileBuf, sz, PUBLICKEY_TYPE, der,
                           0, NULL, NULL);
        }
    }

    if (file != XBADFILE)
        XFCLOSE(file);
    if (dynamic)
        XFREE(fileBuf, NULL, DYNAMIC_TYPE_FILE);

    return ret;
}
/* load pem public key from file into der buffer, return der size or error */
int wc_PemPubKeyToDer(const char* fileName,
                           unsigned char* derBuf, int derSz)
{
    int ret;
    DerBuffer* converted = NULL;
    ret = wc_PemPubKeyToDer_ex(fileName, &converted);
    if (ret == 0) {
        if (converted->length < (word32)derSz) {
            XMEMCPY(derBuf, converted->buffer, converted->length);
            ret = (int)converted->length;
        }
        else
            ret = BUFFER_E;

        FreeDer(&converted);
    }
    return ret;
}
#endif /* WOLFSSL_CERT_EXT || WOLFSSL_PUB_PEM_TO_DER */

#endif /* !NO_FILESYSTEM && WOLFSSL_PEM_TO_DER */

/* Get public key in DER format from a populated DecodedCert struct.
 *
 * Users must call wc_InitDecodedCert() and wc_ParseCert() before calling
 * this API. wc_InitDecodedCert() accepts a DER/ASN.1 encoded certificate.
 * To convert a PEM cert to DER first use wc_CertPemToDer() before calling
 * wc_InitDecodedCert().
 *
 * cert   - populated DecodedCert struct holding X.509 certificate
 * derKey - output buffer to place DER/ASN.1 encoded public key
 * derKeySz [IN/OUT] - size of derKey buffer on input, size of public key
 *                     on return. If derKey is passed in as NULL, derKeySz
 *                     will be set to required buffer size for public key
 *                     and LENGTH_ONLY_E will be returned from function.
 * Returns 0 on success, or negative error code on failure. LENGTH_ONLY_E
 * if derKey is NULL and returning length only.
 */
int wc_GetPubKeyDerFromCert(struct DecodedCert* cert,
                            byte* derKey, word32* derKeySz)
{
    int ret = 0;

    /* derKey may be NULL to return length only */
    if (cert == NULL || derKeySz == NULL ||
        (derKey != NULL && *derKeySz == 0)) {
        return BAD_FUNC_ARG;
    }

    if (cert->publicKey == NULL) {
        WOLFSSL_MSG("DecodedCert does not contain public key\n");
        return BAD_FUNC_ARG;
    }

    /* if derKey is NULL, return required output buffer size in derKeySz */
    if (derKey == NULL) {
        *derKeySz = cert->pubKeySize;
        ret = WC_NO_ERR_TRACE(LENGTH_ONLY_E);
    }

    if (ret == 0) {
        if (cert->pubKeySize > *derKeySz) {
            WOLFSSL_MSG("Output buffer not large enough for public key DER");
            ret = BAD_FUNC_ARG;
        }
        else {
            XMEMCPY(derKey, cert->publicKey, cert->pubKeySize);
            *derKeySz = cert->pubKeySize;
        }
    }

    return ret;
}

#ifdef WOLFSSL_FPKI
/* Search through list for first matching alt name of the same type
 * If 'current' is null then the search starts at the head of the list
 * otherwise the search starts from the node after 'current' alt name.
 * Returns 0 on success
 */
static DNS_entry* FindAltName(struct DecodedCert* cert, int nameType,
    DNS_entry* current)
{
    DNS_entry* entry;

    if (current == NULL) {
        entry = cert->altNames;
    }
    else {
        entry = current->next;
    }

    /* cycle through alt names to check for needed types */
    while (entry != NULL) {
        if (entry->type == nameType) {
            break;
        }
        entry = entry->next;
    }

    return entry;
}


/* returns 0 on success */
int wc_GetUUIDFromCert(struct DecodedCert* cert, byte* uuid, word32* uuidSz)
{
    int ret = WC_NO_ERR_TRACE(ALT_NAME_E);
    DNS_entry* id = NULL;

    do {
        id = FindAltName(cert, ASN_URI_TYPE, id);
        if (id != NULL) {
            /* check if URI string matches expected format for UUID */
            if (id->len != DEFAULT_UUID_SZ) {
                continue; /* size not right not a UUID URI */
            }

            if (XMEMCMP(id->name, "urn:uuid:", 9) != 0) {
                continue; /* beginning text not right for a UUID URI */
            }

            if (uuid == NULL) {
                *uuidSz = (word32)id->len;
                return WC_NO_ERR_TRACE(LENGTH_ONLY_E);
            }

            if ((int)*uuidSz < id->len) {
                return BUFFER_E;
            }

            XMEMCPY(uuid, id->name, (size_t)id->len);
            ret = 0; /* success */
            break;
        }
    } while (id != NULL);

    return ret;
}


/* returns 0 on success */
int wc_GetFASCNFromCert(struct DecodedCert* cert, byte* fascn, word32* fascnSz)
{
    int ret = WC_NO_ERR_TRACE(ALT_NAME_E);
    DNS_entry* id = NULL;

    do {
        id = FindAltName(cert, ASN_OTHER_TYPE, id);
        if (id != NULL && id->oidSum == FASCN_OID) {
            if (fascn == NULL) {
                *fascnSz = (word32)id->len;
                return WC_NO_ERR_TRACE(LENGTH_ONLY_E);
            }

            if ((int)*fascnSz < id->len) {
                return BUFFER_E;
            }

            XMEMCPY(fascn, id->name, (size_t)id->len);
            ret = 0; /* success */
        }
    } while (id != NULL);

    return ret;
}
#endif /* WOLFSSL_FPKI */

#if !defined(NO_RSA) && \
    (defined(WOLFSSL_KEY_TO_DER) || defined(WOLFSSL_CERT_GEN))
/* USER RSA ifdef portions used instead of refactor in consideration for
   possible fips build */
/* Encode a public RSA key to output.
 *
 * X.509: RFC 5280, 4.1 - SubjectPublicKeyInfo
 * PKCS #1: RFC 8017, A.1.1 - RSAPublicKey
 *
 * Encoded data can either be SubjectPublicKeyInfo (with header) or just the key
 * (RSAPublicKey).
 *
 * @param [out] output       Buffer to put encoded data in.
 * @param [in]  key          RSA key object.
 * @param [in]  outLen       Size of the output buffer in bytes.
 * @param [in]  with_header  Whether to include SubjectPublicKeyInfo around key.
 * @return  Size of encoded data in bytes on success.
 * @return  BAD_FUNC_ARG when output or key is NULL, or outLen is less than
 *          minimum length (5 bytes).
 * @return  MEMORY_E when dynamic memory allocation failed.
 */
static int SetRsaPublicKey(byte* output, RsaKey* key, int outLen,
                           int with_header)
{
#ifndef WOLFSSL_ASN_TEMPLATE
    int  nSz, eSz;
    word32 seqSz, algoSz = 0, headSz = 0, bitStringSz = 0, idx;
    byte seq[MAX_SEQ_SZ];
    byte headSeq[MAX_SEQ_SZ];
    byte bitString[1 + MAX_LENGTH_SZ + 1];
    byte algo[MAX_ALGO_SZ]; /* 20 bytes */

    if (key == NULL) {
        return BAD_FUNC_ARG;
    }

    nSz = SetASNIntMP(&key->n, MAX_RSA_INT_SZ, NULL);

    if (nSz < 0)
        return nSz;

    eSz = SetASNIntMP(&key->e, MAX_RSA_INT_SZ, NULL);

    if (eSz < 0)
        return eSz;
    seqSz = SetSequence((word32)(nSz + eSz), seq);

    /* headers */
    if (with_header) {
        algoSz = SetAlgoID(RSAk, algo, oidKeyType, 0);
        bitStringSz = SetBitString(seqSz + (word32)(nSz + eSz), 0, bitString);
        headSz = SetSequence((word32)(nSz + eSz) + seqSz + bitStringSz + algoSz,
                             headSeq);
    }

    /* if getting length only */
    if (output == NULL) {
        return (int)(headSz + algoSz + bitStringSz + seqSz) + nSz + eSz;
    }

    /* check output size */
    if (((int)(headSz + algoSz + bitStringSz + seqSz) + nSz + eSz) > outLen) {
        return BUFFER_E;
    }

    /* write output */
    idx = 0;
    if (with_header) {
        /* header size */
        XMEMCPY(output + idx, headSeq, headSz);
        idx += headSz;
        /* algo */
        XMEMCPY(output + idx, algo, algoSz);
        idx += algoSz;
        /* bit string */
        XMEMCPY(output + idx, bitString, bitStringSz);
        idx += bitStringSz;
    }

    /* seq */
    XMEMCPY(output + idx, seq, seqSz);
    idx += seqSz;
    /* n */
    nSz = SetASNIntMP(&key->n, nSz, output + idx);
    idx += (word32)nSz;
    /* e */
    eSz = SetASNIntMP(&key->e, eSz, output + idx);
    idx += (word32)eSz;

    return (int)idx;
#else
    DECL_ASNSETDATA(dataASN, rsaPublicKeyASN_Length);
    int sz = 0;
    int ret = 0;
    int o = 0;

    /* Check parameter validity. */
    if ((key == NULL) || ((output != NULL) && (outLen < MAX_SEQ_SZ))) {
        ret = BAD_FUNC_ARG;
    }

    CALLOC_ASNSETDATA(dataASN, rsaPublicKeyASN_Length, ret, key->heap);

    if (ret == 0) {
        if (!with_header) {
            /* Start encoding with items after header. */
            o = RSAPUBLICKEYASN_IDX_PUBKEY_RSA_SEQ;
        }
        /* Set OID for RSA key. */
        SetASN_OID(&dataASN[RSAPUBLICKEYASN_IDX_ALGOID_OID], RSAk, oidKeyType);
    #ifdef WC_RSA_PSS
        dataASN[RSAPUBLICKEYASN_IDX_ALGOID_P_SEQ].noOut = 1;
    #endif
        /* Set public key mp_ints. */
        SetASN_MP(&dataASN[RSAPUBLICKEYASN_IDX_PUBKEY_RSA_N], &key->n);
        SetASN_MP(&dataASN[RSAPUBLICKEYASN_IDX_PUBKEY_RSA_E], &key->e);
        /* Calculate size of RSA public key. */
        ret = SizeASN_Items(rsaPublicKeyASN + o, dataASN + o,
                            (int)rsaPublicKeyASN_Length - o, &sz);
    }
    /* Check output buffer is big enough for encoding. */
    if ((ret == 0) && (output != NULL) && (sz > outLen)) {
        ret = BUFFER_E;
    }
    if ((ret == 0) && (output != NULL)) {
        /* Encode RSA public key. */
        SetASN_Items(rsaPublicKeyASN + o, dataASN + o,
                     (int)rsaPublicKeyASN_Length - o, output);
    }
    if (ret == 0) {
        /* Return size of encoding. */
        ret = sz;
    }

    FREE_ASNSETDATA(dataASN, key->heap);
    return ret;
#endif /* WOLFSSL_ASN_TEMPLATE */
}

/* Calculate size of encoded public RSA key in bytes.
 *
 * X.509: RFC 5280, 4.1 - SubjectPublicKeyInfo
 * PKCS #1: RFC 8017, A.1.1 - RSAPublicKey
 *
 * Encoded data can either be SubjectPublicKeyInfo (with header) or just the key
 * (RSAPublicKey).
 *
 * @param [in]  key          RSA key object.
 * @param [in]  with_header  Whether to include SubjectPublicKeyInfo around key.
 * @return  Size of encoded data in bytes on success.
 * @return  BAD_FUNC_ARG when key is NULL.
 * @return  MEMORY_E when dynamic memory allocation failed.
 */
int wc_RsaPublicKeyDerSize(RsaKey* key, int with_header)
{
    return SetRsaPublicKey(NULL, key, 0, with_header);
}

/* Encode public RSA key in DER format.
 *
 * X.509: RFC 5280, 4.1 - SubjectPublicKeyInfo
 * PKCS #1: RFC 8017, A.1.1 - RSAPublicKey
 *
 * @param [in]  key     RSA key object.
 * @param [out] output  Buffer to put encoded data in.
 * @param [in]  inLen   Size of buffer in bytes.
 * @return  Size of encoded data in bytes on success.
 * @return  BAD_FUNC_ARG when key or output is NULL.
 * @return  MEMORY_E when dynamic memory allocation failed.
 */
int wc_RsaKeyToPublicDer(RsaKey* key, byte* output, word32 inLen)
{
    return SetRsaPublicKey(output, key, (int)inLen, 1);
}

/* Returns public DER version of the RSA key. If with_header is 0 then only a
 * seq + n + e is returned in ASN.1 DER format */
int wc_RsaKeyToPublicDer_ex(RsaKey* key, byte* output, word32 inLen,
    int with_header)
{
    return SetRsaPublicKey(output, key, (int)inLen, with_header);
}

#endif /* !NO_RSA && WOLFSSL_KEY_TO_DER */
#endif /* NO_CERTS */

#if !defined(NO_RSA) && defined(WOLFSSL_KEY_TO_DER)

/* Encode private RSA key in DER format.
 *
 * PKCS #1: RFC 8017, A.1.2 - RSAPrivateKey
 *
 * @param [in]  key     RSA key object.
 * @param [out] output  Buffer to put encoded data in.
 * @param [in]  inLen   Size of buffer in bytes.
 * @return  Size of encoded data in bytes on success.
 * @return  BAD_FUNC_ARG when key is NULL or not a private key.
 * @return  MEMORY_E when dynamic memory allocation failed.
 */
int wc_RsaKeyToDer(RsaKey* key, byte* output, word32 inLen)
{
#ifndef WOLFSSL_ASN_TEMPLATE
    int ret = 0, i;
    int mpSz;
    word32 seqSz = 0, verSz = 0, intTotalLen = 0, outLen = 0;
    byte  seq[MAX_SEQ_SZ];
    byte  ver[MAX_VERSION_SZ];
    mp_int* keyInt;
#ifndef WOLFSSL_NO_MALLOC
    word32 rawLen;
    byte* tmps[RSA_INTS];
    word32 sizes[RSA_INTS];
#endif

    if (key == NULL)
        return BAD_FUNC_ARG;

    if (key->type != RSA_PRIVATE)
        return BAD_FUNC_ARG;

#ifndef WOLFSSL_NO_MALLOC
    for (i = 0; i < RSA_INTS; i++)
        tmps[i] = NULL;
#endif

    /* write all big ints from key to DER tmps */
    for (i = 0; i < RSA_INTS; i++) {
        keyInt = GetRsaInt(key, i);
        ret = mp_unsigned_bin_size(keyInt);
        if (ret < 0)
            break;
#ifndef WOLFSSL_NO_MALLOC
        rawLen = (word32)ret + 1;
        ret = 0;
        if (output != NULL) {
            tmps[i] = (byte*)XMALLOC(rawLen + MAX_SEQ_SZ, key->heap,
                                 DYNAMIC_TYPE_RSA);
            if (tmps[i] == NULL) {
                ret = MEMORY_E;
                break;
            }
        }
        mpSz = SetASNIntMP(keyInt, MAX_RSA_INT_SZ, tmps[i]);
#else
        ret = 0;
        mpSz = SetASNIntMP(keyInt, MAX_RSA_INT_SZ, NULL);
#endif
        if (mpSz < 0) {
            ret = mpSz;
            break;
        }
    #ifndef WOLFSSL_NO_MALLOC
        sizes[i] = (word32)mpSz;
    #endif
        intTotalLen += (word32)mpSz;
    }

    if (ret == 0) {
        /* make headers */
        ret = SetMyVersion(0, ver, FALSE);
    }

    if (ret >= 0) {
        verSz = (word32)ret;
        ret = 0;
        seqSz = SetSequence(verSz + intTotalLen, seq);
        outLen = seqSz + verSz + intTotalLen;
        if (output != NULL && outLen > inLen)
            ret = BUFFER_E;
    }
    if (ret == 0 && output != NULL) {
        word32 j;

        /* write to output */
        XMEMCPY(output, seq, seqSz);
        j = seqSz;
        XMEMCPY(output + j, ver, verSz);
        j += verSz;

        for (i = 0; i < RSA_INTS; i++) {
/* copy from tmps if we have malloc, otherwise re-export with buffer */
#ifndef WOLFSSL_NO_MALLOC
            XMEMCPY(output + j, tmps[i], sizes[i]);
            j += sizes[i];
#else
            keyInt = GetRsaInt(key, i);
            ret = mp_unsigned_bin_size(keyInt);
            if (ret < 0)
                break;
            ret = 0;
            /* This won't overrun output due to the outLen check above */
            mpSz = SetASNIntMP(keyInt, MAX_RSA_INT_SZ, output + j);
            if (mpSz < 0) {
                ret = mpSz;
                break;
            }
            j += mpSz;
#endif
        }
    }

#ifndef WOLFSSL_NO_MALLOC
    for (i = 0; i < RSA_INTS; i++) {
        if (tmps[i])
            XFREE(tmps[i], key->heap, DYNAMIC_TYPE_RSA);
    }
#endif

    if (ret == 0)
        ret = (int)outLen;
    return ret;
#else
    DECL_ASNSETDATA(dataASN, rsaKeyASN_Length);
    int i;
    int sz = 0;
    int ret = 0;

    if ((key == NULL) || (key->type != RSA_PRIVATE)) {
        ret = BAD_FUNC_ARG;
    }

    CALLOC_ASNSETDATA(dataASN, rsaKeyASN_Length, ret, key->heap);

    if (ret == 0) {
        /* Set the version. */
        SetASN_Int8Bit(&dataASN[RSAKEYASN_IDX_VER], 0);
        /* Set all the mp_ints in private key. */
        for (i = 0; i < RSA_INTS; i++) {
            SetASN_MP(&dataASN[(byte)RSAKEYASN_IDX_N + i], GetRsaInt(key, i));
        }

        /* Calculate size of RSA private key encoding. */
        ret = SizeASN_Items(rsaKeyASN, dataASN, rsaKeyASN_Length, &sz);
    }
    /* Check output buffer has enough space for encoding. */
    if ((ret == 0) && (output != NULL) && (sz > (int)inLen)) {
        ret = BAD_FUNC_ARG;
    }
    if ((ret == 0) && (output != NULL)) {
        /* Encode RSA private key. */
        SetASN_Items(rsaKeyASN, dataASN, rsaKeyASN_Length, output);
    }

    if (ret == 0) {
        /* Return size of encoding. */
        ret = sz;
    }

    FREE_ASNSETDATA(dataASN, key->heap);
    return ret;
#endif
}

#endif /* !NO_RSA && WOLFSSL_KEY_TO_DER */

#ifndef NO_CERTS

#ifdef WOLFSSL_CERT_GEN

/* Initialize and Set Certificate defaults:
   version    = 3 (0x2)
   serial     = 0
   sigType    = SHA_WITH_RSA
   issuer     = blank
   daysValid  = 500
   selfSigned = 1 (true) use subject as issuer
   subject    = blank
*/
int wc_InitCert_ex(Cert* cert, void* heap, int devId)
{
#ifdef WOLFSSL_MULTI_ATTRIB
    int i = 0;
#endif
    if (cert == NULL) {
        return BAD_FUNC_ARG;
    }

    XMEMSET(cert, 0, sizeof(Cert));

    cert->version    = 2;   /* version 3 is hex 2 */
#ifndef NO_SHA
    cert->sigType    = CTC_SHAwRSA;
#elif !defined(NO_SHA256)
    cert->sigType    = CTC_SHA256wRSA;
#else
    cert->sigType    = 0;
#endif
    cert->daysValid  = 500;
    cert->selfSigned = 1;
    cert->keyType    = RSA_KEY;

    cert->issuer.countryEnc = CTC_PRINTABLE;
    cert->issuer.stateEnc = CTC_UTF8;
    cert->issuer.streetEnc = CTC_UTF8;
    cert->issuer.localityEnc = CTC_UTF8;
    cert->issuer.surEnc = CTC_UTF8;
#ifdef WOLFSSL_CERT_NAME_ALL
    cert->issuer.givenNameEnc = CTC_UTF8;
    cert->issuer.initialsEnc = CTC_UTF8;
    cert->issuer.dnQualifierEnc = CTC_UTF8;
    cert->issuer.dnNameEnc = CTC_UTF8;
#endif
    cert->issuer.orgEnc = CTC_UTF8;
    cert->issuer.unitEnc = CTC_UTF8;
    cert->issuer.commonNameEnc = CTC_UTF8;
    cert->issuer.serialDevEnc = CTC_PRINTABLE;
    cert->issuer.userIdEnc = CTC_UTF8;
    cert->issuer.postalCodeEnc = CTC_UTF8;
#ifdef WOLFSSL_CERT_EXT
    cert->issuer.busCatEnc = CTC_UTF8;
    cert->issuer.joiCEnc = CTC_UTF8;
    cert->issuer.joiStEnc = CTC_UTF8;
#endif

    cert->subject.countryEnc = CTC_PRINTABLE;
    cert->subject.stateEnc = CTC_UTF8;
    cert->subject.streetEnc = CTC_UTF8;
    cert->subject.localityEnc = CTC_UTF8;
    cert->subject.surEnc = CTC_UTF8;
#ifdef WOLFSSL_CERT_NAME_ALL
    cert->subject.givenNameEnc = CTC_UTF8;
    cert->subject.initialsEnc = CTC_UTF8;
    cert->subject.dnQualifierEnc = CTC_UTF8;
    cert->subject.dnNameEnc = CTC_UTF8;
#endif
    cert->subject.orgEnc = CTC_UTF8;
    cert->subject.unitEnc = CTC_UTF8;
    cert->subject.commonNameEnc = CTC_UTF8;
    cert->subject.serialDevEnc = CTC_PRINTABLE;
    cert->subject.userIdEnc = CTC_UTF8;
    cert->subject.postalCodeEnc = CTC_UTF8;
#ifdef WOLFSSL_CERT_EXT
    cert->subject.busCatEnc = CTC_UTF8;
    cert->subject.joiCEnc = CTC_UTF8;
    cert->subject.joiStEnc = CTC_UTF8;
#endif

#ifdef WOLFSSL_MULTI_ATTRIB
    for (i = 0; i < CTC_MAX_ATTRIB; i++) {
        cert->issuer.name[i].type   = CTC_UTF8;
        cert->subject.name[i].type  = CTC_UTF8;
    }
#endif /* WOLFSSL_MULTI_ATTRIB */

    cert->heap = heap;
    (void)devId; /* future */

    return 0;
}

WOLFSSL_ABI
int wc_InitCert(Cert* cert)
{
    return wc_InitCert_ex(cert, NULL, INVALID_DEVID);
}

WOLFSSL_ABI
Cert* wc_CertNew(void* heap)
{
    Cert* certNew;

    certNew = (Cert*)XMALLOC(sizeof(Cert), heap, DYNAMIC_TYPE_CERT);

    if (certNew) {
        if (wc_InitCert_ex(certNew, heap, INVALID_DEVID) != 0) {
            XFREE(certNew, heap, DYNAMIC_TYPE_CERT);
            certNew = NULL;
        }
    }

    return certNew;
}

WOLFSSL_ABI
void  wc_CertFree(Cert* cert)
{
    if (cert) {
         void* heap = cert->heap;

         ForceZero(cert, sizeof(Cert));
         XFREE(cert, heap, DYNAMIC_TYPE_CERT);
         (void)heap;
     }
}

/* DER encoded x509 Certificate */
typedef struct DerCert {
    byte size[MAX_LENGTH_SZ];          /* length encoded */
    byte version[MAX_VERSION_SZ];      /* version encoded */
    byte serial[(int)CTC_SERIAL_SIZE + (int)MAX_LENGTH_SZ]; /* serial number encoded */
    byte sigAlgo[MAX_ALGO_SZ];         /* signature algo encoded */
    byte issuer[WC_ASN_NAME_MAX];         /* issuer  encoded */
    byte subject[WC_ASN_NAME_MAX];        /* subject encoded */
    byte validity[MAX_DATE_SIZE*2 + MAX_SEQ_SZ*2];  /* before and after dates */
    byte publicKey[MAX_PUBLIC_KEY_SZ]; /* rsa public key encoded */
    byte ca[MAX_CA_SZ];                /* basic constraint CA true size */
    byte extensions[MAX_EXTENSIONS_SZ]; /* all extensions */
#ifdef WOLFSSL_CERT_EXT
    byte skid[MAX_KID_SZ];             /* Subject Key Identifier extension */
    byte akid[MAX_KID_SZ
#ifdef WOLFSSL_AKID_NAME
              + sizeof(CertName) + CTC_SERIAL_SIZE
#endif
              ]; /* Authority Key Identifier extension */
    byte keyUsage[MAX_KEYUSAGE_SZ];    /* Key Usage extension */
    byte extKeyUsage[MAX_EXTKEYUSAGE_SZ]; /* Extended Key Usage extension */
#ifndef IGNORE_NETSCAPE_CERT_TYPE
    byte nsCertType[MAX_NSCERTTYPE_SZ]; /* Extended Key Usage extension */
#endif
    byte certPolicies[MAX_CERTPOL_NB*MAX_CERTPOL_SZ]; /* Certificate Policies */
    byte crlInfo[CTC_MAX_CRLINFO_SZ];  /* CRL Distribution Points */
#endif
#ifdef WOLFSSL_CERT_REQ
    byte attrib[MAX_ATTRIB_SZ];        /* Cert req attributes encoded */
    #ifdef WOLFSSL_CUSTOM_OID
    byte extCustom[MAX_ATTRIB_SZ];     /* Encoded user oid and value */
    #endif
#endif
#ifdef WOLFSSL_ALT_NAMES
    byte altNames[CTC_MAX_ALT_SIZE];   /* Alternative Names encoded */
#endif
    int  sizeSz;                       /* encoded size length */
    int  versionSz;                    /* encoded version length */
    int  serialSz;                     /* encoded serial length */
    int  sigAlgoSz;                    /* encoded sig algo length */
    int  issuerSz;                     /* encoded issuer length */
    int  subjectSz;                    /* encoded subject length */
    int  validitySz;                   /* encoded validity length */
    int  publicKeySz;                  /* encoded public key length */
    int  caSz;                         /* encoded CA extension length */
#ifdef WOLFSSL_CERT_EXT
    int  skidSz;                       /* encoded SKID extension length */
    int  akidSz;                       /* encoded SKID extension length */
    int  keyUsageSz;                   /* encoded KeyUsage extension length */
    int  extKeyUsageSz;                /* encoded ExtendedKeyUsage extension length */
#ifndef IGNORE_NETSCAPE_CERT_TYPE
    int  nsCertTypeSz;                 /* encoded Netscape Certificate Type
                                        * extension length */
#endif
    int  certPoliciesSz;               /* encoded CertPolicies extension length*/
    int  crlInfoSz;                    /* encoded CRL Dist Points length */
#endif
#ifdef WOLFSSL_ALT_NAMES
    int  altNamesSz;                   /* encoded AltNames extension length */
#endif
    int  extensionsSz;                 /* encoded extensions total length */
    int  total;                        /* total encoded lengths */
#ifdef WOLFSSL_CERT_REQ
    int  attribSz;
    #ifdef WOLFSSL_CUSTOM_OID
    int  extCustomSz;
    #endif
#endif
} DerCert;


#ifdef WOLFSSL_CERT_REQ
#ifndef WOLFSSL_ASN_TEMPLATE

/* Write a set header to output */
static word32 SetPrintableString(word32 len, byte* output)
{
    output[0] = ASN_PRINTABLE_STRING;
    return SetLength(len, output + 1) + 1;
}

static word32 SetUTF8String(word32 len, byte* output)
{
    output[0] = ASN_UTF8STRING;
    return SetLength(len, output + 1) + 1;
}

#endif
#endif /* WOLFSSL_CERT_REQ */


#ifndef WOLFSSL_CERT_GEN_CACHE
/* wc_SetCert_Free is only public when WOLFSSL_CERT_GEN_CACHE is not defined */
static
#endif
WOLFSSL_ABI
void wc_SetCert_Free(Cert* cert)
{
    if (cert != NULL) {
        cert->der = NULL;
        if (cert->decodedCert) {
            FreeDecodedCert((DecodedCert*)cert->decodedCert);

            XFREE(cert->decodedCert, cert->heap, DYNAMIC_TYPE_DCERT);
            cert->decodedCert = NULL;
        }
    }
}

static int wc_SetCert_LoadDer(Cert* cert, const byte* der, word32 derSz,
    int devId)
{
    int ret;

    if (cert == NULL) {
        ret = BAD_FUNC_ARG;
    }
    else {
        /* Allocate DecodedCert struct and Zero */
        cert->decodedCert = (void*)XMALLOC(sizeof(DecodedCert), cert->heap,
            DYNAMIC_TYPE_DCERT);

        if (cert->decodedCert == NULL) {
            ret = MEMORY_E;
        }
        else {
            XMEMSET(cert->decodedCert, 0, sizeof(DecodedCert));

            InitDecodedCert_ex((DecodedCert*)cert->decodedCert, der, derSz,
                    cert->heap, devId);
            ret = ParseCertRelative((DecodedCert*)cert->decodedCert,
                    CERT_TYPE, 0, NULL, NULL);
            if (ret >= 0) {
                cert->der = (byte*)der;
            }
            else {
                wc_SetCert_Free(cert);
            }
        }
    }

    return ret;
}

#endif /* WOLFSSL_CERT_GEN */

#ifdef WOLFSSL_CERT_GEN

#ifndef NO_ASN_TIME
static WC_INLINE byte itob(int number)
{
    return (byte)(number + 0x30);
}


/* write time to output, format */
static void SetTime(struct tm* date, byte* output)
{
    int i = 0;

    output[i++] = itob((date->tm_year % 10000) / 1000);
    output[i++] = itob((date->tm_year % 1000)  /  100);
    output[i++] = itob((date->tm_year % 100)   /   10);
    output[i++] = itob( date->tm_year % 10);

    output[i++] = itob(date->tm_mon / 10);
    output[i++] = itob(date->tm_mon % 10);

    output[i++] = itob(date->tm_mday / 10);
    output[i++] = itob(date->tm_mday % 10);

    output[i++] = itob(date->tm_hour / 10);
    output[i++] = itob(date->tm_hour % 10);

    output[i++] = itob(date->tm_min / 10);
    output[i++] = itob(date->tm_min % 10);

    output[i++] = itob(date->tm_sec / 10);
    output[i++] = itob(date->tm_sec % 10);

    output[i] = 'Z';  /* Zulu profile */
}
#endif

#ifndef WOLFSSL_ASN_TEMPLATE

/* Copy Dates from cert, return bytes written */
static int CopyValidity(byte* output, Cert* cert)
{
    word32 seqSz;

    WOLFSSL_ENTER("CopyValidity");

    /* headers and output */
    seqSz = SetSequence((word32)(cert->beforeDateSz + cert->afterDateSz),
                        output);
    if (output) {
        XMEMCPY(output + seqSz, cert->beforeDate, (size_t)cert->beforeDateSz);
        XMEMCPY(output + seqSz + cert->beforeDateSz, cert->afterDate,
                (size_t)cert->afterDateSz);
    }
    return (int)seqSz + cert->beforeDateSz + cert->afterDateSz;
}

#endif /* !WOLFSSL_ASN_TEMPLATE */


/* Simple name OID size. */
#define NAME_OID_SZ     3

/* Domain name OIDs. */
static const byte nameOid[][NAME_OID_SZ] = {
    { 0x55, 0x04, ASN_COUNTRY_NAME },
    { 0x55, 0x04, ASN_STATE_NAME },
    { 0x55, 0x04, ASN_STREET_ADDR },
    { 0x55, 0x04, ASN_LOCALITY_NAME },
#ifdef WOLFSSL_CERT_NAME_ALL
    { 0x55, 0x04, ASN_NAME },
    { 0x55, 0x04, ASN_GIVEN_NAME },
    { 0x55, 0x04, ASN_INITIALS },
    { 0x55, 0x04, ASN_DNQUALIFIER },
#endif
    { 0x55, 0x04, ASN_SUR_NAME },
    { 0x55, 0x04, ASN_ORG_NAME },
    { 0x00, 0x00, ASN_DOMAIN_COMPONENT}, /* not actual OID - see dcOid */
                                         /* list all DC values before OUs */
    { 0x55, 0x04, ASN_ORGUNIT_NAME },
    { 0x55, 0x04, ASN_COMMON_NAME },
    { 0x55, 0x04, ASN_SERIAL_NUMBER },
#ifdef WOLFSSL_CERT_EXT
    { 0x55, 0x04, ASN_BUS_CAT },
#endif
    { 0x55, 0x04, ASN_POSTAL_CODE },
    { 0x00, 0x00, ASN_EMAIL_NAME},       /* not actual OID - see attrEmailOid */
    { 0x00, 0x00, ASN_USER_ID},          /* not actual OID - see uidOid */
#ifdef WOLFSSL_CUSTOM_OID
    { 0x00, 0x00, ASN_CUSTOM_NAME} /* OID comes from CertOidField */
#endif
};
#define NAME_ENTRIES (int)(sizeof(nameOid)/NAME_OID_SZ)


/* Get ASN Name from index */
byte GetCertNameId(int idx)
{
    if (idx < NAME_ENTRIES)
        return nameOid[idx][2];
    return 0;
}

/* Get Which Name from index */
const char* GetOneCertName(CertName* name, int idx)
{
    byte type = GetCertNameId(idx);
    switch (type) {
    case ASN_COUNTRY_NAME:
       return name->country;
    case ASN_STATE_NAME:
       return name->state;
    case ASN_STREET_ADDR:
       return name->street;
    case ASN_LOCALITY_NAME:
       return name->locality;
#ifdef WOLFSSL_CERT_NAME_ALL
    case ASN_NAME:
       return name->dnName;
    case ASN_GIVEN_NAME:
       return name->givenName;
    case ASN_INITIALS:
       return name->initials;
    case ASN_DNQUALIFIER:
       return name->dnQualifier;
#endif /* WOLFSSL_CERT_NAME_ALL */
    case ASN_SUR_NAME:
       return name->sur;
    case ASN_ORG_NAME:
       return name->org;
    case ASN_ORGUNIT_NAME:
       return name->unit;
    case ASN_COMMON_NAME:
       return name->commonName;
    case ASN_SERIAL_NUMBER:
       return name->serialDev;
    case ASN_USER_ID:
       return name->userId;
    case ASN_POSTAL_CODE:
       return name->postalCode;
    case ASN_EMAIL_NAME:
       return name->email;
#ifdef WOLFSSL_CERT_EXT
    case ASN_BUS_CAT:
       return name->busCat;
#endif
#ifdef WOLFSSL_CUSTOM_OID
    case ASN_CUSTOM_NAME:
        return (const char*)name->custom.val;
#endif
    default:
       return NULL;
    }
}


/* Get Which Name Encoding from index */
static char GetNameType(CertName* name, int idx)
{
    byte type = GetCertNameId(idx);
    switch (type) {
    case ASN_COUNTRY_NAME:
       return name->countryEnc;
    case ASN_STATE_NAME:
       return name->stateEnc;
    case ASN_STREET_ADDR:
       return name->streetEnc;
    case ASN_LOCALITY_NAME:
       return name->localityEnc;
#ifdef WOLFSSL_CERT_NAME_ALL
    case ASN_NAME:
       return name->dnNameEnc;
    case ASN_GIVEN_NAME:
       return name->givenNameEnc;
    case ASN_INITIALS:
       return name->initialsEnc;
    case ASN_DNQUALIFIER:
       return name->dnQualifierEnc;
#endif /* WOLFSSL_CERT_NAME_ALL */
    case ASN_SUR_NAME:
       return name->surEnc;
    case ASN_ORG_NAME:
       return name->orgEnc;
    case ASN_ORGUNIT_NAME:
       return name->unitEnc;
    case ASN_COMMON_NAME:
       return name->commonNameEnc;
    case ASN_SERIAL_NUMBER:
       return name->serialDevEnc;
    case ASN_USER_ID:
       return name->userIdEnc;
    case ASN_POSTAL_CODE:
       return name->postalCodeEnc;
    case ASN_EMAIL_NAME:
       return 0; /* special */
#ifdef WOLFSSL_CERT_EXT
    case ASN_BUS_CAT:
       return name->busCatEnc;
#endif
#ifdef WOLFSSL_CUSTOM_OID
    case ASN_CUSTOM_NAME:
        return name->custom.enc;
#endif
    default:
       return 0;
    }
}

#ifndef WOLFSSL_ASN_TEMPLATE
/*
 Extensions ::= SEQUENCE OF Extension

 Extension ::= SEQUENCE {
 extnId     OBJECT IDENTIFIER,
 critical   BOOLEAN DEFAULT FALSE,
 extnValue  OCTET STRING }
 */

/* encode all extensions, return total bytes written */
static int SetExtensions(byte* out, word32 outSz, int *IdxInOut,
                         const byte* ext, int extSz)
{
    if (out == NULL || IdxInOut == NULL || ext == NULL)
        return BAD_FUNC_ARG;

    if (outSz < (word32)(*IdxInOut+extSz))
        return BUFFER_E;

    XMEMCPY(&out[*IdxInOut], ext, (size_t)extSz);  /* extensions */
    *IdxInOut += extSz;

    return *IdxInOut;
}

/* encode extensions header, return total bytes written */
static int SetExtensionsHeader(byte* out, word32 outSz, word32 extSz)
{
    byte sequence[MAX_SEQ_SZ];
    byte len[MAX_LENGTH_SZ];
    word32 seqSz, lenSz, idx = 0;

    if (out == NULL)
        return BAD_FUNC_ARG;

    if (outSz < 3)
        return BUFFER_E;

    seqSz = SetSequence(extSz, sequence);

    /* encode extensions length provided */
    lenSz = SetLength(extSz+seqSz, len);

    if (outSz < (word32)(lenSz+seqSz+1))
        return BUFFER_E;

    out[idx++] = ASN_EXTENSIONS; /* extensions id */
    XMEMCPY(&out[idx], len, lenSz);  /* length */
    idx += lenSz;

    XMEMCPY(&out[idx], sequence, seqSz);  /* sequence */
    idx += seqSz;

    return (int)idx;
}


/* encode CA basic constraints true with path length
 * return total bytes written */
static int SetCaWithPathLen(byte* out, word32 outSz, byte pathLen)
{
    /* ASN1->DER sequence for Basic Constraints True and path length */
    const byte caPathLenBasicConstASN1[] = {
        0x30, 0x0F, 0x06, 0x03, 0x55, 0x1D, 0x13, 0x04,
        0x08, 0x30, 0x06, 0x01, 0x01, 0xFF, 0x02, 0x01,
        0x00
    };

    if (out == NULL)
        return BAD_FUNC_ARG;

    if (outSz < sizeof(caPathLenBasicConstASN1))
        return BUFFER_E;

    XMEMCPY(out, caPathLenBasicConstASN1, sizeof(caPathLenBasicConstASN1));

    out[sizeof(caPathLenBasicConstASN1)-1] = pathLen;

    return (int)sizeof(caPathLenBasicConstASN1);
}

/* encode CA basic constraints
 * return total bytes written */
static int SetCaEx(byte* out, word32 outSz, byte isCa)
{
    /* ASN1->DER sequence for Basic Constraints True */
    const byte caBasicConstASN1[] = {
        0x30, 0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x04,
        0x05, 0x30, 0x03, 0x01, 0x01, 0xff
    };

    if (out == NULL)
        return BAD_FUNC_ARG;

    if (outSz < sizeof(caBasicConstASN1))
        return BUFFER_E;

    XMEMCPY(out, caBasicConstASN1, sizeof(caBasicConstASN1));

    if (!isCa) {
        out[sizeof(caBasicConstASN1)-1] = isCa;
    }

    return (int)sizeof(caBasicConstASN1);
}

/* encode CA basic constraints true
 * return total bytes written */
static int SetCa(byte* out, word32 outSz)
{
    return SetCaEx(out, outSz, 1);
}

/* encode basic constraints without CA Boolean
 * return total bytes written */
static int SetBC(byte* out, word32 outSz)
{
    /* ASN1->DER sequence for Basic Constraint without CA Boolean */
 const byte BasicConstASN1[] = {
        0x30, 0x09, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x04,
        0x02, 0x30, 0x00
    };

    if (out == NULL)
        return BAD_FUNC_ARG;

    if (outSz < sizeof(BasicConstASN1))
        return BUFFER_E;

    XMEMCPY(out, BasicConstASN1, sizeof(BasicConstASN1));

    return (int)sizeof(BasicConstASN1);
}
#endif


#ifdef WOLFSSL_CERT_EXT
#ifndef WOLFSSL_ASN_TEMPLATE
/* encode OID and associated value, return total bytes written */
static int SetOidValue(byte* out, word32 outSz, const byte *oid, word32 oidSz,
                       byte *in, word32 inSz)
{
    word32 idx = 0;

    if (out == NULL || oid == NULL || in == NULL)
        return BAD_FUNC_ARG;

    if (outSz < 3)
        return BUFFER_E;

    /* sequence,  + 1 => byte to put value size */
    idx = SetSequence(inSz + oidSz + 1, out);

    if ((idx + inSz + oidSz + 1) > outSz)
        return BUFFER_E;

    XMEMCPY(out+idx, oid, oidSz);
    idx += oidSz;
    out[idx++] = (byte)inSz;
    XMEMCPY(out+idx, in, inSz);

    return (int)(idx+inSz);
}

/* encode Subject Key Identifier, return total bytes written
 * RFC5280 : non-critical */
static int SetSKID(byte* output, word32 outSz, const byte *input, word32 length)
{
    byte skid_len[1 + MAX_LENGTH_SZ];
    byte skid_enc_len[MAX_LENGTH_SZ];
    word32 idx = 0, skid_lenSz, skid_enc_lenSz;
    const byte skid_oid[] = { 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04 };

    if (output == NULL || input == NULL)
        return BAD_FUNC_ARG;

    /* Octet String header */
    skid_lenSz = SetOctetString(length, skid_len);

    /* length of encoded value */
    skid_enc_lenSz = SetLength(length + skid_lenSz, skid_enc_len);

    if (outSz < 3)
        return BUFFER_E;

    idx = SetSequence(length + (word32)sizeof(skid_oid) + skid_lenSz +
                      skid_enc_lenSz, output);

    if ((length + sizeof(skid_oid) + skid_lenSz + skid_enc_lenSz) > outSz)
        return BUFFER_E;

    /* put oid */
    XMEMCPY(output+idx, skid_oid, sizeof(skid_oid));
    idx += sizeof(skid_oid);

    /* put encoded len */
    XMEMCPY(output+idx, skid_enc_len, skid_enc_lenSz);
    idx += skid_enc_lenSz;

    /* put octet header */
    XMEMCPY(output+idx, skid_len, skid_lenSz);
    idx += skid_lenSz;

    /* put value */
    XMEMCPY(output+idx, input, length);
    idx += length;

    return (int)idx;
}

/* encode Authority Key Identifier, return total bytes written
 * RFC5280 : non-critical */
static int SetAKID(byte* output, word32 outSz, byte *input, word32 length,
                   byte rawAkid)
{
    int     enc_valSz;
    byte enc_val_buf[MAX_KID_SZ];
    byte* enc_val;
    const byte akid_oid[] = { 0x06, 0x03, 0x55, 0x1d, 0x23 };
    const byte akid_cs[] = { 0x80 };
    word32 inSeqSz, idx;

    (void)rawAkid;

    if (output == NULL || input == NULL)
        return BAD_FUNC_ARG;

#ifdef WOLFSSL_AKID_NAME
    if (rawAkid) {
        enc_val = input;
        enc_valSz = length;
    }
    else
#endif
    {
        enc_val = enc_val_buf;
        enc_valSz = (int)length + 3 + (int)sizeof(akid_cs);
        if (enc_valSz > (int)sizeof(enc_val_buf))
            return BAD_FUNC_ARG;

        /* sequence for ContentSpec & value */
        enc_valSz = SetOidValue(enc_val, (word32)enc_valSz, akid_cs,
                                sizeof(akid_cs), input, length);
        if (enc_valSz <= 0)
            return enc_valSz;
    }

    /* The size of the extension sequence contents */
    inSeqSz = (word32)sizeof(akid_oid) +
        SetOctetString((word32)enc_valSz, NULL) + (word32)enc_valSz;

    if (SetSeq