/* BEGIN software license
 *
 * msXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright (C) 2009--2020 Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 *
 * This file is part of the msXpertSuite project.
 *
 * The msXpertSuite project is the successor of the massXpert project. This
 * project now includes various independent modules:
 *
 * - massXpert, model polymer chemistries and simulate mass spectrometric data;
 * - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */


/////////////////////// StdLib includes
#include <cstdio>

/////////////////////// Qt includes
#include <QTimer>
#include <QIcon>
#include <QDebug>
#include <QDir>
#include <QStandardPaths>
#include <QLocale>


/////////////////////// pappsomspp includes
#include <pappsomspp/core/msfile/msfilereader.h>
#include <pappsomspp/core/msfile/msfileaccessor.h>
#include <pappsomspp/core/msrun/msrunid.h>
#include <pappsomspp/core/msrun/msrunreader.h>
#include <pappsomspp/core/utils.h>
#include <pappsomspp/core/processing/combiners/tracepluscombiner.h>
#include <pappsomspp/core/processing/combiners/massspectrumpluscombiner.h>
#include <pappsomspp/core/trace/trace.h>
#include <pappsomspp/core/trace/maptrace.h>
#include <pappsomspp/core/processing/combiners/mzintegrationparams.h>


/////////////////////// Local includes
#include "Application.hpp"
#include "ProgramWindow.hpp"

namespace MsXpS
{
namespace MineXpert
{

QString Application::m_userConfigSettingsDirPath  = "";
QString Application::m_userConfigSettingsFilePath = "";

Application::Application(int &argc,
                         char **argv,
                         const QString &application_name)
  : QApplication(argc, argv), m_applicationName(application_name)
{
  // We are dealing with massive amounts of numerical data and the best locale
  // for that is "C".

  QLocale::setDefault(QLocale::C);

  QPixmap splash_pixmap(":/images/splashscreen.png");

  mpa_splash = new QSplashScreen(splash_pixmap, Qt::WindowStaysOnTopHint);
  mpa_splash->show();

  // OLD version QTimer::singleShot(2000, this, SLOT(destroySplash()));
  QTimer::singleShot(2000, this, [this]() {
    delete mpa_splash;
  });

  // We now have to perform a number of initialization steps.

  // We set these, but we do not want these to be used for the
  // QSettings config file, because we do no want that the settings
  // be stored on MS Windows in the HKEY registry. We prefer going
  // for an INI style file in a safe location.
  QCoreApplication::setOrganizationName(m_organizationName);
  QCoreApplication::setOrganizationDomain("msxpertsuite.org");
  QCoreApplication::setApplicationName(m_applicationName);

  // The configuration directory for all the settings of the program

  setUserConfigSettingsDirPath(
    QStandardPaths::writableLocation(QStandardPaths::ConfigLocation));

  if(m_userConfigSettingsDirPath.isEmpty())
    setUserConfigSettingsDirPath(
      QStandardPaths::writableLocation(QStandardPaths::HomeLocation));

  setUserConfigSettingsDirPath(getUserConfigSettingsDirPath() +
                               QDir::separator() + m_applicationName);

  setUserConfigSettingsFilePath(m_userConfigSettingsDirPath +
                                QDir::separator() + "configSettings.ini");

  // qDebug() << "m_userConfigSettingsDirPath:" <<
  // getUserConfigSettingsDirPath(); qDebug() << "m_userConfigSettingsFilePath:"
  //          << getUserConfigSettingsFilePath();

  // On Debian, QStandardPaths::ConfigLocation translates to
  // ~/.config/MineXpert3/configSettings.ini.
  // while
  // on Windows, this translates to %APPDATA%, that is:
  // C:\Users\<User>\AppData\Roaming\MineXpert3\configSettings.ini

  // We now have to initialize libXpertMassCore.
  MsXpS::libXpertMassCore::Utils::initializeConfigSettingsFilePath(
    getUserConfigSettingsFilePath());

#if 0
  // When I changed the icons of MineXpert3, the old icons stubborningly
  // remained the same green frog head. Finally, it turned out that the
  // icon was cached by the desktop ! So changing the desktop filename
  // made the new icon appear correctly.


  // rm ~/.cache/icon-cache.kcache
  // rm ~/.cache/icon-cache.kcache.* 2>/dev/null
  // rm ~/.cache/plasma-theme-kcminit*.cache 2>/dev/null
  // sudo gtk-update-icon-cache -f /usr/share/icons/hicolor
  // rm ~/.cache/kservices6/*.cache 2>/dev/null
  // kbuildsycoca6 --noincremental
  // kquitapp6 plasmashell
  // plasmashell --replace &

  // The trick below, changes the name of the desktop file each time the
  // program is rebuilt, thus forcing to acknowledge that there is a
  // non-cached icon!

  setDesktopFileName(
    QStringLiteral("minexpert3-%1").arg(QDateTime::currentSecsSinceEpoch()));
#endif

  // The default window icon.
  QIcon icon(":/images/icons/svg/MineXpert3.svg");
  setWindowIcon(icon);
}

Application::~Application()
{
}

Application *
Application::getInstance()
{
  return static_cast<Application *>(qApp);
}

void
Application::setProgramWindow(ProgramWindow *program_window_p)
{
  // qDebug() << "Setting program window pointer:" << program_window_p;

  mp_programWindow = program_window_p;
}

ProgramWindow *
Application::getProgramWindow()
{
  // qDebug() << "Returning program window pointer:" << mp_programWindow;
  return mp_programWindow;
}

QString
Application::getApplicationName()
{
  return m_applicationName;
}

void
Application::setDescription(const QString &desc)
{
  m_description = desc;
}

QString
Application::description()
{
  return m_description;
}

void
Application::setUserConfigSettingsDirPath(const QString &dir_path)
{
  m_userConfigSettingsDirPath = dir_path;
}

QString
Application::getUserConfigSettingsDirPath()
{
  return m_userConfigSettingsDirPath;
}

void
Application::setUserConfigSettingsFilePath(const QString &file_path)
{
  m_userConfigSettingsFilePath = file_path;
}

QString
Application::getUserConfigSettingsFilePath()
{
  return m_userConfigSettingsFilePath;
}

void
Application::saveMzIntegrationParamsToSettings(
  const pappso::MzIntegrationParams &mz_integration_params)
{

  QSettings settings(static_cast<Application *>(QCoreApplication::instance())
                       ->getUserConfigSettingsFilePath(),
                     QSettings::IniFormat);

  settings.beginGroup("MzIntegrationParams");

  settings.setValue("MzIntegrationParams", mz_integration_params.toString());

  settings.endGroup();
}

void
Application::eraseMzIntegrationParamsFromSettings()
{

  QSettings settings(static_cast<Application *>(QCoreApplication::instance())
                       ->getUserConfigSettingsFilePath(),
                     QSettings::IniFormat);

  settings.beginGroup("MzIntegrationParams");

  settings.remove("MzIntegrationParams");

  settings.endGroup();
}

// These settings are crucial throughout the whole program.
pappso::MzIntegrationParams::InitializationResult
Application::getMzIntegrationParamsFromSavedSettings(
  pappso::MzIntegrationParams &mz_integration_params)
{
  // qDebug();

  // Try to get the m/z integration parameters if they were stored as settings
  // by the user. Otherwise, return empty ones.

  QSettings settings(static_cast<Application *>(QCoreApplication::instance())
                       ->getUserConfigSettingsFilePath(),
                     QSettings::IniFormat);

  settings.beginGroup("MzIntegrationParams");

  QString mz_integration_params_settings_string =
    settings.value("MzIntegrationParams").toString();

  // qDebug() << "The text used to initialize the MzIntegrationParams is:"
  //          << mz_integration_params_settings_string;

  settings.endGroup();

  // Even if the string is empty, call the function that will set default
  // values.

  pappso::MzIntegrationParams::InitializationResult settings_results =
    mz_integration_params.initialize(mz_integration_params_settings_string);

  return settings_results;
}
} // namespace MineXpert
} // namespace MsXpS
