/*!
 * Matomo - free/libre analytics platform
 *
 * @link    https://matomo.org
 * @license https://www.gnu.org/licenses/gpl-3.0.html GPL v3 or later
 */
import NumberFormatter from './NumberFormatter';

const formats: any = {
  ar: {
    "patternNumber": "#,##0.###",
    "patternPercent": "#,##0%",
    "patternCurrency": "‏#,##0.00 ¤;‏-#,##0.00 ¤",
    "symbolPlus": "‎+",
    "symbolMinus": "‎-",
    "symbolPercent": "‎%‎",
    "symbolGroup": ",",
    "symbolDecimal": ".",
    "patternsCompactNumber": {
      "1000One": "0 ألف",
      "1000Other": "0 ألف",
      "10000One": "00 ألف",
      "10000Other": "00 ألف",
      "100000One": "000 ألف",
      "100000Other": "000 ألف",
      "1000000One": "0 مليون",
      "1000000Other": "0 مليون",
      "10000000One": "00 مليون",
      "10000000Other": "00 مليون",
      "100000000One": "000 مليون",
      "100000000Other": "000 مليون",
      "1000000000One": "0 مليار",
      "1000000000Other": "0 مليار",
      "10000000000One": "00 مليار",
      "10000000000Other": "00 مليار",
      "100000000000One": "000 مليار",
      "100000000000Other": "000 مليار",
      "1000000000000One": "0 ترليون",
      "1000000000000Other": "0 ترليون",
      "10000000000000One": "00 ترليون",
      "10000000000000Other": "00 ترليون",
      "100000000000000One": "000 ترليون",
      "100000000000000Other": "000 ترليون",
      "1000000000000000One": "",
      "1000000000000000Other": "",
      "10000000000000000One": "",
      "10000000000000000Other": "",
      "100000000000000000One": "",
      "100000000000000000Other": "",
      "1000000000000000000One": "",
      "1000000000000000000Other": ""
    },
    "patternsCompactCurrency": {
      "1000One": "0 ألف ¤",
      "1000Other": "0 ألف ¤",
      "10000One": "00 ألف ¤",
      "10000Other": "00 ألف ¤",
      "100000One": "000 ألف ¤",
      "100000Other": "000 ألف ¤",
      "1000000One": "0 مليون ¤",
      "1000000Other": "0 مليون ¤",
      "10000000One": "00 مليون ¤",
      "10000000Other": "00 مليون ¤",
      "100000000One": "000 مليون ¤",
      "100000000Other": "000 مليون ¤",
      "1000000000One": "0 مليار ¤",
      "1000000000Other": "0 مليار ¤",
      "10000000000One": "00 مليار ¤",
      "10000000000Other": "00 مليار ¤",
      "100000000000One": "000 مليار ¤",
      "100000000000Other": "000 مليار ¤",
      "1000000000000One": "0 ترليون ¤",
      "1000000000000Other": "0 ترليون ¤",
      "10000000000000One": "00 ترليون ¤",
      "10000000000000Other": "00 ترليون ¤",
      "100000000000000One": "000 ترليون ¤",
      "100000000000000Other": "000 ترليون ¤",
      "1000000000000000One": "",
      "1000000000000000Other": "",
      "10000000000000000One": "",
      "10000000000000000Other": "",
      "100000000000000000One": "",
      "100000000000000000Other": "",
      "1000000000000000000One": "",
      "1000000000000000000Other": ""
    }
  },
  be: {
    "patternNumber": "#,##0.###",
    "patternPercent": "#,##0 %",
    "patternCurrency": "#,##0.00 ¤",
    "symbolPlus": "+",
    "symbolMinus": "-",
    "symbolPercent": "%",
    "symbolGroup": " ",
    "symbolDecimal": ",",
    "patternsCompactNumber": {
      "1000One": "0 тыс'.'",
      "1000Other": "0 тыс'.'",
      "10000One": "00 тыс'.'",
      "10000Other": "00 тыс'.'",
      "100000One": "000 тыс'.'",
      "100000Other": "000 тыс'.'",
      "1000000One": "0 млн",
      "1000000Other": "0 млн",
      "10000000One": "00 млн",
      "10000000Other": "00 млн",
      "100000000One": "000 млн",
      "100000000Other": "000 млн",
      "1000000000One": "0 млрд",
      "1000000000Other": "0 млрд",
      "10000000000One": "00 млрд",
      "10000000000Other": "00 млрд",
      "100000000000One": "000 млрд",
      "100000000000Other": "000 млрд",
      "1000000000000One": "0 трлн",
      "1000000000000Other": "0 трлн",
      "10000000000000One": "00 трлн",
      "10000000000000Other": "00 трлн",
      "100000000000000One": "000 трлн",
      "100000000000000Other": "000 трлн",
      "1000000000000000One": "",
      "1000000000000000Other": "",
      "10000000000000000One": "",
      "10000000000000000Other": "",
      "100000000000000000One": "",
      "100000000000000000Other": "",
      "1000000000000000000One": "",
      "1000000000000000000Other": ""
    },
    "patternsCompactCurrency": {
      "1000One": "0 тыс'.' ¤",
      "1000Other": "0 тыс'.' ¤",
      "10000One": "00 тыс'.' ¤",
      "10000Other": "00 тыс'.' ¤",
      "100000One": "000 тыс'.' ¤",
      "100000Other": "000 тыс'.' ¤",
      "1000000One": "0 млн ¤",
      "1000000Other": "0 млн ¤",
      "10000000One": "00 млн ¤",
      "10000000Other": "00 млн ¤",
      "100000000One": "000 млн ¤",
      "100000000Other": "000 млн ¤",
      "1000000000One": "0 млрд ¤",
      "1000000000Other": "0 млрд ¤",
      "10000000000One": "00 млрд ¤",
      "10000000000Other": "00 млрд ¤",
      "100000000000One": "000 млрд ¤",
      "100000000000Other": "000 млрд ¤",
      "1000000000000One": "0 трлн ¤",
      "1000000000000Other": "0 трлн ¤",
      "10000000000000One": "00 трлн ¤",
      "10000000000000Other": "00 трлн ¤",
      "100000000000000One": "000 трлн ¤",
      "100000000000000Other": "000 трлн ¤",
      "1000000000000000One": "",
      "1000000000000000Other": "",
      "10000000000000000One": "",
      "10000000000000000Other": "",
      "100000000000000000One": "",
      "100000000000000000Other": "",
      "1000000000000000000One": "",
      "1000000000000000000Other": ""
    }
  },
  de: {
    "patternNumber": "#,##0.###",
    "patternPercent": "#,##0 %",
    "patternCurrency": "#,##0.00 ¤",
    "symbolPlus": "+",
    "symbolMinus": "-",
    "symbolPercent": "%",
    "symbolGroup": ".",
    "symbolDecimal": ",",
    "patternsCompactNumber": {
        "1000One": "0",
        "1000Other": "0",
        "10000One": "0",
        "10000Other": "0",
        "100000One": "0",
        "100000Other": "0",
        "1000000One": "0 Mio'.'",
        "1000000Other": "0 Mio'.'",
        "10000000One": "00 Mio'.'",
        "10000000Other": "00 Mio'.'",
        "100000000One": "000 Mio'.'",
        "100000000Other": "000 Mio'.'",
        "1000000000One": "0 Mrd'.'",
        "1000000000Other": "0 Mrd'.'",
        "10000000000One": "00 Mrd'.'",
        "10000000000Other": "00 Mrd'.'",
        "100000000000One": "000 Mrd'.'",
        "100000000000Other": "000 Mrd'.'",
        "1000000000000One": "0 Bio'.'",
        "1000000000000Other": "0 Bio'.'",
        "10000000000000One": "00 Bio'.'",
        "10000000000000Other": "00 Bio'.'",
        "100000000000000One": "000 Bio'.'",
        "100000000000000Other": "000 Bio'.'",
        "1000000000000000One": "",
        "1000000000000000Other": "",
        "10000000000000000One": "",
        "10000000000000000Other": "",
        "100000000000000000One": "",
        "100000000000000000Other": "",
        "1000000000000000000One": "",
        "1000000000000000000Other": ""
    },
    "patternsCompactCurrency": {
        "1000One": "0",
        "1000Other": "0",
        "10000One": "0",
        "10000Other": "0",
        "100000One": "0",
        "100000Other": "0",
        "1000000One": "0 Mio'.' ¤",
        "1000000Other": "0 Mio'.' ¤",
        "10000000One": "00 Mio'.' ¤",
        "10000000Other": "00 Mio'.' ¤",
        "100000000One": "000 Mio'.' ¤",
        "100000000Other": "000 Mio'.' ¤",
        "1000000000One": "0 Mrd'.' ¤",
        "1000000000Other": "0 Mrd'.' ¤",
        "10000000000One": "00 Mrd'.' ¤",
        "10000000000Other": "00 Mrd'.' ¤",
        "100000000000One": "000 Mrd'.' ¤",
        "100000000000Other": "000 Mrd'.' ¤",
        "1000000000000One": "0 Bio'.' ¤",
        "1000000000000Other": "0 Bio'.' ¤",
        "10000000000000One": "00 Bio'.' ¤",
        "10000000000000Other": "00 Bio'.' ¤",
        "100000000000000One": "000 Bio'.' ¤",
        "100000000000000Other": "000 Bio'.' ¤",
        "1000000000000000One": "",
        "1000000000000000Other": "",
        "10000000000000000One": "",
        "10000000000000000Other": "",
        "100000000000000000One": "",
        "100000000000000000Other": "",
        "1000000000000000000One": "",
        "1000000000000000000Other": ""
    }
},
  en: {
    "patternNumber": "#,##0.###",
    "patternPercent": "#,##0%",
    "patternCurrency": "¤#,##0.00",
    "symbolPlus": "+",
    "symbolMinus": "-",
    "symbolPercent": "%",
    "symbolGroup": ",",
    "symbolDecimal": ".",
    "patternsCompactNumber": {
      "1000One": "0K",
      "1000Other": "0K",
      "10000One": "00K",
      "10000Other": "00K",
      "100000One": "000K",
      "100000Other": "000K",
      "1000000One": "0M",
      "1000000Other": "0M",
      "10000000One": "00M",
      "10000000Other": "00M",
      "100000000One": "000M",
      "100000000Other": "000M",
      "1000000000One": "0B",
      "1000000000Other": "0B",
      "10000000000One": "00B",
      "10000000000Other": "00B",
      "100000000000One": "000B",
      "100000000000Other": "000B",
      "1000000000000One": "0T",
      "1000000000000Other": "0T",
      "10000000000000One": "00T",
      "10000000000000Other": "00T",
      "100000000000000One": "000T",
      "100000000000000Other": "000T",
      "1000000000000000One": "",
      "1000000000000000Other": "",
      "10000000000000000One": "",
      "10000000000000000Other": "",
      "100000000000000000One": "",
      "100000000000000000Other": "",
      "1000000000000000000One": "",
      "1000000000000000000Other": ""
    },
    "patternsCompactCurrency": {
      "1000One": "¤0K",
      "1000Other": "¤0K",
      "10000One": "¤00K",
      "10000Other": "¤00K",
      "100000One": "¤000K",
      "100000Other": "¤000K",
      "1000000One": "¤0M",
      "1000000Other": "¤0M",
      "10000000One": "¤00M",
      "10000000Other": "¤00M",
      "100000000One": "¤000M",
      "100000000Other": "¤000M",
      "1000000000One": "¤0B",
      "1000000000Other": "¤0B",
      "10000000000One": "¤00B",
      "10000000000Other": "¤00B",
      "100000000000One": "¤000B",
      "100000000000Other": "¤000B",
      "1000000000000One": "¤0T",
      "1000000000000Other": "¤0T",
      "10000000000000One": "¤00T",
      "10000000000000Other": "¤00T",
      "100000000000000One": "¤000T",
      "100000000000000Other": "¤000T",
      "1000000000000000One": "",
      "1000000000000000Other": "",
      "10000000000000000One": "",
      "10000000000000000Other": "",
      "100000000000000000One": "",
      "100000000000000000Other": "",
      "1000000000000000000One": "",
      "1000000000000000000Other": ""
    }
  },
  he: {
    "patternNumber": "#,##0.###",
    "patternPercent": "#,##0%",
    "patternCurrency": "‏#,##0.00 ‏¤;‏-#,##0.00 ‏¤",
    "symbolPlus": "‎+",
    "symbolMinus": "‎-",
    "symbolPercent": "%",
    "symbolGroup": ",",
    "symbolDecimal": ".",
    "patternsCompactNumber": {
      "1000One": "0K‏",
      "1000Other": "0K‏",
      "10000One": "00K‏",
      "10000Other": "00K‏",
      "100000One": "000K‏",
      "100000Other": "000K‏",
      "1000000One": "0M‏",
      "1000000Other": "0M‏",
      "10000000One": "00M‏",
      "10000000Other": "00M‏",
      "100000000One": "000M‏",
      "100000000Other": "000M‏",
      "1000000000One": "0B‏",
      "1000000000Other": "0B‏",
      "10000000000One": "00B‏",
      "10000000000Other": "00B‏",
      "100000000000One": "000B‏",
      "100000000000Other": "000B‏",
      "1000000000000One": "0T‏",
      "1000000000000Other": "0T‏",
      "10000000000000One": "00T‏",
      "10000000000000Other": "00T‏",
      "100000000000000One": "000T‏",
      "100000000000000Other": "000T‏",
      "1000000000000000One": "",
      "1000000000000000Other": "",
      "10000000000000000One": "",
      "10000000000000000Other": "",
      "100000000000000000One": "",
      "100000000000000000Other": "",
      "1000000000000000000One": "",
      "1000000000000000000Other": ""
    },
    "patternsCompactCurrency": {
      "1000One": "¤0K‏",
      "1000Other": "¤0K‏",
      "10000One": "¤00K‏",
      "10000Other": "¤00K‏",
      "100000One": "¤000K‏",
      "100000Other": "¤000K‏",
      "1000000One": "¤0M‏",
      "1000000Other": "¤0M‏",
      "10000000One": "¤00M‏",
      "10000000Other": "¤00M‏",
      "100000000One": "¤000M‏",
      "100000000Other": "¤000M‏",
      "1000000000One": "¤0B‏",
      "1000000000Other": "¤0B‏",
      "10000000000One": "¤00B‏",
      "10000000000Other": "¤00B‏",
      "100000000000One": "¤000B‏",
      "100000000000Other": "¤000B‏",
      "1000000000000One": "¤0T‏",
      "1000000000000Other": "¤0T‏",
      "10000000000000One": "¤00T‏",
      "10000000000000Other": "¤00T‏",
      "100000000000000One": "¤000T‏",
      "100000000000000Other": "¤000T‏",
      "1000000000000000One": "",
      "1000000000000000Other": "",
      "10000000000000000One": "",
      "10000000000000000Other": "",
      "100000000000000000One": "",
      "100000000000000000Other": "",
      "1000000000000000000One": "",
      "1000000000000000000Other": ""
    }
  },
  hi: {
    "patternNumber": "#,##,##0.###",
    "patternPercent": "#,##,##0%",
    "patternCurrency": "¤#,##,##0.00",
    "symbolPlus": "+",
    "symbolMinus": "-",
    "symbolPercent": "%",
    "symbolGroup": ",",
    "symbolDecimal": ".",
    "patternsCompactNumber": {
      "1000One": "0 हज़ार",
      "1000Other": "0 हज़ार",
      "10000One": "00 हज़ार",
      "10000Other": "00 हज़ार",
      "100000One": "0 लाख",
      "100000Other": "0 लाख",
      "1000000One": "00 लाख",
      "1000000Other": "00 लाख",
      "10000000One": "0 क॰",
      "10000000Other": "0 क॰",
      "100000000One": "00 क॰",
      "100000000Other": "00 क॰",
      "1000000000One": "0 अ॰",
      "1000000000Other": "0 अ॰",
      "10000000000One": "00 अ॰",
      "10000000000Other": "00 अ॰",
      "100000000000One": "0 ख॰",
      "100000000000Other": "0 ख॰",
      "1000000000000One": "00 ख॰",
      "1000000000000Other": "00 ख॰",
      "10000000000000One": "0 नील",
      "10000000000000Other": "0 नील",
      "100000000000000One": "00 नील",
      "100000000000000Other": "00 नील",
      "1000000000000000One": "",
      "1000000000000000Other": "",
      "10000000000000000One": "",
      "10000000000000000Other": "",
      "100000000000000000One": "",
      "100000000000000000Other": "",
      "1000000000000000000One": "",
      "1000000000000000000Other": ""
    },
    "patternsCompactCurrency": {
      "1000One": "¤0 हज़ार",
      "1000Other": "¤0 हज़ार",
      "10000One": "¤00 हज़ार",
      "10000Other": "¤00 हज़ार",
      "100000One": "¤0 लाख",
      "100000Other": "¤0 लाख",
      "1000000One": "¤00 लाख",
      "1000000Other": "¤00 लाख",
      "10000000One": "¤0 क॰",
      "10000000Other": "¤0 क॰",
      "100000000One": "¤00 क॰",
      "100000000Other": "¤00 क॰",
      "1000000000One": "¤0 अ॰",
      "1000000000Other": "¤0 अ॰",
      "10000000000One": "¤00 अ॰",
      "10000000000Other": "¤00 अ॰",
      "100000000000One": "¤0 ख॰",
      "100000000000Other": "¤0 ख॰",
      "1000000000000One": "¤00 ख॰",
      "1000000000000Other": "¤00 ख॰",
      "10000000000000One": "¤0 नील",
      "10000000000000Other": "¤0 नील",
      "100000000000000One": "¤00 नील",
      "100000000000000Other": "¤00 नील",
      "1000000000000000One": "",
      "1000000000000000Other": "",
      "10000000000000000One": "",
      "10000000000000000Other": "",
      "100000000000000000One": "",
      "100000000000000000Other": "",
      "1000000000000000000One": "",
      "1000000000000000000Other": ""
    }
  },
  ja: {
    "patternNumber": "#,##0.###",
    "patternPercent": "#,##0%",
    "patternCurrency": "¤#,##0.00",
    "symbolPlus": "+",
    "symbolMinus": "-",
    "symbolPercent": "%",
    "symbolGroup": ",",
    "symbolDecimal": ".",
    "patternsCompactNumber": {
      "1000One": "",
      "1000Other": "0",
      "10000One": "",
      "10000Other": "0万",
      "100000One": "",
      "100000Other": "00万",
      "1000000One": "",
      "1000000Other": "000万",
      "10000000One": "",
      "10000000Other": "0000万",
      "100000000One": "",
      "100000000Other": "0億",
      "1000000000One": "",
      "1000000000Other": "00億",
      "10000000000One": "",
      "10000000000Other": "000億",
      "100000000000One": "",
      "100000000000Other": "0000億",
      "1000000000000One": "",
      "1000000000000Other": "0兆",
      "10000000000000One": "",
      "10000000000000Other": "00兆",
      "100000000000000One": "",
      "100000000000000Other": "000兆",
      "1000000000000000One": "",
      "1000000000000000Other": "0000兆",
      "10000000000000000One": "",
      "10000000000000000Other": "0京",
      "100000000000000000One": "",
      "100000000000000000Other": "00京",
      "1000000000000000000One": "",
      "1000000000000000000Other": "000京"
    },
    "patternsCompactCurrency": {
      "1000One": "",
      "1000Other": "0",
      "10000One": "",
      "10000Other": "¤0万",
      "100000One": "",
      "100000Other": "¤00万",
      "1000000One": "",
      "1000000Other": "¤000万",
      "10000000One": "",
      "10000000Other": "¤0000万",
      "100000000One": "",
      "100000000Other": "¤0億",
      "1000000000One": "",
      "1000000000Other": "¤00億",
      "10000000000One": "",
      "10000000000Other": "¤000億",
      "100000000000One": "",
      "100000000000Other": "¤0000億",
      "1000000000000One": "",
      "1000000000000Other": "¤0兆",
      "10000000000000One": "",
      "10000000000000Other": "¤00兆",
      "100000000000000One": "",
      "100000000000000Other": "¤000兆",
      "1000000000000000One": "",
      "1000000000000000Other": "¤0000兆",
      "10000000000000000One": "",
      "10000000000000000Other": "¤0京",
      "100000000000000000One": "",
      "100000000000000000Other": "¤00京",
      "1000000000000000000One": "",
      "1000000000000000000Other": "¤000京"
    }
  },
  lt: {
    "patternNumber": "#,##0.###",
    "patternPercent": "#,##0 %",
    "patternCurrency": "#,##0.00 ¤",
    "symbolPlus": "+",
    "symbolMinus": "−",
    "symbolPercent": "%",
    "symbolGroup": " ",
    "symbolDecimal": ",",
    "patternsCompactNumber": {
      "1000One": "0 tūkst'.'",
      "1000Other": "0 tūkst'.'",
      "10000One": "00 tūkst'.'",
      "10000Other": "00 tūkst'.'",
      "100000One": "000 tūkst'.'",
      "100000Other": "000 tūkst'.'",
      "1000000One": "0 mln'.'",
      "1000000Other": "0 mln'.'",
      "10000000One": "00 mln'.'",
      "10000000Other": "00 mln'.'",
      "100000000One": "000 mln'.'",
      "100000000Other": "000 mln'.'",
      "1000000000One": "0 mlrd'.'",
      "1000000000Other": "0 mlrd'.'",
      "10000000000One": "00 mlrd'.'",
      "10000000000Other": "00 mlrd'.'",
      "100000000000One": "000 mlrd'.'",
      "100000000000Other": "000 mlrd'.'",
      "1000000000000One": "0 trln'.'",
      "1000000000000Other": "0 trln'.'",
      "10000000000000One": "00 trln'.'",
      "10000000000000Other": "00 trln'.'",
      "100000000000000One": "000 trln'.'",
      "100000000000000Other": "000 trln'.'",
      "1000000000000000One": "",
      "1000000000000000Other": "",
      "10000000000000000One": "",
      "10000000000000000Other": "",
      "100000000000000000One": "",
      "100000000000000000Other": "",
      "1000000000000000000One": "",
      "1000000000000000000Other": ""
    },
    "patternsCompactCurrency": {
      "1000One": "0 tūkst'.' ¤",
      "1000Other": "0 tūkst'.' ¤",
      "10000One": "00 tūkst'.' ¤",
      "10000Other": "00 tūkst'.' ¤",
      "100000One": "000 tūkst'.' ¤",
      "100000Other": "000 tūkst'.' ¤",
      "1000000One": "0 mln'.' ¤",
      "1000000Other": "0 mln'.' ¤",
      "10000000One": "00 mln'.' ¤",
      "10000000Other": "00 mln'.' ¤",
      "100000000One": "000 mln'.' ¤",
      "100000000Other": "000 mln'.' ¤",
      "1000000000One": "0 mlrd'.' ¤",
      "1000000000Other": "0 mlrd'.' ¤",
      "10000000000One": "00 mlrd'.' ¤",
      "10000000000Other": "00 mlrd'.' ¤",
      "100000000000One": "000 mlrd'.' ¤",
      "100000000000Other": "000 mlrd'.' ¤",
      "1000000000000One": "0 trln'.' ¤",
      "1000000000000Other": "0 trln'.' ¤",
      "10000000000000One": "00 trln'.' ¤",
      "10000000000000Other": "00 trln'.' ¤",
      "100000000000000One": "000 trln'.' ¤",
      "100000000000000Other": "000 trln'.' ¤",
      "1000000000000000One": "",
      "1000000000000000Other": "",
      "10000000000000000One": "",
      "10000000000000000Other": "",
      "100000000000000000One": "",
      "100000000000000000Other": "",
      "1000000000000000000One": "",
      "1000000000000000000Other": ""
    }
  },
  ro: {
    "patternNumber": "#,##0.###",
    "patternPercent": "#,##0 %",
    "patternCurrency": "#,##0.00 ¤",
    "symbolPlus": "+",
    "symbolMinus": "-",
    "symbolPercent": "%",
    "symbolGroup": ".",
    "symbolDecimal": ",",
    "patternsCompactNumber": {
      "1000One": "0 K",
      "1000Other": "0 K",
      "10000One": "00 K",
      "10000Other": "00 K",
      "100000One": "000 K",
      "100000Other": "000 K",
      "1000000One": "0 mil'.'",
      "1000000Other": "0 mil'.'",
      "10000000One": "00 mil'.'",
      "10000000Other": "00 mil'.'",
      "100000000One": "000 mil'.'",
      "100000000Other": "000 mil'.'",
      "1000000000One": "0 mld'.'",
      "1000000000Other": "0 mld'.'",
      "10000000000One": "00 mld'.'",
      "10000000000Other": "00 mld'.'",
      "100000000000One": "000 mld'.'",
      "100000000000Other": "000 mld'.'",
      "1000000000000One": "0 tril'.'",
      "1000000000000Other": "0 tril'.'",
      "10000000000000One": "00 tril'.'",
      "10000000000000Other": "00 tril'.'",
      "100000000000000One": "000 tril'.'",
      "100000000000000Other": "000 tril'.'",
      "1000000000000000One": "",
      "1000000000000000Other": "",
      "10000000000000000One": "",
      "10000000000000000Other": "",
      "100000000000000000One": "",
      "100000000000000000Other": "",
      "1000000000000000000One": "",
      "1000000000000000000Other": ""
    },
    "patternsCompactCurrency": {
      "1000One": "0 mie ¤",
      "1000Other": "0 mii ¤",
      "10000One": "00 mii ¤",
      "10000Other": "00 mii ¤",
      "100000One": "000 mii ¤",
      "100000Other": "000 mii ¤",
      "1000000One": "0 mil'.' ¤",
      "1000000Other": "0 mil'.' ¤",
      "10000000One": "00 mil'.' ¤",
      "10000000Other": "00 mil'.' ¤",
      "100000000One": "000 mil'.' ¤",
      "100000000Other": "000 mil'.' ¤",
      "1000000000One": "0 mld'.' ¤",
      "1000000000Other": "0 mld'.' ¤",
      "10000000000One": "00 mld'.' ¤",
      "10000000000Other": "00 mld'.' ¤",
      "100000000000One": "000 mld'.' ¤",
      "100000000000Other": "000 mld'.' ¤",
      "1000000000000One": "0 tril'.' ¤",
      "1000000000000Other": "0 tril'.' ¤",
      "10000000000000One": "00 tril'.' ¤",
      "10000000000000Other": "00 tril'.' ¤",
      "100000000000000One": "000 tril'.' ¤",
      "100000000000000Other": "000 tril'.' ¤",
      "1000000000000000One": "",
      "1000000000000000Other": "",
      "10000000000000000One": "",
      "10000000000000000Other": "",
      "100000000000000000One": "",
      "100000000000000000Other": "",
      "1000000000000000000One": "",
      "1000000000000000000Other": ""
    }
  },
};

describe('CoreHome/NumberFormatter', () => {

  const numberTestData: Array<Array<any>> = [
    // english formats
    ['en', 5, 0, 0, '5'],
    ['en', -5, 0, 3, '-5'],
    ['en', 5.299, 0, 0, '5'],
    ['en', 5.2992, 3, 0, '5.299'],
    ['en', 5.6666666666667, 1, 0, '5.7'],
    ['en', 5.07, 1, 0, '5.1'],
    ['en', -50, 3, 3, '-50.000'],
    ['en', 5000, 0, 0, '5,000'],
    ['en', 5000000, 0, 0, '5,000,000'],
    ['en', -5000000, 0, 0, '-5,000,000'],

    // foreign languages
    ['ar', 51239.56, 3, 0, '51,239.56'],
    ['be', 51239.56, 3, 0, '51 239,56'],
    ['de', 51239.56, 3, 0, '51.239,56'],
    ['he', 152551239.56, 3, 0, '152,551,239.56'],
    ['he', -152551239.56, 3, 0, '‎-152,551,239.56'],
    ['hi', 152551239.56, 0, 0, '15,25,51,240'],
    ['lt', -152551239.56, 0, 0, '−152 551 240'],
  ];

  numberTestData.forEach((testdata) => {
    const [ lang, input, maxFractionDigits, minFractionDigits, expected ] = testdata;

    it(`should correctly format number with (${lang}, ${input}, ${maxFractionDigits}, ${minFractionDigits})`, () => {

      window.piwik.numbers = formats[lang];

      const result = NumberFormatter.formatNumber(input as number, maxFractionDigits as number, minFractionDigits as number);

      expect(result).toEqual(expected);
    })
  });

  const compactNumberTestData: Array<Array<any>> = [
    ['en', 100, '100'],
    ['en', 525.22, '525'],
    ['en', 999, '999'],
    ['en', 999.9, '1K'],
    ['en', 1000, '1K'],
    ['en', 1233, '1.2K'],
    ['en', 12330, '12K'],
    ['en', 123306, '123K'],
    ['en', 999506, '1M'],
    ['en', 1233060, '1.2M'],
    ['en', 12330600, '12M'],
    ['en', 123306000, '123M'],
    ['en', 999200000, '999M'],
    ['en', 1233060000, '1.2B'],
    ['en', 12330600000, '12B'],
    ['en', 123306000000, '123B'],
    ['en', 1233060000000, '1.2T'],
    ['en', 12330600000000, '12T'],
    ['en', 123306000000000, '123T'],
    ['en', 999906000000000, '1,000T'],
    ['en', 1233060000000000, '1,233T'],
    ['en', 12330600000000000, '12,331T'],
    ['en', 99999200000000000, '99,999T'],
    ['en', 99999900000000000, '100,000T'],

    ['ja', 1233, '1,233'],
    ['ja', 12330, '1.2万'],
    ['ja', 123306, '12万'],
    ['ja', 1233060, '123万'],
    ['ja', 12330600, '1,233万'],
    ['ja', 123306000, '1.2億'],
    ['ja', 1233060000, '12億'],
    ['ja', 52330600000, '523億'],
    ['ja', 123306000000, '1,233億'],
    ['ja', 1233060000000, '1.2兆'],
    ['ja', 12330600000000, '12兆'],
    ['ja', 123306000000000, '123兆'],
    ['ja', 1233060000000000, '1,233兆'],
    ['ja', 9999060000000000, '9,999兆'],
    ['ja', 12330600000000000, '1.2京'],

    ['ar', 1330600000, '1.3 مليار'],
    ['be', 1330600000, '1,3 млрд'],
    ['de', 1330600000, '1,3 Mrd.'],
    ['hi', 1330600000, '1.3 अ॰'],
    ['lt', 1330600000, '1,3 mlrd.'],
  ];

  compactNumberTestData.forEach((testdata) => {
    const [ lang, input, expected ] = testdata;

    it(`should correctly compact format number with (${lang}, ${input})`, () => {

      window.piwik.numbers = formats[lang];

      const result = NumberFormatter.formatNumberCompact(input as number);

      expect(result).toEqual(expected);
    })
  });

  const percentNumberTestData: Array<Array<any>> = [
    // english formats
    ['en', 5, 0, 0, '5%'],
    ['en', -5, 0, 3, '-5%'],
    ['en', 5.299, 0, 0, '5%'],
    ['en', 5.2992, 3, 0, '5.299%'],
    ['en', -50, 3, 3, '-50.000%'],
    ['en', -50, 1, 1, '-50.0%'],
    ['en', -50.1, 3, 3, '-50.100%'],
    ['en', 5000, 0, 0, '5,000%'],
    ['en', +5000, 0, 0, '5,000%'],
    ['en', 5000000, 0, 0, '5,000,000%'],
    ['en', -5000000, 0, 0, '-5,000,000%'],

    // foreign languages
    ['ar', 51239.56, 3, 0, '51,239.56‎%‎'],
    ['be', 51239.56, 3, 0, '51 239,56 %'],
    ['de', 51239.56, 3, 0, '51.239,56 %'],
    ['he', 152551239.56, 3, 0, '152,551,239.56%'],
    ['hi', 152551239.56, 0, 0, '15,25,51,240%'],
    ['lt', -152551239.56, 0, 0, '−152 551 240 %'],
  ];

  percentNumberTestData.forEach((testdata) => {
    const [ lang, input, maxFractionDigits, minFractionDigits, expected ] = testdata;

    it(`should correctly format percent with (${lang}, ${input}, ${maxFractionDigits}, ${minFractionDigits})`, () => {

      window.piwik.numbers = formats[lang];

      const result = NumberFormatter.formatPercent(input as number, maxFractionDigits as number, minFractionDigits as number);

      expect(result).toEqual(expected);
    })
  });

  const currencyTestData: Array<Array<any>> = [
    // english formats
    ['en', 5, '$', 0, 0, '$5'],
    ['en', -5, '$', 0, 3, '-$5'],
    ['en', 5.299, '$', 0, 0, '$5'],
    ['en', 5.2992, '$', 3, 0, '$5.299'],
    ['en', -50, '$', 3, 3, '-$50.000'],
    ['en', -50, '$', 1, 1, '-$50.0'],
    ['en', -50.1, '$', 3, 3, '-$50.100'],
    ['en', 5000, '$', 0, 0, '$5,000'],
    ['en', +5000, '$', 0, 0, '$5,000'],
    ['en', 5000000, '$', 0, 0, '$5,000,000'],
    ['en', -5000000, '$', 0, 0, '-$5,000,000'],

    // foreign languages
    ['ar', 51239.56, '$', 3, 0, '‏51,239.56 $'],
    ['be', 51239.56, '$', 3, 0, '51 239,56 $'],
    ['de', 51239.56, '$', 3, 0, '51.239,56 $'],
    ['he', -152551239.56, '$', 3, 0, '‏‎-152,551,239.56 ‏$'],
    ['hi', 152551239.56, '$', 0, 0, '$15,25,51,240'],
    ['lt', -152551239.56, '$', 0, 0, '−152 551 240 $'],
  ];

  currencyTestData.forEach((testdata) => {
    const [ lang, input, currency, maxFractionDigits, minFractionDigits, expected ] = testdata;

    it(`should correctly format currency with (${lang}, ${input}, ${currency}, ${maxFractionDigits}, ${minFractionDigits})`, () => {

      window.piwik.numbers = formats[lang];

      const result = NumberFormatter.formatCurrency(input as number, currency as string, maxFractionDigits as number, minFractionDigits as number);

      expect(result).toEqual(expected);
    })
  });

  const compactCurrencyTestData: Array<Array<any>> = [
    ['en', 100, '$100'],
    ['en', 525.22, '$525'],
    ['en', 1000, '$1K'],
    ['en', 1233, '$1.2K'],
    ['en', 12330, '$12K'],
    ['en', 99955, '$100K'],
    ['en', 123306, '$123K'],
    ['en', 3233060, '$3.2M'],
    ['en', 12330600, '$12M'],
    ['en', 123306000, '$123M'],
    ['en', 999930600, '$1B'],
    ['en', 1233060000, '$1.2B'],
    ['en', 52330600000, '$52B'],
    ['en', 123306000000, '$123B'],
    ['en', 1233060000000, '$1.2T'],
    ['en', 72330600000000, '$72T'],
    ['en', 123306000000000, '$123T'],
    ['en', 999906000000000, '$1,000T'],
    ['en', 1233060000000000, '$1,233T'],
    ['en', 12330600000000000, '$12,331T'],

    ['ja', 1233, '$1,233'],
    ['ja', 12330, '$1.2万'],
    ['ja', 123306, '$12万'],
    ['ja', 1233060, '$123万'],
    ['ja', 12330600, '$1,233万'],
    ['ja', 123306000, '$1.2億'],
    ['ja', 1233060000, '$12億'],
    ['ja', 12330600000, '$123億'],
    ['ja', 123306000000, '$1,233億'],
    ['ja', 1233060000000, '$1.2兆'],
    ['ja', 12330600000000, '$12兆'],
    ['ja', 123306000000000, '$123兆'],
    ['ja', 1233060000000000, '$1,233兆'],
    ['ja', 12330600000000000, '$1.2京'],

    ['ar', 1330600000, '1.3 مليار $'],
    ['be', 1330600000, '1,3 млрд $'],
    ['de', 1330600000, '1,3 Mrd. $'],
    ['ro', 1000, '1 mie $'],
    ['ro', 2000, '2 mii $'],
    ['hi', 1330600000, '$1.3 अ॰'],
    ['lt', 1330600000, '1,3 mlrd. $'],
  ];

  compactCurrencyTestData.forEach((testdata) => {
    const [ lang, input, expected ] = testdata;

    it(`should correctly compact format number with (${lang}, ${input})`, () => {

      window.piwik.numbers = formats[lang];

      const result = NumberFormatter.formatCurrencyCompact(input as number, '$');

      expect(result).toEqual(expected);
    })
  });

  const evolutionTestData: Array<Array<any>> = [
    // english formats
    ['en', 5, 0, 0, '+5%'],
    ['en', -5, 0, 3, '-5%'],
    ['en', 5.299, 0, 0, '+5%'],
    ['en', 5.299, 3, 0, '+5.299%'],
    ['en', -50, 3, 3, '-50.000%'],
    ['en', -50, 1, 1, '-50.0%'],
    ['en', -50.1, 3, 3, '-50.100%'],
    ['en', 5000, 0, 0, '+5,000%'],
    ['en', +5000, 0, 0, '+5,000%'],
    ['en', 5000000, 0, 0, '+5,000,000%'],
    ['en', -5000000, 0, 0, '-5,000,000%'],

    // foreign languages
    ['ar', 51239.56, 3, 0, '‎+51,239.56‎%‎'],
    ['be', 51239.56, 3, 0, '+51 239,56 %'],
    ['de', 51239.56, 3, 0, '+51.239,56 %'],
    ['he', 152551239.56, 3, 0, '‎+152,551,239.56%'],
    ['hi', 152551239.56, 0, 0, '+15,25,51,240%'],
    ['lt', -152551239.56, 0, 0, '−152 551 240 %'],
  ];

  evolutionTestData.forEach((testdata) => {
    const [ lang, input, maxFractionDigits, minFractionDigits, expected ] = testdata;

    it(`should correctly format evolution with (${lang}, ${input}, ${maxFractionDigits}, ${minFractionDigits})`, () => {

      window.piwik.numbers = formats[lang];

      const result = NumberFormatter.formatEvolution(input as number, maxFractionDigits as number, minFractionDigits as number);

      expect(result).toEqual(expected);
    })
  });

  const calculateAndFormatEvolutionTestData: Array<Array<any>> = [
    // we test only english, as other formats are already covered by formatEvolution tests
    ['en', 2, 1, false, '+100%'],
    ['en', 25, 100, false, '-75%'],
    ['en', 1, 3, false, '-66.7%'],
    ['en', 1, 3, true, '66.7%'],
    ['en', 10001, 9883, false, '+1.19%'],
    ['en', 100001, 100000, false, '+0.001%'],
    ['en', 100001, 100000, true, '0.001%'],
    ['en', 10000001, 10000000, false, '+0%'],
  ];

  calculateAndFormatEvolutionTestData.forEach((testdata) => {
    const [ lang, input1, input2, noSign, expected ] = testdata;

    it(`should correctly format evolution with (${lang}, ${input1}, ${input2}, ${noSign})`, () => {

      window.piwik.numbers = formats[lang];

      const result = NumberFormatter.calculateAndFormatEvolution(input1 as number, input2 as number, noSign);

      expect(result).toEqual(expected);
    })
  });

  const formattedNumberTestData: Array<Array<any>> = [
    ['en', '+100%', 100],
    ['en', '-75%', -75],
    ['en', '12,245.66', 12245.66],
    ['en', '-0.555', -0.555],
    ['ar', '؜+51٬239.56٪؜', 51239.56],
    ['be', '+51 239,56 %', 51239.56],
    ['de', '+51.239,56 %', 51239.56 ],
    ['de', '-239,56 $%', -239.56 ],
    ['he', '‎+152,551,239.56%', 152551239.56],
    ['he', '‎-152,551,239.56', -152551239.56],
    ['hi', '+15,25,51,240%', 152551240],
    ['lt', '−152 551 240 %', -152551240],
  ];

  formattedNumberTestData.forEach((testdata) => {
    const [ lang, input, expected ] = testdata;

    it(`should correctly parse formatted number with (${lang}, ${input})`, () => {

      window.piwik.numbers = formats[lang];

      const result = NumberFormatter.parseFormattedNumber(input as string);

      expect(result).toEqual(expected);
    })
  });

});
