/* BEGIN software license
 *
 * MsXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright 2009--2026 by Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 *
 * This file is part of the MsXpertSuite project.
 *
 * The MsXpertSuite project is the successor of the massXpert project. This
 * project now includes various independent modules:
 *
 * - massXpert, model polymer chemistries and simulate mass spectrometric data;
 * - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */

#pragma once


/////////////////////// Stdlib includes


/////////////////////// Qt includes
#include <QString>
#include <QObject>
#include <QtQml>


/////////////////////// Local includes
#include "MsXpS/export-import-config.h"

#include "MsXpS/libXpertMassCore/globals.hpp"

namespace MsXpS
{
namespace libXpertMassCore
{

class DECLSPEC Tolerance: public QObject
{
  Q_OBJECT
  Q_PROPERTY(
    double nominal READ getNominal WRITE setNominal NOTIFY nominalChangedSignal)
  Q_PROPERTY(Type type READ getType WRITE setType NOTIFY typeChangedSignal)

  public:
  enum class Type
  {
    NOT_SET = 0x0,
    DALTON,
    RES,
    PPM,
    LAST
  };
  Q_ENUM(Type)

  QMap<Type, QString> m_typeToStringMap = {
    {Type::DALTON, "Dalton"}, {Type::RES, "Res"}, {Type::PPM, "Ppm"}};

  Q_INVOKABLE explicit Tolerance(QObject *parent = nullptr);
  Q_INVOKABLE explicit Tolerance(const QString &text,
                                 QObject *parent = nullptr);
  Q_INVOKABLE explicit Tolerance(Type tolerance_type,
                                 double nominal,
                                 QObject *parent = nullptr);
  Q_INVOKABLE explicit Tolerance(const Tolerance &other,
                                 QObject *parent = nullptr);

  Q_INVOKABLE void
  initialize(double nominal, Tolerance::Type type, QObject *parent = nullptr);
  Q_INVOKABLE void initialize(const Tolerance &tolerance,
                              QObject *parent = nullptr);
  Q_INVOKABLE void initialize(const QString &text, QObject *parent = nullptr);

  Q_INVOKABLE void setNominal(double nominal);
  Q_INVOKABLE double getNominal() const;

  Q_INVOKABLE void setType(Type type);
  Q_INVOKABLE void setType(const QString &text);
  Q_INVOKABLE Type getType() const;
  Q_INVOKABLE Type getType(const QString &text) const;
  Q_INVOKABLE QString getTypeAsString(Tolerance::Type type) const;
  Q_INVOKABLE QString getTypeAsString() const;

  Q_INVOKABLE Tolerance *clone(QObject *parent = nullptr);

  Q_INVOKABLE double calculateWidth(double reason) const;
  Q_INVOKABLE double calculateHalfWidth(double reason) const;

  Q_INVOKABLE double calculateNominal(double width, double mass_or_mz) const;

  Q_INVOKABLE QString toString() const;

  signals:
  void nominalChangedSignal();
  void typeChangedSignal();
  void toleranceChangedSignal(std::pair<double, Tolerance::Type>);

  private:
  double m_nominal = 40000;
  Type m_type      = Type::RES;
};


} // namespace libXpertMassCore
} // namespace MsXpS
