/*
 *  $Id: thumbnail.c 29091 2026-01-06 15:56:49Z yeti-dn $
 *  Copyright (C) 2025 David Necas (Yeti)
 *  E-mail: yeti@gwyddion.net
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <gtk/gtk.h>
#include <gdk/gdkcairo.h>

#include "libgwyddion/macros.h"
#include "libgwyddion/math.h"
#include "libgwyddion/utils.h"
#include "libgwyddion/gwycontainer.h"
#include "libgwyddion/arithmetic.h"
#include "libgwyddion/stats.h"
#include "libgwyui/gwyui.h"

#include "libgwyapp/gwyapp.h"
#include "libgwyapp/module-utils.h"
#include "libgwyapp/gwyappinternal.h"

GwyField*
_gwy_app_create_brick_preview_field(GwyBrick *brick)
{
    GwyField *preview = gwy_brick_new_field_like_xy_plane(brick, FALSE);
    GwyField *mask = gwy_field_new_alike(preview, FALSE);

    gwy_brick_mean_xy_plane(brick, preview);
    gwy_field_mask_outliers_iqr(preview, mask, 3.0, 3.0);
    gwy_field_laplace_solve(preview, mask, -1, 1.0);
    g_object_unref(mask);

    return preview;
}

static gdouble
lawn_reduce_avg(gint ncurves, gint curvelength, const gdouble *curvedata, gpointer user_data)
{
    guint i, idx = GPOINTER_TO_UINT(user_data);
    gdouble s = 0.0;

    gwy_debug("nc %d, clen %d, data %p, idx %d", ncurves, curvelength, curvedata, idx);
    g_return_val_if_fail(idx < ncurves, 0.0);
    if (!curvelength)
        return s;

    curvedata += idx*curvelength;
    for (i = 0; i < curvelength; i++)
        s += curvedata[i];
    return s/curvelength;
}

GwyField*
_gwy_app_create_lawn_preview_field(GwyLawn *lawn)
{
    gint xres = gwy_lawn_get_xres(lawn);
    gint yres = gwy_lawn_get_yres(lawn);
    gdouble xreal = gwy_lawn_get_xreal(lawn);
    gdouble yreal = gwy_lawn_get_yreal(lawn);
    GwyField *preview = gwy_field_new(xres, yres, xreal, yreal, FALSE);
    GwyField *mask = gwy_field_new_alike(preview, FALSE);

    gwy_lawn_reduce_to_plane(lawn, preview, lawn_reduce_avg, GUINT_TO_POINTER(0));
    gwy_field_mask_outliers_iqr(preview, mask, 3.0, 3.0);
    gwy_field_laplace_solve(preview, mask, -1, 1.0);
    g_object_unref(mask);

    gwy_unit_assign(gwy_field_get_unit_z(preview), gwy_lawn_get_unit_curve(lawn, 0));

    return preview;
}

static GwyField*
make_thumbnail_field(GwyField *dfield,
                     gint *width,
                     gint *height)
{
    gint xres, yres;
    gdouble scale;

    xres = gwy_field_get_xres(dfield);
    yres = gwy_field_get_yres(dfield);
    if (xres == *width && yres == *height)
        g_object_ref(dfield);
    else {
        scale = MAX(xres/(gdouble)*width, yres/(gdouble)*height);
        xres = xres/scale;
        yres = yres/scale;
        xres = CLAMP(xres, 2, *width);
        yres = CLAMP(yres, 2, *height);
        dfield = gwy_field_new_resampled(dfield, xres, yres, GWY_INTERPOLATION_NNA);
    }

    *width = xres;
    *height = yres;

    return dfield;
}

static GdkPixbuf*
render_image_thumbnail(GwyField *dfield,
                       const gchar *gradname,
                       GwyColorMappingType mapping,
                       gint width,
                       gint height,
                       gdouble *pmin,
                       gdouble *pmax)
{
    GwyGradient *gradient = gwy_gradients_get_gradient(gradname);
    GwyField *render_field = make_thumbnail_field(dfield, &width, &height);
    GdkPixbuf *pixbuf = gdk_pixbuf_new(GDK_COLORSPACE_RGB, FALSE, 8, width, height);
    gwy_draw_field_pixbuf(pixbuf, render_field, gradient, mapping, pmin, pmax);
    g_object_unref(render_field);

    return pixbuf;
}

static GdkPixbuf*
render_mask_thumbnail(GwyField *dfield,
                      const GwyRGBA *color,
                      gint width,
                      gint height)
{
    GwyField *render_field = make_thumbnail_field(dfield, &width, &height);
    GdkPixbuf *pixbuf = gdk_pixbuf_new(GDK_COLORSPACE_RGB, TRUE, 8, width, height);
    gwy_draw_field_pixbuf_mask(pixbuf, render_field, color);
    g_object_unref(render_field);

    return pixbuf;
}

/**
 * gwy_render_image_thumbnail:
 * @data: A data container.
 * @id: Data channel id.
 * @max_width: Maximum width of the created pixbuf, it must be at least 2.
 * @max_height: Maximum height of the created pixbuf, it must be at least 2.
 *
 * Creates a channel thumbnail.
 *
 * Returns: A newly created pixbuf with channel thumbnail.  It keeps the aspect ratio of the data field while not
 *          exceeding @max_width and @max_height.
 **/
GdkPixbuf*
gwy_render_image_thumbnail(GwyFile *data,
                           gint id,
                           gint max_width,
                           gint max_height)
{
    g_return_val_if_fail(GWY_IS_FILE(data), NULL);
    g_return_val_if_fail(id >= 0, NULL);
    g_return_val_if_fail(max_width > 1 && max_height > 1, NULL);

    GwyField *field, *mfield, *sfield = NULL;
    GdkPixbuf *pixbuf, *mask;

    if (!(field = gwy_file_get_image(data, id)))
        return NULL;

    gwy_container_gis_object(GWY_CONTAINER(data), gwy_file_key_image_picture(id), &sfield);
    const gchar *gradient = gwy_file_get_palette(data, GWY_FILE_IMAGE, id);

    if (sfield)
        pixbuf = render_image_thumbnail(sfield, gradient, GWY_COLOR_MAPPING_FULL, max_width, max_height, NULL, NULL);
    else {
        GwyColorMappingType mapping = gwy_file_get_color_mapping(data, GWY_FILE_IMAGE, id);
        gdouble min, max;
        if (mapping == GWY_COLOR_MAPPING_FIXED) {
            /* render_image_thumbnail() requires non-NULL min and max for the fixed mode. */
            if (!gwy_file_get_fixed_range(data, GWY_FILE_IMAGE, id, &min, NULL))
                min = gwy_field_get_min(field);
            if (!gwy_file_get_fixed_range(data, GWY_FILE_IMAGE, id, NULL, &max))
                max = gwy_field_get_max(field);
        }
        /* Make thumbnails of images with defects nicer. */
        if (mapping == GWY_COLOR_MAPPING_FULL)
            mapping = GWY_COLOR_MAPPING_AUTO;

        pixbuf = render_image_thumbnail(field, gradient, mapping, max_width, max_height, &min, &max);
    }

    mfield = gwy_file_get_image_mask(data, id);
    if (mfield) {
        GwyRGBA color;
        if (!gwy_container_gis_boxed(GWY_CONTAINER(data), GWY_TYPE_RGBA, gwy_file_key_image_mask_color(id), &color))
            gwy_app_settings_get_default_mask_color(&color);
        mask = render_mask_thumbnail(mfield, &color, max_width, max_height);
        gdk_pixbuf_composite(mask, pixbuf,
                             0, 0, gdk_pixbuf_get_width(pixbuf), gdk_pixbuf_get_height(pixbuf), 0, 0,
                             1.0, 1.0, GDK_INTERP_NEAREST, 255);
        g_object_unref(mask);
    }

    return pixbuf;
}

/**
 * gwy_render_volume_thumbnail:
 * @data: A data container.
 * @id: Volume data id.
 * @max_width: Maximum width of the created pixbuf, it must be at least 2.
 * @max_height: Maximum height of the created pixbuf, it must be at least 2.
 *
 * Creates a volume thumbnail.
 *
 * Returns: A newly created pixbuf with volume data thumbnail.  It keeps the aspect ratio of the brick preview while
 *          not exceeding @max_width and @max_height.
 **/
GdkPixbuf*
gwy_render_volume_thumbnail(GwyFile *data,
                            gint id,
                            gint max_width,
                            gint max_height)
{
    g_return_val_if_fail(GWY_IS_FILE(data), NULL);
    g_return_val_if_fail(id >= 0, NULL);
    g_return_val_if_fail(max_width > 1 && max_height > 1, NULL);

    GwyBrick *brick;
    GwyField *dfield = NULL;

    if (!(brick = gwy_file_get_volume(data, id)))
        return NULL;

    if (!gwy_container_gis_object(GWY_CONTAINER(data), gwy_file_key_volume_picture(id), &dfield)) {
        GdkPixbuf *pixbuf = gdk_pixbuf_new(GDK_COLORSPACE_RGB, FALSE, 8, max_width, max_height);
        gdk_pixbuf_fill(pixbuf, 0);
        return pixbuf;
    }

    const gchar *gradient = gwy_file_get_palette(data, GWY_FILE_VOLUME, id);
    return render_image_thumbnail(dfield, gradient, GWY_COLOR_MAPPING_FULL, max_width, max_height, NULL, NULL);
}

/**
 * gwy_render_xyz_thumbnail:
 * @data: A data container.
 * @id: XYZ surface data id.
 * @max_width: Maximum width of the created pixbuf, it must be at least 2.
 * @max_height: Maximum height of the created pixbuf, it must be at least 2.
 *
 * Creates an XYZ data thumbnail.
 *
 * Returns: A newly created pixbuf with XYZ data thumbnail.  It keeps the aspect ratio of the brick preview while not
 *          exceeding @max_width and @max_height.
 **/
GdkPixbuf*
gwy_render_xyz_thumbnail(GwyFile *data,
                         gint id,
                         gint max_width,
                         gint max_height)
{
    g_return_val_if_fail(GWY_IS_CONTAINER(data), NULL);
    g_return_val_if_fail(id >= 0, NULL);
    g_return_val_if_fail(max_width > 1 && max_height > 1, NULL);

    GwySurface *surface;

    if (!(surface = gwy_file_get_xyz(data, id)))
        return NULL;

    const gchar *gradient = gwy_file_get_palette(data, GWY_FILE_XYZ, id);
    GwyField *raster = gwy_field_new(1, 1, 1.0, 1.0, FALSE);
    gwy_preview_surface_to_field(surface, raster, max_width, max_height, 0);
    GdkPixbuf *pixbuf = render_image_thumbnail(raster, gradient, GWY_COLOR_MAPPING_FULL,
                                               max_width, max_height, NULL, NULL);
    g_object_unref(raster);

    return pixbuf;
}

/**
 * gwy_render_graph_thumbnail:
 * @data: A data container.
 * @id: Graph model data id.
 * @max_width: Maximum width of the created pixbuf, it must be at least 2.
 * @max_height: Maximum height of the created pixbuf, it must be at least 2.
 *
 * Creates a graph thumbnail.
 *
 * Note this function needs the GUI running (unlike the other thumbnail functions).  It cannot be used in a console
 * program.
 *
 * Returns: A newly created pixbuf with graph thumbnail.  Since graphs do not have natural width and height, its size
 *          will normally be exactly @max_width and @max_height.
 **/
/* TODO: Make it do something useful without GUI running. */
GdkPixbuf*
gwy_render_graph_thumbnail(GwyFile *data,
                           gint id,
                           gint max_width,
                           gint max_height)
{
    static GwyGraph *graph = NULL;
    static GwyGraphModel *dummy_model = NULL;

    /* Special values indicating cleanup. */
    if (!data && id < 0 && !max_width && !max_height) {
        if (graph) {
            g_object_unref(graph);
            g_object_unref(dummy_model);
        }
        return NULL;
    }

    gint width = 160, height = 120;
    GdkPixbuf *big_pixbuf, *pixbuf;
    GwyGraphModel *gmodel;
    GwyGraphArea *area;
    gdouble min, max, d;
    gboolean is_logscale = FALSE;

    g_return_val_if_fail(GWY_IS_CONTAINER(data), NULL);
    g_return_val_if_fail(id >= 0, NULL);
    g_return_val_if_fail(max_width > 1 && max_height > 1, NULL);

    if (!(gmodel = gwy_file_get_graph(data, id)))
        return NULL;

    if (!gtk_main_level())
        return NULL;

    width = MAX(width, max_width);
    height = MAX(height, max_height);

    if (!graph) {
        graph = GWY_GRAPH(gwy_graph_new(gmodel));
        dummy_model = gwy_graph_model_new();
        gwy_graph_set_axis_visible(graph, GTK_POS_LEFT, FALSE);
        gwy_graph_set_axis_visible(graph, GTK_POS_BOTTOM, FALSE);
        g_object_ref_sink(graph);
        gtk_widget_show_all(GTK_WIDGET(graph));
    }
    else
        gwy_graph_set_model(GWY_GRAPH(graph), gmodel);

    area = GWY_GRAPH_AREA(gwy_graph_get_area(graph));

    GdkRectangle allocation = { .x = 0, .y = 0, .width = width, .height = height };
    gtk_widget_size_allocate(GTK_WIDGET(area), &allocation);
    gtk_widget_show_all(GTK_WIDGET(area));

    gwy_graph_model_get_x_range(gmodel, &min, &max);
    gwy_graph_area_set_x_range(area, min, max);

    g_object_get(gmodel, "y-logarithmic", &is_logscale, NULL);
    gwy_graph_model_get_y_range(gmodel, &min, &max);
    if (is_logscale) {
        if (max > min) {
            d = max/min;
            d = pow(d, 0.07);
            min /= d;
            max *= d;
        }
        else if (max) {
            min = 0.5*max;
            max = 2.0*max;
        }
        else {
            min = 0.1;
            max = 10.0;
        }
    }
    else {
        if (max > min) {
            d = max - min;
            min -= 0.07*d;
            max += 0.07*d;
        }
        else if (max) {
            min = 0.5*max;
            max = 1.5*max;
        }
        else {
            min = -1.0;
            max = 1.0;
        }
    }
    gwy_graph_area_set_y_range(area, min, max);

    cairo_surface_t *surface = gdk_window_create_similar_image_surface(NULL, CAIRO_FORMAT_RGB24, width, height, 0);
    cairo_t *cr = cairo_create(surface);
    // This works:
    GTK_WIDGET_GET_CLASS(area)->draw(GTK_WIDGET(area), cr);
    // This does NOT work because it returns when the widget is now drawable, in particular it must be mapped.
    // TODO: Try again using offscreen windows, now that the coordinate transform in GraphArea is correct. However,
    // it is still a lot more work than just calling the draw method, which we wrote and know what it does.
    //gtk_widget_draw(GTK_WIDGET(area), cr);
    cairo_surface_flush(surface);
    big_pixbuf = gdk_pixbuf_get_from_surface(surface, 0, 0, width, height);
    cairo_destroy(cr);
    cairo_surface_destroy(surface);

    if (width == max_width && height == max_height)
        pixbuf = big_pixbuf;
    else {
        pixbuf = gdk_pixbuf_scale_simple(big_pixbuf, max_width, max_height, GDK_INTERP_BILINEAR);
        g_object_unref(big_pixbuf);
    }

    /* Do not keep reference to the live graph model. */
    gwy_graph_set_model(graph, dummy_model);

    return pixbuf;
}

/**
 * gwy_render_cmap_thumbnail:
 * @data: A data container.
 * @id: Volume data id.
 * @max_width: Maximum width of the created pixbuf, it must be at least 2.
 * @max_height: Maximum height of the created pixbuf, it must be at least 2.
 *
 * Creates a curve map thumbnail.
 *
 * Returns: A newly created pixbuf with curve map data thumbnail.  It keeps the aspect ratio of the lawn preview while
 *          not exceeding @max_width and @max_height.
 **/
GdkPixbuf*
gwy_render_cmap_thumbnail(GwyFile *data,
                          gint id,
                          gint max_width,
                          gint max_height)
{
    g_return_val_if_fail(GWY_IS_CONTAINER(data), NULL);
    g_return_val_if_fail(id >= 0, NULL);
    g_return_val_if_fail(max_width > 1 && max_height > 1, NULL);

    GwyLawn *lawn;
    GwyField *dfield = NULL;

    if (!(lawn = gwy_file_get_cmap(data, id)))
        return NULL;

    if (!gwy_container_gis_object(GWY_CONTAINER(data), gwy_file_key_cmap_picture(id), &dfield)) {
        GdkPixbuf *pixbuf = gdk_pixbuf_new(GDK_COLORSPACE_RGB, FALSE, 8, max_width, max_height);
        gdk_pixbuf_fill(pixbuf, 0);
        return pixbuf;
    }

    const gchar *gradient = gwy_file_get_palette(data, GWY_FILE_CMAP, id);
    return render_image_thumbnail(dfield, gradient, GWY_COLOR_MAPPING_FULL, max_width, max_height, NULL, NULL);
}

/**
 * SECTION: thumbnail
 * @title: Thumbnail
 * @short_description: Utility functions for creating data thumbnails.
 **/

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
